"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lightsail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lightsail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lightsail';
        this.actionList = {
            "AllocateStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html",
                "description": "Creates a static IP address that can be attached to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "AttachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html",
                "description": "Attaches a disk to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    }
                }
            },
            "AttachInstancesToLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html",
                "description": "Attaches one or more instances to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html",
                "description": "Attaches a TLS certificate to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html",
                "description": "Attaches a static IP address to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "CloseInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html",
                "description": "Closes a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html",
                "description": "Copies a snapshot from one AWS Region to another in Amazon Lightsail.",
                "accessLevel": "Write"
            },
            "CreateCloudFormationStack": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html",
                "description": "Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "CreateDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html",
                "description": "Creates a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html",
                "description": "Creates a disk from snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html",
                "description": "Creates a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html",
                "description": "Creates a domain resource for the specified domain name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html",
                "description": "Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "CreateInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html",
                "description": "Creates an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html",
                "description": "Creates one or more instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstancesFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html",
                "description": "Creates one or more instances based on an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html",
                "description": "Creates a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html",
                "description": "Creates a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html",
                "description": "Creates a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "CreateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html",
                "description": "Creates a new relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html",
                "description": "Creates a new relational database from a snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html",
                "description": "Creates a relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html",
                "description": "Deletes a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html",
                "description": "Deletes a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html",
                "description": "Deletes a domain resource and all of its DNS records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html",
                "description": "Deletes a DNS record entry for a domain resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html",
                "description": "Deletes an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html",
                "description": "Deletes an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html",
                "description": "Deletes a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "DeleteKnownHostKeys": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html",
                "description": "Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html",
                "description": "Deletes a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Deletes a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html",
                "description": "Deletes a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html",
                "description": "Deletes relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                }
            },
            "DetachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html",
                "description": "Detaches a disk from an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DetachInstancesFromLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Detaches one or more instances from a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DetachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html",
                "description": "Detaches a static IP from an instance to which it is attached.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "DownloadDefaultKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html",
                "description": "Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "ExportSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html",
                "description": "Exports an Amazon Lightsail snapshot to Amazon EC2.",
                "accessLevel": "Write"
            },
            "GetActiveNames": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html",
                "description": "Returns the names of all active (not deleted) resources.",
                "accessLevel": "Read"
            },
            "GetBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html",
                "description": "Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html",
                "description": "Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetCloudFormationStackRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html",
                "description": "Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "CloudFormationStackRecord": {
                        "required": true
                    }
                }
            },
            "GetDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html",
                "description": "Returns information about a disk.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html",
                "description": "Returns information about a disk snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html",
                "description": "Returns information about all disk snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDisks": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html",
                "description": "Returns information about all disks.",
                "accessLevel": "List"
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html",
                "description": "Returns DNS records for a domain resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetDomains": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html",
                "description": "Returns DNS records for all domain resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetExportSnapshotRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html",
                "description": "Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.",
                "accessLevel": "List",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html",
                "description": "Returns information about an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceAccessDetails": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html",
                "description": "Returns temporary keys you can use to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html",
                "description": "Returns the data points for the specified metric of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstancePortStates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html",
                "description": "Returns the port states of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html",
                "description": "Returns information about an instance snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html",
                "description": "Returns information about all instance snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceState": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html",
                "description": "Returns the state of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html",
                "description": "Returns information about all instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html",
                "description": "Returns information about a key pair.",
                "accessLevel": "List",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetKeyPairs": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html",
                "description": "Returns information about all key pairs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html",
                "description": "Returns information about a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html",
                "description": "Returns the data points for the specified metric of a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerTlsCertificates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Returns information about a load balancer TLS certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancers": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html",
                "description": "Returns information about load balancers.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html",
                "description": "Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperations": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html",
                "description": "Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperationsForResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html",
                "description": "Returns operations for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                }
            },
            "GetRegions": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html",
                "description": "Returns a list of all valid AWS Regions for Amazon Lightsail.",
                "accessLevel": "List"
            },
            "GetRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html",
                "description": "Returns information about a relational database.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html",
                "description": "Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html",
                "description": "Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html",
                "description": "Returns events for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html",
                "description": "Returns events for the specified log stream of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogStreams": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html",
                "description": "Returns the log streams available for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseMasterUserPassword": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html",
                "description": "Returns the master user password of a relational database.",
                "accessLevel": "Write"
            },
            "GetRelationalDatabaseMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html",
                "description": "Returns the data points for the specified metric of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html",
                "description": "Returns the parameters of a relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html",
                "description": "Returns information about a relational database snapshot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html",
                "description": "Returns information about all relational database snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabases": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html",
                "description": "Return information about all relational databases.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html",
                "description": "Returns information about a static IP.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "GetStaticIps": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html",
                "description": "Returns information about all static IPs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html",
                "description": "Imports a public key from a key pair.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "IsVpcPeered": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html",
                "description": "Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.",
                "accessLevel": "Read"
            },
            "OpenInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html",
                "description": "Adds, or opens a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "PeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html",
                "description": "Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.",
                "accessLevel": "Write"
            },
            "PutInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html",
                "description": "Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html",
                "description": "Reboots an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html",
                "description": "Reboots a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "ReleaseStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html",
                "description": "Deletes a static IP.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html",
                "description": "Starts an instance that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StartRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html",
                "description": "Starts a relational database that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html",
                "description": "Stops an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StopRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html",
                "description": "Stops a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnpeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html",
                "description": "Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html",
                "description": "Updates a domain recordset after it is created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "UpdateLoadBalancerAttribute": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Updates a load balancer attribute, such as the health check path and session stickiness.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html",
                "description": "Updates a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html",
                "description": "Updates the parameters of a relational database.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Domain": {
                "name": "Domain",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Instance": {
                "name": "Instance",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "InstanceSnapshot": {
                "name": "InstanceSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "KeyPair": {
                "name": "KeyPair",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StaticIp": {
                "name": "StaticIp",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Disk": {
                "name": "Disk",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DiskSnapshot": {
                "name": "DiskSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LoadBalancer": {
                "name": "LoadBalancer",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PeeredVpc": {
                "name": "PeeredVpc",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}",
                "conditionKeys": []
            },
            "LoadBalancerTlsCertificate": {
                "name": "LoadBalancerTlsCertificate",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}",
                "conditionKeys": []
            },
            "ExportSnapshotRecord": {
                "name": "ExportSnapshotRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}",
                "conditionKeys": []
            },
            "CloudFormationStackRecord": {
                "name": "CloudFormationStackRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}",
                "conditionKeys": []
            },
            "RelationalDatabase": {
                "name": "RelationalDatabase",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "RelationalDatabaseSnapshot": {
                "name": "RelationalDatabaseSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a static IP address that can be attached to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html
     */
    allocateStaticIp() {
        this.add('lightsail:AllocateStaticIp');
        return this;
    }
    /**
     * Attaches a disk to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html
     */
    attachDisk() {
        this.add('lightsail:AttachDisk');
        return this;
    }
    /**
     * Attaches one or more instances to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html
     */
    attachInstancesToLoadBalancer() {
        this.add('lightsail:AttachInstancesToLoadBalancer');
        return this;
    }
    /**
     * Attaches a TLS certificate to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html
     */
    attachLoadBalancerTlsCertificate() {
        this.add('lightsail:AttachLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Attaches a static IP address to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html
     */
    attachStaticIp() {
        this.add('lightsail:AttachStaticIp');
        return this;
    }
    /**
     * Closes a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html
     */
    closeInstancePublicPorts() {
        this.add('lightsail:CloseInstancePublicPorts');
        return this;
    }
    /**
     * Copies a snapshot from one AWS Region to another in Amazon Lightsail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html
     */
    copySnapshot() {
        this.add('lightsail:CopySnapshot');
        return this;
    }
    /**
     * Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html
     */
    createCloudFormationStack() {
        this.add('lightsail:CreateCloudFormationStack');
        return this;
    }
    /**
     * Creates a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html
     */
    createDisk() {
        this.add('lightsail:CreateDisk');
        return this;
    }
    /**
     * Creates a disk from snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html
     */
    createDiskFromSnapshot() {
        this.add('lightsail:CreateDiskFromSnapshot');
        return this;
    }
    /**
     * Creates a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html
     */
    createDiskSnapshot() {
        this.add('lightsail:CreateDiskSnapshot');
        return this;
    }
    /**
     * Creates a domain resource for the specified domain name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html
     */
    createDomain() {
        this.add('lightsail:CreateDomain');
        return this;
    }
    /**
     * Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html
     */
    createDomainEntry() {
        this.add('lightsail:CreateDomainEntry');
        return this;
    }
    /**
     * Creates an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html
     */
    createInstanceSnapshot() {
        this.add('lightsail:CreateInstanceSnapshot');
        return this;
    }
    /**
     * Creates one or more instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html
     */
    createInstances() {
        this.add('lightsail:CreateInstances');
        return this;
    }
    /**
     * Creates one or more instances based on an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html
     */
    createInstancesFromSnapshot() {
        this.add('lightsail:CreateInstancesFromSnapshot');
        return this;
    }
    /**
     * Creates a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html
     */
    createKeyPair() {
        this.add('lightsail:CreateKeyPair');
        return this;
    }
    /**
     * Creates a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html
     */
    createLoadBalancer() {
        this.add('lightsail:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html
     */
    createLoadBalancerTlsCertificate() {
        this.add('lightsail:CreateLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Creates a new relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html
     */
    createRelationalDatabase() {
        this.add('lightsail:CreateRelationalDatabase');
        return this;
    }
    /**
     * Creates a new relational database from a snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html
     */
    createRelationalDatabaseFromSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseFromSnapshot');
        return this;
    }
    /**
     * Creates a relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html
     */
    createRelationalDatabaseSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Deletes a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html
     */
    deleteDisk() {
        this.add('lightsail:DeleteDisk');
        return this;
    }
    /**
     * Deletes a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html
     */
    deleteDiskSnapshot() {
        this.add('lightsail:DeleteDiskSnapshot');
        return this;
    }
    /**
     * Deletes a domain resource and all of its DNS records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('lightsail:DeleteDomain');
        return this;
    }
    /**
     * Deletes a DNS record entry for a domain resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html
     */
    deleteDomainEntry() {
        this.add('lightsail:DeleteDomainEntry');
        return this;
    }
    /**
     * Deletes an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html
     */
    deleteInstance() {
        this.add('lightsail:DeleteInstance');
        return this;
    }
    /**
     * Deletes an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html
     */
    deleteInstanceSnapshot() {
        this.add('lightsail:DeleteInstanceSnapshot');
        return this;
    }
    /**
     * Deletes a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html
     */
    deleteKeyPair() {
        this.add('lightsail:DeleteKeyPair');
        return this;
    }
    /**
     * Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html
     */
    deleteKnownHostKeys() {
        this.add('lightsail:DeleteKnownHostKeys');
        return this;
    }
    /**
     * Deletes a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html
     */
    deleteLoadBalancer() {
        this.add('lightsail:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    deleteLoadBalancerTlsCertificate() {
        this.add('lightsail:DeleteLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Deletes a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html
     */
    deleteRelationalDatabase() {
        this.add('lightsail:DeleteRelationalDatabase');
        return this;
    }
    /**
     * Deletes relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html
     */
    deleteRelationalDatabaseSnapshot() {
        this.add('lightsail:DeleteRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Detaches a disk from an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html
     */
    detachDisk() {
        this.add('lightsail:DetachDisk');
        return this;
    }
    /**
     * Detaches one or more instances from a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    detachInstancesFromLoadBalancer() {
        this.add('lightsail:DetachInstancesFromLoadBalancer');
        return this;
    }
    /**
     * Detaches a static IP from an instance to which it is attached.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html
     */
    detachStaticIp() {
        this.add('lightsail:DetachStaticIp');
        return this;
    }
    /**
     * Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html
     */
    downloadDefaultKeyPair() {
        this.add('lightsail:DownloadDefaultKeyPair');
        return this;
    }
    /**
     * Exports an Amazon Lightsail snapshot to Amazon EC2.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html
     */
    exportSnapshot() {
        this.add('lightsail:ExportSnapshot');
        return this;
    }
    /**
     * Returns the names of all active (not deleted) resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html
     */
    getActiveNames() {
        this.add('lightsail:GetActiveNames');
        return this;
    }
    /**
     * Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html
     */
    getBlueprints() {
        this.add('lightsail:GetBlueprints');
        return this;
    }
    /**
     * Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html
     */
    getBundles() {
        this.add('lightsail:GetBundles');
        return this;
    }
    /**
     * Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html
     */
    getCloudFormationStackRecords() {
        this.add('lightsail:GetCloudFormationStackRecords');
        return this;
    }
    /**
     * Returns information about a disk.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html
     */
    getDisk() {
        this.add('lightsail:GetDisk');
        return this;
    }
    /**
     * Returns information about a disk snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html
     */
    getDiskSnapshot() {
        this.add('lightsail:GetDiskSnapshot');
        return this;
    }
    /**
     * Returns information about all disk snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html
     */
    getDiskSnapshots() {
        this.add('lightsail:GetDiskSnapshots');
        return this;
    }
    /**
     * Returns information about all disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html
     */
    getDisks() {
        this.add('lightsail:GetDisks');
        return this;
    }
    /**
     * Returns DNS records for a domain resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html
     */
    getDomain() {
        this.add('lightsail:GetDomain');
        return this;
    }
    /**
     * Returns DNS records for all domain resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html
     */
    getDomains() {
        this.add('lightsail:GetDomains');
        return this;
    }
    /**
     * Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html
     */
    getExportSnapshotRecords() {
        this.add('lightsail:GetExportSnapshotRecords');
        return this;
    }
    /**
     * Returns information about an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html
     */
    getInstance() {
        this.add('lightsail:GetInstance');
        return this;
    }
    /**
     * Returns temporary keys you can use to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html
     */
    getInstanceAccessDetails() {
        this.add('lightsail:GetInstanceAccessDetails');
        return this;
    }
    /**
     * Returns the data points for the specified metric of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html
     */
    getInstanceMetricData() {
        this.add('lightsail:GetInstanceMetricData');
        return this;
    }
    /**
     * Returns the port states of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html
     */
    getInstancePortStates() {
        this.add('lightsail:GetInstancePortStates');
        return this;
    }
    /**
     * Returns information about an instance snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html
     */
    getInstanceSnapshot() {
        this.add('lightsail:GetInstanceSnapshot');
        return this;
    }
    /**
     * Returns information about all instance snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html
     */
    getInstanceSnapshots() {
        this.add('lightsail:GetInstanceSnapshots');
        return this;
    }
    /**
     * Returns the state of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html
     */
    getInstanceState() {
        this.add('lightsail:GetInstanceState');
        return this;
    }
    /**
     * Returns information about all instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html
     */
    getInstances() {
        this.add('lightsail:GetInstances');
        return this;
    }
    /**
     * Returns information about a key pair.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html
     */
    getKeyPair() {
        this.add('lightsail:GetKeyPair');
        return this;
    }
    /**
     * Returns information about all key pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html
     */
    getKeyPairs() {
        this.add('lightsail:GetKeyPairs');
        return this;
    }
    /**
     * Returns information about a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html
     */
    getLoadBalancer() {
        this.add('lightsail:GetLoadBalancer');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html
     */
    getLoadBalancerMetricData() {
        this.add('lightsail:GetLoadBalancerMetricData');
        return this;
    }
    /**
     * Returns information about a load balancer TLS certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    getLoadBalancerTlsCertificates() {
        this.add('lightsail:GetLoadBalancerTlsCertificates');
        return this;
    }
    /**
     * Returns information about load balancers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html
     */
    getLoadBalancers() {
        this.add('lightsail:GetLoadBalancers');
        return this;
    }
    /**
     * Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html
     */
    getOperation() {
        this.add('lightsail:GetOperation');
        return this;
    }
    /**
     * Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html
     */
    getOperations() {
        this.add('lightsail:GetOperations');
        return this;
    }
    /**
     * Returns operations for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html
     */
    getOperationsForResource() {
        this.add('lightsail:GetOperationsForResource');
        return this;
    }
    /**
     * Returns a list of all valid AWS Regions for Amazon Lightsail.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html
     */
    getRegions() {
        this.add('lightsail:GetRegions');
        return this;
    }
    /**
     * Returns information about a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html
     */
    getRelationalDatabase() {
        this.add('lightsail:GetRelationalDatabase');
        return this;
    }
    /**
     * Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html
     */
    getRelationalDatabaseBlueprints() {
        this.add('lightsail:GetRelationalDatabaseBlueprints');
        return this;
    }
    /**
     * Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html
     */
    getRelationalDatabaseBundles() {
        this.add('lightsail:GetRelationalDatabaseBundles');
        return this;
    }
    /**
     * Returns events for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html
     */
    getRelationalDatabaseEvents() {
        this.add('lightsail:GetRelationalDatabaseEvents');
        return this;
    }
    /**
     * Returns events for the specified log stream of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html
     */
    getRelationalDatabaseLogEvents() {
        this.add('lightsail:GetRelationalDatabaseLogEvents');
        return this;
    }
    /**
     * Returns the log streams available for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html
     */
    getRelationalDatabaseLogStreams() {
        this.add('lightsail:GetRelationalDatabaseLogStreams');
        return this;
    }
    /**
     * Returns the master user password of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html
     */
    getRelationalDatabaseMasterUserPassword() {
        this.add('lightsail:GetRelationalDatabaseMasterUserPassword');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html
     */
    getRelationalDatabaseMetricData() {
        this.add('lightsail:GetRelationalDatabaseMetricData');
        return this;
    }
    /**
     * Returns the parameters of a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html
     */
    getRelationalDatabaseParameters() {
        this.add('lightsail:GetRelationalDatabaseParameters');
        return this;
    }
    /**
     * Returns information about a relational database snapshot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html
     */
    getRelationalDatabaseSnapshot() {
        this.add('lightsail:GetRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Returns information about all relational database snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html
     */
    getRelationalDatabaseSnapshots() {
        this.add('lightsail:GetRelationalDatabaseSnapshots');
        return this;
    }
    /**
     * Return information about all relational databases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html
     */
    getRelationalDatabases() {
        this.add('lightsail:GetRelationalDatabases');
        return this;
    }
    /**
     * Returns information about a static IP.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html
     */
    getStaticIp() {
        this.add('lightsail:GetStaticIp');
        return this;
    }
    /**
     * Returns information about all static IPs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html
     */
    getStaticIps() {
        this.add('lightsail:GetStaticIps');
        return this;
    }
    /**
     * Imports a public key from a key pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html
     */
    importKeyPair() {
        this.add('lightsail:ImportKeyPair');
        return this;
    }
    /**
     * Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html
     */
    isVpcPeered() {
        this.add('lightsail:IsVpcPeered');
        return this;
    }
    /**
     * Adds, or opens a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html
     */
    openInstancePublicPorts() {
        this.add('lightsail:OpenInstancePublicPorts');
        return this;
    }
    /**
     * Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html
     */
    peerVpc() {
        this.add('lightsail:PeerVpc');
        return this;
    }
    /**
     * Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html
     */
    putInstancePublicPorts() {
        this.add('lightsail:PutInstancePublicPorts');
        return this;
    }
    /**
     * Reboots an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html
     */
    rebootInstance() {
        this.add('lightsail:RebootInstance');
        return this;
    }
    /**
     * Reboots a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html
     */
    rebootRelationalDatabase() {
        this.add('lightsail:RebootRelationalDatabase');
        return this;
    }
    /**
     * Deletes a static IP.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html
     */
    releaseStaticIp() {
        this.add('lightsail:ReleaseStaticIp');
        return this;
    }
    /**
     * Starts an instance that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html
     */
    startInstance() {
        this.add('lightsail:StartInstance');
        return this;
    }
    /**
     * Starts a relational database that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html
     */
    startRelationalDatabase() {
        this.add('lightsail:StartRelationalDatabase');
        return this;
    }
    /**
     * Stops an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html
     */
    stopInstance() {
        this.add('lightsail:StopInstance');
        return this;
    }
    /**
     * Stops a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html
     */
    stopRelationalDatabase() {
        this.add('lightsail:StopRelationalDatabase');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html
     */
    tagResource() {
        this.add('lightsail:TagResource');
        return this;
    }
    /**
     * Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html
     */
    unpeerVpc() {
        this.add('lightsail:UnpeerVpc');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html
     */
    untagResource() {
        this.add('lightsail:UntagResource');
        return this;
    }
    /**
     * Updates a domain recordset after it is created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html
     */
    updateDomainEntry() {
        this.add('lightsail:UpdateDomainEntry');
        return this;
    }
    /**
     * Updates a load balancer attribute, such as the health check path and session stickiness.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    updateLoadBalancerAttribute() {
        this.add('lightsail:UpdateLoadBalancerAttribute');
        return this;
    }
    /**
     * Updates a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html
     */
    updateRelationalDatabase() {
        this.add('lightsail:UpdateRelationalDatabase');
        return this;
    }
    /**
     * Updates the parameters of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html
     */
    updateRelationalDatabaseParameters() {
        this.add('lightsail:UpdateRelationalDatabaseParameters');
        return this;
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomain(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInstance(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type InstanceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInstanceSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type KeyPair to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onKeyPair(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StaticIp to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStaticIp(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Disk to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDisk(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type DiskSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDiskSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancer to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onLoadBalancer(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type PeeredVpc to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPeeredVpc(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancerTlsCertificate to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadBalancerTlsCertificate(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ExportSnapshotRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExportSnapshotRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CloudFormationStackRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCloudFormationStackRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabase to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRelationalDatabase(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabaseSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRelationalDatabaseSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Lightsail = Lightsail;
//# sourceMappingURL=data:application/json;base64,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