"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie2';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html",
                "description": "Grants permission to accept an Amazon Macie membership invitation",
                "accessLevel": "Write"
            },
            "ArchiveFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-archive.html",
                "description": "Grants permission to archive one or more findings",
                "accessLevel": "Write"
            },
            "BatchGetCustomDataIdentifiers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html",
                "description": "Grants permission to retrieve information about one or more custom data identifiers",
                "accessLevel": "Read",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "CreateClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html",
                "description": "Grants permission to create and define the settings for a classification job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html",
                "description": "Grants permission to create and define the settings for a custom data identifier",
                "accessLevel": "Write",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html",
                "description": "Grants permission to create and define the settings for a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html",
                "description": "Grants permission to send an Amazon Macie membership invitation",
                "accessLevel": "Write"
            },
            "CreateMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members.html",
                "description": "Grants permission to associate an account with an Amazon Macie master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSampleFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html",
                "description": "Grants permission to create sample findings",
                "accessLevel": "Write"
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html",
                "description": "Grants permission to decline Amazon Macie membership invitations",
                "accessLevel": "Write"
            },
            "DeleteCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html",
                "description": "Grants permission to delete a custom data identifier",
                "accessLevel": "Write",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "DeleteFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to delete a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html",
                "description": "Grants permission to delete Amazon Macie membership invitations",
                "accessLevel": "Write"
            },
            "DeleteMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html",
                "description": "Grants permission to delete the association between an Amazon Macie master account and an account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "DescribeBuckets": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html",
                "description": "Grants permission to retrieve statistical and other data about S3 buckets that Amazon Macie monitors and analyzes",
                "accessLevel": "Read"
            },
            "DescribeClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html",
                "description": "Grants permission to retrieve information about the status and settings for a classification job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                }
            },
            "DescribeOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html",
                "description": "Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization",
                "accessLevel": "Read"
            },
            "DisableMacie": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account",
                "accessLevel": "Write"
            },
            "DisableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to disable an account as a delegated administrator of Amazon Macie for an AWS organization",
                "accessLevel": "Write"
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html",
                "description": "Grants an Amazon Macie member account with permission to disassociate from its master account",
                "accessLevel": "Write"
            },
            "DisassociateMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html",
                "description": "Grants an Amazon Macie master account with permission to disassociate from a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "EnableMacie": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to enable and specify the configuration settings for a new Amazon Macie account",
                "accessLevel": "Write"
            },
            "EnableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to enable an account as a delegated administrator of Amazon Macie for an AWS organization",
                "accessLevel": "Write"
            },
            "GetBucketStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html",
                "description": "Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes",
                "accessLevel": "Read"
            },
            "GetClassificationExportConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html",
                "description": "Grants permission to retrieve the settings for exporting data classification results",
                "accessLevel": "Read"
            },
            "GetCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html",
                "description": "Grants permission to retrieve information about the settings for a custom data identifier",
                "accessLevel": "Read",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "GetFindingStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html",
                "description": "Grants permission to retrieve aggregated statistical data about findings",
                "accessLevel": "Read"
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html",
                "description": "Grants permission to retrieve information about one or more findings",
                "accessLevel": "Read"
            },
            "GetFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to retrieve information about the settings for a findings filter",
                "accessLevel": "Read",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html",
                "description": "Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account",
                "accessLevel": "Read"
            },
            "GetMacieSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account",
                "accessLevel": "Read"
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/master.html",
                "description": "Grants permission to retrieve information about the Amazon Macie master account for an account",
                "accessLevel": "Read"
            },
            "GetMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html",
                "description": "Grants permission to retrieve information about an account that's associated with an Amazon Macie master account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "GetUsageStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html",
                "description": "Grants permission to retrieve quotas and aggregated usage data for one or more accounts",
                "accessLevel": "Read"
            },
            "GetUsageTotals": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/usage.html",
                "description": "Grants permission to retrieve aggregated usage data for an account",
                "accessLevel": "Read"
            },
            "ListClassificationJobs": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html",
                "description": "Grants permission to retrieve information about the status and settings for one or more classification jobs",
                "accessLevel": "List"
            },
            "ListCustomDataIdentifiers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html",
                "description": "Grants permission to retrieve information about all custom data identifiers",
                "accessLevel": "List"
            },
            "ListFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings.html",
                "description": "Grants permission to retrieve a subset of information about one or more findings",
                "accessLevel": "List"
            },
            "ListFindingsFilters": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html",
                "description": "Grants permission to retrieve information about all findings filters",
                "accessLevel": "List"
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html",
                "description": "Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members.html",
                "description": "Grants permission to retrieve information about all the accounts that are associated with an Amazon Macie master account",
                "accessLevel": "List"
            },
            "ListOrganizationAdminAccounts": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to retrieve information about the delegated, Amazon Macie administrator account for an AWS organization",
                "accessLevel": "List"
            },
            "ListTagsForResources": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to retrieve the tags for an Amazon Macie resource or member account",
                "accessLevel": "List"
            },
            "PutClassificationExportConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html",
                "description": "Grants permission to create or update the settings for exporting data classification results",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to add or update the tags for an Amazon Macie resource or member account",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html",
                "description": "Grants permission to test a custom data identifier",
                "accessLevel": "Write"
            },
            "UnarchiveFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-unarchive.html",
                "description": "Grants permission to reactivate (unarchive) one or more findings",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to remove tags from an Amazon Macie resource or member account",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html",
                "description": "Grants permission to cancel a classification job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to update the settings for a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateMacieSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to suspend or re-enable an Amazon Macie account, or update the configuration settings for a Macie account",
                "accessLevel": "Write"
            },
            "UpdateMemberSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html",
                "description": "Grants an Amazon Macie master account with permission to suspend or re-enable a member account",
                "accessLevel": "Write"
            },
            "UpdateOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html",
                "description": "Grants permission to update Amazon Macie configuration settings for an AWS organization",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "ClassificationJob": {
                "name": "ClassificationJob",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:classification-job/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CustomDataIdentifier": {
                "name": "CustomDataIdentifier",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html",
                "arn": "arn:${Partition}:macie2::${Account}:custom-data-identifier/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Member": {
                "name": "Member",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:member/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "FindingsFilter": {
                "name": "FindingsFilter",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:findings-filter/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html
     */
    acceptInvitation() {
        this.add('macie2:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-archive.html
     */
    archiveFindings() {
        this.add('macie2:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more custom data identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html
     */
    batchGetCustomDataIdentifiers() {
        this.add('macie2:BatchGetCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a classification job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html
     */
    createClassificationJob() {
        this.add('macie2:CreateClassificationJob');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html
     */
    createCustomDataIdentifier() {
        this.add('macie2:CreateCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    createFindingsFilter() {
        this.add('macie2:CreateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to send an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    createInvitations() {
        this.add('macie2:CreateInvitations');
        return this;
    }
    /**
     * Grants permission to associate an account with an Amazon Macie master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    createMember() {
        this.add('macie2:CreateMember');
        return this;
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html
     */
    createSampleFindings() {
        this.add('macie2:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to decline Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html
     */
    declineInvitations() {
        this.add('macie2:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    deleteCustomDataIdentifier() {
        this.add('macie2:DeleteCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    deleteFindingsFilter() {
        this.add('macie2:DeleteFindingsFilter');
        return this;
    }
    /**
     * Grants permission to delete Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html
     */
    deleteInvitations() {
        this.add('macie2:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete the association between an Amazon Macie master account and an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    deleteMember() {
        this.add('macie2:DeleteMember');
        return this;
    }
    /**
     * Grants permission to retrieve statistical and other data about S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html
     */
    describeBuckets() {
        this.add('macie2:DescribeBuckets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for a classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    describeClassificationJob() {
        this.add('macie2:DescribeClassificationJob');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    describeOrganizationConfiguration() {
        this.add('macie2:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    disableMacie() {
        this.add('macie2:DisableMacie');
        return this;
    }
    /**
     * Grants permission to disable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    disableOrganizationAdminAccount() {
        this.add('macie2:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie member account with permission to disassociate from its master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html
     */
    disassociateFromMasterAccount() {
        this.add('macie2:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to disassociate from a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html
     */
    disassociateMember() {
        this.add('macie2:DisassociateMember');
        return this;
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    enableMacie() {
        this.add('macie2:EnableMacie');
        return this;
    }
    /**
     * Grants permission to enable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    enableOrganizationAdminAccount() {
        this.add('macie2:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html
     */
    getBucketStatistics() {
        this.add('macie2:GetBucketStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve the settings for exporting data classification results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    getClassificationExportConfiguration() {
        this.add('macie2:GetClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a custom data identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    getCustomDataIdentifier() {
        this.add('macie2:GetCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data about findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html
     */
    getFindingStatistics() {
        this.add('macie2:GetFindingStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html
     */
    getFindings() {
        this.add('macie2:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a findings filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    getFindingsFilter() {
        this.add('macie2:GetFindingsFilter');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html
     */
    getInvitationsCount() {
        this.add('macie2:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    getMacieSession() {
        this.add('macie2:GetMacieSession');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie master account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master.html
     */
    getMasterAccount() {
        this.add('macie2:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Macie master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    getMember() {
        this.add('macie2:GetMember');
        return this;
    }
    /**
     * Grants permission to retrieve quotas and aggregated usage data for one or more accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html
     */
    getUsageStatistics() {
        this.add('macie2:GetUsageStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage.html
     */
    getUsageTotals() {
        this.add('macie2:GetUsageTotals');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for one or more classification jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html
     */
    listClassificationJobs() {
        this.add('macie2:ListClassificationJobs');
        return this;
    }
    /**
     * Grants permission to retrieve information about all custom data identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html
     */
    listCustomDataIdentifiers() {
        this.add('macie2:ListCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings.html
     */
    listFindings() {
        this.add('macie2:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    listFindingsFilters() {
        this.add('macie2:ListFindingsFilters');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    listInvitations() {
        this.add('macie2:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the accounts that are associated with an Amazon Macie master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    listMembers() {
        this.add('macie2:ListMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about the delegated, Amazon Macie administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    listOrganizationAdminAccounts() {
        this.add('macie2:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Macie resource or member account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    listTagsForResources() {
        this.add('macie2:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to create or update the settings for exporting data classification results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    putClassificationExportConfiguration() {
        this.add('macie2:PutClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to add or update the tags for an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    tagResource() {
        this.add('macie2:TagResource');
        return this;
    }
    /**
     * Grants permission to test a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html
     */
    testCustomDataIdentifier() {
        this.add('macie2:TestCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to reactivate (unarchive) one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-unarchive.html
     */
    unarchiveFindings() {
        this.add('macie2:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    untagResource() {
        this.add('macie2:UntagResource');
        return this;
    }
    /**
     * Grants permission to cancel a classification job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    updateClassificationJob() {
        this.add('macie2:UpdateClassificationJob');
        return this;
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    updateFindingsFilter() {
        this.add('macie2:UpdateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to suspend or re-enable an Amazon Macie account, or update the configuration settings for a Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    updateMacieSession() {
        this.add('macie2:UpdateMacieSession');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to suspend or re-enable a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html
     */
    updateMemberSession() {
        this.add('macie2:UpdateMemberSession');
        return this;
    }
    /**
     * Grants permission to update Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    updateOrganizationConfiguration() {
        this.add('macie2:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ClassificationJob to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onClassificationJob(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:classification-job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CustomDataIdentifier to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCustomDataIdentifier(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:custom-data-identifier/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Member to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMember(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:member/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type FindingsFilter to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFindingsFilter(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:findings-filter/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Macie2 = Macie2;
//# sourceMappingURL=data:application/json;base64,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