"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesPinpoint = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpointemailservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesPinpoint extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpointemailservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses-pinpoint';
        this.actionList = {
            "CreateConfigurationSet": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSet.html",
                "description": "Create a configuration set. Configuration sets are groups of rules that you can apply to the emails you send using Amazon Pinpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSetEventDestination.html",
                "description": "Create an event destination",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "CreateDedicatedIpPool": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDedicatedIpPool.html",
                "description": "Create a new pool of dedicated IP addresses",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-ip-pool": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateDeliverabilityTestReport": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDeliverabilityTestReport.html",
                "description": "Create a new predictive inbox placement test.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateEmailIdentity": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateEmailIdentity.html",
                "description": "Verifies an email identity for use with Amazon Pinpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteConfigurationSet": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSet.html",
                "description": "Delete an existing configuration set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "DeleteConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSetEventDestination.html",
                "description": "Delete an event destination",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "DeleteDedicatedIpPool": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteDedicatedIpPool.html",
                "description": "Delete a dedicated IP pool",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-ip-pool": {
                        "required": true
                    }
                }
            },
            "DeleteEmailIdentity": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteEmailIdentity.html",
                "description": "Deletes an email identity that you previously verified for use with Amazon Pinpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "GetAccount": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetAccount.html",
                "description": "Obtain information about the email-sending status and capabilities",
                "accessLevel": "Read"
            },
            "GetBlacklistReports": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetBlacklistReports.html",
                "description": "Retrieve a list of the blacklists that your dedicated IP addresses appear on",
                "accessLevel": "Read"
            },
            "GetConfigurationSet": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSet.html",
                "description": "Get information about an existing configuration set",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "GetConfigurationSetEventDestinations": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSetEventDestinations.html",
                "description": "Retrieve a list of event destinations that are associated with a configuration set",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "GetDedicatedIp": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIp.html",
                "description": "Get information about a dedicated IP address",
                "accessLevel": "Read"
            },
            "GetDedicatedIps": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIps.html",
                "description": "List the dedicated IP addresses that are associated with your Amazon Pinpoint account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dedicated-ip-pool": {
                        "required": true
                    }
                }
            },
            "GetDeliverabilityDashboardOptions": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityDashboardOptions.html",
                "description": "Show the status of the Deliverability dashboard",
                "accessLevel": "Read"
            },
            "GetDeliverabilityTestReport": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityTestReport.html",
                "description": "Retrieve the results of a predictive inbox placement test",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deliverability-test-report": {
                        "required": true
                    }
                }
            },
            "GetDomainStatisticsReport": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDomainStatisticsReport.html",
                "description": "Retrieve inbox placement and engagement rates for the domains that you use to send email",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "GetEmailIdentity": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetEmailIdentity.html",
                "description": "Provides information about a specific identity associated with your Amazon Pinpoint account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "ListConfigurationSets": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListConfigurationSets.html",
                "description": "List all of the configuration sets associated with your Amazon Pinpoint account in the current region",
                "accessLevel": "List"
            },
            "ListDedicatedIpPools": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDedicatedIpPools.html",
                "description": "List all of the dedicated IP pools that exist in your Amazon Pinpoint account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListDeliverabilityTestReports": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDeliverabilityTestReports.html",
                "description": "Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses",
                "accessLevel": "List"
            },
            "ListEmailIdentities": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListEmailIdentities.html",
                "description": "Returns a list of all of the email identities that are associated with your Amazon Pinpoint account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListTagsForResource.html",
                "description": "Retrieve a list of the tags (keys and values) that are associated with a specific resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configuration-set": {
                        "required": false
                    },
                    "dedicated-ip-pool": {
                        "required": false
                    },
                    "deliverability-test-report": {
                        "required": false
                    },
                    "identity": {
                        "required": false
                    }
                }
            },
            "PutAccountDedicatedIpWarmupAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountDedicatedIpWarmupAttributes.html",
                "description": "Enable or disable the automatic warm-up feature for dedicated IP addresses",
                "accessLevel": "Write"
            },
            "PutAccountSendingAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountSendingAttributes.html",
                "description": "Enable or disable the ability of your account to send email",
                "accessLevel": "Write"
            },
            "PutConfigurationSetDeliveryOptions": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetDeliveryOptions.html",
                "description": "Associate a configuration set with a dedicated IP pool",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "PutConfigurationSetReputationOptions": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetReputationOptions.html",
                "description": "Enable or disable collection of reputation metrics for emails that you send using a particular configuration set in a specific AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "PutConfigurationSetSendingOptions": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetSendingOptions.html",
                "description": "Enable or disable email sending for messages that use a particular configuration set in a specific AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "PutConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetTrackingOptions.html",
                "description": "Specify a custom domain to use for open and click tracking elements in email that you send using Amazon Pinpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            },
            "PutDedicatedIpInPool": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpInPool.html",
                "description": "Move a dedicated IP address to an existing dedicated IP pool",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-ip-pool": {
                        "required": true
                    }
                }
            },
            "PutDedicatedIpWarmupAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpWarmupAttributes.html",
                "description": "Put Dedicated IP warm up attributes",
                "accessLevel": "Write"
            },
            "PutDeliverabilityDashboardOption": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDeliverabilityDashboardOption.html",
                "description": "Enable or disable the Deliverability dashboard",
                "accessLevel": "Write"
            },
            "PutEmailIdentityDkimAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityDkimAttributes.html",
                "description": "Used to enable or disable DKIM authentication for an email identity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "PutEmailIdentityFeedbackAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityFeedbackAttributes.html",
                "description": "Used to enable or disable feedback forwarding for an identity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "PutEmailIdentityMailFromAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityMailFromAttributes.html",
                "description": "Used to enable or disable the custom Mail-From domain configuration for an email identity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "SendEmail": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_SendEmail.html",
                "description": "Sends an email message",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_TagResource.html",
                "description": "Add one or more tags (keys and values) to a specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "configuration-set": {
                        "required": false
                    },
                    "dedicated-ip-pool": {
                        "required": false
                    },
                    "deliverability-test-report": {
                        "required": false
                    },
                    "identity": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UntagResource.html",
                "description": "Remove one or more tags (keys and values) from a specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "configuration-set": {
                        "required": false
                    },
                    "dedicated-ip-pool": {
                        "required": false
                    },
                    "deliverability-test-report": {
                        "required": false
                    },
                    "identity": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UpdateConfigurationSetEventDestination.html",
                "description": "Update the configuration of an event destination for a configuration set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration-set": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "configuration-set": {
                "name": "configuration-set",
                "url": "",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dedicated-ip-pool": {
                "name": "dedicated-ip-pool",
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DedicatedIp.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:dedicated-ip-pool/${CustomVerificationEmailTemplateName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deliverability-test-report": {
                "name": "deliverability-test-report",
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeliverabilityTestReport.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:deliverability-test-report/${CustomVerificationEmailTemplateName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "event-destination": {
                "name": "event-destination",
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_EventDestination.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_IdentityInfo.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Create a configuration set. Configuration sets are groups of rules that you can apply to the emails you send using Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSet.html
     */
    createConfigurationSet() {
        this.add('ses-pinpoint:CreateConfigurationSet');
        return this;
    }
    /**
     * Create an event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    createConfigurationSetEventDestination() {
        this.add('ses-pinpoint:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDedicatedIpPool.html
     */
    createDedicatedIpPool() {
        this.add('ses-pinpoint:CreateDedicatedIpPool');
        return this;
    }
    /**
     * Create a new predictive inbox placement test.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDeliverabilityTestReport.html
     */
    createDeliverabilityTestReport() {
        this.add('ses-pinpoint:CreateDeliverabilityTestReport');
        return this;
    }
    /**
     * Verifies an email identity for use with Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateEmailIdentity.html
     */
    createEmailIdentity() {
        this.add('ses-pinpoint:CreateEmailIdentity');
        return this;
    }
    /**
     * Delete an existing configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSet.html
     */
    deleteConfigurationSet() {
        this.add('ses-pinpoint:DeleteConfigurationSet');
        return this;
    }
    /**
     * Delete an event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    deleteConfigurationSetEventDestination() {
        this.add('ses-pinpoint:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteDedicatedIpPool.html
     */
    deleteDedicatedIpPool() {
        this.add('ses-pinpoint:DeleteDedicatedIpPool');
        return this;
    }
    /**
     * Deletes an email identity that you previously verified for use with Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteEmailIdentity.html
     */
    deleteEmailIdentity() {
        this.add('ses-pinpoint:DeleteEmailIdentity');
        return this;
    }
    /**
     * Obtain information about the email-sending status and capabilities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetAccount.html
     */
    getAccount() {
        this.add('ses-pinpoint:GetAccount');
        return this;
    }
    /**
     * Retrieve a list of the blacklists that your dedicated IP addresses appear on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetBlacklistReports.html
     */
    getBlacklistReports() {
        this.add('ses-pinpoint:GetBlacklistReports');
        return this;
    }
    /**
     * Get information about an existing configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSet.html
     */
    getConfigurationSet() {
        this.add('ses-pinpoint:GetConfigurationSet');
        return this;
    }
    /**
     * Retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSetEventDestinations.html
     */
    getConfigurationSetEventDestinations() {
        this.add('ses-pinpoint:GetConfigurationSetEventDestinations');
        return this;
    }
    /**
     * Get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIp.html
     */
    getDedicatedIp() {
        this.add('ses-pinpoint:GetDedicatedIp');
        return this;
    }
    /**
     * List the dedicated IP addresses that are associated with your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIps.html
     */
    getDedicatedIps() {
        this.add('ses-pinpoint:GetDedicatedIps');
        return this;
    }
    /**
     * Show the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityDashboardOptions.html
     */
    getDeliverabilityDashboardOptions() {
        this.add('ses-pinpoint:GetDeliverabilityDashboardOptions');
        return this;
    }
    /**
     * Retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityTestReport.html
     */
    getDeliverabilityTestReport() {
        this.add('ses-pinpoint:GetDeliverabilityTestReport');
        return this;
    }
    /**
     * Retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDomainStatisticsReport.html
     */
    getDomainStatisticsReport() {
        this.add('ses-pinpoint:GetDomainStatisticsReport');
        return this;
    }
    /**
     * Provides information about a specific identity associated with your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetEmailIdentity.html
     */
    getEmailIdentity() {
        this.add('ses-pinpoint:GetEmailIdentity');
        return this;
    }
    /**
     * List all of the configuration sets associated with your Amazon Pinpoint account in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListConfigurationSets.html
     */
    listConfigurationSets() {
        this.add('ses-pinpoint:ListConfigurationSets');
        return this;
    }
    /**
     * List all of the dedicated IP pools that exist in your Amazon Pinpoint account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDedicatedIpPools.html
     */
    listDedicatedIpPools() {
        this.add('ses-pinpoint:ListDedicatedIpPools');
        return this;
    }
    /**
     * Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDeliverabilityTestReports.html
     */
    listDeliverabilityTestReports() {
        this.add('ses-pinpoint:ListDeliverabilityTestReports');
        return this;
    }
    /**
     * Returns a list of all of the email identities that are associated with your Amazon Pinpoint account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListEmailIdentities.html
     */
    listEmailIdentities() {
        this.add('ses-pinpoint:ListEmailIdentities');
        return this;
    }
    /**
     * Retrieve a list of the tags (keys and values) that are associated with a specific resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('ses-pinpoint:ListTagsForResource');
        return this;
    }
    /**
     * Enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    putAccountDedicatedIpWarmupAttributes() {
        this.add('ses-pinpoint:PutAccountDedicatedIpWarmupAttributes');
        return this;
    }
    /**
     * Enable or disable the ability of your account to send email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountSendingAttributes.html
     */
    putAccountSendingAttributes() {
        this.add('ses-pinpoint:PutAccountSendingAttributes');
        return this;
    }
    /**
     * Associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetDeliveryOptions.html
     */
    putConfigurationSetDeliveryOptions() {
        this.add('ses-pinpoint:PutConfigurationSetDeliveryOptions');
        return this;
    }
    /**
     * Enable or disable collection of reputation metrics for emails that you send using a particular configuration set in a specific AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetReputationOptions.html
     */
    putConfigurationSetReputationOptions() {
        this.add('ses-pinpoint:PutConfigurationSetReputationOptions');
        return this;
    }
    /**
     * Enable or disable email sending for messages that use a particular configuration set in a specific AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetSendingOptions.html
     */
    putConfigurationSetSendingOptions() {
        this.add('ses-pinpoint:PutConfigurationSetSendingOptions');
        return this;
    }
    /**
     * Specify a custom domain to use for open and click tracking elements in email that you send using Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetTrackingOptions.html
     */
    putConfigurationSetTrackingOptions() {
        this.add('ses-pinpoint:PutConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpInPool.html
     */
    putDedicatedIpInPool() {
        this.add('ses-pinpoint:PutDedicatedIpInPool');
        return this;
    }
    /**
     * Put Dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpWarmupAttributes.html
     */
    putDedicatedIpWarmupAttributes() {
        this.add('ses-pinpoint:PutDedicatedIpWarmupAttributes');
        return this;
    }
    /**
     * Enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDeliverabilityDashboardOption.html
     */
    putDeliverabilityDashboardOption() {
        this.add('ses-pinpoint:PutDeliverabilityDashboardOption');
        return this;
    }
    /**
     * Used to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityDkimAttributes.html
     */
    putEmailIdentityDkimAttributes() {
        this.add('ses-pinpoint:PutEmailIdentityDkimAttributes');
        return this;
    }
    /**
     * Used to enable or disable feedback forwarding for an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityFeedbackAttributes.html
     */
    putEmailIdentityFeedbackAttributes() {
        this.add('ses-pinpoint:PutEmailIdentityFeedbackAttributes');
        return this;
    }
    /**
     * Used to enable or disable the custom Mail-From domain configuration for an email identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityMailFromAttributes.html
     */
    putEmailIdentityMailFromAttributes() {
        this.add('ses-pinpoint:PutEmailIdentityMailFromAttributes');
        return this;
    }
    /**
     * Sends an email message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_SendEmail.html
     */
    sendEmail() {
        this.add('ses-pinpoint:SendEmail');
        return this;
    }
    /**
     * Add one or more tags (keys and values) to a specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('ses-pinpoint:TagResource');
        return this;
    }
    /**
     * Remove one or more tags (keys and values) from a specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('ses-pinpoint:UntagResource');
        return this;
    }
    /**
     * Update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    updateConfigurationSetEventDestination() {
        this.add('ses-pinpoint:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DedicatedIp.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDedicatedIpPool(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:dedicated-ip-pool/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeliverabilityTestReport.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDeliverabilityTestReport(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:deliverability-test-report/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.SesPinpoint = SesPinpoint;
//# sourceMappingURL=data:application/json;base64,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