"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Polly = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Polly extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'polly';
        this.actionList = {
            "DeleteLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html",
                "description": "Deletes the specified pronunciation lexicon stored in an AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "DescribeVoices": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html",
                "description": "Returns the list of voices that are available for use when requesting speech synthesis.",
                "accessLevel": "List"
            },
            "GetLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html",
                "description": "Returns the content of the specified pronunciation lexicon stored in an AWS Region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "GetSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html",
                "description": "Enables the user to get information about specific speech synthesis task.",
                "accessLevel": "Read"
            },
            "ListLexicons": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html",
                "description": "Returns a list of pronunciation lexicons stored in an AWS Region.",
                "accessLevel": "List"
            },
            "ListSpeechSynthesisTasks": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html",
                "description": "Enables the user to list requested speech synthesis tasks.",
                "accessLevel": "List"
            },
            "PutLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html",
                "description": "Stores a pronunciation lexicon in an AWS Region.",
                "accessLevel": "Write"
            },
            "StartSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html",
                "description": "Enables the user to synthesize long inputs to provided S3 location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            },
            "SynthesizeSpeech": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html",
                "description": "Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "lexicon": {
                "name": "lexicon",
                "url": "https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html",
                "arn": "arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html
     */
    deleteLexicon() {
        this.add('polly:DeleteLexicon');
        return this;
    }
    /**
     * Returns the list of voices that are available for use when requesting speech synthesis.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html
     */
    describeVoices() {
        this.add('polly:DescribeVoices');
        return this;
    }
    /**
     * Returns the content of the specified pronunciation lexicon stored in an AWS Region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html
     */
    getLexicon() {
        this.add('polly:GetLexicon');
        return this;
    }
    /**
     * Enables the user to get information about specific speech synthesis task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html
     */
    getSpeechSynthesisTask() {
        this.add('polly:GetSpeechSynthesisTask');
        return this;
    }
    /**
     * Returns a list of pronunciation lexicons stored in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html
     */
    listLexicons() {
        this.add('polly:ListLexicons');
        return this;
    }
    /**
     * Enables the user to list requested speech synthesis tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html
     */
    listSpeechSynthesisTasks() {
        this.add('polly:ListSpeechSynthesisTasks');
        return this;
    }
    /**
     * Stores a pronunciation lexicon in an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html
     */
    putLexicon() {
        this.add('polly:PutLexicon');
        return this;
    }
    /**
     * Enables the user to synthesize long inputs to provided S3 location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html
     */
    startSpeechSynthesisTask() {
        this.add('polly:StartSpeechSynthesisTask');
        return this;
    }
    /**
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html
     */
    synthesizeSpeech() {
        this.add('polly:SynthesizeSpeech');
        return this;
    }
    /**
     * Adds a resource of type lexicon to the statement
     *
     * https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html
     *
     * @param lexiconName - Identifier for the lexiconName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLexicon(lexiconName, account, region, partition) {
        var arn = 'arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}';
        arn = arn.replace('${LexiconName}', lexiconName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Polly = Polly;
//# sourceMappingURL=data:application/json;base64,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