"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspaces = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workspaces](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workspaces extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workspaces](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkspaces.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workspaces';
        this.actionList = {
            "AssociateIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html",
                "description": "Associates the specified IP access control group with the specified directory.",
                "accessLevel": "Write"
            },
            "AuthorizeIpRules": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html",
                "description": "Adds one or more rules to the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "CreateIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html",
                "description": "Creates an IP access control group.",
                "accessLevel": "Write"
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html",
                "description": "Creates tags for a WorkSpace.",
                "accessLevel": "Tagging"
            },
            "CreateWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html",
                "description": "Creates one or more WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    },
                    "workspacebundle": {
                        "required": true
                    }
                }
            },
            "DeleteIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html",
                "description": "Deletes the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html",
                "description": "Deletes tags from a Workspace.",
                "accessLevel": "Write"
            },
            "DeleteWorkspaceImage": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html",
                "description": "Deletes the specified workspace image.",
                "accessLevel": "Write"
            },
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html",
                "description": "Retrieves a list that describes the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "List"
            },
            "DescribeAccountModifications": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html",
                "description": "Retrieves a list that describes modifications to the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "List"
            },
            "DescribeClientProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html",
                "description": "Describe client properties about the specified resources.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    }
                }
            },
            "DescribeIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html",
                "description": "Retrieves information about the IP access control groups of your account in the region.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html",
                "description": "Describes tags for a WorkSpace.",
                "accessLevel": "List"
            },
            "DescribeWorkspaceBundles": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html",
                "description": "Obtains information about the WorkSpace bundles that are available to your account in the specified region.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workspacebundle": {
                        "required": true
                    }
                }
            },
            "DescribeWorkspaceDirectories": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html",
                "description": "Retrieves information about the AWS Directory Service directories in the region that are registered with Amazon WorkSpaces and are available to your account.",
                "accessLevel": "List"
            },
            "DescribeWorkspaceImages": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html",
                "description": "Retrieves a list that describes one or more specified images.",
                "accessLevel": "List"
            },
            "DescribeWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html",
                "description": "Obtains information about the specified WorkSpaces.",
                "accessLevel": "List"
            },
            "DescribeWorkspacesConnectionStatus": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html",
                "description": "Describes the connection status of a specified WorkSpace.",
                "accessLevel": "Read"
            },
            "DisassociateIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html",
                "description": "Disassociates the specified IP access control group from the specified directory.",
                "accessLevel": "Write"
            },
            "ImportWorkspaceImage": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html",
                "description": "Import a licensed EC2 image to into Amazon WorkSpaces.",
                "accessLevel": "Write"
            },
            "ListAvailableManagementCidrRanges": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html",
                "description": "List available CIDR ranges for a CIDR range constraint.",
                "accessLevel": "List"
            },
            "ModifyAccount": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html",
                "description": "Modify the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "Write"
            },
            "ModifyClientProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html",
                "description": "Modify the client properties of a specified resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    }
                }
            },
            "ModifyWorkspaceProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html",
                "description": "Modifies the WorkSpace properties, including the running mode and AutoStop time.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "ModifyWorkspaceState": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html",
                "description": "Modify the state of specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RebootWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html",
                "description": "Reboots the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RebuildWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html",
                "description": "Rebuilds the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RevokeIpRules": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html",
                "description": "Removes one or more rules from the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "StartWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html",
                "description": "Starts the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "StopWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html",
                "description": "Stops the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "TerminateWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html",
                "description": "Terminates the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "UpdateRulesOfIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html",
                "description": "Replaces the current rules of the specified IP access control group with the specified rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "workspacebundle": {
                "name": "workspacebundle",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspacebundle/${BundleId}",
                "conditionKeys": []
            },
            "workspaceipgroup": {
                "name": "workspaceipgroup",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspaceipgroup/${GroupId}",
                "conditionKeys": []
            },
            "directoryid": {
                "name": "directoryid",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": []
            },
            "workspaceid": {
                "name": "workspaceid",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspace/${WorkspaceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the specified IP access control group with the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html
     */
    associateIpGroups() {
        this.add('workspaces:AssociateIpGroups');
        return this;
    }
    /**
     * Adds one or more rules to the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html
     */
    authorizeIpRules() {
        this.add('workspaces:AuthorizeIpRules');
        return this;
    }
    /**
     * Creates an IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html
     */
    createIpGroup() {
        this.add('workspaces:CreateIpGroup');
        return this;
    }
    /**
     * Creates tags for a WorkSpace.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html
     */
    createTags() {
        this.add('workspaces:CreateTags');
        return this;
    }
    /**
     * Creates one or more WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html
     */
    createWorkspaces() {
        this.add('workspaces:CreateWorkspaces');
        return this;
    }
    /**
     * Deletes the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html
     */
    deleteIpGroup() {
        this.add('workspaces:DeleteIpGroup');
        return this;
    }
    /**
     * Deletes tags from a Workspace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html
     */
    deleteTags() {
        this.add('workspaces:DeleteTags');
        return this;
    }
    /**
     * Deletes the specified workspace image.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html
     */
    deleteWorkspaceImage() {
        this.add('workspaces:DeleteWorkspaceImage');
        return this;
    }
    /**
     * Retrieves a list that describes the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html
     */
    describeAccount() {
        this.add('workspaces:DescribeAccount');
        return this;
    }
    /**
     * Retrieves a list that describes modifications to the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html
     */
    describeAccountModifications() {
        this.add('workspaces:DescribeAccountModifications');
        return this;
    }
    /**
     * Describe client properties about the specified resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html
     */
    describeClientProperties() {
        this.add('workspaces:DescribeClientProperties');
        return this;
    }
    /**
     * Retrieves information about the IP access control groups of your account in the region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html
     */
    describeIpGroups() {
        this.add('workspaces:DescribeIpGroups');
        return this;
    }
    /**
     * Describes tags for a WorkSpace.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html
     */
    describeTags() {
        this.add('workspaces:DescribeTags');
        return this;
    }
    /**
     * Obtains information about the WorkSpace bundles that are available to your account in the specified region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html
     */
    describeWorkspaceBundles() {
        this.add('workspaces:DescribeWorkspaceBundles');
        return this;
    }
    /**
     * Retrieves information about the AWS Directory Service directories in the region that are registered with Amazon WorkSpaces and are available to your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html
     */
    describeWorkspaceDirectories() {
        this.add('workspaces:DescribeWorkspaceDirectories');
        return this;
    }
    /**
     * Retrieves a list that describes one or more specified images.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html
     */
    describeWorkspaceImages() {
        this.add('workspaces:DescribeWorkspaceImages');
        return this;
    }
    /**
     * Obtains information about the specified WorkSpaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html
     */
    describeWorkspaces() {
        this.add('workspaces:DescribeWorkspaces');
        return this;
    }
    /**
     * Describes the connection status of a specified WorkSpace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html
     */
    describeWorkspacesConnectionStatus() {
        this.add('workspaces:DescribeWorkspacesConnectionStatus');
        return this;
    }
    /**
     * Disassociates the specified IP access control group from the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html
     */
    disassociateIpGroups() {
        this.add('workspaces:DisassociateIpGroups');
        return this;
    }
    /**
     * Import a licensed EC2 image to into Amazon WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html
     */
    importWorkspaceImage() {
        this.add('workspaces:ImportWorkspaceImage');
        return this;
    }
    /**
     * List available CIDR ranges for a CIDR range constraint.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html
     */
    listAvailableManagementCidrRanges() {
        this.add('workspaces:ListAvailableManagementCidrRanges');
        return this;
    }
    /**
     * Modify the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html
     */
    modifyAccount() {
        this.add('workspaces:ModifyAccount');
        return this;
    }
    /**
     * Modify the client properties of a specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html
     */
    modifyClientProperties() {
        this.add('workspaces:ModifyClientProperties');
        return this;
    }
    /**
     * Modifies the WorkSpace properties, including the running mode and AutoStop time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html
     */
    modifyWorkspaceProperties() {
        this.add('workspaces:ModifyWorkspaceProperties');
        return this;
    }
    /**
     * Modify the state of specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html
     */
    modifyWorkspaceState() {
        this.add('workspaces:ModifyWorkspaceState');
        return this;
    }
    /**
     * Reboots the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html
     */
    rebootWorkspaces() {
        this.add('workspaces:RebootWorkspaces');
        return this;
    }
    /**
     * Rebuilds the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html
     */
    rebuildWorkspaces() {
        this.add('workspaces:RebuildWorkspaces');
        return this;
    }
    /**
     * Removes one or more rules from the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html
     */
    revokeIpRules() {
        this.add('workspaces:RevokeIpRules');
        return this;
    }
    /**
     * Starts the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html
     */
    startWorkspaces() {
        this.add('workspaces:StartWorkspaces');
        return this;
    }
    /**
     * Stops the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html
     */
    stopWorkspaces() {
        this.add('workspaces:StopWorkspaces');
        return this;
    }
    /**
     * Terminates the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html
     */
    terminateWorkspaces() {
        this.add('workspaces:TerminateWorkspaces');
        return this;
    }
    /**
     * Replaces the current rules of the specified IP access control group with the specified rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html
     */
    updateRulesOfIpGroup() {
        this.add('workspaces:UpdateRulesOfIpGroup');
        return this;
    }
    /**
     * Adds a resource of type workspacebundle to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param bundleId - Identifier for the bundleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspacebundle(bundleId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspacebundle/${BundleId}';
        arn = arn.replace('${BundleId}', bundleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceipgroup to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceipgroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspaceipgroup/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directoryid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectoryid(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceid(workspaceId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspace/${WorkspaceId}';
        arn = arn.replace('${WorkspaceId}', workspaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Workspaces = Workspaces;
//# sourceMappingURL=data:application/json;base64,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