"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Amplify = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [amplify](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsamplify.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Amplify extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [amplify](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsamplify.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'amplify';
        this.actionList = {
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new backend environment for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new Branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a deployment for manual deploy apps. (Apps are not connected to repository)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "CreateDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a new DomainAssociation on an App",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a new webhook on an App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete an existing Amplify App by appId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "DeleteDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a DomainAssociation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete a job, for an Amplify branch, part of Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "DeleteWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete a webhook by id.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GenerateAccessLogs": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Generate website access logs for a specific time range via a pre-signed URL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves an existing Amplify App by appId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetArtifactUrl": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves artifact info that corresponds to a artifactId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves a backend environment for an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves a branch for an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "GetDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves domain info that corresponds to an appId and domainName.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Get a job for a branch, part of an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "GetWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves webhook info that corresponds to a webhookId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListApps": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists existing Amplify Apps.",
                "accessLevel": "List"
            },
            "ListArtifacts": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List artifacts with an app, a branch, a job and an artifact type.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListBackendEnvironments": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists backend environments for an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListBranches": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists branches for an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListDomainAssociations": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List domains with an app",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List Jobs for a branch, part of an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "ListWebHooks": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List webhooks on an App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "StartDeployment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Start a deployment for manual deploy apps. (Apps are not connected to repository)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "StartJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Starts a new job for a branch, part of an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "StopJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Stop a job that is in progress, for an Amplify branch, part of Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "This action tags an AWS Amplify Console resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "branches": {
                        "required": false
                    },
                    "jobs": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "This action removes a tag from an AWS Amplify Console resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "branches": {
                        "required": false
                    },
                    "jobs": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Updates an existing Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Updates a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "UpdateDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Update a DomainAssociation on an App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "UpdateWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Update a webhook.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "apps": {
                "name": "apps",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "branches": {
                "name": "branches",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "jobs": {
                "name": "jobs",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}/jobs/${JobId}",
                "conditionKeys": []
            },
            "domains": {
                "name": "domains",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/domains/${DomainName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a new Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createApp() {
        this.add('amplify:CreateApp');
        return this;
    }
    /**
     * Creates a new backend environment for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createBackendEnvironment() {
        this.add('amplify:CreateBackendEnvironment');
        return this;
    }
    /**
     * Creates a new Branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createBranch() {
        this.add('amplify:CreateBranch');
        return this;
    }
    /**
     * Create a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createDeployment() {
        this.add('amplify:CreateDeployment');
        return this;
    }
    /**
     * Create a new DomainAssociation on an App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createDomainAssociation() {
        this.add('amplify:CreateDomainAssociation');
        return this;
    }
    /**
     * Create a new webhook on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createWebHook() {
        this.add('amplify:CreateWebHook');
        return this;
    }
    /**
     * Delete an existing Amplify App by appId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteApp() {
        this.add('amplify:DeleteApp');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteBackendEnvironment() {
        this.add('amplify:DeleteBackendEnvironment');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteBranch() {
        this.add('amplify:DeleteBranch');
        return this;
    }
    /**
     * Deletes a DomainAssociation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteDomainAssociation() {
        this.add('amplify:DeleteDomainAssociation');
        return this;
    }
    /**
     * Delete a job, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteJob() {
        this.add('amplify:DeleteJob');
        return this;
    }
    /**
     * Delete a webhook by id.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteWebHook() {
        this.add('amplify:DeleteWebHook');
        return this;
    }
    /**
     * Generate website access logs for a specific time range via a pre-signed URL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    generateAccessLogs() {
        this.add('amplify:GenerateAccessLogs');
        return this;
    }
    /**
     * Retrieves an existing Amplify App by appId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getApp() {
        this.add('amplify:GetApp');
        return this;
    }
    /**
     * Retrieves artifact info that corresponds to a artifactId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getArtifactUrl() {
        this.add('amplify:GetArtifactUrl');
        return this;
    }
    /**
     * Retrieves a backend environment for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getBackendEnvironment() {
        this.add('amplify:GetBackendEnvironment');
        return this;
    }
    /**
     * Retrieves a branch for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getBranch() {
        this.add('amplify:GetBranch');
        return this;
    }
    /**
     * Retrieves domain info that corresponds to an appId and domainName.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getDomainAssociation() {
        this.add('amplify:GetDomainAssociation');
        return this;
    }
    /**
     * Get a job for a branch, part of an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getJob() {
        this.add('amplify:GetJob');
        return this;
    }
    /**
     * Retrieves webhook info that corresponds to a webhookId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getWebHook() {
        this.add('amplify:GetWebHook');
        return this;
    }
    /**
     * Lists existing Amplify Apps.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listApps() {
        this.add('amplify:ListApps');
        return this;
    }
    /**
     * List artifacts with an app, a branch, a job and an artifact type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listArtifacts() {
        this.add('amplify:ListArtifacts');
        return this;
    }
    /**
     * Lists backend environments for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listBackendEnvironments() {
        this.add('amplify:ListBackendEnvironments');
        return this;
    }
    /**
     * Lists branches for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listBranches() {
        this.add('amplify:ListBranches');
        return this;
    }
    /**
     * List domains with an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listDomainAssociations() {
        this.add('amplify:ListDomainAssociations');
        return this;
    }
    /**
     * List Jobs for a branch, part of an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listJobs() {
        this.add('amplify:ListJobs');
        return this;
    }
    /**
     * List webhooks on an App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listWebHooks() {
        this.add('amplify:ListWebHooks');
        return this;
    }
    /**
     * Start a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    startDeployment() {
        this.add('amplify:StartDeployment');
        return this;
    }
    /**
     * Starts a new job for a branch, part of an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    startJob() {
        this.add('amplify:StartJob');
        return this;
    }
    /**
     * Stop a job that is in progress, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    stopJob() {
        this.add('amplify:StopJob');
        return this;
    }
    /**
     * This action tags an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    tagResource() {
        this.add('amplify:TagResource');
        return this;
    }
    /**
     * This action removes a tag from an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    untagResource() {
        this.add('amplify:UntagResource');
        return this;
    }
    /**
     * Updates an existing Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateApp() {
        this.add('amplify:UpdateApp');
        return this;
    }
    /**
     * Updates a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateBranch() {
        this.add('amplify:UpdateBranch');
        return this;
    }
    /**
     * Update a DomainAssociation on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateDomainAssociation() {
        this.add('amplify:UpdateDomainAssociation');
        return this;
    }
    /**
     * Update a webhook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateWebHook() {
        this.add('amplify:UpdateWebHook');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type branches to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBranches(appId, branchName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type jobs to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobs(appId, branchName, jobId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}/jobs/${JobId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type domains to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomains(appId, domainName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/domains/${DomainName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Amplify = Amplify;
//# sourceMappingURL=data:application/json;base64,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