"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcmPca = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AcmPca extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm-pca';
        this.actionList = {
            "CreateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html",
                "description": "Creates an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html",
                "description": "Creates an audit report for an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "CreatePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html",
                "description": "Creates a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeleteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html",
                "description": "Deletes an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeletePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html",
                "description": "Deletes a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html",
                "description": "Returns a list of the configuration and status fields contained in the specified ACM Private CA.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html",
                "description": "Returns the status and information about an ACM Private CA audit report.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCsr": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html",
                "description": "Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ImportCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html",
                "description": "Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "IssueCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html",
                "description": "Issues an ACM Private CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "acm-pca:TemplateArn"
                ]
            },
            "ListCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html",
                "description": "Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.",
                "accessLevel": "List"
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the permissions that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html",
                "description": "Lists the tags that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RestoreCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html",
                "description": "Restores an ACM Private CA from the deleted state to the state it was in when deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RevokeCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html",
                "description": "Revokes a certificate issued by an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "TagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html",
                "description": "Adds one or more tags to an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html",
                "description": "Remove one or more tags from an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html",
                "description": "Updates the configuration of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "certificate-authority": {
                "name": "certificate-authority",
                "url": "https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations",
                "arn": "arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html
     */
    createCertificateAuthority() {
        this.add('acm-pca:CreateCertificateAuthority');
        return this;
    }
    /**
     * Creates an audit report for an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html
     */
    createCertificateAuthorityAuditReport() {
        this.add('acm-pca:CreateCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Creates a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html
     */
    createPermission() {
        this.add('acm-pca:CreatePermission');
        return this;
    }
    /**
     * Deletes an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html
     */
    deleteCertificateAuthority() {
        this.add('acm-pca:DeleteCertificateAuthority');
        return this;
    }
    /**
     * Deletes a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html
     */
    deletePermission() {
        this.add('acm-pca:DeletePermission');
        return this;
    }
    /**
     * Returns a list of the configuration and status fields contained in the specified ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html
     */
    describeCertificateAuthority() {
        this.add('acm-pca:DescribeCertificateAuthority');
        return this;
    }
    /**
     * Returns the status and information about an ACM Private CA audit report.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html
     */
    describeCertificateAuthorityAuditReport() {
        this.add('acm-pca:DescribeCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html
     */
    getCertificate() {
        this.add('acm-pca:GetCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html
     */
    getCertificateAuthorityCertificate() {
        this.add('acm-pca:GetCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html
     */
    getCertificateAuthorityCsr() {
        this.add('acm-pca:GetCertificateAuthorityCsr');
        return this;
    }
    /**
     * Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html
     */
    importCertificateAuthorityCertificate() {
        this.add('acm-pca:ImportCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Issues an ACM Private CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html
     */
    issueCertificate() {
        this.add('acm-pca:IssueCertificate');
        return this;
    }
    /**
     * Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html
     */
    listCertificateAuthorities() {
        this.add('acm-pca:ListCertificateAuthorities');
        return this;
    }
    /**
     * Lists the permissions that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html
     */
    listPermissions() {
        this.add('acm-pca:ListPermissions');
        return this;
    }
    /**
     * Lists the tags that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html
     */
    listTags() {
        this.add('acm-pca:ListTags');
        return this;
    }
    /**
     * Restores an ACM Private CA from the deleted state to the state it was in when deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html
     */
    restoreCertificateAuthority() {
        this.add('acm-pca:RestoreCertificateAuthority');
        return this;
    }
    /**
     * Revokes a certificate issued by an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html
     */
    revokeCertificate() {
        this.add('acm-pca:RevokeCertificate');
        return this;
    }
    /**
     * Adds one or more tags to an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html
     */
    tagCertificateAuthority() {
        this.add('acm-pca:TagCertificateAuthority');
        return this;
    }
    /**
     * Remove one or more tags from an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html
     */
    untagCertificateAuthority() {
        this.add('acm-pca:UntagCertificateAuthority');
        return this;
    }
    /**
     * Updates the configuration of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html
     */
    updateCertificateAuthority() {
        this.add('acm-pca:UpdateCertificateAuthority');
        return this;
    }
    /**
     * Adds a resource of type certificate-authority to the statement
     *
     * https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations
     *
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCertificateAuthority(certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}';
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters issue certificate requests based on the presence of TemplateArn in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateArn(value, operator) {
        return this.if(`acm-pca:TemplateArn`, value, operator || 'StringLike');
    }
}
exports.AcmPca = AcmPca;
//# sourceMappingURL=data:application/json;base64,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