"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicediscovery = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicediscovery extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicediscovery';
        this.actionList = {
            "CreateHttpNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html",
                "description": "Creates an HTTP namespace.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePrivateDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html",
                "description": "Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePublicDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html",
                "description": "Creates a public namespace based on DNS, which will be visible on the internet.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html",
                "description": "Creates a service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:NamespaceArn",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html",
                "description": "Deletes a specified namespace.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "DeleteService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html",
                "description": "Deletes a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "DeregisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html",
                "description": "Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "DiscoverInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html",
                "description": "Discovers registered instances for a specified namespace and service.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:NamespaceName",
                    "servicediscovery:ServiceName"
                ]
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html",
                "description": "Gets information about a specified instance.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetInstancesHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html",
                "description": "Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html",
                "description": "Gets information about a namespace.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html",
                "description": "Gets information about a specific operation.",
                "accessLevel": "Read"
            },
            "GetService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html",
                "description": "Gets the settings for a specified service.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "ListInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html",
                "description": "Gets summary information about the instances that were registered with a specified service.",
                "accessLevel": "List",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "ListNamespaces": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html",
                "description": "Gets information about the namespaces.",
                "accessLevel": "List"
            },
            "ListOperations": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html",
                "description": "Lists operations that match the criteria that you specify.",
                "accessLevel": "List"
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html",
                "description": "Gets settings for all the services that match specified filters.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html",
                "description": "Lists tags for the specified resource.",
                "accessLevel": "List"
            },
            "RegisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html",
                "description": "Registers an instance based on the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html",
                "description": "Adds one or more tags to the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html",
                "description": "Removes one or more tags from the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateInstanceCustomHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html",
                "description": "Updates the current health status for an instance that has a custom health check.",
                "accessLevel": "Write",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "UpdateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html",
                "description": "Updates the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "namespace": {
                "name": "namespace",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "service": {
                "name": "service",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an HTTP namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html
     */
    createHttpNamespace() {
        this.add('servicediscovery:CreateHttpNamespace');
        return this;
    }
    /**
     * Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html
     */
    createPrivateDnsNamespace() {
        this.add('servicediscovery:CreatePrivateDnsNamespace');
        return this;
    }
    /**
     * Creates a public namespace based on DNS, which will be visible on the internet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html
     */
    createPublicDnsNamespace() {
        this.add('servicediscovery:CreatePublicDnsNamespace');
        return this;
    }
    /**
     * Creates a service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html
     */
    createService() {
        this.add('servicediscovery:CreateService');
        return this;
    }
    /**
     * Deletes a specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html
     */
    deleteNamespace() {
        this.add('servicediscovery:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html
     */
    deleteService() {
        this.add('servicediscovery:DeleteService');
        return this;
    }
    /**
     * Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html
     */
    deregisterInstance() {
        this.add('servicediscovery:DeregisterInstance');
        return this;
    }
    /**
     * Discovers registered instances for a specified namespace and service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html
     */
    discoverInstances() {
        this.add('servicediscovery:DiscoverInstances');
        return this;
    }
    /**
     * Gets information about a specified instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html
     */
    getInstance() {
        this.add('servicediscovery:GetInstance');
        return this;
    }
    /**
     * Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html
     */
    getInstancesHealthStatus() {
        this.add('servicediscovery:GetInstancesHealthStatus');
        return this;
    }
    /**
     * Gets information about a namespace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html
     */
    getNamespace() {
        this.add('servicediscovery:GetNamespace');
        return this;
    }
    /**
     * Gets information about a specific operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html
     */
    getOperation() {
        this.add('servicediscovery:GetOperation');
        return this;
    }
    /**
     * Gets the settings for a specified service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html
     */
    getService() {
        this.add('servicediscovery:GetService');
        return this;
    }
    /**
     * Gets summary information about the instances that were registered with a specified service.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html
     */
    listInstances() {
        this.add('servicediscovery:ListInstances');
        return this;
    }
    /**
     * Gets information about the namespaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html
     */
    listNamespaces() {
        this.add('servicediscovery:ListNamespaces');
        return this;
    }
    /**
     * Lists operations that match the criteria that you specify.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html
     */
    listOperations() {
        this.add('servicediscovery:ListOperations');
        return this;
    }
    /**
     * Gets settings for all the services that match specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html
     */
    listServices() {
        this.add('servicediscovery:ListServices');
        return this;
    }
    /**
     * Lists tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('servicediscovery:ListTagsForResource');
        return this;
    }
    /**
     * Registers an instance based on the settings in a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html
     */
    registerInstance() {
        this.add('servicediscovery:RegisterInstance');
        return this;
    }
    /**
     * Adds one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('servicediscovery:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('servicediscovery:UntagResource');
        return this;
    }
    /**
     * Updates the current health status for an instance that has a custom health check.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html
     */
    updateInstanceCustomHealthStatus() {
        this.add('servicediscovery:UpdateInstanceCustomHealthStatus');
        return this;
    }
    /**
     * Updates the settings in a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html
     */
    updateService() {
        this.add('servicediscovery:UpdateService');
        return this;
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html
     *
     * @param namespaceId - Identifier for the namespaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onNamespace(namespaceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}';
        arn = arn.replace('${NamespaceId}', namespaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onService(serviceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}';
        arn = arn.replace('${ServiceId}', serviceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceArn(value, operator) {
        return this.if(`servicediscovery:NamespaceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceName(value, operator) {
        return this.if(`servicediscovery:NamespaceName`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`servicediscovery:ServiceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceName(value, operator) {
        return this.if(`servicediscovery:ServiceName`, value, operator || 'StringLike');
    }
}
exports.Servicediscovery = Servicediscovery;
//# sourceMappingURL=data:application/json;base64,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