"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codecommit = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codecommit](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodecommit.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codecommit extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codecommit](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodecommit.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codecommit';
        this.actionList = {
            "AssociateApprovalRuleTemplateWithRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html",
                "description": "Grants permission to associate an approval rule template with a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchAssociateApprovalRuleTemplateWithRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html",
                "description": "Grants permission to associate an approval rule template with multiple repositories in a single operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchDescribeMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html",
                "description": "Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchDisassociateApprovalRuleTemplateFromRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html",
                "description": "Grants permission to remove the association between an approval rule template and multiple repositories in a single operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetCommits": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html",
                "description": "Grants permission to get return information about one or more commits in an AWS CodeCommit repository.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetPullRequests": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr",
                "description": "Grants permission to return information about one or more pull requests in an AWS CodeCommit repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html",
                "description": "Grants permission to get information about multiple repositories",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CancelUploadArchive": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html",
                "description": "Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests",
                "accessLevel": "Write"
            },
            "CreateBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html",
                "description": "Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "CreateCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html",
                "description": "Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "CreatePullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html",
                "description": "Grants permission to create a pull request in the specified repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreatePullRequestApprovalRule": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html",
                "description": "Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html",
                "description": "Grants permission to create an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUnreferencedMergeCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html",
                "description": "Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeleteApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html",
                "description": "Grants permission to delete an approval rule template",
                "accessLevel": "Write"
            },
            "DeleteBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html",
                "description": "Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeleteCommentContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html",
                "description": "Grants permission to delete the content of a comment made on a change, file, or commit in a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html",
                "description": "Grants permission to delete a specified file from a specified branch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeletePullRequestApprovalRule": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html",
                "description": "Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html",
                "description": "Grants permission to delete an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribeMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html",
                "description": "Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribePullRequestEvents": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html",
                "description": "Grants permission to return information about one or more pull request events",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DisassociateApprovalRuleTemplateFromRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html",
                "description": "Grants permission to remove the association between an approval rule template and a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "EvaluatePullRequestApprovalRules": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html",
                "description": "Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html",
                "description": "Grants permission to return information about an approval rule template",
                "accessLevel": "Read"
            },
            "GetBlob": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html",
                "description": "Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html",
                "description": "Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetComment": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html",
                "description": "Grants permission to get the content of a comment made on a change, file, or commit in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentReactions": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html",
                "description": "Grants permission to get the reactions on a comment",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentsForComparedCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html",
                "description": "Grants permission to get information about comments made on the comparison between two commits",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentsForPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html",
                "description": "Grants permission to get comments made on a pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html",
                "description": "Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommitHistory": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to get information about the history of commits in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommitsFromMergeBase": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr",
                "description": "Grants permission to get information about the difference between commits in the context of a potential merge",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetDifferences": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html",
                "description": "Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html",
                "description": "Grants permission to return the base-64 encoded contents of a specified file and its metadata",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetFolder": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html",
                "description": "Grants permission to return the contents of a specified folder in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetMergeCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html",
                "description": "Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "GetMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html",
                "description": "Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetMergeOptions": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html",
                "description": "Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetObjectIdentifier": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to resolve blobs, trees, and commits to their identifier",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html",
                "description": "Grants permission to get information about a pull request in a specified repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequestApprovalStates": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html",
                "description": "Grants permission to retrieve the current approvals on an inputted pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequestOverrideState": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html",
                "description": "Grants permission to retrieve the current override state of a given pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetReferences": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html",
                "description": "Grants permission to get information about an AWS CodeCommit repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html",
                "description": "Grants permission to get information about triggers configured for a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetTree": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetUploadArchiveStatus": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GitPull": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git",
                "description": "Grants permission to pull information from an AWS CodeCommit repository to a local repo",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GitPush": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git",
                "description": "Grants permission to push information from a local repo to an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "ListApprovalRuleTemplates": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html",
                "description": "Grants permission to list all approval rule templates in an AWS Region for the AWS account",
                "accessLevel": "List"
            },
            "ListAssociatedApprovalRuleTemplatesForRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html",
                "description": "Grants permission to list approval rule templates that are associated with a repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListBranches": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html",
                "description": "Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListPullRequests": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html",
                "description": "Grants permission to list pull requests for a specified repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html",
                "description": "Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account",
                "accessLevel": "List"
            },
            "ListRepositoriesForApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html",
                "description": "Grants permission to list repositories that are associated with an approval rule template",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the resource attached to a CodeCommit resource ARN",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                }
            },
            "MergeBranchesByFastForward": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the fast-forward merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergeBranchesBySquash": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the squash merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergeBranchesByThreeWay": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the three-way merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestByFastForward": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestBySquash": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestByThreeWay": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "OverridePullRequestApprovalRules": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html",
                "description": "Grants permission to override all approval rules for a pull request, including approval rules created by a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentForComparedCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html",
                "description": "Grants permission to post a comment on the comparison between two commits",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentForPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html",
                "description": "Grants permission to post a comment on a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentReply": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html",
                "description": "Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PutCommentReaction": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html",
                "description": "Grants permission to post a reaction on a comment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PutFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html",
                "description": "Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "PutRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html",
                "description": "Grants permission to create, update, or delete triggers for a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to attach resource tags to a CodeCommit resource ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html",
                "description": "Grants permission to test the functionality of repository triggers by sending information to the trigger target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to disassociate resource tags from a CodeCommit resource ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApprovalRuleTemplateContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html",
                "description": "Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests",
                "accessLevel": "Write"
            },
            "UpdateApprovalRuleTemplateDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html",
                "description": "Grants permission to update the description of approval rule templates",
                "accessLevel": "Write"
            },
            "UpdateApprovalRuleTemplateName": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html",
                "description": "Grants permission to update the name of approval rule templates",
                "accessLevel": "Write"
            },
            "UpdateComment": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html",
                "description": "Grants permission to update the contents of a comment if the identity matches the identity used to create the comment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateDefaultBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html",
                "description": "Grants permission to change the default branch in an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestApprovalRuleContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html",
                "description": "Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestApprovalState": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html",
                "description": "Grants permission to update the approval state for pull requests",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html",
                "description": "Grants permission to update the description of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestStatus": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html",
                "description": "Grants permission to update the status of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestTitle": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html",
                "description": "Grants permission to update the title of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateRepositoryDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html",
                "description": "Grants permission to change the description of an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateRepositoryName": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html",
                "description": "Grants permission to change the name of an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UploadArchive": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "repository": {
                "name": "repository",
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate an approval rule template with a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html
     */
    associateApprovalRuleTemplateWithRepository() {
        this.add('codecommit:AssociateApprovalRuleTemplateWithRepository');
        return this;
    }
    /**
     * Grants permission to associate an approval rule template with multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html
     */
    batchAssociateApprovalRuleTemplateWithRepositories() {
        this.add('codecommit:BatchAssociateApprovalRuleTemplateWithRepositories');
        return this;
    }
    /**
     * Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html
     */
    batchDescribeMergeConflicts() {
        this.add('codecommit:BatchDescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html
     */
    batchDisassociateApprovalRuleTemplateFromRepositories() {
        this.add('codecommit:BatchDisassociateApprovalRuleTemplateFromRepositories');
        return this;
    }
    /**
     * Grants permission to get return information about one or more commits in an AWS CodeCommit repository.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html
     */
    batchGetCommits() {
        this.add('codecommit:BatchGetCommits');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull requests in an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    batchGetPullRequests() {
        this.add('codecommit:BatchGetPullRequests');
        return this;
    }
    /**
     * Grants permission to get information about multiple repositories
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html
     */
    batchGetRepositories() {
        this.add('codecommit:BatchGetRepositories');
        return this;
    }
    /**
     * Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    cancelUploadArchive() {
        this.add('codecommit:CancelUploadArchive');
        return this;
    }
    /**
     * Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html
     */
    createApprovalRuleTemplate() {
        this.add('codecommit:CreateApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html
     */
    createBranch() {
        this.add('codecommit:CreateBranch');
        return this;
    }
    /**
     * Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html
     */
    createCommit() {
        this.add('codecommit:CreateCommit');
        return this;
    }
    /**
     * Grants permission to create a pull request in the specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html
     */
    createPullRequest() {
        this.add('codecommit:CreatePullRequest');
        return this;
    }
    /**
     * Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html
     */
    createPullRequestApprovalRule() {
        this.add('codecommit:CreatePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to create an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html
     */
    createRepository() {
        this.add('codecommit:CreateRepository');
        return this;
    }
    /**
     * Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html
     */
    createUnreferencedMergeCommit() {
        this.add('codecommit:CreateUnreferencedMergeCommit');
        return this;
    }
    /**
     * Grants permission to delete an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html
     */
    deleteApprovalRuleTemplate() {
        this.add('codecommit:DeleteApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html
     */
    deleteBranch() {
        this.add('codecommit:DeleteBranch');
        return this;
    }
    /**
     * Grants permission to delete the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html
     */
    deleteCommentContent() {
        this.add('codecommit:DeleteCommentContent');
        return this;
    }
    /**
     * Grants permission to delete a specified file from a specified branch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html
     */
    deleteFile() {
        this.add('codecommit:DeleteFile');
        return this;
    }
    /**
     * Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html
     */
    deletePullRequestApprovalRule() {
        this.add('codecommit:DeletePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to delete an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html
     */
    deleteRepository() {
        this.add('codecommit:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html
     */
    describeMergeConflicts() {
        this.add('codecommit:DescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull request events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html
     */
    describePullRequestEvents() {
        this.add('codecommit:DescribePullRequestEvents');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html
     */
    disassociateApprovalRuleTemplateFromRepository() {
        this.add('codecommit:DisassociateApprovalRuleTemplateFromRepository');
        return this;
    }
    /**
     * Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html
     */
    evaluatePullRequestApprovalRules() {
        this.add('codecommit:EvaluatePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to return information about an approval rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html
     */
    getApprovalRuleTemplate() {
        this.add('codecommit:GetApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html
     */
    getBlob() {
        this.add('codecommit:GetBlob');
        return this;
    }
    /**
     * Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html
     */
    getBranch() {
        this.add('codecommit:GetBranch');
        return this;
    }
    /**
     * Grants permission to get the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html
     */
    getComment() {
        this.add('codecommit:GetComment');
        return this;
    }
    /**
     * Grants permission to get the reactions on a comment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html
     */
    getCommentReactions() {
        this.add('codecommit:GetCommentReactions');
        return this;
    }
    /**
     * Grants permission to get information about comments made on the comparison between two commits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html
     */
    getCommentsForComparedCommit() {
        this.add('codecommit:GetCommentsForComparedCommit');
        return this;
    }
    /**
     * Grants permission to get comments made on a pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html
     */
    getCommentsForPullRequest() {
        this.add('codecommit:GetCommentsForPullRequest');
        return this;
    }
    /**
     * Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html
     */
    getCommit() {
        this.add('codecommit:GetCommit');
        return this;
    }
    /**
     * Grants permission to get information about the history of commits in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    getCommitHistory() {
        this.add('codecommit:GetCommitHistory');
        return this;
    }
    /**
     * Grants permission to get information about the difference between commits in the context of a potential merge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    getCommitsFromMergeBase() {
        this.add('codecommit:GetCommitsFromMergeBase');
        return this;
    }
    /**
     * Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html
     */
    getDifferences() {
        this.add('codecommit:GetDifferences');
        return this;
    }
    /**
     * Grants permission to return the base-64 encoded contents of a specified file and its metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html
     */
    getFile() {
        this.add('codecommit:GetFile');
        return this;
    }
    /**
     * Grants permission to return the contents of a specified folder in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html
     */
    getFolder() {
        this.add('codecommit:GetFolder');
        return this;
    }
    /**
     * Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html
     */
    getMergeCommit() {
        this.add('codecommit:GetMergeCommit');
        return this;
    }
    /**
     * Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html
     */
    getMergeConflicts() {
        this.add('codecommit:GetMergeConflicts');
        return this;
    }
    /**
     * Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html
     */
    getMergeOptions() {
        this.add('codecommit:GetMergeOptions');
        return this;
    }
    /**
     * Grants permission to resolve blobs, trees, and commits to their identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    getObjectIdentifier() {
        this.add('codecommit:GetObjectIdentifier');
        return this;
    }
    /**
     * Grants permission to get information about a pull request in a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html
     */
    getPullRequest() {
        this.add('codecommit:GetPullRequest');
        return this;
    }
    /**
     * Grants permission to retrieve the current approvals on an inputted pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html
     */
    getPullRequestApprovalStates() {
        this.add('codecommit:GetPullRequestApprovalStates');
        return this;
    }
    /**
     * Grants permission to retrieve the current override state of a given pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html
     */
    getPullRequestOverrideState() {
        this.add('codecommit:GetPullRequestOverrideState');
        return this;
    }
    /**
     * Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    getReferences() {
        this.add('codecommit:GetReferences');
        return this;
    }
    /**
     * Grants permission to get information about an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html
     */
    getRepository() {
        this.add('codecommit:GetRepository');
        return this;
    }
    /**
     * Grants permission to get information about triggers configured for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html
     */
    getRepositoryTriggers() {
        this.add('codecommit:GetRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    getTree() {
        this.add('codecommit:GetTree');
        return this;
    }
    /**
     * Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    getUploadArchiveStatus() {
        this.add('codecommit:GetUploadArchiveStatus');
        return this;
    }
    /**
     * Grants permission to pull information from an AWS CodeCommit repository to a local repo
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    gitPull() {
        this.add('codecommit:GitPull');
        return this;
    }
    /**
     * Grants permission to push information from a local repo to an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    gitPush() {
        this.add('codecommit:GitPush');
        return this;
    }
    /**
     * Grants permission to list all approval rule templates in an AWS Region for the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html
     */
    listApprovalRuleTemplates() {
        this.add('codecommit:ListApprovalRuleTemplates');
        return this;
    }
    /**
     * Grants permission to list approval rule templates that are associated with a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html
     */
    listAssociatedApprovalRuleTemplatesForRepository() {
        this.add('codecommit:ListAssociatedApprovalRuleTemplatesForRepository');
        return this;
    }
    /**
     * Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html
     */
    listBranches() {
        this.add('codecommit:ListBranches');
        return this;
    }
    /**
     * Grants permission to list pull requests for a specified repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html
     */
    listPullRequests() {
        this.add('codecommit:ListPullRequests');
        return this;
    }
    /**
     * Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html
     */
    listRepositories() {
        this.add('codecommit:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list repositories that are associated with an approval rule template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html
     */
    listRepositoriesForApprovalRuleTemplate() {
        this.add('codecommit:ListRepositoriesForApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to list the resource attached to a CodeCommit resource ARN
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('codecommit:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the fast-forward merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html
     */
    mergeBranchesByFastForward() {
        this.add('codecommit:MergeBranchesByFastForward');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the squash merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html
     */
    mergeBranchesBySquash() {
        this.add('codecommit:MergeBranchesBySquash');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the three-way merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html
     */
    mergeBranchesByThreeWay() {
        this.add('codecommit:MergeBranchesByThreeWay');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html
     */
    mergePullRequestByFastForward() {
        this.add('codecommit:MergePullRequestByFastForward');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html
     */
    mergePullRequestBySquash() {
        this.add('codecommit:MergePullRequestBySquash');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html
     */
    mergePullRequestByThreeWay() {
        this.add('codecommit:MergePullRequestByThreeWay');
        return this;
    }
    /**
     * Grants permission to override all approval rules for a pull request, including approval rules created by a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html
     */
    overridePullRequestApprovalRules() {
        this.add('codecommit:OverridePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to post a comment on the comparison between two commits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html
     */
    postCommentForComparedCommit() {
        this.add('codecommit:PostCommentForComparedCommit');
        return this;
    }
    /**
     * Grants permission to post a comment on a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html
     */
    postCommentForPullRequest() {
        this.add('codecommit:PostCommentForPullRequest');
        return this;
    }
    /**
     * Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html
     */
    postCommentReply() {
        this.add('codecommit:PostCommentReply');
        return this;
    }
    /**
     * Grants permission to post a reaction on a comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html
     */
    putCommentReaction() {
        this.add('codecommit:PutCommentReaction');
        return this;
    }
    /**
     * Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html
     */
    putFile() {
        this.add('codecommit:PutFile');
        return this;
    }
    /**
     * Grants permission to create, update, or delete triggers for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html
     */
    putRepositoryTriggers() {
        this.add('codecommit:PutRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to attach resource tags to a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('codecommit:TagResource');
        return this;
    }
    /**
     * Grants permission to test the functionality of repository triggers by sending information to the trigger target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html
     */
    testRepositoryTriggers() {
        this.add('codecommit:TestRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to disassociate resource tags from a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('codecommit:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html
     */
    updateApprovalRuleTemplateContent() {
        this.add('codecommit:UpdateApprovalRuleTemplateContent');
        return this;
    }
    /**
     * Grants permission to update the description of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html
     */
    updateApprovalRuleTemplateDescription() {
        this.add('codecommit:UpdateApprovalRuleTemplateDescription');
        return this;
    }
    /**
     * Grants permission to update the name of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html
     */
    updateApprovalRuleTemplateName() {
        this.add('codecommit:UpdateApprovalRuleTemplateName');
        return this;
    }
    /**
     * Grants permission to update the contents of a comment if the identity matches the identity used to create the comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html
     */
    updateComment() {
        this.add('codecommit:UpdateComment');
        return this;
    }
    /**
     * Grants permission to change the default branch in an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html
     */
    updateDefaultBranch() {
        this.add('codecommit:UpdateDefaultBranch');
        return this;
    }
    /**
     * Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html
     */
    updatePullRequestApprovalRuleContent() {
        this.add('codecommit:UpdatePullRequestApprovalRuleContent');
        return this;
    }
    /**
     * Grants permission to update the approval state for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html
     */
    updatePullRequestApprovalState() {
        this.add('codecommit:UpdatePullRequestApprovalState');
        return this;
    }
    /**
     * Grants permission to update the description of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html
     */
    updatePullRequestDescription() {
        this.add('codecommit:UpdatePullRequestDescription');
        return this;
    }
    /**
     * Grants permission to update the status of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html
     */
    updatePullRequestStatus() {
        this.add('codecommit:UpdatePullRequestStatus');
        return this;
    }
    /**
     * Grants permission to update the title of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html
     */
    updatePullRequestTitle() {
        this.add('codecommit:UpdatePullRequestTitle');
        return this;
    }
    /**
     * Grants permission to change the description of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html
     */
    updateRepositoryDescription() {
        this.add('codecommit:UpdateRepositoryDescription');
        return this;
    }
    /**
     * Grants permission to change the name of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html
     */
    updateRepositoryName() {
        this.add('codecommit:UpdateRepositoryName');
        return this;
    }
    /**
     * Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    uploadArchive() {
        this.add('codecommit:UploadArchive');
        return this;
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRepository(repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}';
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by Git reference to specified AWS CodeCommit actions
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/how-to-conditional-branch.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReferences(value, operator) {
        return this.if(`codecommit:References`, value, operator || 'StringLike');
    }
}
exports.Codecommit = Codecommit;
//# sourceMappingURL=data:application/json;base64,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