"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Signer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [signer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodesigningforamazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Signer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [signer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodesigningforamazonfreertos.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'signer';
        this.actionList = {
            "CancelSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_CancelSigningProfile.html",
                "description": "Cancels a signing profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "DescribeSigningJob": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_DescribeSigningJob.html",
                "description": "Describe a signing job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "signing-job": {
                        "required": true
                    }
                }
            },
            "GetSigningPlatform": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_GetSigningPlatform.html",
                "description": "Retrieves a signing platform.",
                "accessLevel": "Read"
            },
            "GetSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_GetSigningProfile.html",
                "description": "Retrieves a signing profile.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "ListSigningJobs": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningJobs.html",
                "description": "List signing jobs.",
                "accessLevel": "List"
            },
            "ListSigningPlatforms": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningPlatforms.html",
                "description": "List all signing platforms.",
                "accessLevel": "List"
            },
            "ListSigningProfiles": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningProfiles.html",
                "description": "List all signing profile associated with the account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListTagsForResource.html",
                "description": "Lists the tags associated with the Signing Profile resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "PutSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_PutSigningProfile.html",
                "description": "Creates a new signing profile if not exists.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartSigningJob": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_StartSigningJob.html",
                "description": "Starts a code signing request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_TagResource.html",
                "description": "Adds one or more tags to an Signing Profile resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_UntagResource.html",
                "description": "Removes one or more tags from an Signing Profile resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "signing-profile": {
                "name": "signing-profile",
                "url": "https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html",
                "arn": "arn:${Partition}:signer:${Region}::/signing-profiles/${ProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "signing-job": {
                "name": "signing-job",
                "url": "https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html",
                "arn": "arn:${Partition}:signer:${Region}::/signing-jobs/${JobId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels a signing profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_CancelSigningProfile.html
     */
    cancelSigningProfile() {
        this.add('signer:CancelSigningProfile');
        return this;
    }
    /**
     * Describe a signing job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_DescribeSigningJob.html
     */
    describeSigningJob() {
        this.add('signer:DescribeSigningJob');
        return this;
    }
    /**
     * Retrieves a signing platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningPlatform.html
     */
    getSigningPlatform() {
        this.add('signer:GetSigningPlatform');
        return this;
    }
    /**
     * Retrieves a signing profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningProfile.html
     */
    getSigningProfile() {
        this.add('signer:GetSigningProfile');
        return this;
    }
    /**
     * List signing jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningJobs.html
     */
    listSigningJobs() {
        this.add('signer:ListSigningJobs');
        return this;
    }
    /**
     * List all signing platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningPlatforms.html
     */
    listSigningPlatforms() {
        this.add('signer:ListSigningPlatforms');
        return this;
    }
    /**
     * List all signing profile associated with the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningProfiles.html
     */
    listSigningProfiles() {
        this.add('signer:ListSigningProfiles');
        return this;
    }
    /**
     * Lists the tags associated with the Signing Profile resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('signer:ListTagsForResource');
        return this;
    }
    /**
     * Creates a new signing profile if not exists.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_PutSigningProfile.html
     */
    putSigningProfile() {
        this.add('signer:PutSigningProfile');
        return this;
    }
    /**
     * Starts a code signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_StartSigningJob.html
     */
    startSigningJob() {
        this.add('signer:StartSigningJob');
        return this;
    }
    /**
     * Adds one or more tags to an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('signer:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('signer:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type signing-profile to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param profileName - Identifier for the profileName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSigningProfile(profileName, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-profiles/${ProfileName}';
        arn = arn.replace('${ProfileName}', profileName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type signing-job to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param jobId - Identifier for the jobId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSigningJob(jobId, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Signer = Signer;
//# sourceMappingURL=data:application/json;base64,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