"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codestar = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codestar](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestar.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codestar extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codestar](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestar.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codestar';
        this.actionList = {
            "AssociateTeamMember": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_AssociateTeamMember.html",
                "description": "Adds a user to the team for an AWS CodeStar project.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateProject.html",
                "description": "Creates a project with minimal structure, customer policies, and no resources.",
                "accessLevel": "Permissions management",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUserProfile": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateUserProfile.html",
                "description": "Creates a profile for a user that includes user preferences, display name, and email.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteExtendedAccess": {
                "url": "",
                "description": "Grants access to extended delete APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteProject.html",
                "description": "Deletes a project, including project resources. Does not delete users associated with the project, but does delete the IAM roles that allowed access to the project.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteUserProfile": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteUserProfile.html",
                "description": "Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such as display name and email address. It does not delete the history of that user, for example the history of commits made by that user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DescribeProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeProject.html",
                "description": "Describes a project and its resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeUserProfile": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeUserProfile.html",
                "description": "Describes a user in AWS CodeStar and the user attributes across all projects.",
                "accessLevel": "Read"
            },
            "DisassociateTeamMember": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_DisassociateTeamMember.html",
                "description": "Removes a user from a project. Removing a user from a project also removes the IAM policies from that user that allowed access to the project and its resources.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetExtendedAccess": {
                "url": "",
                "description": "Grants access to extended read APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListProjects.html",
                "description": "Lists all projects in CodeStar associated with your AWS account.",
                "accessLevel": "List"
            },
            "ListResources": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListResources.html",
                "description": "Lists all resources associated with a project in CodeStar.",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListTagsForProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTagsForProject.html",
                "description": "Lists the tags associated with a project in CodeStar.",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListTeamMembers": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTeamMembers.html",
                "description": "Lists all team members associated with a project.",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListUserProfiles": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListUserProfiles.html",
                "description": "Lists user profiles in AWS CodeStar.",
                "accessLevel": "List"
            },
            "PutExtendedAccess": {
                "url": "",
                "description": "Grants access to extended write APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "TagProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_TagProject.html",
                "description": "Adds tags to a project in CodeStar.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_UntagProject.html",
                "description": "Removes tags from a project in CodeStar.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateProject.html",
                "description": "Updates a project in CodeStar.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateTeamMember": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateTeamMember.html",
                "description": "Updates team member attributes within a CodeStar project.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateUserProfile": {
                "url": "https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateUserProfile.html",
                "description": "Updates a profile for a user that includes user preferences, display name, and email.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/codestar/latest/userguide/working-with-projects.html",
                "arn": "arn:${Partition}:codestar:${Region}:${Account}:project/${ProjectId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "user": {
                "name": "user",
                "url": "",
                "arn": "arn:${Partition}:iam:${Region}:${Account}:user/${UserNameWithPath}",
                "conditionKeys": [
                    "iam:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds a user to the team for an AWS CodeStar project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_AssociateTeamMember.html
     */
    associateTeamMember() {
        this.add('codestar:AssociateTeamMember');
        return this;
    }
    /**
     * Creates a project with minimal structure, customer policies, and no resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateProject.html
     */
    createProject() {
        this.add('codestar:CreateProject');
        return this;
    }
    /**
     * Creates a profile for a user that includes user preferences, display name, and email.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateUserProfile.html
     */
    createUserProfile() {
        this.add('codestar:CreateUserProfile');
        return this;
    }
    /**
     * Grants access to extended delete APIs.
     *
     * Access Level: Write
     */
    deleteExtendedAccess() {
        this.add('codestar:DeleteExtendedAccess');
        return this;
    }
    /**
     * Deletes a project, including project resources. Does not delete users associated with the project, but does delete the IAM roles that allowed access to the project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteProject.html
     */
    deleteProject() {
        this.add('codestar:DeleteProject');
        return this;
    }
    /**
     * Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such as display name and email address. It does not delete the history of that user, for example the history of commits made by that user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteUserProfile.html
     */
    deleteUserProfile() {
        this.add('codestar:DeleteUserProfile');
        return this;
    }
    /**
     * Describes a project and its resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeProject.html
     */
    describeProject() {
        this.add('codestar:DescribeProject');
        return this;
    }
    /**
     * Describes a user in AWS CodeStar and the user attributes across all projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeUserProfile.html
     */
    describeUserProfile() {
        this.add('codestar:DescribeUserProfile');
        return this;
    }
    /**
     * Removes a user from a project. Removing a user from a project also removes the IAM policies from that user that allowed access to the project and its resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DisassociateTeamMember.html
     */
    disassociateTeamMember() {
        this.add('codestar:DisassociateTeamMember');
        return this;
    }
    /**
     * Grants access to extended read APIs.
     *
     * Access Level: Read
     */
    getExtendedAccess() {
        this.add('codestar:GetExtendedAccess');
        return this;
    }
    /**
     * Lists all projects in CodeStar associated with your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListProjects.html
     */
    listProjects() {
        this.add('codestar:ListProjects');
        return this;
    }
    /**
     * Lists all resources associated with a project in CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListResources.html
     */
    listResources() {
        this.add('codestar:ListResources');
        return this;
    }
    /**
     * Lists the tags associated with a project in CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTagsForProject.html
     */
    listTagsForProject() {
        this.add('codestar:ListTagsForProject');
        return this;
    }
    /**
     * Lists all team members associated with a project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTeamMembers.html
     */
    listTeamMembers() {
        this.add('codestar:ListTeamMembers');
        return this;
    }
    /**
     * Lists user profiles in AWS CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListUserProfiles.html
     */
    listUserProfiles() {
        this.add('codestar:ListUserProfiles');
        return this;
    }
    /**
     * Grants access to extended write APIs.
     *
     * Access Level: Write
     */
    putExtendedAccess() {
        this.add('codestar:PutExtendedAccess');
        return this;
    }
    /**
     * Adds tags to a project in CodeStar.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_TagProject.html
     */
    tagProject() {
        this.add('codestar:TagProject');
        return this;
    }
    /**
     * Removes tags from a project in CodeStar.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UntagProject.html
     */
    untagProject() {
        this.add('codestar:UntagProject');
        return this;
    }
    /**
     * Updates a project in CodeStar.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateProject.html
     */
    updateProject() {
        this.add('codestar:UpdateProject');
        return this;
    }
    /**
     * Updates team member attributes within a CodeStar project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateTeamMember.html
     */
    updateTeamMember() {
        this.add('codestar:UpdateTeamMember');
        return this;
    }
    /**
     * Updates a profile for a user that includes user preferences, display name, and email.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateUserProfile.html
     */
    updateUserProfile() {
        this.add('codestar:UpdateUserProfile');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codestar/latest/userguide/working-with-projects.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - iam:ResourceTag/${TagKey}
     */
    onUser(userNameWithPath, account, region, partition) {
        var arn = 'arn:${Partition}:iam:${Region}:${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iam:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Codestar = Codestar;
//# sourceMappingURL=data:application/json;base64,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