"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticbeanstalk = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselasticbeanstalk.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticbeanstalk extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselasticbeanstalk.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticbeanstalk';
        this.actionList = {
            "AbortEnvironmentUpdate": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AbortEnvironmentUpdate.html",
                "description": "Grants permission to cancel in-progress environment configuration update or application version deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html",
                "description": "Grants permission to add tags to an Elastic Beanstalk resource and to update tag values.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "applicationversion": {
                        "required": false
                    },
                    "configurationtemplate": {
                        "required": false
                    },
                    "environment": {
                        "required": false
                    },
                    "platform": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ApplyEnvironmentManagedAction": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ApplyEnvironmentManagedAction.html",
                "description": "Grants permission to apply a scheduled managed action immediately.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "AssociateEnvironmentOperationsRole": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AssociateEnvironmentOperationsRole.html",
                "description": "Grants permission to associate an operations role with an environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "CheckDNSAvailability": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CheckDNSAvailability.html",
                "description": "Grants permission to check CNAME availability.",
                "accessLevel": "Read"
            },
            "ComposeEnvironments": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ComposeEnvironments.html",
                "description": "Grants permission to create or update a group of environments, each running a separate component of a single application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    },
                    "applicationversion": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "CreateApplication": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplication.html",
                "description": "Grants permission to create a new application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateApplicationVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplicationVersion.html",
                "description": "Grants permission to create an application version for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    },
                    "applicationversion": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConfigurationTemplate": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateConfigurationTemplate.html",
                "description": "Grants permission to create a configuration template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "elasticbeanstalk:FromApplication",
                    "elasticbeanstalk:FromApplicationVersion",
                    "elasticbeanstalk:FromConfigurationTemplate",
                    "elasticbeanstalk:FromEnvironment",
                    "elasticbeanstalk:FromSolutionStack",
                    "elasticbeanstalk:FromPlatform",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEnvironment": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateEnvironment.html",
                "description": "Grants permission to launch an environment for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "elasticbeanstalk:FromApplicationVersion",
                    "elasticbeanstalk:FromConfigurationTemplate",
                    "elasticbeanstalk:FromSolutionStack",
                    "elasticbeanstalk:FromPlatform",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePlatformVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreatePlatformVersion.html",
                "description": "Grants permission to create a new version of a custom platform.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "platform": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStorageLocation": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateStorageLocation.html",
                "description": "Grants permission to create the Amazon S3 storage location for the account.",
                "accessLevel": "Write"
            },
            "DeleteApplication": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplication.html",
                "description": "Grants permission to delete an application along with all associated versions and configurations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplicationVersion.html",
                "description": "Grants permission to delete an application version from an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applicationversion": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DeleteConfigurationTemplate": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteConfigurationTemplate.html",
                "description": "Grants permission to delete a configuration template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DeleteEnvironmentConfiguration": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteEnvironmentConfiguration.html",
                "description": "Grants permission to delete the draft configuration associated with the running environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DeletePlatformVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeletePlatformVersion.html",
                "description": "Grants permission to delete a version of a custom platform.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "platform": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeAccountAttributes.html",
                "description": "Grants permission to retrieve a list of account attributes, including resource quotas.",
                "accessLevel": "Read"
            },
            "DescribeApplicationVersions": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplicationVersions.html",
                "description": "Grants permission to retrieve a list of application versions stored in an AWS Elastic Beanstalk storage bucket.",
                "accessLevel": "List",
                "resourceTypes": {
                    "applicationversion": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeApplications": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplications.html",
                "description": "Grants permission to retrieve the descriptions of existing applications.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": false
                    }
                }
            },
            "DescribeConfigurationOptions": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationOptions.html",
                "description": "Grants permission to retrieve descriptions of environment configuration options.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "solutionstack": {
                        "required": false
                    }
                }
            },
            "DescribeConfigurationSettings": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationSettings.html",
                "description": "Grants permission to retrieve a description of the settings for a configuration set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeEnvironmentHealth": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentHealth.html",
                "description": "Grants permission to retrieve information about the overall health of an environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": false
                    }
                }
            },
            "DescribeEnvironmentManagedActionHistory": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActionHistory.html",
                "description": "Grants permission to retrieve a list of an environment's completed and failed managed actions.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeEnvironmentManagedActions": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActions.html",
                "description": "Grants permission to retrieve a list of an environment's upcoming and in-progress managed actions.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeEnvironmentResources": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentResources.html",
                "description": "Grants permission to retrieve a list of AWS resources for an environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeEnvironments": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironments.html",
                "description": "Grants permission to retrieve descriptions for existing environments.",
                "accessLevel": "List",
                "resourceTypes": {
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEvents.html",
                "description": "Grants permission to retrieve a list of event descriptions matching a set of criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "applicationversion": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "configurationtemplate": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "DescribeInstancesHealth": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeInstancesHealth.html",
                "description": "Grants permission to retrieve more detailed information about the health of environment instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": false
                    }
                }
            },
            "DescribePlatformVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribePlatformVersion.html",
                "description": "Grants permission to retrieve a description of a platform version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "platform": {
                        "required": false
                    }
                }
            },
            "DisassociateEnvironmentOperationsRole": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DisassociateEnvironmentOperationsRole.html",
                "description": "Grants permission to disassociate an operations role with an environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "ListAvailableSolutionStacks": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html",
                "description": "Grants permission to retrieve a list of the available solution stack names.",
                "accessLevel": "List",
                "resourceTypes": {
                    "solutionstack": {
                        "required": false
                    }
                }
            },
            "ListPlatformBranches": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformBranches.html",
                "description": "Grants permission to retrieve a list of the available platform branches.",
                "accessLevel": "List"
            },
            "ListPlatformVersions": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformVersions.html",
                "description": "Grants permission to retrieve a list of the available platforms.",
                "accessLevel": "List",
                "resourceTypes": {
                    "platform": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListTagsForResource.html",
                "description": "Grants permission to retrieve a list of tags of an Elastic Beanstalk resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "applicationversion": {
                        "required": false
                    },
                    "configurationtemplate": {
                        "required": false
                    },
                    "environment": {
                        "required": false
                    },
                    "platform": {
                        "required": false
                    }
                }
            },
            "RebuildEnvironment": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RebuildEnvironment.html",
                "description": "Grants permission to delete and recreate all of the AWS resources for an environment and to force a restart.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html",
                "description": "Grants permission to remove tags from an Elastic Beanstalk resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "applicationversion": {
                        "required": false
                    },
                    "configurationtemplate": {
                        "required": false
                    },
                    "environment": {
                        "required": false
                    },
                    "platform": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "RequestEnvironmentInfo": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RequestEnvironmentInfo.html",
                "description": "Grants permission to initiate a request to compile information of the deployed environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "RestartAppServer": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RestartAppServer.html",
                "description": "Grants permission to request an environment to restart the application container server running on each Amazon EC2 instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "RetrieveEnvironmentInfo": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RetrieveEnvironmentInfo.html",
                "description": "Grants permission to retrieve the compiled information from a RequestEnvironmentInfo request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "SwapEnvironmentCNAMEs": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_SwapEnvironmentCNAMEs.html",
                "description": "Grants permission to swap the CNAMEs of two environments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "elasticbeanstalk:FromEnvironment"
                ]
            },
            "TerminateEnvironment": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_TerminateEnvironment.html",
                "description": "Grants permission to terminate an environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "UpdateApplication": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplication.html",
                "description": "Grants permission to update an application with specified properties.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UpdateApplicationResourceLifecycle": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationResourceLifecycle.html",
                "description": "Grants permission to update the application version lifecycle policy associated with the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UpdateApplicationVersion": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationVersion.html",
                "description": "Grants permission to update an application version with specified properties.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applicationversion": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            },
            "UpdateConfigurationTemplate": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateConfigurationTemplate.html",
                "description": "Grants permission to update a configuration template with specified properties or configuration option values.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "elasticbeanstalk:FromApplication",
                    "elasticbeanstalk:FromApplicationVersion",
                    "elasticbeanstalk:FromConfigurationTemplate",
                    "elasticbeanstalk:FromEnvironment",
                    "elasticbeanstalk:FromSolutionStack",
                    "elasticbeanstalk:FromPlatform"
                ]
            },
            "UpdateEnvironment": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateEnvironment.html",
                "description": "Grants permission to update an environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                },
                "conditions": [
                    "elasticbeanstalk:FromApplicationVersion",
                    "elasticbeanstalk:FromConfigurationTemplate",
                    "elasticbeanstalk:FromSolutionStack",
                    "elasticbeanstalk:FromPlatform"
                ]
            },
            "ValidateConfigurationSettings": {
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ValidateConfigurationSettings.html",
                "description": "Grants permission to check the validity of a set of configuration settings for a configuration template or an environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configurationtemplate": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    },
                    "environment": {
                        "required": false,
                        "conditions": [
                            "elasticbeanstalk:InApplication"
                        ]
                    }
                }
            }
        };
        this.resourceTypes = {
            "application": {
                "name": "application",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}:${Account}:application/${ApplicationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "applicationversion": {
                "name": "applicationversion",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}:${Account}:applicationversion/${ApplicationName}/${VersionLabel}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "elasticbeanstalk:InApplication"
                ]
            },
            "configurationtemplate": {
                "name": "configurationtemplate",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}:${Account}:configurationtemplate/${ApplicationName}/${TemplateName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "elasticbeanstalk:InApplication"
                ]
            },
            "environment": {
                "name": "environment",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}:${Account}:environment/${ApplicationName}/${EnvironmentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "elasticbeanstalk:InApplication"
                ]
            },
            "solutionstack": {
                "name": "solutionstack",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}::solutionstack/${SolutionStackName}",
                "conditionKeys": []
            },
            "platform": {
                "name": "platform",
                "url": "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html",
                "arn": "arn:${Partition}:elasticbeanstalk:${Region}::platform/${PlatformNameWithVersion}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to cancel in-progress environment configuration update or application version deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AbortEnvironmentUpdate.html
     */
    abortEnvironmentUpdate() {
        this.add('elasticbeanstalk:AbortEnvironmentUpdate');
        return this;
    }
    /**
     * Grants permission to add tags to an Elastic Beanstalk resource and to update tag values.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    addTags() {
        this.add('elasticbeanstalk:AddTags');
        return this;
    }
    /**
     * Grants permission to apply a scheduled managed action immediately.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ApplyEnvironmentManagedAction.html
     */
    applyEnvironmentManagedAction() {
        this.add('elasticbeanstalk:ApplyEnvironmentManagedAction');
        return this;
    }
    /**
     * Grants permission to associate an operations role with an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AssociateEnvironmentOperationsRole.html
     */
    associateEnvironmentOperationsRole() {
        this.add('elasticbeanstalk:AssociateEnvironmentOperationsRole');
        return this;
    }
    /**
     * Grants permission to check CNAME availability.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CheckDNSAvailability.html
     */
    checkDNSAvailability() {
        this.add('elasticbeanstalk:CheckDNSAvailability');
        return this;
    }
    /**
     * Grants permission to create or update a group of environments, each running a separate component of a single application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ComposeEnvironments.html
     */
    composeEnvironments() {
        this.add('elasticbeanstalk:ComposeEnvironments');
        return this;
    }
    /**
     * Grants permission to create a new application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplication.html
     */
    createApplication() {
        this.add('elasticbeanstalk:CreateApplication');
        return this;
    }
    /**
     * Grants permission to create an application version for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplicationVersion.html
     */
    createApplicationVersion() {
        this.add('elasticbeanstalk:CreateApplicationVersion');
        return this;
    }
    /**
     * Grants permission to create a configuration template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateConfigurationTemplate.html
     */
    createConfigurationTemplate() {
        this.add('elasticbeanstalk:CreateConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to launch an environment for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateEnvironment.html
     */
    createEnvironment() {
        this.add('elasticbeanstalk:CreateEnvironment');
        return this;
    }
    /**
     * Grants permission to create a new version of a custom platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreatePlatformVersion.html
     */
    createPlatformVersion() {
        this.add('elasticbeanstalk:CreatePlatformVersion');
        return this;
    }
    /**
     * Grants permission to create the Amazon S3 storage location for the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateStorageLocation.html
     */
    createStorageLocation() {
        this.add('elasticbeanstalk:CreateStorageLocation');
        return this;
    }
    /**
     * Grants permission to delete an application along with all associated versions and configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplication.html
     */
    deleteApplication() {
        this.add('elasticbeanstalk:DeleteApplication');
        return this;
    }
    /**
     * Grants permission to delete an application version from an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplicationVersion.html
     */
    deleteApplicationVersion() {
        this.add('elasticbeanstalk:DeleteApplicationVersion');
        return this;
    }
    /**
     * Grants permission to delete a configuration template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteConfigurationTemplate.html
     */
    deleteConfigurationTemplate() {
        this.add('elasticbeanstalk:DeleteConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to delete the draft configuration associated with the running environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteEnvironmentConfiguration.html
     */
    deleteEnvironmentConfiguration() {
        this.add('elasticbeanstalk:DeleteEnvironmentConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a version of a custom platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeletePlatformVersion.html
     */
    deletePlatformVersion() {
        this.add('elasticbeanstalk:DeletePlatformVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of account attributes, including resource quotas.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('elasticbeanstalk:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to retrieve a list of application versions stored in an AWS Elastic Beanstalk storage bucket.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplicationVersions.html
     */
    describeApplicationVersions() {
        this.add('elasticbeanstalk:DescribeApplicationVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the descriptions of existing applications.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplications.html
     */
    describeApplications() {
        this.add('elasticbeanstalk:DescribeApplications');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions of environment configuration options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationOptions.html
     */
    describeConfigurationOptions() {
        this.add('elasticbeanstalk:DescribeConfigurationOptions');
        return this;
    }
    /**
     * Grants permission to retrieve a description of the settings for a configuration set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationSettings.html
     */
    describeConfigurationSettings() {
        this.add('elasticbeanstalk:DescribeConfigurationSettings');
        return this;
    }
    /**
     * Grants permission to retrieve information about the overall health of an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentHealth.html
     */
    describeEnvironmentHealth() {
        this.add('elasticbeanstalk:DescribeEnvironmentHealth');
        return this;
    }
    /**
     * Grants permission to retrieve a list of an environment's completed and failed managed actions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActionHistory.html
     */
    describeEnvironmentManagedActionHistory() {
        this.add('elasticbeanstalk:DescribeEnvironmentManagedActionHistory');
        return this;
    }
    /**
     * Grants permission to retrieve a list of an environment's upcoming and in-progress managed actions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActions.html
     */
    describeEnvironmentManagedActions() {
        this.add('elasticbeanstalk:DescribeEnvironmentManagedActions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS resources for an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentResources.html
     */
    describeEnvironmentResources() {
        this.add('elasticbeanstalk:DescribeEnvironmentResources');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions for existing environments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironments.html
     */
    describeEnvironments() {
        this.add('elasticbeanstalk:DescribeEnvironments');
        return this;
    }
    /**
     * Grants permission to retrieve a list of event descriptions matching a set of criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('elasticbeanstalk:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to retrieve more detailed information about the health of environment instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeInstancesHealth.html
     */
    describeInstancesHealth() {
        this.add('elasticbeanstalk:DescribeInstancesHealth');
        return this;
    }
    /**
     * Grants permission to retrieve a description of a platform version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribePlatformVersion.html
     */
    describePlatformVersion() {
        this.add('elasticbeanstalk:DescribePlatformVersion');
        return this;
    }
    /**
     * Grants permission to disassociate an operations role with an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DisassociateEnvironmentOperationsRole.html
     */
    disassociateEnvironmentOperationsRole() {
        this.add('elasticbeanstalk:DisassociateEnvironmentOperationsRole');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available solution stack names.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html
     */
    listAvailableSolutionStacks() {
        this.add('elasticbeanstalk:ListAvailableSolutionStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available platform branches.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformBranches.html
     */
    listPlatformBranches() {
        this.add('elasticbeanstalk:ListPlatformBranches');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the available platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformVersions.html
     */
    listPlatformVersions() {
        this.add('elasticbeanstalk:ListPlatformVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags of an Elastic Beanstalk resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('elasticbeanstalk:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to delete and recreate all of the AWS resources for an environment and to force a restart.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RebuildEnvironment.html
     */
    rebuildEnvironment() {
        this.add('elasticbeanstalk:RebuildEnvironment');
        return this;
    }
    /**
     * Grants permission to remove tags from an Elastic Beanstalk resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    removeTags() {
        this.add('elasticbeanstalk:RemoveTags');
        return this;
    }
    /**
     * Grants permission to initiate a request to compile information of the deployed environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RequestEnvironmentInfo.html
     */
    requestEnvironmentInfo() {
        this.add('elasticbeanstalk:RequestEnvironmentInfo');
        return this;
    }
    /**
     * Grants permission to request an environment to restart the application container server running on each Amazon EC2 instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RestartAppServer.html
     */
    restartAppServer() {
        this.add('elasticbeanstalk:RestartAppServer');
        return this;
    }
    /**
     * Grants permission to retrieve the compiled information from a RequestEnvironmentInfo request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RetrieveEnvironmentInfo.html
     */
    retrieveEnvironmentInfo() {
        this.add('elasticbeanstalk:RetrieveEnvironmentInfo');
        return this;
    }
    /**
     * Grants permission to swap the CNAMEs of two environments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_SwapEnvironmentCNAMEs.html
     */
    swapEnvironmentCNAMEs() {
        this.add('elasticbeanstalk:SwapEnvironmentCNAMEs');
        return this;
    }
    /**
     * Grants permission to terminate an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_TerminateEnvironment.html
     */
    terminateEnvironment() {
        this.add('elasticbeanstalk:TerminateEnvironment');
        return this;
    }
    /**
     * Grants permission to update an application with specified properties.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplication.html
     */
    updateApplication() {
        this.add('elasticbeanstalk:UpdateApplication');
        return this;
    }
    /**
     * Grants permission to update the application version lifecycle policy associated with the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationResourceLifecycle.html
     */
    updateApplicationResourceLifecycle() {
        this.add('elasticbeanstalk:UpdateApplicationResourceLifecycle');
        return this;
    }
    /**
     * Grants permission to update an application version with specified properties.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationVersion.html
     */
    updateApplicationVersion() {
        this.add('elasticbeanstalk:UpdateApplicationVersion');
        return this;
    }
    /**
     * Grants permission to update a configuration template with specified properties or configuration option values.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateConfigurationTemplate.html
     */
    updateConfigurationTemplate() {
        this.add('elasticbeanstalk:UpdateConfigurationTemplate');
        return this;
    }
    /**
     * Grants permission to update an environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateEnvironment.html
     */
    updateEnvironment() {
        this.add('elasticbeanstalk:UpdateEnvironment');
        return this;
    }
    /**
     * Grants permission to check the validity of a set of configuration settings for a configuration template or an environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ValidateConfigurationSettings.html
     */
    validateConfigurationSettings() {
        this.add('elasticbeanstalk:ValidateConfigurationSettings');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:application/${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type applicationversion to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param versionLabel - Identifier for the versionLabel.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - elasticbeanstalk:InApplication
     */
    onApplicationversion(applicationName, versionLabel, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:applicationversion/${ApplicationName}/${VersionLabel}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${VersionLabel}', versionLabel);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type configurationtemplate to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - elasticbeanstalk:InApplication
     */
    onConfigurationtemplate(applicationName, templateName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:configurationtemplate/${ApplicationName}/${TemplateName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param environmentName - Identifier for the environmentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - elasticbeanstalk:InApplication
     */
    onEnvironment(applicationName, environmentName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}:${Account}:environment/${ApplicationName}/${EnvironmentName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${EnvironmentName}', environmentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type solutionstack to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param solutionStackName - Identifier for the solutionStackName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSolutionstack(solutionStackName, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}::solutionstack/${SolutionStackName}';
        arn = arn.replace('${SolutionStackName}', solutionStackName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type platform to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param platformNameWithVersion - Identifier for the platformNameWithVersion.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPlatform(platformNameWithVersion, region, partition) {
        var arn = 'arn:${Partition}:elasticbeanstalk:${Region}::platform/${PlatformNameWithVersion}';
        arn = arn.replace('${PlatformNameWithVersion}', platformNameWithVersion);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by an application as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromApplication(value, operator) {
        return this.if(`elasticbeanstalk:FromApplication`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by an application version as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromApplicationVersion(value, operator) {
        return this.if(`elasticbeanstalk:FromApplicationVersion`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by a configuration template as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromConfigurationTemplate(value, operator) {
        return this.if(`elasticbeanstalk:FromConfigurationTemplate`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by an environment as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromEnvironment(value, operator) {
        return this.if(`elasticbeanstalk:FromEnvironment`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by a platform as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromPlatform(value, operator) {
        return this.if(`elasticbeanstalk:FromPlatform`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by a solution stack as a dependency or a constraint on an input parameter.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFromSolutionStack(value, operator) {
        return this.if(`elasticbeanstalk:FromSolutionStack`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the application that contains the resource that the action operates on.
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifInApplication(value, operator) {
        return this.if(`elasticbeanstalk:InApplication`, value, operator || 'ArnEquals');
    }
}
exports.Elasticbeanstalk = Elasticbeanstalk;
//# sourceMappingURL=data:application/json;base64,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