"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Medialive = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [medialive](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmedialive.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Medialive extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [medialive](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmedialive.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'medialive';
        this.actionList = {
            "BatchUpdateSchedule": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/batching-actions.html",
                "description": "Grants permission to add and remove actions from a channel's schedule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "CreateChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/creating-channel-scratch.html",
                "description": "Grants permission to create a channel",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    },
                    "input": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInput": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html",
                "description": "Grants permission to create an input",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "input": {
                        "required": true
                    },
                    "input-security-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInputSecurityGroup": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html",
                "description": "Grants permission to create an input security group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "input-security-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/creating-multiplex.html",
                "description": "Grants permission to create a multiplex",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/tagging.html",
                "description": "Grants permission to create tags for channels, inputs, input security groups, multiplexes, and reservations.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    },
                    "input-security-group": {
                        "required": false
                    },
                    "multiplex": {
                        "required": false
                    },
                    "reservation": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html",
                "description": "Grants permission to delete a channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteInput": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/delete-input.html",
                "description": "Grants permission to delete an input",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DeleteInputSecurityGroup": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/delete-input-security-group.html",
                "description": "Grants permission to delete an input security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input-security-group": {
                        "required": true
                    }
                }
            },
            "DeleteMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex.html",
                "description": "Grants permission to delete a multiplex",
                "accessLevel": "Write",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                }
            },
            "DeleteReservation": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/deleting-reservations.html",
                "description": "Grants permission to delete an expired reservation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reservation": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/tagging.html",
                "description": "Grants permission to delete tags from channels, inputs, input security groups, multiplexes, and reservations.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    },
                    "input-security-group": {
                        "required": false
                    },
                    "multiplex": {
                        "required": false
                    },
                    "reservation": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DescribeChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html",
                "description": "Grants permission to get details about a channel",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DescribeInput": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html",
                "description": "Grants permission to describe an input",
                "accessLevel": "Read",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DescribeInputDevice": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html",
                "description": "Grants permission to describe an input device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "input-device": {
                        "required": true
                    }
                }
            },
            "DescribeInputSecurityGroup": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html",
                "description": "Grants permission to describe an input security group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "input-security-group": {
                        "required": true
                    }
                }
            },
            "DescribeMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/viewing-multiplex-configuration.html",
                "description": "Grants permission to describe a multiplex",
                "accessLevel": "Read",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                }
            },
            "DescribeOffering": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html",
                "description": "Grants permission to get details about a reservation offering",
                "accessLevel": "Read",
                "resourceTypes": {
                    "offering": {
                        "required": true
                    }
                }
            },
            "DescribeReservation": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html",
                "description": "Grants permission to get details about a reservation",
                "accessLevel": "Read",
                "resourceTypes": {
                    "reservation": {
                        "required": true
                    }
                }
            },
            "DescribeSchedule": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/viewing-actions-schedule.html",
                "description": "Grants permission to view a list of actions scheduled on a channel.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "ListChannels": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html",
                "description": "Grants permission to list channels",
                "accessLevel": "List"
            },
            "ListInputDevices": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html",
                "description": "Grants permission to list input devices",
                "accessLevel": "List"
            },
            "ListInputSecurityGroups": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html",
                "description": "Grants permission to list input security groups",
                "accessLevel": "List"
            },
            "ListInputs": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html",
                "description": "Grants permission to list inputs",
                "accessLevel": "List"
            },
            "ListMultiplexes": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/viewing-multiplex-configuration.html",
                "description": "Grants permission to list multiplexes",
                "accessLevel": "List"
            },
            "ListOfferings": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html",
                "description": "Grants permission to list reservation offerings",
                "accessLevel": "List"
            },
            "ListReservations": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html",
                "description": "Grants permission to list reservations",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/tagging.html",
                "description": "Grants permission to list tags for channels, inputs, input security groups, multiplexes, and reservations.",
                "accessLevel": "List",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    },
                    "input-security-group": {
                        "required": false
                    },
                    "multiplex": {
                        "required": false
                    },
                    "reservation": {
                        "required": false
                    }
                }
            },
            "PurchaseOffering": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html",
                "description": "Grants permission to purchase a reservation offering",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "offering": {
                        "required": true
                    },
                    "reservation": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html",
                "description": "Grants permission to start a channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "StartMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-a-multiplex.html",
                "description": "Grants permission to start a multiplex",
                "accessLevel": "Write",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                }
            },
            "StopChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html",
                "description": "Grants permission to stop a channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "StopMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-a-multiplex.html",
                "description": "Grants permission to stop a multiplex",
                "accessLevel": "Write",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                }
            },
            "UpdateChannel": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html",
                "description": "Grants permission to update a channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "UpdateChannelClass": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html",
                "description": "Grants permission to update the class of a channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "UpdateInput": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html",
                "description": "Grants permission to update an input",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "UpdateInputDevice": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html",
                "description": "Grants permission to update an input device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input-device": {
                        "required": true
                    }
                }
            },
            "UpdateInputSecurityGroup": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html",
                "description": "Grants permission to update an input security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input-security-group": {
                        "required": true
                    }
                }
            },
            "UpdateMultiplex": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex.html",
                "description": "Grants permission to update a multiplex",
                "accessLevel": "Write",
                "resourceTypes": {
                    "multiplex": {
                        "required": true
                    }
                }
            },
            "UpdateReservation": {
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/reservations.html",
                "description": "Grants permission to update a reservation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reservation": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/channels.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:channel:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "input": {
                "name": "input",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/inputs.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:input:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "input-device": {
                "name": "input-device",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/inputdevices.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:inputDevice:${ResourceName}",
                "conditionKeys": []
            },
            "input-security-group": {
                "name": "input-security-group",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/inputsecuritygroups.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:inputSecurityGroup:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "multiplex": {
                "name": "multiplex",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/multiplexes.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:multiplex:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "reservation": {
                "name": "reservation",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/reservations.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:reservation:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "offering": {
                "name": "offering",
                "url": "https://docs.aws.amazon.com/medialive/latest/ug/input-output-reservations.html",
                "arn": "arn:${Partition}:medialive:${Region}:${Account}:offering:${ResourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add and remove actions from a channel's schedule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/batching-actions.html
     */
    batchUpdateSchedule() {
        this.add('medialive:BatchUpdateSchedule');
        return this;
    }
    /**
     * Grants permission to create a channel
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-channel-scratch.html
     */
    createChannel() {
        this.add('medialive:CreateChannel');
        return this;
    }
    /**
     * Grants permission to create an input
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html
     */
    createInput() {
        this.add('medialive:CreateInput');
        return this;
    }
    /**
     * Grants permission to create an input security group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html
     */
    createInputSecurityGroup() {
        this.add('medialive:CreateInputSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a multiplex
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-multiplex.html
     */
    createMultiplex() {
        this.add('medialive:CreateMultiplex');
        return this;
    }
    /**
     * Grants permission to create tags for channels, inputs, input security groups, multiplexes, and reservations.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    createTags() {
        this.add('medialive:CreateTags');
        return this;
    }
    /**
     * Grants permission to delete a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    deleteChannel() {
        this.add('medialive:DeleteChannel');
        return this;
    }
    /**
     * Grants permission to delete an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input.html
     */
    deleteInput() {
        this.add('medialive:DeleteInput');
        return this;
    }
    /**
     * Grants permission to delete an input security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input-security-group.html
     */
    deleteInputSecurityGroup() {
        this.add('medialive:DeleteInputSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex.html
     */
    deleteMultiplex() {
        this.add('medialive:DeleteMultiplex');
        return this;
    }
    /**
     * Grants permission to delete an expired reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/deleting-reservations.html
     */
    deleteReservation() {
        this.add('medialive:DeleteReservation');
        return this;
    }
    /**
     * Grants permission to delete tags from channels, inputs, input security groups, multiplexes, and reservations.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    deleteTags() {
        this.add('medialive:DeleteTags');
        return this;
    }
    /**
     * Grants permission to get details about a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    describeChannel() {
        this.add('medialive:DescribeChannel');
        return this;
    }
    /**
     * Grants permission to describe an input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    describeInput() {
        this.add('medialive:DescribeInput');
        return this;
    }
    /**
     * Grants permission to describe an input device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html
     */
    describeInputDevice() {
        this.add('medialive:DescribeInputDevice');
        return this;
    }
    /**
     * Grants permission to describe an input security group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    describeInputSecurityGroup() {
        this.add('medialive:DescribeInputSecurityGroup');
        return this;
    }
    /**
     * Grants permission to describe a multiplex
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-multiplex-configuration.html
     */
    describeMultiplex() {
        this.add('medialive:DescribeMultiplex');
        return this;
    }
    /**
     * Grants permission to get details about a reservation offering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    describeOffering() {
        this.add('medialive:DescribeOffering');
        return this;
    }
    /**
     * Grants permission to get details about a reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    describeReservation() {
        this.add('medialive:DescribeReservation');
        return this;
    }
    /**
     * Grants permission to view a list of actions scheduled on a channel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-actions-schedule.html
     */
    describeSchedule() {
        this.add('medialive:DescribeSchedule');
        return this;
    }
    /**
     * Grants permission to list channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    listChannels() {
        this.add('medialive:ListChannels');
        return this;
    }
    /**
     * Grants permission to list input devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html
     */
    listInputDevices() {
        this.add('medialive:ListInputDevices');
        return this;
    }
    /**
     * Grants permission to list input security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    listInputSecurityGroups() {
        this.add('medialive:ListInputSecurityGroups');
        return this;
    }
    /**
     * Grants permission to list inputs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    listInputs() {
        this.add('medialive:ListInputs');
        return this;
    }
    /**
     * Grants permission to list multiplexes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-multiplex-configuration.html
     */
    listMultiplexes() {
        this.add('medialive:ListMultiplexes');
        return this;
    }
    /**
     * Grants permission to list reservation offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    listOfferings() {
        this.add('medialive:ListOfferings');
        return this;
    }
    /**
     * Grants permission to list reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    listReservations() {
        this.add('medialive:ListReservations');
        return this;
    }
    /**
     * Grants permission to list tags for channels, inputs, input security groups, multiplexes, and reservations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    listTagsForResource() {
        this.add('medialive:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to purchase a reservation offering
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    purchaseOffering() {
        this.add('medialive:PurchaseOffering');
        return this;
    }
    /**
     * Grants permission to start a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    startChannel() {
        this.add('medialive:StartChannel');
        return this;
    }
    /**
     * Grants permission to start a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-a-multiplex.html
     */
    startMultiplex() {
        this.add('medialive:StartMultiplex');
        return this;
    }
    /**
     * Grants permission to stop a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    stopChannel() {
        this.add('medialive:StopChannel');
        return this;
    }
    /**
     * Grants permission to stop a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-a-multiplex.html
     */
    stopMultiplex() {
        this.add('medialive:StopMultiplex');
        return this;
    }
    /**
     * Grants permission to update a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    updateChannel() {
        this.add('medialive:UpdateChannel');
        return this;
    }
    /**
     * Grants permission to update the class of a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    updateChannelClass() {
        this.add('medialive:UpdateChannelClass');
        return this;
    }
    /**
     * Grants permission to update an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    updateInput() {
        this.add('medialive:UpdateInput');
        return this;
    }
    /**
     * Grants permission to update an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-device.html
     */
    updateInputDevice() {
        this.add('medialive:UpdateInputDevice');
        return this;
    }
    /**
     * Grants permission to update an input security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    updateInputSecurityGroup() {
        this.add('medialive:UpdateInputSecurityGroup');
        return this;
    }
    /**
     * Grants permission to update a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex.html
     */
    updateMultiplex() {
        this.add('medialive:UpdateMultiplex');
        return this;
    }
    /**
     * Grants permission to update a reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     */
    updateReservation() {
        this.add('medialive:UpdateReservation');
        return this;
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/channels.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onChannel(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:channel:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/inputs.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInput(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:input:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input-device to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/inputdevices.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInputDevice(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:inputDevice:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input-security-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/inputsecuritygroups.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInputSecurityGroup(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:inputSecurityGroup:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type multiplex to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/multiplexes.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMultiplex(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:multiplex:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reservation to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onReservation(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:reservation:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type offering to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/input-output-reservations.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOffering(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:medialive:${Region}:${Account}:offering:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Medialive = Medialive;
//# sourceMappingURL=data:application/json;base64,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