"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glue = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glue extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glue';
        this.actionList = {
            "BatchCreatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition",
                "description": "Grants permission to create one or more partitions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection",
                "description": "Grants permission to delete one or more connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "BatchDeletePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition",
                "description": "Grants permission to delete one or more partitions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable",
                "description": "Grants permission to delete one or more tables",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchDeleteTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion",
                "description": "Grants permission to delete one or more versions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "BatchGetCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers",
                "description": "Grants permission to retrieve one or more crawlers",
                "accessLevel": "Read"
            },
            "BatchGetDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints",
                "description": "Grants permission to retrieve one or more development endpoints",
                "accessLevel": "Read"
            },
            "BatchGetJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs",
                "description": "Grants permission to retrieve one or more jobs",
                "accessLevel": "Read"
            },
            "BatchGetPartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition",
                "description": "Grants permission to retrieve one or more partitions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "BatchGetTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers",
                "description": "Grants permission to retrieve one or more triggers",
                "accessLevel": "Read"
            },
            "BatchGetWorkflows": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows",
                "description": "Grants permission to retrieve one or more workflows",
                "accessLevel": "Read"
            },
            "BatchStopJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun",
                "description": "Grants permission to stop one or more job runs for a job",
                "accessLevel": "Write"
            },
            "CancelMLTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun",
                "description": "Grants permission to stop a running ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "CreateClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier",
                "description": "Grants permission to create a classifier",
                "accessLevel": "Write"
            },
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection",
                "description": "Grants permission to create a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "CreateCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler",
                "description": "Grants permission to create a crawler",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase",
                "description": "Grants permission to create a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "CreateDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint",
                "description": "Grants permission to create a development endpoint",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob",
                "description": "Grants permission to create a job",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform",
                "description": "Grants permission to create an ML Transform",
                "accessLevel": "Write"
            },
            "CreatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition",
                "description": "Grants permission to create a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateScript": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript",
                "description": "Grants permission to create a script",
                "accessLevel": "Write"
            },
            "CreateSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration",
                "description": "Grants permission to create a security configuration",
                "accessLevel": "Write"
            },
            "CreateTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable",
                "description": "Grants permission to create a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "CreateTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger",
                "description": "Grants permission to create a trigger",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction",
                "description": "Grants permission to create a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "CreateWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow",
                "description": "Grants permission to create a workflow",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier",
                "description": "Grants permission to delete a classifier",
                "accessLevel": "Write"
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection",
                "description": "Grants permission to delete a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "DeleteCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler",
                "description": "Grants permission to delete a crawler",
                "accessLevel": "Write"
            },
            "DeleteDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase",
                "description": "Grants permission to delete a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "DeleteDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint",
                "description": "Grants permission to delete a development endpoint",
                "accessLevel": "Write"
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob",
                "description": "Grants permission to delete a job",
                "accessLevel": "Write"
            },
            "DeleteMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform",
                "description": "Grants permission to delete an ML Transform",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "DeletePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition",
                "description": "Grants permission to delete a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy",
                "description": "Grants permission to delete a resource policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "DeleteSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration",
                "description": "Grants permission to delete a security configuration",
                "accessLevel": "Write"
            },
            "DeleteTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable",
                "description": "Grants permission to delete a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "DeleteTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion",
                "description": "Grants permission to delete a version of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "DeleteTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger",
                "description": "Grants permission to delete a trigger",
                "accessLevel": "Write"
            },
            "DeleteUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction",
                "description": "Grants permission to delete a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "DeleteWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow",
                "description": "Grants permission to delete a workflow",
                "accessLevel": "Write"
            },
            "GetCatalogImportStatus": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus",
                "description": "Grants permission to retrieve the catalog import status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "GetClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier",
                "description": "Grants permission to retrieve a classifier",
                "accessLevel": "Read"
            },
            "GetClassifiers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers",
                "description": "Grants permission to list all classifiers",
                "accessLevel": "Read"
            },
            "GetConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection",
                "description": "Grants permission to retrieve a connection",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "GetConnections": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections",
                "description": "Grants permission to retrieve a list of connections",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "GetCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler",
                "description": "Grants permission to retrieve a crawler",
                "accessLevel": "Read"
            },
            "GetCrawlerMetrics": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics",
                "description": "Grants permission to retrieve metrics about crawlers",
                "accessLevel": "Read"
            },
            "GetCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers",
                "description": "Grants permission to retrieve all crawlers",
                "accessLevel": "Read"
            },
            "GetDataCatalogEncryptionSettings": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings",
                "description": "Grants permission to retrieve catalog encryption settings",
                "accessLevel": "Read"
            },
            "GetDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase",
                "description": "Grants permission to retrieve a database",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "GetDatabases": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases",
                "description": "Grants permission to retrieve all databases",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "GetDataflowGraph": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph",
                "description": "Grants permission to transform a script into a directed acyclic graph (DAG)",
                "accessLevel": "Read"
            },
            "GetDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint",
                "description": "Grants permission to retrieve a development endpoint",
                "accessLevel": "Read"
            },
            "GetDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints",
                "description": "Grants permission to retrieve all development endpoints",
                "accessLevel": "Read"
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob",
                "description": "Grants permission to retrieve a job",
                "accessLevel": "Read"
            },
            "GetJobBookmark": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark",
                "description": "Grants permission to retrieve a job bookmark",
                "accessLevel": "Read"
            },
            "GetJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun",
                "description": "Grants permission to retrieve a job run",
                "accessLevel": "Read"
            },
            "GetJobRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns",
                "description": "Grants permission to retrieve all job runs of a job",
                "accessLevel": "Read"
            },
            "GetJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs",
                "description": "Grants permission to retrieve all current jobs",
                "accessLevel": "Read"
            },
            "GetMLTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun",
                "description": "Grants permission to retrieve an ML Task Run",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTaskRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns",
                "description": "Grants permission to retrieve all ML Task Runs",
                "accessLevel": "List",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform",
                "description": "Grants permission to retrieve an ML Transform",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "GetMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms",
                "description": "Grants permission to retrieve all ML Transforms",
                "accessLevel": "List"
            },
            "GetMapping": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping",
                "description": "Grants permission to create a mapping",
                "accessLevel": "Write"
            },
            "GetPartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition",
                "description": "Grants permission to retrieve a partition",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetPartitions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions",
                "description": "Grants permission to retrieve the partitions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetPlan": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan",
                "description": "Grants permission to retrieve a mapping for a script",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy",
                "description": "Grants permission to retrieve a resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "GetSecurityConfiguration": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration",
                "description": "Grants permission to retrieve a security configuration",
                "accessLevel": "Read"
            },
            "GetSecurityConfigurations": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations",
                "description": "Grants permission to retrieve one or more security configurations",
                "accessLevel": "Read"
            },
            "GetTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable",
                "description": "Grants permission to retrieve a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetTableVersion": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion",
                "description": "Grants permission to retrieve a version of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "GetTableVersions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions",
                "description": "Grants permission to retrieve a list of versions of a table",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    },
                    "tableversion": {
                        "required": true
                    }
                }
            },
            "GetTables": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables",
                "description": "Grants permission to retrieve the tables in a database",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "GetTags": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource",
                "description": "Grants permission to retrieve all tags associated with a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                }
            },
            "GetTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger",
                "description": "Grants permission to retrieve a trigger",
                "accessLevel": "Read"
            },
            "GetTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers",
                "description": "Grants permission to retrieve the triggers associated with a job",
                "accessLevel": "Read"
            },
            "GetUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction",
                "description": "Grants permission to retrieve a function definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "GetUserDefinedFunctions": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions",
                "description": "Grants permission to retrieve multiple function definitions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "GetWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow",
                "description": "Grants permission to retrieve a workflow",
                "accessLevel": "Read"
            },
            "GetWorkflowRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun",
                "description": "Grants permission to retrieve a workflow run",
                "accessLevel": "Read"
            },
            "GetWorkflowRunProperties": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties",
                "description": "Grants permission to retrieve workflow run properties",
                "accessLevel": "Read"
            },
            "GetWorkflowRuns": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns",
                "description": "Grants permission to retrieve all runs of a workflow",
                "accessLevel": "Read"
            },
            "ImportCatalogToGlue": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue",
                "description": "Grants permission to import an Athena data catalog into AWS Glue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "ListCrawlers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers",
                "description": "Grants permission to retrieve all crawlers",
                "accessLevel": "List"
            },
            "ListDevEndpoints": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints",
                "description": "Grants permission to retrieve all development endpoints",
                "accessLevel": "List"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs",
                "description": "Grants permission to retrieve all current jobs",
                "accessLevel": "List"
            },
            "ListMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms",
                "description": "Grants permission to retrieve all ML Transforms",
                "accessLevel": "List"
            },
            "ListTriggers": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers",
                "description": "Grants permission to retrieve all triggers",
                "accessLevel": "List"
            },
            "ListWorkflows": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows",
                "description": "Grants permission to retrieve all workflows",
                "accessLevel": "List"
            },
            "PutDataCatalogEncryptionSettings": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings",
                "description": "Grants permission to update catalog encryption settings",
                "accessLevel": "Write"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy",
                "description": "Grants permission to update a resource policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    }
                }
            },
            "PutWorkflowRunProperties": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties",
                "description": "Grants permission to update workflow run properties",
                "accessLevel": "Write"
            },
            "ResetJobBookmark": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark",
                "description": "Grants permission to reset a job bookmark",
                "accessLevel": "Write"
            },
            "SearchTables": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables",
                "description": "Grants permission to retrieve the tables in the catalog",
                "accessLevel": "Read",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "StartCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler",
                "description": "Grants permission to start a crawler",
                "accessLevel": "Write"
            },
            "StartCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule",
                "description": "Grants permission to change the schedule state of a crawler to SCHEDULED",
                "accessLevel": "Write"
            },
            "StartExportLabelsTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun",
                "description": "Grants permission to start an Export Labels ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartImportLabelsTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun",
                "description": "Grants permission to start an Import Labels ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartJobRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun",
                "description": "Grants permission to start running a job",
                "accessLevel": "Write"
            },
            "StartMLEvaluationTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun",
                "description": "Grants permission to start an Evaluation ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartMLLabelingSetGenerationTaskRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun",
                "description": "Grants permission to start a Labeling Set Generation ML Task Run",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "StartTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger",
                "description": "Grants permission to start a trigger",
                "accessLevel": "Write"
            },
            "StartWorkflowRun": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun",
                "description": "Grants permission to start running a workflow",
                "accessLevel": "Write"
            },
            "StopCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler",
                "description": "Grants permission to stop a running crawler",
                "accessLevel": "Write"
            },
            "StopCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule",
                "description": "Grants permission to set the schedule state of a crawler to NOT_SCHEDULED",
                "accessLevel": "Write"
            },
            "StopTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger",
                "description": "Grants permission to stop a trigger",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource",
                "description": "Grants permission to add tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource",
                "description": "Grants permission to remove tags associated with a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "crawler": {
                        "required": false
                    },
                    "devendpoint": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "trigger": {
                        "required": false
                    },
                    "workflow": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateClassifier": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier",
                "description": "Grants permission to update a classifier",
                "accessLevel": "Write"
            },
            "UpdateConnection": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection",
                "description": "Grants permission to update a connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "connection": {
                        "required": true
                    }
                }
            },
            "UpdateCrawler": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler",
                "description": "Grants permission to update a crawler",
                "accessLevel": "Write"
            },
            "UpdateCrawlerSchedule": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule",
                "description": "Grants permission to update the schedule of a crawler",
                "accessLevel": "Write"
            },
            "UpdateDatabase": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase",
                "description": "Grants permission to update a database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    }
                }
            },
            "UpdateDevEndpoint": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint",
                "description": "Grants permission to update a development endpoint",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob",
                "description": "Grants permission to update a job",
                "accessLevel": "Write"
            },
            "UpdateMLTransform": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform",
                "description": "Grants permission to update an ML Transform",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            },
            "UpdatePartition": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition",
                "description": "Grants permission to update a partition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTable": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable",
                "description": "Grants permission to update a table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "table": {
                        "required": true
                    }
                }
            },
            "UpdateTrigger": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger",
                "description": "Grants permission to update a trigger",
                "accessLevel": "Write"
            },
            "UpdateUserDefinedFunction": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction",
                "description": "Grants permission to update a function definition",
                "accessLevel": "Write",
                "resourceTypes": {
                    "catalog": {
                        "required": true
                    },
                    "database": {
                        "required": true
                    },
                    "userdefinedfunction": {
                        "required": true
                    }
                }
            },
            "UpdateWorkflow": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow",
                "description": "Grants permission to update a workflow",
                "accessLevel": "Write"
            },
            "UseMLTransforms": {
                "url": "https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html",
                "description": "Grants permission to use an ML Transform from within a Glue ETL Script",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mlTransform": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "catalog": {
                "name": "catalog",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:catalog",
                "conditionKeys": []
            },
            "database": {
                "name": "database",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}",
                "conditionKeys": []
            },
            "table": {
                "name": "table",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}",
                "conditionKeys": []
            },
            "tableversion": {
                "name": "tableversion",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}",
                "conditionKeys": []
            },
            "connection": {
                "name": "connection",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}",
                "conditionKeys": []
            },
            "userdefinedfunction": {
                "name": "userdefinedfunction",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}",
                "conditionKeys": []
            },
            "devendpoint": {
                "name": "devendpoint",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:job/${JobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "trigger": {
                "name": "trigger",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "crawler": {
                "name": "crawler",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "workflow": {
                "name": "workflow",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "mlTransform": {
                "name": "mlTransform",
                "url": "https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html",
                "arn": "arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition
     */
    batchCreatePartition() {
        this.add('glue:BatchCreatePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection
     */
    batchDeleteConnection() {
        this.add('glue:BatchDeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition
     */
    batchDeletePartition() {
        this.add('glue:BatchDeletePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more tables
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable
     */
    batchDeleteTable() {
        this.add('glue:BatchDeleteTable');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    batchDeleteTableVersion() {
        this.add('glue:BatchDeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve one or more crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers
     */
    batchGetCrawlers() {
        this.add('glue:BatchGetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints
     */
    batchGetDevEndpoints() {
        this.add('glue:BatchGetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve one or more jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs
     */
    batchGetJobs() {
        this.add('glue:BatchGetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve one or more partitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition
     */
    batchGetPartition() {
        this.add('glue:BatchGetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve one or more triggers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers
     */
    batchGetTriggers() {
        this.add('glue:BatchGetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more workflows
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows
     */
    batchGetWorkflows() {
        this.add('glue:BatchGetWorkflows');
        return this;
    }
    /**
     * Grants permission to stop one or more job runs for a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun
     */
    batchStopJobRun() {
        this.add('glue:BatchStopJobRun');
        return this;
    }
    /**
     * Grants permission to stop a running ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun
     */
    cancelMLTaskRun() {
        this.add('glue:CancelMLTaskRun');
        return this;
    }
    /**
     * Grants permission to create a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier
     */
    createClassifier() {
        this.add('glue:CreateClassifier');
        return this;
    }
    /**
     * Grants permission to create a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection
     */
    createConnection() {
        this.add('glue:CreateConnection');
        return this;
    }
    /**
     * Grants permission to create a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler
     */
    createCrawler() {
        this.add('glue:CreateCrawler');
        return this;
    }
    /**
     * Grants permission to create a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase
     */
    createDatabase() {
        this.add('glue:CreateDatabase');
        return this;
    }
    /**
     * Grants permission to create a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint
     */
    createDevEndpoint() {
        this.add('glue:CreateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob
     */
    createJob() {
        this.add('glue:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform
     */
    createMLTransform() {
        this.add('glue:CreateMLTransform');
        return this;
    }
    /**
     * Grants permission to create a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition
     */
    createPartition() {
        this.add('glue:CreatePartition');
        return this;
    }
    /**
     * Grants permission to create a script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript
     */
    createScript() {
        this.add('glue:CreateScript');
        return this;
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration
     */
    createSecurityConfiguration() {
        this.add('glue:CreateSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable
     */
    createTable() {
        this.add('glue:CreateTable');
        return this;
    }
    /**
     * Grants permission to create a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger
     */
    createTrigger() {
        this.add('glue:CreateTrigger');
        return this;
    }
    /**
     * Grants permission to create a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction
     */
    createUserDefinedFunction() {
        this.add('glue:CreateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to create a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow
     */
    createWorkflow() {
        this.add('glue:CreateWorkflow');
        return this;
    }
    /**
     * Grants permission to delete a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier
     */
    deleteClassifier() {
        this.add('glue:DeleteClassifier');
        return this;
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection
     */
    deleteConnection() {
        this.add('glue:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler
     */
    deleteCrawler() {
        this.add('glue:DeleteCrawler');
        return this;
    }
    /**
     * Grants permission to delete a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase
     */
    deleteDatabase() {
        this.add('glue:DeleteDatabase');
        return this;
    }
    /**
     * Grants permission to delete a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint
     */
    deleteDevEndpoint() {
        this.add('glue:DeleteDevEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob
     */
    deleteJob() {
        this.add('glue:DeleteJob');
        return this;
    }
    /**
     * Grants permission to delete an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform
     */
    deleteMLTransform() {
        this.add('glue:DeleteMLTransform');
        return this;
    }
    /**
     * Grants permission to delete a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition
     */
    deletePartition() {
        this.add('glue:DeletePartition');
        return this;
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy
     */
    deleteResourcePolicy() {
        this.add('glue:DeleteResourcePolicy');
        return this;
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration
     */
    deleteSecurityConfiguration() {
        this.add('glue:DeleteSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable
     */
    deleteTable() {
        this.add('glue:DeleteTable');
        return this;
    }
    /**
     * Grants permission to delete a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    deleteTableVersion() {
        this.add('glue:DeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to delete a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger
     */
    deleteTrigger() {
        this.add('glue:DeleteTrigger');
        return this;
    }
    /**
     * Grants permission to delete a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction
     */
    deleteUserDefinedFunction() {
        this.add('glue:DeleteUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow
     */
    deleteWorkflow() {
        this.add('glue:DeleteWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve the catalog import status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus
     */
    getCatalogImportStatus() {
        this.add('glue:GetCatalogImportStatus');
        return this;
    }
    /**
     * Grants permission to retrieve a classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier
     */
    getClassifier() {
        this.add('glue:GetClassifier');
        return this;
    }
    /**
     * Grants permission to list all classifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers
     */
    getClassifiers() {
        this.add('glue:GetClassifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection
     */
    getConnection() {
        this.add('glue:GetConnection');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections
     */
    getConnections() {
        this.add('glue:GetConnections');
        return this;
    }
    /**
     * Grants permission to retrieve a crawler
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler
     */
    getCrawler() {
        this.add('glue:GetCrawler');
        return this;
    }
    /**
     * Grants permission to retrieve metrics about crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics
     */
    getCrawlerMetrics() {
        this.add('glue:GetCrawlerMetrics');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers
     */
    getCrawlers() {
        this.add('glue:GetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve catalog encryption settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings
     */
    getDataCatalogEncryptionSettings() {
        this.add('glue:GetDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase
     */
    getDatabase() {
        this.add('glue:GetDatabase');
        return this;
    }
    /**
     * Grants permission to retrieve all databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases
     */
    getDatabases() {
        this.add('glue:GetDatabases');
        return this;
    }
    /**
     * Grants permission to transform a script into a directed acyclic graph (DAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph
     */
    getDataflowGraph() {
        this.add('glue:GetDataflowGraph');
        return this;
    }
    /**
     * Grants permission to retrieve a development endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint
     */
    getDevEndpoint() {
        this.add('glue:GetDevEndpoint');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints
     */
    getDevEndpoints() {
        this.add('glue:GetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob
     */
    getJob() {
        this.add('glue:GetJob');
        return this;
    }
    /**
     * Grants permission to retrieve a job bookmark
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark
     */
    getJobBookmark() {
        this.add('glue:GetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve a job run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun
     */
    getJobRun() {
        this.add('glue:GetJobRun');
        return this;
    }
    /**
     * Grants permission to retrieve all job runs of a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns
     */
    getJobRuns() {
        this.add('glue:GetJobRuns');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs
     */
    getJobs() {
        this.add('glue:GetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Task Run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun
     */
    getMLTaskRun() {
        this.add('glue:GetMLTaskRun');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Task Runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns
     */
    getMLTaskRuns() {
        this.add('glue:GetMLTaskRuns');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform
     */
    getMLTransform() {
        this.add('glue:GetMLTransform');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms
     */
    getMLTransforms() {
        this.add('glue:GetMLTransforms');
        return this;
    }
    /**
     * Grants permission to create a mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping
     */
    getMapping() {
        this.add('glue:GetMapping');
        return this;
    }
    /**
     * Grants permission to retrieve a partition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition
     */
    getPartition() {
        this.add('glue:GetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve the partitions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions
     */
    getPartitions() {
        this.add('glue:GetPartitions');
        return this;
    }
    /**
     * Grants permission to retrieve a mapping for a script
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan
     */
    getPlan() {
        this.add('glue:GetPlan');
        return this;
    }
    /**
     * Grants permission to retrieve a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy
     */
    getResourcePolicy() {
        this.add('glue:GetResourcePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration
     */
    getSecurityConfiguration() {
        this.add('glue:GetSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve one or more security configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations
     */
    getSecurityConfigurations() {
        this.add('glue:GetSecurityConfigurations');
        return this;
    }
    /**
     * Grants permission to retrieve a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable
     */
    getTable() {
        this.add('glue:GetTable');
        return this;
    }
    /**
     * Grants permission to retrieve a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion
     */
    getTableVersion() {
        this.add('glue:GetTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions
     */
    getTableVersions() {
        this.add('glue:GetTableVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables
     */
    getTables() {
        this.add('glue:GetTables');
        return this;
    }
    /**
     * Grants permission to retrieve all tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    getTags() {
        this.add('glue:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a trigger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger
     */
    getTrigger() {
        this.add('glue:GetTrigger');
        return this;
    }
    /**
     * Grants permission to retrieve the triggers associated with a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers
     */
    getTriggers() {
        this.add('glue:GetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve a function definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction
     */
    getUserDefinedFunction() {
        this.add('glue:GetUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to retrieve multiple function definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions
     */
    getUserDefinedFunctions() {
        this.add('glue:GetUserDefinedFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow
     */
    getWorkflow() {
        this.add('glue:GetWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun
     */
    getWorkflowRun() {
        this.add('glue:GetWorkflowRun');
        return this;
    }
    /**
     * Grants permission to retrieve workflow run properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties
     */
    getWorkflowRunProperties() {
        this.add('glue:GetWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to retrieve all runs of a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns
     */
    getWorkflowRuns() {
        this.add('glue:GetWorkflowRuns');
        return this;
    }
    /**
     * Grants permission to import an Athena data catalog into AWS Glue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue
     */
    importCatalogToGlue() {
        this.add('glue:ImportCatalogToGlue');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers
     */
    listCrawlers() {
        this.add('glue:ListCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints
     */
    listDevEndpoints() {
        this.add('glue:ListDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs
     */
    listJobs() {
        this.add('glue:ListJobs');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms
     */
    listMLTransforms() {
        this.add('glue:ListMLTransforms');
        return this;
    }
    /**
     * Grants permission to retrieve all triggers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers
     */
    listTriggers() {
        this.add('glue:ListTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve all workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows
     */
    listWorkflows() {
        this.add('glue:ListWorkflows');
        return this;
    }
    /**
     * Grants permission to update catalog encryption settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings
     */
    putDataCatalogEncryptionSettings() {
        this.add('glue:PutDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy
     */
    putResourcePolicy() {
        this.add('glue:PutResourcePolicy');
        return this;
    }
    /**
     * Grants permission to update workflow run properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties
     */
    putWorkflowRunProperties() {
        this.add('glue:PutWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to reset a job bookmark
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark
     */
    resetJobBookmark() {
        this.add('glue:ResetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in the catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables
     */
    searchTables() {
        this.add('glue:SearchTables');
        return this;
    }
    /**
     * Grants permission to start a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler
     */
    startCrawler() {
        this.add('glue:StartCrawler');
        return this;
    }
    /**
     * Grants permission to change the schedule state of a crawler to SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule
     */
    startCrawlerSchedule() {
        this.add('glue:StartCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to start an Export Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun
     */
    startExportLabelsTaskRun() {
        this.add('glue:StartExportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start an Import Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun
     */
    startImportLabelsTaskRun() {
        this.add('glue:StartImportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start running a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun
     */
    startJobRun() {
        this.add('glue:StartJobRun');
        return this;
    }
    /**
     * Grants permission to start an Evaluation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun
     */
    startMLEvaluationTaskRun() {
        this.add('glue:StartMLEvaluationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a Labeling Set Generation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun
     */
    startMLLabelingSetGenerationTaskRun() {
        this.add('glue:StartMLLabelingSetGenerationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger
     */
    startTrigger() {
        this.add('glue:StartTrigger');
        return this;
    }
    /**
     * Grants permission to start running a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun
     */
    startWorkflowRun() {
        this.add('glue:StartWorkflowRun');
        return this;
    }
    /**
     * Grants permission to stop a running crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler
     */
    stopCrawler() {
        this.add('glue:StopCrawler');
        return this;
    }
    /**
     * Grants permission to set the schedule state of a crawler to NOT_SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule
     */
    stopCrawlerSchedule() {
        this.add('glue:StopCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to stop a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger
     */
    stopTrigger() {
        this.add('glue:StopTrigger');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource
     */
    tagResource() {
        this.add('glue:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags associated with a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    untagResource() {
        this.add('glue:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier
     */
    updateClassifier() {
        this.add('glue:UpdateClassifier');
        return this;
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection
     */
    updateConnection() {
        this.add('glue:UpdateConnection');
        return this;
    }
    /**
     * Grants permission to update a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler
     */
    updateCrawler() {
        this.add('glue:UpdateCrawler');
        return this;
    }
    /**
     * Grants permission to update the schedule of a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule
     */
    updateCrawlerSchedule() {
        this.add('glue:UpdateCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to update a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase
     */
    updateDatabase() {
        this.add('glue:UpdateDatabase');
        return this;
    }
    /**
     * Grants permission to update a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint
     */
    updateDevEndpoint() {
        this.add('glue:UpdateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob
     */
    updateJob() {
        this.add('glue:UpdateJob');
        return this;
    }
    /**
     * Grants permission to update an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform
     */
    updateMLTransform() {
        this.add('glue:UpdateMLTransform');
        return this;
    }
    /**
     * Grants permission to update a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition
     */
    updatePartition() {
        this.add('glue:UpdatePartition');
        return this;
    }
    /**
     * Grants permission to update a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable
     */
    updateTable() {
        this.add('glue:UpdateTable');
        return this;
    }
    /**
     * Grants permission to update a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger
     */
    updateTrigger() {
        this.add('glue:UpdateTrigger');
        return this;
    }
    /**
     * Grants permission to update a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction
     */
    updateUserDefinedFunction() {
        this.add('glue:UpdateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to update a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow
     */
    updateWorkflow() {
        this.add('glue:UpdateWorkflow');
        return this;
    }
    /**
     * Grants permission to use an ML Transform from within a Glue ETL Script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html
     */
    useMLTransforms() {
        this.add('glue:UseMLTransforms');
        return this;
    }
    /**
     * Adds a resource of type catalog to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCatalog(account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:catalog';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatabase(databaseName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(databaseName, tableName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tableversion to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param tableVersionName - Identifier for the tableVersionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTableversion(databaseName, tableName, tableVersionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${TableVersionName}', tableVersionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param connectionName - Identifier for the connectionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}';
        arn = arn.replace('${ConnectionName}', connectionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type userdefinedfunction to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param userDefinedFunctionName - Identifier for the userDefinedFunctionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUserdefinedfunction(databaseName, userDefinedFunctionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${UserDefinedFunctionName}', userDefinedFunctionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type devendpoint to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param devEndpointName - Identifier for the devEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDevendpoint(devEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}';
        arn = arn.replace('${DevEndpointName}', devEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onJob(jobName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:job/${JobName}';
        arn = arn.replace('${JobName}', jobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trigger to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param triggerName - Identifier for the triggerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTrigger(triggerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}';
        arn = arn.replace('${TriggerName}', triggerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type crawler to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param crawlerName - Identifier for the crawlerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCrawler(crawlerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}';
        arn = arn.replace('${CrawlerName}', crawlerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onWorkflow(workflowName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}';
        arn = arn.replace('${WorkflowName}', workflowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mlTransform to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param transformId - Identifier for the transformId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMlTransform(transformId, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}';
        arn = arn.replace('${TransformId}', transformId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Glue = Glue;
//# sourceMappingURL=data:application/json;base64,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