"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsMarketplace = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplace.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AwsMarketplace extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplace.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'aws-marketplace';
        this.actionList = {
            "AcceptAgreementApprovalRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to approve an incoming subscription request (for providers who provide products that require subscription verification).",
                "accessLevel": "Write"
            },
            "CancelAgreementRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to cancel pending subscription requests for products that require subscription verification.",
                "accessLevel": "Write"
            },
            "DescribeAgreement": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Returns metadata about the agreement.",
                "accessLevel": "Read"
            },
            "GetAgreementApprovalRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to view the details of their incoming subscription requests (for providers who provide products that require subscription verification).",
                "accessLevel": "Read"
            },
            "GetAgreementRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to view the details of their subscription requests for data products that require subscription verification.",
                "accessLevel": "Read"
            },
            "GetAgreementTerms": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Returns a list of terms for an agreement.",
                "accessLevel": "List"
            },
            "ListAgreementApprovalRequests": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to list their incoming subscription requests (for providers who provide products that require subscription verification).",
                "accessLevel": "List"
            },
            "ListAgreementRequests": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to list their subscription requests for products that require subscription verification.",
                "accessLevel": "List"
            },
            "RejectAgreementApprovalRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to decline an incoming subscription requests (for providers who provide products that require subscription verification).",
                "accessLevel": "Write"
            },
            "SearchAgreements": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to search their agreements.",
                "accessLevel": "List"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to subscribe to AWS Marketplace products. Includes the ability to send a subscription request for products that require subscription verification. Includes the ability to enable auto-renewal for an existing subscription.",
                "accessLevel": "Write"
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to remove subscriptions to AWS Marketplace products. Includes the ability to disable auto-renewal for an existing subscription.",
                "accessLevel": "Write"
            },
            "UpdateAgreementApprovalRequest": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to make changes to an incoming subscription request, including the ability to delete the prospective subscriber's information (for providers who provide products that require subscription verification).",
                "accessLevel": "Write"
            },
            "ViewSubscriptions": {
                "url": "https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions",
                "description": "Allows users to see their account's subscriptions.",
                "accessLevel": "List"
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Allows users to approve an incoming subscription request (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    acceptAgreementApprovalRequest() {
        this.add('aws-marketplace:AcceptAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to cancel pending subscription requests for products that require subscription verification.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    cancelAgreementRequest() {
        this.add('aws-marketplace:CancelAgreementRequest');
        return this;
    }
    /**
     * Returns metadata about the agreement.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    describeAgreement() {
        this.add('aws-marketplace:DescribeAgreement');
        return this;
    }
    /**
     * Allows users to view the details of their incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    getAgreementApprovalRequest() {
        this.add('aws-marketplace:GetAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to view the details of their subscription requests for data products that require subscription verification.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    getAgreementRequest() {
        this.add('aws-marketplace:GetAgreementRequest');
        return this;
    }
    /**
     * Returns a list of terms for an agreement.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    getAgreementTerms() {
        this.add('aws-marketplace:GetAgreementTerms');
        return this;
    }
    /**
     * Allows users to list their incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    listAgreementApprovalRequests() {
        this.add('aws-marketplace:ListAgreementApprovalRequests');
        return this;
    }
    /**
     * Allows users to list their subscription requests for products that require subscription verification.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    listAgreementRequests() {
        this.add('aws-marketplace:ListAgreementRequests');
        return this;
    }
    /**
     * Allows users to decline an incoming subscription requests (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    rejectAgreementApprovalRequest() {
        this.add('aws-marketplace:RejectAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to search their agreements.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    searchAgreements() {
        this.add('aws-marketplace:SearchAgreements');
        return this;
    }
    /**
     * Allows users to subscribe to AWS Marketplace products. Includes the ability to send a subscription request for products that require subscription verification. Includes the ability to enable auto-renewal for an existing subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    subscribe() {
        this.add('aws-marketplace:Subscribe');
        return this;
    }
    /**
     * Allows users to remove subscriptions to AWS Marketplace products. Includes the ability to disable auto-renewal for an existing subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    unsubscribe() {
        this.add('aws-marketplace:Unsubscribe');
        return this;
    }
    /**
     * Allows users to make changes to an incoming subscription request, including the ability to delete the prospective subscriber's information (for providers who provide products that require subscription verification).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    updateAgreementApprovalRequest() {
        this.add('aws-marketplace:UpdateAgreementApprovalRequest');
        return this;
    }
    /**
     * Allows users to see their account's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/controlling-access/ControllingAccessToAWSMarketplaceSubscriptions.html#SummaryOfAWSMarketplaceSubscriptionsPermissions
     */
    viewSubscriptions() {
        this.add('aws-marketplace:ViewSubscriptions');
        return this;
    }
    /**
     * Enables you to control access based on the type of the agreement.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAgreementType(value, operator) {
        return this.if(`aws-marketplace:AgreementType`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the party type of the agreement.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPartyType(value, operator) {
        return this.if(`aws-marketplace:PartyType`, value, operator || 'StringLike');
    }
}
exports.AwsMarketplace = AwsMarketplace;
//# sourceMappingURL=data:application/json;base64,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