"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsMarketplaceCatalog = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [aws-marketplace-catalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplacecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AwsMarketplaceCatalog extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [aws-marketplace-catalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplacecatalog.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'aws-marketplace-catalog';
        this.actionList = {
            "CancelChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_CancelChangeSet.html",
                "description": "Cancels a running change set.",
                "accessLevel": "Write"
            },
            "CompleteTask": {
                "url": "",
                "description": "Complete an existing task and submit the content to the associated change.",
                "accessLevel": "Write"
            },
            "DescribeChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeChangeSet.html",
                "description": "Returns the details of an existing change set.",
                "accessLevel": "Read"
            },
            "DescribeEntity": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeEntity.html",
                "description": "Returns the details of an existing entity.",
                "accessLevel": "Read"
            },
            "DescribeTask": {
                "url": "",
                "description": "Returns the details of an existing task.",
                "accessLevel": "Read"
            },
            "ListChangeSets": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListChangeSets.html",
                "description": "Lists existing change sets.",
                "accessLevel": "Read"
            },
            "ListEntities": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListEntities.html",
                "description": "Lists existing entities.",
                "accessLevel": "Read"
            },
            "ListTasks": {
                "url": "",
                "description": "Lists existing tasks.",
                "accessLevel": "List"
            },
            "StartChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_StartChangeSet.html",
                "description": "Requests a new change set.",
                "accessLevel": "Write",
                "conditions": [
                    "catalog:ChangeType"
                ]
            },
            "UpdateTask": {
                "url": "",
                "description": "Update the content of an existing task.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Entity": {
                "name": "Entity",
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_DescribeEntity.html#API_DescribeEntity_ResponseSyntax",
                "arn": "arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/${EntityType}/${ResourceId}",
                "conditionKeys": []
            },
            "ChangeSet": {
                "name": "ChangeSet",
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_ResponseSyntax",
                "arn": "arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/ChangeSet/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels a running change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_CancelChangeSet.html
     */
    cancelChangeSet() {
        this.add('aws-marketplace-catalog:CancelChangeSet');
        return this;
    }
    /**
     * Complete an existing task and submit the content to the associated change.
     *
     * Access Level: Write
     */
    completeTask() {
        this.add('aws-marketplace-catalog:CompleteTask');
        return this;
    }
    /**
     * Returns the details of an existing change set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeChangeSet.html
     */
    describeChangeSet() {
        this.add('aws-marketplace-catalog:DescribeChangeSet');
        return this;
    }
    /**
     * Returns the details of an existing entity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeEntity.html
     */
    describeEntity() {
        this.add('aws-marketplace-catalog:DescribeEntity');
        return this;
    }
    /**
     * Returns the details of an existing task.
     *
     * Access Level: Read
     */
    describeTask() {
        this.add('aws-marketplace-catalog:DescribeTask');
        return this;
    }
    /**
     * Lists existing change sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListChangeSets.html
     */
    listChangeSets() {
        this.add('aws-marketplace-catalog:ListChangeSets');
        return this;
    }
    /**
     * Lists existing entities.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListEntities.html
     */
    listEntities() {
        this.add('aws-marketplace-catalog:ListEntities');
        return this;
    }
    /**
     * Lists existing tasks.
     *
     * Access Level: List
     */
    listTasks() {
        this.add('aws-marketplace-catalog:ListTasks');
        return this;
    }
    /**
     * Requests a new change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_StartChangeSet.html
     */
    startChangeSet() {
        this.add('aws-marketplace-catalog:StartChangeSet');
        return this;
    }
    /**
     * Update the content of an existing task.
     *
     * Access Level: Write
     */
    updateTask() {
        this.add('aws-marketplace-catalog:UpdateTask');
        return this;
    }
    /**
     * Adds a resource of type Entity to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_DescribeEntity.html#API_DescribeEntity_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param entityType - Identifier for the entityType.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEntity(catalog, entityType, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/${EntityType}/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${EntityType}', entityType);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ChangeSet to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChangeSet(catalog, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/ChangeSet/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to verify change type in the StartChangeSet request.
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/api-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeType(value, operator) {
        return this.if(`catalog:ChangeType`, value, operator || 'StringLike');
    }
}
exports.AwsMarketplaceCatalog = AwsMarketplaceCatalog;
//# sourceMappingURL=data:application/json;base64,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