"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ram = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ram extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ram';
        this.actionList = {
            "AcceptResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html",
                "description": "Accept the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "AssociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html",
                "description": "Associates resource(s) and/or principal(s) to a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "AssociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html",
                "description": "Associate a Permission with a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "CreateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html",
                "description": "Create resource share with provided resource(s) and/or principal(s)",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn",
                    "ram:RequestedAllowsExternalPrincipals",
                    "ram:Principal"
                ]
            },
            "DeleteResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html",
                "description": "Delete resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "DisassociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html",
                "description": "Disassociates resource(s) and/or principal(s) from a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "DisassociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html",
                "description": "Disassociate a Permission from a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "EnableSharingWithAwsOrganization": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html",
                "description": "Grants permission to access customer's organization and create a SLR in the customer's account",
                "accessLevel": "Write"
            },
            "GetPermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html",
                "description": "Gets the contents of an AWS RAM permission",
                "accessLevel": "Read",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:PermissionArn"
                ]
            },
            "GetResourcePolicies": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html",
                "description": "Gets the policies for the specified resources that you own and have shared",
                "accessLevel": "Read"
            },
            "GetResourceShareAssociations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html",
                "description": "Get a set of resource share associations from a provided list or with a specified status of the specified type",
                "accessLevel": "Read"
            },
            "GetResourceShareInvitations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html",
                "description": "Get resource share invitations by the specified invitation arn or those for the resource share",
                "accessLevel": "Read"
            },
            "GetResourceShares": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html",
                "description": "Get a set of resource shares from a provided list or with a specified status",
                "accessLevel": "Read"
            },
            "ListPendingInvitationResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html",
                "description": "Lists the resources in a resource share that is shared with you but that the invitation is still pending for",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                }
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the AWS RAM permissions",
                "accessLevel": "List"
            },
            "ListPrincipals": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html",
                "description": "Lists the principals that you have shared resources with or that have shared resources with you",
                "accessLevel": "List"
            },
            "ListResourceSharePermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html",
                "description": "List the Permissions associated with a Resource Share",
                "accessLevel": "List",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "ListResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html",
                "description": "Lists the resources that you added to a resource shares or the resources that are shared with you",
                "accessLevel": "List"
            },
            "RejectResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html",
                "description": "Reject the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html",
                "description": "Tag the specified resources share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html",
                "description": "Untag the specified resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html",
                "description": "Update attributes of the resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:RequestedAllowsExternalPrincipals"
                ]
            }
        };
        this.resourceTypes = {
            "resource-share": {
                "name": "resource-share",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName"
                ]
            },
            "resource-share-invitation": {
                "name": "resource-share-invitation",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}",
                "conditionKeys": []
            },
            "permission": {
                "name": "permission",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html",
                "arn": "arn:${Partition}:ram::${Account}:permission/${ResourcePath}",
                "conditionKeys": [
                    "ram:PermissionArn"
                ]
            }
        };
    }
    /**
     * Accept the specified resource share invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html
     */
    acceptResourceShareInvitation() {
        this.add('ram:AcceptResourceShareInvitation');
        return this;
    }
    /**
     * Associates resource(s) and/or principal(s) to a resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html
     */
    associateResourceShare() {
        this.add('ram:AssociateResourceShare');
        return this;
    }
    /**
     * Associate a Permission with a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html
     */
    associateResourceSharePermission() {
        this.add('ram:AssociateResourceSharePermission');
        return this;
    }
    /**
     * Create resource share with provided resource(s) and/or principal(s)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html
     */
    createResourceShare() {
        this.add('ram:CreateResourceShare');
        return this;
    }
    /**
     * Delete resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html
     */
    deleteResourceShare() {
        this.add('ram:DeleteResourceShare');
        return this;
    }
    /**
     * Disassociates resource(s) and/or principal(s) from a resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html
     */
    disassociateResourceShare() {
        this.add('ram:DisassociateResourceShare');
        return this;
    }
    /**
     * Disassociate a Permission from a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html
     */
    disassociateResourceSharePermission() {
        this.add('ram:DisassociateResourceSharePermission');
        return this;
    }
    /**
     * Grants permission to access customer's organization and create a SLR in the customer's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html
     */
    enableSharingWithAwsOrganization() {
        this.add('ram:EnableSharingWithAwsOrganization');
        return this;
    }
    /**
     * Gets the contents of an AWS RAM permission
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html
     */
    getPermission() {
        this.add('ram:GetPermission');
        return this;
    }
    /**
     * Gets the policies for the specified resources that you own and have shared
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html
     */
    getResourcePolicies() {
        this.add('ram:GetResourcePolicies');
        return this;
    }
    /**
     * Get a set of resource share associations from a provided list or with a specified status of the specified type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html
     */
    getResourceShareAssociations() {
        this.add('ram:GetResourceShareAssociations');
        return this;
    }
    /**
     * Get resource share invitations by the specified invitation arn or those for the resource share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html
     */
    getResourceShareInvitations() {
        this.add('ram:GetResourceShareInvitations');
        return this;
    }
    /**
     * Get a set of resource shares from a provided list or with a specified status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html
     */
    getResourceShares() {
        this.add('ram:GetResourceShares');
        return this;
    }
    /**
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html
     */
    listPendingInvitationResources() {
        this.add('ram:ListPendingInvitationResources');
        return this;
    }
    /**
     * Lists the AWS RAM permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html
     */
    listPermissions() {
        this.add('ram:ListPermissions');
        return this;
    }
    /**
     * Lists the principals that you have shared resources with or that have shared resources with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html
     */
    listPrincipals() {
        this.add('ram:ListPrincipals');
        return this;
    }
    /**
     * List the Permissions associated with a Resource Share
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html
     */
    listResourceSharePermissions() {
        this.add('ram:ListResourceSharePermissions');
        return this;
    }
    /**
     * Lists the resources that you added to a resource shares or the resources that are shared with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html
     */
    listResources() {
        this.add('ram:ListResources');
        return this;
    }
    /**
     * Reject the specified resource share invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html
     */
    rejectResourceShareInvitation() {
        this.add('ram:RejectResourceShareInvitation');
        return this;
    }
    /**
     * Tag the specified resources share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('ram:TagResource');
        return this;
    }
    /**
     * Untag the specified resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('ram:UntagResource');
        return this;
    }
    /**
     * Update attributes of the resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html
     */
    updateResourceShare() {
        this.add('ram:UpdateResourceShare');
        return this;
    }
    /**
     * Adds a resource of type resource-share to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ram:AllowsExternalPrincipals
     *  - ram:ResourceShareName
     */
    onResourceShare(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resource-share-invitation to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceShareInvitation(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ram:PermissionArn
     */
    onPermission(resourcePath, account, partition) {
        var arn = 'arn:${Partition}:ram::${Account}:permission/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Indicates that the action can only be performed on resource shares that allow or deny sharing with external principals. For example, specify true if the action can only be performed on resource shares that allow sharing with external principals. External principals are AWS accounts that are outside of its AWS organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowsExternalPrincipals(value) {
        return this.if(`ram:AllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on a resource using the specified Permission ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPermissionArn(value, operator) {
        return this.if(`ram:PermissionArn`, value, operator || 'ArnEquals');
    }
    /**
     * Principals with the specified format can be associated to or disassociated from a resource share
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`ram:Principal`, value, operator || 'StringLike');
    }
    /**
     * The request must have the specified value for 'allowExternalPrincipals'. External principals are AWS accounts that are outside of its AWS Organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestedAllowsExternalPrincipals(value) {
        return this.if(`ram:RequestedAllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on the specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedResourceType(value, operator) {
        return this.if(`ram:RequestedResourceType`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on a resource with the specified ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResourceArn(value, operator) {
        return this.if(`ram:ResourceArn`, value, operator || 'ArnEquals');
    }
    /**
     * Indicates that the action can only be performed on a resource share with the specified name.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceShareName(value, operator) {
        return this.if(`ram:ResourceShareName`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resource shares owned by a specific account. For example, you can use this condition key to specify which resource share invitations can be accepted or rejected based on the resource share owner’s account ID.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareOwnerAccountId(value, operator) {
        return this.if(`ram:ShareOwnerAccountId`, value, operator || 'StringLike');
    }
}
exports.Ram = Ram;
//# sourceMappingURL=data:application/json;base64,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