"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Robomaker = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Robomaker extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'robomaker';
        this.actionList = {
            "BatchDescribeSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html",
                "description": "Describe multiple simulation jobs",
                "accessLevel": "Read"
            },
            "CancelDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html",
                "description": "Cancel a deployment job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentJob": {
                        "required": true
                    }
                }
            },
            "CancelSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html",
                "description": "Cancel a simulation job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "CancelSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html",
                "description": "Cancel a simulation job batch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJobBatch": {
                        "required": true
                    }
                }
            },
            "CreateDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html",
                "description": "Create a deployment job",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html",
                "description": "Create a deployment fleet that represents a logical group of robots running the same robot application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html",
                "description": "Create a robot that can be registered to a fleet",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html",
                "description": "Create a robot application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobotApplicationVersion": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html",
                "description": "Create a snapshot of a robot application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "CreateSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html",
                "description": "Create a simulation application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateSimulationApplicationVersion": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html",
                "description": "Create a snapshot of a simulation application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "CreateSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html",
                "description": "Create a simulation job",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html",
                "description": "Delete a deployment fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "DeleteRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html",
                "description": "Delete a robot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robot": {
                        "required": true
                    }
                }
            },
            "DeleteRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html",
                "description": "Delete a robot application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "DeleteSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html",
                "description": "Delete a simulation application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "DeregisterRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html",
                "description": "Deregister a robot from a fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    },
                    "robot": {
                        "required": true
                    }
                }
            },
            "DescribeDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html",
                "description": "Describe a deployment job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentJob": {
                        "required": true
                    }
                }
            },
            "DescribeFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html",
                "description": "Describe a deployment fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "DescribeRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html",
                "description": "Describe a robot",
                "accessLevel": "Read",
                "resourceTypes": {
                    "robot": {
                        "required": true
                    }
                }
            },
            "DescribeRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html",
                "description": "Describe a robot application",
                "accessLevel": "Read",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html",
                "description": "Describe a simulation application",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html",
                "description": "Describe a simulation job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html",
                "description": "Describe a simulation job batch",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationJobBatch": {
                        "required": true
                    }
                }
            },
            "ListDeploymentJobs": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html",
                "description": "List deployment jobs",
                "accessLevel": "List"
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html",
                "description": "List fleets",
                "accessLevel": "List"
            },
            "ListRobotApplications": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html",
                "description": "List robot applications",
                "accessLevel": "List"
            },
            "ListRobots": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html",
                "description": "List robots",
                "accessLevel": "List"
            },
            "ListSimulationApplications": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html",
                "description": "List simulation applications",
                "accessLevel": "List"
            },
            "ListSimulationJobBatches": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html",
                "description": "List simulation job batches",
                "accessLevel": "List"
            },
            "ListSimulationJobs": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html",
                "description": "List simulation jobs",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html",
                "description": "List tags for a RoboMaker resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                }
            },
            "RegisterRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html",
                "description": "Register a robot to a fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    },
                    "robot": {
                        "required": true
                    }
                }
            },
            "RestartSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html",
                "description": "Restart a running simulation job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "StartSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html",
                "description": "Create a simulation job batch",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "SyncDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html",
                "description": "Ensures the most recently deployed robot application is deployed to all robots in the fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html",
                "description": "Add tags to a RoboMaker resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html",
                "description": "Remove tags from a RoboMaker resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html",
                "description": "Update a robot application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "UpdateSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html",
                "description": "Update a simulation application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "robotApplication": {
                "name": "robotApplication",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:robot-application/${ApplicationName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationApplication": {
                "name": "simulationApplication",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-application/${ApplicationName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationJob": {
                "name": "simulationJob",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-job/${SimulationJobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationJobBatch": {
                "name": "simulationJobBatch",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-job-batch/${SimulationJobBatchId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deploymentJob": {
                "name": "deploymentJob",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:deployment-job/${DeploymentJobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "robot": {
                "name": "robot",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:robot/${RobotName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deploymentFleet": {
                "name": "deploymentFleet",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:deployment-fleet/${FleetName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Describe multiple simulation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html
     */
    batchDescribeSimulationJob() {
        this.add('robomaker:BatchDescribeSimulationJob');
        return this;
    }
    /**
     * Cancel a deployment job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html
     */
    cancelDeploymentJob() {
        this.add('robomaker:CancelDeploymentJob');
        return this;
    }
    /**
     * Cancel a simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html
     */
    cancelSimulationJob() {
        this.add('robomaker:CancelSimulationJob');
        return this;
    }
    /**
     * Cancel a simulation job batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html
     */
    cancelSimulationJobBatch() {
        this.add('robomaker:CancelSimulationJobBatch');
        return this;
    }
    /**
     * Create a deployment job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html
     */
    createDeploymentJob() {
        this.add('robomaker:CreateDeploymentJob');
        return this;
    }
    /**
     * Create a deployment fleet that represents a logical group of robots running the same robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html
     */
    createFleet() {
        this.add('robomaker:CreateFleet');
        return this;
    }
    /**
     * Create a robot that can be registered to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html
     */
    createRobot() {
        this.add('robomaker:CreateRobot');
        return this;
    }
    /**
     * Create a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html
     */
    createRobotApplication() {
        this.add('robomaker:CreateRobotApplication');
        return this;
    }
    /**
     * Create a snapshot of a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html
     */
    createRobotApplicationVersion() {
        this.add('robomaker:CreateRobotApplicationVersion');
        return this;
    }
    /**
     * Create a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html
     */
    createSimulationApplication() {
        this.add('robomaker:CreateSimulationApplication');
        return this;
    }
    /**
     * Create a snapshot of a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html
     */
    createSimulationApplicationVersion() {
        this.add('robomaker:CreateSimulationApplicationVersion');
        return this;
    }
    /**
     * Create a simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html
     */
    createSimulationJob() {
        this.add('robomaker:CreateSimulationJob');
        return this;
    }
    /**
     * Delete a deployment fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('robomaker:DeleteFleet');
        return this;
    }
    /**
     * Delete a robot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html
     */
    deleteRobot() {
        this.add('robomaker:DeleteRobot');
        return this;
    }
    /**
     * Delete a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html
     */
    deleteRobotApplication() {
        this.add('robomaker:DeleteRobotApplication');
        return this;
    }
    /**
     * Delete a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html
     */
    deleteSimulationApplication() {
        this.add('robomaker:DeleteSimulationApplication');
        return this;
    }
    /**
     * Deregister a robot from a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html
     */
    deregisterRobot() {
        this.add('robomaker:DeregisterRobot');
        return this;
    }
    /**
     * Describe a deployment job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html
     */
    describeDeploymentJob() {
        this.add('robomaker:DescribeDeploymentJob');
        return this;
    }
    /**
     * Describe a deployment fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html
     */
    describeFleet() {
        this.add('robomaker:DescribeFleet');
        return this;
    }
    /**
     * Describe a robot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html
     */
    describeRobot() {
        this.add('robomaker:DescribeRobot');
        return this;
    }
    /**
     * Describe a robot application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html
     */
    describeRobotApplication() {
        this.add('robomaker:DescribeRobotApplication');
        return this;
    }
    /**
     * Describe a simulation application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html
     */
    describeSimulationApplication() {
        this.add('robomaker:DescribeSimulationApplication');
        return this;
    }
    /**
     * Describe a simulation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html
     */
    describeSimulationJob() {
        this.add('robomaker:DescribeSimulationJob');
        return this;
    }
    /**
     * Describe a simulation job batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html
     */
    describeSimulationJobBatch() {
        this.add('robomaker:DescribeSimulationJobBatch');
        return this;
    }
    /**
     * List deployment jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html
     */
    listDeploymentJobs() {
        this.add('robomaker:ListDeploymentJobs');
        return this;
    }
    /**
     * List fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html
     */
    listFleets() {
        this.add('robomaker:ListFleets');
        return this;
    }
    /**
     * List robot applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html
     */
    listRobotApplications() {
        this.add('robomaker:ListRobotApplications');
        return this;
    }
    /**
     * List robots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html
     */
    listRobots() {
        this.add('robomaker:ListRobots');
        return this;
    }
    /**
     * List simulation applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html
     */
    listSimulationApplications() {
        this.add('robomaker:ListSimulationApplications');
        return this;
    }
    /**
     * List simulation job batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html
     */
    listSimulationJobBatches() {
        this.add('robomaker:ListSimulationJobBatches');
        return this;
    }
    /**
     * List simulation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html
     */
    listSimulationJobs() {
        this.add('robomaker:ListSimulationJobs');
        return this;
    }
    /**
     * List tags for a RoboMaker resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('robomaker:ListTagsForResource');
        return this;
    }
    /**
     * Register a robot to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html
     */
    registerRobot() {
        this.add('robomaker:RegisterRobot');
        return this;
    }
    /**
     * Restart a running simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html
     */
    restartSimulationJob() {
        this.add('robomaker:RestartSimulationJob');
        return this;
    }
    /**
     * Create a simulation job batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html
     */
    startSimulationJobBatch() {
        this.add('robomaker:StartSimulationJobBatch');
        return this;
    }
    /**
     * Ensures the most recently deployed robot application is deployed to all robots in the fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html
     */
    syncDeploymentJob() {
        this.add('robomaker:SyncDeploymentJob');
        return this;
    }
    /**
     * Add tags to a RoboMaker resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('robomaker:TagResource');
        return this;
    }
    /**
     * Remove tags from a RoboMaker resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('robomaker:UntagResource');
        return this;
    }
    /**
     * Update a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html
     */
    updateRobotApplication() {
        this.add('robomaker:UpdateRobotApplication');
        return this;
    }
    /**
     * Update a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html
     */
    updateSimulationApplication() {
        this.add('robomaker:UpdateSimulationApplication');
        return this;
    }
    /**
     * Adds a resource of type robotApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRobotApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSimulationApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html
     *
     * @param simulationJobId - Identifier for the simulationJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSimulationJob(simulationJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job/${SimulationJobId}';
        arn = arn.replace('${SimulationJobId}', simulationJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJobBatch to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html
     *
     * @param simulationJobBatchId - Identifier for the simulationJobBatchId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSimulationJobBatch(simulationJobBatchId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job-batch/${SimulationJobBatchId}';
        arn = arn.replace('${SimulationJobBatchId}', simulationJobBatchId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html
     *
     * @param deploymentJobId - Identifier for the deploymentJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDeploymentJob(deploymentJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-job/${DeploymentJobId}';
        arn = arn.replace('${DeploymentJobId}', deploymentJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type robot to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html
     *
     * @param robotName - Identifier for the robotName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRobot(robotName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot/${RobotName}/${CreatedOnEpoch}';
        arn = arn.replace('${RobotName}', robotName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentFleet to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param fleetName - Identifier for the fleetName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDeploymentFleet(fleetName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-fleet/${FleetName}/${CreatedOnEpoch}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Robomaker = Robomaker;
//# sourceMappingURL=data:application/json;base64,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