"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Shield = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [shield](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsshield.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Shield extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [shield](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsshield.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'shield';
        this.actionList = {
            "AssociateDRTLogBucket": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTLogBucket.html",
                "description": "Authorizes the DDoS Response team to access the specified Amazon S3 bucket containing your flow logs",
                "accessLevel": "Write"
            },
            "AssociateDRTRole": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTRole.html",
                "description": "Authorizes the DDoS Response team using the specified role, to access your AWS account to assist with DDoS attack mitigation during potential attacks",
                "accessLevel": "Write"
            },
            "CreateProtection": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateProtection.html",
                "description": "Activate DDoS protection service for a given resource ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "protection": {
                        "required": true
                    }
                }
            },
            "CreateSubscription": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html",
                "description": "Activate subscription",
                "accessLevel": "Write"
            },
            "DeleteProtection": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteProtection.html",
                "description": "Delete an existing protection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "protection": {
                        "required": true
                    }
                }
            },
            "DeleteSubscription": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteSubscription.html",
                "description": "Deactivate subscription",
                "accessLevel": "Write"
            },
            "DescribeAttack": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeAttack.html",
                "description": "Get attack details",
                "accessLevel": "Read",
                "resourceTypes": {
                    "attack": {
                        "required": true
                    }
                }
            },
            "DescribeDRTAccess": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeDRTAccess.html",
                "description": "Returns the current role and list of Amazon S3 log buckets used by the DDoS Response team to access your AWS account while assisting with attack mitigation",
                "accessLevel": "Read"
            },
            "DescribeEmergencyContactSettings": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeEmergencyContactSettings.html",
                "description": "Lists the email addresses that the DRT can use to contact you during a suspected attack",
                "accessLevel": "Read"
            },
            "DescribeProtection": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeProtection.html",
                "description": "Get protection details",
                "accessLevel": "Read",
                "resourceTypes": {
                    "protection": {
                        "required": true
                    }
                }
            },
            "DescribeSubscription": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeSubscription.html",
                "description": "Get subscription details, such as start time",
                "accessLevel": "Read"
            },
            "DisassociateDRTLogBucket": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTLogBucket.html",
                "description": "Removes the DDoS Response team's access to the specified Amazon S3 bucket containing your flow logs",
                "accessLevel": "Write"
            },
            "DisassociateDRTRole": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTRole.html",
                "description": "Removes the DDoS Response team's access to your AWS account",
                "accessLevel": "Write"
            },
            "GetSubscriptionState": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_GetSubscriptionState.html",
                "description": "Get subscription state",
                "accessLevel": "Read"
            },
            "ListAttacks": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListAttacks.html",
                "description": "List all existing attacks",
                "accessLevel": "List"
            },
            "ListProtections": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListProtections.html",
                "description": "List all existing protections",
                "accessLevel": "List"
            },
            "UpdateEmergencyContactSettings": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateEmergencyContactSettings.html",
                "description": "Updates the details of the list of email addresses that the DRT can use to contact you during a suspected attack",
                "accessLevel": "Write"
            },
            "UpdateSubscription": {
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateSubscription.html",
                "description": "Updates the details of an existing subscription",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "attack": {
                "name": "attack",
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AttackDetail.html",
                "arn": "arn:${Partition}:shield::${Account}:attack/${Id}",
                "conditionKeys": []
            },
            "protection": {
                "name": "protection",
                "url": "https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_Protection.html",
                "arn": "arn:${Partition}:shield::${Account}:protection/${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Authorizes the DDoS Response team to access the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTLogBucket.html
     */
    associateDRTLogBucket() {
        this.add('shield:AssociateDRTLogBucket');
        return this;
    }
    /**
     * Authorizes the DDoS Response team using the specified role, to access your AWS account to assist with DDoS attack mitigation during potential attacks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTRole.html
     */
    associateDRTRole() {
        this.add('shield:AssociateDRTRole');
        return this;
    }
    /**
     * Activate DDoS protection service for a given resource ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateProtection.html
     */
    createProtection() {
        this.add('shield:CreateProtection');
        return this;
    }
    /**
     * Activate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html
     */
    createSubscription() {
        this.add('shield:CreateSubscription');
        return this;
    }
    /**
     * Delete an existing protection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteProtection.html
     */
    deleteProtection() {
        this.add('shield:DeleteProtection');
        return this;
    }
    /**
     * Deactivate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteSubscription.html
     */
    deleteSubscription() {
        this.add('shield:DeleteSubscription');
        return this;
    }
    /**
     * Get attack details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeAttack.html
     */
    describeAttack() {
        this.add('shield:DescribeAttack');
        return this;
    }
    /**
     * Returns the current role and list of Amazon S3 log buckets used by the DDoS Response team to access your AWS account while assisting with attack mitigation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeDRTAccess.html
     */
    describeDRTAccess() {
        this.add('shield:DescribeDRTAccess');
        return this;
    }
    /**
     * Lists the email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeEmergencyContactSettings.html
     */
    describeEmergencyContactSettings() {
        this.add('shield:DescribeEmergencyContactSettings');
        return this;
    }
    /**
     * Get protection details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeProtection.html
     */
    describeProtection() {
        this.add('shield:DescribeProtection');
        return this;
    }
    /**
     * Get subscription details, such as start time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeSubscription.html
     */
    describeSubscription() {
        this.add('shield:DescribeSubscription');
        return this;
    }
    /**
     * Removes the DDoS Response team's access to the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTLogBucket.html
     */
    disassociateDRTLogBucket() {
        this.add('shield:DisassociateDRTLogBucket');
        return this;
    }
    /**
     * Removes the DDoS Response team's access to your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTRole.html
     */
    disassociateDRTRole() {
        this.add('shield:DisassociateDRTRole');
        return this;
    }
    /**
     * Get subscription state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_GetSubscriptionState.html
     */
    getSubscriptionState() {
        this.add('shield:GetSubscriptionState');
        return this;
    }
    /**
     * List all existing attacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListAttacks.html
     */
    listAttacks() {
        this.add('shield:ListAttacks');
        return this;
    }
    /**
     * List all existing protections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListProtections.html
     */
    listProtections() {
        this.add('shield:ListProtections');
        return this;
    }
    /**
     * Updates the details of the list of email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateEmergencyContactSettings.html
     */
    updateEmergencyContactSettings() {
        this.add('shield:UpdateEmergencyContactSettings');
        return this;
    }
    /**
     * Updates the details of an existing subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateSubscription.html
     */
    updateSubscription() {
        this.add('shield:UpdateSubscription');
        return this;
    }
    /**
     * Adds a resource of type attack to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AttackDetail.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAttack(id, account, partition) {
        var arn = 'arn:${Partition}:shield::${Account}:attack/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type protection to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_Protection.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProtection(id, account, partition) {
        var arn = 'arn:${Partition}:shield::${Account}:protection/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Shield = Shield;
//# sourceMappingURL=data:application/json;base64,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