"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElasticloadbalancingV2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticloadbalancing-v2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_elasticloadbalancingv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ElasticloadbalancingV2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticloadbalancing-v2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_elasticloadbalancingv2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticloadbalancing-v2';
        this.actionList = {
            "AddListenerCertificates": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_AddListenerCertificates.html",
                "description": "Adds the specified certificates to the specified secure listener",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener/app": {
                        "required": true
                    },
                    "listener/net": {
                        "required": true
                    }
                }
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_AddTags.html",
                "description": "Adds the specified tags to the specified load balancer. Each load balancer can have a maximum of 10 tags",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    },
                    "targetgroup": {
                        "required": false
                    }
                }
            },
            "CreateListener": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateListener.html",
                "description": "Creates a listener for the specified Application Load Balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "CreateLoadBalancer": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateLoadBalancer.html",
                "description": "Creates a load balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "CreateRule": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateRule.html",
                "description": "Creates a rule for the specified listener",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener/app": {
                        "required": true
                    },
                    "listener/net": {
                        "required": true
                    }
                }
            },
            "CreateTargetGroup": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateTargetGroup.html",
                "description": "Creates a target group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "DeleteListener": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteListener.html",
                "description": "Deletes the specified listener",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener/app": {
                        "required": true
                    },
                    "listener/net": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancer": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteLoadBalancer.html",
                "description": "Deletes the specified load balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteRule.html",
                "description": "Deletes the specified rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener-rule/app": {
                        "required": true
                    },
                    "listener-rule/net": {
                        "required": true
                    }
                }
            },
            "DeleteTargetGroup": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteTargetGroup.html",
                "description": "Deletes the specified target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "DeregisterTargets": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeregisterTargets.html",
                "description": "Deregisters the specified targets from the specified target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "DescribeAccountLimits": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeAccountLimits.html",
                "description": "Describes the Elastic Load Balancing resource limits for the AWS account",
                "accessLevel": "Read"
            },
            "DescribeListenerCertificates": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeListenerCertificates.html",
                "description": "Describes the certificates for the specified secure listener",
                "accessLevel": "Read"
            },
            "DescribeListeners": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeListeners.html",
                "description": "Describes the specified listeners or the listeners for the specified Application Load Balancer",
                "accessLevel": "Read"
            },
            "DescribeLoadBalancerAttributes": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancerAttributes.html",
                "description": "Describes the attributes for the specified load balancer",
                "accessLevel": "Read"
            },
            "DescribeLoadBalancers": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html",
                "description": "Describes the specified the load balancers. If no load balancers are specified, the call describes all of your load balancers",
                "accessLevel": "Read"
            },
            "DescribeRules": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeRules.html",
                "description": "Describes the specified rules or the rules for the specified listener",
                "accessLevel": "Read"
            },
            "DescribeSSLPolicies": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeSSLPolicies.html",
                "description": "Describes the specified policies or all policies used for SSL negotiation",
                "accessLevel": "Read"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the tags associated with the specified load balancers",
                "accessLevel": "Read"
            },
            "DescribeTargetGroupAttributes": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroupAttributes.html",
                "description": "Describes the attributes for the specified target group",
                "accessLevel": "Read"
            },
            "DescribeTargetGroups": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html",
                "description": "Describes the specified target groups or all of your target groups",
                "accessLevel": "Read"
            },
            "DescribeTargetHealth": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetHealth.html",
                "description": "Describes the health of the specified targets or all of your targets",
                "accessLevel": "Read"
            },
            "ModifyListener": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyListener.html",
                "description": "Modifies the specified properties of the specified listener",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener/app": {
                        "required": true
                    },
                    "listener/net": {
                        "required": true
                    }
                }
            },
            "ModifyLoadBalancerAttributes": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyLoadBalancerAttributes.html",
                "description": "Modifies the attributes of the specified load balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "ModifyRule": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyRule.html",
                "description": "Modifies the specified rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener-rule/app": {
                        "required": true
                    },
                    "listener-rule/net": {
                        "required": true
                    }
                }
            },
            "ModifyTargetGroup": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyTargetGroup.html",
                "description": "Modifies the health checks used when evaluating the health state of the targets in the specified target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "ModifyTargetGroupAttributes": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyTargetGroupAttributes.html",
                "description": "Modifies the specified attributes of the specified target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "RegisterTargets": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RegisterTargets.html",
                "description": "Registers the specified targets with the specified target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "targetgroup": {
                        "required": true
                    }
                }
            },
            "RemoveListenerCertificates": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RemoveListenerCertificates.html",
                "description": "Removes the specified certificates of the specified secure listener",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener/app": {
                        "required": true
                    },
                    "listener/net": {
                        "required": true
                    }
                }
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RemoveTags.html",
                "description": "Removes one or more tags from the specified load balancer",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    },
                    "targetgroup": {
                        "required": false
                    }
                }
            },
            "SetIpAddressType": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetIpAddressType.html",
                "description": "Not found",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "SetRulePriorities": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetRulePriorities.html",
                "description": "Sets the priorities of the specified rules",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener-rule/app": {
                        "required": true
                    },
                    "listener-rule/net": {
                        "required": true
                    }
                }
            },
            "SetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetSecurityGroups.html",
                "description": "Associates the specified security groups with the specified load balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "SetSubnets": {
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetSubnets.html",
                "description": "Enables the Availability Zone for the specified subnets for the specified load balancer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loadbalancer/app/": {
                        "required": false
                    },
                    "loadbalancer/net/": {
                        "required": false
                    }
                }
            },
            "SetWebAcl": {
                "url": "",
                "description": "Gives WebAcl permission to WAF",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "listener/app": {
                "name": "listener/app",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener/app/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}",
                "conditionKeys": []
            },
            "listener-rule/app": {
                "name": "listener-rule/app",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-update-rules.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener-rule/app/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}/${ListenerRuleId}",
                "conditionKeys": []
            },
            "listener/net": {
                "name": "listener/net",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener/net/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}",
                "conditionKeys": []
            },
            "listener-rule/net": {
                "name": "listener-rule/net",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-update-rules.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener-rule/net/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}/${ListenerRuleId}",
                "conditionKeys": []
            },
            "loadbalancer/app/": {
                "name": "loadbalancer/app/",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/introduction.html#application-load-balancer-overview",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "elasticloadbalancing:ResourceTag/${TagKey}"
                ]
            },
            "loadbalancer/net/": {
                "name": "loadbalancer/net/",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/network/introduction.html#network-load-balancer-overview",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/net/${LoadBalancerName}/${LoadBalancerId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "elasticloadbalancing:ResourceTag/${TagKey}"
                ]
            },
            "targetgroup": {
                "name": "targetgroup",
                "url": "https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:targetgroup/${TargetGroupName}/${TargetGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "elasticloadbalancing:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds the specified certificates to the specified secure listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_AddListenerCertificates.html
     */
    addListenerCertificates() {
        this.add('elasticloadbalancing-v2:AddListenerCertificates');
        return this;
    }
    /**
     * Adds the specified tags to the specified load balancer. Each load balancer can have a maximum of 10 tags
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_AddTags.html
     */
    addTags() {
        this.add('elasticloadbalancing-v2:AddTags');
        return this;
    }
    /**
     * Creates a listener for the specified Application Load Balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateListener.html
     */
    createListener() {
        this.add('elasticloadbalancing-v2:CreateListener');
        return this;
    }
    /**
     * Creates a load balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateLoadBalancer.html
     */
    createLoadBalancer() {
        this.add('elasticloadbalancing-v2:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a rule for the specified listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateRule.html
     */
    createRule() {
        this.add('elasticloadbalancing-v2:CreateRule');
        return this;
    }
    /**
     * Creates a target group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_CreateTargetGroup.html
     */
    createTargetGroup() {
        this.add('elasticloadbalancing-v2:CreateTargetGroup');
        return this;
    }
    /**
     * Deletes the specified listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteListener.html
     */
    deleteListener() {
        this.add('elasticloadbalancing-v2:DeleteListener');
        return this;
    }
    /**
     * Deletes the specified load balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteLoadBalancer.html
     */
    deleteLoadBalancer() {
        this.add('elasticloadbalancing-v2:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes the specified rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteRule.html
     */
    deleteRule() {
        this.add('elasticloadbalancing-v2:DeleteRule');
        return this;
    }
    /**
     * Deletes the specified target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeleteTargetGroup.html
     */
    deleteTargetGroup() {
        this.add('elasticloadbalancing-v2:DeleteTargetGroup');
        return this;
    }
    /**
     * Deregisters the specified targets from the specified target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DeregisterTargets.html
     */
    deregisterTargets() {
        this.add('elasticloadbalancing-v2:DeregisterTargets');
        return this;
    }
    /**
     * Describes the Elastic Load Balancing resource limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeAccountLimits.html
     */
    describeAccountLimits() {
        this.add('elasticloadbalancing-v2:DescribeAccountLimits');
        return this;
    }
    /**
     * Describes the certificates for the specified secure listener
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeListenerCertificates.html
     */
    describeListenerCertificates() {
        this.add('elasticloadbalancing-v2:DescribeListenerCertificates');
        return this;
    }
    /**
     * Describes the specified listeners or the listeners for the specified Application Load Balancer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeListeners.html
     */
    describeListeners() {
        this.add('elasticloadbalancing-v2:DescribeListeners');
        return this;
    }
    /**
     * Describes the attributes for the specified load balancer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancerAttributes.html
     */
    describeLoadBalancerAttributes() {
        this.add('elasticloadbalancing-v2:DescribeLoadBalancerAttributes');
        return this;
    }
    /**
     * Describes the specified the load balancers. If no load balancers are specified, the call describes all of your load balancers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html
     */
    describeLoadBalancers() {
        this.add('elasticloadbalancing-v2:DescribeLoadBalancers');
        return this;
    }
    /**
     * Describes the specified rules or the rules for the specified listener
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeRules.html
     */
    describeRules() {
        this.add('elasticloadbalancing-v2:DescribeRules');
        return this;
    }
    /**
     * Describes the specified policies or all policies used for SSL negotiation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeSSLPolicies.html
     */
    describeSSLPolicies() {
        this.add('elasticloadbalancing-v2:DescribeSSLPolicies');
        return this;
    }
    /**
     * Describes the tags associated with the specified load balancers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('elasticloadbalancing-v2:DescribeTags');
        return this;
    }
    /**
     * Describes the attributes for the specified target group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroupAttributes.html
     */
    describeTargetGroupAttributes() {
        this.add('elasticloadbalancing-v2:DescribeTargetGroupAttributes');
        return this;
    }
    /**
     * Describes the specified target groups or all of your target groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html
     */
    describeTargetGroups() {
        this.add('elasticloadbalancing-v2:DescribeTargetGroups');
        return this;
    }
    /**
     * Describes the health of the specified targets or all of your targets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetHealth.html
     */
    describeTargetHealth() {
        this.add('elasticloadbalancing-v2:DescribeTargetHealth');
        return this;
    }
    /**
     * Modifies the specified properties of the specified listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyListener.html
     */
    modifyListener() {
        this.add('elasticloadbalancing-v2:ModifyListener');
        return this;
    }
    /**
     * Modifies the attributes of the specified load balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyLoadBalancerAttributes.html
     */
    modifyLoadBalancerAttributes() {
        this.add('elasticloadbalancing-v2:ModifyLoadBalancerAttributes');
        return this;
    }
    /**
     * Modifies the specified rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyRule.html
     */
    modifyRule() {
        this.add('elasticloadbalancing-v2:ModifyRule');
        return this;
    }
    /**
     * Modifies the health checks used when evaluating the health state of the targets in the specified target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyTargetGroup.html
     */
    modifyTargetGroup() {
        this.add('elasticloadbalancing-v2:ModifyTargetGroup');
        return this;
    }
    /**
     * Modifies the specified attributes of the specified target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_ModifyTargetGroupAttributes.html
     */
    modifyTargetGroupAttributes() {
        this.add('elasticloadbalancing-v2:ModifyTargetGroupAttributes');
        return this;
    }
    /**
     * Registers the specified targets with the specified target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RegisterTargets.html
     */
    registerTargets() {
        this.add('elasticloadbalancing-v2:RegisterTargets');
        return this;
    }
    /**
     * Removes the specified certificates of the specified secure listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RemoveListenerCertificates.html
     */
    removeListenerCertificates() {
        this.add('elasticloadbalancing-v2:RemoveListenerCertificates');
        return this;
    }
    /**
     * Removes one or more tags from the specified load balancer
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_RemoveTags.html
     */
    removeTags() {
        this.add('elasticloadbalancing-v2:RemoveTags');
        return this;
    }
    /**
     * Not found
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetIpAddressType.html
     */
    setIpAddressType() {
        this.add('elasticloadbalancing-v2:SetIpAddressType');
        return this;
    }
    /**
     * Sets the priorities of the specified rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetRulePriorities.html
     */
    setRulePriorities() {
        this.add('elasticloadbalancing-v2:SetRulePriorities');
        return this;
    }
    /**
     * Associates the specified security groups with the specified load balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetSecurityGroups.html
     */
    setSecurityGroups() {
        this.add('elasticloadbalancing-v2:SetSecurityGroups');
        return this;
    }
    /**
     * Enables the Availability Zone for the specified subnets for the specified load balancer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_SetSubnets.html
     */
    setSubnets() {
        this.add('elasticloadbalancing-v2:SetSubnets');
        return this;
    }
    /**
     * Gives WebAcl permission to WAF
     *
     * Access Level: Write
     */
    setWebAcl() {
        this.add('elasticloadbalancing-v2:SetWebAcl');
        return this;
    }
    /**
     * Adds a resource of type listener/app to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onListenerApp(loadBalancerName, loadBalancerId, listenerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener/app/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener-rule/app to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-update-rules.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param listenerId - Identifier for the listenerId.
     * @param listenerRuleId - Identifier for the listenerRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onListenerRuleApp(loadBalancerName, loadBalancerId, listenerId, listenerRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener-rule/app/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}/${ListenerRuleId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${ListenerRuleId}', listenerRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener/net to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onListenerNet(loadBalancerName, loadBalancerId, listenerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener/net/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener-rule/net to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-update-rules.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param listenerId - Identifier for the listenerId.
     * @param listenerRuleId - Identifier for the listenerRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onListenerRuleNet(loadBalancerName, loadBalancerId, listenerId, listenerRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:listener-rule/net/${LoadBalancerName}/${LoadBalancerId}/${ListenerId}/${ListenerRuleId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${ListenerRuleId}', listenerRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/introduction.html#application-load-balancer-overview
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - elasticloadbalancing:ResourceTag/${TagKey}
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/net/ to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/network/introduction.html#network-load-balancer-overview
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - elasticloadbalancing:ResourceTag/${TagKey}
     */
    onLoadbalancerNet(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/net/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type targetgroup to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html
     *
     * @param targetGroupName - Identifier for the targetGroupName.
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - elasticloadbalancing:ResourceTag/${TagKey}
     */
    onTargetgroup(targetGroupName, targetGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:targetgroup/${TargetGroupName}/${TargetGroupId}';
        arn = arn.replace('${TargetGroupName}', targetGroupName);
        arn = arn.replace('${TargetGroupId}', targetGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A tag key and value pair.
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`elasticloadbalancing:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.ElasticloadbalancingV2 = ElasticloadbalancingV2;
//# sourceMappingURL=data:application/json;base64,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