"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudsearch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudsearch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudsearch';
        this.actionList = {
            "AddTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_AddTags.html",
                "description": "Attaches resource tags to an Amazon CloudSearch domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "BuildSuggesters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_BuildSuggesters.html",
                "description": "Indexes the search suggestions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_CreateDomain.html",
                "description": "Creates a new search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineAnalysisScheme": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineAnalysisScheme.html",
                "description": "Configures an analysis scheme that can be applied to a text or text-array field to define language-specific text processing options.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineExpression": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineExpression.html",
                "description": "Configures an Expression for the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineIndexField": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineIndexField.html",
                "description": "Configures an IndexField for the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineSuggester": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineSuggester.html",
                "description": "Configures a suggester for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteAnalysisScheme": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteAnalysisScheme.html",
                "description": "Deletes an analysis scheme.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteDomain.html",
                "description": "Permanently deletes a search domain and all of its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteExpression": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteExpression.html",
                "description": "Removes an Expression from the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteIndexField": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteIndexField.html",
                "description": "Removes an IndexField from the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteSuggester": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteSuggester.html",
                "description": "Deletes a suggester.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeAnalysisSchemes": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAnalysisSchemes.html",
                "description": "Gets the analysis schemes configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeAvailabilityOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAvailabilityOptions.html",
                "description": "Gets the availability options configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeDomainEndpointOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomainEndpointOptions.html",
                "description": "Gets the domain endpoint options configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeDomains": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomains.html",
                "description": "Gets information about the search domains owned by this account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeExpressions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeExpressions.html",
                "description": "Gets the expressions configured for the search domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeIndexFields": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeIndexFields.html",
                "description": "Gets information about the index fields configured for the search domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeScalingParameters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeScalingParameters.html",
                "description": "Gets the scaling parameters configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeServiceAccessPolicies": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeServiceAccessPolicies.html",
                "description": "Gets information about the access policies that control access to the domain's document and search endpoints.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeSuggesters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeSuggesters.html",
                "description": "Gets the suggesters configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "IndexDocuments": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_IndexDocuments.html",
                "description": "Tells the search domain to start indexing its documents using the latest indexing options.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomainNames": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListDomainNames.html",
                "description": "Lists all search domains owned by an account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListTags.html",
                "description": "Displays all of the resource tags for an Amazon CloudSearch domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_RemoveTags.html",
                "description": "Removes the specified resource tags from an Amazon ES domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateAvailabilityOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateAvailabilityOptions.html",
                "description": "Configures the availability options for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateDomainEndpointOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateDomainEndpointOptions.html",
                "description": "Configures the domain endpoint options for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateScalingParameters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateScalingParameters.html",
                "description": "Configures scaling parameters for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateServiceAccessPolicies": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateServiceAccessPolicies.html",
                "description": "Configures the access rules that control access to the domain's document and search endpoints.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "document": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the document service operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "search": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the search operations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "suggest": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the suggest operations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html",
                "arn": "arn:${Partition}:cloudsearch:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Attaches resource tags to an Amazon CloudSearch domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_AddTags.html
     */
    addTags() {
        this.add('cloudsearch:AddTags');
        return this;
    }
    /**
     * Indexes the search suggestions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_BuildSuggesters.html
     */
    buildSuggesters() {
        this.add('cloudsearch:BuildSuggesters');
        return this;
    }
    /**
     * Creates a new search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_CreateDomain.html
     */
    createDomain() {
        this.add('cloudsearch:CreateDomain');
        return this;
    }
    /**
     * Configures an analysis scheme that can be applied to a text or text-array field to define language-specific text processing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineAnalysisScheme.html
     */
    defineAnalysisScheme() {
        this.add('cloudsearch:DefineAnalysisScheme');
        return this;
    }
    /**
     * Configures an Expression for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineExpression.html
     */
    defineExpression() {
        this.add('cloudsearch:DefineExpression');
        return this;
    }
    /**
     * Configures an IndexField for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineIndexField.html
     */
    defineIndexField() {
        this.add('cloudsearch:DefineIndexField');
        return this;
    }
    /**
     * Configures a suggester for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineSuggester.html
     */
    defineSuggester() {
        this.add('cloudsearch:DefineSuggester');
        return this;
    }
    /**
     * Deletes an analysis scheme.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteAnalysisScheme.html
     */
    deleteAnalysisScheme() {
        this.add('cloudsearch:DeleteAnalysisScheme');
        return this;
    }
    /**
     * Permanently deletes a search domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('cloudsearch:DeleteDomain');
        return this;
    }
    /**
     * Removes an Expression from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteExpression.html
     */
    deleteExpression() {
        this.add('cloudsearch:DeleteExpression');
        return this;
    }
    /**
     * Removes an IndexField from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteIndexField.html
     */
    deleteIndexField() {
        this.add('cloudsearch:DeleteIndexField');
        return this;
    }
    /**
     * Deletes a suggester.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteSuggester.html
     */
    deleteSuggester() {
        this.add('cloudsearch:DeleteSuggester');
        return this;
    }
    /**
     * Gets the analysis schemes configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAnalysisSchemes.html
     */
    describeAnalysisSchemes() {
        this.add('cloudsearch:DescribeAnalysisSchemes');
        return this;
    }
    /**
     * Gets the availability options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAvailabilityOptions.html
     */
    describeAvailabilityOptions() {
        this.add('cloudsearch:DescribeAvailabilityOptions');
        return this;
    }
    /**
     * Gets the domain endpoint options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomainEndpointOptions.html
     */
    describeDomainEndpointOptions() {
        this.add('cloudsearch:DescribeDomainEndpointOptions');
        return this;
    }
    /**
     * Gets information about the search domains owned by this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomains.html
     */
    describeDomains() {
        this.add('cloudsearch:DescribeDomains');
        return this;
    }
    /**
     * Gets the expressions configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeExpressions.html
     */
    describeExpressions() {
        this.add('cloudsearch:DescribeExpressions');
        return this;
    }
    /**
     * Gets information about the index fields configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeIndexFields.html
     */
    describeIndexFields() {
        this.add('cloudsearch:DescribeIndexFields');
        return this;
    }
    /**
     * Gets the scaling parameters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeScalingParameters.html
     */
    describeScalingParameters() {
        this.add('cloudsearch:DescribeScalingParameters');
        return this;
    }
    /**
     * Gets information about the access policies that control access to the domain's document and search endpoints.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeServiceAccessPolicies.html
     */
    describeServiceAccessPolicies() {
        this.add('cloudsearch:DescribeServiceAccessPolicies');
        return this;
    }
    /**
     * Gets the suggesters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeSuggesters.html
     */
    describeSuggesters() {
        this.add('cloudsearch:DescribeSuggesters');
        return this;
    }
    /**
     * Tells the search domain to start indexing its documents using the latest indexing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_IndexDocuments.html
     */
    indexDocuments() {
        this.add('cloudsearch:IndexDocuments');
        return this;
    }
    /**
     * Lists all search domains owned by an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListDomainNames.html
     */
    listDomainNames() {
        this.add('cloudsearch:ListDomainNames');
        return this;
    }
    /**
     * Displays all of the resource tags for an Amazon CloudSearch domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListTags.html
     */
    listTags() {
        this.add('cloudsearch:ListTags');
        return this;
    }
    /**
     * Removes the specified resource tags from an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_RemoveTags.html
     */
    removeTags() {
        this.add('cloudsearch:RemoveTags');
        return this;
    }
    /**
     * Configures the availability options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateAvailabilityOptions.html
     */
    updateAvailabilityOptions() {
        this.add('cloudsearch:UpdateAvailabilityOptions');
        return this;
    }
    /**
     * Configures the domain endpoint options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateDomainEndpointOptions.html
     */
    updateDomainEndpointOptions() {
        this.add('cloudsearch:UpdateDomainEndpointOptions');
        return this;
    }
    /**
     * Configures scaling parameters for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateScalingParameters.html
     */
    updateScalingParameters() {
        this.add('cloudsearch:UpdateScalingParameters');
        return this;
    }
    /**
     * Configures the access rules that control access to the domain's document and search endpoints.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateServiceAccessPolicies.html
     */
    updateServiceAccessPolicies() {
        this.add('cloudsearch:UpdateServiceAccessPolicies');
        return this;
    }
    /**
     * Allows access to the document service operations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    document() {
        this.add('cloudsearch:document');
        return this;
    }
    /**
     * Allows access to the search operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    search() {
        this.add('cloudsearch:search');
        return this;
    }
    /**
     * Allows access to the suggest operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    suggest() {
        this.add('cloudsearch:suggest');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudsearch:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudsearch = Cloudsearch;
//# sourceMappingURL=data:application/json;base64,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