"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dax = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dax](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodbacceleratordax.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dax extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dax](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondynamodbacceleratordax.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dax';
        this.actionList = {
            "BatchGetItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html",
                "description": "The BatchGetItem action returns the attributes of one or more items from one or more tables.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "BatchWriteItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html",
                "description": "The BatchWriteItem action operation puts or deletes multiple items in one or more tables.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ConditionCheckItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html",
                "description": "The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateCluster.html",
                "description": "The CreateCluster action creates a DAX cluster.",
                "accessLevel": "Write",
                "dependentActions": [
                    "dax:CreateParameterGroup",
                    "dax:CreateSubnetGroup",
                    "ec2:CreateNetworkInterface",
                    "ec2:DeleteNetworkInterface",
                    "ec2:DescribeNetworkInterfaces",
                    "ec2:DescribeSecurityGroups",
                    "ec2:DescribeSubnets",
                    "ec2:DescribeVpcs",
                    "iam:GetRole",
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "CreateParameterGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateParameterGroup.html",
                "description": "The CreateParameterGroup action creates collection of parameters that you apply to all of the nodes in a DAX cluster.",
                "accessLevel": "Write"
            },
            "CreateSubnetGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateSubnetGroup.html",
                "description": "The CreateSubnetGroup action creates a new subnet group.",
                "accessLevel": "Write"
            },
            "DecreaseReplicationFactor": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DecreaseReplicationFactor.html",
                "description": "The DecreaseReplicationFactor action removes one or more nodes from a DAX cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteCluster.html",
                "description": "The DeleteCluster action deletes a previously provisioned DAX cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html",
                "description": "The DeleteItem action deletes a single item in a table by primary key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "dax:EnclosingOperation"
                ]
            },
            "DeleteParameterGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteParameterGroup.html",
                "description": "The DeleteParameterGroup action deletes the specified parameter group.",
                "accessLevel": "Write"
            },
            "DeleteSubnetGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteSubnetGroup.html",
                "description": "The DeleteSubnetGroup action deletes a subnet group.",
                "accessLevel": "Write"
            },
            "DescribeClusters": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeClusters.html",
                "description": "The DescribeClusters action returns information about all provisioned DAX clusters.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": false
                    }
                }
            },
            "DescribeDefaultParameters": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeDefaultParameters.html",
                "description": "The DescribeDefaultParameters action returns the default system parameter information for DAX.",
                "accessLevel": "List"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeEvents.html",
                "description": "The DescribeEvents action returns events related to DAX clusters and parameter groups.",
                "accessLevel": "List"
            },
            "DescribeParameterGroups": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameterGroups.html",
                "description": "The DescribeParameterGroups action returns a list of parameter group descriptions.",
                "accessLevel": "List"
            },
            "DescribeParameters": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameters.html",
                "description": "The DescribeParameters action returns the detailed parameter list for a particular parameter group.",
                "accessLevel": "Read"
            },
            "DescribeSubnetGroups": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeSubnetGroups.html",
                "description": "The DescribeSubnetGroups action returns a list of subnet group descriptions.",
                "accessLevel": "List"
            },
            "GetItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html",
                "description": "The GetItem action returns a set of attributes for the item with the given primary key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "dax:EnclosingOperation"
                ]
            },
            "IncreaseReplicationFactor": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_IncreaseReplicationFactor.html",
                "description": "The IncreaseReplicationFactor action adds one or more nodes to a DAX cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_ListTags.html",
                "description": "The ListTags action returns a list all of the tags for a DAX cluster.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "PutItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html",
                "description": "The PutItem action creates a new item, or replaces an old item with a new item.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "dax:EnclosingOperation"
                ]
            },
            "Query": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html",
                "description": "The Query action finds items based on primary key values. You can query any table or secondary index that has a composite primary key (a partition key and a sort key).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "RebootNode": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_RebootNode.html",
                "description": "The RebootNode action reboots a single node of a DAX cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "Scan": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html",
                "description": "The Scan action returns one or more items and item attributes by accessing every item in a table or a secondary index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_TagResource.html",
                "description": "The TagResource action associates a set of tags with a DAX resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UntagResource.html",
                "description": "The UntagResource action removes the association of tags from a DAX resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UpdateCluster": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateCluster.html",
                "description": "The UpdateCluster action modifies the settings for a DAX cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UpdateItem": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html",
                "description": "The UpdateItem action edits an existing item's attributes, or adds a new item to the table if it does not already exist.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "dax:EnclosingOperation"
                ]
            },
            "UpdateParameterGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateParameterGroup.html",
                "description": "The UpdateParameterGroup action modifies the parameters of a parameter group.",
                "accessLevel": "Write"
            },
            "UpdateSubnetGroup": {
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateSubnetGroup.html",
                "description": "The UpdateSubnetGroup action modifies an existing subnet group.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "application": {
                "name": "application",
                "url": "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DAX.access-control.html",
                "arn": "arn:${Partition}:dax:${Region}:${Account}:cache/${ClusterName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * The BatchGetItem action returns the attributes of one or more items from one or more tables.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    batchGetItem() {
        this.add('dax:BatchGetItem');
        return this;
    }
    /**
     * The BatchWriteItem action operation puts or deletes multiple items in one or more tables.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    batchWriteItem() {
        this.add('dax:BatchWriteItem');
        return this;
    }
    /**
     * The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    conditionCheckItem() {
        this.add('dax:ConditionCheckItem');
        return this;
    }
    /**
     * The CreateCluster action creates a DAX cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dax:CreateParameterGroup
     * - dax:CreateSubnetGroup
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:GetRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateCluster.html
     */
    createCluster() {
        this.add('dax:CreateCluster');
        return this;
    }
    /**
     * The CreateParameterGroup action creates collection of parameters that you apply to all of the nodes in a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateParameterGroup.html
     */
    createParameterGroup() {
        this.add('dax:CreateParameterGroup');
        return this;
    }
    /**
     * The CreateSubnetGroup action creates a new subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateSubnetGroup.html
     */
    createSubnetGroup() {
        this.add('dax:CreateSubnetGroup');
        return this;
    }
    /**
     * The DecreaseReplicationFactor action removes one or more nodes from a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DecreaseReplicationFactor.html
     */
    decreaseReplicationFactor() {
        this.add('dax:DecreaseReplicationFactor');
        return this;
    }
    /**
     * The DeleteCluster action deletes a previously provisioned DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteCluster.html
     */
    deleteCluster() {
        this.add('dax:DeleteCluster');
        return this;
    }
    /**
     * The DeleteItem action deletes a single item in a table by primary key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    deleteItem() {
        this.add('dax:DeleteItem');
        return this;
    }
    /**
     * The DeleteParameterGroup action deletes the specified parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteParameterGroup.html
     */
    deleteParameterGroup() {
        this.add('dax:DeleteParameterGroup');
        return this;
    }
    /**
     * The DeleteSubnetGroup action deletes a subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteSubnetGroup.html
     */
    deleteSubnetGroup() {
        this.add('dax:DeleteSubnetGroup');
        return this;
    }
    /**
     * The DescribeClusters action returns information about all provisioned DAX clusters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeClusters.html
     */
    describeClusters() {
        this.add('dax:DescribeClusters');
        return this;
    }
    /**
     * The DescribeDefaultParameters action returns the default system parameter information for DAX.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeDefaultParameters.html
     */
    describeDefaultParameters() {
        this.add('dax:DescribeDefaultParameters');
        return this;
    }
    /**
     * The DescribeEvents action returns events related to DAX clusters and parameter groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeEvents.html
     */
    describeEvents() {
        this.add('dax:DescribeEvents');
        return this;
    }
    /**
     * The DescribeParameterGroups action returns a list of parameter group descriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameterGroups.html
     */
    describeParameterGroups() {
        this.add('dax:DescribeParameterGroups');
        return this;
    }
    /**
     * The DescribeParameters action returns the detailed parameter list for a particular parameter group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameters.html
     */
    describeParameters() {
        this.add('dax:DescribeParameters');
        return this;
    }
    /**
     * The DescribeSubnetGroups action returns a list of subnet group descriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeSubnetGroups.html
     */
    describeSubnetGroups() {
        this.add('dax:DescribeSubnetGroups');
        return this;
    }
    /**
     * The GetItem action returns a set of attributes for the item with the given primary key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    getItem() {
        this.add('dax:GetItem');
        return this;
    }
    /**
     * The IncreaseReplicationFactor action adds one or more nodes to a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_IncreaseReplicationFactor.html
     */
    increaseReplicationFactor() {
        this.add('dax:IncreaseReplicationFactor');
        return this;
    }
    /**
     * The ListTags action returns a list all of the tags for a DAX cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_ListTags.html
     */
    listTags() {
        this.add('dax:ListTags');
        return this;
    }
    /**
     * The PutItem action creates a new item, or replaces an old item with a new item.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    putItem() {
        this.add('dax:PutItem');
        return this;
    }
    /**
     * The Query action finds items based on primary key values. You can query any table or secondary index that has a composite primary key (a partition key and a sort key).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    query() {
        this.add('dax:Query');
        return this;
    }
    /**
     * The RebootNode action reboots a single node of a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_RebootNode.html
     */
    rebootNode() {
        this.add('dax:RebootNode');
        return this;
    }
    /**
     * The Scan action returns one or more items and item attributes by accessing every item in a table or a secondary index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    scan() {
        this.add('dax:Scan');
        return this;
    }
    /**
     * The TagResource action associates a set of tags with a DAX resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_TagResource.html
     */
    tagResource() {
        this.add('dax:TagResource');
        return this;
    }
    /**
     * The UntagResource action removes the association of tags from a DAX resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UntagResource.html
     */
    untagResource() {
        this.add('dax:UntagResource');
        return this;
    }
    /**
     * The UpdateCluster action modifies the settings for a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateCluster.html
     */
    updateCluster() {
        this.add('dax:UpdateCluster');
        return this;
    }
    /**
     * The UpdateItem action edits an existing item's attributes, or adds a new item to the table if it does not already exist.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    updateItem() {
        this.add('dax:UpdateItem');
        return this;
    }
    /**
     * The UpdateParameterGroup action modifies the parameters of a parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateParameterGroup.html
     */
    updateParameterGroup() {
        this.add('dax:UpdateParameterGroup');
        return this;
    }
    /**
     * The UpdateSubnetGroup action modifies an existing subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateSubnetGroup.html
     */
    updateSubnetGroup() {
        this.add('dax:UpdateSubnetGroup');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DAX.access-control.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplication(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:dax:${Region}:${Account}:cache/${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa.
     *
     * Applies to actions:
     * - .deleteItem()
     * - .getItem()
     * - .putItem()
     * - .updateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`dax:EnclosingOperation`, value, operator || 'StringLike');
    }
}
exports.Dax = Dax;
//# sourceMappingURL=data:application/json;base64,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