"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glacier = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glacier](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glacier extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glacier](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glacier';
        this.actionList = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html",
                "description": "Aborts a multipart upload identified by the upload ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AbortVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html",
                "description": "Aborts the vault locking process if the vault lock is not in the Locked state",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AddTagsToVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html",
                "description": "Adds the specified tags to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html",
                "description": "Completes a multipart upload process",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html",
                "description": "Completes the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CreateVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html",
                "description": "Creates a new vault with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html",
                "description": "Deletes an archive from a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "DeleteVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html",
                "description": "Deletes a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html",
                "description": "Deletes the access policy associated with the specified vault",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html",
                "description": "Deletes the notification configuration set for a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html",
                "description": "Returns information about a job you previously initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html",
                "description": "Returns information about a vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html",
                "description": "Returns the current data retrieval policy for the account and region specified in the GET request",
                "accessLevel": "Read"
            },
            "GetJobOutput": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html",
                "description": "Downloads the output of the job you initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html",
                "description": "Retrieves the access-policy subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html",
                "description": "Retrieves attributes from the lock-policy subresource set on the specified vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html",
                "description": "Retrieves the notification-configuration subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html",
                "description": "Initiates a job of the specified type",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "InitiateMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html",
                "description": "Initiates a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html",
                "description": "Initiates the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html",
                "description": "Lists jobs for a vault that are in-progress and jobs that have recently finished",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListMultipartUploads": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html",
                "description": "Lists in-progress multipart uploads for the specified vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListParts": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html",
                "description": "Lists the parts of an archive that have been uploaded in a specific multipart upload",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html",
                "description": "This operation lists the provisioned capacity for the specified AWS account.",
                "accessLevel": "List"
            },
            "ListTagsForVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html",
                "description": "Lists all the tags attached to a vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListVaults": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html",
                "description": "Lists all vaults",
                "accessLevel": "List"
            },
            "PurchaseProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html",
                "description": "This operation purchases a provisioned capacity unit for an AWS account.",
                "accessLevel": "Write"
            },
            "RemoveTagsFromVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html",
                "description": "Removes one or more tags from the set of tags attached to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html",
                "description": "Sets and then enacts a data retrieval policy in the region specified in the PUT request",
                "accessLevel": "Permissions management"
            },
            "SetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html",
                "description": "Configures an access policy for a vault and will overwrite an existing policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html",
                "description": "Configures vault notifications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html",
                "description": "Adds an archive to a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadMultipartPart": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html",
                "description": "Uploads a part of an archive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "vault": {
                "name": "vault",
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html",
                "arn": "arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Aborts a multipart upload identified by the upload ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html
     */
    abortMultipartUpload() {
        this.add('glacier:AbortMultipartUpload');
        return this;
    }
    /**
     * Aborts the vault locking process if the vault lock is not in the Locked state
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html
     */
    abortVaultLock() {
        this.add('glacier:AbortVaultLock');
        return this;
    }
    /**
     * Adds the specified tags to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html
     */
    addTagsToVault() {
        this.add('glacier:AddTagsToVault');
        return this;
    }
    /**
     * Completes a multipart upload process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html
     */
    completeMultipartUpload() {
        this.add('glacier:CompleteMultipartUpload');
        return this;
    }
    /**
     * Completes the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html
     */
    completeVaultLock() {
        this.add('glacier:CompleteVaultLock');
        return this;
    }
    /**
     * Creates a new vault with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html
     */
    createVault() {
        this.add('glacier:CreateVault');
        return this;
    }
    /**
     * Deletes an archive from a vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArchiveAgeInDays()
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html
     */
    deleteArchive() {
        this.add('glacier:DeleteArchive');
        return this;
    }
    /**
     * Deletes a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html
     */
    deleteVault() {
        this.add('glacier:DeleteVault');
        return this;
    }
    /**
     * Deletes the access policy associated with the specified vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html
     */
    deleteVaultAccessPolicy() {
        this.add('glacier:DeleteVaultAccessPolicy');
        return this;
    }
    /**
     * Deletes the notification configuration set for a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html
     */
    deleteVaultNotifications() {
        this.add('glacier:DeleteVaultNotifications');
        return this;
    }
    /**
     * Returns information about a job you previously initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html
     */
    describeJob() {
        this.add('glacier:DescribeJob');
        return this;
    }
    /**
     * Returns information about a vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html
     */
    describeVault() {
        this.add('glacier:DescribeVault');
        return this;
    }
    /**
     * Returns the current data retrieval policy for the account and region specified in the GET request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html
     */
    getDataRetrievalPolicy() {
        this.add('glacier:GetDataRetrievalPolicy');
        return this;
    }
    /**
     * Downloads the output of the job you initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html
     */
    getJobOutput() {
        this.add('glacier:GetJobOutput');
        return this;
    }
    /**
     * Retrieves the access-policy subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html
     */
    getVaultAccessPolicy() {
        this.add('glacier:GetVaultAccessPolicy');
        return this;
    }
    /**
     * Retrieves attributes from the lock-policy subresource set on the specified vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html
     */
    getVaultLock() {
        this.add('glacier:GetVaultLock');
        return this;
    }
    /**
     * Retrieves the notification-configuration subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html
     */
    getVaultNotifications() {
        this.add('glacier:GetVaultNotifications');
        return this;
    }
    /**
     * Initiates a job of the specified type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifArchiveAgeInDays()
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html
     */
    initiateJob() {
        this.add('glacier:InitiateJob');
        return this;
    }
    /**
     * Initiates a multipart upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html
     */
    initiateMultipartUpload() {
        this.add('glacier:InitiateMultipartUpload');
        return this;
    }
    /**
     * Initiates the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html
     */
    initiateVaultLock() {
        this.add('glacier:InitiateVaultLock');
        return this;
    }
    /**
     * Lists jobs for a vault that are in-progress and jobs that have recently finished
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html
     */
    listJobs() {
        this.add('glacier:ListJobs');
        return this;
    }
    /**
     * Lists in-progress multipart uploads for the specified vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html
     */
    listMultipartUploads() {
        this.add('glacier:ListMultipartUploads');
        return this;
    }
    /**
     * Lists the parts of an archive that have been uploaded in a specific multipart upload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html
     */
    listParts() {
        this.add('glacier:ListParts');
        return this;
    }
    /**
     * This operation lists the provisioned capacity for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html
     */
    listProvisionedCapacity() {
        this.add('glacier:ListProvisionedCapacity');
        return this;
    }
    /**
     * Lists all the tags attached to a vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html
     */
    listTagsForVault() {
        this.add('glacier:ListTagsForVault');
        return this;
    }
    /**
     * Lists all vaults
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html
     */
    listVaults() {
        this.add('glacier:ListVaults');
        return this;
    }
    /**
     * This operation purchases a provisioned capacity unit for an AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html
     */
    purchaseProvisionedCapacity() {
        this.add('glacier:PurchaseProvisionedCapacity');
        return this;
    }
    /**
     * Removes one or more tags from the set of tags attached to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html
     */
    removeTagsFromVault() {
        this.add('glacier:RemoveTagsFromVault');
        return this;
    }
    /**
     * Sets and then enacts a data retrieval policy in the region specified in the PUT request
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html
     */
    setDataRetrievalPolicy() {
        this.add('glacier:SetDataRetrievalPolicy');
        return this;
    }
    /**
     * Configures an access policy for a vault and will overwrite an existing policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html
     */
    setVaultAccessPolicy() {
        this.add('glacier:SetVaultAccessPolicy');
        return this;
    }
    /**
     * Configures vault notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html
     */
    setVaultNotifications() {
        this.add('glacier:SetVaultNotifications');
        return this;
    }
    /**
     * Adds an archive to a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html
     */
    uploadArchive() {
        this.add('glacier:UploadArchive');
        return this;
    }
    /**
     * Uploads a part of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html
     */
    uploadMultipartPart() {
        this.add('glacier:UploadMultipartPart');
        return this;
    }
    /**
     * Adds a resource of type vault to the statement
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html
     *
     * @param vaultName - Identifier for the vaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVault(vaultName, account, region, partition) {
        var arn = 'arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}';
        arn = arn.replace('${VaultName}', vaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * How long an archive has been stored in the vault, in days.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .deleteArchive()
     * - .initiateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArchiveAgeInDays(value, operator) {
        return this.if(`glacier:ArchiveAgeInDays`, value, operator || 'StringLike');
    }
    /**
     * A customer-defined tag.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`glacier:ResourceTag/`, value, operator || 'StringLike');
    }
}
exports.Glacier = Glacier;
//# sourceMappingURL=data:application/json;base64,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