"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacie.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie2';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html",
                "description": "Grants permission to accept an Amazon Macie membership invitation",
                "accessLevel": "Write"
            },
            "ArchiveFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-archive.html",
                "description": "Grants permission to archive one or more findings",
                "accessLevel": "Write"
            },
            "BatchGetCustomDataIdentifiers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html",
                "description": "Grants permission to retrieve information about one or more custom data identifiers",
                "accessLevel": "Read",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "CreateClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html",
                "description": "Grants permission to create and define the settings for a classification job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html",
                "description": "Grants permission to create and define the settings for a custom data identifier",
                "accessLevel": "Write",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html",
                "description": "Grants permission to create and define the settings for a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html",
                "description": "Grants permission to send an Amazon Macie membership invitation",
                "accessLevel": "Write"
            },
            "CreateMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members.html",
                "description": "Grants permission to associate an account with an Amazon Macie master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSampleFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html",
                "description": "Grants permission to create sample findings",
                "accessLevel": "Write"
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html",
                "description": "Grants permission to decline Amazon Macie membership invitations",
                "accessLevel": "Write"
            },
            "DeleteCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html",
                "description": "Grants permission to delete a custom data identifier",
                "accessLevel": "Write",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "DeleteFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to delete a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html",
                "description": "Grants permission to delete Amazon Macie membership invitations",
                "accessLevel": "Write"
            },
            "DeleteMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html",
                "description": "Grants permission to delete the association between an Amazon Macie master account and an account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "DescribeBuckets": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html",
                "description": "Grants permission to retrieve statistical and other data about S3 buckets that Amazon Macie monitors and analyzes",
                "accessLevel": "Read"
            },
            "DescribeClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html",
                "description": "Grants permission to retrieve information about the status and settings for a classification job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                }
            },
            "DescribeOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html",
                "description": "Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization",
                "accessLevel": "Read"
            },
            "DisableMacie": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account",
                "accessLevel": "Write"
            },
            "DisableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to disable an account as a delegated administrator of Amazon Macie for an AWS organization",
                "accessLevel": "Write"
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html",
                "description": "Grants an Amazon Macie member account with permission to disassociate from its master account",
                "accessLevel": "Write"
            },
            "DisassociateMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html",
                "description": "Grants an Amazon Macie master account with permission to disassociate from a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "EnableMacie": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to enable and specify the configuration settings for a new Amazon Macie account",
                "accessLevel": "Write"
            },
            "EnableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to enable an account as a delegated administrator of Amazon Macie for an AWS organization",
                "accessLevel": "Write"
            },
            "GetBucketStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html",
                "description": "Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes",
                "accessLevel": "Read"
            },
            "GetClassificationExportConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html",
                "description": "Grants permission to retrieve the settings for exporting data classification results",
                "accessLevel": "Read"
            },
            "GetCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html",
                "description": "Grants permission to retrieve information about the settings for a custom data identifier",
                "accessLevel": "Read",
                "resourceTypes": {
                    "CustomDataIdentifier": {
                        "required": true
                    }
                }
            },
            "GetFindingStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html",
                "description": "Grants permission to retrieve aggregated statistical data about findings",
                "accessLevel": "Read"
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html",
                "description": "Grants permission to retrieve information about one or more findings",
                "accessLevel": "Read"
            },
            "GetFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to retrieve information about the settings for a findings filter",
                "accessLevel": "Read",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html",
                "description": "Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account",
                "accessLevel": "Read"
            },
            "GetMacieSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account",
                "accessLevel": "Read"
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/master.html",
                "description": "Grants permission to retrieve information about the Amazon Macie master account for an account",
                "accessLevel": "Read"
            },
            "GetMember": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html",
                "description": "Grants permission to retrieve information about an account that's associated with an Amazon Macie master account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Member": {
                        "required": true
                    }
                }
            },
            "GetUsageStatistics": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html",
                "description": "Grants permission to retrieve quotas and aggregated usage data for one or more accounts",
                "accessLevel": "Read"
            },
            "GetUsageTotals": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/usage.html",
                "description": "Grants permission to retrieve aggregated usage data for an account",
                "accessLevel": "Read"
            },
            "ListClassificationJobs": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html",
                "description": "Grants permission to retrieve information about the status and settings for one or more classification jobs",
                "accessLevel": "List"
            },
            "ListCustomDataIdentifiers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html",
                "description": "Grants permission to retrieve information about all custom data identifiers",
                "accessLevel": "List"
            },
            "ListFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings.html",
                "description": "Grants permission to retrieve a subset of information about one or more findings",
                "accessLevel": "List"
            },
            "ListFindingsFilters": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html",
                "description": "Grants permission to retrieve information about all findings filters",
                "accessLevel": "List"
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html",
                "description": "Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/members.html",
                "description": "Grants permission to retrieve information about all the accounts that are associated with an Amazon Macie master account",
                "accessLevel": "List"
            },
            "ListOrganizationAdminAccounts": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin.html",
                "description": "Grants permission to retrieve information about the delegated, Amazon Macie administrator account for an AWS organization",
                "accessLevel": "List"
            },
            "ListTagsForResources": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to retrieve the tags for an Amazon Macie resource or member account",
                "accessLevel": "List"
            },
            "PutClassificationExportConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html",
                "description": "Grants permission to create or update the settings for exporting data classification results",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to add or update the tags for an Amazon Macie resource or member account",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestCustomDataIdentifier": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html",
                "description": "Grants permission to test a custom data identifier",
                "accessLevel": "Write"
            },
            "UnarchiveFindings": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findings-unarchive.html",
                "description": "Grants permission to reactivate (unarchive) one or more findings",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html",
                "description": "Grants permission to remove tags from an Amazon Macie resource or member account",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateClassificationJob": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html",
                "description": "Grants permission to cancel a classification job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ClassificationJob": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateFindingsFilter": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html",
                "description": "Grants permission to update the settings for a findings filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "FindingsFilter": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateMacieSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie.html",
                "description": "Grants permission to suspend or re-enable an Amazon Macie account, or update the configuration settings for a Macie account",
                "accessLevel": "Write"
            },
            "UpdateMemberSession": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html",
                "description": "Grants an Amazon Macie master account with permission to suspend or re-enable a member account",
                "accessLevel": "Write"
            },
            "UpdateOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html",
                "description": "Grants permission to update Amazon Macie configuration settings for an AWS organization",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "ClassificationJob": {
                "name": "ClassificationJob",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:classification-job/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CustomDataIdentifier": {
                "name": "CustomDataIdentifier",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html",
                "arn": "arn:${Partition}:macie2::${Account}:custom-data-identifier/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Member": {
                "name": "Member",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:member/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "FindingsFilter": {
                "name": "FindingsFilter",
                "url": "https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html",
                "arn": "arn:${Partition}:macie2::${Account}:findings-filter/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html
     */
    acceptInvitation() {
        this.add('macie2:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-archive.html
     */
    archiveFindings() {
        this.add('macie2:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more custom data identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html
     */
    batchGetCustomDataIdentifiers() {
        this.add('macie2:BatchGetCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a classification job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html
     */
    createClassificationJob() {
        this.add('macie2:CreateClassificationJob');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a custom data identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html
     */
    createCustomDataIdentifier() {
        this.add('macie2:CreateCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    createFindingsFilter() {
        this.add('macie2:CreateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to send an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    createInvitations() {
        this.add('macie2:CreateInvitations');
        return this;
    }
    /**
     * Grants permission to associate an account with an Amazon Macie master account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    createMember() {
        this.add('macie2:CreateMember');
        return this;
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html
     */
    createSampleFindings() {
        this.add('macie2:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to decline Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html
     */
    declineInvitations() {
        this.add('macie2:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    deleteCustomDataIdentifier() {
        this.add('macie2:DeleteCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    deleteFindingsFilter() {
        this.add('macie2:DeleteFindingsFilter');
        return this;
    }
    /**
     * Grants permission to delete Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html
     */
    deleteInvitations() {
        this.add('macie2:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete the association between an Amazon Macie master account and an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    deleteMember() {
        this.add('macie2:DeleteMember');
        return this;
    }
    /**
     * Grants permission to retrieve statistical and other data about S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html
     */
    describeBuckets() {
        this.add('macie2:DescribeBuckets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for a classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    describeClassificationJob() {
        this.add('macie2:DescribeClassificationJob');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    describeOrganizationConfiguration() {
        this.add('macie2:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    disableMacie() {
        this.add('macie2:DisableMacie');
        return this;
    }
    /**
     * Grants permission to disable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    disableOrganizationAdminAccount() {
        this.add('macie2:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie member account with permission to disassociate from its master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html
     */
    disassociateFromMasterAccount() {
        this.add('macie2:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to disassociate from a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html
     */
    disassociateMember() {
        this.add('macie2:DisassociateMember');
        return this;
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    enableMacie() {
        this.add('macie2:EnableMacie');
        return this;
    }
    /**
     * Grants permission to enable an account as a delegated administrator of Amazon Macie for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    enableOrganizationAdminAccount() {
        this.add('macie2:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html
     */
    getBucketStatistics() {
        this.add('macie2:GetBucketStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve the settings for exporting data classification results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    getClassificationExportConfiguration() {
        this.add('macie2:GetClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a custom data identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    getCustomDataIdentifier() {
        this.add('macie2:GetCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated statistical data about findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html
     */
    getFindingStatistics() {
        this.add('macie2:GetFindingStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve information about one or more findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html
     */
    getFindings() {
        this.add('macie2:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about the settings for a findings filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    getFindingsFilter() {
        this.add('macie2:GetFindingsFilter');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html
     */
    getInvitationsCount() {
        this.add('macie2:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    getMacieSession() {
        this.add('macie2:GetMacieSession');
        return this;
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie master account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master.html
     */
    getMasterAccount() {
        this.add('macie2:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Macie master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    getMember() {
        this.add('macie2:GetMember');
        return this;
    }
    /**
     * Grants permission to retrieve quotas and aggregated usage data for one or more accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html
     */
    getUsageStatistics() {
        this.add('macie2:GetUsageStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage.html
     */
    getUsageTotals() {
        this.add('macie2:GetUsageTotals');
        return this;
    }
    /**
     * Grants permission to retrieve information about the status and settings for one or more classification jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html
     */
    listClassificationJobs() {
        this.add('macie2:ListClassificationJobs');
        return this;
    }
    /**
     * Grants permission to retrieve information about all custom data identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html
     */
    listCustomDataIdentifiers() {
        this.add('macie2:ListCustomDataIdentifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings.html
     */
    listFindings() {
        this.add('macie2:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    listFindingsFilters() {
        this.add('macie2:ListFindingsFilters');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    listInvitations() {
        this.add('macie2:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve information about all the accounts that are associated with an Amazon Macie master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    listMembers() {
        this.add('macie2:ListMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about the delegated, Amazon Macie administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    listOrganizationAdminAccounts() {
        this.add('macie2:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Macie resource or member account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    listTagsForResources() {
        this.add('macie2:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to create or update the settings for exporting data classification results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    putClassificationExportConfiguration() {
        this.add('macie2:PutClassificationExportConfiguration');
        return this;
    }
    /**
     * Grants permission to add or update the tags for an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    tagResource() {
        this.add('macie2:TagResource');
        return this;
    }
    /**
     * Grants permission to test a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html
     */
    testCustomDataIdentifier() {
        this.add('macie2:TestCustomDataIdentifier');
        return this;
    }
    /**
     * Grants permission to reactivate (unarchive) one or more findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-unarchive.html
     */
    unarchiveFindings() {
        this.add('macie2:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from an Amazon Macie resource or member account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    untagResource() {
        this.add('macie2:UntagResource');
        return this;
    }
    /**
     * Grants permission to cancel a classification job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    updateClassificationJob() {
        this.add('macie2:UpdateClassificationJob');
        return this;
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    updateFindingsFilter() {
        this.add('macie2:UpdateFindingsFilter');
        return this;
    }
    /**
     * Grants permission to suspend or re-enable an Amazon Macie account, or update the configuration settings for a Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    updateMacieSession() {
        this.add('macie2:UpdateMacieSession');
        return this;
    }
    /**
     * Grants an Amazon Macie master account with permission to suspend or re-enable a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html
     */
    updateMemberSession() {
        this.add('macie2:UpdateMemberSession');
        return this;
    }
    /**
     * Grants permission to update Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    updateOrganizationConfiguration() {
        this.add('macie2:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ClassificationJob to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClassificationJob(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:classification-job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CustomDataIdentifier to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomDataIdentifier(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:custom-data-identifier/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Member to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMember(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:member/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type FindingsFilter to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFindingsFilter(resourceId, account, partition) {
        var arn = 'arn:${Partition}:macie2::${Account}:findings-filter/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createClassificationJob()
     * - .createCustomDataIdentifier()
     * - .createFindingsFilter()
     * - .createMember()
     * - .tagResource()
     * - .updateClassificationJob()
     * - .updateFindingsFilter()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on tag key-value pairs that are associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ClassificationJob
     * - CustomDataIdentifier
     * - Member
     * - FindingsFilter
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createClassificationJob()
     * - .createCustomDataIdentifier()
     * - .createFindingsFilter()
     * - .createMember()
     * - .tagResource()
     * - .untagResource()
     * - .updateClassificationJob()
     * - .updateFindingsFilter()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Macie2 = Macie2;
//# sourceMappingURL=data:application/json;base64,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