"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicediscovery = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicediscovery extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicediscovery';
        this.actionList = {
            "CreateHttpNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html",
                "description": "Creates an HTTP namespace.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePrivateDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html",
                "description": "Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePublicDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html",
                "description": "Creates a public namespace based on DNS, which will be visible on the internet.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html",
                "description": "Creates a service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:NamespaceArn",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html",
                "description": "Deletes a specified namespace.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "DeleteService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html",
                "description": "Deletes a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "DeregisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html",
                "description": "Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "DiscoverInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html",
                "description": "Discovers registered instances for a specified namespace and service.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:NamespaceName",
                    "servicediscovery:ServiceName"
                ]
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html",
                "description": "Gets information about a specified instance.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetInstancesHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html",
                "description": "Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html",
                "description": "Gets information about a namespace.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html",
                "description": "Gets information about a specific operation.",
                "accessLevel": "Read"
            },
            "GetService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html",
                "description": "Gets the settings for a specified service.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "ListInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html",
                "description": "Gets summary information about the instances that were registered with a specified service.",
                "accessLevel": "List",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "ListNamespaces": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html",
                "description": "Gets information about the namespaces.",
                "accessLevel": "List"
            },
            "ListOperations": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html",
                "description": "Lists operations that match the criteria that you specify.",
                "accessLevel": "List"
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html",
                "description": "Gets settings for all the services that match specified filters.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html",
                "description": "Lists tags for the specified resource.",
                "accessLevel": "List"
            },
            "RegisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html",
                "description": "Registers an instance based on the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html",
                "description": "Adds one or more tags to the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html",
                "description": "Removes one or more tags from the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateInstanceCustomHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html",
                "description": "Updates the current health status for an instance that has a custom health check.",
                "accessLevel": "Write",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "UpdateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html",
                "description": "Updates the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "namespace": {
                "name": "namespace",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "service": {
                "name": "service",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an HTTP namespace.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html
     */
    createHttpNamespace() {
        this.add('servicediscovery:CreateHttpNamespace');
        return this;
    }
    /**
     * Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html
     */
    createPrivateDnsNamespace() {
        this.add('servicediscovery:CreatePrivateDnsNamespace');
        return this;
    }
    /**
     * Creates a public namespace based on DNS, which will be visible on the internet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html
     */
    createPublicDnsNamespace() {
        this.add('servicediscovery:CreatePublicDnsNamespace');
        return this;
    }
    /**
     * Creates a service.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespaceArn()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html
     */
    createService() {
        this.add('servicediscovery:CreateService');
        return this;
    }
    /**
     * Deletes a specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html
     */
    deleteNamespace() {
        this.add('servicediscovery:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html
     */
    deleteService() {
        this.add('servicediscovery:DeleteService');
        return this;
    }
    /**
     * Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html
     */
    deregisterInstance() {
        this.add('servicediscovery:DeregisterInstance');
        return this;
    }
    /**
     * Discovers registered instances for a specified namespace and service.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespaceName()
     * - .ifServiceName()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html
     */
    discoverInstances() {
        this.add('servicediscovery:DiscoverInstances');
        return this;
    }
    /**
     * Gets information about a specified instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html
     */
    getInstance() {
        this.add('servicediscovery:GetInstance');
        return this;
    }
    /**
     * Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html
     */
    getInstancesHealthStatus() {
        this.add('servicediscovery:GetInstancesHealthStatus');
        return this;
    }
    /**
     * Gets information about a namespace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html
     */
    getNamespace() {
        this.add('servicediscovery:GetNamespace');
        return this;
    }
    /**
     * Gets information about a specific operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html
     */
    getOperation() {
        this.add('servicediscovery:GetOperation');
        return this;
    }
    /**
     * Gets the settings for a specified service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html
     */
    getService() {
        this.add('servicediscovery:GetService');
        return this;
    }
    /**
     * Gets summary information about the instances that were registered with a specified service.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html
     */
    listInstances() {
        this.add('servicediscovery:ListInstances');
        return this;
    }
    /**
     * Gets information about the namespaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html
     */
    listNamespaces() {
        this.add('servicediscovery:ListNamespaces');
        return this;
    }
    /**
     * Lists operations that match the criteria that you specify.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html
     */
    listOperations() {
        this.add('servicediscovery:ListOperations');
        return this;
    }
    /**
     * Gets settings for all the services that match specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html
     */
    listServices() {
        this.add('servicediscovery:ListServices');
        return this;
    }
    /**
     * Lists tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('servicediscovery:ListTagsForResource');
        return this;
    }
    /**
     * Registers an instance based on the settings in a specified service.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html
     */
    registerInstance() {
        this.add('servicediscovery:RegisterInstance');
        return this;
    }
    /**
     * Adds one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('servicediscovery:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('servicediscovery:UntagResource');
        return this;
    }
    /**
     * Updates the current health status for an instance that has a custom health check.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html
     */
    updateInstanceCustomHealthStatus() {
        this.add('servicediscovery:UpdateInstanceCustomHealthStatus');
        return this;
    }
    /**
     * Updates the settings in a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html
     */
    updateService() {
        this.add('servicediscovery:UpdateService');
        return this;
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html
     *
     * @param namespaceId - Identifier for the namespaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNamespace(namespaceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}';
        arn = arn.replace('${NamespaceId}', namespaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onService(serviceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}';
        arn = arn.replace('${ServiceId}', serviceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createHttpNamespace()
     * - .createPrivateDnsNamespace()
     * - .createPublicDnsNamespace()
     * - .createService()
     * - .tagResource()
     * - .untagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - namespace
     * - service
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createHttpNamespace()
     * - .createPrivateDnsNamespace()
     * - .createPublicDnsNamespace()
     * - .createService()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .createService()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceArn(value, operator) {
        return this.if(`servicediscovery:NamespaceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .discoverInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceName(value, operator) {
        return this.if(`servicediscovery:NamespaceName`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .deregisterInstance()
     * - .getInstance()
     * - .getInstancesHealthStatus()
     * - .listInstances()
     * - .registerInstance()
     * - .updateInstanceCustomHealthStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`servicediscovery:ServiceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .discoverInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceName(value, operator) {
        return this.if(`servicediscovery:ServiceName`, value, operator || 'StringLike');
    }
}
exports.Servicediscovery = Servicediscovery;
//# sourceMappingURL=data:application/json;base64,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