"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodestarConnections = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarconnections.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodestarConnections extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarconnections.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codestar-connections';
        this.actionList = {
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateConnection.html",
                "description": "Grants permission to create a Connection resource",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-connections:ProviderType"
                ]
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteConnection.html",
                "description": "Grants permission to delete a Connection resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                }
            },
            "GetConnection": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetConnection.html",
                "description": "Grants permission to get details about a Connection resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                }
            },
            "GetIndividualAccessToken": {
                "url": "",
                "description": "Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection",
                "accessLevel": "Read",
                "dependentActions": [
                    "codestar-connections:StartOAuthHandshake"
                ],
                "conditions": [
                    "codestar-connections:ProviderType"
                ]
            },
            "GetInstallationUrl": {
                "url": "",
                "description": "Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection",
                "accessLevel": "Read",
                "conditions": [
                    "codestar-connections:ProviderType"
                ]
            },
            "ListConnections": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListConnections.html",
                "description": "Grants permission to list Connection resources",
                "accessLevel": "List",
                "conditions": [
                    "codestar-connections:ProviderTypeFilter"
                ]
            },
            "ListInstallationTargets": {
                "url": "",
                "description": "Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection",
                "accessLevel": "List",
                "dependentActions": [
                    "codestar-connections:GetIndividualAccessToken",
                    "codestar-connections:StartOAuthHandshake"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListTagsForResource.html",
                "description": "Gets the set of key-value pairs that are used to manage the resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                }
            },
            "PassConnection": {
                "url": "",
                "description": "Grants permission to pass a Connection resource to an AWS service that accepts a Connection ARN as input, such as codepipeline:CreatePipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                },
                "conditions": [
                    "codestar-connections:PassedToService"
                ]
            },
            "StartOAuthHandshake": {
                "url": "",
                "description": "Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection",
                "accessLevel": "Read",
                "conditions": [
                    "codestar-connections:ProviderType"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_TagResource.html",
                "description": "Adds to or modifies the tags of the given resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from an AWS resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateConnectionInstallation": {
                "url": "",
                "description": "Grants permission to update a Connection resource with an installation of the CodeStar Connections App",
                "accessLevel": "Write",
                "dependentActions": [
                    "codestar-connections:GetIndividualAccessToken",
                    "codestar-connections:GetInstallationUrl",
                    "codestar-connections:ListInstallationTargets",
                    "codestar-connections:StartOAuthHandshake"
                ],
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                },
                "conditions": [
                    "codestar-connections:InstallationId"
                ]
            },
            "UseConnection": {
                "url": "",
                "description": "Grants permission to use a Connection resource to call provider actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Connection": {
                        "required": true
                    }
                },
                "conditions": [
                    "codestar-connections:FullRepositoryId",
                    "codestar-connections:ProviderAction",
                    "codestar-connections:ProviderPermissionsRequired"
                ]
            }
        };
        this.resourceTypes = {
            "Connection": {
                "name": "Connection",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/connections.html/API_Connection.html",
                "arn": "arn:${Partition}:codestar-connections:${Region}:${Account}:connection/${ConnectionId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create a Connection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateConnection.html
     */
    createConnection() {
        this.add('codestar-connections:CreateConnection');
        return this;
    }
    /**
     * Grants permission to delete a Connection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteConnection.html
     */
    deleteConnection() {
        this.add('codestar-connections:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to get details about a Connection resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetConnection.html
     */
    getConnection() {
        this.add('codestar-connections:GetConnection');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * Dependent actions:
     * - codestar-connections:StartOAuthHandshake
     */
    getIndividualAccessToken() {
        this.add('codestar-connections:GetIndividualAccessToken');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     */
    getInstallationUrl() {
        this.add('codestar-connections:GetInstallationUrl');
        return this;
    }
    /**
     * Grants permission to list Connection resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListConnections.html
     */
    listConnections() {
        this.add('codestar-connections:ListConnections');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: List
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:StartOAuthHandshake
     */
    listInstallationTargets() {
        this.add('codestar-connections:ListInstallationTargets');
        return this;
    }
    /**
     * Gets the set of key-value pairs that are used to manage the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('codestar-connections:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to pass a Connection resource to an AWS service that accepts a Connection ARN as input, such as codepipeline:CreatePipeline
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     */
    passConnection() {
        this.add('codestar-connections:PassConnection');
        return this;
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     */
    startOAuthHandshake() {
        this.add('codestar-connections:StartOAuthHandshake');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('codestar-connections:TagResource');
        return this;
    }
    /**
     * Removes tags from an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('codestar-connections:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a Connection resource with an installation of the CodeStar Connections App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstallationId()
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:GetInstallationUrl
     * - codestar-connections:ListInstallationTargets
     * - codestar-connections:StartOAuthHandshake
     */
    updateConnectionInstallation() {
        this.add('codestar-connections:UpdateConnectionInstallation');
        return this;
    }
    /**
     * Grants permission to use a Connection resource to call provider actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFullRepositoryId()
     * - .ifProviderAction()
     * - .ifProviderPermissionsRequired()
     */
    useConnection() {
        this.add('codestar-connections:UseConnection');
        return this;
    }
    /**
     * Adds a resource of type Connection to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections.html/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar-connections:${Region}:${Account}:connection/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createConnection()
     * - .tagResource()
     * - .untagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createConnection()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the branch name that is passed in the request. Applies only to UseConnection requests for access to a specific repository branch
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranchName(value, operator) {
        return this.if(`codestar-connections:BranchName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the repository that is passed in the request. Applies only to UseConnection requests for access to a specific repository
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * Applies to actions:
     * - .useConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFullRepositoryId(value, operator) {
        return this.if(`codestar-connections:FullRepositoryId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the third-party ID (such as the Bitbucket App installation ID for CodeStar Connections) that is used to update a Connection. Allows you to restrict which third-party App installations can be used to make a Connection
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions-handshake
     *
     * Applies to actions:
     * - .updateConnectionInstallation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstallationId(value, operator) {
        return this.if(`codestar-connections:InstallationId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the third-party repository. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwnerId(value, operator) {
        return this.if(`codestar-connections:OwnerId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service to which the principal is allowed to pass a Connection
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-passconnection
     *
     * Applies to actions:
     * - .passConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`codestar-connections:PassedToService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the provider action in a UseConnection request such as ListRepositories. See documentation for all valid values
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use-provider
     *
     * Applies to actions:
     * - .useConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderAction(value, operator) {
        return this.if(`codestar-connections:ProviderAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the write permissions of a provider action in a UseConnection request. Valid types include read_only and read_write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * Applies to actions:
     * - .useConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderPermissionsRequired(value, operator) {
        return this.if(`codestar-connections:ProviderPermissionsRequired`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider passed in the request
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions
     *
     * Applies to actions:
     * - .createConnection()
     * - .getIndividualAccessToken()
     * - .getInstallationUrl()
     * - .startOAuthHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderType(value, operator) {
        return this.if(`codestar-connections:ProviderType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider used to filter results
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-permissions-actions
     *
     * Applies to actions:
     * - .listConnections()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderTypeFilter(value, operator) {
        return this.if(`codestar-connections:ProviderTypeFilter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the repository name that is passed in the request. Applies only to UseConnection requests for creating new repositories
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/connections-permissions.html#connections-use
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepositoryName(value, operator) {
        return this.if(`codestar-connections:RepositoryName`, value, operator || 'StringLike');
    }
}
exports.CodestarConnections = CodestarConnections;
//# sourceMappingURL=data:application/json;base64,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