"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Config = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Config extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'config';
        this.actionList = {
            "BatchGetAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html",
                "description": "Returns the current configuration items for resources that are present in your AWS Config aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "BatchGetResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html",
                "description": "Returns the current configuration for one or more requested resources",
                "accessLevel": "Read"
            },
            "DeleteAggregationAuthorization": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html",
                "description": "Deletes the authorization granted to the specified configuration aggregator account in a specified region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": true
                    }
                }
            },
            "DeleteConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html",
                "description": "Deletes the specified AWS Config rule and all of its evaluation results",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DeleteConfigurationAggregator": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html",
                "description": "Deletes the specified configuration aggregator and the aggregated data associated with the aggregator",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DeleteConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html",
                "description": "Deletes the configuration recorder",
                "accessLevel": "Write"
            },
            "DeleteConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html",
                "description": "Deletes the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack.",
                "accessLevel": "Write"
            },
            "DeleteDeliveryChannel": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html",
                "description": "Deletes the delivery channel",
                "accessLevel": "Write"
            },
            "DeleteEvaluationResults": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html",
                "description": "Deletes the evaluation results for the specified Config rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DeleteOrganizationConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html",
                "description": "Deletes the specified organization config rule and all of its evaluation results from all member accounts in that organization.",
                "accessLevel": "Write"
            },
            "DeleteOrganizationConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html",
                "description": "Deletes the specified organization conformance pack and all of its evaluation results from all member accounts in that organization.",
                "accessLevel": "Write"
            },
            "DeletePendingAggregationRequest": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html",
                "description": "Deletes pending authorization requests for a specified aggregator account in a specified region",
                "accessLevel": "Write"
            },
            "DeleteRemediationConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html",
                "description": "Deletes the remediation configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html",
                "description": "Deletes one or more remediation exceptions for specific resource keys for a specific AWS Config Rule.",
                "accessLevel": "Write"
            },
            "DeleteRetentionConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html",
                "description": "Deletes the retention configuration",
                "accessLevel": "Write"
            },
            "DeliverConfigSnapshot": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html",
                "description": "Schedules delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel",
                "accessLevel": "Read"
            },
            "DescribeAggregateComplianceByConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html",
                "description": "Returns a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DescribeAggregationAuthorizations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html",
                "description": "Returns a list of authorizations granted to various aggregator accounts and regions",
                "accessLevel": "List"
            },
            "DescribeComplianceByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html",
                "description": "Indicates whether the specified AWS Config rules are compliant",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeComplianceByResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html",
                "description": "Indicates whether the specified AWS resources are compliant",
                "accessLevel": "List"
            },
            "DescribeConfigRuleEvaluationStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html",
                "description": "Returns status information for each of your AWS managed Config rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html",
                "description": "Returns details about your AWS Config rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeConfigurationAggregatorSourcesStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html",
                "description": "Returns status information for sources within an aggregator",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DescribeConfigurationAggregators": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html",
                "description": "Returns the details of one or more configuration aggregators",
                "accessLevel": "List"
            },
            "DescribeConfigurationRecorderStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html",
                "description": "Returns the current status of the specified configuration recorder",
                "accessLevel": "List"
            },
            "DescribeConfigurationRecorders": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html",
                "description": "Returns the name of one or more specified configuration recorders",
                "accessLevel": "List"
            },
            "DescribeConformancePackCompliance": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html",
                "description": "Returns compliance information for each rule in that conformance pack.",
                "accessLevel": "Read"
            },
            "DescribeConformancePackStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html",
                "description": "Provides one or more conformance packs deployment status.",
                "accessLevel": "Read"
            },
            "DescribeConformancePacks": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html",
                "description": "Returns a list of one or more conformance packs.",
                "accessLevel": "Read"
            },
            "DescribeDeliveryChannelStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html",
                "description": "Returns the current status of the specified delivery channel",
                "accessLevel": "List"
            },
            "DescribeDeliveryChannels": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html",
                "description": "Returns details about the specified delivery channel",
                "accessLevel": "List"
            },
            "DescribeOrganizationConfigRuleStatuses": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html",
                "description": "Provides organization config rule deployment status for an organization.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html",
                "description": "Returns a list of organization config rules.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConformancePackStatuses": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html",
                "description": "Provides organization conformance pack deployment status for an organization.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConformancePacks": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html",
                "description": "Returns a list of organization conformance packs.",
                "accessLevel": "Read"
            },
            "DescribePendingAggregationRequests": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html",
                "description": "Returns a list of all pending aggregation requests",
                "accessLevel": "List"
            },
            "DescribeRemediationConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html",
                "description": "Returns the details of one or more remediation configurations",
                "accessLevel": "List",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DescribeRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html",
                "description": "Returns the details of one or more remediation exceptions.",
                "accessLevel": "List"
            },
            "DescribeRemediationExecutionStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html",
                "description": "Provides a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed",
                "accessLevel": "List",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DescribeRetentionConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html",
                "description": "Returns the details of one or more retention configurations",
                "accessLevel": "List"
            },
            "GetAggregateComplianceDetailsByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html",
                "description": "Returns the evaluation results for the specified AWS Config rule for a specific resource in a rule",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateConfigRuleComplianceSummary": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html",
                "description": "Returns the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateDiscoveredResourceCounts": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html",
                "description": "Returns the resource counts across accounts and regions that are present in your AWS Config aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html",
                "description": "Returns configuration item that is aggregated for your specific resource in a specific source account and region",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetComplianceDetailsByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html",
                "description": "Returns the evaluation results for the specified AWS Config rule",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "GetComplianceDetailsByResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html",
                "description": "Returns the evaluation results for the specified AWS resource",
                "accessLevel": "Read"
            },
            "GetComplianceSummaryByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html",
                "description": "Returns the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each",
                "accessLevel": "Read"
            },
            "GetComplianceSummaryByResourceType": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html",
                "description": "Returns the number of resources that are compliant and the number that are noncompliant",
                "accessLevel": "Read"
            },
            "GetConformancePackComplianceDetails": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html",
                "description": "Returns compliance details of a conformance pack for all AWS resources that are monitered by conformance pack.",
                "accessLevel": "Read"
            },
            "GetConformancePackComplianceSummary": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html",
                "description": "Provides compliance summary for one or more conformance packs.",
                "accessLevel": "Read"
            },
            "GetDiscoveredResourceCounts": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html",
                "description": "Returns the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account",
                "accessLevel": "Read"
            },
            "GetOrganizationConfigRuleDetailedStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html",
                "description": "Returns detailed status for each member account within an organization for a given organization config rule.",
                "accessLevel": "Read"
            },
            "GetOrganizationConformancePackDetailedStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html",
                "description": "Returns detailed status for each member account within an organization for a given organization conformance pack.",
                "accessLevel": "Read"
            },
            "GetResourceConfigHistory": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html",
                "description": "Returns a list of configuration items for the specified resource",
                "accessLevel": "Read"
            },
            "ListAggregateDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html",
                "description": "Accepts a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "ListDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html",
                "description": "Accepts a resource type and returns a list of resource identifiers for the resources of that type",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html",
                "description": "List the tags for AWS Config resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    }
                }
            },
            "PutAggregationAuthorization": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html",
                "description": "Authorizes the aggregator account and region to collect data from the source account and region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html",
                "description": "Adds or updates an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigurationAggregator": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html",
                "description": "Creates and updates the configuration aggregator with the selected source accounts and regions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html",
                "description": "Creates a new configuration recorder to record the selected resource configurations",
                "accessLevel": "Write"
            },
            "PutConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html",
                "description": "Creates or updates a conformance pack.",
                "accessLevel": "Write"
            },
            "PutDeliveryChannel": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html",
                "description": "Creates a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic",
                "accessLevel": "Write"
            },
            "PutEvaluations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html",
                "description": "Used by an AWS Lambda function to deliver evaluation results to AWS Config",
                "accessLevel": "Write"
            },
            "PutOrganizationConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html",
                "description": "Adds or updates organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations.",
                "accessLevel": "Write"
            },
            "PutOrganizationConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html",
                "description": "Adds or updates organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations.",
                "accessLevel": "Write"
            },
            "PutRemediationConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html",
                "description": "Adds or updates the remediation configuration with a specific AWS Config rule with the selected target or action",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "PutRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html",
                "description": "Adds or updates remediation exceptions for specific resources for a specific AWS Config rule.",
                "accessLevel": "Write"
            },
            "PutRetentionConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html",
                "description": "Creates and updates the retention configuration with details about retention period (number of days) that AWS Config stores your historical information",
                "accessLevel": "Write"
            },
            "SelectAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html",
                "description": "Accepts a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties.",
                "accessLevel": "Read"
            },
            "SelectResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html",
                "description": "Accepts a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties",
                "accessLevel": "Read"
            },
            "StartConfigRulesEvaluation": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html",
                "description": "Evaluates your resources against the specified Config rules",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "StartConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html",
                "description": "Starts recording configurations of the AWS resources you have selected to record in your AWS account",
                "accessLevel": "Write"
            },
            "StartRemediationExecution": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html",
                "description": "Runs an on-demand remediation for the specified AWS Config rules against the last known remediation configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "StopConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html",
                "description": "Stops recording configurations of the AWS resources you have selected to record in your AWS account",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html",
                "description": "Associates the specified tags to a resource with the specified resourceArn",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    },
                    "ConformancePack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html",
                "description": "Deletes specified tags from a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    },
                    "ConformancePack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "AggregationAuthorization": {
                "name": "AggregationAuthorization",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_AggregationAuthorization.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:aggregation-authorization/${AggregatorAccount}/${AggregatorRegion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConfigurationAggregator": {
                "name": "ConfigurationAggregator",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigurationAggregator.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:config-aggregator/${AggregatorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConfigRule": {
                "name": "ConfigRule",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigRule.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:config-rule/${ConfigRuleId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConformancePack": {
                "name": "ConformancePack",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConformancePack.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:conformance-pack/${ConformancePackName}/${ConformancePackId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "OrganizationConfigRule": {
                "name": "OrganizationConfigRule",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConfigRule.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:organization-config-rule/${OrganizationConfigRuleId}",
                "conditionKeys": []
            },
            "OrganizationConformancePack": {
                "name": "OrganizationConformancePack",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConformancePack.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:organization-conformance-pack/${OrganizationConformancePackId}",
                "conditionKeys": []
            },
            "RemediationConfiguration": {
                "name": "RemediationConfiguration",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_RemediationConfiguration.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:remediation-configuration/${RemediationConfigurationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Returns the current configuration items for resources that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html
     */
    batchGetAggregateResourceConfig() {
        this.add('config:BatchGetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the current configuration for one or more requested resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html
     */
    batchGetResourceConfig() {
        this.add('config:BatchGetResourceConfig');
        return this;
    }
    /**
     * Deletes the authorization granted to the specified configuration aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html
     */
    deleteAggregationAuthorization() {
        this.add('config:DeleteAggregationAuthorization');
        return this;
    }
    /**
     * Deletes the specified AWS Config rule and all of its evaluation results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html
     */
    deleteConfigRule() {
        this.add('config:DeleteConfigRule');
        return this;
    }
    /**
     * Deletes the specified configuration aggregator and the aggregated data associated with the aggregator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html
     */
    deleteConfigurationAggregator() {
        this.add('config:DeleteConfigurationAggregator');
        return this;
    }
    /**
     * Deletes the configuration recorder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html
     */
    deleteConfigurationRecorder() {
        this.add('config:DeleteConfigurationRecorder');
        return this;
    }
    /**
     * Deletes the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html
     */
    deleteConformancePack() {
        this.add('config:DeleteConformancePack');
        return this;
    }
    /**
     * Deletes the delivery channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html
     */
    deleteDeliveryChannel() {
        this.add('config:DeleteDeliveryChannel');
        return this;
    }
    /**
     * Deletes the evaluation results for the specified Config rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html
     */
    deleteEvaluationResults() {
        this.add('config:DeleteEvaluationResults');
        return this;
    }
    /**
     * Deletes the specified organization config rule and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html
     */
    deleteOrganizationConfigRule() {
        this.add('config:DeleteOrganizationConfigRule');
        return this;
    }
    /**
     * Deletes the specified organization conformance pack and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html
     */
    deleteOrganizationConformancePack() {
        this.add('config:DeleteOrganizationConformancePack');
        return this;
    }
    /**
     * Deletes pending authorization requests for a specified aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html
     */
    deletePendingAggregationRequest() {
        this.add('config:DeletePendingAggregationRequest');
        return this;
    }
    /**
     * Deletes the remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html
     */
    deleteRemediationConfiguration() {
        this.add('config:DeleteRemediationConfiguration');
        return this;
    }
    /**
     * Deletes one or more remediation exceptions for specific resource keys for a specific AWS Config Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html
     */
    deleteRemediationExceptions() {
        this.add('config:DeleteRemediationExceptions');
        return this;
    }
    /**
     * Deletes the retention configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html
     */
    deleteRetentionConfiguration() {
        this.add('config:DeleteRetentionConfiguration');
        return this;
    }
    /**
     * Schedules delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html
     */
    deliverConfigSnapshot() {
        this.add('config:DeliverConfigSnapshot');
        return this;
    }
    /**
     * Returns a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html
     */
    describeAggregateComplianceByConfigRules() {
        this.add('config:DescribeAggregateComplianceByConfigRules');
        return this;
    }
    /**
     * Returns a list of authorizations granted to various aggregator accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html
     */
    describeAggregationAuthorizations() {
        this.add('config:DescribeAggregationAuthorizations');
        return this;
    }
    /**
     * Indicates whether the specified AWS Config rules are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html
     */
    describeComplianceByConfigRule() {
        this.add('config:DescribeComplianceByConfigRule');
        return this;
    }
    /**
     * Indicates whether the specified AWS resources are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html
     */
    describeComplianceByResource() {
        this.add('config:DescribeComplianceByResource');
        return this;
    }
    /**
     * Returns status information for each of your AWS managed Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html
     */
    describeConfigRuleEvaluationStatus() {
        this.add('config:DescribeConfigRuleEvaluationStatus');
        return this;
    }
    /**
     * Returns details about your AWS Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html
     */
    describeConfigRules() {
        this.add('config:DescribeConfigRules');
        return this;
    }
    /**
     * Returns status information for sources within an aggregator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html
     */
    describeConfigurationAggregatorSourcesStatus() {
        this.add('config:DescribeConfigurationAggregatorSourcesStatus');
        return this;
    }
    /**
     * Returns the details of one or more configuration aggregators
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html
     */
    describeConfigurationAggregators() {
        this.add('config:DescribeConfigurationAggregators');
        return this;
    }
    /**
     * Returns the current status of the specified configuration recorder
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html
     */
    describeConfigurationRecorderStatus() {
        this.add('config:DescribeConfigurationRecorderStatus');
        return this;
    }
    /**
     * Returns the name of one or more specified configuration recorders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html
     */
    describeConfigurationRecorders() {
        this.add('config:DescribeConfigurationRecorders');
        return this;
    }
    /**
     * Returns compliance information for each rule in that conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html
     */
    describeConformancePackCompliance() {
        this.add('config:DescribeConformancePackCompliance');
        return this;
    }
    /**
     * Provides one or more conformance packs deployment status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html
     */
    describeConformancePackStatus() {
        this.add('config:DescribeConformancePackStatus');
        return this;
    }
    /**
     * Returns a list of one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html
     */
    describeConformancePacks() {
        this.add('config:DescribeConformancePacks');
        return this;
    }
    /**
     * Returns the current status of the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html
     */
    describeDeliveryChannelStatus() {
        this.add('config:DescribeDeliveryChannelStatus');
        return this;
    }
    /**
     * Returns details about the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html
     */
    describeDeliveryChannels() {
        this.add('config:DescribeDeliveryChannels');
        return this;
    }
    /**
     * Provides organization config rule deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html
     */
    describeOrganizationConfigRuleStatuses() {
        this.add('config:DescribeOrganizationConfigRuleStatuses');
        return this;
    }
    /**
     * Returns a list of organization config rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html
     */
    describeOrganizationConfigRules() {
        this.add('config:DescribeOrganizationConfigRules');
        return this;
    }
    /**
     * Provides organization conformance pack deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html
     */
    describeOrganizationConformancePackStatuses() {
        this.add('config:DescribeOrganizationConformancePackStatuses');
        return this;
    }
    /**
     * Returns a list of organization conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html
     */
    describeOrganizationConformancePacks() {
        this.add('config:DescribeOrganizationConformancePacks');
        return this;
    }
    /**
     * Returns a list of all pending aggregation requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html
     */
    describePendingAggregationRequests() {
        this.add('config:DescribePendingAggregationRequests');
        return this;
    }
    /**
     * Returns the details of one or more remediation configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html
     */
    describeRemediationConfigurations() {
        this.add('config:DescribeRemediationConfigurations');
        return this;
    }
    /**
     * Returns the details of one or more remediation exceptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html
     */
    describeRemediationExceptions() {
        this.add('config:DescribeRemediationExceptions');
        return this;
    }
    /**
     * Provides a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html
     */
    describeRemediationExecutionStatus() {
        this.add('config:DescribeRemediationExecutionStatus');
        return this;
    }
    /**
     * Returns the details of one or more retention configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html
     */
    describeRetentionConfigurations() {
        this.add('config:DescribeRetentionConfigurations');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule for a specific resource in a rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html
     */
    getAggregateComplianceDetailsByConfigRule() {
        this.add('config:GetAggregateComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html
     */
    getAggregateConfigRuleComplianceSummary() {
        this.add('config:GetAggregateConfigRuleComplianceSummary');
        return this;
    }
    /**
     * Returns the resource counts across accounts and regions that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html
     */
    getAggregateDiscoveredResourceCounts() {
        this.add('config:GetAggregateDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns configuration item that is aggregated for your specific resource in a specific source account and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html
     */
    getAggregateResourceConfig() {
        this.add('config:GetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html
     */
    getComplianceDetailsByConfigRule() {
        this.add('config:GetComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html
     */
    getComplianceDetailsByResource() {
        this.add('config:GetComplianceDetailsByResource');
        return this;
    }
    /**
     * Returns the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html
     */
    getComplianceSummaryByConfigRule() {
        this.add('config:GetComplianceSummaryByConfigRule');
        return this;
    }
    /**
     * Returns the number of resources that are compliant and the number that are noncompliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html
     */
    getComplianceSummaryByResourceType() {
        this.add('config:GetComplianceSummaryByResourceType');
        return this;
    }
    /**
     * Returns compliance details of a conformance pack for all AWS resources that are monitered by conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html
     */
    getConformancePackComplianceDetails() {
        this.add('config:GetConformancePackComplianceDetails');
        return this;
    }
    /**
     * Provides compliance summary for one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html
     */
    getConformancePackComplianceSummary() {
        this.add('config:GetConformancePackComplianceSummary');
        return this;
    }
    /**
     * Returns the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html
     */
    getDiscoveredResourceCounts() {
        this.add('config:GetDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization config rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html
     */
    getOrganizationConfigRuleDetailedStatus() {
        this.add('config:GetOrganizationConfigRuleDetailedStatus');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html
     */
    getOrganizationConformancePackDetailedStatus() {
        this.add('config:GetOrganizationConformancePackDetailedStatus');
        return this;
    }
    /**
     * Returns a list of configuration items for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html
     */
    getResourceConfigHistory() {
        this.add('config:GetResourceConfigHistory');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html
     */
    listAggregateDiscoveredResources() {
        this.add('config:ListAggregateDiscoveredResources');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers for the resources of that type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html
     */
    listDiscoveredResources() {
        this.add('config:ListDiscoveredResources');
        return this;
    }
    /**
     * List the tags for AWS Config resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('config:ListTagsForResource');
        return this;
    }
    /**
     * Authorizes the aggregator account and region to collect data from the source account and region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html
     */
    putAggregationAuthorization() {
        this.add('config:PutAggregationAuthorization');
        return this;
    }
    /**
     * Adds or updates an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html
     */
    putConfigRule() {
        this.add('config:PutConfigRule');
        return this;
    }
    /**
     * Creates and updates the configuration aggregator with the selected source accounts and regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html
     */
    putConfigurationAggregator() {
        this.add('config:PutConfigurationAggregator');
        return this;
    }
    /**
     * Creates a new configuration recorder to record the selected resource configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html
     */
    putConfigurationRecorder() {
        this.add('config:PutConfigurationRecorder');
        return this;
    }
    /**
     * Creates or updates a conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html
     */
    putConformancePack() {
        this.add('config:PutConformancePack');
        return this;
    }
    /**
     * Creates a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html
     */
    putDeliveryChannel() {
        this.add('config:PutDeliveryChannel');
        return this;
    }
    /**
     * Used by an AWS Lambda function to deliver evaluation results to AWS Config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html
     */
    putEvaluations() {
        this.add('config:PutEvaluations');
        return this;
    }
    /**
     * Adds or updates organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html
     */
    putOrganizationConfigRule() {
        this.add('config:PutOrganizationConfigRule');
        return this;
    }
    /**
     * Adds or updates organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html
     */
    putOrganizationConformancePack() {
        this.add('config:PutOrganizationConformancePack');
        return this;
    }
    /**
     * Adds or updates the remediation configuration with a specific AWS Config rule with the selected target or action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html
     */
    putRemediationConfigurations() {
        this.add('config:PutRemediationConfigurations');
        return this;
    }
    /**
     * Adds or updates remediation exceptions for specific resources for a specific AWS Config rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html
     */
    putRemediationExceptions() {
        this.add('config:PutRemediationExceptions');
        return this;
    }
    /**
     * Creates and updates the retention configuration with details about retention period (number of days) that AWS Config stores your historical information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html
     */
    putRetentionConfiguration() {
        this.add('config:PutRetentionConfiguration');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html
     */
    selectAggregateResourceConfig() {
        this.add('config:SelectAggregateResourceConfig');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html
     */
    selectResourceConfig() {
        this.add('config:SelectResourceConfig');
        return this;
    }
    /**
     * Evaluates your resources against the specified Config rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html
     */
    startConfigRulesEvaluation() {
        this.add('config:StartConfigRulesEvaluation');
        return this;
    }
    /**
     * Starts recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html
     */
    startConfigurationRecorder() {
        this.add('config:StartConfigurationRecorder');
        return this;
    }
    /**
     * Runs an on-demand remediation for the specified AWS Config rules against the last known remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html
     */
    startRemediationExecution() {
        this.add('config:StartRemediationExecution');
        return this;
    }
    /**
     * Stops recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html
     */
    stopConfigurationRecorder() {
        this.add('config:StopConfigurationRecorder');
        return this;
    }
    /**
     * Associates the specified tags to a resource with the specified resourceArn
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('config:TagResource');
        return this;
    }
    /**
     * Deletes specified tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('config:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type AggregationAuthorization to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_AggregationAuthorization.html
     *
     * @param aggregatorAccount - Identifier for the aggregatorAccount.
     * @param aggregatorRegion - Identifier for the aggregatorRegion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAggregationAuthorization(aggregatorAccount, aggregatorRegion, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:aggregation-authorization/${AggregatorAccount}/${AggregatorRegion}';
        arn = arn.replace('${AggregatorAccount}', aggregatorAccount);
        arn = arn.replace('${AggregatorRegion}', aggregatorRegion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigurationAggregator to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigurationAggregator.html
     *
     * @param aggregatorId - Identifier for the aggregatorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationAggregator(aggregatorId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-aggregator/${AggregatorId}';
        arn = arn.replace('${AggregatorId}', aggregatorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigRule.html
     *
     * @param configRuleId - Identifier for the configRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigRule(configRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-rule/${ConfigRuleId}';
        arn = arn.replace('${ConfigRuleId}', configRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConformancePack.html
     *
     * @param conformancePackName - Identifier for the conformancePackName.
     * @param conformancePackId - Identifier for the conformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConformancePack(conformancePackName, conformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:conformance-pack/${ConformancePackName}/${ConformancePackId}';
        arn = arn.replace('${ConformancePackName}', conformancePackName);
        arn = arn.replace('${ConformancePackId}', conformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConfigRule.html
     *
     * @param organizationConfigRuleId - Identifier for the organizationConfigRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConfigRule(organizationConfigRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-config-rule/${OrganizationConfigRuleId}';
        arn = arn.replace('${OrganizationConfigRuleId}', organizationConfigRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConformancePack.html
     *
     * @param organizationConformancePackId - Identifier for the organizationConformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConformancePack(organizationConformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-conformance-pack/${OrganizationConformancePackId}';
        arn = arn.replace('${OrganizationConformancePackId}', organizationConformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RemediationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_RemediationConfiguration.html
     *
     * @param remediationConfigurationId - Identifier for the remediationConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRemediationConfiguration(remediationConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:remediation-configuration/${RemediationConfigurationId}';
        arn = arn.replace('${RemediationConfigurationId}', remediationConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .putAggregationAuthorization()
     * - .putConfigRule()
     * - .putConfigurationAggregator()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - AggregationAuthorization
     * - ConfigurationAggregator
     * - ConfigRule
     * - ConformancePack
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .putAggregationAuthorization()
     * - .putConfigRule()
     * - .putConfigurationAggregator()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Config = Config;
//# sourceMappingURL=data:application/json;base64,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