"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kms';
        this.actionList = {
            "CancelKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html",
                "description": "Controls permission to cancel the scheduled deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ConnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html",
                "description": "Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html",
                "description": "Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "CreateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html",
                "description": "Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.",
                "accessLevel": "Write",
                "dependentActions": [
                    "cloudhsm:DescribeClusters"
                ]
            },
            "CreateGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html",
                "description": "Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantConstraintType",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "CreateKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html",
                "description": "Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.",
                "accessLevel": "Write",
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CustomerMasterKeySpec",
                    "kms:CustomerMasterKeyUsage",
                    "kms:KeyOrigin"
                ]
            },
            "Decrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html",
                "description": "Controls permission to decrypt ciphertext that was encrypted under a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html",
                "description": "Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DeleteCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html",
                "description": "Controls permission to delete a custom key store.",
                "accessLevel": "Write"
            },
            "DeleteImportedKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html",
                "description": "Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DescribeCustomKeyStores": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html",
                "description": "Controls permission to view detailed information about custom key stores in the account and region.",
                "accessLevel": "Read"
            },
            "DescribeKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html",
                "description": "Controls permission to view detailed information about a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html",
                "description": "Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html",
                "description": "Controls permission to disable automatic rotation of a customer managed customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisconnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html",
                "description": "Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "EnableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html",
                "description": "Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "EnableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html",
                "description": "Controls permission to enable automatic rotation of the cryptographic material in a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Encrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html",
                "description": "Controls permission to use the specified customer master key to encrypt data and data keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html",
                "description": "Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPair": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html",
                "description": "Controls permission to use the customer master key to generate data key pairs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPairWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateRandom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html",
                "description": "Controls permission to get a cryptographically secure random byte string from AWS KMS.",
                "accessLevel": "Write"
            },
            "GetKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html",
                "description": "Controls permission to view the key policy for the specified customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetKeyRotationStatus": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html",
                "description": "Controls permission to determine whether automatic key rotation is enabled on the customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetParametersForImport": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html",
                "description": "Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService",
                    "kms:WrappingAlgorithm",
                    "kms:WrappingKeySpec"
                ]
            },
            "GetPublicKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html",
                "description": "Controls permission to download the public key of an asymmetric customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ImportKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html",
                "description": "Controls permission to import cryptographic material into a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ExpirationModel",
                    "kms:ValidTo",
                    "kms:ViaService"
                ]
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html",
                "description": "Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "List"
            },
            "ListGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html",
                "description": "Controls permission to view all grants for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ListKeyPolicies": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html",
                "description": "Controls permission to view the names of key policies for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListKeys": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html",
                "description": "Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.",
                "accessLevel": "List"
            },
            "ListResourceTags": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html",
                "description": "Controls permission to view all tags that are attached to a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListRetirableGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html",
                "description": "Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "PutKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html",
                "description": "Controls permission to replace the key policy for the specified customer master key.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ReEncryptFrom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "ReEncryptTo": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "RetireGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html",
                "description": "Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "RevokeGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html",
                "description": "Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ScheduleKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html",
                "description": "Controls permission to schedule deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Sign": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html",
                "description": "Controls permission to produce a digital signature for a message.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html",
                "description": "Controls permission to create or update tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html",
                "description": "Controls permission to delete tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html",
                "description": "Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html",
                "description": "Controls permission to change the properties of a custom key store.",
                "accessLevel": "Write"
            },
            "UpdateKeyDescription": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html",
                "description": "Controls permission to delete or change the description of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Verify": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html",
                "description": "Controls permission to use the specified customer master key to verify digital signatures.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            }
        };
        this.resourceTypes = {
            "alias": {
                "name": "alias",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}",
                "conditionKeys": []
            },
            "key": {
                "name": "key",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Controls permission to cancel the scheduled deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html
     */
    cancelKeyDeletion() {
        this.add('kms:CancelKeyDeletion');
        return this;
    }
    /**
     * Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html
     */
    connectCustomKeyStore() {
        this.add('kms:ConnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html
     */
    createAlias() {
        this.add('kms:CreateAlias');
        return this;
    }
    /**
     * Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudhsm:DescribeClusters
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html
     */
    createCustomKeyStore() {
        this.add('kms:CreateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantConstraintType()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html
     */
    createGrant() {
        this.add('kms:CreateGrant');
        return this;
    }
    /**
     * Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCustomerMasterKeySpec()
     * - .ifCustomerMasterKeyUsage()
     * - .ifKeyOrigin()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html
     */
    createKey() {
        this.add('kms:CreateKey');
        return this;
    }
    /**
     * Controls permission to decrypt ciphertext that was encrypted under a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html
     */
    decrypt() {
        this.add('kms:Decrypt');
        return this;
    }
    /**
     * Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('kms:DeleteAlias');
        return this;
    }
    /**
     * Controls permission to delete a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html
     */
    deleteCustomKeyStore() {
        this.add('kms:DeleteCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html
     */
    deleteImportedKeyMaterial() {
        this.add('kms:DeleteImportedKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view detailed information about custom key stores in the account and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html
     */
    describeCustomKeyStores() {
        this.add('kms:DescribeCustomKeyStores');
        return this;
    }
    /**
     * Controls permission to view detailed information about a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html
     */
    describeKey() {
        this.add('kms:DescribeKey');
        return this;
    }
    /**
     * Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html
     */
    disableKey() {
        this.add('kms:DisableKey');
        return this;
    }
    /**
     * Controls permission to disable automatic rotation of a customer managed customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html
     */
    disableKeyRotation() {
        this.add('kms:DisableKeyRotation');
        return this;
    }
    /**
     * Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html
     */
    disconnectCustomKeyStore() {
        this.add('kms:DisconnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html
     */
    enableKey() {
        this.add('kms:EnableKey');
        return this;
    }
    /**
     * Controls permission to enable automatic rotation of the cryptographic material in a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html
     */
    enableKeyRotation() {
        this.add('kms:EnableKeyRotation');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to encrypt data and data keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html
     */
    encrypt() {
        this.add('kms:Encrypt');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html
     */
    generateDataKey() {
        this.add('kms:GenerateDataKey');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html
     */
    generateDataKeyPair() {
        this.add('kms:GenerateDataKeyPair');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html
     */
    generateDataKeyPairWithoutPlaintext() {
        this.add('kms:GenerateDataKeyPairWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html
     */
    generateDataKeyWithoutPlaintext() {
        this.add('kms:GenerateDataKeyWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to get a cryptographically secure random byte string from AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html
     */
    generateRandom() {
        this.add('kms:GenerateRandom');
        return this;
    }
    /**
     * Controls permission to view the key policy for the specified customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html
     */
    getKeyPolicy() {
        this.add('kms:GetKeyPolicy');
        return this;
    }
    /**
     * Controls permission to determine whether automatic key rotation is enabled on the customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html
     */
    getKeyRotationStatus() {
        this.add('kms:GetKeyRotationStatus');
        return this;
    }
    /**
     * Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     * - .ifWrappingAlgorithm()
     * - .ifWrappingKeySpec()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html
     */
    getParametersForImport() {
        this.add('kms:GetParametersForImport');
        return this;
    }
    /**
     * Controls permission to download the public key of an asymmetric customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html
     */
    getPublicKey() {
        this.add('kms:GetPublicKey');
        return this;
    }
    /**
     * Controls permission to import cryptographic material into a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifExpirationModel()
     * - .ifValidTo()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html
     */
    importKeyMaterial() {
        this.add('kms:ImportKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html
     */
    listAliases() {
        this.add('kms:ListAliases');
        return this;
    }
    /**
     * Controls permission to view all grants for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html
     */
    listGrants() {
        this.add('kms:ListGrants');
        return this;
    }
    /**
     * Controls permission to view the names of key policies for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html
     */
    listKeyPolicies() {
        this.add('kms:ListKeyPolicies');
        return this;
    }
    /**
     * Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html
     */
    listKeys() {
        this.add('kms:ListKeys');
        return this;
    }
    /**
     * Controls permission to view all tags that are attached to a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html
     */
    listResourceTags() {
        this.add('kms:ListResourceTags');
        return this;
    }
    /**
     * Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html
     */
    listRetirableGrants() {
        this.add('kms:ListRetirableGrants');
        return this;
    }
    /**
     * Controls permission to replace the key policy for the specified customer master key.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html
     */
    putKeyPolicy() {
        this.add('kms:PutKeyPolicy');
        return this;
    }
    /**
     * Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    reEncryptFrom() {
        this.add('kms:ReEncryptFrom');
        return this;
    }
    /**
     * Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    reEncryptTo() {
        this.add('kms:ReEncryptTo');
        return this;
    }
    /**
     * Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html
     */
    retireGrant() {
        this.add('kms:RetireGrant');
        return this;
    }
    /**
     * Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html
     */
    revokeGrant() {
        this.add('kms:RevokeGrant');
        return this;
    }
    /**
     * Controls permission to schedule deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html
     */
    scheduleKeyDeletion() {
        this.add('kms:ScheduleKeyDeletion');
        return this;
    }
    /**
     * Controls permission to produce a digital signature for a message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html
     */
    sign() {
        this.add('kms:Sign');
        return this;
    }
    /**
     * Controls permission to create or update tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('kms:TagResource');
        return this;
    }
    /**
     * Controls permission to delete tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('kms:UntagResource');
        return this;
    }
    /**
     * Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html
     */
    updateAlias() {
        this.add('kms:UpdateAlias');
        return this;
    }
    /**
     * Controls permission to change the properties of a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html
     */
    updateCustomKeyStore() {
        this.add('kms:UpdateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete or change the description of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html
     */
    updateKeyDescription() {
        this.add('kms:UpdateKeyDescription');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to verify digital signatures.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html
     */
    verify() {
        this.add('kms:Verify');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlias(alias, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}';
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Controls access to the CreateKey and PutKeyPolicy operations based on the value of the BypassPolicyLockoutSafetyCheck parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-bypass-policy-lockout-safety-check
     *
     * Applies to actions:
     * - .createKey()
     * - .putKeyPolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBypassPolicyLockoutSafetyCheck(value) {
        return this.if(`kms:BypassPolicyLockoutSafetyCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to specified AWS KMS operations based on the AWS account ID of the caller. You can use this condition key to allow or deny access to all IAM users and roles in an AWS account in a single policy statement.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-caller-account
     *
     * Applies to actions:
     * - .cancelKeyDeletion()
     * - .createAlias()
     * - .createGrant()
     * - .decrypt()
     * - .deleteAlias()
     * - .deleteImportedKeyMaterial()
     * - .describeKey()
     * - .disableKey()
     * - .disableKeyRotation()
     * - .enableKey()
     * - .enableKeyRotation()
     * - .encrypt()
     * - .generateDataKey()
     * - .generateDataKeyPair()
     * - .generateDataKeyPairWithoutPlaintext()
     * - .generateDataKeyWithoutPlaintext()
     * - .getKeyPolicy()
     * - .getKeyRotationStatus()
     * - .getParametersForImport()
     * - .getPublicKey()
     * - .importKeyMaterial()
     * - .listGrants()
     * - .listKeyPolicies()
     * - .listResourceTags()
     * - .putKeyPolicy()
     * - .reEncryptFrom()
     * - .reEncryptTo()
     * - .revokeGrant()
     * - .scheduleKeyDeletion()
     * - .sign()
     * - .tagResource()
     * - .untagResource()
     * - .updateAlias()
     * - .updateKeyDescription()
     * - .verify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCallerAccount(value, operator) {
        return this.if(`kms:CallerAccount`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the CustomerMasterKeySpec property of the CMK that is created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-spec
     *
     * Applies to actions:
     * - .createKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeySpec(value, operator) {
        return this.if(`kms:CustomerMasterKeySpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the KeyUsage property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-usage
     *
     * Applies to actions:
     * - .createKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeyUsage(value, operator) {
        return this.if(`kms:CustomerMasterKeyUsage`, value, operator || 'StringLike');
    }
    /**
     * Controls access to GenerateDataKeyPair and GenerateDataKeyPairWithoutPlaintext operations based on the value of the DataKeyPairSpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-data-key-pair-spec
     *
     * Applies to actions:
     * - .generateDataKeyPair()
     * - .generateDataKeyPairWithoutPlaintext()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataKeyPairSpec(value, operator) {
        return this.if(`kms:DataKeyPairSpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to encryption operations based on the value of the encryption algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm
     *
     * Applies to actions:
     * - .decrypt()
     * - .encrypt()
     * - .generateDataKey()
     * - .generateDataKeyPair()
     * - .generateDataKeyPairWithoutPlaintext()
     * - .generateDataKeyWithoutPlaintext()
     * - .reEncryptFrom()
     * - .reEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionAlgorithm(value, operator) {
        return this.if(`kms:EncryptionAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access based on the presence of specified keys in the encryption context. The encryption context is an optional element in a cryptographic operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-context-keys
     *
     * Applies to actions:
     * - .decrypt()
     * - .encrypt()
     * - .generateDataKey()
     * - .generateDataKeyPair()
     * - .generateDataKeyPairWithoutPlaintext()
     * - .generateDataKeyWithoutPlaintext()
     * - .reEncryptFrom()
     * - .reEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContextKeys(value, operator) {
        return this.if(`kms:EncryptionContextKeys`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ExpirationModel parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-expiration-model
     *
     * Applies to actions:
     * - .importKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExpirationModel(value, operator) {
        return this.if(`kms:ExpirationModel`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grant constraint in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-constraint-type
     *
     * Applies to actions:
     * - .createGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantConstraintType(value, operator) {
        return this.if(`kms:GrantConstraintType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation when the request comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-is-for-aws-resource
     *
     * Applies to actions:
     * - .createGrant()
     * - .listGrants()
     * - .revokeGrant()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifGrantIsForAWSResource(value) {
        return this.if(`kms:GrantIsForAWSResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the operations in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantOperations(value, operator) {
        return this.if(`kms:GrantOperations`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grantee principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grantee-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGranteePrincipal(value, operator) {
        return this.if(`kms:GranteePrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the Origin property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-key-origin
     *
     * Applies to actions:
     * - .createKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyOrigin(value, operator) {
        return this.if(`kms:KeyOrigin`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the value of the MessageType parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-message-type
     *
     * Applies to actions:
     * - .sign()
     * - .verify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMessageType(value, operator) {
        return this.if(`kms:MessageType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ReEncrypt operation when it uses the same customer master key that was used for the Encrypt operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-reencrypt-on-same-key
     *
     * Applies to actions:
     * - .reEncryptFrom()
     * - .reEncryptTo()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifReEncryptOnSameKey(value) {
        return this.if(`kms:ReEncryptOnSameKey`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the retiring principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-retiring-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRetiringPrincipal(value, operator) {
        return this.if(`kms:RetiringPrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the signing algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm
     *
     * Applies to actions:
     * - .sign()
     * - .verify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSigningAlgorithm(value, operator) {
        return this.if(`kms:SigningAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ValidTo parameter in the request. You can use this condition key to allow users to import key material only when it expires by the specified date.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-valid-to
     *
     * Applies to actions:
     * - .importKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifValidTo(value, operator) {
        return this.if(`kms:ValidTo`, value, operator || 'NumericEquals');
    }
    /**
     * Controls access when a request made on the principal's behalf comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-via-service
     *
     * Applies to actions:
     * - .cancelKeyDeletion()
     * - .createAlias()
     * - .createGrant()
     * - .decrypt()
     * - .deleteAlias()
     * - .deleteImportedKeyMaterial()
     * - .describeKey()
     * - .disableKey()
     * - .disableKeyRotation()
     * - .enableKey()
     * - .enableKeyRotation()
     * - .encrypt()
     * - .generateDataKey()
     * - .generateDataKeyPair()
     * - .generateDataKeyPairWithoutPlaintext()
     * - .generateDataKeyWithoutPlaintext()
     * - .getKeyPolicy()
     * - .getKeyRotationStatus()
     * - .getParametersForImport()
     * - .getPublicKey()
     * - .importKeyMaterial()
     * - .listGrants()
     * - .listKeyPolicies()
     * - .listResourceTags()
     * - .putKeyPolicy()
     * - .reEncryptFrom()
     * - .reEncryptTo()
     * - .revokeGrant()
     * - .scheduleKeyDeletion()
     * - .sign()
     * - .tagResource()
     * - .untagResource()
     * - .updateAlias()
     * - .updateKeyDescription()
     * - .verify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaService(value, operator) {
        return this.if(`kms:ViaService`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingAlgorithm parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-algorithm
     *
     * Applies to actions:
     * - .getParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingAlgorithm(value, operator) {
        return this.if(`kms:WrappingAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingKeySpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-key-spec
     *
     * Applies to actions:
     * - .getParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeySpec(value, operator) {
        return this.if(`kms:WrappingKeySpec`, value, operator || 'StringLike');
    }
}
exports.Kms = Kms;
//# sourceMappingURL=data:application/json;base64,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