"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Organizations = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Organizations extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'organizations';
        this.actionList = {
            "AcceptHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html",
                "description": "Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html",
                "description": "Grants permission to attach a policy to a root, an organizational unit, or an individual account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    },
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "CancelHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html",
                "description": "Grants permission to cancel a handshake.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request.",
                "accessLevel": "Write"
            },
            "CreateGovCloudAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html",
                "description": "Grants permission to create an AWS GovCloud (US) account.",
                "accessLevel": "Write"
            },
            "CreateOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html",
                "description": "Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the master account of the new organization.",
                "accessLevel": "Write"
            },
            "CreateOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html",
                "description": "Grants permission to create an organizational unit (OU) within a root or parent OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html",
                "description": "Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DeclineHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html",
                "description": "Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "DeleteOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html",
                "description": "Grants permission to delete the organization.",
                "accessLevel": "Write"
            },
            "DeleteOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html",
                "description": "Grants permission to delete an organizational unit from a root or another OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html",
                "description": "Grants permission to delete a policy from your organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DeregisterDelegatedAdministrator": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html",
                "description": "Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html",
                "description": "Grants permission to retrieve Organizations-related details about the specified account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                }
            },
            "DescribeCreateAccountStatus": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html",
                "description": "Grants permission to retrieve the current status of an asynchronous request to create an account.",
                "accessLevel": "Read"
            },
            "DescribeEffectivePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html",
                "description": "Grants permission to retrieve the effective policy for an account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DescribeHandshake": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html",
                "description": "Grants permission to retrieve details about a previously requested handshake.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "handshake": {
                        "required": true
                    }
                }
            },
            "DescribeOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html",
                "description": "Grants permission to retrieves details about the organization that the calling credentials belong to.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html",
                "description": "Grants permission to retrieve details about an organizational unit (OU).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "DescribePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html",
                "description": "Grants permission to retrieves details about a policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html",
                "description": "Grants permission to detach a policy from a target root, organizational unit, or account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    },
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "DisableAWSServiceAccess": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html",
                "description": "Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "DisablePolicyType": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html",
                "description": "Grants permission to disable an organization policy type in a root.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "root": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "EnableAWSServiceAccess": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html",
                "description": "Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "EnableAllFeatures": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html",
                "description": "Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features.",
                "accessLevel": "Write"
            },
            "EnablePolicyType": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html",
                "description": "Grants permission to enable a policy type in a root.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "root": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "InviteAccountToOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html",
                "description": "Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": false
                    }
                }
            },
            "LeaveOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html",
                "description": "Grants permission to remove a member account from its parent organization.",
                "accessLevel": "Write"
            },
            "ListAWSServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html",
                "description": "Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization.",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list all of the the accounts in the organization.",
                "accessLevel": "List"
            },
            "ListAccountsForParent": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html",
                "description": "Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU).",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListChildren": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html",
                "description": "Grants permission to list all of the OUs or accounts that are contained in a parent OU or root.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListCreateAccountStatus": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html",
                "description": "Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization.",
                "accessLevel": "List"
            },
            "ListDelegatedAdministrators": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html",
                "description": "Grants permission to list the AWS accounts that are designated as delegated administrators in this organization.",
                "accessLevel": "List",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "ListDelegatedServicesForAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html",
                "description": "Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization.",
                "accessLevel": "List"
            },
            "ListHandshakesForAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html",
                "description": "Grants permission to list all of the handshakes that are associated with an account.",
                "accessLevel": "List"
            },
            "ListHandshakesForOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html",
                "description": "Grants permission to list the handshakes that are associated with the organization.",
                "accessLevel": "List"
            },
            "ListOrganizationalUnitsForParent": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html",
                "description": "Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root.",
                "accessLevel": "List",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "ListParents": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html",
                "description": "Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    }
                }
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html",
                "description": "Grants permission to list all of the policies in an organization.",
                "accessLevel": "List",
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "ListPoliciesForTarget": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html",
                "description": "Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "account": {
                        "required": false
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "ListRoots": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html",
                "description": "Grants permission to list all of the roots that are defined in the organization.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list all tags for the specified resource.",
                "accessLevel": "List"
            },
            "ListTargetsForPolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html",
                "description": "Grants permission to list all the roots, OUs, and accounts to which a policy is attached.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            },
            "MoveAccount": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html",
                "description": "Grants permission to move an account from its current root or OU to another parent root or OU.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    },
                    "organizationalunit": {
                        "required": false
                    },
                    "root": {
                        "required": false
                    }
                }
            },
            "RegisterDelegatedAdministrator": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html",
                "description": "Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal.",
                "accessLevel": "Write",
                "conditions": [
                    "organizations:ServicePrincipal"
                ]
            },
            "RemoveAccountFromOrganization": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html",
                "description": "Grants permission to removes the specified account from the organization.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "account": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add one or more tags to the specified resource.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags from the specified resource.",
                "accessLevel": "Tagging"
            },
            "UpdateOrganizationalUnit": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html",
                "description": "Grants permission to rename an organizational unit (OU).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "organizationalunit": {
                        "required": true
                    }
                }
            },
            "UpdatePolicy": {
                "url": "https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html",
                "description": "Grants permission to update an existing policy with a new name, description, or content.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "organizations:PolicyType"
                ]
            }
        };
        this.resourceTypes = {
            "account": {
                "name": "account",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:account/o-${OrganizationId}/${AccountId}",
                "conditionKeys": []
            },
            "handshake": {
                "name": "handshake",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:handshake/o-${OrganizationId}/${HandshakeType}/h-${HandshakeId}",
                "conditionKeys": []
            },
            "organization": {
                "name": "organization",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:organization/o-${OrganizationId}",
                "conditionKeys": []
            },
            "organizationalunit": {
                "name": "organizationalunit",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:policy/o-${OrganizationId}/${PolicyType}/p-${PolicyId}",
                "conditionKeys": []
            },
            "awspolicy": {
                "name": "awspolicy",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::aws:policy/${PolicyType}/p-${PolicyId}",
                "conditionKeys": []
            },
            "root": {
                "name": "root",
                "url": "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html",
                "arn": "arn:${Partition}:organizations::${MasterAccountId}:root/o-${OrganizationId}/r-${RootId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html
     */
    acceptHandshake() {
        this.add('organizations:AcceptHandshake');
        return this;
    }
    /**
     * Grants permission to attach a policy to a root, an organizational unit, or an individual account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html
     */
    attachPolicy() {
        this.add('organizations:AttachPolicy');
        return this;
    }
    /**
     * Grants permission to cancel a handshake.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html
     */
    cancelHandshake() {
        this.add('organizations:CancelHandshake');
        return this;
    }
    /**
     * Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html
     */
    createAccount() {
        this.add('organizations:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create an AWS GovCloud (US) account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html
     */
    createGovCloudAccount() {
        this.add('organizations:CreateGovCloudAccount');
        return this;
    }
    /**
     * Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the master account of the new organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html
     */
    createOrganization() {
        this.add('organizations:CreateOrganization');
        return this;
    }
    /**
     * Grants permission to create an organizational unit (OU) within a root or parent OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html
     */
    createOrganizationalUnit() {
        this.add('organizations:CreateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html
     */
    createPolicy() {
        this.add('organizations:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html
     */
    declineHandshake() {
        this.add('organizations:DeclineHandshake');
        return this;
    }
    /**
     * Grants permission to delete the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html
     */
    deleteOrganization() {
        this.add('organizations:DeleteOrganization');
        return this;
    }
    /**
     * Grants permission to delete an organizational unit from a root or another OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html
     */
    deleteOrganizationalUnit() {
        this.add('organizations:DeleteOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to delete a policy from your organization.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('organizations:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html
     */
    deregisterDelegatedAdministrator() {
        this.add('organizations:DeregisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to retrieve Organizations-related details about the specified account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html
     */
    describeAccount() {
        this.add('organizations:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the current status of an asynchronous request to create an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html
     */
    describeCreateAccountStatus() {
        this.add('organizations:DescribeCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the effective policy for an account.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html
     */
    describeEffectivePolicy() {
        this.add('organizations:DescribeEffectivePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve details about a previously requested handshake.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html
     */
    describeHandshake() {
        this.add('organizations:DescribeHandshake');
        return this;
    }
    /**
     * Grants permission to retrieves details about the organization that the calling credentials belong to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    describeOrganization() {
        this.add('organizations:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve details about an organizational unit (OU).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html
     */
    describeOrganizationalUnit() {
        this.add('organizations:DescribeOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to retrieves details about a policy.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html
     */
    describePolicy() {
        this.add('organizations:DescribePolicy');
        return this;
    }
    /**
     * Grants permission to detach a policy from a target root, organizational unit, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html
     */
    detachPolicy() {
        this.add('organizations:DetachPolicy');
        return this;
    }
    /**
     * Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html
     */
    disableAWSServiceAccess() {
        this.add('organizations:DisableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to disable an organization policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html
     */
    disablePolicyType() {
        this.add('organizations:DisablePolicyType');
        return this;
    }
    /**
     * Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html
     */
    enableAWSServiceAccess() {
        this.add('organizations:EnableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html
     */
    enableAllFeatures() {
        this.add('organizations:EnableAllFeatures');
        return this;
    }
    /**
     * Grants permission to enable a policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html
     */
    enablePolicyType() {
        this.add('organizations:EnablePolicyType');
        return this;
    }
    /**
     * Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html
     */
    inviteAccountToOrganization() {
        this.add('organizations:InviteAccountToOrganization');
        return this;
    }
    /**
     * Grants permission to remove a member account from its parent organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html
     */
    leaveOrganization() {
        this.add('organizations:LeaveOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html
     */
    listAWSServiceAccessForOrganization() {
        this.add('organizations:ListAWSServiceAccessForOrganization');
        return this;
    }
    /**
     * Grants permission to list all of the the accounts in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html
     */
    listAccounts() {
        this.add('organizations:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html
     */
    listAccountsForParent() {
        this.add('organizations:ListAccountsForParent');
        return this;
    }
    /**
     * Grants permission to list all of the OUs or accounts that are contained in a parent OU or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html
     */
    listChildren() {
        this.add('organizations:ListChildren');
        return this;
    }
    /**
     * Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html
     */
    listCreateAccountStatus() {
        this.add('organizations:ListCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts that are designated as delegated administrators in this organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html
     */
    listDelegatedAdministrators() {
        this.add('organizations:ListDelegatedAdministrators');
        return this;
    }
    /**
     * Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html
     */
    listDelegatedServicesForAccount() {
        this.add('organizations:ListDelegatedServicesForAccount');
        return this;
    }
    /**
     * Grants permission to list all of the handshakes that are associated with an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html
     */
    listHandshakesForAccount() {
        this.add('organizations:ListHandshakesForAccount');
        return this;
    }
    /**
     * Grants permission to list the handshakes that are associated with the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html
     */
    listHandshakesForOrganization() {
        this.add('organizations:ListHandshakesForOrganization');
        return this;
    }
    /**
     * Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html
     */
    listOrganizationalUnitsForParent() {
        this.add('organizations:ListOrganizationalUnitsForParent');
        return this;
    }
    /**
     * Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html
     */
    listParents() {
        this.add('organizations:ListParents');
        return this;
    }
    /**
     * Grants permission to list all of the policies in an organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('organizations:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html
     */
    listPoliciesForTarget() {
        this.add('organizations:ListPoliciesForTarget');
        return this;
    }
    /**
     * Grants permission to list all of the roots that are defined in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html
     */
    listRoots() {
        this.add('organizations:ListRoots');
        return this;
    }
    /**
     * Grants permission to list all tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('organizations:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list all the roots, OUs, and accounts to which a policy is attached.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html
     */
    listTargetsForPolicy() {
        this.add('organizations:ListTargetsForPolicy');
        return this;
    }
    /**
     * Grants permission to move an account from its current root or OU to another parent root or OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html
     */
    moveAccount() {
        this.add('organizations:MoveAccount');
        return this;
    }
    /**
     * Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html
     */
    registerDelegatedAdministrator() {
        this.add('organizations:RegisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to removes the specified account from the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html
     */
    removeAccountFromOrganization() {
        this.add('organizations:RemoveAccountFromOrganization');
        return this;
    }
    /**
     * Grants permission to add one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('organizations:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('organizations:UntagResource');
        return this;
    }
    /**
     * Grants permission to rename an organizational unit (OU).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html
     */
    updateOrganizationalUnit() {
        this.add('organizations:UpdateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to update an existing policy with a new name, description, or content.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html
     */
    updatePolicy() {
        this.add('organizations:UpdatePolicy');
        return this;
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccount(masterAccountId, organizationId, accountId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:account/o-${OrganizationId}/${AccountId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type handshake to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param handshakeType - Identifier for the handshakeType.
     * @param handshakeId - Identifier for the handshakeId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHandshake(masterAccountId, organizationId, handshakeType, handshakeId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:handshake/o-${OrganizationId}/${HandshakeType}/h-${HandshakeId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${HandshakeType}', handshakeType);
        arn = arn.replace('${HandshakeId}', handshakeId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganization(masterAccountId, organizationId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:organization/o-${OrganizationId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organizationalunit to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param organizationalUnitId - Identifier for the organizationalUnitId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationalunit(masterAccountId, organizationId, organizationalUnitId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${OrganizationalUnitId}', organizationalUnitId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(masterAccountId, organizationId, policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:policy/o-${OrganizationId}/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type awspolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAwspolicy(policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::aws:policy/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type root to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param rootId - Identifier for the rootId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRoot(masterAccountId, organizationId, rootId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:root/o-${OrganizationId}/r-${RootId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${RootId}', rootId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to filter the request to only the specified policy type names.
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .attachPolicy()
     * - .createPolicy()
     * - .deletePolicy()
     * - .describeEffectivePolicy()
     * - .describePolicy()
     * - .detachPolicy()
     * - .disablePolicyType()
     * - .enablePolicyType()
     * - .listPolicies()
     * - .listPoliciesForTarget()
     * - .listTargetsForPolicy()
     * - .updatePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPolicyType(value, operator) {
        return this.if(`organizations:PolicyType`, value, operator || 'StringLike');
    }
    /**
     * Enables you to filter the request to only the specified service principal names.
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .deregisterDelegatedAdministrator()
     * - .disableAWSServiceAccess()
     * - .enableAWSServiceAccess()
     * - .listDelegatedAdministrators()
     * - .registerDelegatedAdministrator()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServicePrincipal(value, operator) {
        return this.if(`organizations:ServicePrincipal`, value, operator || 'StringLike');
    }
}
exports.Organizations = Organizations;
//# sourceMappingURL=data:application/json;base64,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