"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Robomaker = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Robomaker extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [robomaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsrobomaker.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'robomaker';
        this.actionList = {
            "BatchDescribeSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html",
                "description": "Describe multiple simulation jobs",
                "accessLevel": "Read"
            },
            "CancelDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html",
                "description": "Cancel a deployment job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentJob": {
                        "required": true
                    }
                }
            },
            "CancelSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html",
                "description": "Cancel a simulation job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "CancelSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html",
                "description": "Cancel a simulation job batch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJobBatch": {
                        "required": true
                    }
                }
            },
            "CreateDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html",
                "description": "Create a deployment job",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html",
                "description": "Create a deployment fleet that represents a logical group of robots running the same robot application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html",
                "description": "Create a robot that can be registered to a fleet",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html",
                "description": "Create a robot application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateRobotApplicationVersion": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html",
                "description": "Create a snapshot of a robot application",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ],
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "CreateSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html",
                "description": "Create a simulation application",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateSimulationApplicationVersion": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html",
                "description": "Create a snapshot of a simulation application",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ],
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "CreateSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html",
                "description": "Create a simulation job",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html",
                "description": "Delete a deployment fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "DeleteRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html",
                "description": "Delete a robot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robot": {
                        "required": true
                    }
                }
            },
            "DeleteRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html",
                "description": "Delete a robot application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "DeleteSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html",
                "description": "Delete a simulation application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "DeregisterRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html",
                "description": "Deregister a robot from a fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    },
                    "robot": {
                        "required": true
                    }
                }
            },
            "DescribeDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html",
                "description": "Describe a deployment job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentJob": {
                        "required": true
                    }
                }
            },
            "DescribeFleet": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html",
                "description": "Describe a deployment fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "DescribeRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html",
                "description": "Describe a robot",
                "accessLevel": "Read",
                "resourceTypes": {
                    "robot": {
                        "required": true
                    }
                }
            },
            "DescribeRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html",
                "description": "Describe a robot application",
                "accessLevel": "Read",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html",
                "description": "Describe a simulation application",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html",
                "description": "Describe a simulation job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "DescribeSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html",
                "description": "Describe a simulation job batch",
                "accessLevel": "Read",
                "resourceTypes": {
                    "simulationJobBatch": {
                        "required": true
                    }
                }
            },
            "ListDeploymentJobs": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html",
                "description": "List deployment jobs",
                "accessLevel": "List"
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html",
                "description": "List fleets",
                "accessLevel": "List"
            },
            "ListRobotApplications": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html",
                "description": "List robot applications",
                "accessLevel": "List"
            },
            "ListRobots": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html",
                "description": "List robots",
                "accessLevel": "List"
            },
            "ListSimulationApplications": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html",
                "description": "List simulation applications",
                "accessLevel": "List"
            },
            "ListSimulationJobBatches": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html",
                "description": "List simulation job batches",
                "accessLevel": "List"
            },
            "ListSimulationJobs": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html",
                "description": "List simulation jobs",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html",
                "description": "List tags for a RoboMaker resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                }
            },
            "RegisterRobot": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html",
                "description": "Register a robot to a fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    },
                    "robot": {
                        "required": true
                    }
                }
            },
            "RestartSimulationJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html",
                "description": "Restart a running simulation job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationJob": {
                        "required": true
                    }
                }
            },
            "StartSimulationJobBatch": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html",
                "description": "Create a simulation job batch",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "SyncDeploymentJob": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html",
                "description": "Ensures the most recently deployed robot application is deployed to all robots in the fleet",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html",
                "description": "Add tags to a RoboMaker resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html",
                "description": "Remove tags from a RoboMaker resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentFleet": {
                        "required": false
                    },
                    "deploymentJob": {
                        "required": false
                    },
                    "robot": {
                        "required": false
                    },
                    "robotApplication": {
                        "required": false
                    },
                    "simulationApplication": {
                        "required": false
                    },
                    "simulationJob": {
                        "required": false
                    },
                    "simulationJobBatch": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateRobotApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html",
                "description": "Update a robot application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "robotApplication": {
                        "required": true
                    }
                }
            },
            "UpdateSimulationApplication": {
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html",
                "description": "Update a simulation application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "simulationApplication": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "robotApplication": {
                "name": "robotApplication",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:robot-application/${ApplicationName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationApplication": {
                "name": "simulationApplication",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-application/${ApplicationName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationJob": {
                "name": "simulationJob",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-job/${SimulationJobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "simulationJobBatch": {
                "name": "simulationJobBatch",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:simulation-job-batch/${SimulationJobBatchId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deploymentJob": {
                "name": "deploymentJob",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:deployment-job/${DeploymentJobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "robot": {
                "name": "robot",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:robot/${RobotName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deploymentFleet": {
                "name": "deploymentFleet",
                "url": "https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html",
                "arn": "arn:${Partition}:robomaker:${Region}:${Account}:deployment-fleet/${FleetName}/${CreatedOnEpoch}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Describe multiple simulation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html
     */
    batchDescribeSimulationJob() {
        this.add('robomaker:BatchDescribeSimulationJob');
        return this;
    }
    /**
     * Cancel a deployment job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html
     */
    cancelDeploymentJob() {
        this.add('robomaker:CancelDeploymentJob');
        return this;
    }
    /**
     * Cancel a simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html
     */
    cancelSimulationJob() {
        this.add('robomaker:CancelSimulationJob');
        return this;
    }
    /**
     * Cancel a simulation job batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html
     */
    cancelSimulationJobBatch() {
        this.add('robomaker:CancelSimulationJobBatch');
        return this;
    }
    /**
     * Create a deployment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html
     */
    createDeploymentJob() {
        this.add('robomaker:CreateDeploymentJob');
        return this;
    }
    /**
     * Create a deployment fleet that represents a logical group of robots running the same robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html
     */
    createFleet() {
        this.add('robomaker:CreateFleet');
        return this;
    }
    /**
     * Create a robot that can be registered to a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html
     */
    createRobot() {
        this.add('robomaker:CreateRobot');
        return this;
    }
    /**
     * Create a robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html
     */
    createRobotApplication() {
        this.add('robomaker:CreateRobotApplication');
        return this;
    }
    /**
     * Create a snapshot of a robot application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html
     */
    createRobotApplicationVersion() {
        this.add('robomaker:CreateRobotApplicationVersion');
        return this;
    }
    /**
     * Create a simulation application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html
     */
    createSimulationApplication() {
        this.add('robomaker:CreateSimulationApplication');
        return this;
    }
    /**
     * Create a snapshot of a simulation application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html
     */
    createSimulationApplicationVersion() {
        this.add('robomaker:CreateSimulationApplicationVersion');
        return this;
    }
    /**
     * Create a simulation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html
     */
    createSimulationJob() {
        this.add('robomaker:CreateSimulationJob');
        return this;
    }
    /**
     * Delete a deployment fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('robomaker:DeleteFleet');
        return this;
    }
    /**
     * Delete a robot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html
     */
    deleteRobot() {
        this.add('robomaker:DeleteRobot');
        return this;
    }
    /**
     * Delete a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html
     */
    deleteRobotApplication() {
        this.add('robomaker:DeleteRobotApplication');
        return this;
    }
    /**
     * Delete a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html
     */
    deleteSimulationApplication() {
        this.add('robomaker:DeleteSimulationApplication');
        return this;
    }
    /**
     * Deregister a robot from a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html
     */
    deregisterRobot() {
        this.add('robomaker:DeregisterRobot');
        return this;
    }
    /**
     * Describe a deployment job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html
     */
    describeDeploymentJob() {
        this.add('robomaker:DescribeDeploymentJob');
        return this;
    }
    /**
     * Describe a deployment fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html
     */
    describeFleet() {
        this.add('robomaker:DescribeFleet');
        return this;
    }
    /**
     * Describe a robot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html
     */
    describeRobot() {
        this.add('robomaker:DescribeRobot');
        return this;
    }
    /**
     * Describe a robot application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html
     */
    describeRobotApplication() {
        this.add('robomaker:DescribeRobotApplication');
        return this;
    }
    /**
     * Describe a simulation application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html
     */
    describeSimulationApplication() {
        this.add('robomaker:DescribeSimulationApplication');
        return this;
    }
    /**
     * Describe a simulation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html
     */
    describeSimulationJob() {
        this.add('robomaker:DescribeSimulationJob');
        return this;
    }
    /**
     * Describe a simulation job batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html
     */
    describeSimulationJobBatch() {
        this.add('robomaker:DescribeSimulationJobBatch');
        return this;
    }
    /**
     * List deployment jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html
     */
    listDeploymentJobs() {
        this.add('robomaker:ListDeploymentJobs');
        return this;
    }
    /**
     * List fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html
     */
    listFleets() {
        this.add('robomaker:ListFleets');
        return this;
    }
    /**
     * List robot applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html
     */
    listRobotApplications() {
        this.add('robomaker:ListRobotApplications');
        return this;
    }
    /**
     * List robots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html
     */
    listRobots() {
        this.add('robomaker:ListRobots');
        return this;
    }
    /**
     * List simulation applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html
     */
    listSimulationApplications() {
        this.add('robomaker:ListSimulationApplications');
        return this;
    }
    /**
     * List simulation job batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html
     */
    listSimulationJobBatches() {
        this.add('robomaker:ListSimulationJobBatches');
        return this;
    }
    /**
     * List simulation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html
     */
    listSimulationJobs() {
        this.add('robomaker:ListSimulationJobs');
        return this;
    }
    /**
     * List tags for a RoboMaker resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('robomaker:ListTagsForResource');
        return this;
    }
    /**
     * Register a robot to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html
     */
    registerRobot() {
        this.add('robomaker:RegisterRobot');
        return this;
    }
    /**
     * Restart a running simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html
     */
    restartSimulationJob() {
        this.add('robomaker:RestartSimulationJob');
        return this;
    }
    /**
     * Create a simulation job batch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html
     */
    startSimulationJobBatch() {
        this.add('robomaker:StartSimulationJobBatch');
        return this;
    }
    /**
     * Ensures the most recently deployed robot application is deployed to all robots in the fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html
     */
    syncDeploymentJob() {
        this.add('robomaker:SyncDeploymentJob');
        return this;
    }
    /**
     * Add tags to a RoboMaker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('robomaker:TagResource');
        return this;
    }
    /**
     * Remove tags from a RoboMaker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('robomaker:UntagResource');
        return this;
    }
    /**
     * Update a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html
     */
    updateRobotApplication() {
        this.add('robomaker:UpdateRobotApplication');
        return this;
    }
    /**
     * Update a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html
     */
    updateSimulationApplication() {
        this.add('robomaker:UpdateSimulationApplication');
        return this;
    }
    /**
     * Adds a resource of type robotApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobotApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationApplication(applicationName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-application/${ApplicationName}/${CreatedOnEpoch}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html
     *
     * @param simulationJobId - Identifier for the simulationJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJob(simulationJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job/${SimulationJobId}';
        arn = arn.replace('${SimulationJobId}', simulationJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type simulationJobBatch to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html
     *
     * @param simulationJobBatchId - Identifier for the simulationJobBatchId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJobBatch(simulationJobBatchId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:simulation-job-batch/${SimulationJobBatchId}';
        arn = arn.replace('${SimulationJobBatchId}', simulationJobBatchId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html
     *
     * @param deploymentJobId - Identifier for the deploymentJobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentJob(deploymentJobId, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-job/${DeploymentJobId}';
        arn = arn.replace('${DeploymentJobId}', deploymentJobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type robot to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html
     *
     * @param robotName - Identifier for the robotName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobot(robotName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:robot/${RobotName}/${CreatedOnEpoch}';
        arn = arn.replace('${RobotName}', robotName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentFleet to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param fleetName - Identifier for the fleetName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentFleet(fleetName, createdOnEpoch, account, region, partition) {
        var arn = 'arn:${Partition}:robomaker:${Region}:${Account}:deployment-fleet/${FleetName}/${CreatedOnEpoch}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${CreatedOnEpoch}', createdOnEpoch);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Applies to actions:
     * - .createDeploymentJob()
     * - .createFleet()
     * - .createRobot()
     * - .createRobotApplication()
     * - .createSimulationApplication()
     * - .createSimulationJob()
     * - .startSimulationJobBatch()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Applies to resource types:
     * - robotApplication
     * - simulationApplication
     * - simulationJob
     * - simulationJobBatch
     * - deploymentJob
     * - robot
     * - deploymentFleet
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Applies to actions:
     * - .createDeploymentJob()
     * - .createFleet()
     * - .createRobot()
     * - .createRobotApplication()
     * - .createSimulationApplication()
     * - .createSimulationJob()
     * - .startSimulationJobBatch()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Robomaker = Robomaker;
//# sourceMappingURL=data:application/json;base64,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