"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.actionList = {};
        this.useNotActions = false;
        this.actions = [];
        this.cdkActionsApplied = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            statement[mode] = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            const uniqueActions = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * @param actions Actions that will be added to the statement.
     */
    add(action) {
        this.actions.push(action);
        return this;
    }
    /**
     * Adds actions to the statement, matching an `AccessLevel` or regular expression.
     *
     * When no value is passed, all actions of the service will be added.
     */
    allActions(...rules) {
        if (rules.length) {
            rules.forEach((rule) => {
                for (const [name, action] of Object.entries(this.actionList)) {
                    if (typeof rule === 'object') {
                        //assume it's a regex
                        if (rule.test(name)) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                    else {
                        // assume it's an AccessLevel
                        if (rule == action.accessLevel) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                }
            });
        }
        else {
            this.add(`${this.servicePrefix}:*`);
        }
        return this;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
//# sourceMappingURL=data:application/json;base64,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