"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointGroup = exports.EndpointConfiguration = void 0;
const cdk = require("@aws-cdk/core");
const ga = require("./globalaccelerator.generated");
/**
 * The class for endpoint configuration
 */
class EndpointConfiguration extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        props.endpointGroup._linkEndpoint(this);
    }
    /**
     * render the endpoint configuration for the endpoint group
     */
    renderEndpointConfiguration() {
        return {
            clientIpPreservationEnabled: this.props.clientIpReservation,
            endpointId: this.props.endpointId,
            weight: this.props.weight,
        };
    }
}
exports.EndpointConfiguration = EndpointConfiguration;
/**
 * EndpointGroup construct
 */
class EndpointGroup extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The array of the endpoints in this endpoint group
         */
        this.endpoints = new Array();
        const resource = new ga.CfnEndpointGroup(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            endpointGroupRegion: (_a = props.region) !== null && _a !== void 0 ? _a : cdk.Stack.of(this).region,
            endpointConfigurations: cdk.Lazy.anyValue({ produce: () => this.renderEndpoints() }, { omitEmptyArray: true }),
        });
        this.endpointGroupArn = resource.attrEndpointGroupArn;
        this.endpointGroupName = (_b = props.endpointGroupName) !== null && _b !== void 0 ? _b : resource.logicalId;
    }
    /**
     * import from ARN
     */
    static fromEndpointGroupArn(scope, id, endpointGroupArn) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointGroupArn = endpointGroupArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add an endpoint
     */
    addEndpoint(id, endpointId, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointGroup: this,
            endpointId,
            ...props,
        });
    }
    /**
     * Add an Elastic Load Balancer as an endpoint in this endpoint group
     */
    addLoadBalancer(id, lb, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: lb.loadBalancerArn,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EIP as an endpoint in this endpoint group
     */
    addElasticIpAddress(id, eip, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: eip.attrAllocationId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EC2 Instance as an endpoint in this endpoint group
     */
    addEc2Instance(id, instance, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: instance.instanceId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Links a endpoint to this endpoint group
     * @internal
     */
    _linkEndpoint(endpoint) {
        this.endpoints.push(endpoint);
    }
    renderEndpoints() {
        return this.endpoints.map(e => e.renderEndpointConfiguration());
    }
}
exports.EndpointGroup = EndpointGroup;
//# sourceMappingURL=data:application/json;base64,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