# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['poethepoet',
 'poethepoet.completion',
 'poethepoet.executor',
 'poethepoet.helpers',
 'poethepoet.task']

package_data = \
{'': ['*']}

install_requires = \
['pastel>=0.2.1,<0.3.0', 'tomli>=1.2.2,<2.0.0']

extras_require = \
{'poetry_plugin': ['poetry>=1.0,<2.0']}

entry_points = \
{'console_scripts': ['poe = poethepoet:main'],
 'poetry.application.plugin': ['poethepoet = poethepoet.plugin:PoetryPlugin']}

setup_kwargs = {
    'name': 'poethepoet',
    'version': '0.12.2',
    'description': 'A task runner that works well with poetry.',
    'long_description': '************\nPoe the Poet\n************\n\nA task runner that works well with poetry.\n\n.. role:: bash(code)\n   :language: bash\n.. role:: fish(code)\n   :language: fish\n.. role:: zsh(code)\n   :language: zsh\n.. role:: toml(code)\n   :language: toml\n.. role:: python(code)\n   :language: python\n.. |•| unicode:: ✅ 0xA0 0xA0\n   :trim:\n\nFeatures\n========\n\n|•| Straight forward declaration of project tasks in your pyproject.toml (kind of like npm scripts)\n\n|•| Task are run in poetry\'s virtualenv (or another env you specify)\n\n|•| Shell completion of task names (and global options too for zsh)\n\n|•| Can be used standalone or as a poetry plugin\n\n|•| Tasks can be commands (with or without a shell) or references to python functions (like tool.poetry.scripts)\n\n|•| Short and sweet commands with extra arguments passed to the task :bash:`poe [options] task [task_args]`, or you can define arguments explicitly.\n\n|•| Tasks can specify and reference environment variables as if they were evaluated by a shell\n\n|•| Tasks are self documenting, with optional help messages (just run poe without arguments)\n\n|•| Tasks can be defined as a sequence of other tasks\n\n|•| Works with .env files\n\n\nInstallation\n============\n\n1.\n  Install the CLI\n\n  .. code-block:: bash\n\n    pipx install poethepoet\n\n  Or use pip to install into any python environment\n\n  .. code-block:: bash\n\n    pip install poethepoet\n\n2.\n  Or into your project (so it works inside poetry shell):\n\n  .. code-block:: bash\n\n    poetry add --dev poethepoet\n\n3.\n  Or into poetry as a plugin (requires poetry >= 1.2)\n\n  .. code-block:: bash\n\n    poetry plugin add poethepoet[poetry_plugin]\n\nEnable tab completion for your shell\n------------------------------------\n\nPoe comes with tab completion scripts for bash, zsh, and fish to save you keystrokes.\nHow to install them will depend on your shell setup.\n\nZsh\n~~~\n\n.. code-block:: zsh\n\n  # oh-my-zsh\n  mkdir -p ~/.oh-my-zsh/completions\n  poe _zsh_completion > ~/.oh-my-zsh/completions/_poe\n\n  # without oh-my-zsh\n  mkdir -p ~/.zfunc/\n  poe _zsh_completion > ~/.zfunc/_poetry\n\nNote that you\'ll need to start a new shell for the new completion script to be loaded.\nIf it still doesn\'t work try adding a call to :bash:`compinit` to the end of your zshrc\nfile.\n\nBash\n~~~~\n\n.. code-block:: bash\n\n  # System bash\n  poe _bash_completion > /etc/bash_completion.d/poe.bash-completion\n\n  # Homebrew bash\n  poe _bash_completion > $(brew --prefix)/etc/bash_completion.d/poe.bash-completion\n\n\nHow to ensure installed bash completions are enabled may vary depending on your system.\n\nFish\n~~~~\n\n.. code-block:: fish\n\n  # Fish\n  poe _fish_completion > ~/.config/fish/completions/poe.fish\n\n  # Homebrew fish\n  poe _fish_completion > (brew --prefix)/share/fish/vendor_completions.d/poe.fish\n\n\nBasic Usage\n===========\n\nDefine tasks in your pyproject.toml\n-----------------------------------\n\n`See a real example <https://github.com/nat-n/poethepoet/blob/master/pyproject.toml>`_\n\n.. code-block:: toml\n\n  [tool.poe.tasks]\n  test   = "pytest --cov=poethepoet"                                # simple command based task\n  serve  = { script = "my_app.service:run(debug=True)" }            # python script based task\n  tunnel = { shell = "ssh -N -L 0.0.0.0:8080:$PROD:8080 $PROD &" }  # (posix) shell based task\n\nRun tasks with the poe cli\n--------------------------\n\n.. code-block:: bash\n\n  poe test\n\nBy default additional arguments are passed to the task so\n\n.. code-block:: bash\n\n  poe test -v tests/favorite_test.py\n\nresults in the following being run inside poetry\'s virtualenv\n\n.. code-block:: bash\n\n  pytest --cov=poethepoet -v tests/favorite_test.py\n\nYou can also run it like so if you fancy\n\n.. code-block:: bash\n\n  python -m poethepoet [options] task [task_args]\n\nOr use it as a poetry plugin (for poetry >= 1.2) like so\n\n.. code-block:: bash\n\n  poetry plugin add poethepoet[poetry_plugin]\n  poetry poe [options] task [task_args]\n\nOr just install it as a dev dependency with poetry and run it like\n\n.. code-block:: bash\n\n  poetry add --dev poethepoet\n  poetry run poe [options] task [task_args]\n\nThough in that case you might like to define :bash:`alias poe=\'poetry run poe\'`.\n\nTypes of task\n=============\n\nThere are four types of task: simple commands *(cmd)*, python scripts *(script)*, shell\nscripts *(shell)*, and sequence tasks *(sequence)*.\n\n- **Command tasks** contain a single command that will be executed without a shell.\n  This covers most basic use cases for example:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n    format = "black ."  # strings are interpreted as commands by default\n    clean = """\n    # Multiline commands including comments work too. Unescaped whitespace is ignored.\n    rm -rf .coverage\n           .mypy_cache\n           .pytest_cache\n           dist\n           ./**/__pycache__\n    """\n    lint = { "cmd": "pylint poethepoet" }  # Inline tables with a cmd key work too\n    greet = "echo Hello $USER"  # Environment variables work, even though there\'s no shell!\n\n- **Script tasks** contain a reference to a python callable to import and execute, for\n  example:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n    fetch-assets = { "script" = "my_package.assets:fetch" }\n    fetch-images = { "script" = "my_package.assets:fetch(only=\'images\', log=environ[\'LOG_PATH\'])" }\n\n  As in the second example, is it possible to hard code literal arguments to the target\n  callable. In fact a subset of python syntax, operators, and globals can be used inline\n  to define the arguments to the function using normal python syntax, including environ\n  (from the os package) to access environment variables that are available to the task.\n\n  If extra arguments are passed to task on the command line (and no CLI args are\n  declared), then they will be available within the called python function via\n  :python:`sys.argv`.\n\n- **Shell tasks** are similar to simple command tasks except that they are executed\n  inside a new shell, and can consist of multiple separate commands, command\n  substitution, pipes, background processes, etc.\n\n  An example use case for this might be opening some ssh tunnels in the background with\n  one task and closing them with another like so:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n    pfwd = { "shell" = "ssh -N -L 0.0.0.0:8080:$STAGING:8080 $STAGING & ssh -N -L 0.0.0.0:5432:$STAGINGDB:5432 $STAGINGDB &" }\n    pfwdstop = { "shell" = "kill $(pgrep -f "ssh -N -L .*:(8080|5432)")" }\n\n  By default poe attempts to find a posix shell (sh, bash, or zsh in that order) on the system and uses that. When running on windows, this might not always be possible. If bash is not found on the path on windows then poe will explicitly look for `Git bash <https://gitforwindows.org>`_ at the usual location.\n\n  **Using different types of shell/interpreter**\n\n  It is also possible to specify an alternative interpreter (or list of compatible interpreters ordered by preference) to be invoked to execute shell task content. For example if you only expect the task to be executed on windows or other environments with powershell installed then you can specify a powershell based task like so:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks.install-poetry]\n    shell = """\n    (Invoke-WebRequest -Uri https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py -UseBasicParsing).Content | python -\n    """\n    interpreter = "pwsh"\n\n  If your task content is restricted to syntax that is valid for both posix shells and powershell then you can maximise increase the likelihood of it working on any system by specifying the interpreter as:\n\n  .. code-block:: toml\n\n    interpreter = ["posix", "pwsh"]\n\n  It is also possible to specify python code as the shell task code as in the following example. However it is recommended to use a *script* task rather than writing complex code inline within your pyproject.toml.\n\n  .. code-block:: toml\n\n    [tool.poe.tasks.time]\n    shell = """\n    from datetime import datetime\n\n    print(datetime.now())\n    """\n    interpreter = "python"\n\n  The following interpreter values may be used:\n\n  posix\n      This is the default behavoir, equivalent to ["sh", "bash", "zsh"], meaning that poe will try to find sh, and fallback to bash, then zsh.\n  sh\n      Use the basic posix shell. This is often an alias for bash or dash depending on the operating system.\n  bash\n      Uses whatever version of bash can be found. This is usually the most portable option.\n  zsh\n      Uses whatever version of zsh can be found.\n  fish\n      Uses whatever version of fish can be found.\n  pwsh\n      Uses powershell version 6 or higher.\n  powershell\n      Uses the newest version of powershell that can be found.\n\n  The default value can be changed with the global *shell_interpreter* option as described below.\n\n- **Composite tasks** are defined as a sequence of other tasks as an array.\n\n  By default the contents of the array are interpreted as references to other tasks\n  (actually a ref task type), though this behaviour can be altered by setting the global\n  :toml:`default_array_item_task_type` option to the name of another task type such as\n  *cmd*, or by setting the :toml:`default_item_type` option locally on the sequence task.\n\n  **An example task with references**\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n\n    test = "pytest --cov=src"\n    build = "poetry build"\n    _publish = "poetry publish"\n    release = ["test", "build", "_publish"]\n\n  Note that tasks with names prefixed with :code:`_` are not included in the\n  documentation or directly executable, but can be useful for cases where a task is only\n  needed for referencing from another task.\n\n  **An example task with inline tasks expressed via inline tables**\n\n  .. code-block:: toml\n\n    release = [\n      { cmd = "pytest --cov=src" },\n      { script = "devtasks:build" },\n      { ref = "_publish" },\n    ]\n\n  **An example task with inline tasks expressed via an array of tables**\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n\n      [[tool.poe.tasks.release]]\n      cmd = "pytest --cov=src"\n\n      [[tool.poe.tasks.release]]\n      script = "devtasks:build"\n\n      [[tool.poe.tasks.release]]\n      ref = "_publish"\n\n  **An example task with inline script subtasks using default_item_type**\n\n  .. code-block:: toml\n\n    release.sequence = [\n      "devtasks:run_tests(all=True)",\n      "devtasks:build",\n      "devtasks:publish",\n    ]\n    release.default_item_type = "script"\n\n  A failure (non-zero result) will result in the rest of the tasks in the sequence not\n  being executed, unless the :toml:`ignore_fail` option is set on the task to\n  :toml:`true` or :toml:`"return_zero"` like so:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n    attempts.sequence = ["task1", "task2", "task3"]\n    attempts.ignore_fail = "return_zero"\n\n  If you want to run all the subtasks in the sequence but return non-zero result in the\n  end of the sequence if any of the subtasks have failed you can set :toml:`ignore_fail`\n  option to the :toml:`return_non_zero` value like so:\n\n  .. code-block:: toml\n\n    [tool.poe.tasks]\n    attempts.sequence = ["task1", "task2", "task3"]\n    attempts.ignore_fail = "return_non_zero"\n\nTask level configuration\n========================\n\nTask help text\n--------------\n\nYou can specify help text to be shown alongside the task name in the list of available\ntasks (such as when executing poe with no arguments), by adding a help key like so:\n\n.. code-block:: toml\n\n    [tool.poe.tasks]\n    style = {cmd = "black . --check --diff", help = "Check code style"}\n\nEnvironment variables\n---------------------\n\nYou can specify arbitrary environment variables to be set for a task by providing the\nenv key like so:\n\n.. code-block:: toml\n\n    [tool.poe.tasks]\n    serve.script = "myapp:run"\n    serve.env = { PORT = "9001" }\n\nNotice this example uses deep keys which can be more convenient but aren\'t as well\nsupported by some toml implementations.\n\nThe above example can be modified to only set the `PORT` variable if it is not already\nset by replacing the last line with the following:\n\n.. code-block:: toml\n\n    serve.env.PORT.default = "9001"\n\n\nYou can also specify an env file (with bash-like syntax) to load per task like so:\n\n.. code-block:: bash\n\n    # .env\n    STAGE=dev\n    PASSWORD=\'!@#$%^&*(\'\n\n.. code-block:: toml\n\n    [tool.poe.tasks]\n    serve.script  = "myapp:run"\n    serve.envfile = ".env"\n\nDeclaring CLI arguments\n-----------------------\n\nBy default extra arguments passed to the poe CLI following the task name are appended to\nthe end of a cmd task, or exposed as sys.argv in a script task (but will cause an error\nfor shell or sequence tasks). Alternatively it is possible to define named arguments\nthat a task should accept, which will be documented in the help for that task, and\nexposed to the task in a way the makes the most sense for that task type.\n\nIn general named arguments can take one of the following three forms:\n\n- **positional arguments** which are provided directly following the name of the task like\n   :bash:`poe task-name arg-value`\n\n- **option arguments** which are provided like\n   :bash:`poe task-name --option-name arg-value`\n\n- **flags** which are either provided or not, but don\'t accept a value like\n   :bash:`poe task-name --flag`\n\nThe value for the named argument is then accessible by name within the task content,\nthough exactly how will depend on the type of the task as detailed below.\n\n\nConfiguration syntax\n~~~~~~~~~~~~~~~~~~~~\n\nNamed arguments are configured by declaring the *args* task option as either an array or\na subtable.\n\n\nArray configuration syntax\n""""""""""""""""""""""""""\n\nThe array form may contain string items which are interpreted as an option argument with\nthe given name.\n\n.. code-block:: toml\n\n    [tool.poe.tasks.serve]\n    cmd = "myapp:run"\n    args = ["host", "port"]\n\nThis example can be invoked as\n\n.. code-block:: bash\n\n    poe serve --host 0.0.0.0 --port 8001\n\nItems in the array can also be inline tables to allow for more configuration to be\nprovided to the task like so:\n\n.. code-block:: toml\n\n    [tool.poe.tasks.serve]\n    cmd = "myapp:run"\n    args = [{ name = "host", default = "localhost" }, { name = "port", default = "9000" }]\n\nYou can also use the toml syntax for an array of tables like so:\n\n.. code-block:: toml\n\n    [tool.poe.tasks.serve]\n    cmd = "myapp:run"\n    help = "Run the application server"\n\n      [[tool.poe.tasks.serve.args]]\n      name = "host"\n      options = ["-h", "--host"]\n      help = "The host on which to expose the service"\n      default = "localhost"\n\n      [[tool.poe.tasks.serve.args]]\n      name = "port"\n      options = ["-p", "--port"]\n      help = "The port on which to expose the service"\n      default = "8000"\n\n\nTable configuration syntax\n""""""""""""""""""""""""""\n\nYou can also use the toml syntax for subtables like so:\n\n.. code-block:: toml\n\n    [tool.poe.tasks.serve]\n    cmd = "myapp:run"\n    help = "Run the application server"\n\n      [tool.poe.tasks.serve.args.host]\n      options = ["-h", "--host"]\n      help = "The host on which to expose the service"\n      default = "localhost"\n\n      [tool.poe.tasks.serve.args.port]\n      options = ["-p", "--port"]\n      help = "The port on which to expose the service"\n      default = "8000"\n\nWhen using this form the *name* option is no longer applicable because the key for the\nargument within the args table is taken as the name.\n\nTask argument options\n~~~~~~~~~~~~~~~~~~~~~\n\nNamed arguments support the following configuration options:\n\n- **default** : Union[str, int, float, bool]\n   The value to use if the argument is not provided. This option has no effect if the\n   required option is set to true.\n\n- **help** : str\n   A short description of the argument to include in the documentation of the task.\n\n- **name** : str\n   The name of the task. Only applicable when *args* is an array.\n\n- **options** : List[str]\n   A list of options to accept for this argument, similar to\n   `argsparse name or flags <https://docs.python.org/3/library/argparse.html#name-or-flags>`_.\n   If not provided then the name of the argument is used. You can use this option to\n   expose a different name to the CLI vs the name that is used inside the task, or to\n   specify long and short forms of the CLI option, e.g. ["-h", "--help"].\n\n- **positional** : bool\n   If set to true then the argument becomes a position argument instead of an option\n   argument. Note that positional arguments may not have type *bool*.\n\n- **required** : bool\n   If true then not providing the argument will result in an error. Arguments are not\n   required by default.\n\n- **type** : str\n   The type that the provided value will be cast to. The set of acceptable options is\n   {"string", "float", "integer", "boolean"}. If not provided then the default behaviour\n   is to keep values as strings. Setting the type to "bool" makes the resulting argument\n   a flag that if provided will set the value to the boolean opposite of the default\n   value – i.e. *true* if no default value is given, or false if :toml:`default = true`.\n\n\nArguments for cmd and shell tasks\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nFor cmd and shell tasks the values are exposed to the task as environment variables. For\nexample given the following configuration:\n\n.. code-block:: toml\n\n  [tool.poe.tasks.passby]\n  shell = """\n  echo "hello $planet";\n  echo "goodbye $planet";\n  """\n  help = "Pass by a planet!"\n\n    [[tool.poe.tasks.passby.args]]\n    name = "planet"\n    help = "Name of the planet to pass"\n    default = "earth"\n    options = ["-p", "--planet"]\n\nThe resulting task can be run like:\n\n.. code-block:: bash\n\n  poe passby --planet mars\n\nArguments for script tasks\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nArguments can be defined for script tasks in the same way, but how they are exposed to\nthe underlying python function depends on how the script is defined.\n\nIn the following example, since no parenthesis are included for the referenced function,\nall provided args will be passed to the function as kwargs:\n\n.. code-block:: toml\n\n  [tool.poe.tasks]\n  build = { script = "project.util:build", args = ["dest", "version"] }\n\nYou can also control exactly how values are passed to the python function as\ndemonstrated in the following example:\n\n.. code-block:: toml\n\n  [tool.poe.tasks]\n  build = { script = "project.util:build(dest, build_version=version, verbose=True)", args = ["dest", "version"]\n\nArguments for sequence tasks\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nArguments can be passed to the tasks referenced from a sequence task as in the following\nexample.\n\n.. code-block:: toml\n\n  [tool.poe.tasks]\n  build = { script = "util:build_app", args = [{ name = "target", positional = true }] }\n\n  [tool.poe.tasks.check]\n  sequence = ["build ${target}", { script = "util:run_tests(environ[\'target\'])" }]\n  args = ["target"]\n\nThis works by setting the argument values as environment variables for the subtasks,\nwhich can be read at runtime, but also referenced in the task definition as\ndemonstrated in the above example for a *ref* task and *script* task.\n\nProject-wide configuration options\n==================================\n\nGlobal environment variables\n----------------------------\n\nYou can configure environment variables to be set for all poe tasks in the\npyproject.toml file by specifying :toml:`tool.poe.env` like so\n\n.. code-block:: toml\n\n  [tool.poe.env]\n  VAR1 = "FOO"\n  VAR2 = "BAR"\n\nAs for the task level option, you can indicated that a variable should only be set if\nnot already set like so:\n\n.. code-block:: toml\n\n  [tool.poe.env]\n  VAR1.default = "FOO"\n\nYou can also specify an env file (with bash-like syntax) to load for all tasks like so:\n\n.. code-block:: bash\n\n    # .env\n    STAGE=dev\n    PASSWORD=\'!@#$%^&*(\'\n\n.. code-block:: toml\n\n    [tool.poe]\n    envfile = ".env"\n\nDefault command verbosity\n-------------------------\n\nYou can alter the verbosity level for poe commands by passing :bash:`--quiet` /\n:bash:`-q` (which decreases verbosity) or :bash:`--verbose` / :bash:`-v` (which\nincreases verbosity) on the CLI.\n\nIf you want to change the default verbosity level for all commands, you can use\nthe :toml:`tool.poe.verbose` option in pyproject.toml like so:\n\n.. code-block:: toml\n\n  [tool.poe]\n  verbosity = -1\n\n:toml:`-1` is the quietest and :toml:`1` is the most verbose. :toml:`0` is the\ndefault.\n\nNote that the command line arguments are incremental: :bash:`-q` subtracts one\nfrom the default verbosity, and :bash:`-v` adds one. So setting the default\nverbosity to :toml:`-1` and passing :bash:`-v -v` on the command line is\nequivalent to setting the verbosity to :toml:`0` and just passing :bash:`-v`.\n\nRun poe from anywhere\n---------------------\n\nBy default poe will detect when you\'re inside a project with a pyproject.toml in the\nroot. However if you want to run it from elsewhere then that is supported by using the\n:bash:`--root` option to specify an alternate location for the toml file. The task will\nrun with the given location as the current working directory.\n\nIn all cases the path to project root (where the pyproject.toml resides) will be\navailable as :bash:`$POE_ROOT` within the command line and process.\n\nChange the default task type\n----------------------------\n\nBy default tasks defined as strings are interpreted as shell commands, and script tasks\nrequire the more verbose table syntax to specify. For example:\n\n.. code-block:: toml\n\n  my_cmd_task = "cmd args"\n  my_script_task = { "script" = "my_package.my_module:run" }\n\nThis behaviour can be reversed by setting the :toml:`default_task_type` option in your\npyproject.toml like so:\n\n.. code-block:: toml\n\n  [tool.poe]\n  default_task_type = "script"\n\n  [tool.poe.tasks]\n  my_cmd_task = { "cmd" = "cmd args" }\n  my_script_task = "my_package.my_module:run"\n\nChange the executor type\n------------------------\n\nYou can configure poe to use a specific executor by setting\n:toml:`tool.poe.executor.type`. Valid values include:\n\n- **auto**: to automatically use the most appropriate of the following executors in order\n- **poetry**: to run tasks in the poetry managed environment\n- **virtualenv**: to run tasks in the indicated virtualenv (or else "./.venv" if present)\n- **simple**: to run tasks without doing any specific environment setup\n\nThe default behaviour is auto.\n\nFor example the following configuration will cause poe to ignore the poetry environment\n(if present), and instead use the virtualenv at the given location relative to the\nparent directory.\n\n.. code-block:: toml\n\n  [tool.poe.executor]\n  type = "virtualenv"\n  location = "myvenv"\n\nSee below for more details.\n\nChange the default shell interpreter\n------------------------------------\n\nNormally shell tasks are executed using a posix shell by default (see section for shell\ntasks above). This default can be overridden to something else by setting the\n*shell_interpreter* global option. In the following example we configure all shell tasks\nto use *fish* by default.\n\n.. code-block:: toml\n\n  tool.poe.shell_interpreter = "fish"\n\n  [tool.poe.tasks.fibonacci]\n  help = "Output the fibonacci sequence up to 89"\n  shell = """\n    function fib --argument-names max n0 n1\n      if test $max -ge $n0\n        echo $n0\n        fib $max $n1 (math $n0 + $n1)\n      end\n    end\n\n    fib 89 1 1\n  """\n\nLoad tasks from another file\n============================\n\nThere are some scenarios where one might wish to define tasks outside of pyproject.toml.\nFor example, if you want to share tasks between projects via git modules, generate tasks\ndefinitions dynamically, or simply have a lot of tasks and don\'t want the pyproject.toml\nto get too large. This can be achieved by creating a toml or json file within your\nproject directory structure including the same structure for tasks as used in\npyproject.toml\n\nFor example:\n\n.. code-block:: toml\n\n  # acme_common/shared_tasks.toml\n  [tool.poe.tasks.build-image]\n  cmd = "docker build"\n\n\n.. code-block:: toml\n\n  [tool.poe]\n  # this references a file from a git submodule\n  include = "modules/acme_common/shared_tasks.toml"\n\nImported files may also specify environment variables via\n:code:`tool.poe.envfile` or entries for :code:`tool.poe.env`.\n\nIt\'s also possible to include tasks from multiple files by providing a list like\nso:\n\n.. code-block:: toml\n\n  [tool.poe]\n  include = ["modules/acme_common/shared_tasks.toml", "generated_tasks.json"]\n\nFiles are loaded in the order specified. If an item already exists then the included value it ignored.\n\nIf a referenced file is missing then poe ignores it without error, though\nfailure to read the contents will result in failure.\n\nUsage as a poetry plugin\n========================\n\nDepending on how you manage your python environments you may also wish to use Poe the\nPoet in the form of a poetry plugin. This requires installing `poethepoet[poetry_plugin]`\neither into the same environment as poetry or into poetry itself.\n`See the poetry docs <https://python-poetry.org/docs/master/plugins/#using-plugins>`_\nfor more details.\n\nDue to how the poetry CLI works (using `cleo <https://github.com/sdispater/cleo>`_ — a\nfeatureful but highly opinionated  CLI framework) there exist a few minor limitations\nwhen used in this way.\n\n1.\n  Normally the poe CLI allows tasks to accept any arguments, either by defining the\n  expected options or by passing any command line tokens following the task name to the\n  task at runtime. This is not supported by cleo. The plugin implements a workaround\n  that mostly works, but still if the `--no-plugins` option is provided *anywhere* in\n  the command line then the poe plugin will never be invoked.\n\n2.\n  Poetry comes with its own\n  `command line completion <https://python-poetry.org/docs/#enable-tab-completion-for-bash-fish-or-zsh>`_,\n  but poe\'s command line completion won\'t work.\n\n3.\n  If you declare named arguments for your poe tasks then these are included in the\n  documentation when poe is invoked without any arguments. However the inline\n  documentation for poetry commands contains only the task names and help text.\n\nTherefore it is recommended to use the poe CLI tool directly if you don\'t mind having\nit installed onto your path.\n\nConfiguring the plugin\n----------------------\n\nBy default the poetry plugin will register *poe* as a command prefix so tasks can be\ninvoked like:\n\n.. code-block:: sh\n\n  poetry poe [task_name] [task_args]\n\nAnd the poe documentation can be viewed via:\n\n.. code-block:: bash\n\n  poetry poe\n\nIt is also possible to modify this behavoir, to either have a different command prefix\nor none at all by setting the :toml:`poetry_command` global option in your\npyproject.toml like so:\n\n.. code-block:: toml\n\n  [tool.poe]\n  poetry_command = ""\n\nIn this case poe tasks will be registered as top level commands on poetry and can be\ninvoked simply as:\n\n.. code-block:: sh\n\n  poetry [task_name]\n\n.. warning::\n    Whatever :toml:`tool.poe.poetry_command` is set to must not already exist as a\n    poetry command!\n\n    Additionally if setting it to the emtpy string then care must be taken to avoid\n    defining any poe tasks that conflict with any other built in or plugin provided\n    poetry command.\n\nHooking into poetry commands\n----------------------------\n\nIt is also possible to configure a task to be run before or after a specific poetry\ncommand by declaring the poetry_hooks global option like so:\n\n.. code-block:: toml\n\n  [tool.poe.poetry_hooks]\n  pre_build  = "prep-assets --verbosity=5"\n  post_build = "archive-build"\n\n  [tool.poe.tasks.prep-assets]\n  script = "scripts:prepare_assets"\n  help   = "Optimise static assets for inclusion in the build"\n\n  [tool.poe.tasks.archive-build]\n  script = "scripts:archive_build"\n  help   = "Upload the latest build version to archive server"\n\nIn this example the :code:`prep-assets` task will be run as the first step in calling\n:bash:`poetry build` with an argument passed as if the task were being called via the\npoe CLI. We\'ve also configured the :code:`archive-build` task to be run after every\nsuccessful build.\n\nIf a task fails when running as a hook, then the poetry command will exit with an error.\nIf it is a *pre* hook then this will cause the actual poetry command not to execute.\nThis behaviour may be useful for running checks before :bash:`poetry publish`\n\nHooks can be disabled for a single invocation by passing the :bash:`--no-plugins` option\nto poetry.\n\nNamespaced commands like :bash:`poetry env info` can be specified with underscores like so:\n\n.. code-block:: toml\n\n  [tool.poe.poetry_hooks]\n  post_env_info = "info"\n\n\nUsage without poetry\n====================\n\nPoe the Poet was originally intended for use alongside poetry. But it works just as\nwell with any other kind of virtualenv, or simply as a general purpose way to define\nhandy tasks for use within a certain directory structure! This behaviour is configurable\nvia the :toml:`tool.poe.executor` global option (see above).\n\nBy default poe will run tasks in the poetry managed environment, if the pyproject.toml\ncontains a :toml:`tool.poetry` section. If it doesn\'t then poe looks for a virtualenv to\nuse from :bash:`./.venv` or :bash:`./venv` relative to the pyproject.toml file.\nOtherwise it falls back to running tasks without any special environment management.\n\nComposing tasks into graphs (Experimental)\n==========================================\n\nYou can define tasks that depend on other tasks, and optionally capture and reuse the\noutput of those tasks, thus defining an execution graph of tasks. This is done by using\nthe *deps* task option, or if you want to capture the output of the upstream task to\npass it to the present task then specify the *uses* option, as demonstrated below.\n\n.. code-block:: toml\n\n  [tool.poe.tasks]\n  _website_bucket_name.shell = """\n    aws cloudformation describe-stacks \\\n      --stack-name $AWS_SAM_STACK_NAME \\\n      --query "Stacks[0].Outputs[?(@.OutputKey == \'FrontendS3Bucket\')].OutputValue" \\\n    | jq -cr \'select(0)[0]\'\n  """\n\n  [tool.poe.tasks.build-backend]\n  help = "Build the backend"\n  sequence = [\n    {cmd = "poetry export -f requirements.txt --output src/requirements.txt"},\n    {cmd = "sam build"},\n  ]\n\n  [tool.poe.tasks.build-frontend]\n  help = "Build the frontend"\n  cmd = "npm --prefix client run build"\n\n  [tool.poe.tasks.shipit]\n  help = "Build and deploy the app"\n  sequence = [\n    "sam deploy --config-env $SAM_ENV_NAME",\n    "aws s3 sync --delete ./client/build s3://${BUCKET_NAME}"\n  ]\n  default_item_type = "cmd"\n  deps = ["build-frontend", "build-backend"]\n  uses = { BUCKET_NAME = "_website_bucket_name" }\n\nIn this example the *shipit* task depends on the *build-frontend* *build-backend*, which\nmeans that these tasks get executed before the *shipit* task. It also declares that it\nuses the output of the hidden *_website_bucket_name* task, which means that this also\ngets executed, but its output it captured and then made available to the *shipit* task\nas the environment variable BUCKET_NAME.\n\nThis feature is experimental. There may be edge cases that aren\'t handled well, so\nfeedback is requested. Some details of the implementation or API may be altered in\nfuture versions.\n\nSupported python versions\n=========================\n\nPoe the Poet officially supports python >3.6.2, and is tested with python 3.6 to 3.9 on\nmacOS, linux and windows.\n\nContributing\n============\n\nThere\'s plenty to do, come say hi in\n`the issues <https://github.com/nat-n/poethepoet/issues>`_! 👋\n\nAlso check out the\n`CONTRIBUTING.MD <https://github.com/nat-n/poethepoet/blob/main/.github/CONTRIBUTING.md>`_ 🤓\n\nLicence\n=======\n\nMIT.\n',
    'author': 'Nat Noordanus',
    'author_email': 'n@natn.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nat-n/poethepoet',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2',
}


setup(**setup_kwargs)
