import numpy as np
import scipy.linalg

import materia as mtr


class SymmetryOperation:
    def __init__(self, matrix=None, determinant=None, trace=None, axis=None):
        if matrix is not None:
            self.matrix, _ = scipy.linalg.polar(matrix)
        else:
            if axis is None:
                self.matrix, _ = scipy.linalg.polar(
                    determinant * np.eye(3).astype("float64")
                )
            else:
                cos_theta = max(min((trace - determinant) / 2, 1), -1)
                sin_theta = max(min(np.sqrt(1 - cos_theta ** 2), 1), -1)

                u1, u2, u3 = axis
                K = np.array([[0, -u3, u2], [u3, 0, -u1], [-u2, u1, 0]])

                rotation = (
                    np.eye(3) + sin_theta * K + (1 - cos_theta) * (K @ K)
                ).astype("float64")
                reflection = np.eye(3) - (determinant == -1) * 2 * np.outer(axis, axis)

                self.matrix, _ = scipy.linalg.polar(rotation @ reflection)

    def __eq__(self, other):
        # print(self.axis or [])
        # print(other.axis or [])
        # return np.allclose((self.det,self.tr),(other.det,other.tr)) and np.allclose(self.axis if self.axis is not None else [],other.axis if other.axis is not None else [])
        return hasattr(other, "matrix") and np.allclose(
            self.matrix, other.matrix, atol=1e-3
        )

    @property
    def det(self):
        return int(round(np.linalg.det(self.matrix)))

    @property
    def tr(self):
        return np.trace(self.matrix)

    @property
    def cos_theta(self):
        return max(min((self.tr - np.sign(self.det)) / 2, 1), -1)

    @property
    def axis(self):
        if np.isclose(abs(self.tr), 3):
            return None

        S = self.matrix + self.matrix.T - (self.tr - np.sign(self.det)) * np.eye(3)
        S[np.isclose(S, 0)] = 0

        axis = np.sqrt(
            np.abs(np.diag(S) / (3 - np.sign(self.det) * self.tr))
        )  # scipy.linalg.null_space(R-det*np.eye(3)).squeeze()
        u = mtr.perpendicular_vector(axis)
        axis *= np.sign(np.dot(axis, np.cross(u, self.matrix @ u)))

        return axis

    @property
    def inverse(self):
        return SymmetryOperation(matrix=self.matrix.T)

    def is_symmetry_of(self, structure, symprec):
        transformed_atomic_positions = (
            self.matrix @ structure.centered_atomic_positions.value
        )
        return set(
            tuple(row) for row in transformed_atomic_positions.T.round(symprec)
        ) == set(
            tuple(row)
            for row in structure.centered_atomic_positions.value.T.round(symprec)
        )

    @property
    def order(self):
        return mtr.periodicity(self.matrix)

    def __mul__(self, other):
        return SymmetryOperation(matrix=self.matrix @ other.matrix)


class Inversion(SymmetryOperation):
    def __init__(self):
        determinant = -1
        trace = -3
        axis = None
        super().__init__(determinant=determinant, trace=trace, axis=axis)


class Reflection(SymmetryOperation):
    def __init__(self, axis):
        determinant = -1
        trace = 1
        super().__init__(determinant=determinant, trace=trace, axis=axis)


class ProperRotation(SymmetryOperation):
    def __init__(self, order, axis):
        determinant = 1
        trace = 2 * np.cos(2 * np.pi / order) + determinant
        super().__init__(determinant=determinant, trace=trace, axis=axis)

    def __repr__(self) -> str:
        return f"ProperRotation(order={self.order})"


class ImproperRotation(SymmetryOperation):
    def __init__(self, order, axis):
        determinant = -1
        trace = 2 * np.cos(2 * np.pi / order) + determinant
        super().__init__(determinant=determinant, trace=trace, axis=axis)
