from enum import Enum

# Converts a hexadecimal string to a base 10 integer.
def hex_to_int(hex_to_convert):
    return int(hex_to_convert, 16)


# Reverses the endianness of a hexadecimal string.
def reverse_endianness(hex_to_reverse):
    assert (len(hex_to_reverse) % 2) == 0
    hex_pairs = []
    for i in range(len(hex_to_reverse) // 2):
        hex_pairs.append(hex_to_reverse[2 * i] + hex_to_reverse[2 * i + 1])
    reversed_hex_pairs = hex_pairs[::-1]
    output_string = ""
    for pair in reversed_hex_pairs:
        output_string += pair
    return output_string


# Converts an integer to a hexadecimal string.
def int_to_hex(int_to_convert):
    return (hex(int_to_convert)[2:]).upper()


# Pads a hexadecimal string with 0's until it meets the provided length.
def pad_hex(hex_to_pad, num_hex_characters):
    return_hex = hex_to_pad
    while len(return_hex) < num_hex_characters:
        return_hex = "0" + return_hex
    return return_hex


class Direction(Enum):
    RIGHT = 0x00
    LEFT = 0x01
    DOWN = 0x02
    UP = 0x03


class DoorData:
    def __init__(self, door_hex):
        # Pointer to this DDB entry's location in memory
        # Not an actually contained in the DDB entry
        # Relative to bank 83 (0x010000 in file)
        self.door_ptr = hex_to_int(reverse_endianness(door_hex[0:4]))
        # Pointer to the room this door leads to
        # Relative to bank 8f (0x070000 in file)
        self.room_ptr = hex_to_int(reverse_endianness(door_hex[4:8]))
        # Bitflags used to handle state.
        # 0x40 if destination is different region
        # 0x00 if destination is in same region
        self.bitflag = hex_to_int(reverse_endianness(door_hex[8:10]))
        # Direction door goes to
        self.direction = hex_to_int(reverse_endianness(door_hex[10:12]))
        # X position of the destination door cap in the room
        self.door_cap_x = hex_to_int(reverse_endianness(door_hex[12:14]))
        # Y position of the destination door cap in the room
        self.door_cap_y = hex_to_int(reverse_endianness(door_hex[14:16]))
        # Automap X position of the destination door cap in the room
        self.screen_x = hex_to_int(reverse_endianness(door_hex[16:18]))
        # Automap Y position of the destination door cap in the room
        self.screen_y = hex_to_int(reverse_endianness(door_hex[18:20]))
        # How far Samus should spawn in front of the destination door
        # Set to 0x0080 to use the default distnace
        self.distance_to_spawn = hex_to_int(reverse_endianness(door_hex[20:24]))
        # Pointer to ASM to run
        self.door_asm_pointer = hex_to_int(reverse_endianness(door_hex[24:28]))

        # Boolean - marks whether this door has a direction mismatch with the destination door
        self.door_mismatch = False
        # Boolean - marks whether this door has had its destination altered
        self.door_redirect = False
        # Defines an override position for Samus in the case of incompatible door transition
        self.override_x = 0x0000
        self.override_y = 0x0000
        # Extra ASM to be tacked onto this object for specific rooms.
        # Hardcoded, externally to this.
        self.extra_asm_pointer = 0x0000

    # dest_room_addr - the address of the room we want this room to lead to
    # dest_door_index - the index of the door data for the door we want to emerge from
    def change_destination(self, dest_room_addr, dest_door_index):
        assert SMRooms.addr_to_num_doors[dest_room_addr] > dest_door_index
        original_direction = self.direction

        # Alter bitflag to match
        # Compare region of source to that of destination
        source_room_addr = SMRooms.door_addr_to_room_dict[self.door_ptr]
        if SMRooms.addr_to_region[source_room_addr] == SMRooms.addr_to_region[dest_room_addr]:
            self.bitflag &= 0xFF - 0x04
        else:
            self.bitflag |= 0x40

        # Set direction and position of destination
        # First find the door data that originally lead to our destination
        # We do this using the hardcoded set of vanilla door data.

        # First get the data for where the door we want to come out of would ordinarily lead back to.
        dest_room_door_data = DoorData((SMRooms.addr_to_door_data_dict[dest_room_addr])[dest_door_index])

        # Next get all of the door data from the room it leads back to,
        # Then find the data for the door that would take us to our destination in vanilla.
        vanilla_source_room_addr = SMRooms.door_addr_to_room_dict[dest_room_door_data.door_ptr]
        # print(int_to_hex(vanilla_source_room_addr))
        vanilla_source_room_door_data = None
        # Get all doors in the destination
        for door_data in [DoorData(door_hex) for door_hex in SMRooms.addr_to_door_data_dict[vanilla_source_room_addr]]:
            # For each of those doors, go to the room they lead to, and search each door to find one that leads back to destination room.
            for return_door_data in [
                DoorData(subdoor_hex) for subdoor_hex in SMRooms.addr_to_door_data_dict[door_data.room_ptr]
            ]:
                if return_door_data.room_ptr == dest_room_addr:
                    if dest_room_door_data.room_ptr == SMRooms.door_addr_to_room_dict[return_door_data.door_ptr]:
                        vanilla_source_room_door_data = return_door_data
                        break

        self.direction = vanilla_source_room_door_data.direction
        self.door_cap_x = vanilla_source_room_door_data.door_cap_x
        self.door_cap_y = vanilla_source_room_door_data.door_cap_y
        self.screen_x = vanilla_source_room_door_data.screen_x
        self.screen_y = vanilla_source_room_door_data.screen_y
        self.distance_to_spawn = vanilla_source_room_door_data.distance_to_spawn
        self.door_asm_pointer = vanilla_source_room_door_data.door_asm_pointer

        self.door_redirect = True

        # Validate the door directions
        # If the door that leads to our destination in Vanilla has a different direction from this door,
        # Then there is a door mismatch.
        if vanilla_source_room_door_data.direction != original_direction:

            self.distance_to_spawn = 0x00
            self.door_mismatch = True

            # Get direction of doorcap
            true_direction = self.direction
            if true_direction >= 4:
                true_direction -= 4

            # Calculate an appropriate position for Samus (in pixels)
            # We start with the position of doorcap in pixels
            # The position stored in the DDB is in tiles
            samus_x = 0x10 * self.door_cap_x + 4
            samus_y = 0x10 * self.door_cap_y + 40

            # Offset Samus by one tile in appropriate direction
            # If the direction is LEFT, that means the door cap will be to the left of Samus, and thus we would move her right.
            # Similarly, we will want to move Samus in the opposite direction to distance her from the door cap.
            # So to move her away from the door we want to move her in the opposite direction.
            if true_direction == Direction.RIGHT.value:
                samus_x -= 0x10
            elif true_direction == Direction.LEFT.value:
                samus_x += 0x10
            elif true_direction == Direction.DOWN.value:
                # samus_y -= 0x10
                samus_x += 0x20
            elif true_direction == Direction.UP.value:
                samus_y += 0x40
                samus_x += 0x20

            self.override_x = samus_x
            self.override_y = samus_y
            # Modify transition direction and doorcap behavior.
            # Change direction to be the direction that you would ordinarily come into this room with,
            # Rather than the direction the original door data would be sending you in.
            # Also remove the door cap closing behind the player.
            # If horizontal, keep the original direction and the door cap.
            if DoorData.is_dir_horizontal(self.direction) and DoorData.is_dir_horizontal(
                vanilla_source_room_door_data.direction
            ):
                if self.direction >= 4:
                    self.direction -= 4
            else:
                self.direction = original_direction
                if self.direction >= 4:
                    self.direction -= 4

        # Finally, change the room pointer to match the destination
        self.room_ptr = dest_room_addr

    @staticmethod
    def is_dir_horizontal(direction):
        if direction >= 4:
            direction -= 4
        return direction <= 1

    # Note that the door pointer is not an actual part of the DDB entry
    # It just points to where it sits in memory.
    def write_ddb_entry_to_file(self, f):
        f.seek(0x010000 + self.door_ptr)
        f.write(self.room_ptr.to_bytes(2, "little"))
        f.write(self.bitflag.to_bytes(1, "little"))
        f.write(self.direction.to_bytes(1, "little"))
        f.write(self.door_cap_x.to_bytes(1, "little"))
        f.write(self.door_cap_y.to_bytes(1, "little"))
        f.write(self.screen_x.to_bytes(1, "little"))
        f.write(self.screen_y.to_bytes(1, "little"))
        f.write(self.distance_to_spawn.to_bytes(2, "little"))
        f.write(self.door_asm_pointer.to_bytes(2, "little"))

    def to_string(self):
        print(pad_hex(int_to_hex(self.door_ptr), 4))
        print(pad_hex(int_to_hex(self.room_ptr), 4))
        print(pad_hex(int_to_hex(self.bitflag), 2))
        print(pad_hex(int_to_hex(self.direction), 2))
        print(pad_hex(int_to_hex(self.door_cap_x), 2))
        print(pad_hex(int_to_hex(self.door_cap_y), 2))
        print(pad_hex(int_to_hex(self.screen_x), 2))
        print(pad_hex(int_to_hex(self.screen_y), 2))
        print(pad_hex(int_to_hex(self.distance_to_spawn), 4))
        print(pad_hex(int_to_hex(self.door_asm_pointer), 4))
        print("")


class SMRooms:
    addr_to_num_doors = {
        0x91F8: 4,
        0x92B3: 2,
        0x92FD: 7,
        0x93AA: 1,
        0x93D5: 1,
        0x93FE: 6,
        0x9461: 2,
        0x948C: 3,
        0x94CC: 3,
        0x94FD: 2,
        0x9552: 2,
        0x957D: 2,
        0x95A8: 2,
        0x95D4: 2,
        0x95FF: 2,
        0x962A: 3,
        0x965B: 2,
        0x968F: 2,
        0x96BA: 5,
        0x975C: 2,
        0x97B5: 3,
        0x9804: 1,
        0x9879: 2,
        0x98E2: 2,
        0x990D: 2,
        0x9938: 3,
        0x9969: 2,
        0x9994: 1,
        0x99BD: 4,
        0x99F9: 2,
        0x9A44: 2,
        0x9A90: 1,
        0x9AD9: 11,
        0x9B5B: 2,
        0x9B9D: 2,
        0x9BC8: 2,
        0x9C07: 1,
        0x9C35: 1,
        0x9C5E: 2,
        0x9C89: 1,
        0x9CB3: 3,
        0x9D19: 9,
        0x9D9C: 2,
        0x9DC7: 2,
        0x9E11: 2,
        0x9E52: 3,
        0x9E9F: 4,
        0x9F11: 3,
        0x9F64: 2,
        0x9FBA: 2,
        0x9FE5: 2,
        0xA011: 4,
        0xA051: 1,
        0xA07B: 1,
        0xA0A4: 2,
        0xA0D2: 1,
        0xA107: 1,
        0xA130: 2,
        0xA15B: 1,
        0xA184: 1,
        0xA1AD: 2,
        0xA1D8: 1,
        0xA201: 1,
        0xA22A: 1,
        0xA253: 5,
        0xA293: 2,
        0xA2CE: 1,
        0xA2F7: 2,
        0xA322: 7,
        0xA37C: 1,
        0xA3AE: 1,
        0xA3DD: 2,
        0xA408: 3,
        0xA447: 1,
        0xA471: 3,
        0xA4B1: 1,
        0xA4DA: 3,
        0xA521: 2,
        0xA56B: 3,
        0xA59F: 2,
        0xA5ED: 2,
        0xA618: 1,
        0xA641: 1,
        0xA66A: 3,
        0xA6A1: 4,
        0xA6E2: 1,
        0xA70B: 1,
        0xA734: 1,
        0xA75D: 2,
        0xA788: 2,
        0xA7B3: 2,
        0xA7DE: 8,
        0xA815: 4,
        0xA865: 2,
        0xA890: 1,
        0xA8B9: 3,
        0xA8F8: 2,
        0xA923: 5,
        0xA98D: 2,
        0xA9E5: 1,
        0xAA0E: 2,
        0xAA41: 2,
        0xAA82: 4,
        0xAAB5: 1,
        0xAADE: 1,
        0xAB07: 4,
        0xAB3B: 1,
        0xAB64: 2,
        0xAB8F: 2,
        0xABD2: 2,
        0xAC00: 2,
        0xAC2B: 2,
        0xAC5A: 1,
        0xAC83: 2,
        0xACB3: 7,
        0xACF0: 2,
        0xAD1B: 1,
        0xAD5E: 5,
        0xADAD: 3,
        0xADDE: 1,
        0xAE07: 2,
        0xAE32: 2,
        0xAE74: 4,
        0xAEB4: 2,
        0xAEDF: 3,
        0xAF14: 2,
        0xAF3F: 4,
        0xAF72: 3,
        0xAFA3: 2,
        0xAFCE: 3,
        0xAFFB: 2,
        0xB026: 2,
        0xB051: 1,
        0xB07A: 2,
        0xB0B4: 1,
        0xB0DD: 1,
        0xB106: 2,
        0xB139: 2,
        0xB167: 2,
        0xB192: 1,
        0xB1BB: 1,
        0xB1E5: 2,
        0xB236: 4,
        0xB283: 2,
        0xB2DA: 2,
        0xB305: 1,
        0xB32E: 2,
        0xB37A: 2,
        0xB3A5: 5,
        0xB3E1: 1,
        0xB40A: 2,
        0xB457: 2,
        0xB482: 2,
        0xB4AD: 3,
        0xB4E5: 2,
        0xB510: 3,
        0xB55A: 2,
        0xB585: 4,
        0xB5D5: 2,
        0xB62B: 2,
        0xB656: 2,
        0xB698: 1,
        0xB6C1: 3,
        0xB6EE: 3,
        0xB741: 1,
        0xC98E: 3,
        0xCA08: 2,
        0xCA52: 3,
        0xCAAE: 1,
        0xCAF6: 7,
        0xCB8B: 2,
        0xCBD5: 3,
        0xCC27: 1,
        0xCC6F: 3,
        0xCCCB: 1,
        0xCD13: 1,
        0xCD5C: 2,
        0xCDA8: 1,
        0xCDF1: 1,
        0xCE40: 2,
        0xCE8A: 1,
        0xCED2: 1,
        0xCEFB: 4,
        0xCF54: 2,
        0xCF80: 3,
        0xCFC9: 5,
        0xD017: 4,
        0xD055: 1,
        0xD08A: 2,
        0xD0B9: 6,
        0xD104: 2,
        0xD13B: 1,
        0xD16D: 2,
        0xD1A3: 3,
        0xD1DD: 3,
        0xD21C: 4,
        0xD252: 2,
        0xD27E: 2,
        0xD2AA: 1,
        0xD2D9: 2,
        0xD30B: 4,
        0xD340: 4,
        0xD387: 2,
        0xD3B6: 1,
        0xD3DF: 1,
        0xD408: 2,
        0xD433: 3,
        0xD461: 3,
        0xD48E: 3,
        0xD4C2: 3,
        0xD4EF: 2,
        0xD51E: 2,
        0xD54D: 2,
        0xD57A: 2,
        0xD5A7: 6,
        0xD5EC: 2,
        0xD617: 2,
        0xD646: 4,
        0xD69A: 2,
        0xD6D0: 1,
        0xD6FD: 1,
        0xD72A: 3,
        0xD765: 1,
        0xD78F: 2,
        0xD7E4: 4,
        0xD81A: 2,
        0xD845: 1,
        0xD86E: 1,
        0xD898: 2,
        0xD8C5: 2,
        0xD913: 4,
        0xD95E: 2,
        0xD9AA: 1,
        0xD9D4: 1,
        0xD9FE: 2,
        0xDA2B: 2,
        0xDA60: 2,
        0xDAAE: 4,
        0xDAE1: 2,
        0xDB31: 2,
        0xDB7D: 2,
        0xDBCD: 2,
        0xDC19: 2,
        0xDC65: 2,
        0xDCB1: 2,
        0xDCFF: 3,
        0xDD2E: 1,
        0xDD58: 2,
        0xDDC4: 2,
        0xDDF3: 3,
        0xDE23: 1,
        0xDE4D: 2,
        0xDE7A: 2,
        0xDEA7: 2,
        0xDEDE: 2,
        0xDF1B: 1,
        0xDF45: 1,
        0xDF8D: 2,
        0xDFD7: 2,
        0xE021: 2,
        0xE06B: 2,
        0xE0B5: 1,
        0xE82C: 4,
    }

    addr_to_region = {
        0x91F8: "Crateria",
        0x92B3: "Crateria",
        0x92FD: "Crateria",
        0x93AA: "Crateria",
        0x93D5: "Crateria",
        0x93FE: "Crateria",
        0x9461: "Crateria",
        0x948C: "Crateria",
        0x94CC: "Crateria",
        0x94FD: "Crateria",
        0x9552: "Crateria",
        0x957D: "Crateria",
        0x95A8: "Crateria",
        0x95D4: "Crateria",
        0x95FF: "Crateria",
        0x962A: "Crateria",
        0x965B: "Crateria",
        0x968F: "Crateria",
        0x96BA: "Crateria",
        0x975C: "Crateria",
        0x97B5: "Crateria",
        0x9804: "Crateria",
        0x9879: "Crateria",
        0x98E2: "Crateria",
        0x990D: "Crateria",
        0x9938: "Crateria",
        0x9969: "Crateria",
        0x9994: "Crateria",
        0x99BD: "Crateria",
        0x99F9: "Crateria",
        0x9A44: "Crateria",
        0x9A90: "Crateria",
        0x9AD9: "Brinstar",
        0x9B5B: "Brinstar",
        0x9B9D: "Brinstar",
        0x9BC8: "Brinstar",
        0x9C07: "Brinstar",
        0x9C35: "Brinstar",
        0x9C5E: "Brinstar",
        0x9C89: "Brinstar",
        0x9CB3: "Brinstar",
        0x9D19: "Brinstar",
        0x9D9C: "Brinstar",
        0x9DC7: "Brinstar",
        0x9E11: "Brinstar",
        0x9E52: "Brinstar",
        0x9E9F: "Brinstar",
        0x9F11: "Brinstar",
        0x9F64: "Brinstar",
        0x9FBA: "Brinstar",
        0x9FE5: "Brinstar",
        0xA011: "Brinstar",
        0xA051: "Brinstar",
        0xA07B: "Brinstar",
        0xA0A4: "Brinstar",
        0xA0D2: "Brinstar",
        0xA107: "Brinstar",
        0xA130: "Brinstar",
        0xA15B: "Brinstar",
        0xA184: "Brinstar",
        0xA1AD: "Brinstar",
        0xA1D8: "Brinstar",
        0xA201: "Brinstar",
        0xA22A: "Brinstar",
        0xA253: "Brinstar",
        0xA293: "Brinstar",
        0xA2CE: "Brinstar",
        0xA2F7: "Brinstar",
        0xA322: "Brinstar",
        0xA37C: "Brinstar",
        0xA3AE: "Brinstar",
        0xA3DD: "Brinstar",
        0xA408: "Brinstar",
        0xA447: "Brinstar",
        0xA471: "Brinstar",
        0xA4B1: "Brinstar",
        0xA4DA: "Brinstar",
        0xA521: "Brinstar",
        0xA56B: "Brinstar",
        0xA59F: "Brinstar",
        0xA5ED: "Crateria",
        0xA618: "Brinstar",
        0xA641: "Brinstar",
        0xA66A: "Crateria",
        0xA6A1: "Brinstar",
        0xA6E2: "Brinstar",
        0xA70B: "Brinstar",
        0xA734: "Brinstar",
        0xA75D: "Norfair",
        0xA788: "Norfair",
        0xA7B3: "Norfair",
        0xA7DE: "Norfair",
        0xA815: "Norfair",
        0xA865: "Norfair",
        0xA890: "Norfair",
        0xA8B9: "Norfair",
        0xA8F8: "Norfair",
        0xA923: "Norfair",
        0xA98D: "Norfair",
        0xA9E5: "Norfair",
        0xAA0E: "Norfair",
        0xAA41: "Norfair",
        0xAA82: "Norfair",
        0xAAB5: "Norfair",
        0xAADE: "Norfair",
        0xAB07: "Norfair",
        0xAB3B: "Norfair",
        0xAB64: "Norfair",
        0xAB8F: "Norfair",
        0xABD2: "Norfair",
        0xAC00: "Norfair",
        0xAC2B: "Norfair",
        0xAC5A: "Norfair",
        0xAC83: "Norfair",
        0xACB3: "Norfair",
        0xACF0: "Norfair",
        0xAD1B: "Norfair",
        0xAD5E: "Norfair",
        0xADAD: "Norfair",
        0xADDE: "Norfair",
        0xAE07: "Norfair",
        0xAE32: "Norfair",
        0xAE74: "Norfair",
        0xAEB4: "Norfair",
        0xAEDF: "Norfair",
        0xAF14: "Norfair",
        0xAF3F: "Norfair",
        0xAF72: "Norfair",
        0xAFA3: "Norfair",
        0xAFCE: "Norfair",
        0xAFFB: "Norfair",
        0xB026: "Norfair",
        0xB051: "Norfair",
        0xB07A: "Norfair",
        0xB0B4: "Norfair",
        0xB0DD: "Norfair",
        0xB106: "Norfair",
        0xB139: "Norfair",
        0xB167: "Norfair",
        0xB192: "Norfair",
        0xB1BB: "Norfair",
        0xB1E5: "Norfair",
        0xB236: "Norfair",
        0xB283: "Norfair",
        0xB2DA: "Norfair",
        0xB305: "Norfair",
        0xB32E: "Norfair",
        0xB37A: "Norfair",
        0xB3A5: "Norfair",
        0xB3E1: "Norfair",
        0xB40A: "Norfair",
        0xB457: "Norfair",
        0xB482: "Norfair",
        0xB4AD: "Norfair",
        0xB4E5: "Norfair",
        0xB510: "Norfair",
        0xB55A: "Norfair",
        0xB585: "Norfair",
        0xB5D5: "Norfair",
        0xB62B: "Norfair",
        0xB656: "Norfair",
        0xB698: "Norfair",
        0xB6C1: "Norfair",
        0xB6EE: "Norfair",
        0xB741: "Norfair",
        0xC98E: "Wrecked Ship",
        0xCA08: "Wrecked Ship",
        0xCA52: "Wrecked Ship",
        0xCAAE: "Wrecked Ship",
        0xCAF6: "Wrecked Ship",
        0xCB8B: "Wrecked Ship",
        0xCBD5: "Wrecked Ship",
        0xCC27: "Wrecked Ship",
        0xCC6F: "Wrecked Ship",
        0xCCCB: "Wrecked Ship",
        0xCD13: "Wrecked Ship",
        0xCD5C: "Wrecked Ship",
        0xCDA8: "Wrecked Ship",
        0xCDF1: "Wrecked Ship",
        0xCE40: "Wrecked Ship",
        0xCE8A: "Wrecked Ship",
        0xCED2: "Maridia",
        0xCEFB: "Maridia",
        0xCF54: "Maridia",
        0xCF80: "Maridia",
        0xCFC9: "Maridia",
        0xD017: "Maridia",
        0xD055: "Maridia",
        0xD08A: "Maridia",
        0xD0B9: "Maridia",
        0xD104: "Maridia",
        0xD13B: "Maridia",
        0xD16D: "Maridia",
        0xD1A3: "Maridia",
        0xD1DD: "Maridia",
        0xD21C: "Maridia",
        0xD252: "Maridia",
        0xD27E: "Maridia",
        0xD2AA: "Maridia",
        0xD2D9: "Maridia",
        0xD30B: "Maridia",
        0xD340: "Maridia",
        0xD387: "Maridia",
        0xD3B6: "Maridia",
        0xD3DF: "Maridia",
        0xD408: "Maridia",
        0xD433: "Maridia",
        0xD461: "Maridia",
        0xD48E: "Maridia",
        0xD4C2: "Maridia",
        0xD4EF: "Maridia",
        0xD51E: "Maridia",
        0xD54D: "Maridia",
        0xD57A: "Maridia",
        0xD5A7: "Maridia",
        0xD5EC: "Maridia",
        0xD617: "Maridia",
        0xD646: "Maridia",
        0xD69A: "Maridia",
        0xD6D0: "Maridia",
        0xD6FD: "Maridia",
        0xD72A: "Maridia",
        0xD765: "Maridia",
        0xD78F: "Maridia",
        0xD7E4: "Maridia",
        0xD81A: "Maridia",
        0xD845: "Maridia",
        0xD86E: "Maridia",
        0xD898: "Maridia",
        0xD8C5: "Maridia",
        0xD913: "Maridia",
        0xD95E: "Maridia",
        0xD9AA: "Maridia",
        0xD9D4: "Maridia",
        0xD9FE: "Maridia",
        0xDA2B: "Maridia",
        0xDA60: "Maridia",
        0xDAAE: "Tourian",
        0xDAE1: "Tourian",
        0xDB31: "Tourian",
        0xDB7D: "Tourian",
        0xDBCD: "Tourian",
        0xDC19: "Tourian",
        0xDC65: "Tourian",
        0xDCB1: "Tourian",
        0xDCFF: "Tourian",
        0xDD2E: "Tourian",
        0xDD58: "Tourian",
        0xDDC4: "Tourian",
        0xDDF3: "Tourian",
        0xDE23: "Tourian",
        0xDE4D: "Tourian",
        0xDE7A: "Tourian",
        0xDEA7: "Tourian",
        0xDEDE: "Tourian",
        0xDF1B: "Tourian",
        0xDF45: "Ceres Station",
        0xDF8D: "Ceres Station",
        0xDFD7: "Ceres Station",
        0xE021: "Ceres Station",
        0xE06B: "Ceres Station",
        0xE0B5: "Ceres Station",
        0xE82C: "Debug",
    }

    room_name_to_addr_dict = {
        "Landing Site": 0x91F8,
        "Gauntlet Access": 0x92B3,
        "Parlor": 0x92FD,
        "Crateria Power Bomb Room": 0x93AA,
        "Parlor Save Station": 0x93D5,
        "West Ocean": 0x93FE,
        "Bowling Alley Path": 0x9461,
        "Crateria Kihunter Room": 0x948C,
        "Forgotten Highway Elevator": 0x94CC,
        "East Ocean": 0x94FD,
        "Forgotten Highway Kago Room": 0x9552,
        "Crab Maze": 0x957D,
        "Forgotten Highway Elbow": 0x95A8,
        "Crateria Tube": 0x95D4,
        "Moat": 0x95FF,
        "Elevator to Red Brinstar": 0x962A,
        "Gauntlet": 0x965B,
        "Bowling Alley Path": 0x968F,
        "Climb": 0x96BA,
        "Old Mother Brain": 0x975C,
        "Elevator to Blue Brinstar": 0x97B5,
        "Bomb Torizo Room": 0x9804,
        "Flyway": 0x9879,
        "Crateria Map Access": 0x98E2,
        "Terminator Room": 0x990D,
        "Elevator to Green Brinstar": 0x9938,
        "Lower Mushrooms": 0x9969,
        "Crateria Map Station": 0x9994,
        "Green Pirates Shaft": 0x99BD,
        "Crateria Super Room": 0x99F9,
        "Final Missile Bombway": 0x9A44,
        "Final Missile Room": 0x9A90,
        "Green Brinstar Main Shaft & Etecoon Room": 0x9AD9,
        "Spore Spawn Super Room": 0x9B5B,
        "Green Brinstar Map Access": 0x9B9D,
        "Early Supers Room": 0x9BC8,
        "Green Brinstar Reserve Tank Room": 0x9C07,
        "Brinstar Map Station": 0x9C35,
        "Green Brinstar Fireflea Room": 0x9C5E,
        "West Brinstar Missile Station": 0x9C89,
        "Dachora Room": 0x9CB3,
        "Big Pink": 0x9D19,
        "Spore Spawn Kihunter Room": 0x9D9C,
        "Spore Spawn Room": 0x9DC7,
        "Pink Brinstar Power Bomb Room": 0x9E11,
        "Green Hill Zone": 0x9E52,
        "Retro Brinstar": 0x9E9F,
        "Construction Zone": 0x9F11,
        "Blue Brinstar Energy Tank Room": 0x9F64,
        "Noob Bridge": 0x9FBA,
        "Green Brinstar Beetom Room": 0x9FE5,
        "Etecoon Energy Tank Room": 0xA011,
        "Etecoon Super Missile Room": 0xA051,
        "West Brinstar Energy Station": 0xA07B,
        "Spore Spawn Farming Room": 0xA0A4,
        "Waterway": 0xA0D2,
        "First Missiles": 0xA107,
        "Pink Brinstar Hopper Room": 0xA130,
        "Pink Brinstar Energy Tank Room": 0xA15B,
        "Big Pink Save Station": 0xA184,
        "Blue Brinstar Boulder Room": 0xA1AD,
        "Blue Brinstar Double Missile Room": 0xA1D8,
        "Green Brinstar Upper Save Station": 0xA201,
        "Green Brinstar Lower Save Station": 0xA22A,
        "Red Tower": 0xA253,
        "Red Brinstar Fireflea Room": 0xA293,
        "X-Ray Scope Room": 0xA2CE,
        "Red Brinstar Badway": 0xA2F7,
        "Red Brinstar Zoro Room": 0xA322,
        "Red Brinstar Beta Power Bomb Room": 0xA37C,
        "Red Brinstar Alpha Power Bomb Room": 0xA3AE,
        "Red Brinstar Bat Room": 0xA3DD,
        "Red Brinstar Below Spazer": 0xA408,
        "Spazer Room": 0xA447,
        "Warehouse Zeela Room": 0xA471,
        "Warehouse Beetom Room": 0xA4B1,
        "Warehouse Kihunter Room": 0xA4DA,
        "Mini Kraid Room": 0xA521,
        "Kraid Gadora Room": 0xA56B,
        "Kraid's Lair": 0xA59F,
        "Gold Four Hallway": 0xA5ED,
        "Red Tower Energy Charge Station": 0xA618,
        "Warehouse Resupply Station": 0xA641,
        "Gold Four Room": 0xA66A,
        "Warehouse Entrance": 0xA6A1,
        "Varia Suit Room": 0xA6E2,
        "Warehouse Save Station": 0xA70B,
        "Red Tower Save Station": 0xA734,
        "Ice Beam Acid Room": 0xA75D,
        "Cathedral": 0xA788,
        "Cathedral Access": 0xA7B3,
        "Business Center": 0xA7DE,
        "Ice Beam Gate Room": 0xA815,
        "Ice Beam Tutorial Room": 0xA865,
        "Ice Beam Room": 0xA890,
        "Ice Beam Snake Room": 0xA8B9,
        "Crumble Shaft": 0xA8F8,
        "Crocomire Speedway": 0xA923,
        "Crocomire's Room": 0xA98D,
        "Hi-Jump Room": 0xA9E5,
        "Crocomire Escape": 0xAA0E,
        "Hi-Jump Energy Tank Room": 0xAA41,
        "Post Crocomire Farming Room": 0xAA82,
        "Post Crocomire Save Station": 0xAAB5,
        "Post Crocomire Power Bomb Room": 0xAADE,
        "Post Crocomire Shaft": 0xAB07,
        "Cosine Missile Room": 0xAB3B,
        "Grapple Tutorial Room 3": 0xAB64,
        "Post Crocomire Jump Room": 0xAB8F,
        "Grapple Tutorial Room 2": 0xABD2,
        "Grapple Tutorial Room 1": 0xAC00,
        "Grapple Beam Room": 0xAC2B,
        "Bubble Mountain Reserve": 0xAC5A,
        "Bubble Mountain Reserve Access": 0xAC83,
        "Bubble Mountain": 0xACB3,
        "Speed Booster Hall": 0xACF0,
        "Speed Booster Room": 0xAD1B,
        "Single Chamber": 0xAD5E,
        "Double Chamber": 0xADAD,
        "Wave Beam Room": 0xADDE,
        "Spiky Platforms Tunnel": 0xAE07,
        "Volcano": 0xAE32,
        "Kronic Boost Room": 0xAE74,
        "Magdollite Tunnel": 0xAEB4,
        "Purple Shaft": 0xAEDF,
        "Lava Dive": 0xAF14,
        "Elevator to Lower Norfair": 0xAF3F,
        "Upper Norfair Farming Room": 0xAF72,
        "Rising Tide": 0xAFA3,
        "Acid Snakes Tunnel": 0xAFCE,
        "Spiky Acid Snakes Tunnel": 0xAFFB,
        "Peanut Butter Refill": 0xB026,
        "Purple Farming Room": 0xB051,
        "Bat Cave": 0xB07A,
        "Norfair Map Station": 0xB0B4,
        "Bubble Mountain Save Station": 0xB0DD,
        "Norfair Speedway": 0xB106,
        "Red Pirate Shaft": 0xB139,
        "Business Center Save Station": 0xB167,
        "Crocomire Save Station": 0xB192,
        "Elevator to Lower Norfair Save Station": 0xB1BB,
        "Acid Statue Room": 0xB1E5,
        "Main Hall": 0xB236,
        "Golden Torizo's Room": 0xB283,
        "Fast Ripper Room": 0xB2DA,
        "Screw Attack Energy Refill": 0xB305,
        "Ridley's Room": 0xB32E,
        "Ridley Farming Room": 0xB37A,
        "Fast Pillars Setup Room": 0xB3A5,
        "Unused Room": 0xB3E1,
        "Famous Trademarked Mouse Room": 0xB40A,
        "Pillar Room": 0xB457,
        "Plowerhouse Room": 0xB482,
        "The Worst Room in the Game": 0xB4AD,
        "Amphitheatre": 0xB4E5,
        "Lower Norfair Springball Maze Room": 0xB510,
        "Lower Norfair Escape Power Bomb Room": 0xB55A,
        "Red Kihunter Shaft": 0xB585,
        "Wasteland": 0xB5D5,
        "Metal Pirates Room": 0xB62B,
        "Three Musketeers' Room": 0xB656,
        "Ridley Energy Tank Room": 0xB698,
        "Screw Attack Room": 0xB6C1,
        "Lower Norfair Fireflea Room": 0xB6EE,
        "Red Kihunter Shaft Save Station": 0xB741,
        "Bowling Alley": 0xC98E,
        "Wrecked Ship Entrance": 0xCA08,
        "Attic": 0xCA52,
        "Assembly Line": 0xCAAE,
        "Wrecked Ship Main Shaft": 0xCAF6,
        "Spiky Death Room": 0xCB8B,
        "Electric Death Room": 0xCBD5,
        "Wrecked Ship Energy Tank Room": 0xCC27,
        "Basement": 0xCC6F,
        "Wrecked Ship Map Station": 0xCCCB,
        "Phantoon's Room": 0xCD13,
        "Sponge Bath": 0xCD5C,
        "Wrecked Ship West Super Room": 0xCDA8,
        "Wrecked Ship East Super Room": 0xCDF1,
        "Gravity Suit Room": 0xCE40,
        "Shaft Save Station": 0xCE8A,
        "Maridia Glass Tube Save Station": 0xCED2,
        "Maridia Glass Tube": 0xCEFB,
        "Red Brinstar West Tunnel": 0xCF54,
        "Red Brinstar East Tunnel": 0xCF80,
        "Main Street": 0xCFC9,
        "Fish Tank": 0xD017,
        "Mama Turtle Room": 0xD055,
        "Crab Tunnel": 0xD08A,
        "Mt Everest": 0xD0B9,
        "Red Fish Room": 0xD104,
        "Watering Hole": 0xD13B,
        "Northwest Maridia Bug Room": 0xD16D,
        "Crab Shaft": 0xD1A3,
        "Pseudo Plasma Spark Room": 0xD1DD,
        "Crab Hole": 0xD21C,
        "West Sand Hall Tunnel": 0xD252,
        "Plasma Tutorial Room": 0xD27E,
        "Plasma Room": 0xD2AA,
        "Thread the Needle Room": 0xD2D9,
        "Maridia Elevator Room": 0xD30B,
        "Plasma Spark Room": 0xD340,
        "Plasma Climb": 0xD387,
        "Maridia Map Room": 0xD3B6,
        "Maridia Elevator Save Station": 0xD3DF,
        "Maridia Transit": 0xD408,
        "Bug Sand Hole": 0xD433,
        "Sandy Hall West": 0xD461,
        "Oasis": 0xD48E,
        "Sandy Hall East": 0xD4C2,
        "West Sandtrap": 0xD4EF,
        "East Sandtrap": 0xD51E,
        "West Aqueduct Quicksand Room": 0xD54D,
        "East Aqueduct Quicksand Room": 0xD57A,
        "Aqueduct": 0xD5A7,
        "Butterfly Room": 0xD5EC,
        "Botwoon Hallway": 0xD617,
        "Pants Room": 0xD646,
        "Pants Room West Half": 0xD69A,
        "Springball Room": 0xD6D0,
        "Below Botwoon Energy Tank": 0xD6FD,
        "Colosseum": 0xD72A,
        "Aqueduct Save Station": 0xD765,
        "Precious Room": 0xD78F,
        "Botwoon Energy Tank Room": 0xD7E4,
        "Colosseum Save Station": 0xD81A,
        "Halfie Climb Missile Refill": 0xD845,
        "Plasma Beach Quicksand Room": 0xD86E,
        "Botwoon Quicksand Room": 0xD898,
        "Shaktool Room": 0xD8C5,
        "Halfie Climb Room": 0xD913,
        "Botwoon's Room": 0xD95E,
        "Space Jump Room": 0xD9AA,
        "Colosseum Energy Refill": 0xD9D4,
        "West Cactus Alley Room": 0xD9FE,
        "East Cactus Alley Room": 0xDA2B,
        "Draygon's Room": 0xDA60,
        "Tourian Entrance": 0xDAAE,
        "Metroid Room 1": 0xDAE1,
        "Metroid Room 2": 0xDB31,
        "Metroid Room 3": 0xDB7D,
        "Metroid Room 4": 0xDBCD,
        "Blue Hopper Room": 0xDC19,
        "Dust Torizo Room": 0xDC65,
        "Big Boy Room": 0xDCB1,
        "Seaweed Room": 0xDCFF,
        "Tourian Ressuply Station": 0xDD2E,
        "Mother Brain's Room": 0xDD58,
        "Tourian Gadora Room": 0xDDC4,
        "Rinka Shaft": 0xDDF3,
        "Rinka Shaft Save Station": 0xDE23,
        "Tourian Escape Room 1": 0xDE4D,
        "Tourian Escape Room 2": 0xDE7A,
        "Tourian Escape Room 3": 0xDEA7,
        "Tourian Escape Room 4": 0xDEDE,
        "Upper Tourian Save Station": 0xDF1B,
        "Elevator Shaft": 0xDF45,
        "Hallway A": 0xDF8D,
        "Hallway B": 0xDFD7,
        "Hallway C": 0xE021,
        "Hallway D": 0xE06B,
        "Sample Storage": 0xE0B5,
        "Debug Room": 0xE82C,
    }

    addr_to_room_name_dict = {
        0x91F8: "Landing Site",
        0x92B3: "Gauntlet Access",
        0x92FD: "Parlor",
        0x93AA: "Crateria Power Bomb Room",
        0x93D5: "Parlor Save Station",
        0x93FE: "West Ocean",
        0x9461: "Bowling Alley Path",
        0x948C: "Crateria Kihunter Room",
        0x94CC: "Forgotten Highway Elevator",
        0x94FD: "East Ocean",
        0x9552: "Forgotten Highway Kago Room",
        0x957D: "Crab Maze",
        0x95A8: "Forgotten Highway Elbow",
        0x95D4: "Crateria Tube",
        0x95FF: "Moat",
        0x962A: "Elevator to Red Brinstar",
        0x965B: "Gauntlet",
        0x968F: "Bowling Alley Path",
        0x96BA: "Climb",
        0x975C: "Old Mother Brain",
        0x97B5: "Elevator to Blue Brinstar",
        0x9804: "Bomb Torizo Room",
        0x9879: "Flyway",
        0x98E2: "Crateria Map Access",
        0x990D: "Terminator Room",
        0x9938: "Elevator to Green Brinstar",
        0x9969: "Lower Mushrooms",
        0x9994: "Crateria Map Station",
        0x99BD: "Green Pirates Shaft",
        0x99F9: "Crateria Super Room",
        0x9A44: "Final Missile Bombway",
        0x9A90: "Final Missile Room",
        0x9AD9: "Green Brinstar Main Shaft & Etecoon Room",
        0x9B5B: "Spore Spawn Super Room",
        0x9B9D: "Green Brinstar Map Access",
        0x9BC8: "Early Supers Room",
        0x9C07: "Green Brinstar Reserve Tank Room",
        0x9C35: "Brinstar Map Station",
        0x9C5E: "Green Brinstar Fireflea Room",
        0x9C89: "West Brinstar Missile Station",
        0x9CB3: "Dachora Room",
        0x9D19: "Big Pink",
        0x9D9C: "Spore Spawn Kihunter Room",
        0x9DC7: "Spore Spawn Room",
        0x9E11: "Pink Brinstar Power Bomb Room",
        0x9E52: "Green Hill Zone",
        0x9E9F: "Retro Brinstar",
        0x9F11: "Construction Zone",
        0x9F64: "Blue Brinstar Energy Tank Room",
        0x9FBA: "Noob Bridge",
        0x9FE5: "Green Brinstar Beetom Room",
        0xA011: "Etecoon Energy Tank Room",
        0xA051: "Etecoon Super Missile Room",
        0xA07B: "West Brinstar Energy Station",
        0xA0A4: "Spore Spawn Farming Room",
        0xA0D2: "Waterway",
        0xA107: "First Missiles",
        0xA130: "Pink Brinstar Hopper Room",
        0xA15B: "Pink Brinstar Energy Tank Room",
        0xA184: "Big Pink Save Station",
        0xA1AD: "Blue Brinstar Boulder Room",
        0xA1D8: "Blue Brinstar Double Missile Room",
        0xA201: "Green Brinstar Upper Save Station",
        0xA22A: "Green Brinstar Lower Save Station",
        0xA253: "Red Tower",
        0xA293: "Red Brinstar Fireflea Room",
        0xA2CE: "X-Ray Scope Room",
        0xA2F7: "Red Brinstar Badway",
        0xA322: "Red Brinstar Zoro Room",
        0xA37C: "Red Brinstar Beta Power Bomb Room",
        0xA3AE: "Red Brinstar Alpha Power Bomb Room",
        0xA3DD: "Red Brinstar Bat Room",
        0xA408: "Red Brinstar Below Spazer",
        0xA447: "Spazer Room",
        0xA471: "Warehouse Zeela Room",
        0xA4B1: "Warehouse Beetom Room",
        0xA4DA: "Warehouse Kihunter Room",
        0xA521: "Mini Kraid Room",
        0xA56B: "Kraid Gadora Room",
        0xA59F: "Kraid's Lair",
        0xA5ED: "Gold Four Hallway",
        0xA618: "Red Tower Energy Charge Station",
        0xA641: "Warehouse Resupply Station",
        0xA66A: "Gold Four Room",
        0xA6A1: "Warehouse Entrance",
        0xA6E2: "Varia Suit Room",
        0xA70B: "Warehouse Save Station",
        0xA734: "Red Tower Save Station",
        0xA75D: "Ice Beam Acid Room",
        0xA788: "Cathedral",
        0xA7B3: "Cathedral Access",
        0xA7DE: "Business Center",
        0xA815: "Ice Beam Gate Room",
        0xA865: "Ice Beam Tutorial Room",
        0xA890: "Ice Beam Room",
        0xA8B9: "Ice Beam Snake Room",
        0xA8F8: "Crumble Shaft",
        0xA923: "Crocomire Speedway",
        0xA98D: "Crocomire's Room",
        0xA9E5: "Hi-Jump Room",
        0xAA0E: "Crocomire Escape",
        0xAA41: "Hi-Jump Energy Tank Room",
        0xAA82: "Post Crocomire Farming Room",
        0xAAB5: "Post Crocomire Save Station",
        0xAADE: "Post Crocomire Power Bomb Room",
        0xAB07: "Post Crocomire Shaft",
        0xAB3B: "Cosine Missile Room",
        0xAB64: "Grapple Tutorial Room 3",
        0xAB8F: "Post Crocomire Jump Room",
        0xABD2: "Grapple Tutorial Room 2",
        0xAC00: "Grapple Tutorial Room 1",
        0xAC2B: "Grapple Beam Room",
        0xAC5A: "Bubble Mountain Reserve",
        0xAC83: "Bubble Mountain Reserve Access",
        0xACB3: "Bubble Mountain",
        0xACF0: "Speed Booster Hall",
        0xAD1B: "Speed Booster Room",
        0xAD5E: "Single Chamber",
        0xADAD: "Double Chamber",
        0xADDE: "Wave Beam Room",
        0xAE07: "Spiky Platforms Tunnel",
        0xAE32: "Volcano",
        0xAE74: "Kronic Boost Room",
        0xAEB4: "Magdollite Tunnel",
        0xAEDF: "Purple Shaft",
        0xAF14: "Lava Dive",
        0xAF3F: "Elevator to Lower Norfair",
        0xAF72: "Upper Norfair Farming Room",
        0xAFA3: "Rising Tide",
        0xAFCE: "Acid Snakes Tunnel",
        0xAFFB: "Spiky Acid Snakes Tunnel",
        0xB026: "Peanut Butter Refill",
        0xB051: "Purple Farming Room",
        0xB07A: "Bat Cave",
        0xB0B4: "Norfair Map Station",
        0xB0DD: "Bubble Mountain Save Station",
        0xB106: "Norfair Speedway",
        0xB139: "Red Pirate Shaft",
        0xB167: "Business Center Save Station",
        0xB192: "Crocomire Save Station",
        0xB1BB: "Elevator to Lower Norfair Save Station",
        0xB1E5: "Acid Statue Room",
        0xB236: "Main Hall",
        0xB283: "Golden Torizo's Room",
        0xB2DA: "Fast Ripper Room",
        0xB305: "Screw Attack Energy Refill",
        0xB32E: "Ridley's Room",
        0xB37A: "Ridley Farming Room",
        0xB3A5: "Fast Pillars Setup Room",
        0xB3E1: "Unused Room",
        0xB40A: "Famous Trademarked Mouse Room",
        0xB457: "Pillar Room",
        0xB482: "Plowerhouse Room",
        0xB4AD: "The Worst Room in the Game",
        0xB4E5: "Amphitheatre",
        0xB510: "Lower Norfair Springball Maze Room",
        0xB55A: "Lower Norfair Escape Power Bomb Room",
        0xB585: "Red Kihunter Shaft",
        0xB5D5: "Wasteland",
        0xB62B: "Metal Pirates Room",
        0xB656: "Three Musketeers' Room",
        0xB698: "Ridley Energy Tank Room",
        0xB6C1: "Screw Attack Room",
        0xB6EE: "Lower Norfair Fireflea Room",
        0xB741: "Red Kihunter Shaft Save Station",
        0xC98E: "Bowling Alley",
        0xCA08: "Wrecked Ship Entrance",
        0xCA52: "Attic",
        0xCAAE: "Assembly Line",
        0xCAF6: "Wrecked Ship Main Shaft",
        0xCB8B: "Spiky Death Room",
        0xCBD5: "Electric Death Room",
        0xCC27: "Wrecked Ship Energy Tank Room",
        0xCC6F: "Basement",
        0xCCCB: "Wrecked Ship Map Station",
        0xCD13: "Phantoon's Room",
        0xCD5C: "Sponge Bath",
        0xCDA8: "Wrecked Ship West Super Room",
        0xCDF1: "Wrecked Ship East Super Room",
        0xCE40: "Gravity Suit Room",
        0xCE8A: "Shaft Save Station",
        0xCED2: "Maridia Glass Tube Save Station",
        0xCEFB: "Maridia Glass Tube",
        0xCF54: "Red Brinstar West Tunnel",
        0xCF80: "Red Brinstar East Tunnel",
        0xCFC9: "Main Street",
        0xD017: "Fish Tank",
        0xD055: "Mama Turtle Room",
        0xD08A: "Crab Tunnel",
        0xD0B9: "Mt Everest",
        0xD104: "Red Fish Room",
        0xD13B: "Watering Hole",
        0xD16D: "Northwest Maridia Bug Room",
        0xD1A3: "Crab Shaft",
        0xD1DD: "Pseudo Plasma Spark Room",
        0xD21C: "Crab Hole",
        0xD252: "West Sand Hall Tunnel",
        0xD27E: "Plasma Tutorial Room",
        0xD2AA: "Plasma Room",
        0xD2D9: "Thread the Needle Room",
        0xD30B: "Maridia Elevator Room",
        0xD340: "Plasma Spark Room",
        0xD387: "Plasma Climb",
        0xD3B6: "Maridia Map Room",
        0xD3DF: "Maridia Elevator Save Station",
        0xD408: "Maridia Transit",
        0xD433: "Bug Sand Hole",
        0xD461: "Sandy Hall West",
        0xD48E: "Oasis",
        0xD4C2: "Sandy Hall East",
        0xD4EF: "West Sandtrap",
        0xD51E: "East Sandtrap",
        0xD54D: "West Aqueduct Quicksand Room",
        0xD57A: "East Aqueduct Quicksand Room",
        0xD5A7: "Aqueduct",
        0xD5EC: "Butterfly Room",
        0xD617: "Botwoon Hallway",
        0xD646: "Pants Room",
        0xD69A: "Pants Room West Half",
        0xD6D0: "Springball Room",
        0xD6FD: "Below Botwoon Energy Tank",
        0xD72A: "Colosseum",
        0xD765: "Aqueduct Save Station",
        0xD78F: "Precious Room",
        0xD7E4: "Botwoon Energy Tank Room",
        0xD81A: "Colosseum Save Station",
        0xD845: "Halfie Climb Missile Refill",
        0xD86E: "Plasma Beach Quicksand Room",
        0xD898: "Botwoon Quicksand Room",
        0xD8C5: "Shaktool Room",
        0xD913: "Halfie Climb Room",
        0xD95E: "Botwoon's Room",
        0xD9AA: "Space Jump Room",
        0xD9D4: "Colosseum Energy Refill",
        0xD9FE: "West Cactus Alley Room",
        0xDA2B: "East Cactus Alley Room",
        0xDA60: "Draygon's Room",
        0xDAAE: "Tourian Entrance",
        0xDAE1: "Metroid Room 1",
        0xDB31: "Metroid Room 2",
        0xDB7D: "Metroid Room 3",
        0xDBCD: "Metroid Room 4",
        0xDC19: "Blue Hopper Room",
        0xDC65: "Dust Torizo Room",
        0xDCB1: "Big Boy Room",
        0xDCFF: "Seaweed Room",
        0xDD2E: "Tourian Ressuply Station",
        0xDD58: "Mother Brain's Room",
        0xDDC4: "Tourian Gadora Room",
        0xDDF3: "Rinka Shaft",
        0xDE23: "Rinka Shaft Save Station",
        0xDE4D: "Tourian Escape Room 1",
        0xDE7A: "Tourian Escape Room 2",
        0xDEA7: "Tourian Escape Room 3",
        0xDEDE: "Tourian Escape Room 4",
        0xDF1B: "Upper Tourian Save Station",
        0xDF45: "Elevator Shaft",
        0xDF8D: "Hallway A",
        0xDFD7: "Hallway B",
        0xE021: "Hallway C",
        0xE06B: "Hallway D",
        0xE0B5: "Sample Storage",
        0xE82C: "Debug Room",
    }

    addr_to_door_data_dict = {
        0x91F8: [
            "1689FD9200054E06040000800000",
            "2289D49500000000000000800000",
            "2E89B39200054E06040000800000",
            "3A89AA9300040106000000800000",
        ],
        0x92B3: ["4689F891000401260002008097B9", "52895B9600055E06050000800000"],
        0x92FD: [
            "5E890D9900055E06050000800000",
            "6A89F89100040146000400800000",
            "7689E29800040106000000800000",
            "8289799800040106000000800000",
            "8E89BA9600061602010000800000",
            "9A89D59300050E06000000800000",
            "A689449A00051E06010000800000",
        ],
        0x93AA: ["B289F89100058E16080100800000"],
        0x93D5: ["BE89FD92000411260102008081B9"],
        0x93FE: [
            "CA89FF9500051E06010000800000",
            "D68908CA500401060000008071B9",
            "E289619400040106000000800000",
            "EE8952CA40040106000000800000",
            "FA898EC94004210602000080B3B9",
            "068A40CE40040106000000800000",
        ],
        0x9461: ["128AFE9300052E26020200800000", "1E8A8F9600040106000000800000"],
        0x948C: ["2A8AD49500010000000000800000", "368AFF9500040106000000800000", "428A2A9600060602000000800000"],
        0x94CC: ["4E8AA8950007060D0000C0010000", "5A8A0BD3D0020000000000000000", "FC880000F8910003000004000080"],
        0x94FD: ["668AD5CB40050E16000100800000", "728A529500040106000000800000"],
        0x9552: ["7E8AFD9400056E46060400800000", "8A8A7D9500063602030000800000"],
        0x957D: ["968A52950007063D0003C0010000", "AE8AA89500050E06000000800000"],
        0x95A8: ["A28A7D9500040116000100800000", "BA8ACC9400060602000000800000"],
        0x95D4: ["C68AF89100058E46080400800000", "D28A8C9400040106000000800000"],
        0x95FF: ["DE8A8C9400052E06020000800000", "EA8AFE9300040146000400800000"],
        0x962A: ["F68A8C940007162D0102C001F1B9", "028B22A3E00200000000000021BA", "FC880000F8910003000004000080"],
        0x965B: ["0E8BB39200040106000000800000", "1A8BBD9900050E06000000802CBA"],
        0x968F: ["268B619400051E06010000800000", "328B8EC9400401160001A000CAB9"],
        0x96BA: [
            "3E8BFD920007164D0104C00181B9",
            "4A8BF99900040106000000800000",
            "568BF99900040176000700800000",
            "628B5C9700040106000000800000",
            "6E8BDEDE00052E16020100800000",
        ],
        0x975C: ["7A8BBA9600051E86010800800000", "868BB59700040106000000800000"],
        0x97B5: ["928B5C9700052E06020000800000", "9E8B9F9EF0020000050000000000", "FC880000F8910003000004000080"],
        0x9804: ["AA8B799800052E06020000800000"],
        0x9879: ["B68BFD9200053E2603020080A2B9", "C28B049800040106000000800000"],
        0x98E2: ["CE8BFD9200051E360103008081B9", "DA8B949900040106000000800000"],
        0x990D: ["E68BBD9900050E46000400800000", "F28BFD9200040106000000808CB9"],
        0x9938: ["FE8B699900040106000000800000", "0A8CD99AC00200000000000025BD", "FC880000F8910003000004000080"],
        0x9969: ["168CBD9900040166000600800000", "228C389900050E06000000800000"],
        0x9994: ["2E8CE29800052E06020000800000"],
        0x99BD: [
            "3A8C0D9900040126000200800000",
            "468C699900053E06030000800000",
            "528CEDA5000401060000008000FE",
            "5E8C5B96000401060000008016BA",
        ],
        0x99F9: ["6A8CBA9600052E060200008000BA", "768CBA9600052E76020700800BBA"],
        0x9A44: ["828C909A00050E06000000800000", "8E8CFD92000411360103008081B9"],
        0x9A90: ["9A8C449A00040106000000800000"],
        0x9AD9: [
            "A68C3899D00300000000000036BE",
            "B28C9D9B00052E06020000800000",
            "BE8CE59F00050E06000000800000",
            "CA8C5E9C00052E06020000800000",
            "D68CC89B00040116000100800000",
            "E28CB39C00040106000000806CBD",
            "EE8CD99A000401760107008025BD",
            "FA8C11A000054E160401008016BD",
            "068DD99A00050E760007008007BD",
            "FC880000F8910003000004000080",
            "128D01A200050E06000000800000",
        ],
        0x9B5B: ["1E8DA4A000052E06020000800000", "2A8DC79D00050E06000000800000"],
        0x9B9D: ["368D359C00050E06000000800000", "428DD99A00040146000400800000"],
        0x9BC8: ["4E8DD99A00050E46000400800000", "5A8D079C00040106000000800000"],
        0x9C07: ["668DC89B00052E16020100800000"],
        0x9C35: ["728D9D9B00040106000000800000"],
        0x9C5E: ["7E8D899C00050E06000000800000", "8A8DD99A00040166000600800000"],
        0x9C89: ["968D5E9C00040116000100800000"],
        0x9CB3: ["A28DD99A00050E66000600800000", "AE8D199D00042126020200800000", "BA8D7BA000050E06000000800000"],
        0x9D19: [
            "C68D9C9D00040106000000800000",
            "D28DB39C00056E06060000800000",
            "DE8D119E00051E160101008030BD",
            "EA8D529E00040106000000800000",
            "F68D84A100050E06000000800000",
            "028E119E00011E06010000800000",
            "0E8ED2A000056E06060000800000",
            "1A8E30A100040116000100800000",
            "268EA4A000040106000000800000",
        ],
        0x9D9C: ["328E199D00053E06030000800000", "3E8EC79D0007062E000200800000"],
        0x9DC7: ["4A8E5B9B0004010600000080C0BD", "568E9C9D00063603030000800000"],
        0x9E11: ["628E199D00000000020300800000", "6E8E199D00042146020400800000"],
        0x9E52: ["7A8E199D00053E66030600800000", "868E9F9E00040126000200800000", "928EBA9F00040106000000800000"],
        0x9E9F: [
            "9E8E529E00051E06010000800000",
            "AA8E119F00040106000000800000",
            "B68EB597E0030000000000000000",
            "FC880000F8910003000004000080",
        ],
        0x9F11: ["C28E9F9E00057E26070200800000", "CE8E649F00040126000200800000", "DA8E07A100050E06000000800000"],
        0x9F64: ["E68E119F00050E06000000800000", "F28EADA100051E06010000800000"],
        0x9FBA: ["FE8E529E00057E36070300800000", "0A8F53A200040146000400800000"],
        0x9FE5: ["168FD99A000401A6000A008007BD", "228F11A000051E06010000800000"],
        0xA011: [
            "2E8FE59F00040106000000800000",
            "3A8F51A000050E06000000800000",
            "468FD99A000421B6020B008025BD",
            "528F2AA200050E06000000800000",
        ],
        0xA051: ["5E8F11A000040106000000800000"],
        0xA07B: ["6A8FB39C000401660006008050BD"],
        0xA0A4: ["768F5B9B00040186000800800000", "828F199D00054E56040500805BBD"],
        0xA0D2: ["8E8F199D000401960009008077BD"],
        0xA107: ["A68F119F000401160001008025BE"],
        0xA130: ["B28F199D00053E46030400800000", "BE8F5BA100040106000000800000"],
        0xA15B: ["CA8F30A100051E16010100800000"],
        0xA184: ["D68F199D000411060100008000BE"],
        0xA1AD: ["E28F649F00042106020000808ABD", "EE8FD8A100050E06000000800000"],
        0xA1D8: ["FA8FADA100040106000000800000"],
        0xA201: ["0690D99A00040156000500800000"],
        0xA22A: ["129011A0000401160001008016BD"],
        0xA253: [
            "1E90F7A200040106000000800000",
            "2A90BA9F00055E06050000800000",
            "369093A200057E06070000800000",
            "4290DDA300040106000000800000",
            "4E9018A600050E06000000800000",
        ],
        0xA293: ["5A90CEA200051E06010000800000", "669053A200040166000600800000"],
        0xA2CE: ["729093A200040106000000800000"],
        0xA2F7: ["7E9053A200050E06000000800000", "8A9022A300040156000500801ABE"],
        0xA322: [
            "9690AEA300052E06020000800000",
            "A2907CA300051E06010000800000",
            "AE90F7A200052E06020000800000",
            "BA902A96F0030000000000000000",
            "C69004D14004010600000080AFBD",
            "FC880000F8910003000004000080",
            "D29034A700040106000000800000",
        ],
        0xA37C: ["DE9022A300040136000300800000"],
        0xA3AE: ["EA9022A300040176000700800BBE"],
        0xA3DD: ["F69053A200050E9600090080A0BD", "029108A400040116000100800000"],
        0xA408: ["0E91DDA300051E06010000800000", "1A9154CF40040106000000800000", "269147A400040106000000800000"],
        0xA447: ["329108A400051E0601000080F1BD"],
        0xA471: ["3E91A1A600052E06020000803FBD", "4A91B1A400050E06000000800000", "5691DAA40007060C000040020000"],
        0xA4B1: ["629171A400040116000100800000"],
        0xA4DA: ["6E9171A4000616130101C0010000", "7A9121A500040106000000800000", "86910BA700040106000000800000"],
        0xA521: ["9291DAA400051E16010100800000", "9E916BA500040116000100800000"],
        0xA56B: ["AA9121A500055E06050000800000", "B6919FA500040116000100800000", "C29141A600040106000000800000"],
        0xA59F: ["CE916BA500051E16010100800000", "DA91E2A600040106000000800000"],
        0xA5ED: ["E691BD9900050E66000600800000", "F2916AA600040106000000800000"],
        0xA618: ["FE9153A20004019600090080A0BD"],
        0xA641: ["0A926BA500050E060000008095BD"],
        0xA66A: ["1692EDA500054E06040000800000", "2292AEDAD0020000000000000000", "FC880000F8910003000004000080"],
        0xA6A1: [
            "2E9280CF40050E1600010080D1BD",
            "3A9271A400040106000000800000",
            "4692DEA7C0020000000000000000",
            "FC880000F8910003000004000080",
        ],
        0xA6E2: ["52929FA500051E16010100800000"],
        0xA70B: ["5E92DAA400053E0603000080E2BD"],
        0xA734: ["6A9222A300050E46000400800000"],
        0xA75D: ["769215A800043126030200800000", "8292B9A800050E26000200800000"],
        0xA788: ["8E92B3A700052E06020000800000", "9A92A3AF00040106000000800000"],
        0xA7B3: ["A692DEA700050E36000300800000", "B29288A700040106000000800000"],
        0xA7DE: [
            "BE9215A800056E26060200800000",
            "CA92B3A700040106000000800000",
            "D69241AA00051E06010000800000",
            "E2920EAA00040106000000800000",
            "EE92A1A6D0030000000000000000",
            "FC880000F8910003000004000080",
            "FA9267B100040106000000800000",
            "0693B4B000050E06000000800000",
        ],
        0xA815: [
            "129365A800051E06010000800000",
            "1E935DA700051E06010000800000",
            "2A93DEA700040136000300800000",
            "3693F8A800050E06000000800000",
        ],
        0xA865: ["4293B9A800050E06000000800000", "4E9315A80004310603000080FABF"],
        0xA890: ["5A93B9A800051E16010100803AC0"],
        0xA8B9: ["66935DA700040106000000800000", "729365A800040106000000800000", "7E9390A800040106000000800000"],
        0xA8F8: ["8A9315A80004013600030080EFC0", "969323A900040106000000800000"],
        0xA923: [
            "A293F8A800050E36000300800000",
            "AE930EAA00053E16030100800000",
            "BA9392B100040106000000800000",
            "C693CEAF00040106000000800000",
            "D2938DA900063602030000800000",
        ],
        0xA98D: ["DE9382AA00051E06010000800000", "EA9323A90007C62D0C02C0010000"],
        0xA9E5: ["F69341AA000401160001008016C0"],
        0xAA0E: ["0294DEA700050E66000600800000", "0E9423A90004C1060C0000800000"],
        0xAA41: ["1A94DEA700040156000500800000", "2694E5A900050E06000000800000"],
        0xAA82: [
            "32948DA90004010600000080DABF",
            "3E94DEAA00050E06000000800000",
            "4A9407AB00060602000000800000",
            "5694B5AA00040106000000800000",
        ],
        0xAAB5: ["629482AA00051E16010100800000"],
        0xAADE: ["6E9482AA00040106000000800000"],
        0xAB07: [
            "7A9482AA0007061D0001C0010000",
            "869464AB00052E06020000800000",
            "92943BAB00040106000000800000",
            "9E948FAB00066622060240010000",
        ],
        0xAB3B: ["AA9407AB00050E36000300800000"],
        0xAB64: ["B694D2AB00050E06000000800000", "C29407AB00040106000000800000"],
        0xAB8F: ["CE9407AB0007064D0004C0010000", "DA942BAC00050E26000200800000"],
        0xABD2: ["E69400AC00051E06010000800000", "F29464AB00040106000000800000"],
        0xAC00: ["FE942BAC00050E06000000800000", "0A95D2AB00040126000200800000"],
        0xAC2B: ["16958FAB00040116000100800000", "229500AC00040106000000800000"],
        0xAC5A: ["2E9583AC0004010600000080D3C0"],
        0xAC83: ["3A95B3AC00040106000000800000", "46955AAC00051E06010000800000"],
        0xACB3: [
            "529583AC00051E06010000800000",
            "5E95A3AF00054E06040000800000",
            "6A9572AF00051E06010000800000",
            "7695DFAE0006060200000080DEC0",
            "82955EAD00040106000000800000",
            "8E957AB000040116000100800000",
            "9A95DDB000050E06000000800000",
        ],
        0xACF0: ["A6957AB000050E0600000080E9BF", "B2951BAD00040106000000800000"],
        0xAD1B: ["BE95F0AC0005BE160B0100800000"],
        0xAD5E: [
            "CA95B3AC00051E16010100800000",
            "D695ADAD00040106000000800000",
            "E295ADAD00040116000100800000",
            "EE9507AE00040106000000800000",
            "FA9556B600041106010000800000",
        ],
        0xADAD: ["06965EAD00050E16000100800000", "12965EAD00050E26000200800000", "1E96DEAD00040106000000800000"],
        0xADDE: ["2A96ADAD00053E06030000800000"],
        0xAE07: ["36965EAD00050E36000300800000", "429632AE00042106020000800000"],
        0xAE32: ["4E9607AE00053E06030000800000", "5A9674AE00051E06010000800000"],
        0xAE74: [
            "6696B4AE00052E06020000800000",
            "729632AE000401260002008025C0",
            "7E9614AF00053E06030000800000",
            "8A96FBAF00053E06030000800000",
        ],
        0xAEB4: ["9696DFAE00050E26000200800000", "A29674AE00041106010000800000"],
        0xAEDF: ["AE96B3AC0007063D0003C0010000", "BA96B4AE00040106000000800000", "C69651B000040106000000800000"],
        0xAF14: ["D29674AE00041126010200800000", "DE963FAF00050E06000000800000"],
        0xAF3F: [
            "EA9614AF00040106000000800000",
            "F69636B2A0020000040000000000",
            "FC880000F8910003000004000080",
            "0297BBB100050E06000000800000",
        ],
        0xAF72: ["0E9706B100057E06070000800000", "1A97B3AC00040136000300800000", "269739B100050E06000000800000"],
        0xAFA3: ["329788A700052E16020100800000", "3E97B3AC00040126000200800000"],
        0xAFCE: ["4A9723A90005CE260C0200800000", "569726B000040106000000800000", "629739B10007062D000200020000"],
        0xAFFB: ["6E9726B000050E06000000800000", "7A9774AE0004011600010080FAC0"],
        0xB026: ["8697CEAF00053E06030000800000", "9297FBAF00040106000000800000"],
        0xB051: ["9E97DFAE00050E16000100800000"],
        0xB07A: ["AA97B3AC00051E06010000800000", "B697F0AC00040106000000800000"],
        0xB0B4: ["C297DEA700040146000400800000"],
        0xB0DD: ["CE97B3AC00040116000100800000"],
        0xB106: ["DA9767B100050E06000000800000", "E69772AF00040106000000800000"],
        0xB139: ["F29772AF00040116000100800000", "FE97CEAF00063602030040010000"],
        0xB167: ["0A9806B100040106000000800000", "1698DEA700050E56000500800000"],
        0xB192: ["229823A90005AE160C0100800000"],
        0xB1BB: ["2E983FAF00040106000000800000"],
        0xB1E5: ["3A9883B2000401060000008089C0", "469836B200040126000200800000"],
        0xB236: [
            "5298E5B100051E06010000800000",
            "5E98A5B300040116000100800000",
            "6A983FAFB0030000000000000000",
            "FC880000F8910003000004000080",
        ],
        0xB283: ["7698E5B100052E26020200809EBF", "8298C1B600040126000200800000"],
        0xB2DA: ["8E98C1B600050E06000000800000", "9A98A5B3000401260002008056C0"],
        0xB305: ["A698C1B600050E16000100800000"],
        0xB32E: ["B29898B600050E06000000800000", "BE987AB300040106000000800000"],
        0xB37A: ["CA982EB300050E06000000800000", "D69882B400040106000000800000"],
        0xB3A5: [
            "E29836B200057E26070200800000",
            "EE980AB400040136000300800000",
            "FA9836B200057E26070200800000",
            "0699DAB200053E06030000800000",
            "129957B400040106000000800000",
        ],
        0xB3E1: ["1E99A5B300050E06000000800000"],
        0xB40A: ["2A99A5B300050E06000000800000", "3699ADB40004010600000080A2C0"],
        0xB457: ["4299A5B300050E260002008056C0", "4E99ADB400040156000500800000"],
        0xB482: ["5A997AB300052E06020000800000", "66992BB600040106000000800000"],
        0xB4AD: ["72990AB400053E06030000800BC1", "7E99E5B400040116000100800000", "8A9957B400053E06030000800000"],
        0xB4E5: ["9699ADB400050E16000100800000", "A29985B5000401060000008067C0"],
        0xB510: ["AE9956B600053E26030200800000", "BA99EEB60004010600000080ADC0", "C6995AB500060602000000800000"],
        0xB55A: ["D299EEB600051E060100008078C0", "DE9910B50007460D0400C001BBBF"],
        0xB585: [
            "EA99D5B500065602050000800000",
            "F699E5B400053E06030000800000",
            "029AEEB600041136010300800000",
            "0E9A41B700040106000000800000",
        ],
        0xB5D5: ["1A9A2BB600052E06020000800000", "269A85B50007264D020400020000"],
        0xB62B: ["329A82B400052E06020000800000", "3E9AD5B500041126010200804BC0"],
        0xB656: ["4A9A5EAD00055E06050000800000", "569A10B500040106000000800000"],
        0xB698: ["629A2EB300040116000100800000"],
        0xB6C1: ["6E9ADAB200040106000000800000", "7A9A05B300040106000000800000", "869A83B200051E16010100800000"],
        0xB6EE: ["929A10B500051E16010100800000", "9E9A5AB500040106000000800000", "AA9A85B500050E0600000080C2C0"],
        0xB741: ["B69A85B500050E36000300800000"],
        0xC98E: ["8CA1FE9340057E16070100800000", "98A18F9640050E06000000800000", "A4A140CE00050E06000000800000"],
        0xCA08: ["B0A1FE9340057E46070400800000", "BCA1F6CA00044136040300800000"],
        0xCA52: ["C8A1F6CA00064602040000800000", "D4A1AECA00040106000000800000", "E0A1FE9340057E06070000800000"],
        0xCAAE: ["ECA152CA00056E06060000800000"],
        0xCAF6: [
            "F8A108CA00053E0603000080D8E1",
            "04A25CCD00040106000000800000",
            "10A2A8CD00050E06000000800000",
            "1CA26FCC00062602020000800000",
            "28A252CA0007460E0400C0010000",
            "34A2F1CD00040106000000800000",
            "40A28ACE00040106000000800000",
        ],
        0xCB8B: ["4CA25CCD00051E0601000080E8E1", "58A2D5CB00040126000200800000"],
        0xCBD5: ["64A2FD9440040146000400800000", "70A28BCB00051E06010000800000", "7CA227CC00052E06020000800000"],
        0xCC27: ["88A2D5CB0004010600000080F3E1"],
        0xCC6F: ["94A2F6CA0007467D0407C0011AE2", "A0A2CBCC00050E06000000800000", "ACA213CD00040106000000800000"],
        0xCCCB: ["B8A26FCC00040106000000800000"],
        0xCD13: ["C4A26FCC00054E0604000080FEE1"],
        0xCD5C: ["D0A2F6CA00054E46040400800000", "DCA28BCB00040106000000800000"],
        0xCDA8: ["E8A2F6CA00044166040600800000"],
        0xCDF1: ["F4A2F6CA00055E66050600800FE2"],
        0xCE40: ["00A3FE9340055E36050300800000", "0CA38EC9000411260102008029E2"],
        0xCE8A: ["18A3F6CA00054E36040300800000"],
        0xCED2: ["24A3FBCE00050E26000200800000"],
        0xCEFB: [
            "30A3C9CF0007167D010700020000",
            "3CA354CF00050E06000000800000",
            "48A380CF000401160001008045E3",
            "54A3D2CE00040106000000800000",
        ],
        0xCF54: ["60A3FBCE000000000001008078E3", "6CA308A440051E16010100800000"],
        0xCF80: ["78A3FBCE000100000001008078E3", "84A3A1A640040106000000800000", "90A31CD2000401160001008056E3"],
        0xCFC9: [
            "9CA3FBCE00060602000070010000",
            "A8A38AD000040106000000800000",
            "B4A317D000040126000200800000",
            "C0A3B9D000040106000000800000",
            "CCA3B9D000000000010200800000",
        ],
        0xD017: [
            "D8A3C9CF00052E66020600800000",
            "E4A355D000040136000300800000",
            "F0A3B9D00007163D0103C0010000",
            "FCA3B9D00007463D0403C0010000",
        ],
        0xD055: ["08A417D000053E26030200800000"],
        0xD08A: ["14A4C9CF00051E76010700800000", "20A41CD200000000000000800000"],
        0xD0B9: [
            "2CA404D10007261D0201C0010000",
            "38A4C9CF00051E06010000800000",
            "44A417D000060602000000800000",
            "50A417D000063602030000800000",
            "5CA4C9CF00010000020200800000",
            "68A4A3D100040126000200800000",
        ],
        0xD104: ["74A4B9D000062602020000800000", "80A422A340052E360203008067E3"],
        0xD13B: ["8CA46DD100040106000040010000"],
        0xD16D: ["98A43BD100051E06010000800000", "A4A4DDD100040116000100800000"],
        0xD1A3: ["B0A4B9D000055E06050000800000", "BCA4DDD10007062D0002C0010000", "C8A4A7D500040116000100800000"],
        0xD1DD: ["D4A46DD100053E16030100800000", "E0A4A3D100060602000000800000", "ECA440D300000000000200800000"],
        0xD21C: [
            "F8A48AD000053E06030000800000",
            "04A552D200040106000000800000",
            "10A580CF00053E06030000800000",
            "1CA5B6D300040106000000800000",
        ],
        0xD252: ["28A51CD200010000000000800000", "34A561D400040106000000800000"],
        0xD27E: ["40A587D300050E06000000800000", "4CA5AAD200040106000000800000"],
        0xD2AA: ["58A57ED200050E06000000800000"],
        0xD2D9: ["64A533D400050E06000000800000", "70A50BD300040156000500800000"],
        0xD30B: [
            "7CA5D9D200056E06060000800000",
            "88A5DFD300040106000000800000",
            "94A5CC94D0030000000000000000",
            "8AA10000FE9340057E1607010080",
        ],
        0xD340: [
            "A0A5ECD500040106000000800000",
            "ACA508D4000200000000008091E2",
            "B8A587D300040136000300800000",
            "C4A533D400040106000000800000",
        ],
        0xD387: ["D0A540D300052E16020100800000", "DCA57ED200040106000000800000"],
        0xD3B6: ["E8A51CD200050E160001008056E3"],
        0xD3DF: ["F4A50BD300050E46000400800000"],
        0xD408: ["00A68ED4000606020000008009E3", "0CA640D30007062D0002000201E3"],
        0xD433: ["18A6D9D200040106000000800000", "24A66ED800020000000000800000", "30A640D300053E36030300800000"],
        0xD461: ["3CA652D200050E06000000800000", "48A68ED400040116000100800000", "54A6EFD400030000000100800000"],
        0xD48E: ["60A661D400053E06030000800000", "6CA6C2D400040106000000800000", "78A608D400030000000900806CE2"],
        0xD4C2: ["84A68ED400050E16000100800000", "90A646D60004013600030080A3E3", "9CA61ED500030000010100800000"],
        0xD4EF: ["A8A64DD500030000000100800000", "B4A661D400020000020000800000"],
        0xD51E: ["C0A67AD500030000000100800000", "CCA6C2D400020000010000800000"],
        0xD54D: ["D8A6A7D500030000010200800000", "E4A6EFD400020000010000800000"],
        0xD57A: ["F0A6A7D500030000030200800000", "FCA61ED500020000000000800000"],
        0xD5A7: [
            "08A7A3D100051E360103008098E3",
            "14A74DD500020000000000800000",
            "20A77AD500020000000000800000",
            "2CA717D60007060D0000C0010000",
            "38A7FDD600040106000000800000",
            "44A765D700050E06000000800000",
        ],
        0xD5EC: ["50A740D300053E56030500800000", "5CA7FED900040116000100800000"],
        0xD617: ["68A7A7D500060602000000800000", "74A75ED900040106000000800000"],
        0xD646: [
            "80A7C2D400052E06020000800000",
            "8CA79AD600040126000200800000",
            "98A7C5D800040106000000800000",
            "A4A746D600050E3600030080B9E3",
        ],
        0xD69A: ["B0A746D600050E3600030080A3E3", "BCA7C5D800040106000000800000"],
        0xD6D0: ["C8A7C5D800053E06030000800000"],
        0xD6FD: ["D4A7A7D500055E16050100800000"],
        0xD72A: ["E0A713D900010000000000800000", "ECA71AD800040106000000800000", "F8A78FD700040106000000800000"],
        0xD765: ["28A8A7D500040126000200800000"],
        0xD78F: ["34A82AD700056E16060100800000", "40A860DA00051E06010000800000"],
        0xD7E4: [
            "4CA85ED900051E06010000808DE3",
            "58A898D800020000010000800000",
            "64A898D800020000000000800000",
            "70A813D900040126000200800000",
        ],
        0xD81A: ["7CA8D4D900040106000000800000", "88A82AD700056E06060000800000"],
        0xD845: ["94A813D900054E260402008018E3"],
        0xD86E: ["A0A8ECD500020000000000800000"],
        0xD898: ["ACA8FDD600020000020000800000", "B8A8FDD600020000030000800000"],
        0xD8C5: ["C4A89AD600050E1600010080C8E3", "D0A8D0D600040106000000800000"],
        0xD913: [
            "DCA8E4D700056E06060000800000",
            "E8A82AD700040106000000800000",
            "F4A845D800040106000000800000",
            "00A92BDA00054E16040100800000",
        ],
        0xD95E: ["0CA917D600053E06030000800000", "18A9E4D700040106000000800000"],
        0xD9AA: ["24A960DA00040116000100800000"],
        0xD9D4: ["30A91AD800050E06000000800000"],
        0xD9FE: ["3CA9ECD500050E06000000800000", "48A92BDA00040106000000800000"],
        0xDA2B: ["54A9FED900050E06000000800000", "60A913D900040116000100800000"],
        0xDA60: ["6CA98FD70004012600020080D9E3", "78A9AAD900050E06000000800000"],
        0xDAAE: [
            "84A9E1DA00055E06050000800000",
            "90A96AA6D003000000010000C0E4",
            "8AA10000FE9340057E1607010080",
            "9CA91BDF00040106000000800000",
        ],
        0xDAE1: ["A8A9AEDA00040136000300800000", "B4A931DB00050E06000000800000"],
        0xDB31: ["C0A9E1DA00040106000000800000", "CCA97DDB00040106000000800000"],
        0xDB7D: ["D8A931DB00050E16000100800000", "E4A9CDDB00040106000000800000"],
        0xDBCD: ["F0A97DDB00055E06050000800000", "FCA919DC00061603010000800000"],
        0xDC19: ["08AACDDB0007061E0001C0010000", "14AA65DC00051E06010000800000"],
        0xDC65: ["20AA19DC00040106000000800000", "2CAAB1DC00053E06030000800000"],
        0xDCB1: ["38AA65DC00040106000000800000", "44AAFFDC00010000000000800000"],
        0xDCFF: ["50AAB1DC00000000000000800000", "5CAAC4DD00040106000000800000", "68AA2EDD00050E06000000800000"],
        0xDD2E: ["74AAFFDC00040116000100800000"],
        0xDD58: ["80AAF3DD00040126000200800000", "8CAA4DDE00091F06010000800000"],
        0xDDC4: ["98AAFFDC00050E16000100800000", "A4AAF3DD00040106000000800000"],
        0xDDF3: ["B0AAC4DD00053E06030000800000", "BCAA23DE00050E06000000800000", "C8AA58DD00053E06030000800000"],
        0xDE23: ["D4AAF3DD00040116000100800000"],
        0xDE4D: ["E0AA58DD00040106000000800000", "ECAA7ADE00060603000000800000"],
        0xDE7A: ["F8AA4DDE0007060C000000020000", "04ABA7DE00040116000100800000"],
        0xDEA7: ["10AB7ADE00050E16000100800000", "1CABDEDE00040136000300800000"],
        0xDEDE: ["28ABA7DE00055E06050000800000", "34ABBA964004018600080080CFE4"],
        0xDF1B: ["40ABAEDA00050E36000300800000"],
        0xDF45: ["4CAB8DDF000000000000008013E5"],
        0xDF8D: ["58AB45DF0001000000020080E0E4", "64ABD7DF00000000000000800000"],
        0xDFD7: ["70AB8DDF00010000010000800000", "7CAB21E000000000000000800000"],
        0xE021: ["88ABD7DF00010000000100800000", "94AB6BE000000000000000800000"],
        0xE06B: ["A0AB21E000010000010000800000", "ACABB5E000000000000000800000"],
        0xE0B5: ["B8AB6BE000010000010000800000"],
        0xE82C: [
            "C4AB2CE80500000100000100002C",
            "CFAB2CE80500000101000100002C",
            "DAAB2CE80400000000000100002C",
            "E5AB2CE804000000010001000000",
        ],
    }

    door_addr_to_room_dict = {
        0x8916: 0x91F8,
        0x8922: 0x91F8,
        0x892E: 0x91F8,
        0x893A: 0x91F8,
        0x8946: 0x92B3,
        0x8952: 0x92B3,
        0x895E: 0x92FD,
        0x896A: 0x92FD,
        0x8976: 0x92FD,
        0x8982: 0x92FD,
        0x898E: 0x92FD,
        0x899A: 0x92FD,
        0x89A6: 0x92FD,
        0x89B2: 0x93AA,
        0x89BE: 0x93D5,
        0x89CA: 0x93FE,
        0x89D6: 0x93FE,
        0x89E2: 0x93FE,
        0x89EE: 0x93FE,
        0x89FA: 0x93FE,
        0x8A06: 0x93FE,
        0x8A12: 0x9461,
        0x8A1E: 0x9461,
        0x8A2A: 0x948C,
        0x8A36: 0x948C,
        0x8A42: 0x948C,
        0x8A4E: 0x94CC,
        0x8A5A: 0x94CC,
        0x88FC: 0x94CC,
        0x8A66: 0x94FD,
        0x8A72: 0x94FD,
        0x8A7E: 0x9552,
        0x8A8A: 0x9552,
        0x8A96: 0x957D,
        0x8AAE: 0x957D,
        0x8AA2: 0x95A8,
        0x8ABA: 0x95A8,
        0x8AC6: 0x95D4,
        0x8AD2: 0x95D4,
        0x8ADE: 0x95FF,
        0x8AEA: 0x95FF,
        0x8AF6: 0x962A,
        0x8B02: 0x962A,
        0x88FC: 0x962A,
        0x8B0E: 0x965B,
        0x8B1A: 0x965B,
        0x8B26: 0x968F,
        0x8B32: 0x968F,
        0x8B3E: 0x96BA,
        0x8B4A: 0x96BA,
        0x8B56: 0x96BA,
        0x8B62: 0x96BA,
        0x8B6E: 0x96BA,
        0x8B7A: 0x975C,
        0x8B86: 0x975C,
        0x8B92: 0x97B5,
        0x8B9E: 0x97B5,
        0x88FC: 0x97B5,
        0x8BAA: 0x9804,
        0x8BB6: 0x9879,
        0x8BC2: 0x9879,
        0x8BCE: 0x98E2,
        0x8BDA: 0x98E2,
        0x8BE6: 0x990D,
        0x8BF2: 0x990D,
        0x8BFE: 0x9938,
        0x8C0A: 0x9938,
        0x88FC: 0x9938,
        0x8C16: 0x9969,
        0x8C22: 0x9969,
        0x8C2E: 0x9994,
        0x8C3A: 0x99BD,
        0x8C46: 0x99BD,
        0x8C52: 0x99BD,
        0x8C5E: 0x99BD,
        0x8C6A: 0x99F9,
        0x8C76: 0x99F9,
        0x8C82: 0x9A44,
        0x8C8E: 0x9A44,
        0x8C9A: 0x9A90,
        0x8CA6: 0x9AD9,
        0x8CB2: 0x9AD9,
        0x8CBE: 0x9AD9,
        0x8CCA: 0x9AD9,
        0x8CD6: 0x9AD9,
        0x8CE2: 0x9AD9,
        0x8CEE: 0x9AD9,
        0x8CFA: 0x9AD9,
        0x8D06: 0x9AD9,
        0x88FC: 0x9AD9,
        0x8D12: 0x9AD9,
        0x8D1E: 0x9B5B,
        0x8D2A: 0x9B5B,
        0x8D36: 0x9B9D,
        0x8D42: 0x9B9D,
        0x8D4E: 0x9BC8,
        0x8D5A: 0x9BC8,
        0x8D66: 0x9C07,
        0x8D72: 0x9C35,
        0x8D7E: 0x9C5E,
        0x8D8A: 0x9C5E,
        0x8D96: 0x9C89,
        0x8DA2: 0x9CB3,
        0x8DAE: 0x9CB3,
        0x8DBA: 0x9CB3,
        0x8DC6: 0x9D19,
        0x8DD2: 0x9D19,
        0x8DDE: 0x9D19,
        0x8DEA: 0x9D19,
        0x8DF6: 0x9D19,
        0x8E02: 0x9D19,
        0x8E0E: 0x9D19,
        0x8E1A: 0x9D19,
        0x8E26: 0x9D19,
        0x8E32: 0x9D9C,
        0x8E3E: 0x9D9C,
        0x8E4A: 0x9DC7,
        0x8E56: 0x9DC7,
        0x8E62: 0x9E11,
        0x8E6E: 0x9E11,
        0x8E7A: 0x9E52,
        0x8E86: 0x9E52,
        0x8E92: 0x9E52,
        0x8E9E: 0x9E9F,
        0x8EAA: 0x9E9F,
        0x8EB6: 0x9E9F,
        0x88FC: 0x9E9F,
        0x8EC2: 0x9F11,
        0x8ECE: 0x9F11,
        0x8EDA: 0x9F11,
        0x8EE6: 0x9F64,
        0x8EF2: 0x9F64,
        0x8EFE: 0x9FBA,
        0x8F0A: 0x9FBA,
        0x8F16: 0x9FE5,
        0x8F22: 0x9FE5,
        0x8F2E: 0xA011,
        0x8F3A: 0xA011,
        0x8F46: 0xA011,
        0x8F52: 0xA011,
        0x8F5E: 0xA051,
        0x8F6A: 0xA07B,
        0x8F76: 0xA0A4,
        0x8F82: 0xA0A4,
        0x8F8E: 0xA0D2,
        0x8FA6: 0xA107,
        0x8FB2: 0xA130,
        0x8FBE: 0xA130,
        0x8FCA: 0xA15B,
        0x8FD6: 0xA184,
        0x8FE2: 0xA1AD,
        0x8FEE: 0xA1AD,
        0x8FFA: 0xA1D8,
        0x9006: 0xA201,
        0x9012: 0xA22A,
        0x901E: 0xA253,
        0x902A: 0xA253,
        0x9036: 0xA253,
        0x9042: 0xA253,
        0x904E: 0xA253,
        0x905A: 0xA293,
        0x9066: 0xA293,
        0x9072: 0xA2CE,
        0x907E: 0xA2F7,
        0x908A: 0xA2F7,
        0x9096: 0xA322,
        0x90A2: 0xA322,
        0x90AE: 0xA322,
        0x90BA: 0xA322,
        0x90C6: 0xA322,
        0x88FC: 0xA322,
        0x90D2: 0xA322,
        0x90DE: 0xA37C,
        0x90EA: 0xA3AE,
        0x90F6: 0xA3DD,
        0x9102: 0xA3DD,
        0x910E: 0xA408,
        0x911A: 0xA408,
        0x9126: 0xA408,
        0x9132: 0xA447,
        0x913E: 0xA471,
        0x914A: 0xA471,
        0x9156: 0xA471,
        0x9162: 0xA4B1,
        0x916E: 0xA4DA,
        0x917A: 0xA4DA,
        0x9186: 0xA4DA,
        0x9192: 0xA521,
        0x919E: 0xA521,
        0x91AA: 0xA56B,
        0x91B6: 0xA56B,
        0x91C2: 0xA56B,
        0x91CE: 0xA59F,
        0x91DA: 0xA59F,
        0x91E6: 0xA5ED,
        0x91F2: 0xA5ED,
        0x91FE: 0xA618,
        0x920A: 0xA641,
        0x9216: 0xA66A,
        0x9222: 0xA66A,
        0x88FC: 0xA66A,
        0x922E: 0xA6A1,
        0x923A: 0xA6A1,
        0x9246: 0xA6A1,
        0x88FC: 0xA6A1,
        0x9252: 0xA6E2,
        0x925E: 0xA70B,
        0x926A: 0xA734,
        0x9276: 0xA75D,
        0x9282: 0xA75D,
        0x928E: 0xA788,
        0x929A: 0xA788,
        0x92A6: 0xA7B3,
        0x92B2: 0xA7B3,
        0x92BE: 0xA7DE,
        0x92CA: 0xA7DE,
        0x92D6: 0xA7DE,
        0x92E2: 0xA7DE,
        0x92EE: 0xA7DE,
        0x88FC: 0xA7DE,
        0x92FA: 0xA7DE,
        0x9306: 0xA7DE,
        0x9312: 0xA815,
        0x931E: 0xA815,
        0x932A: 0xA815,
        0x9336: 0xA815,
        0x9342: 0xA865,
        0x934E: 0xA865,
        0x935A: 0xA890,
        0x9366: 0xA8B9,
        0x9372: 0xA8B9,
        0x937E: 0xA8B9,
        0x938A: 0xA8F8,
        0x9396: 0xA8F8,
        0x93A2: 0xA923,
        0x93AE: 0xA923,
        0x93BA: 0xA923,
        0x93C6: 0xA923,
        0x93D2: 0xA923,
        0x93DE: 0xA98D,
        0x93EA: 0xA98D,
        0x93F6: 0xA9E5,
        0x9402: 0xAA0E,
        0x940E: 0xAA0E,
        0x941A: 0xAA41,
        0x9426: 0xAA41,
        0x9432: 0xAA82,
        0x943E: 0xAA82,
        0x944A: 0xAA82,
        0x9456: 0xAA82,
        0x9462: 0xAAB5,
        0x946E: 0xAADE,
        0x947A: 0xAB07,
        0x9486: 0xAB07,
        0x9492: 0xAB07,
        0x949E: 0xAB07,
        0x94AA: 0xAB3B,
        0x94B6: 0xAB64,
        0x94C2: 0xAB64,
        0x94CE: 0xAB8F,
        0x94DA: 0xAB8F,
        0x94E6: 0xABD2,
        0x94F2: 0xABD2,
        0x94FE: 0xAC00,
        0x950A: 0xAC00,
        0x9516: 0xAC2B,
        0x9522: 0xAC2B,
        0x952E: 0xAC5A,
        0x953A: 0xAC83,
        0x9546: 0xAC83,
        0x9552: 0xACB3,
        0x955E: 0xACB3,
        0x956A: 0xACB3,
        0x9576: 0xACB3,
        0x9582: 0xACB3,
        0x958E: 0xACB3,
        0x959A: 0xACB3,
        0x95A6: 0xACF0,
        0x95B2: 0xACF0,
        0x95BE: 0xAD1B,
        0x95CA: 0xAD5E,
        0x95D6: 0xAD5E,
        0x95E2: 0xAD5E,
        0x95EE: 0xAD5E,
        0x95FA: 0xAD5E,
        0x9606: 0xADAD,
        0x9612: 0xADAD,
        0x961E: 0xADAD,
        0x962A: 0xADDE,
        0x9636: 0xAE07,
        0x9642: 0xAE07,
        0x964E: 0xAE32,
        0x965A: 0xAE32,
        0x9666: 0xAE74,
        0x9672: 0xAE74,
        0x967E: 0xAE74,
        0x968A: 0xAE74,
        0x9696: 0xAEB4,
        0x96A2: 0xAEB4,
        0x96AE: 0xAEDF,
        0x96BA: 0xAEDF,
        0x96C6: 0xAEDF,
        0x96D2: 0xAF14,
        0x96DE: 0xAF14,
        0x96EA: 0xAF3F,
        0x96F6: 0xAF3F,
        0x88FC: 0xAF3F,
        0x9702: 0xAF3F,
        0x970E: 0xAF72,
        0x971A: 0xAF72,
        0x9726: 0xAF72,
        0x9732: 0xAFA3,
        0x973E: 0xAFA3,
        0x974A: 0xAFCE,
        0x9756: 0xAFCE,
        0x9762: 0xAFCE,
        0x976E: 0xAFFB,
        0x977A: 0xAFFB,
        0x9786: 0xB026,
        0x9792: 0xB026,
        0x979E: 0xB051,
        0x97AA: 0xB07A,
        0x97B6: 0xB07A,
        0x97C2: 0xB0B4,
        0x97CE: 0xB0DD,
        0x97DA: 0xB106,
        0x97E6: 0xB106,
        0x97F2: 0xB139,
        0x97FE: 0xB139,
        0x980A: 0xB167,
        0x9816: 0xB167,
        0x9822: 0xB192,
        0x982E: 0xB1BB,
        0x983A: 0xB1E5,
        0x9846: 0xB1E5,
        0x9852: 0xB236,
        0x985E: 0xB236,
        0x986A: 0xB236,
        0x88FC: 0xB236,
        0x9876: 0xB283,
        0x9882: 0xB283,
        0x988E: 0xB2DA,
        0x989A: 0xB2DA,
        0x98A6: 0xB305,
        0x98B2: 0xB32E,
        0x98BE: 0xB32E,
        0x98CA: 0xB37A,
        0x98D6: 0xB37A,
        0x98E2: 0xB3A5,
        0x98EE: 0xB3A5,
        0x98FA: 0xB3A5,
        0x9906: 0xB3A5,
        0x9912: 0xB3A5,
        0x991E: 0xB3E1,
        0x992A: 0xB40A,
        0x9936: 0xB40A,
        0x9942: 0xB457,
        0x994E: 0xB457,
        0x995A: 0xB482,
        0x9966: 0xB482,
        0x9972: 0xB4AD,
        0x997E: 0xB4AD,
        0x998A: 0xB4AD,
        0x9996: 0xB4E5,
        0x99A2: 0xB4E5,
        0x99AE: 0xB510,
        0x99BA: 0xB510,
        0x99C6: 0xB510,
        0x99D2: 0xB55A,
        0x99DE: 0xB55A,
        0x99EA: 0xB585,
        0x99F6: 0xB585,
        0x9A02: 0xB585,
        0x9A0E: 0xB585,
        0x9A1A: 0xB5D5,
        0x9A26: 0xB5D5,
        0x9A32: 0xB62B,
        0x9A3E: 0xB62B,
        0x9A4A: 0xB656,
        0x9A56: 0xB656,
        0x9A62: 0xB698,
        0x9A6E: 0xB6C1,
        0x9A7A: 0xB6C1,
        0x9A86: 0xB6C1,
        0x9A92: 0xB6EE,
        0x9A9E: 0xB6EE,
        0x9AAA: 0xB6EE,
        0x9AB6: 0xB741,
        0xA18C: 0xC98E,
        0xA198: 0xC98E,
        0xA1A4: 0xC98E,
        0xA1B0: 0xCA08,
        0xA1BC: 0xCA08,
        0xA1C8: 0xCA52,
        0xA1D4: 0xCA52,
        0xA1E0: 0xCA52,
        0xA1EC: 0xCAAE,
        0xA1F8: 0xCAF6,
        0xA204: 0xCAF6,
        0xA210: 0xCAF6,
        0xA21C: 0xCAF6,
        0xA228: 0xCAF6,
        0xA234: 0xCAF6,
        0xA240: 0xCAF6,
        0xA24C: 0xCB8B,
        0xA258: 0xCB8B,
        0xA264: 0xCBD5,
        0xA270: 0xCBD5,
        0xA27C: 0xCBD5,
        0xA288: 0xCC27,
        0xA294: 0xCC6F,
        0xA2A0: 0xCC6F,
        0xA2AC: 0xCC6F,
        0xA2B8: 0xCCCB,
        0xA2C4: 0xCD13,
        0xA2D0: 0xCD5C,
        0xA2DC: 0xCD5C,
        0xA2E8: 0xCDA8,
        0xA2F4: 0xCDF1,
        0xA300: 0xCE40,
        0xA30C: 0xCE40,
        0xA318: 0xCE8A,
        0xA324: 0xCED2,
        0xA330: 0xCEFB,
        0xA33C: 0xCEFB,
        0xA348: 0xCEFB,
        0xA354: 0xCEFB,
        0xA360: 0xCF54,
        0xA36C: 0xCF54,
        0xA378: 0xCF80,
        0xA384: 0xCF80,
        0xA390: 0xCF80,
        0xA39C: 0xCFC9,
        0xA3A8: 0xCFC9,
        0xA3B4: 0xCFC9,
        0xA3C0: 0xCFC9,
        0xA3CC: 0xCFC9,
        0xA3D8: 0xD017,
        0xA3E4: 0xD017,
        0xA3F0: 0xD017,
        0xA3FC: 0xD017,
        0xA408: 0xD055,
        0xA414: 0xD08A,
        0xA420: 0xD08A,
        0xA42C: 0xD0B9,
        0xA438: 0xD0B9,
        0xA444: 0xD0B9,
        0xA450: 0xD0B9,
        0xA45C: 0xD0B9,
        0xA468: 0xD0B9,
        0xA474: 0xD104,
        0xA480: 0xD104,
        0xA48C: 0xD13B,
        0xA498: 0xD16D,
        0xA4A4: 0xD16D,
        0xA4B0: 0xD1A3,
        0xA4BC: 0xD1A3,
        0xA4C8: 0xD1A3,
        0xA4D4: 0xD1DD,
        0xA4E0: 0xD1DD,
        0xA4EC: 0xD1DD,
        0xA4F8: 0xD21C,
        0xA504: 0xD21C,
        0xA510: 0xD21C,
        0xA51C: 0xD21C,
        0xA528: 0xD252,
        0xA534: 0xD252,
        0xA540: 0xD27E,
        0xA54C: 0xD27E,
        0xA558: 0xD2AA,
        0xA564: 0xD2D9,
        0xA570: 0xD2D9,
        0xA57C: 0xD30B,
        0xA588: 0xD30B,
        0xA594: 0xD30B,
        0xA18A: 0xD30B,
        0xA5A0: 0xD340,
        0xA5AC: 0xD340,
        0xA5B8: 0xD340,
        0xA5C4: 0xD340,
        0xA5D0: 0xD387,
        0xA5DC: 0xD387,
        0xA5E8: 0xD3B6,
        0xA5F4: 0xD3DF,
        0xA600: 0xD408,
        0xA60C: 0xD408,
        0xA618: 0xD433,
        0xA624: 0xD433,
        0xA630: 0xD433,
        0xA63C: 0xD461,
        0xA648: 0xD461,
        0xA654: 0xD461,
        0xA660: 0xD48E,
        0xA66C: 0xD48E,
        0xA678: 0xD48E,
        0xA684: 0xD4C2,
        0xA690: 0xD4C2,
        0xA69C: 0xD4C2,
        0xA6A8: 0xD4EF,
        0xA6B4: 0xD4EF,
        0xA6C0: 0xD51E,
        0xA6CC: 0xD51E,
        0xA6D8: 0xD54D,
        0xA6E4: 0xD54D,
        0xA6F0: 0xD57A,
        0xA6FC: 0xD57A,
        0xA708: 0xD5A7,
        0xA714: 0xD5A7,
        0xA720: 0xD5A7,
        0xA72C: 0xD5A7,
        0xA738: 0xD5A7,
        0xA744: 0xD5A7,
        0xA750: 0xD5EC,
        0xA75C: 0xD5EC,
        0xA768: 0xD617,
        0xA774: 0xD617,
        0xA780: 0xD646,
        0xA78C: 0xD646,
        0xA798: 0xD646,
        0xA7A4: 0xD646,
        0xA7B0: 0xD69A,
        0xA7BC: 0xD69A,
        0xA7C8: 0xD6D0,
        0xA7D4: 0xD6FD,
        0xA7E0: 0xD72A,
        0xA7EC: 0xD72A,
        0xA7F8: 0xD72A,
        0xA828: 0xD765,
        0xA834: 0xD78F,
        0xA840: 0xD78F,
        0xA84C: 0xD7E4,
        0xA858: 0xD7E4,
        0xA864: 0xD7E4,
        0xA870: 0xD7E4,
        0xA87C: 0xD81A,
        0xA888: 0xD81A,
        0xA894: 0xD845,
        0xA8A0: 0xD86E,
        0xA8AC: 0xD898,
        0xA8B8: 0xD898,
        0xA8C4: 0xD8C5,
        0xA8D0: 0xD8C5,
        0xA8DC: 0xD913,
        0xA8E8: 0xD913,
        0xA8F4: 0xD913,
        0xA900: 0xD913,
        0xA90C: 0xD95E,
        0xA918: 0xD95E,
        0xA924: 0xD9AA,
        0xA930: 0xD9D4,
        0xA93C: 0xD9FE,
        0xA948: 0xD9FE,
        0xA954: 0xDA2B,
        0xA960: 0xDA2B,
        0xA96C: 0xDA60,
        0xA978: 0xDA60,
        0xA984: 0xDAAE,
        0xA990: 0xDAAE,
        0xA18A: 0xDAAE,
        0xA99C: 0xDAAE,
        0xA9A8: 0xDAE1,
        0xA9B4: 0xDAE1,
        0xA9C0: 0xDB31,
        0xA9CC: 0xDB31,
        0xA9D8: 0xDB7D,
        0xA9E4: 0xD87D,
        0xA9F0: 0xDBCD,
        0xA9FC: 0xDBCD,
        0xAA08: 0xDC19,
        0xAA14: 0xDC19,
        0xAA20: 0xDC65,
        0xAA2C: 0xDC65,
        0xAA38: 0xDCB1,
        0xAA44: 0xDCB1,
        0xAA50: 0xDCFF,
        0xAA5C: 0xDCFF,
        0xAA68: 0xDCFF,
        0xAA74: 0xDD2E,
        0xAA80: 0xDD58,
        0xAA8C: 0xDD58,
        0xAA98: 0xDDC4,
        0xAAA4: 0xDDC4,
        0xAAB0: 0xDDF3,
        0xAABC: 0xDDF3,
        0xAAC8: 0xDDF3,
        0xAAD4: 0xDE23,
        0xAAE0: 0xDE4D,
        0xAAEC: 0xDE4D,
        0xAAF8: 0xDE7A,
        0xAB04: 0xDE7A,
        0xAB10: 0xDEA7,
        0xAB1C: 0xDEA7,
        0xAB28: 0xDEDE,
        0xAB34: 0xDEDE,
        0xAB40: 0xDF1B,
        0xAB4C: 0xDF45,
        0xAB58: 0xDF8D,
        0xAB64: 0xDF8D,
        0xAB70: 0xDFD7,
        0xAB7C: 0xDFD7,
        0xAB88: 0xE021,
        0xAB94: 0xE021,
        0xABA0: 0xE06B,
        0xABAC: 0xE06B,
        0xABB8: 0xE0B5,
        0xABC4: 0xE82C,
        0xABCF: 0xE82C,
        0xABDA: 0xE82C,
        0xABE5: 0xE82C,
    }
