import { Construct, Resource } from '@aws-cdk/core';
import { SubscriptionFilter } from './subscription-filter';
import { ITopic } from './topic-base';
/**
 * Options for creating a new subscription
 */
export interface SubscriptionOptions {
    /**
     * What type of subscription to add.
     */
    readonly protocol: SubscriptionProtocol;
    /**
     * The subscription endpoint.
     *
     * The meaning of this value depends on the value for 'protocol'.
     */
    readonly endpoint: string;
    /**
     * true if raw message delivery is enabled for the subscription. Raw messages are free of JSON formatting and can be
     * sent to HTTP/S and Amazon SQS endpoints. For more information, see GetSubscriptionAttributes in the Amazon Simple
     * Notification Service API Reference.
     *
     * @default false
     */
    readonly rawMessageDelivery?: boolean;
    /**
     * The filter policy.
     *
     * @default - all messages are delivered
     */
    readonly filterPolicy?: {
        [attribute: string]: SubscriptionFilter;
    };
    /**
     * The region where the topic resides, in the case of cross-region subscriptions
     * @link https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sns-subscription.html#cfn-sns-subscription-region
     * @default - the region where the CloudFormation stack is being deployed.
     */
    readonly region?: string;
}
/**
 * Properties for creating a new subscription
 */
export interface SubscriptionProps extends SubscriptionOptions {
    /**
     * The topic to subscribe to.
     */
    readonly topic: ITopic;
}
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
export declare class Subscription extends Resource {
    private readonly filterPolicy?;
    constructor(scope: Construct, id: string, props: SubscriptionProps);
}
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
export declare enum SubscriptionProtocol {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    HTTP = "http",
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    HTTPS = "https",
    /**
     * Notifications are sent via email.
     */
    EMAIL = "email",
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    EMAIL_JSON = "email-json",
    /**
     * Notification is delivered by SMS
     */
    SMS = "sms",
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SQS = "sqs",
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    APPLICATION = "application",
    /**
     * Notifications trigger a Lambda function.
     */
    LAMBDA = "lambda"
}
