"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_substrings_1 = require("@udondan/common-substrings");
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.accessLevelList = {};
        this.useNotActions = false;
        this.actions = [];
        this.cdkActionsApplied = false;
        this.isCompact = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            if (this.isCompact) {
                this.compactActions();
            }
            const actions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            statement[mode] = actions.length > 1 ? actions : actions[0];
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            if (this.isCompact) {
                this.compactActions();
            }
            const uniqueActions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * If the action does not contain a colon, the action will be prefixed with the service prefix of the class, e.g. `ec2:`
     *
     * @param action Actions that will be added to the statement.
     */
    to(action) {
        if (this.servicePrefix.length && action.indexOf(':') < 0) {
            action = this.servicePrefix + ':' + action;
        }
        this.actions.push(action);
        return this;
    }
    /**
     * Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.to(`${this.servicePrefix}:*`);
        }
        else {
            this.to('*');
        }
        return this;
    }
    /**
     * Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions. The regular expressions need to be in [Perl/JavaScript literal style](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions) and need to be passed as strings,
     * For example:
     * ```typescript
     * allMatchingActions('/vpn/i')
     * ```
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [_, actions] of Object.entries(this.accessLevelList)) {
                actions.forEach((action) => {
                    if (action.match(RegexParser(expression))) {
                        this.to(`${this.servicePrefix}:${action}`);
                    }
                });
            }
        });
        return this;
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        if (accessLevel in this.accessLevelList) {
            this.accessLevelList[accessLevel].forEach((action) => {
                this.to(`${this.servicePrefix}:${action}`);
            });
        }
        return this;
    }
    /**
     * Condense action list down to a list of patterns.
     *
     * Using this method can help to reduce the policy size.
     *
     * For example, all actions with access level `list` could be reduced to a small pattern `List*`.
     */
    compact() {
        this.isCompact = true;
        return this;
    }
    compactActions() {
        // actions that will be included, service prefix is removed
        const includeActions = this.actions.map((elem) => {
            return elem.substr(elem.indexOf(':') + 1);
        });
        // actions that will not be included, the opposite of includeActions
        const excludeActions = [];
        for (const [_, actions] of Object.entries(this.accessLevelList)) {
            actions.forEach((action) => {
                if (!includeActions.includes(action)) {
                    excludeActions.push(`^${action}$`);
                }
            });
        }
        // will contain the index of elements that are covered by substrings
        let covered = [];
        const subs = common_substrings_1.default(includeActions.map((action) => {
            return `^${action}$`;
        }), {
            minLength: 3,
            minOccurrence: 2,
        })
            .filter((sub) => {
            // remove all substrings, that match an action we have not selected
            for (let action of excludeActions) {
                if (action.includes(sub.name)) {
                    return false;
                }
            }
            return true;
        })
            .sort((a, b) => {
            // sort list by the number of matches
            if (a.source.length < b.source.length)
                return 1;
            if (a.source.length > b.source.length)
                return -1;
            return 0;
        })
            .filter((sub) => {
            // removes substrings, that have already been covered by other substrings
            const sources = sub.source.filter((source) => {
                return !covered.includes(source);
            });
            if (sources.length > 1) {
                //add list of sources to the global list of covered actions
                covered = covered.concat(sources);
                return true;
            }
            return false;
        });
        // stores the list of patterns
        const compactActionList = [];
        subs.forEach((sub) => {
            compactActionList.push(`${this.servicePrefix}:*${sub.name}*`
                .replace('*^', '')
                .replace('$*', ''));
            sub.source.forEach((source) => {
                includeActions[source] = ''; // invalidate, will be filtered later
            });
        });
        includeActions
            .filter((action) => {
            // remove elements that have been covered by patterns, we invalidated them above
            return action.length > 0;
        })
            .forEach((action) => {
            // add actions that have not been covered by patterns to the new action list
            compactActionList.push(`${this.servicePrefix}:${action}`);
        });
        // we're done. override action list
        this.actions = compactActionList;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithActions[_a] = { fqn: "iam-floyd.PolicyStatementWithActions", version: "0.325.0" };
//# sourceMappingURL=data:application/json;base64,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