import numpy
from scipy.optimize import curve_fit


#
# misc
#

def atomic_symbols():
    return [
        'Vacuum', 'H', 'He', 'Li', 'Be', 'B', 'C', 'N', 'O', 'F', 'Ne',
        'Na', 'Mg', 'Al', 'Si', 'P', 'S', 'Cl', 'Ar', 'K', 'Ca',
        'Sc', 'Ti', 'V', 'Cr', 'Mn', 'Fe', 'Co', 'Ni', 'Cu', 'Zn',
        'Ga', 'Ge', 'As', 'Se', 'Br', 'Kr', 'Rb', 'Sr', 'Y', 'Zr',
        'Nb', 'Mo', 'Tc', 'Ru', 'Rh', 'Pd', 'Ag', 'Cd', 'In', 'Sn',
        'Sb', 'Te', 'I', 'Xe', 'Cs', 'Ba', 'La', 'Ce', 'Pr', 'Nd',
        'Pm', 'Sm', 'Eu', 'Gd', 'Tb', 'Dy', 'Ho', 'Er', 'Tm', 'Yb',
        'Lu', 'Hf', 'Ta', 'W', 'Re', 'Os', 'Ir', 'Pt', 'Au', 'Hg',
        'Tl', 'Pb', 'Bi', 'Po', 'At', 'Rn', 'Fr', 'Ra', 'Ac', 'Th',
        'Pa', 'U', 'Np', 'Pu', 'Am', 'Cm', 'Bk', 'Cf', 'Es', 'Fm',
        'Md', 'No', 'Lr', 'Rf', 'Db', 'Sg', 'Bh', 'Hs', 'Mt', 'Ds',
        'Rg', 'Uub', 'Uut', 'Uuq', 'Uup', 'Uuh', 'Uus', 'Uuo']

def atomic_names():
    return [
        'Vacuum',
        'Hydrogen', 'Helium', 'Lithium', 'Beryllium', 'Boron',
        'Carbon', 'Nitrogen', 'Oxygen', 'Fluorine', 'Neon',
        'Sodium', 'Magnesium', 'Aluminum', 'Silicon', 'Phosphorus',
        'Sulfur', 'Chlorine', 'Argon', 'Potassium', 'Calcium',
        'Scandium', 'Titanium', 'Vanadium', 'Chromium', 'Manganese',
        'Iron', 'Cobalt', 'Nickel', 'Copper', 'Zinc',
        'Gallium', 'Germanium', 'Arsenic', 'Selenium', 'Bromine',
        'Krypton', 'Rubidium', 'Strontium', 'Yttrium', 'Zirconium',
        'Niobium', 'Molybdenum', 'Technetium', 'Ruthenium', 'Rhodium',
        'Palladium', 'Silver', 'Cadmium', 'Indium', 'Tin',
        'Antimony', 'Tellurium', 'Iodine', 'Xenon', 'Cesium',
        'Barium', 'Lanthanum', 'Cerium', 'Praseodymium', 'Neodymium',
        'Promethium', 'Samarium', 'Europium', 'Gadolinium', 'Terbium',
        'Dysprosium', 'Holmium', 'Erbium', 'Thulium', 'Ytterbium',
        'Lutetium', 'Hafnium', 'Tantalum', 'Tungsten', 'Rhenium',
        'Osmium', 'Iridium', 'Platinum', 'Gold', 'Mercury',
        'Thallium', 'Lead', 'Bismuth', 'Polonium', 'Astatine',
        'Radon', 'Francium', 'Radium', 'Actinium', 'Thorium',
        'Protactinium', 'Uranium', 'Neptunium', 'Plutonium', 'Americium',
        'Curium', 'Berkelium', 'Californium', 'Einsteinium', 'Fermium',
        'Mendelevium', 'Nobelium', 'Lawrencium', 'Rutherfordium', 'Dubnium',
        'Seaborgium', 'Bohrium', 'Hassium', 'Meitnerium', 'Darmstadtium',
        'Roentgenium', 'Ununbium', 'Ununtrium', 'Ununquadium', 'Ununpentium',
        'Ununhexium', 'Ununseptium', 'Ununoctium']

def atomic_number(symbol):
    return atomic_symbols().index(symbol)

def parse_formula(formula, verbose=False):

    import re
    if formula.count('(') == 0:
        pass
    elif formula.count('(') == 1:
        if verbose: print("Found parentheses")
        match = re.search(r'\([\x00-\xFF]*\)(\d\.?\d?)', formula)
        subformula = match.group(0)

        if verbose: print("    >>>>>>> subformula", subformula)

        match = re.search(r'\(([^\)]*)\)', subformula)
        subformula_inside_parentheses = match.group(0)[1:-1]
        if verbose: print("    >>>>>>> subformula inside parentheses", subformula_inside_parentheses)

        match = re.search(r'\)\d\.?\d?', subformula)
        times = float(match.group(0)[1:])
        if verbose: print("    >>>>>>> times", times)

        zetas, fatomic = parse_formula(subformula_inside_parentheses, verbose=verbose)
        if verbose: print("    >>>> zetas, fatomic:", zetas, fatomic)
        string = ''
        for i, element in enumerate(zetas):
            string += atomic_symbols()[element] + "%g" % (fatomic[i] * times)

            if verbose: print("    expanded: ", string, "replaced", formula.replace(subformula, string))

        return parse_formula(formula.replace(subformula, string), verbose=verbose)
    else:
        raise NotImplementedError("wrong formula %s: multiple parentheses not implemented" % formula)

    tmp = re.findall(r'([A-Z][a-z]*)(\d*\.?\d?)', formula)

    fatomic = []
    zetas = []
    for element, str_number in tmp:
        if str_number == '':
            number = 1
        else:
            number = float(str_number)

        fatomic.append(float(number))
        zetas.append(atomic_number(element))

    return zetas, fatomic


#=======================================================================================================================
#  f utilities
#=======================================================================================================================


#TODO: in future we will use xraylib, by now we use this.
def f0_xop(Z):
    """
    Returns the f0 value as stired in the f0_xop DABAX file:

    Parametrization of f0 (the non-dispersive part of the atomic scttering factor) vs sin(theta/lambda).
    This file contains a Waasmaier&Kirfel-like parametrization
    for the f0 data evaluated by Kissel using modified relativistic form
    factor [1]. For the fitting, model and the data error as in [2] are used.
    The parametrization was calculating by fitting the
    original data in the RTAB database [1] (also in the DABAX file
    f0_mf_Kissel.dat) using the MPFIT IDL procedure of Markwardt
    (http://cow.physics.wisc.edu/~craigm/idl/idl.html) usind the
    function [2]:
       f0[k] = c + [SUM a_i*EXP(-b_i*(k^2)) ]
                   i=1,5
     where k = sin(theta) / lambda and c, a_i and b_i
     are the coefficients tabulated in this file (in columns:
     a1  a2  a3  a4  a5  c  b1  b2  b3  b4  b5

    References
    [1] L. Kissel, Radiation physics and chemistry 59 (2000) 185-200
        available at:
        http://www-phys.llnl.gov/Research/scattering/RTAB.html
    [2] D. Waasmaier & A. Kirfel (Acta Cryst. (1995). A51, 416-413)


     Remarks:
     1) This file contains data for only neutral atoms. For ionic states
        other DABAX files (like f0_WaasKirf.dat) could be used.
     2) The coefficients in this file are prepared for xop2.1 and are
        unpublished.
     3) We created this file for being used as a default for XOP2.1. The
        reasons were two:
        i) It is more practical to use a parametrization rather than to use
           the direct data in file f0_mf.dat because of spped and storage
           reasons.
        ii) The previous defaulty file for XOP 2.0 was f0_WaasKirf.dat, which
           contained a published parametrization [2], but the original data
           is from the International Tables for X-ray Crystallography, vol C,
           (1995) Edited by AJC Wilson, Kluwer Academic Press, table 6.1.1.3
           which contains data from multiple origins (and references). We
           prefer to use evaluated data in the same conditions for all elements
           to keep some consistency in their origin.
           Additionally, Kissel data for f1f2 and CrossSec are also used in XOP.

    Column description: a1  a2  a3  a4  a5  c  b1  b2  b3 b4  b5

    :param Z: the atomic number
    :return: the 11 coefficients for buiding f0
    """
    tmp = [
    [ 0.30426303,     0.45440815,     0.07977026,     0.15268426,     0.00871624,     0.00006937,     9.61768510,    23.52152246,     3.33237545,    53.49579285,     0.80301704],
    [ 0.79971010,     0.63926998,     0.25291139,     0.27260951,     0.03475295,     0.00044196,    10.35354478,     3.93677741,     1.38554540,    25.50380187,     0.34363759],
    [ 0.93188799,     0.17034671,     0.71602201,     0.35839485,     0.82020547,     0.00260460,     4.17494209,     0.33818181,    85.52950781,   175.66836887,     1.37984909],
    [ 1.35769775,     0.71952631,     0.76849159,     0.14970777,     0.99985829,     0.00347440,    37.89541686,     0.66643295,    92.11060309,     0.17800503,     1.90321664],
    [ 2.11021585,     0.94826030,     1.03175074,     0.17991800,     0.72039282,     0.00538888,    21.36228681,     1.17425000,    65.42872639,     0.12888999,     0.44259026],
    [ 2.75491071,     1.06083859,     1.39000885,   -61.38969569,     0.72824752,    61.44921510,    13.55853062,     0.76518509,    43.40152845,    -0.00002249,     0.23756415],
    [ 0.59731711,     2.28313328,     0.63580984,     2.26859990,     1.18920099,     0.02241857,     0.13665445,    17.86124963,    47.32019642,     7.67047709,     0.49262938],
    [ 2.85010001,     2.54410664,     0.64919585,     0.79765894,     1.12558170,     0.02853916,    12.89518279,     5.45538842,     0.11296751,    35.32893278,     0.38721225],
    [ 3.40542650,     2.84112029,     0.71156027,     0.94857450,     1.05115146,     0.03584797,     9.84065316,     4.07801707,     0.09701872,    27.57669889,     0.31753486],
    [ 3.93418627,     3.17292749,     0.78988890,     1.08878497,     0.96122888,     0.04476065,     7.82568355,     3.17536738,     0.08620371,    22.24803408,     0.27139624],
    [ 4.67888021,     3.31246434,     1.20302749,     1.11030781,     0.60080617,     0.07903453,     2.97160025,     8.01401365,     0.09917076,   119.87596938,     0.36875620],
    [ 4.85834741,     1.50402245,     1.53555848,     0.95416075,     3.00404753,     0.13060425,     4.35307904,    95.26343316,     0.11044872,    31.99246407,     1.75670443],
    [ 4.68669359,     2.38879059,     1.52287056,     1.07903978,     3.16354871,     0.14361992,     3.40015051,    36.83643520,     0.09557493,   114.27524168,     1.47492947],
    [ 4.98816795,     3.35710271,     1.50292204,     1.22172882,     2.76143663,     0.15142442,     2.53600438,    29.97580504,     0.08254945,    88.73513838,     1.16712390],
    [ 2.10509260,     4.36716206,     1.48028761,     1.33409382,     5.53780604,     0.15604602,     0.89460753,    24.41881968,     0.07145192,    69.85860169,     1.90671824],
    [ 6.20239600,     5.39538819,     1.45887725,     1.42806579,     1.33894393,     0.15420384,     1.46090299,    20.14653917,     0.06164271,    56.73801104,     0.64755460],
    [ 1.42704046,     6.77857375,     6.41364632,     1.52938606,     0.68790919,     0.13835865,     0.05185139,     1.14744676,    16.83234374,    47.08267330,     0.39585031],
    [ 7.12496467,     6.96362577,     0.40535492,     1.69961663,     1.50025574,     0.27745601,     0.91908882,    14.15475105,    14.15474574,    39.07709506,     0.06417165],
    [ 8.37197122,     7.07989232,     1.01347196,     0.77223727,     1.46667316,     0.26287737,    11.89949237,     0.78056461,   198.44072308,    41.56210501,     0.05445260],
    [ 8.64392994,     1.46357896,     1.56066861,     1.05545005,     7.04009562,     0.19999674,     9.69070226,     0.04305143,    72.04776233,   166.55421240,     0.66621783],
    [ 1.68383652,     1.47480433,     1.37047425,     9.24659290,     7.02207943,     0.16099853,    49.92976638,     0.03653814,   134.31889317,     8.45519757,     0.58503649],
    [ 9.88746586,     1.50747326,     1.90635535,     1.54310395,     7.00376164,     0.10558296,     7.44889667,     0.03084452,    38.16657035,   115.68708856,     0.51848158],
    [10.51693874,     1.57393530,     2.16699802,     1.69193955,     6.98396214,     0.01430187,     6.58802517,     0.02534021,    30.05889048,   101.44514004,     0.46207981],
    [11.07627441,     1.58619669,     2.93288168,     1.37491086,     6.96664643,     0.00459082,     5.94437112,     0.02328655,    20.59115302,    96.88722975,     0.41908215],
    [11.73712145,     1.82995875,     2.81141301,     1.90065220,     6.94066322,    -0.28355091,     5.23367741,     0.01636240,    20.53719746,    82.01631881,     0.37401385],
    [12.30434401,     2.32437385,     3.18807287,     1.99400470,     6.92612145,    -0.80737786,     4.69095987,     0.01074597,    17.39159075,    74.61601310,     0.33856083],
    [12.87035966,     4.43881362,     3.58294665,     2.06698965,     6.91586723,    -2.95253425,     4.22745007,     0.00461408,    15.08330800,    68.44820515,     0.30788472],
    [13.40782101,     6.90042218,     3.99539727,     2.14745733,   283.90205482,  -282.43818444,     3.82465894,     0.28165040,    13.17017325,    62.81493664,     0.00006082],
    [13.75399998,   297.69542224,     5.21908965,     1.58299921,     6.87528047,  -296.22007511,     3.49108135,     0.00005604,    11.26181897,    62.07734534,     0.26153431],
    [14.43242183,     6.83363443,     4.86041106,     2.31023153,   518.74776544,  -517.28462315,     3.16729215,     0.24008925,    10.34604629,    53.61746648,     0.00002943],
    [15.57306569,     6.80777698,     4.48158313,     2.55416620,   237.18610925,  -235.71998143,     2.94104939,     0.22275367,    11.54501025,    63.56972773,     0.00006123],
    [16.29213242,   271.89501093,     4.17901049,     3.17804713,     6.76929988,  -270.43761631,     2.69207459,     0.00004947,    12.10034542,    55.59088173,     0.20581071],
    [16.79861850,   336.43256975,     4.18039017,     3.72954296,     6.72535523,  -334.99650264,     2.44950901,     0.00003586,    12.83743210,    47.67503824,     0.18952317],
    [17.16712180,   336.96736021,     4.71747652,     3.89487231,     6.68380463,  -335.56546318,     2.22463287,     0.00003080,    14.03902313,    42.86632703,     0.17412989],
    [17.40231392,     5.93891344,   354.54400524,     3.52445295,     6.64817985,  -353.19765239,     2.01532010,    15.16569556,     0.00002303,    40.69766838,     0.15923958],
    [17.53267157,     7.44816522,   267.89934293,     2.98742575,     6.61999042,  -266.63403399,     1.82191497,    15.41761348,     0.00002029,    39.30642110,     0.14476941],
    [ 8.87288109,     1.56926290,     6.66073618,     1.08275119,    17.57340057,     1.08665457,    14.34390518,    35.97531308,     0.12903646,   209.51615220,     1.64279492],
    [17.59670810,     9.94862630,     6.04462400,     2.61652635,     0.54472165,     1.07329335,     1.49577834,    13.12002382,     0.11972149,   119.44032856,     0.11972149],
    [17.65536751,    10.43931521,     5.65885161,     3.13610892,     0.86203672,     1.06017183,     1.36826402,    12.12449612,     0.11133571,    95.69676914,     0.11133576],
    [17.70896284,    11.09168541,     5.76424243,     3.49865793,     0.68990337,     1.04561396,     1.25477542,    11.37544137,     0.10372418,    80.89514696,     0.10372414],
    [17.82410279,    12.34652807,     4.93750714,     3.19055176,     1.46016577,     1.02630932,     1.15717637,    11.35742646,     0.09662928,    69.39554321,     0.09662930],
    [ 6.17787753,    17.84044504,    13.24473559,     3.35857771,     0.15154195,     0.99960433,     0.08963202,     1.06333880,    10.67451656,    61.58100361,     0.08963181],
    [17.74278282,     3.79497777,     0.79808174,    13.24774635,     6.24221603,     0.95794137,     0.97113299,    42.86390771,   132.10191226,     9.26979654,     0.08236923],
    [ 6.18155275,    17.80169136,    14.63083035,     3.59439161,     0.64428903,     0.91877951,     0.07624173,     0.89717697,     9.12737711,    36.30062507,   130.83719947],
    [ 6.12780040,    17.79870560,     3.70551025,     0.61262679,    15.64710027,     0.86745289,     0.07028489,     0.82885382,    33.92215621,   126.42588650,     8.60698534],
    [ 6.09085570,     4.68870690,    17.15641530,     3.84300712,    13.15123080,     0.80254284,     0.06457257,     0.76863619,     8.38997512,    34.11521928,     0.76863617],
    [ 6.05724103,    17.76734177,     3.85910747,     0.54976811,    17.79132902,     0.70865971,     0.05857511,     7.66424959,    30.20483695,   121.19639558,     0.70979037],
    [ 6.04036427,    18.38160827,     4.20316493,     0.75471621,    17.74206099,     0.59841397,     0.05297111,     7.01093020,    30.72014689,   106.80764634,     0.65631719],
    [ 6.06154651,    19.00060821,     4.45915941,     1.04270193,    17.69494384,     0.44547768,     0.04732772,     6.43011640,    31.38897338,   129.84310238,     0.60802939],
    [19.44406366,     6.14105459,     5.04240100,     1.19460095,    17.63952092,     0.22942056,     5.87081521,     0.04153040,    32.37864163,   114.94916342,     0.56396218],
    [ 6.01179028,     6.25268840,    19.69743565,     1.16859113,    17.56401079,    -0.01748380,    32.17182048,     0.03634623,     5.33340196,   103.27699871,     0.52334570],
    [ 7.27011998,     6.55126218,    19.86580814,     0.95229359,    17.52010619,    -0.49670130,    31.27680875,     0.02990172,     4.84670590,   100.88905997,     0.48406725],
    [19.93807829,     7.09598483,     8.55254548,     0.81322939,    17.46003963,    -1.21189020,     4.40199492,     0.02378768,    29.37558678,    96.37033204,     0.44869824],
    [19.95850428,     8.36420076,     9.90402346,     0.65245953,    17.40931309,    -2.65593783,     4.00466253,     0.01705338,    27.35376466,    96.62465241,     0.41685323],
    [17.37569453,    12.13586009,    10.55273124,     1.29778936,    19.87680486,    -6.62800954,     0.38664179,     0.00966981,    23.44864968,   220.56960899,     3.64363491],
    [19.82277247,    17.37211521,    10.64912854,     2.42340529,    51.54683902,   -46.21974320,     3.34809910,     0.36080812,    20.04789088,   161.05858343,     0.00185019],
    [19.96684466,  2616.77020357,    11.08605976,     2.97061929,    17.30250838, -2611.52103692,     3.11105960,     0.00003320,    18.89301922,   126.40106401,     0.33961309],
    [17.29595781,  4345.36340933,    20.94755728,     2.49690482,    11.57180940, -4340.11668462,     0.32551631,     0.00001928,     3.04729063,   146.75357391,    17.76655273],
    [21.56386679,    17.23892858,    11.96705606,     2.55058606,  4645.78355064, -4640.56446888,     2.91250166,     0.30985101,    16.75073376,   139.14334527,     0.00001725],
    [17.18849734,  5256.12632558,    12.33127623,     2.58243269,    22.21683582, -5250.92543157,     0.29580553,     0.00001461,    15.88877581,   133.41685838,     2.79253798],
    [17.14790054,  5467.09133967,    12.67590249,     2.61445255,    22.89314617, -5461.92264966,     0.28225651,     0.00001335,    15.09888908,   128.17517281,     2.67966474],
    [23.57776892,  4336.95334631,    13.01186004,     2.65112019,    17.09246464, -4331.80653530,     2.57205344,     0.00001606,    14.35914195,   123.21157749,     0.26967441],
    [17.07135563,  5126.49069186,    13.33466886,     2.65426567,    24.27237487, -5121.36316596,     0.25889588,     0.00001297,    13.73081795,   120.24251547,     2.47764087],
    [24.64625088,    16.99747848,    13.36466048,     3.34514581,  4543.57338365, -4538.49039355,     2.34262325,     0.24679775,    12.90194424,    95.39774603,     0.00001379],
    [25.66612063,  4192.75632504,    13.95736847,     2.72815093,    17.00044486, -4187.69180909,     2.29102865,     0.00001424,    12.50615875,   111.05763849,     0.23787894],
    [26.38323252,  4398.24494424,    14.26736529,     2.76532287,    16.97289395, -4393.23947589,     2.20308248,     0.00001260,    11.95434176,   106.89618428,     0.22734974],
    [27.09405383,    16.93749917,    14.58634079,     2.80145206,  3698.14032223, -3693.18846082,     2.11867358,     0.21749694,    11.43043354,   102.92181598,     0.00001396],
    [27.81053249,  4104.64353871,    14.88325935,     2.83483528,    16.89331430, -4099.71721767,     2.04181579,     0.00001196,    10.95422544,    99.42542607,     0.20889623],
    [28.52850085,  2445.57145486,    15.22240776,     2.86116529,    16.92020688, -2440.77861007,     1.96383816,     0.00001716,    10.48926663,    96.46598501,     0.19884908],
    [29.24441972,  2212.27369164,    15.49510957,     2.92480251,    16.89099012, -2207.52991878,     1.89336193,     0.00001759,    10.05627636,    91.69307079,     0.19084871],
    [29.69788948,    15.47177518,  2097.86718749,     3.60694158,    16.82362572, -2093.19263162,     1.80307602,     9.47929223,     0.00001698,    77.64749351,     0.18222067],
    [30.11062139,    15.40407022,  2274.03155954,     4.38097094,    16.78436412, -2269.46340922,     1.71565360,     8.96253008,     0.00001361,    63.95107044,     0.17352112],
    [30.57052118,    15.57376402,  1353.88450796,     4.87958160,    16.74532612, -1349.42935153,     1.63430749,     8.68104102,     0.00001949,    56.83069217,     0.16513675],
    [30.98784842,    15.84365824,  1085.37020439,     5.34491722,    16.75914099, -1081.10872457,     1.55377182,     8.43294944,     0.00001745,    50.66121806,     0.15616127],
    [31.36764074,    16.20662904,  1140.89802522,     5.70714498,    16.65069139, -1136.66620413,     1.48081504,     8.26474305,     0.00001603,    45.63908603,     0.14966164],
    [31.75193454,    16.78095382,   410.94969397,     5.94974979,    16.75723964,  -407.05071307,     1.40570152,     8.12076847,     0.00001747,    42.41784435,     0.13977093],
    [31.76739790,     1.41742145,    16.74659192,    16.21969187,     6.34032022,     3.65295339,     1.32218506,    81.15672344,     0.13119994,     7.35439190,    26.57488749],
    [30.98992203,    18.15501717,    16.67920729,     6.30866056,     1.30977395,     3.63949590,     1.27222945,     7.83558995,     0.12654931,    36.86734925,     1.27222941],
    [16.88254232,    19.37581635,    32.62146048,     6.01941144,   432.23434409,  -429.07568590,     0.11690541,     7.87648778,     1.20474478,    32.97062267,    -0.00003099],
    [16.43979396,    19.91863890,    27.76463255,     6.41022928,     4.90481781,     3.58984976,     0.11673200,     7.55331808,     1.15426441,    33.30365931,     1.15426437],
    [16.27949672,    19.61124539,    32.52808287,     1.21893192,     6.82864702,     3.56321128,     0.11193237,     6.83979837,     1.09072759,   118.61695437,    25.09178834],
    [16.15573594,    32.56471040,     6.74799686,     1.62318006,    20.37867502,     3.53108830,     0.10731472,     1.03828673,    25.84002437,   104.03280242,     6.51727715],
    [16.05513439,    32.57219633,     7.10079270,     1.82264739,    20.92330655,     3.49908832,     0.10310727,     0.99041508,    26.79675111,    92.39516651,     6.16859916],
    [15.92966393,    32.53463621,    21.26597071,     1.85348130,     7.90481807,     3.45521535,     0.09871917,     0.94233879,     5.77485734,    84.25758041,    27.14259799],
    [15.82334418,    32.46748196,     9.01250570,     1.72434540,    21.46356994,     3.42272786,     0.09486320,     0.89843768,    27.00385753,    79.63992570,     5.39445822],
    [15.71714768,    32.36554794,    21.55254018,     1.54577949,    10.31957180,     3.38282129,     0.09111052,     0.85619893,     5.01523482,    76.65511543,    26.25222808],
    [15.61356578,    32.29005100,    21.58808798,     1.52079712,    11.49239752,     3.33918799,     0.08744016,     0.81690795,     4.68491604,   187.13075688,    24.42129865],
    [32.26137613,    21.47411433,    11.54875240,     2.70070705,    15.53353668,     3.29342205,     0.78242755,     4.39338862,    21.26815521,   145.78453108,     0.08389021],
    [15.44129352,    32.19461688,    21.67018557,    11.62375491,     3.60839956,     3.23867141,     0.08037991,     0.74781467,     4.14033299,    19.84028498,   113.68244239],
    [15.35041360,    32.15264867,    21.98489988,     4.20304734,    11.87984118,     3.17150887,     0.07685215,     0.71518502,     3.92354148,    96.69516744,    19.25785705],
    [32.41734383,    22.05085313,    13.10871855,     3.72910194,    15.28123528,     3.11865061,     0.68922416,     3.86872813,    17.74443730,   101.90118122,     0.07385957],
    [15.20314929,    32.53208188,    13.81186567,     3.76642798,    22.31042147,     3.04559348,     0.07066435,     0.66171756,    16.79213936,    97.72588956,     3.75457984],
    [32.64327679,    22.60430101,    14.43752068,     3.83052598,    15.12931302,     2.98794434,     0.63635924,     3.65636186,    15.88803466,    93.50959958,     0.06783444],
    [32.89114822,    23.09839499,    15.57640146,     3.06849154,    15.06334425,     2.89811885,     0.61290312,     3.63716663,    15.44356030,   103.61518316,     0.06473308],
    [33.02310917,    23.59414755,    16.08710719,     3.05160429,    15.00738866,     2.79583742,     0.58946182,     3.56398355,    14.80962925,   101.26355106,     0.06167575],
    [14.92952838,    33.03254055,    24.00228529,     3.79349958,    16.07933216,     2.68191187,     0.05858286,     0.56470268,     3.42619535,    86.81878918,    13.93081918],
    [14.87484815,    33.25429616,    24.75369621,     3.05890997,    16.98364667,     2.55564102,     0.05561473,     0.54346670,     3.41941648,    95.53275618,    13.62900417],
    [33.35395184,    25.38419399,    17.37894160,     3.08843097,    14.82268135,     2.41071058,     0.52171734,     3.34704782,    13.07655187,    91.79465535,     0.05263332],
    ]
    tmp = numpy.array(tmp)
    return tmp[Z-1].copy()

#
# f0
#

def calculate_f0_from_f0coeff(f0coeff, ratio):
    icentral = len(f0coeff) // 2
    F0 = f0coeff[icentral]
    for i in range(icentral):
        F0 += f0coeff[i] * numpy.exp(-1.0 * f0coeff[i + icentral + 1] * ratio ** 2)
    return F0

def f0_interpolate_coefficients(charge, charge_list, coefficient_list, verbose=False):
    #
    # f0 data
    #

    nitems = len(charge_list)

    if nitems == 1:
        print("Warning: no interpolating of charge: only one value available for Z=%d" % Z)
        return coefficient_list[0]

    charge_list_difference = []
    for i in range(nitems):
        charge_list_difference.append(charge_list[i] - charge)

    charge_list_difference = numpy.array(charge_list_difference)  # convert to numpy array

    if numpy.abs(charge_list_difference).min() == 0:
        idx = numpy.abs(charge_list_difference).argmin()
        if verbose: print("No interpolating needed: returning value for charge = ", charge_list[idx])
        return coefficient_list[idx]

    # get the closer charge values, no matter of the sign

    sorted_indices = numpy.argsort(numpy.abs(charge_list_difference))
    sorted_index_0 = sorted_indices[0]
    sorted_index_1 = sorted_indices[1]
    delta_data = charge_list[sorted_index_1] - charge_list[sorted_index_0]
    delta_charge = charge - charge_list[sorted_index_0]
    delta = delta_charge / delta_data
    if verbose: print("Interpolating charge %g = %s + %g (%s - %s)" % (charge,
                                                                       charge_list[sorted_index_0],
                                                                       delta,
                                                                       charge_list[sorted_index_1],
                                                                       charge_list[sorted_index_0]))

    # interpolate to get the f0 for the wanted charge

    q = numpy.linspace(0.0, 2.0, 100)
    f0_0 = calculate_f0_from_f0coeff(coefficient_list[sorted_index_0], q)
    f0_1 = calculate_f0_from_f0coeff(coefficient_list[sorted_index_1], q)
    f0 = f0_0 + delta * (f0_1 - f0_0)

    #
    # fit
    #
    try:
        popt, pcov = curve_fit(__f0func, q, f0, p0=coefficient_list[sorted_index_0], maxfev=20000)
        if verbose: print("fitted: ", popt)

        return popt
    except:
        if verbose: print("Error: failed to fit coefficients for fractional charge. Returning the ones of charge ",
                          charge_list[sorted_index_0])
        return coefficient_list[sorted_index_0]

def __f0func(q, a1, a2, a3, a4, a5, a6, a7, a8, a9):
    return calculate_f0_from_f0coeff([a1, a2, a3, a4, a5, a6, a7, a8, a9], q)


def f0_xop_with_fractional_charge_data(Z):
    a = []
    a.append({'Z': 1 ,'charge_list': [0.0, -1] ,'coefficient_list': [[0.493002, 0.322912, 0.140191, 0.04081, 0.0030380001, 10.5109, 26.1257, 3.14236, 57.7997], [0.897661, 0.565616, 0.415815, 0.116973, 0.002389, 53.1368, 15.187, 186.576, 3.56709]] })
    a.append({'Z': 2 ,'charge_list': [0.0] ,'coefficient_list': [[0.8734, 0.6309, 0.3112, 0.178, 0.0063999998, 9.1037, 3.3568, 22.9276, 0.9821]] })
    a.append({'Z': 3 ,'charge_list': [0.0, 1] ,'coefficient_list': [[1.1282, 0.7508, 0.6175, 0.4653, 0.037700001, 3.9546, 1.0524, 85.3905, 168.261], [0.6968, 0.7888, 0.3414, 0.1563, 0.0167, 4.6237, 1.9557, 0.6316, 10.0953]] })
    a.append({'Z': 4 ,'charge_list': [0.0, 2] ,'coefficient_list': [[1.5919, 1.1278, 0.5391, 0.7029, 0.0385, 43.6427, 1.8623, 103.483, 0.542], [6.2603, 0.8849, 0.7993, 0.1647, -6.1092, 0.0027000001, 0.8313, 2.2758, 5.1146]] })
    a.append({'Z': 5 ,'charge_list': [0.0, 3] ,'coefficient_list': [[2.0545, 1.3326, 1.0979, 0.7068, -0.1932, 23.2185, 1.021, 60.3498, 0.1403], [0.136717, 0.860415, -0.000445298, 0.928984, 0.0743566, 3.50974, 1.39902, 78.1147, 0.460361]] })
    a.append({'Z': 6 ,'charge_list': [0.0] ,'coefficient_list': [[2.31, 1.02, 1.5886, 0.865, 0.2156, 20.8439, 10.2075, 0.5687, 51.6512]] })
    a.append({'Z': 7 ,'charge_list': [0.0] ,'coefficient_list': [[12.2126, 3.1322, 2.0125, 1.1663, -11.529, 0.0057000001, 9.8933, 28.9975, 0.5826]] })
    a.append({'Z': 8 ,'charge_list': [0.0, -1] ,'coefficient_list': [[3.0485, 2.2868, 1.5463, 0.867, 0.2508, 13.2771, 5.7011, 0.3239, 32.9089], [4.1916, 1.63969, 1.52673, -20.307, 21.9412, 12.8573, 4.17236, 47.0179, -0.01404]] })
    a.append({'Z': 9 ,'charge_list': [0.0, -1] ,'coefficient_list': [[3.5392, 2.6412, 1.517, 1.0243, 0.2776, 10.2825, 4.2944, 0.2615, 26.1476], [3.6322, 3.51057, 1.26064, 0.940706, 0.653396, 5.27756, 14.7353, 0.442258, 47.3437]] })
    a.append({'Z': 10 ,'charge_list': [0.0] ,'coefficient_list': [[3.9553, 3.1125, 1.4546, 1.1251, 0.3515, 8.4042, 3.4262, 0.2306, 21.7184]] })
    a.append({'Z': 11 ,'charge_list': [0.0, 1] ,'coefficient_list': [[4.7626, 3.1736, 1.2674, 1.1128, 0.676, 3.285, 8.8422, 0.3136, 129.424], [3.2565, 3.9362, 1.3998, 1.0032, 0.404, 2.6671, 6.1153, 0.2001, 14.039]] })
    a.append({'Z': 12 ,'charge_list': [0.0, 2] ,'coefficient_list': [[5.4204, 2.1735, 1.2269, 2.3073, 0.8584, 2.8275, 79.2611, 0.3808, 7.1937], [3.4988, 3.8378, 1.3284, 0.8497, 0.4853, 2.1676, 4.7542, 0.185, 10.1411]] })
    a.append({'Z': 13 ,'charge_list': [0.0, 3] ,'coefficient_list': [[6.4202, 1.9002, 1.5936, 1.9646, 1.1151, 3.0387, 0.7426, 31.5472, 85.0886], [4.17448, 3.3876, 1.20296, 0.528137, 0.706786, 1.93816, 4.14553, 0.228753, 8.28524]] })
    a.append({'Z': 14 ,'charge_list': [0.0, 4] ,'coefficient_list': [[6.2915, 3.0353, 1.9891, 1.541, 1.1407, 2.4386, 32.3337, 0.6785, 81.6937], [4.43918, 3.20345, 1.19453, 0.41653, 0.746297, 1.64167, 3.43757, 0.2149, 6.65365]] })
    a.append({'Z': 15 ,'charge_list': [0.0] ,'coefficient_list': [[6.4345, 4.1791, 1.78, 1.4908, 1.1149, 1.9067, 27.157, 0.526, 68.1645]] })
    a.append({'Z': 16 ,'charge_list': [0.0] ,'coefficient_list': [[6.9053, 5.2034, 1.4379, 1.5863, 0.8669, 1.4679, 22.2151, 0.2536, 56.172]] })
    a.append({'Z': 17 ,'charge_list': [0.0, -1] ,'coefficient_list': [[11.4604, 7.1964, 6.2556, 1.6455, -9.5574, 0.0104, 1.1662, 18.5194, 47.7784], [18.2915, 7.2084, 6.5337, 2.3386, -16.378, 0.0066, 1.1717, 19.5424, 60.4486]] })
    a.append({'Z': 18 ,'charge_list': [0.0] ,'coefficient_list': [[7.4845, 6.7723, 0.6539, 1.6442, 1.4445, 0.9072, 14.8407, 43.8983, 33.3929]] })
    a.append({'Z': 19 ,'charge_list': [0.0, 1] ,'coefficient_list': [[8.2186, 7.4398, 1.0519, 0.8659, 1.4228, 12.7949, 0.7748, 213.187, 41.6841], [7.9578, 7.4917, 6.359, 1.1915, -4.9978, 12.6331, 0.7674, -0.0020000001, 31.9128]] })
    a.append({'Z': 20 ,'charge_list': [0.0, 2] ,'coefficient_list': [[8.6266, 7.3873, 1.5899, 1.0211, 1.3751, 10.4421, 0.6599, 85.7484, 178.437], [15.6348, 7.9518, 8.4372, 0.8537, -14.875, -0.0074, 0.6089, 10.3116, 25.9905]] })
    a.append({'Z': 21 ,'charge_list': [0.0, 3] ,'coefficient_list': [[9.189, 7.3679, 1.6409, 1.468, 1.3329, 9.0213, 0.5729, 136.108, 51.3531], [13.4008, 8.0273, 1.65943, 1.57936, -6.6667, 0.29854, 7.9629, -0.28604, 16.0662]] })
    a.append({'Z': 22 ,'charge_list': [0.0, 2, 3, 4] ,'coefficient_list': [[9.7595, 7.3558, 1.6991, 1.9021, 1.2807, 7.8508, 0.5, 35.6338, 116.105], [9.11423, 7.62174, 2.2793, 0.087898999, 0.897155, 7.5243, 0.457585, 19.5361, 61.6558], [17.7344, 8.73816, 5.25691, 1.92134, -14.652, 0.22061, 7.04716, -0.15762, 15.9768], [19.5114, 8.23473, 2.01341, 1.5208, -13.28, 0.178847, 6.67018, -0.29263, 12.9464]] })
    a.append({'Z': 23 ,'charge_list': [0.0, 2, 3, 5] ,'coefficient_list': [[10.2971, 7.3511, 2.0703, 2.0571, 1.2199, 6.8657, 0.4385, 26.8938, 102.478], [10.106, 7.3541, 2.2884, 0.022299999, 1.2298, 6.8818, 0.4409, 20.3004, 115.122], [9.43141, 7.7419, 2.15343, 0.016865, 0.656565, 6.39535, 0.383349, 15.1908, 63.969], [15.6887, 8.14208, 2.03081, -9.576, 1.7143, 0.679003, 5.40135, 9.97278, 0.940464]] })
    a.append({'Z': 24 ,'charge_list': [0.0, 2, 3] ,'coefficient_list': [[10.6406, 7.3537, 3.324, 1.4922, 1.1832, 6.1038, 0.392, 20.2626, 98.7399], [9.54034, 7.7509, 3.58274, 0.509107, 0.616898, 5.66078, 0.344261, 13.3075, 32.4224], [9.6809, 7.81136, 2.87603, 0.113575, 0.518275, 5.59463, 0.334393, 12.8288, 32.8761]] })
    a.append({'Z': 25 ,'charge_list': [0.0, 2, 3, 4] ,'coefficient_list': [[11.2819, 7.3573, 3.0193, 2.2441, 1.0896, 5.3409, 0.3432, 17.8674, 83.7543], [10.8061, 7.362, 3.5268, 0.2184, 1.0874, 5.2796, 0.3435, 14.343, 41.3235], [9.84521, 7.87194, 3.56531, 0.323613, 0.393974, 4.91797, 0.294393, 10.8171, 24.1281], [9.96253, 7.97057, 2.76067, 0.054446999, 0.251877, 4.8485, 0.283303, 10.4852, 27.573]] })
    a.append({'Z': 26 ,'charge_list': [0.0, 3] ,'coefficient_list': [[11.7695, 7.3573, 3.5222, 2.3045, 1.0369, 4.7611, 0.3072, 15.3535, 76.8805], [11.1764, 7.3863, 3.3948, 0.072400004, 0.9707, 4.6147, 0.3005, 11.6729, 38.5566]] })
    a.append({'Z': 27 ,'charge_list': [0.0, 2, 3] ,'coefficient_list': [[12.2841, 7.3409, 4.0034, 2.3488, 1.0118, 4.2791, 0.2784, 13.5359, 71.1692], [11.2296, 7.3883, 4.7393, 0.7108, 0.9324, 4.1231, 0.2726, 10.2443, 25.6466], [10.338, 7.88173, 4.76795, 0.725591, 0.286667, 3.90969, 0.238668, 8.35583, 18.3491]] })
    a.append({'Z': 28 ,'charge_list': [0.0, 2, 3] ,'coefficient_list': [[12.8376, 7.292, 4.4438, 2.38, 1.0341, 3.8785, 0.2565, 12.1763, 66.3421], [11.4166, 7.4005, 5.3442, 0.9773, 0.8614, 3.6766, 0.2449, 8.873, 22.1626], [10.7806, 7.75868, 5.22746, 0.847114, 0.386044, 3.5477, 0.22314, 7.64468, 16.9673]] })
    a.append({'Z': 29 ,'charge_list': [0.0, 1, 2] ,'coefficient_list': [[13.338, 7.1676, 5.6158, 1.6735, 1.191, 3.5828, 0.247, 11.3966, 64.8126], [11.9475, 7.3573, 6.2455, 1.5578, 0.89, 3.3669, 0.2274, 8.6625, 25.8487], [11.8168, 7.11181, 5.78135, 1.14523, 1.14431, 3.37484, 0.244078, 7.9876, 19.897]] })
    a.append({'Z': 30 ,'charge_list': [0.0, 2] ,'coefficient_list': [[14.0743, 7.0318, 5.1652, 2.41, 1.3041, 3.2655, 0.2333, 10.3163, 58.7097], [11.9719, 7.3862, 6.4668, 1.394, 0.7807, 2.9946, 0.2031, 7.0826, 18.0995]] })
    a.append({'Z': 31 ,'charge_list': [0.0, 3] ,'coefficient_list': [[15.2354, 6.7006, 4.3591, 2.9623, 1.7189, 3.0669, 0.2412, 10.7805, 61.4135], [12.692, 6.69883, 6.06692, 1.0066, 1.53545, 2.81262, 0.22789, 6.36441, 14.4122]] })
    a.append({'Z': 32 ,'charge_list': [0.0, 4] ,'coefficient_list': [[16.0816, 6.3747, 3.7068, 3.683, 2.1313, 2.8509, 0.2516, 11.4468, 54.7625], [12.9172, 6.70003, 6.06791, 0.859041, 1.45572, 2.53718, 0.205855, 5.47913, 11.603]] })
    a.append({'Z': 33 ,'charge_list': [0.0] ,'coefficient_list': [[16.6723, 6.0701, 3.4313, 4.2779, 2.531, 2.6345, 0.2647, 12.9479, 47.7972]] })
    a.append({'Z': 34 ,'charge_list': [0.0] ,'coefficient_list': [[17.0006, 5.8196, 3.9731, 4.3543, 2.8409, 2.4098, 0.2726, 15.2372, 43.8163]] })
    a.append({'Z': 35 ,'charge_list': [0.0, -1] ,'coefficient_list': [[17.1789, 5.2358, 5.6377, 3.9851, 2.9557, 2.1723, 16.5796, 0.2609, 41.4328], [17.1718, 6.3338, 5.5754, 3.7272, 3.1776, 2.2059, 19.3345, 0.2871, 58.1535]] })
    a.append({'Z': 36 ,'charge_list': [0.0] ,'coefficient_list': [[17.3555, 6.7286, 5.5493, 3.5375, 2.825, 1.9384, 16.5623, 0.2261, 39.3972]] })
    a.append({'Z': 37 ,'charge_list': [0.0, 1] ,'coefficient_list': [[17.1784, 9.6435, 5.1399, 1.5292, 3.4873, 1.7888, 17.3151, 0.2748, 164.934], [17.5816, 7.6598, 5.8981, 2.7817, 2.0782, 1.7139, 14.7957, 0.1603, 31.2087]] })
    a.append({'Z': 38 ,'charge_list': [0.0, 2] ,'coefficient_list': [[17.5663, 9.8184, 5.422, 2.6694, 2.5064, 1.5564, 14.0988, 0.1664, 132.376], [18.0874, 8.1373, 2.5654, -34.193, 41.4025, 1.4907, 12.6963, 24.5651, -0.0138]] })
    a.append({'Z': 39 ,'charge_list': [0.0, 3] ,'coefficient_list': [[17.776, 10.2946, 5.72629, 3.26588, 1.91213, 1.4029, 12.8006, 0.125599, 104.354], [17.9268, 9.1531, 1.76795, -33.108, 40.2602, 1.35417, 11.2145, 22.6599, -0.01319]] })
    a.append({'Z': 40 ,'charge_list': [0.0, 4] ,'coefficient_list': [[17.8765, 10.948, 5.41732, 3.65721, 2.06929, 1.27618, 11.916, 0.117622, 87.6627], [18.1668, 10.0562, 1.01118, -2.6479, 9.41454, 1.2148, 10.1483, 21.6054, -0.10276]] })
    a.append({'Z': 41 ,'charge_list': [0.0, 3, 5] ,'coefficient_list': [[17.6142, 12.0144, 4.04183, 3.53346, 3.75591, 1.18865, 11.766, 0.204785, 69.7957], [19.8812, 18.0653, 11.0177, 1.94715, -12.912, 0.019175, 1.13305, 10.1621, 28.3389], [17.9163, 13.3417, 10.799, 0.337905, -6.3934, 1.12446, 0.028781001, 9.28206, 25.7228]] })
    a.append({'Z': 42 ,'charge_list': [0.0, 3, 5, 6] ,'coefficient_list': [[3.7025, 17.2356, 12.8876, 3.7429, 4.3875, 0.2772, 1.0958, 11.004, 61.6584], [21.1664, 18.2017, 11.7423, 2.30951, -14.421, 0.014734, 1.03031, 9.53659, 26.6307], [21.0149, 18.0992, 11.4632, 0.740625, -14.316, 0.014345, 1.02238, 8.78809, 23.3452], [17.8871, 11.175, 6.57891, 0.0, 0.344941, 1.03649, 8.48061, 0.058881, 0.0]] })
    a.append({'Z': 43 ,'charge_list': [0.0] ,'coefficient_list': [[19.1301, 11.0948, 4.64901, 2.71263, 5.40428, 0.864132, 8.14487, 21.5707, 86.8472]] })
    a.append({'Z': 44 ,'charge_list': [0.0, 3] ,'coefficient_list': [[19.2674, 12.9182, 4.86337, 1.56756, 5.37874, 0.80852, 8.43467, 24.7997, 94.2928], [18.5638, 13.2885, 9.32602, 3.00964, -3.1892, 0.847329, 8.37164, 0.017662, 22.887]] })
    a.append({'Z': 45 ,'charge_list': [0.0, 3, 4] ,'coefficient_list': [[19.2957, 14.3501, 4.73425, 1.28918, 5.328, 0.751536, 8.21758, 25.8749, 98.6062], [18.8785, 14.1259, 3.32515, -6.1989, 11.8678, 0.764252, 7.84438, 21.2487, -0.01036], [18.8545, 13.9806, 2.53464, -5.6526, 11.2835, 0.760825, 7.62436, 19.3317, -0.0102]] })
    a.append({'Z': 46 ,'charge_list': [0.0, 2, 4] ,'coefficient_list': [[19.3319, 15.5017, 5.29537, 0.605844, 5.26593, 0.698655, 7.98929, 25.2052, 76.8986], [19.1701, 15.2096, 4.32234, 0.0, 5.2916, 0.696219, 7.55573, 22.5057, 0.0], [19.2493, 14.79, 2.89289, -7.9492, 13.0174, 0.683839, 7.14833, 17.9144, 0.0051270002]] })
    a.append({'Z': 47 ,'charge_list': [0.0, 1, 2] ,'coefficient_list': [[19.2808, 16.6885, 4.8045, 1.0463, 5.179, 0.6446, 7.4726, 24.6605, 99.8156], [19.1812, 15.9719, 5.27475, 0.357534, 5.21572, 0.646179, 7.19123, 21.7326, 66.1147], [19.1643, 16.2456, 4.3709, 0.0, 5.21404, 0.645643, 7.18544, 21.4072, 0.0]] })
    a.append({'Z': 48 ,'charge_list': [0.0, 2] ,'coefficient_list': [[19.2214, 17.6444, 4.461, 1.6029, 5.0694, 0.5946, 6.9089, 24.7008, 87.4825], [19.1514, 17.2535, 4.47128, 0.0, 5.11937, 0.597922, 6.80639, 20.2521, 0.0]] })
    a.append({'Z': 49 ,'charge_list': [0.0, 3] ,'coefficient_list': [[19.1624, 18.5596, 4.2948, 2.0396, 4.9391, 0.5476, 6.3776, 25.8499, 92.8029], [19.1045, 18.1108, 3.78897, 0.0, 4.99635, 0.551522, 6.3247, 17.3595, 0.0]] })
    a.append({'Z': 50 ,'charge_list': [0.0, 2, 4] ,'coefficient_list': [[19.1889, 19.1005, 4.4585, 2.4663, 4.7821, 5.8303, 0.5031, 26.8909, 83.9571], [19.1094, 19.0548, 4.5648, 0.487, 4.7861, 0.5036, 5.8378, 23.3752, 62.2061], [18.9333, 19.7131, 3.4182, 0.019300001, 3.9182, 5.764, 0.4655, 14.0049, -0.7583]] })
    a.append({'Z': 51 ,'charge_list': [0.0, 3, 5] ,'coefficient_list': [[19.6418, 19.0455, 5.0371, 2.6827, 4.5909, 5.3034, 0.4607, 27.9074, 75.2825], [18.9755, 18.933, 5.10789, 0.288753, 4.69626, 0.467196, 5.22126, 19.5902, 55.5113], [19.8685, 19.0302, 2.41253, 0.0, 4.69263, 5.44853, 0.467973, 14.1259, 0.0]] })
    a.append({'Z': 52 ,'charge_list': [0.0] ,'coefficient_list': [[19.9644, 19.0138, 6.14487, 2.5239, 4.352, 4.81742, 0.420885, 28.5284, 70.8403]] })
    a.append({'Z': 53 ,'charge_list': [0.0, -1] ,'coefficient_list': [[20.1472, 18.9949, 7.5138, 2.2735, 4.0712, 4.347, 0.3814, 27.766, 66.8776], [20.2332, 18.997, 7.8069, 2.8868, 4.0714, 4.3579, 0.3815, 29.5259, 84.9304]] })
    a.append({'Z': 54 ,'charge_list': [0.0] ,'coefficient_list': [[20.2933, 19.0298, 8.9767, 1.99, 3.7118, 3.9282, 0.344, 26.4659, 64.2658]] })
    a.append({'Z': 55 ,'charge_list': [0.0, 1] ,'coefficient_list': [[20.3892, 19.1062, 10.662, 1.4953, 3.3352, 3.569, 0.3107, 24.3879, 213.904], [20.3524, 19.1278, 10.2821, 0.9615, 3.2791, 3.552, 0.3086, 23.7128, 59.4565]] })
    a.append({'Z': 56 ,'charge_list': [0.0, 2] ,'coefficient_list': [[20.3361, 19.297, 10.888, 2.6959, 2.7731, 3.216, 0.2756, 20.2073, 167.202], [20.1807, 19.1136, 10.9054, 0.773634, 3.02902, 3.21367, 0.28331, 20.0558, 51.746]] })
    a.append({'Z': 57 ,'charge_list': [0.0, 3] ,'coefficient_list': [[20.578, 19.599, 11.3727, 3.28719, 2.14678, 2.94817, 0.244475, 18.7726, 133.124], [20.2489, 19.3763, 11.6323, 0.336048, 2.4086, 2.9207, 0.250698, 17.8211, 54.9453]] })
    a.append({'Z': 58 ,'charge_list': [0.0, 3, 4] ,'coefficient_list': [[21.1671, 19.7695, 11.8513, 3.33049, 1.86264, 2.81219, 0.226836, 17.6083, 127.113], [20.8036, 19.559, 11.9369, 0.612376, 2.09013, 2.77691, 0.23154, 16.5408, 43.1692], [20.3235, 19.8186, 12.1233, 0.144583, 1.5918, 2.65941, 0.21885, 15.7992, 62.2355]] })
    a.append({'Z': 59 ,'charge_list': [0.0, 3, 4] ,'coefficient_list': [[22.044, 19.6697, 12.3856, 2.82428, 2.0583, 2.77393, 0.222087, 16.7669, 143.644], [21.3727, 19.7491, 12.1329, 0.97518, 1.77132, 2.6452, 0.214299, 15.323, 36.4065], [20.9413, 20.0539, 12.4668, 0.296689, 1.24285, 2.54467, 0.202481, 14.8137, 45.4643]] })
    a.append({'Z': 60 ,'charge_list': [0.0, 3] ,'coefficient_list': [[22.6845, 19.6847, 12.774, 2.85137, 1.98486, 2.66248, 0.210628, 15.885, 137.903], [21.961, 19.9339, 12.12, 1.51031, 1.47588, 2.52722, 0.199237, 14.1783, 30.8717]] })
    a.append({'Z': 61 ,'charge_list': [0.0, 3] ,'coefficient_list': [[23.3405, 19.6095, 13.1235, 2.87516, 2.02876, 2.5627, 0.202088, 15.1009, 132.721], [22.5527, 20.1108, 12.0671, 2.07492, 1.19499, 2.4174, 0.185769, 13.1275, 27.4491]] })
    a.append({'Z': 62 ,'charge_list': [0.0, 3] ,'coefficient_list': [[24.0042, 19.4258, 13.4396, 2.89604, 2.20963, 2.47274, 0.196451, 14.3996, 128.007], [23.1504, 20.2599, 11.9202, 2.71488, 0.954586, 2.31641, 0.174081, 12.1571, 24.8242]] })
    a.append({'Z': 63 ,'charge_list': [0.0, 2, 3] ,'coefficient_list': [[24.6274, 19.0886, 13.7603, 2.9227, 2.5745, 2.3879, 0.1942, 13.7546, 123.174], [24.0063, 19.9504, 11.8034, 3.87243, 1.36389, 2.27783, 0.17353, 11.6096, 26.5156], [23.7497, 20.3745, 11.8509, 3.26503, 0.759344, 2.22258, 0.16394, 11.311, 22.9966]] })
    a.append({'Z': 64 ,'charge_list': [0.0, 3] ,'coefficient_list': [[25.0709, 19.0798, 13.8518, 3.54545, 2.4196, 2.25341, 0.181951, 12.9331, 101.398], [24.3466, 20.4208, 11.8708, 3.7149, 0.645089, 2.13553, 0.155525, 10.5782, 21.7029]] })
    a.append({'Z': 65 ,'charge_list': [0.0, 3] ,'coefficient_list': [[25.8976, 18.2185, 14.3167, 2.95354, 3.58324, 2.24256, 0.196143, 12.6648, 115.362], [24.9559, 20.3271, 12.2471, 3.773, 0.691967, 2.05601, 0.149525, 10.0499, 21.2773]] })
    a.append({'Z': 66 ,'charge_list': [0.0, 3] ,'coefficient_list': [[26.507, 17.6383, 14.5596, 2.96577, 4.29728, 2.1802, 0.202172, 12.1899, 111.874], [25.5395, 20.2861, 11.9812, 4.50073, 0.68969, 1.9804, 0.143384, 9.34972, 19.581]] })
    a.append({'Z': 67 ,'charge_list': [0.0, 3] ,'coefficient_list': [[26.9049, 17.294, 14.5583, 3.63837, 4.56796, 2.07051, 0.19794, 11.4407, 92.6566], [26.1296, 20.0994, 11.9788, 4.93676, 0.852795, 1.91072, 0.139358, 8.80018, 18.5908]] })
    a.append({'Z': 68 ,'charge_list': [0.0, 3] ,'coefficient_list': [[27.6563, 16.4285, 14.9779, 2.98233, 5.92046, 2.07356, 0.223545, 11.3604, 105.703], [26.722, 19.7748, 12.1506, 5.17379, 1.17613, 1.84659, 0.13729, 8.36225, 17.8974]] })
    a.append({'Z': 69 ,'charge_list': [0.0, 3] ,'coefficient_list': [[28.1819, 15.8851, 15.1542, 2.98706, 6.75621, 2.02859, 0.238849, 10.9975, 102.961], [27.3083, 19.332, 12.3339, 5.38348, 1.63929, 1.78711, 0.136974, 7.96778, 17.2922]] })
    a.append({'Z': 70 ,'charge_list': [0.0, 2, 3] ,'coefficient_list': [[28.6641, 15.4345, 15.3087, 2.98963, 7.56672, 1.9889, 0.257119, 10.6647, 100.417], [28.1209, 17.6817, 13.3335, 5.14657, 3.70983, 1.78503, 0.15997, 8.18304, 20.39], [27.8917, 18.7614, 12.6072, 5.47647, 2.26001, 1.73272, 0.13879, 7.64412, 16.8153]] })
    a.append({'Z': 71 ,'charge_list': [0.0, 3] ,'coefficient_list': [[28.9476, 15.2208, 15.1, 3.71601, 7.97628, 1.90182, 9.98519, 0.261033, 84.3298], [28.4628, 18.121, 12.8429, 5.59415, 2.97573, 1.68216, 0.142292, 7.33727, 16.3535]] })
    a.append({'Z': 72 ,'charge_list': [0.0, 4] ,'coefficient_list': [[29.144, 15.1726, 14.7586, 4.30013, 8.58154, 1.83262, 9.5999, 0.275116, 72.029], [28.8131, 18.4601, 12.7285, 5.59927, 2.39699, 1.59136, 0.128903, 6.76232, 14.0366]] })
    a.append({'Z': 73 ,'charge_list': [0.0, 5] ,'coefficient_list': [[29.2024, 15.2293, 14.5135, 4.76492, 9.24354, 1.77333, 9.37046, 0.295977, 63.3644], [29.1587, 18.8407, 12.8268, 5.38695, 1.78555, 1.50711, 0.116741, 6.31524, 12.4244]] })
    a.append({'Z': 74 ,'charge_list': [0.0, 6] ,'coefficient_list': [[29.0818, 15.43, 14.4327, 5.11982, 9.8875, 1.72029, 9.2259, 0.321703, 57.056], [29.4936, 19.3763, 13.0544, 5.06412, 1.01074, 1.42755, 0.104621, 5.93667, 11.1972]] })
    a.append({'Z': 75 ,'charge_list': [0.0] ,'coefficient_list': [[28.7621, 15.7189, 14.5564, 5.44174, 10.472, 1.67191, 9.09227, 0.3505, 52.0861]] })
    a.append({'Z': 76 ,'charge_list': [0.0, 4] ,'coefficient_list': [[28.1894, 16.155, 14.9305, 5.67589, 11.0005, 1.62903, 8.97948, 0.382661, 48.1647], [30.419, 15.2637, 14.7458, 5.06795, 6.49804, 1.37113, 6.84706, 0.165191, 18.003]] })
    a.append({'Z': 77 ,'charge_list': [0.0, 3, 4] ,'coefficient_list': [[27.3049, 16.7296, 15.6115, 5.83377, 11.4722, 1.59279, 8.86553, 0.417916, 45.0011], [30.4156, 15.862, 13.6145, 5.82008, 8.27903, 1.34323, 7.10909, 0.204633, 20.3254], [30.7058, 15.5512, 14.2326, 5.53672, 6.96824, 1.30923, 6.71983, 0.167252, 17.4911]] })
    a.append({'Z': 78 ,'charge_list': [0.0, 2, 4] ,'coefficient_list': [[27.0059, 17.7639, 15.7131, 5.7837, 11.6883, 1.51293, 8.81174, 0.424593, 38.6103], [29.8429, 16.7224, 13.2153, 6.35234, 9.85329, 1.32927, 7.38979, 0.263297, 22.9426], [30.9612, 15.9829, 13.7348, 5.92034, 7.39534, 1.24813, 6.60834, 0.16864, 16.9392]] })
    a.append({'Z': 79 ,'charge_list': [0.0, 1, 3] ,'coefficient_list': [[16.8819, 18.5913, 25.5582, 5.86, 12.0658, 0.4611, 8.6216, 1.4826, 36.3956], [28.0109, 17.8204, 14.3359, 6.58077, 11.2299, 1.35321, 7.7395, 0.356752, 26.4043], [30.6886, 16.9029, 12.7801, 6.52354, 9.0968, 1.2199, 6.82872, 0.212867, 18.659]] })
    a.append({'Z': 80 ,'charge_list': [0.0, 1, 2] ,'coefficient_list': [[20.6809, 19.0417, 21.6575, 5.9676, 12.6089, 0.545, 8.4484, 1.5729, 38.3246], [25.0853, 18.4973, 16.8883, 6.48216, 12.0205, 1.39507, 7.65105, 0.443378, 28.2262], [29.5641, 18.06, 12.8374, 6.89912, 10.6268, 1.21152, 7.05639, 0.284738, 20.7482]] })
    a.append({'Z': 81 ,'charge_list': [0.0, 1, 3] ,'coefficient_list': [[27.5446, 19.1584, 15.538, 5.52593, 13.1746, 0.65515, 8.70751, 1.96347, 45.8149], [21.3985, 20.4723, 18.7478, 6.82847, 12.5258, 1.4711, 0.517394, 7.43463, 28.8482], [30.8695, 18.3841, 11.9328, 7.00574, 9.8027, 1.1008, 6.53852, 0.219074, 17.2114]] })
    a.append({'Z': 82 ,'charge_list': [0.0, 2, 4] ,'coefficient_list': [[31.0617, 13.0637, 18.442, 5.9696, 13.4118, 0.6902, 2.3576, 8.618, 47.2579], [21.7886, 19.5682, 19.1406, 7.01107, 12.4734, 1.3366, 0.488383, 6.7727, 23.8132], [32.1244, 18.8003, 12.0175, 6.96886, 8.08428, 1.00566, 6.10926, 0.147041, 14.714]] })
    a.append({'Z': 83 ,'charge_list': [0.0, 3, 5] ,'coefficient_list': [[33.3689, 12.951, 16.5877, 6.4692, 13.5782, 0.704, 2.9238, 8.7937, 48.0093], [21.8053, 19.5026, 19.1053, 7.10295, 12.4711, 1.2356, 6.24149, 0.469999, 20.3185], [33.5364, 25.0946, 19.2497, 6.91555, -6.7994, 0.91654, 0.039042, 5.71414, 12.8285]] })
    a.append({'Z': 84 ,'charge_list': [0.0] ,'coefficient_list': [[34.6726, 15.4733, 13.1138, 7.02588, 13.677, 0.700999, 3.55078, 9.55642, 47.0045]] })
    a.append({'Z': 85 ,'charge_list': [0.0] ,'coefficient_list': [[35.3163, 19.0211, 9.49887, 7.42518, 13.7108, 0.68587, 3.97458, 11.3824, 45.4715]] })
    a.append({'Z': 86 ,'charge_list': [0.0] ,'coefficient_list': [[35.5631, 21.2816, 8.0037, 7.4433, 13.6905, 0.6631, 4.0691, 14.0422, 44.2473]] })
    a.append({'Z': 87 ,'charge_list': [0.0] ,'coefficient_list': [[35.9299, 23.0547, 12.1439, 2.11253, 13.7247, 0.646453, 4.17619, 23.1052, 150.645]] })
    a.append({'Z': 88 ,'charge_list': [0.0, 2] ,'coefficient_list': [[35.763, 22.9064, 12.4739, 3.21097, 13.6211, 0.616341, 3.87135, 19.9887, 142.325], [35.215, 21.67, 7.91342, 7.65078, 13.5431, 0.604909, 3.5767, 12.601, 29.8436]] })
    a.append({'Z': 89 ,'charge_list': [0.0, 3] ,'coefficient_list': [[35.6597, 23.1032, 12.5977, 4.08655, 13.5266, 0.589092, 3.65155, 18.599, 117.02], [35.1736, 22.1112, 8.19216, 7.05545, 13.4637, 0.579689, 3.41437, 12.9187, 25.9443]] })
    a.append({'Z': 90 ,'charge_list': [0.0, 4] ,'coefficient_list': [[35.5645, 23.4219, 12.7473, 4.80703, 13.4314, 0.563359, 3.46204, 17.8309, 99.1722], [35.1007, 22.4418, 9.78554, 5.29444, 13.376, 0.555054, 3.24498, 13.4661, 23.9533]] })
    a.append({'Z': 91 ,'charge_list': [0.0] ,'coefficient_list': [[35.8847, 23.2948, 14.1891, 4.17287, 13.4287, 0.547751, 3.41519, 16.9235, 105.251]] })
    a.append({'Z': 92 ,'charge_list': [0.0, 3, 4, 6] ,'coefficient_list': [[36.0228, 23.4128, 14.9491, 4.188, 13.3966, 0.5293, 3.3253, 16.0927, 100.613], [35.5747, 22.5259, 12.2165, 5.37073, 13.3092, 0.52048, 3.12293, 12.7148, 26.3394], [35.3715, 22.5326, 12.0291, 4.7984, 13.2671, 0.516598, 3.05053, 12.5723, 23.4582], [34.8509, 22.7584, 14.0099, 1.21457, 13.1665, 0.507079, 2.8903, 13.1767, 25.2017]] })
    a.append({'Z': 93 ,'charge_list': [0.0, 3, 4, 6] ,'coefficient_list': [[36.1874, 23.5964, 15.6402, 4.1855, 13.3573, 0.511929, 3.25396, 15.3622, 97.4908], [35.0136, 22.7286, 14.3884, 1.75669, 13.113, 0.48981, 2.81099, 12.33, 22.6581], [36.5254, 23.8083, 16.7707, 3.47947, 13.3812, 0.499384, 3.26371, 14.9455, 105.98], [35.7074, 22.613, 12.9898, 5.43227, 13.2544, 0.502322, 3.03807, 12.1449, 25.4928]] })
    a.append({'Z': 94 ,'charge_list': [0.0, 3, 4, 6] ,'coefficient_list': [[35.5103, 22.5787, 12.7766, 4.92159, 13.2116, 0.498626, 2.96627, 11.9484, 22.7502], [35.84, 22.7169, 13.5807, 5.66016, 13.1991, 0.484938, 2.96118, 11.5331, 24.3992], [35.6493, 22.646, 13.3595, 5.18831, 13.1555, 0.481422, 2.8902, 11.316, 21.8301], [35.1736, 22.7181, 14.7635, 2.28678, 13.0582, 0.473204, 2.73848, 11.553, 20.9303]] })
    a.append({'Z': 95 ,'charge_list': [0.0] ,'coefficient_list': [[36.6706, 24.0992, 17.3415, 3.49331, 13.3592, 0.483629, 3.20647, 14.3136, 102.273]] })
    a.append({'Z': 96 ,'charge_list': [0.0] ,'coefficient_list': [[36.6488, 24.4096, 17.399, 4.21665, 13.2887, 0.465154, 3.08997, 13.4346, 88.4834]] })
    a.append({'Z': 97 ,'charge_list': [0.0] ,'coefficient_list': [[36.7881, 24.7736, 17.8919, 4.23284, 13.2754, 0.451018, 3.04619, 12.8946, 86.003]] })
    a.append({'Z': 98 ,'charge_list': [0.0] ,'coefficient_list': [[36.9185, 25.1995, 18.3317, 4.24391, 13.2674, 0.437533, 3.00775, 12.4044, 83.7881]] })

    if Z != a[Z-1]['Z']:
        raise Exception("Mismatch of indices!")
    return a[Z-1]

def f0_xop_with_fractional_charge(Z, charge=0.0):
    """

    """
    if charge == 0.0: return f0_xop(Z)
    f0_data = f0_xop_with_fractional_charge_data(Z)
    return f0_interpolate_coefficients(charge,
                                       f0_data['charge_list'],
                                       f0_data['coefficient_list']
                                       )


#=======================================================================================================================
#  CRYSTAL UTILITIES
#=======================================================================================================================

def bragg_metrictensor(a,b,c,a1,a2,a3,RETURN_REAL_SPACE=0,RETURN_VOLUME=0,HKL=None):
    """
    Returns the metric tensor in the reciprocal space

    :param a: unit cell a
    :param b: unit cell b
    :param c: unit cell c
    :param a1: unit cell alpha
    :param a2: unit cell beta
    :param a3: unit cell gamma
    :param RETURN_REAL_SPACE: set to 1 for returning metric tensor in real space
    :param RETURN_VOLUME: set to 1 to return the unit cell volume in Angstroms^3
    :param HKL: if !=None, returns the d-spacing for the corresponding [H,K,L] reflection
    :return: the returned value depends on the keywords used. If RETURN_REAL_SPACE=0,RETURN_VOLUME=0, and HKL=None
             then retuns the metric tensor in reciprocal space.
    """
    # input cell a,b,c,alpha,beta,gamma; angles in degrees
    a1 *= numpy.pi / 180.0
    a2 *= numpy.pi / 180.0
    a3 *= numpy.pi / 180.0
    # ;
    # ; tensor in real space
    # ;
    g = numpy.array( [ [a*a, a*b*numpy.cos(a3), a*c*numpy.cos(a2)], \
          [a*b*numpy.cos(a3), b*b, b*c*numpy.cos(a1)], \
          [a*c*numpy.cos(a2), b*c*numpy.cos(a1), c*c]] )

    if RETURN_REAL_SPACE: return g
    # print("g: ",g)

    # ;
    # ; volume of the lattice
    # ;
    volume2 = numpy.linalg.det(g)
    volume = numpy.sqrt(volume2)

    # print("Volume of unit cell: %g A^3",volume)

    if RETURN_VOLUME: return volume

    # ;
    # ; tensor in reciprocal space
    # ;
    ginv = numpy.linalg.inv(g)
    # ;print,gInv
    #

    # itmp = where(abs(ginv) LT 1d-8)
    # IF itmp[0] NE -1 THEN ginv[itmp]=0D

    itmp = numpy.where(numpy.abs(ginv) < 1e-8)
    ginv[itmp] = 0.0

    # print("ginv: ",ginv)

    if HKL != None:
    #   ; computes d-spacing
        dd = numpy.dot( numpy.array(HKL) , numpy.dot( ginv , numpy.array(HKL)))
        #
        # print("DD: ", dd)
        dd1 = 1.0 / numpy.sqrt(dd)
        # print("D-spacing: ",dd1)
        return dd1
    else:
        return ginv

def lorentz(theta_bragg_deg,return_what=0):
    """
    This function returns the Lorentz factor, polarization factor (unpolarized beam), geometric factor,
    or a combination of them.

    :param theta_bragg_deg: Bragg angle in degrees
    :param return_what: A flag indicating the returned variable:
                        0: (default) PolFac*lorentzFac
                        1: PolFac
                        2: lorentzFac
                        3: geomFac
    :return: a scalar value
    """
    tr = theta_bragg_deg * numpy.pi / 180.
    polarization_factor = 0.5 * (1.0 + (numpy.cos(2.0 * tr))**2)
    lorentz_factor = 1.0 / numpy.sin(2.0 * tr)
    geometrical_factor = 1.0 * numpy.cos(tr) / numpy.sin(2.0 * tr)

    if return_what == 0:
        return polarization_factor*lorentz_factor
    elif return_what == 1:
        return polarization_factor
    elif return_what == 2:
        return lorentz_factor
    elif return_what == 3:
        return geometrical_factor
    elif return_what == 4:
        return polarization_factor*lorentz_factor*geometrical_factor

#
#
#
#
#


if __name__ == "__main__":

    print(len(f0_xop(14)), f0_xop(14))

    print(len(f0_xop_with_fractional_charge(14,1.5)), f0_xop_with_fractional_charge(14,1.5))