# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/API_Prediction.ipynb (unless otherwise specified).

__all__ = ["Prediction"]

# Cell

from typing import *

# Internal Cell

import os

import pandas as pd
from fastcore.foundation import patch

from ..logger import get_logger, set_level
from ..helper import (
    get_data,
    post_data,
    delete_data,
    add_ready_column,
    generate_df,
    get_attributes_from_instances,
)

from .client import Client
from .progress_status import ProgressStatus

# Internal Cell

logger = get_logger(__name__)

# Cell


class Prediction(ProgressStatus):
    """A class to manage and download the predictions.

    The `Prediction` class is automatically instantiated by calling the `Model.predict` method of a `Model` instance.
    Currently, it is the only way to instantiate this class.
    """

    BASIC_PRED_COLS = ["id", "created", "total_steps", "completed_steps"]
    ALL_PRED_COLS = BASIC_PRED_COLS + ["model_id", "datasource_id", "error"]

    def __init__(
        self,
        id: int,
        datasource_id: Optional[int] = None,
        model_id: Optional[int] = None,
        created: Optional[pd.Timestamp] = None,
        total_steps: Optional[int] = None,
        completed_steps: Optional[int] = None,
        error: Optional[str] = None,
        disabled: Optional[bool] = None,
    ):
        """Constructs a new `Prediction` instance

        Warning:
            Do not construct this object directly by calling the constructor, instead please use
            `Model.predict` method of the Model instance.

        Args:
            id: Prediction ID in the server.
            datasource_id: DataSource ID in the server.
            model_id: Model ID in the server.
            created: Prediction creation date.
            total_steps: No of steps needed to complete the model prediction.
            completed_steps: No of steps completed so far in the model prediction.
            error: Error message while making the prediction.
            disabled: Flag to indicate the status of the prediction.
        """
        self.id = id
        self.datasource_id = datasource_id
        self.model_id = model_id
        self.created = created
        self.total_steps = total_steps
        self.completed_steps = completed_steps
        self.error = error
        self.disabled = disabled
        ProgressStatus.__init__(self, relative_url=f"/prediction/{self.id}")

    @staticmethod
    def ls(
        offset: int = 0,
        limit: int = 100,
        disabled: bool = False,
        completed: bool = False,
    ) -> List["Prediction"]:
        """Return the list of Prediction instances available in the server.

        Args:
            offset: The number of predictions to offset at the beginning. If None, then the default value **0** will be used.
            limit: The maximum number of predictions to return from the server. If None,
                then the default value **100** will be used.
            disabled: If set to **True**, then only the deleted predictions will be returned. Else, the default value
                **False** will be used to return only the list of active predictions.
            completed: If set to **True**, then only the predictions that are successfully processed in server will be returned.
                Else, the default value **False** will be used to return all the predictions.

        Returns:
            A list of Prediction instances available in the server.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to list the available predictions:

        ```python
        Prediction.ls()
        ```
        """
        lists = Client._get_data(
            relative_url=f"/prediction/?disabled={disabled}&completed={completed}&offset={offset}&limit={limit}"
        )

        predx = [
            Prediction(
                id=pred["id"],
                model_id=pred["model_id"],
                datasource_id=pred["datasource_id"],
                created=pred["created"],
                total_steps=pred["total_steps"],
                completed_steps=pred["completed_steps"],
                error=pred["error"],
                disabled=pred["disabled"],
            )
            for pred in lists
        ]

        return predx

    @staticmethod
    def as_df(predx: List["Prediction"]) -> pd.DataFrame:
        """Return the details of prediction instances as a pandas dataframe.

        Args:
            predx: List of prediction instances.

        Returns:
            Details of all the prediction in a dataframe.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example get the details of available prediction:

        ```python
        predictionx = Prediction.ls()
        Prediction.as_df(predictionx)
        ```
        """
        lists = get_attributes_from_instances(predx, Prediction.BASIC_PRED_COLS)  # type: ignore

        df = generate_df(lists, Prediction.BASIC_PRED_COLS)

        return add_ready_column(df)

    def details(self) -> pd.DataFrame:
        raise NotImplementedError()

    def to_pandas(self) -> pd.DataFrame:
        raise NotImplementedError()

    def delete(self) -> pd.DataFrame:
        raise NotImplementedError()


#     def push(self, data_source: "airt.components.datasource.DataSource"): # type: ignore
#         raise NotImplementedError()

# Cell


@patch
def details(self: Prediction) -> pd.DataFrame:
    """Return the details of a prediction.

    Returns:
        A pandas DataFrame encapsulating the details of the prediction.

    Raises:
        ConnectionError: If the server address is invalid or not reachable.

    An example to get details of a prediction:

    ```python
    prediction.details()
    ```
    """
    details = Client._get_data(relative_url=f"/prediction/{self.id}")

    details_df = pd.DataFrame(details, index=[0])[Prediction.ALL_PRED_COLS]

    return add_ready_column(details_df)


# Cell


@patch
def delete(self: Prediction) -> pd.DataFrame:
    """Delete a prediction from the server.

    Returns:
        A pandas DataFrame encapsulating the details of the deleted prediction.

    Raises:
        ConnectionError: If the server address is invalid or not reachable.

    An example to delete a prediction:

    ```python
    prediction.delete()
    ```
    """
    predictions = Client._delete_data(relative_url=f"/prediction/{self.id}")

    predictions_df = pd.DataFrame(predictions, index=[0])[Prediction.BASIC_PRED_COLS]

    return add_ready_column(predictions_df)


# Cell


@patch
def to_pandas(self: Prediction) -> pd.DataFrame:
    """Return the prediction results as a pandas DataFrame

    Returns:
        A pandas DataFrame encapsulating the results of the prediction.

    Raises:
        ConnectionError: If the server address is invalid or not reachable.

    An example to convert the prediction results into a pandas DataFrame:

    ```python
    predictions.to_pandas()
    ```
    """
    response = Client._get_data(relative_url=f"/prediction/{self.id}/pandas")
    keys = list(response.keys())
    keys.remove("Score")
    index_name = keys[0]
    return (
        pd.DataFrame(response)
        .set_index(index_name)
        .sort_values("Score", ascending=False)
    )


# Cell


# @patch
# def push(self: Prediction, data_source: "airt.components.datasource.DataSource") -> ProgressStatus: # type: ignore
#     """A function to push the prediction results into the target data source.

#     For more information on the supported data sources, please refer to the documentation on `DataSource` class

#     Args:
#         data_source: An instance of the `DataSource` class that encapsulates the data.

#     Returns:
#         An instance of `ProgressStatus` class. `ProgressStatus` is a base class for querying status of a remote operation. For more information
#         please refer to `ProgressStatus` class documentation.

#     Raises:
#         ValueError: If the input parameters to the API are invalid.
#         ConnectionError: If the server address is invalid or not reachable.


#     The below example illustrates pushing the prediction results to a database:

#     ```python
#     from datetime import timedelta

#     Client.get_token()

#     data_source_s3 = DataSource.s3(
#         uri="s3://test-airt-service/ecommerce_behavior"
#     )
#     data_source_s3.pull().progress_bar()

#     model = data_source_s3.train(
#         client_column="user_id",
#         target_column="event_type",
#         target="*purchase",
#         predict_after=timedelta(hours=3),
#     )

#     predictions = model.predict()

#     data_source_pred = DataSource.s3(
#         uri="s3://target-bucket"
#     )

#     progress = predictions.push(data_source_pred)
#     progress.progress_bar()
#     ```
#     """
#     response = Client._post_data(
#         relative_url=f"/prediction/{self.id}/push",
#         data=dict(data_id=data_source.id),
#     )
#     return ProgressStatus(relative_url=f"/prediction/push/{int(response['id'])}")
