"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const events = require("@aws-cdk/aws-events");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const destinations = require("../lib");
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
const lambdaProps = {
    code: new lambda.InlineCode('foo'),
    handler: 'index.handler',
    runtime: lambda.Runtime.NODEJS_10_X,
};
test('event bus as destination', () => {
    // GIVEN
    const eventBus = new events.EventBus(stack, 'EventBus');
    // WHEN
    new lambda.Function(stack, 'Function', Object.assign(Object.assign({}, lambdaProps), { onSuccess: new destinations.EventBridgeDestination(eventBus) }));
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'EventBus7B8748AA',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'events:PutEvents',
                    Effect: 'Allow',
                    Resource: '*'
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('event bus as destination defaults to default event bus', () => {
    // WHEN
    new lambda.Function(stack, 'Function', Object.assign(Object.assign({}, lambdaProps), { onSuccess: new destinations.EventBridgeDestination() }));
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':events:',
                            {
                                Ref: 'AWS::Region'
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId'
                            },
                            ':event-bus/default'
                        ]
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'events:PutEvents',
                    Effect: 'Allow',
                    Resource: '*'
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('lambda as destination', () => {
    // GIVEN
    const successLambda = new lambda.Function(stack, 'SuccessFunction', Object.assign({}, lambdaProps));
    // WHEN
    new lambda.Function(stack, 'Function', Object.assign(Object.assign({}, lambdaProps), { onSuccess: new destinations.LambdaDestination(successLambda) }));
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'SuccessFunction93C61D39',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'lambda:InvokeFunction',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'SuccessFunction93C61D39',
                            'Arn'
                        ]
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('sns as destination', () => {
    // GIVEN
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    new lambda.Function(stack, 'Function', Object.assign(Object.assign({}, lambdaProps), { onSuccess: new destinations.SnsDestination(topic) }));
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    Ref: 'TopicBFC7AF6E'
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'sns:Publish',
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'TopicBFC7AF6E'
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('sqs as destination', () => {
    // GIVEN
    const queue = new sqs.Queue(stack, 'Queue');
    // WHEN
    new lambda.Function(stack, 'Function', Object.assign(Object.assign({}, lambdaProps), { onSuccess: new destinations.SqsDestination(queue) }));
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'Queue4A7E3555',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'sqs:SendMessage',
                        'sqs:GetQueueAttributes',
                        'sqs:GetQueueUrl'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Queue4A7E3555',
                            'Arn'
                        ]
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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