import itertools
from numbers import Number
from typing import Any, Callable, Dict, List, Optional, Tuple, Union, cast

import numpy as np

import great_expectations.exceptions as ge_exceptions
from great_expectations.core.batch import Batch, BatchRequest, RuntimeBatchRequest
from great_expectations.rule_based_profiler.helpers.util import (
    NP_EPSILON,
    compute_bootstrap_quantiles_point_estimate,
    compute_quantiles,
    get_parameter_value_and_validate_return_type,
)
from great_expectations.rule_based_profiler.parameter_builder import (
    AttributedResolvedMetrics,
    MetricMultiBatchParameterBuilder,
    MetricValues,
)
from great_expectations.rule_based_profiler.types import (
    Domain,
    ParameterContainer,
    ParameterNode,
)
from great_expectations.util import is_numeric

MAX_DECIMALS: int = 9

DEFAULT_BOOTSTRAP_NUM_RESAMPLES: int = 9999


class NumericMetricRangeMultiBatchParameterBuilder(MetricMultiBatchParameterBuilder):
    """
    A Multi-Batch implementation for obtaining the range estimation bounds for a resolved (evaluated) numeric metric,
    using domain_kwargs, value_kwargs, metric_name, and false_positive_rate (tolerance) as arguments.

    This Multi-Batch ParameterBuilder is general in the sense that any metric that computes numbers can be accommodated.
    On the other hand, it is specific in the sense that the parameter names will always have the semantics of numeric
    ranges, which will incorporate the requirements, imposed by the configured false_positive_rate tolerances.

    The implementation supports two methods of estimating parameter values from data:
    * bootstrapped (default) -- a statistical technique (see "https://en.wikipedia.org/wiki/Bootstrapping_(statistics)")
    * one-shot -- assumes that metric values, computed on batch data, are normally distributed and computes the mean
      and the standard error using the queried batches as the single sample of the distribution (fast, but inaccurate).
    """

    RECOGNIZED_SAMPLING_METHOD_NAMES: set = {
        "oneshot",
        "bootstrap",
    }

    RECOGNIZED_TRUNCATE_DISTRIBUTION_KEYS: set = {
        "lower_bound",
        "upper_bound",
    }

    def __init__(
        self,
        name: str,
        metric_name: str,
        metric_domain_kwargs: Optional[Union[str, dict]] = None,
        metric_value_kwargs: Optional[Union[str, dict]] = None,
        enforce_numeric_metric: Union[str, bool] = True,
        replace_nan_with_zero: Union[str, bool] = True,
        reduce_scalar_metric: Union[str, bool] = True,
        false_positive_rate: Union[str, float] = 5.0e-2,
        estimator: str = "bootstrap",
        num_bootstrap_samples: Optional[Union[str, int]] = None,
        bootstrap_random_seed: Optional[Union[str, int]] = None,
        round_decimals: Optional[Union[str, int]] = None,
        truncate_values: Optional[
            Union[str, Dict[str, Union[Optional[int], Optional[float]]]]
        ] = None,
        evaluation_parameter_builder_configs: Optional[List[dict]] = None,
        json_serialize: Union[str, bool] = True,
        batch_list: Optional[List[Batch]] = None,
        batch_request: Optional[
            Union[str, BatchRequest, RuntimeBatchRequest, dict]
        ] = None,
        data_context: Optional["DataContext"] = None,  # noqa: F821
    ):
        """
        Args:
            name: the name of this parameter -- this is user-specified parameter name (from configuration);
            it is not the fully-qualified parameter name; a fully-qualified parameter name must start with "$parameter."
            and may contain one or more subsequent parts (e.g., "$parameter.<my_param_from_config>.<metric_name>").
            metric_name: the name of a metric used in MetricConfiguration (must be a supported and registered metric)
            metric_domain_kwargs: used in MetricConfiguration
            metric_value_kwargs: used in MetricConfiguration
            enforce_numeric_metric: used in MetricConfiguration to insure that metric computations return numeric values
            replace_nan_with_zero: if False, then if the computed metric gives NaN, then exception is raised; otherwise,
            if True (default), then if the computed metric gives NaN, then it is converted to the 0.0 (float) value.
            reduce_scalar_metric: if True (default), then reduces computation of 1-dimensional metric to scalar value.
            false_positive_rate: user-configured fraction between 0 and 1 expressing desired false positive rate for
            identifying unexpected values as judged by the upper- and lower- quantiles of the observed metric data.
            estimator: choice of the estimation algorithm: "oneshot" (one observation) or "bootstrap" (default)
            num_bootstrap_samples: Applicable only for the "bootstrap" sampling method -- if omitted (default), then
            9999 is used (default in "https://docs.scipy.org/doc/scipy/reference/generated/scipy.stats.bootstrap.html").
            round_decimals: user-configured non-negative integer indicating the number of decimals of the
            rounding precision of the computed parameter values (i.e., min_value, max_value) prior to packaging them on
            output.  If omitted, then no rounding is performed, unless the computed value is already an integer.
            truncate_values: user-configured directive for whether or not to allow the computed parameter values
            (i.e., lower_bound, upper_bound) to take on values outside the specified bounds when packaged on output.
            evaluation_parameter_builder_configs: ParameterBuilder configurations, executing and making whose respective
            ParameterBuilder objects' outputs available (as fully-qualified parameter names) is pre-requisite.
            These "ParameterBuilder" configurations help build parameters needed for this "ParameterBuilder".
            json_serialize: If True (default), convert computed value to JSON prior to saving results.
            batch_list: explicitly passed Batch objects for parameter computation (take precedence over batch_request).
            batch_request: specified in ParameterBuilder configuration to get Batch objects for parameter computation.
            data_context: DataContext
        """
        super().__init__(
            name=name,
            metric_name=metric_name,
            metric_domain_kwargs=metric_domain_kwargs,
            metric_value_kwargs=metric_value_kwargs,
            enforce_numeric_metric=enforce_numeric_metric,
            replace_nan_with_zero=replace_nan_with_zero,
            reduce_scalar_metric=reduce_scalar_metric,
            evaluation_parameter_builder_configs=evaluation_parameter_builder_configs,
            json_serialize=json_serialize,
            batch_list=batch_list,
            batch_request=batch_request,
            data_context=data_context,
        )

        self._false_positive_rate = false_positive_rate

        self._estimator = estimator

        self._num_bootstrap_samples = num_bootstrap_samples

        self._bootstrap_random_seed = bootstrap_random_seed

        self._round_decimals = round_decimals

        if not truncate_values:
            truncate_values = {
                "lower_bound": None,
                "upper_bound": None,
            }
        else:
            if not isinstance(truncate_values, str):
                truncate_values_keys: set = set(truncate_values.keys())
                if (
                    not truncate_values_keys
                    <= NumericMetricRangeMultiBatchParameterBuilder.RECOGNIZED_TRUNCATE_DISTRIBUTION_KEYS
                ):
                    raise ge_exceptions.ProfilerExecutionError(
                        message=f"""Unrecognized truncate_values key(s) in {self.__class__.__name__}:
"{str(truncate_values_keys - NumericMetricRangeMultiBatchParameterBuilder.RECOGNIZED_TRUNCATE_DISTRIBUTION_KEYS)}" \
detected.
"""
                    )

        self._truncate_values = truncate_values

    """
    Full getter/setter accessors for needed properties are for configuring MetricMultiBatchParameterBuilder dynamically.
    """

    @property
    def false_positive_rate(self) -> Union[str, float]:
        return self._false_positive_rate

    @property
    def estimator(self) -> str:
        return self._estimator

    @property
    def num_bootstrap_samples(self) -> Optional[Union[str, int]]:
        return self._num_bootstrap_samples

    @property
    def bootstrap_random_seed(self) -> Optional[Union[str, int]]:
        return self._bootstrap_random_seed

    @property
    def round_decimals(self) -> Optional[Union[str, int]]:
        return self._round_decimals

    @property
    def truncate_values(
        self,
    ) -> Optional[Union[str, Dict[str, Union[Optional[int], Optional[float]]]]]:
        return self._truncate_values

    def _build_parameters(
        self,
        parameter_container: ParameterContainer,
        domain: Domain,
        variables: Optional[ParameterContainer] = None,
        parameters: Optional[Dict[str, ParameterContainer]] = None,
    ) -> Tuple[Any, dict]:
        """
         Builds ParameterContainer object that holds ParameterNode objects with attribute name-value pairs and optional
         details.

         return: Tuple containing computed_parameter_value and parameter_computation_details metadata.

         The algorithm operates according to the following steps:
         1. Obtain batch IDs of interest using DataContext and BatchRequest (unless passed explicitly as argument). Note
         that this specific BatchRequest was specified as part of configuration for the present ParameterBuilder class.
         2. Set up metric_domain_kwargs and metric_value_kwargs (using configuration and/or variables and parameters).
         3. Instantiate the Validator object corresponding to BatchRequest (with a temporary expectation_suite_name) in
            order to have access to all Batch objects, on each of which the specified metric_name will be computed.
         4. Perform metric computations and obtain the result in the array-like form (one metric value per each Batch).
         5. Using the configured directives and heuristics, determine whether or not the ranges should be clipped.
         6. Using the configured directives and heuristics, determine if return values should be rounded to an integer.
         7. Convert the multi-dimensional metric computation results to a numpy array (for further computations).
         Steps 8 -- 10 are for the "oneshot" sampling method only (the "bootstrap" method achieves same automatically):
         8. Compute the mean and the standard deviation of the metric (aggregated over all the gathered Batch objects).
         9. Compute number of standard deviations (as floating point) needed (around the mean) to achieve the specified
            false_positive_rate (note that false_positive_rate of 0.0 would result in infinite number of standard deviations,
            hence it is "nudged" by small quantity "epsilon" above 0.0 if false_positive_rate of 0.0 appears as argument).
            (Please refer to "https://en.wikipedia.org/wiki/Normal_distribution" and references therein for background.)
        10. Compute the "band" around the mean as the min_value and max_value (to be used in ExpectationConfiguration).
        11. Return [low, high] for the desired metric as estimated by the specified sampling method.
        12. Set up the arguments and call build_parameter_container() to store the parameter as part of "rule state".
        """
        # Obtain false_positive_rate from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        false_positive_rate: np.float64 = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=self.false_positive_rate,
            expected_return_type=(float, np.float64),
            variables=variables,
            parameters=parameters,
        )
        if not (0.0 <= false_positive_rate <= 1.0):
            raise ge_exceptions.ProfilerExecutionError(
                message=f"The confidence level for {self.__class__.__name__} is outside of [0.0, 1.0] closed interval."
            )

        # Obtain estimator directive from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        estimator: str = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=self.estimator,
            expected_return_type=str,
            variables=variables,
            parameters=parameters,
        )
        if (
            estimator
            not in NumericMetricRangeMultiBatchParameterBuilder.RECOGNIZED_SAMPLING_METHOD_NAMES
        ):
            raise ge_exceptions.ProfilerExecutionError(
                message=f"""The directive "estimator" for {self.__class__.__name__} can be only one of
{NumericMetricRangeMultiBatchParameterBuilder.RECOGNIZED_SAMPLING_METHOD_NAMES} ("{estimator}" was detected).
"""
            )

        estimator_func: Callable
        etimator_kwargs: dict
        if estimator == "bootstrap":
            estimator_func = self._get_bootstrap_estimate
            estimator_kwargs = {
                "false_positive_rate": false_positive_rate,
                "num_bootstrap_samples": self.num_bootstrap_samples,
            }
        else:
            estimator_func = self._get_deterministic_estimate
            estimator_kwargs = {
                "false_positive_rate": false_positive_rate,
            }

        # Compute metric value for each Batch object.
        super().build_parameters(
            parameter_container=parameter_container,
            domain=domain,
            variables=variables,
            parameters=parameters,
            parameter_computation_impl=super()._build_parameters,
            json_serialize=False,
        )

        # Retrieve metric values for all Batch objects.
        parameter_node: ParameterNode = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=self.fully_qualified_parameter_name,
            expected_return_type=None,
            variables=variables,
            parameters=parameters,
        )
        metric_values: MetricValues
        if isinstance(parameter_node.value, list):
            num_parameter_node_value_elements: int = len(parameter_node.value)
            if not (num_parameter_node_value_elements == 1):
                raise ge_exceptions.ProfilerExecutionError(
                    message=f'Length of "AttributedResolvedMetrics" list for {self.__class__.__name__} must be exactly 1 ({num_parameter_node_value_elements} elements found).'
                )

            attributed_resolved_metrics: AttributedResolvedMetrics = (
                parameter_node.value[0]
            )
            metric_values = attributed_resolved_metrics.metric_values
        else:
            metric_values = parameter_node.value

        metric_value_range: np.ndarray = self._estimate_metric_value_range(
            metric_values=metric_values,
            estimator_func=estimator_func,
            domain=domain,
            variables=variables,
            parameters=parameters,
            **estimator_kwargs,
        )

        return (
            metric_value_range,
            parameter_node.details,
        )

    def _estimate_metric_value_range(
        self,
        metric_values: np.ndarray,
        estimator_func: Callable,
        domain: Optional[Domain] = None,
        variables: Optional[ParameterContainer] = None,
        parameters: Optional[Dict[str, ParameterContainer]] = None,
        **kwargs,
    ) -> np.ndarray:
        """
        This method accepts an estimator Callable and data samples in the format "N x R^m", where "N" (most significant
        dimension) is the number of measurements (e.g., one per Batch of data), while "R^m" is the multi-dimensional
        metric, whose values are being estimated.  Thus, for each element in the "R^m" hypercube, an "N"-dimensional
        vector of sample measurements is constructed and given to the estimator to apply its specific algorithm for
        computing the range of values in this vector.  Estimator algorithms differ based on their use of data samples.
        """
        truncate_values: Dict[str, Number] = self._get_truncate_values_using_heuristics(
            metric_values=metric_values,
            domain=domain,
            variables=variables,
            parameters=parameters,
        )
        lower_bound: Optional[float] = truncate_values.get("lower_bound")
        upper_bound: Optional[float] = truncate_values.get("upper_bound")

        round_decimals: int = self._get_round_decimals_using_heuristics(
            metric_values=metric_values,
            domain=domain,
            variables=variables,
            parameters=parameters,
        )

        min_value: Number
        max_value: Number

        lower_quantile: Number
        upper_quantile: Number

        # Outer-most dimension is data samples (e.g., one per Batch); the rest are dimensions of the actual metric.
        metric_value_shape: tuple = metric_values.shape[1:]

        # Generate all permutations of indexes for accessing every element of the multi-dimensional metric.
        metric_value_shape_idx: int
        axes: List[np.ndarray] = [
            np.indices(dimensions=(metric_value_shape_idx,))[0]
            for metric_value_shape_idx in metric_value_shape
        ]
        metric_value_indices: List[tuple] = list(itertools.product(*tuple(axes)))

        # Generate all permutations of indexes for accessing estimates of every element of the multi-dimensional metric.
        # Prefixing multi-dimensional index with "(slice(None, None, None),)" is equivalent to "[:,]" access.
        metric_value_idx: tuple
        metric_value_vector_indices: List[tuple] = [
            (slice(None, None, None),) + metric_value_idx
            for metric_value_idx in metric_value_indices
        ]

        # Since range includes min and max values, value range estimate contains 2-element least-significant dimension.
        metric_value_range_shape: tuple = metric_value_shape + (2,)
        # Initialize value range estimate for multi-dimensional metric to all trivial values (to be updated in situ).
        metric_value_range: np.ndarray = np.zeros(shape=metric_value_range_shape)

        metric_value_vector: np.ndarray
        metric_value_range_min_idx: tuple
        metric_value_range_max_idx: tuple
        # Traverse indices of sample vectors corresponding to every element of multi-dimensional metric.
        for metric_value_idx in metric_value_vector_indices:
            # Obtain "N"-element-long vector of samples for each element of multi-dimensional metric.
            metric_value_vector = metric_values[metric_value_idx]
            if np.all(np.isclose(metric_value_vector, metric_value_vector[0])):
                # Computation is unnecessary if distribution is degenerate.
                lower_quantile = upper_quantile = metric_value_vector[0]
            else:
                # Compute low and high estimates for vector of samples for given element of multi-dimensional metric.
                lower_quantile, upper_quantile = estimator_func(
                    metric_values=metric_value_vector,
                    domain=domain,
                    variables=variables,
                    parameters=parameters,
                    **kwargs,
                )

            if round_decimals == 0:
                min_value = round(float(cast(float, lower_quantile)))
                max_value = round(float(cast(float, upper_quantile)))
            else:
                min_value = round(float(cast(float, lower_quantile)), round_decimals)
                max_value = round(float(cast(float, upper_quantile)), round_decimals)

            if lower_bound is not None:
                min_value = max(cast(float, min_value), lower_bound)

            if upper_bound is not None:
                max_value = min(cast(float, max_value), upper_bound)

            # Obtain index of metric element (by discarding "N"-element samples dimension).
            metric_value_idx = metric_value_idx[1:]

            # Compute indices for min and max value range estimates.
            metric_value_range_min_idx = metric_value_idx + (
                slice(0, 1, None),
            )  # appends "[0]" element
            metric_value_range_max_idx = metric_value_idx + (
                slice(1, 2, None),
            )  # appends "[0]" element

            # Store computed min and max value estimates into allocated range estimate for multi-dimensional metric.
            metric_value_range[metric_value_range_min_idx] = min_value
            metric_value_range[metric_value_range_max_idx] = max_value

        # As a simplification, apply reduction to scalar in case of one-dimensional metric (for convenience).
        if metric_value_range.shape[0] == 1:
            metric_value_range = metric_value_range[0]

        return metric_value_range

    def _get_truncate_values_using_heuristics(
        self,
        metric_values: np.ndarray,
        domain: Domain,
        *,
        variables: Optional[ParameterContainer] = None,
        parameters: Optional[Dict[str, ParameterContainer]] = None,
    ) -> Dict[str, Union[Optional[int], Optional[float]]]:
        # Obtain truncate_values directive from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        truncate_values: Dict[
            str, Optional[Number]
        ] = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=self.truncate_values,
            expected_return_type=dict,
            variables=variables,
            parameters=parameters,
        )

        distribution_boundary: Optional[Union[int, float]]
        if not all(
            [
                (
                    distribution_boundary is None
                    or is_numeric(value=distribution_boundary)
                )
                for distribution_boundary in truncate_values.values()
            ]
        ):
            raise ge_exceptions.ProfilerExecutionError(
                message=f"""The directive "truncate_values" for {self.__class__.__name__} must specify the
[lower_bound, upper_bound] closed interval, where either boundary is a numeric value (or None).
"""
            )

        lower_bound: Optional[Number] = truncate_values.get("lower_bound")
        upper_bound: Optional[Number] = truncate_values.get("upper_bound")

        if lower_bound is None and np.all(np.greater(metric_values, NP_EPSILON)):
            lower_bound = 0.0

        if upper_bound is None and np.all(np.less(metric_values, (-NP_EPSILON))):
            upper_bound = 0.0

        return {
            "lower_bound": lower_bound,
            "upper_bound": upper_bound,
        }

    def _get_round_decimals_using_heuristics(
        self,
        metric_values: np.ndarray,
        domain: Domain,
        variables: Optional[ParameterContainer] = None,
        parameters: Optional[Dict[str, ParameterContainer]] = None,
    ) -> int:
        # Obtain round_decimals directive from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        round_decimals: Optional[int] = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=self.round_decimals,
            expected_return_type=None,
            variables=variables,
            parameters=parameters,
        )
        if round_decimals is None:
            round_decimals = MAX_DECIMALS
        else:
            if not isinstance(round_decimals, int) or (round_decimals < 0):
                raise ge_exceptions.ProfilerExecutionError(
                    message=f"""The directive "round_decimals" for {self.__class__.__name__} can be 0 or a
positive integer, or must be omitted (or set to None).
"""
                )

        if np.issubdtype(metric_values.dtype, np.integer):
            round_decimals = 0

        return round_decimals

    @staticmethod
    def _get_bootstrap_estimate(
        metric_values: np.ndarray,
        domain: Domain,
        variables: Optional[ParameterContainer] = None,
        parameters: Optional[Dict[str, ParameterContainer]] = None,
        **kwargs,
    ) -> Tuple[Number, Number]:
        false_positive_rate: np.float64 = kwargs.get("false_positive_rate", 5.0e-2)

        # Obtain num_bootstrap_samples override from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        num_bootstrap_samples: Optional[
            int
        ] = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=kwargs.get("num_bootstrap_samples"),
            expected_return_type=None,
            variables=variables,
            parameters=parameters,
        )

        n_resamples: int
        if num_bootstrap_samples is None:
            n_resamples = DEFAULT_BOOTSTRAP_NUM_RESAMPLES
        else:
            n_resamples = num_bootstrap_samples

        # Obtain random_seed override from "rule state" (i.e., variables and parameters); from instance variable otherwise.
        random_seed: Optional[int] = get_parameter_value_and_validate_return_type(
            domain=domain,
            parameter_reference=kwargs.get("bootstrap_random_seed"),
            expected_return_type=None,
            variables=variables,
            parameters=parameters,
        )

        return compute_bootstrap_quantiles_point_estimate(
            metric_values=metric_values,
            false_positive_rate=false_positive_rate,
            n_resamples=n_resamples,
            random_seed=random_seed,
        )

    @staticmethod
    def _get_deterministic_estimate(
        metric_values: np.ndarray,
        **kwargs,
    ) -> Tuple[Number, Number]:
        false_positive_rate: np.float64 = kwargs.get("false_positive_rate", 5.0e-2)

        return compute_quantiles(
            metric_values=metric_values,
            false_positive_rate=false_positive_rate,
        )
