"""
The Gauss-Patterson quadrature is a nested family which begins with the
Gauss-Legendre rules of orders 1 and 3, and then successively inserts one new
abscissa in each subinterval. Thus, after the second rule, the Gauss-Patterson
rules do not have the super-high precision of the Gauss-Legendre rules. They
trade this precision in exchange for the advantages of nestedness. This means
that Gauss-Patterson rules are only available for orders of 1, 3, 7, 15, 31,
63, 127, 255 or 511.

Sources
---------

This code is an adapted of John Burkardt's implementation in Fortran. The
algorithm is taken from "Handbook of Computational Methods for Integration" by
P. Kythe and M. Schaeferkotter, and `The Optimal Addition of Points to
Quadrature Formulae`_.

.. _The Optimal Addition of Points to Quadrature Formulae: \
    https://www.jstor.org/stable/2004583

Example usage
-------------

With increasing order::

    >>> distribution = chaospy.Beta(2, 2, lower=-1, upper=1)
    >>> for order in range(3):  # doctest: +NORMALIZE_WHITESPACE
    ...     abscissas, weights = chaospy.generate_quadrature(
    ...         order, distribution, rule="gauss_patterson")
    ...     print(abscissas.round(2), weights.round(2))
    [[0.]] [1.]
    [[-0.77  0.    0.77]] [0.17 0.67 0.17]
    [[-0.96 -0.77 -0.43  0.    0.43  0.77  0.96]]
     [0.01 0.08 0.24 0.34 0.24 0.08 0.01]
"""
import numpy
import chaospy

from .combine import combine


def quad_gauss_patterson(order, domain):
    """
    Generate sets abscissas and weights for Gauss-Patterson quadrature.

    Args:
        order (int):
            The quadrature order. Must be in the interval (0, 8).
        domain (chaospy.distributions.baseclass.Distribution, numpy.ndarray):
            Either distribution or bounding of interval to integrate over.

    Returns:
        (numpy.ndarray, numpy.ndarray):
            abscissas:
                The quadrature points for where to evaluate the model function
                with ``abscissas.shape == (len(dist), N)`` where ``N`` is the
                number of samples.
            weights:
                The quadrature weights with ``weights.shape == (N,)``.

    Example:
        >>> abscissas, weights = chaospy.quad_gauss_patterson(
        ...     2, chaospy.Uniform(0, 1))
        >>> abscissas.round(4)
        array([[0.0198, 0.1127, 0.2829, 0.5   , 0.7171, 0.8873, 0.9802]])
        >>> weights.round(4)
        array([0.0523, 0.1342, 0.2007, 0.2255, 0.2007, 0.1342, 0.0523])
    """
    if isinstance(domain, chaospy.Distribution):
        abscissas, weights = quad_gauss_patterson(
            order, (domain.lower, domain.upper))
        weights *= domain.pdf(abscissas).flatten()
        weights /= numpy.sum(weights)
        return abscissas, weights

    lower, upper = numpy.array(domain)
    lower = numpy.asarray(lower).flatten()
    upper = numpy.asarray(upper).flatten()

    if len(lower) > 1:

        values = [quad_gauss_patterson(order[i], (domain[0][i], domain[1][i]))
                  for i in range(len(domain[0]))]

        abscissas = [_[0][0] for _ in values]
        weights = [_[1] for _ in values]
        abscissas = combine(abscissas).T
        weights = numpy.prod(combine(weights), -1)

        return abscissas, weights

    order = sorted(PATTERSON_VALUES.keys())[order]
    abscissas, weights = PATTERSON_VALUES[order]

    abscissas = .5*(abscissas*(upper-lower)+upper+lower)
    abscissas = abscissas.reshape(1, abscissas.size)
    weights /= numpy.sum(weights)

    return abscissas, weights


PATTERSON_VALUES = {
    0 : ((0e+00,), (2.0e+00,)),
    3 : ((
        -0.77459666924148337704e+00, 0.0e+00, 0.77459666924148337704e+00,
    ), (
        0.555555555555555555556e+00, 0.888888888888888888889e+00,
        0.555555555555555555556e+00,
    )),
    7 : ((
        -0.96049126870802028342e+00, -0.77459666924148337704e+00,
        -0.43424374934680255800e+00, 0.0e+00, 0.43424374934680255800e+00,
        0.77459666924148337704e+00, 0.96049126870802028342e+00,
    ), (
        0.104656226026467265194e+00, 0.268488089868333440729e+00,
        0.401397414775962222905e+00, 0.450916538658474142345e+00,
        0.401397414775962222905e+00, 0.268488089868333440729e+00,
        0.104656226026467265194e+00,
    )),
    15 : ((
        -0.99383196321275502221e+00, -0.96049126870802028342e+00,
        -0.88845923287225699889e+00, -0.77459666924148337704e+00,
        -0.62110294673722640294e+00, -0.43424374934680255800e+00,
        -0.22338668642896688163e+00, 0.0e+00, 0.22338668642896688163e+00,
        0.43424374934680255800e+00, 0.62110294673722640294e+00,
        0.77459666924148337704e+00, 0.88845923287225699889e+00,
        0.96049126870802028342e+00, 0.99383196321275502221e+00,
    ), (
        0.0170017196299402603390e+00, 0.0516032829970797396969e+00,
        0.0929271953151245376859e+00, 0.134415255243784220360e+00,
        0.171511909136391380787e+00, 0.200628529376989021034e+00,
        0.219156858401587496404e+00, 0.225510499798206687386e+00,
        0.219156858401587496404e+00, 0.200628529376989021034e+00,
        0.171511909136391380787e+00, 0.134415255243784220360e+00,
        0.0929271953151245376859e+00, 0.0516032829970797396969e+00,
        0.0170017196299402603390e+00,
    )),
    31 : ((
        -0.99909812496766759766e+00, -0.99383196321275502221e+00,
        -0.98153114955374010687e+00, -0.96049126870802028342e+00,
        -0.92965485742974005667e+00, -0.88845923287225699889e+00,
        -0.83672593816886873550e+00, -0.77459666924148337704e+00,
        -0.70249620649152707861e+00, -0.62110294673722640294e+00,
        -0.53131974364437562397e+00, -0.43424374934680255800e+00,
        -0.33113539325797683309e+00, -0.22338668642896688163e+00,
        -0.11248894313318662575e+00, 0.0e+00, 0.11248894313318662575e+00,
        0.22338668642896688163e+00, 0.33113539325797683309e+00,
        0.43424374934680255800e+00, 0.53131974364437562397e+00,
        0.62110294673722640294e+00, 0.70249620649152707861e+00,
        0.77459666924148337704e+00, 0.83672593816886873550e+00,
        0.88845923287225699889e+00, 0.92965485742974005667e+00,
        0.96049126870802028342e+00, 0.98153114955374010687e+00,
        0.99383196321275502221e+00, 0.99909812496766759766e+00,
    ), (
        0.00254478079156187441540e+00, 0.00843456573932110624631e+00,
        0.0164460498543878109338e+00, 0.0258075980961766535646e+00,
        0.0359571033071293220968e+00, 0.0464628932617579865414e+00,
        0.0569795094941233574122e+00, 0.0672077542959907035404e+00,
        0.0768796204990035310427e+00, 0.0857559200499903511542e+00,
        0.0936271099812644736167e+00, 0.100314278611795578771e+00,
        0.105669893580234809744e+00, 0.109578421055924638237e+00,
        0.111956873020953456880e+00, 0.112755256720768691607e+00,
        0.111956873020953456880e+00, 0.109578421055924638237e+00,
        0.105669893580234809744e+00, 0.100314278611795578771e+00,
        0.0936271099812644736167e+00, 0.0857559200499903511542e+00,
        0.0768796204990035310427e+00, 0.0672077542959907035404e+00,
        0.0569795094941233574122e+00, 0.0464628932617579865414e+00,
        0.0359571033071293220968e+00, 0.0258075980961766535646e+00,
        0.0164460498543878109338e+00, 0.00843456573932110624631e+00,
        0.00254478079156187441540e+00,
    )),
    63 : ((
        -0.99987288812035761194e+00, -0.99909812496766759766e+00,
        -0.99720625937222195908e+00, -0.99383196321275502221e+00,
        -0.98868475754742947994e+00, -0.98153114955374010687e+00,
        -0.97218287474858179658e+00, -0.96049126870802028342e+00,
        -0.94634285837340290515e+00, -0.92965485742974005667e+00,
        -0.91037115695700429250e+00, -0.88845923287225699889e+00,
        -0.86390793819369047715e+00, -0.83672593816886873550e+00,
        -0.80694053195021761186e+00, -0.77459666924148337704e+00,
        -0.73975604435269475868e+00, -0.70249620649152707861e+00,
        -0.66290966002478059546e+00, -0.62110294673722640294e+00,
        -0.57719571005204581484e+00, -0.53131974364437562397e+00,
        -0.48361802694584102756e+00, -0.43424374934680255800e+00,
        -0.38335932419873034692e+00, -0.33113539325797683309e+00,
        -0.27774982202182431507e+00, -0.22338668642896688163e+00,
        -0.16823525155220746498e+00, -0.11248894313318662575e+00,
        -0.056344313046592789972e+00, 0.0e+00, 0.056344313046592789972e+00,
        0.11248894313318662575e+00, 0.16823525155220746498e+00,
        0.22338668642896688163e+00, 0.27774982202182431507e+00,
        0.33113539325797683309e+00, 0.38335932419873034692e+00,
        0.43424374934680255800e+00, 0.48361802694584102756e+00,
        0.53131974364437562397e+00, 0.57719571005204581484e+00,
        0.62110294673722640294e+00, 0.66290966002478059546e+00,
        0.70249620649152707861e+00, 0.73975604435269475868e+00,
        0.77459666924148337704e+00, 0.80694053195021761186e+00,
        0.83672593816886873550e+00, 0.86390793819369047715e+00,
        0.88845923287225699889e+00, 0.91037115695700429250e+00,
        0.92965485742974005667e+00, 0.94634285837340290515e+00,
        0.96049126870802028342e+00, 0.97218287474858179658e+00,
        0.98153114955374010687e+00, 0.98868475754742947994e+00,
        0.99383196321275502221e+00, 0.99720625937222195908e+00,
        0.99909812496766759766e+00, 0.99987288812035761194e+00,
    ), (
        0.000363221481845530659694e+00, 0.00126515655623006801137e+00,
        0.00257904979468568827243e+00, 0.00421763044155885483908e+00,
        0.00611550682211724633968e+00, 0.00822300795723592966926e+00,
        0.0104982469096213218983e+00, 0.0129038001003512656260e+00,
        0.0154067504665594978021e+00, 0.0179785515681282703329e+00,
        0.0205942339159127111492e+00, 0.0232314466399102694433e+00,
        0.0258696793272147469108e+00, 0.0284897547458335486125e+00,
        0.0310735511116879648799e+00, 0.0336038771482077305417e+00,
        0.0360644327807825726401e+00, 0.0384398102494555320386e+00,
        0.0407155101169443189339e+00, 0.0428779600250077344929e+00,
        0.0449145316536321974143e+00, 0.0468135549906280124026e+00,
        0.0485643304066731987159e+00, 0.0501571393058995374137e+00,
        0.0515832539520484587768e+00, 0.0528349467901165198621e+00,
        0.0539054993352660639269e+00, 0.0547892105279628650322e+00,
        0.0554814043565593639878e+00, 0.0559784365104763194076e+00,
        0.0562776998312543012726e+00, 0.0563776283603847173877e+00,
        0.0562776998312543012726e+00, 0.0559784365104763194076e+00,
        0.0554814043565593639878e+00, 0.0547892105279628650322e+00,
        0.0539054993352660639269e+00, 0.0528349467901165198621e+00,
        0.0515832539520484587768e+00, 0.0501571393058995374137e+00,
        0.0485643304066731987159e+00, 0.0468135549906280124026e+00,
        0.0449145316536321974143e+00, 0.0428779600250077344929e+00,
        0.0407155101169443189339e+00, 0.0384398102494555320386e+00,
        0.0360644327807825726401e+00, 0.0336038771482077305417e+00,
        0.0310735511116879648799e+00, 0.0284897547458335486125e+00,
        0.0258696793272147469108e+00, 0.0232314466399102694433e+00,
        0.0205942339159127111492e+00, 0.0179785515681282703329e+00,
        0.0154067504665594978021e+00, 0.0129038001003512656260e+00,
        0.0104982469096213218983e+00, 0.00822300795723592966926e+00,
        0.00611550682211724633968e+00, 0.00421763044155885483908e+00,
        0.00257904979468568827243e+00, 0.00126515655623006801137e+00,
        0.000363221481845530659694e+00,
    )),
    127 : ((
        -0.99998243035489159858e+00, -0.99987288812035761194e+00,
        -0.99959879967191068325e+00, -0.99909812496766759766e+00,
        -0.99831663531840739253e+00, -0.99720625937222195908e+00,
        -0.99572410469840718851e+00, -0.99383196321275502221e+00,
        -0.99149572117810613240e+00, -0.98868475754742947994e+00,
        -0.98537149959852037111e+00, -0.98153114955374010687e+00,
        -0.97714151463970571416e+00, -0.97218287474858179658e+00,
        -0.96663785155841656709e+00, -0.96049126870802028342e+00,
        -0.95373000642576113641e+00, -0.94634285837340290515e+00,
        -0.93832039777959288365e+00, -0.92965485742974005667e+00,
        -0.92034002547001242073e+00, -0.91037115695700429250e+00,
        -0.89974489977694003664e+00, -0.88845923287225699889e+00,
        -0.87651341448470526974e+00, -0.86390793819369047715e+00,
        -0.85064449476835027976e+00, -0.83672593816886873550e+00,
        -0.82215625436498040737e+00, -0.80694053195021761186e+00,
        -0.79108493379984836143e+00, -0.77459666924148337704e+00,
        -0.75748396638051363793e+00, -0.73975604435269475868e+00,
        -0.72142308537009891548e+00, -0.70249620649152707861e+00,
        -0.68298743109107922809e+00, -0.66290966002478059546e+00,
        -0.64227664250975951377e+00, -0.62110294673722640294e+00,
        -0.59940393024224289297e+00, -0.57719571005204581484e+00,
        -0.55449513263193254887e+00, -0.53131974364437562397e+00,
        -0.50768775753371660215e+00, -0.48361802694584102756e+00,
        -0.45913001198983233287e+00, -0.43424374934680255800e+00,
        -0.40897982122988867241e+00, -0.38335932419873034692e+00,
        -0.35740383783153215238e+00, -0.33113539325797683309e+00,
        -0.30457644155671404334e+00, -0.27774982202182431507e+00,
        -0.25067873030348317661e+00, -0.22338668642896688163e+00,
        -0.19589750271110015392e+00, -0.16823525155220746498e+00,
        -0.14042423315256017459e+00, -0.11248894313318662575e+00,
        -0.084454040083710883710e+00, -0.056344313046592789972e+00,
        -0.028184648949745694339e+00, 0.0e+00, 0.028184648949745694339e+00,
        0.056344313046592789972e+00, 0.084454040083710883710e+00,
        0.11248894313318662575e+00, 0.14042423315256017459e+00,
        0.16823525155220746498e+00, 0.19589750271110015392e+00,
        0.22338668642896688163e+00, 0.25067873030348317661e+00,
        0.27774982202182431507e+00, 0.30457644155671404334e+00,
        0.33113539325797683309e+00, 0.35740383783153215238e+00,
        0.38335932419873034692e+00, 0.40897982122988867241e+00,
        0.43424374934680255800e+00, 0.45913001198983233287e+00,
        0.48361802694584102756e+00, 0.50768775753371660215e+00,
        0.53131974364437562397e+00, 0.55449513263193254887e+00,
        0.57719571005204581484e+00, 0.59940393024224289297e+00,
        0.62110294673722640294e+00, 0.64227664250975951377e+00,
        0.66290966002478059546e+00, 0.68298743109107922809e+00,
        0.70249620649152707861e+00, 0.72142308537009891548e+00,
        0.73975604435269475868e+00, 0.75748396638051363793e+00,
        0.77459666924148337704e+00, 0.79108493379984836143e+00,
        0.80694053195021761186e+00, 0.82215625436498040737e+00,
        0.83672593816886873550e+00, 0.85064449476835027976e+00,
        0.86390793819369047715e+00, 0.87651341448470526974e+00,
        0.88845923287225699889e+00, 0.89974489977694003664e+00,
        0.91037115695700429250e+00, 0.92034002547001242073e+00,
        0.92965485742974005667e+00, 0.93832039777959288365e+00,
        0.94634285837340290515e+00, 0.95373000642576113641e+00,
        0.96049126870802028342e+00, 0.96663785155841656709e+00,
        0.97218287474858179658e+00, 0.97714151463970571416e+00,
        0.98153114955374010687e+00, 0.98537149959852037111e+00,
        0.98868475754742947994e+00, 0.99149572117810613240e+00,
        0.99383196321275502221e+00, 0.99572410469840718851e+00,
        0.99720625937222195908e+00, 0.99831663531840739253e+00,
        0.99909812496766759766e+00, 0.99959879967191068325e+00,
        0.99987288812035761194e+00, 0.99998243035489159858e+00,
    ), (
        0.0000505360952078625176247e+00, 0.000180739564445388357820e+00,
        0.000377746646326984660274e+00, 0.000632607319362633544219e+00,
        0.000938369848542381500794e+00, 0.00128952408261041739210e+00,
        0.00168114286542146990631e+00, 0.00210881524572663287933e+00,
        0.00256876494379402037313e+00, 0.00305775341017553113613e+00,
        0.00357289278351729964938e+00, 0.00411150397865469304717e+00,
        0.00467105037211432174741e+00, 0.00524912345480885912513e+00,
        0.00584344987583563950756e+00, 0.00645190005017573692280e+00,
        0.00707248999543355546805e+00, 0.00770337523327974184817e+00,
        0.00834283875396815770558e+00, 0.00898927578406413572328e+00,
        0.00964117772970253669530e+00, 0.0102971169579563555237e+00,
        0.0109557333878379016480e+00, 0.0116157233199551347270e+00,
        0.0122758305600827700870e+00, 0.0129348396636073734547e+00,
        0.0135915710097655467896e+00, 0.0142448773729167743063e+00,
        0.0148936416648151820348e+00, 0.0155367755558439824399e+00,
        0.0161732187295777199419e+00, 0.0168019385741038652709e+00,
        0.0174219301594641737472e+00, 0.0180322163903912863201e+00,
        0.0186318482561387901863e+00, 0.0192199051247277660193e+00,
        0.0197954950480974994880e+00, 0.0203577550584721594669e+00,
        0.0209058514458120238522e+00, 0.0214389800125038672465e+00,
        0.0219563663053178249393e+00, 0.0224572658268160987071e+00,
        0.0229409642293877487608e+00, 0.0234067774953140062013e+00,
        0.0238540521060385400804e+00, 0.0242821652033365993580e+00,
        0.0246905247444876769091e+00, 0.0250785696529497687068e+00,
        0.0254457699654647658126e+00, 0.0257916269760242293884e+00,
        0.0261156733767060976805e+00, 0.0264174733950582599310e+00,
        0.0266966229274503599062e+00, 0.0269527496676330319634e+00,
        0.0271855132296247918192e+00, 0.0273946052639814325161e+00,
        0.0275797495664818730349e+00, 0.0277407021782796819939e+00,
        0.0278772514766137016085e+00, 0.0279892182552381597038e+00,
        0.0280764557938172466068e+00, 0.0281388499156271506363e+00,
        0.0281763190330166021307e+00, 0.0281888141801923586938e+00,
        0.0281763190330166021307e+00, 0.0281388499156271506363e+00,
        0.0280764557938172466068e+00, 0.0279892182552381597038e+00,
        0.0278772514766137016085e+00, 0.0277407021782796819939e+00,
        0.0275797495664818730349e+00, 0.0273946052639814325161e+00,
        0.0271855132296247918192e+00, 0.0269527496676330319634e+00,
        0.0266966229274503599062e+00, 0.0264174733950582599310e+00,
        0.0261156733767060976805e+00, 0.0257916269760242293884e+00,
        0.0254457699654647658126e+00, 0.0250785696529497687068e+00,
        0.0246905247444876769091e+00, 0.0242821652033365993580e+00,
        0.0238540521060385400804e+00, 0.0234067774953140062013e+00,
        0.0229409642293877487608e+00, 0.0224572658268160987071e+00,
        0.0219563663053178249393e+00, 0.0214389800125038672465e+00,
        0.0209058514458120238522e+00, 0.0203577550584721594669e+00,
        0.0197954950480974994880e+00, 0.0192199051247277660193e+00,
        0.0186318482561387901863e+00, 0.0180322163903912863201e+00,
        0.0174219301594641737472e+00, 0.0168019385741038652709e+00,
        0.0161732187295777199419e+00, 0.0155367755558439824399e+00,
        0.0148936416648151820348e+00, 0.0142448773729167743063e+00,
        0.0135915710097655467896e+00, 0.0129348396636073734547e+00,
        0.0122758305600827700870e+00, 0.0116157233199551347270e+00,
        0.0109557333878379016480e+00, 0.0102971169579563555237e+00,
        0.00964117772970253669530e+00, 0.00898927578406413572328e+00,
        0.00834283875396815770558e+00, 0.00770337523327974184817e+00,
        0.00707248999543355546805e+00, 0.00645190005017573692280e+00,
        0.00584344987583563950756e+00, 0.00524912345480885912513e+00,
        0.00467105037211432174741e+00, 0.00411150397865469304717e+00,
        0.00357289278351729964938e+00, 0.00305775341017553113613e+00,
        0.00256876494379402037313e+00, 0.00210881524572663287933e+00,
        0.00168114286542146990631e+00, 0.00128952408261041739210e+00,
        0.000938369848542381500794e+00, 0.000632607319362633544219e+00,
        0.000377746646326984660274e+00, 0.000180739564445388357820e+00,
        0.0000505360952078625176247e+00,
    )),
    255 : ((
        -0.99999759637974846462e+00, -0.99998243035489159858e+00,
        -0.99994399620705437576e+00, -0.99987288812035761194e+00,
        -0.99976049092443204733e+00, -0.99959879967191068325e+00,
        -0.99938033802502358193e+00, -0.99909812496766759766e+00,
        -0.99874561446809511470e+00, -0.99831663531840739253e+00,
        -0.99780535449595727456e+00, -0.99720625937222195908e+00,
        -0.99651414591489027385e+00, -0.99572410469840718851e+00,
        -0.99483150280062100052e+00, -0.99383196321275502221e+00,
        -0.99272134428278861533e+00, -0.99149572117810613240e+00,
        -0.99015137040077015918e+00, -0.98868475754742947994e+00,
        -0.98709252795403406719e+00, -0.98537149959852037111e+00,
        -0.98351865757863272876e+00, -0.98153114955374010687e+00,
        -0.97940628167086268381e+00, -0.97714151463970571416e+00,
        -0.97473445975240266776e+00, -0.97218287474858179658e+00,
        -0.96948465950245923177e+00, -0.96663785155841656709e+00,
        -0.96364062156981213252e+00, -0.96049126870802028342e+00,
        -0.95718821610986096274e+00, -0.95373000642576113641e+00,
        -0.95011529752129487656e+00, -0.94634285837340290515e+00,
        -0.94241156519108305981e+00, -0.93832039777959288365e+00,
        -0.93406843615772578800e+00, -0.92965485742974005667e+00,
        -0.92507893290707565236e+00, -0.92034002547001242073e+00,
        -0.91543758715576504064e+00, -0.91037115695700429250e+00,
        -0.90514035881326159519e+00, -0.89974489977694003664e+00,
        -0.89418456833555902286e+00, -0.88845923287225699889e+00,
        -0.88256884024734190684e+00, -0.87651341448470526974e+00,
        -0.87029305554811390585e+00, -0.86390793819369047715e+00,
        -0.85735831088623215653e+00, -0.85064449476835027976e+00,
        -0.84376688267270860104e+00, -0.83672593816886873550e+00,
        -0.82952219463740140018e+00, -0.82215625436498040737e+00,
        -0.81462878765513741344e+00, -0.80694053195021761186e+00,
        -0.79909229096084140180e+00, -0.79108493379984836143e+00,
        -0.78291939411828301639e+00, -0.77459666924148337704e+00,
        -0.76611781930376009072e+00, -0.75748396638051363793e+00,
        -0.74869629361693660282e+00, -0.73975604435269475868e+00,
        -0.73066452124218126133e+00, -0.72142308537009891548e+00,
        -0.71203315536225203459e+00, -0.70249620649152707861e+00,
        -0.69281376977911470289e+00, -0.68298743109107922809e+00,
        -0.67301883023041847920e+00, -0.66290966002478059546e+00,
        -0.65266166541001749610e+00, -0.64227664250975951377e+00,
        -0.63175643771119423041e+00, -0.62110294673722640294e+00,
        -0.61031811371518640016e+00, -0.59940393024224289297e+00,
        -0.58836243444766254143e+00, -0.57719571005204581484e+00,
        -0.56590588542365442262e+00, -0.55449513263193254887e+00,
        -0.54296566649831149049e+00, -0.53131974364437562397e+00,
        -0.51955966153745702199e+00, -0.50768775753371660215e+00,
        -0.49570640791876146017e+00, -0.48361802694584102756e+00,
        -0.47142506587165887693e+00, -0.45913001198983233287e+00,
        -0.44673538766202847374e+00, -0.43424374934680255800e+00,
        -0.42165768662616330006e+00, -0.40897982122988867241e+00,
        -0.39621280605761593918e+00, -0.38335932419873034692e+00,
        -0.37042208795007823014e+00, -0.35740383783153215238e+00,
        -0.34430734159943802278e+00, -0.33113539325797683309e+00,
        -0.31789081206847668318e+00, -0.30457644155671404334e+00,
        -0.29119514851824668196e+00, -0.27774982202182431507e+00,
        -0.26424337241092676194e+00, -0.25067873030348317661e+00,
        -0.23705884558982972721e+00, -0.22338668642896688163e+00,
        -0.20966523824318119477e+00, -0.19589750271110015392e+00,
        -0.18208649675925219825e+00, -0.16823525155220746498e+00,
        -0.15434681148137810869e+00, -0.14042423315256017459e+00,
        -0.12647058437230196685e+00, -0.11248894313318662575e+00,
        -0.098482396598119202090e+00, -0.084454040083710883710e+00,
        -0.070406976042855179063e+00, -0.056344313046592789972e+00,
        -0.042269164765363603212e+00, -0.028184648949745694339e+00,
        -0.014093886410782462614e+00, 0.0e+00, 0.014093886410782462614e+00,
        0.028184648949745694339e+00, 0.042269164765363603212e+00,
        0.056344313046592789972e+00, 0.070406976042855179063e+00,
        0.084454040083710883710e+00, 0.098482396598119202090e+00,
        0.11248894313318662575e+00, 0.12647058437230196685e+00,
        0.14042423315256017459e+00, 0.15434681148137810869e+00,
        0.16823525155220746498e+00, 0.18208649675925219825e+00,
        0.19589750271110015392e+00, 0.20966523824318119477e+00,
        0.22338668642896688163e+00, 0.23705884558982972721e+00,
        0.25067873030348317661e+00, 0.26424337241092676194e+00,
        0.27774982202182431507e+00, 0.29119514851824668196e+00,
        0.30457644155671404334e+00, 0.31789081206847668318e+00,
        0.33113539325797683309e+00, 0.34430734159943802278e+00,
        0.35740383783153215238e+00, 0.37042208795007823014e+00,
        0.38335932419873034692e+00, 0.39621280605761593918e+00,
        0.40897982122988867241e+00, 0.42165768662616330006e+00,
        0.43424374934680255800e+00, 0.44673538766202847374e+00,
        0.45913001198983233287e+00, 0.47142506587165887693e+00,
        0.48361802694584102756e+00, 0.49570640791876146017e+00,
        0.50768775753371660215e+00, 0.51955966153745702199e+00,
        0.53131974364437562397e+00, 0.54296566649831149049e+00,
        0.55449513263193254887e+00, 0.56590588542365442262e+00,
        0.57719571005204581484e+00, 0.58836243444766254143e+00,
        0.59940393024224289297e+00, 0.61031811371518640016e+00,
        0.62110294673722640294e+00, 0.63175643771119423041e+00,
        0.64227664250975951377e+00, 0.65266166541001749610e+00,
        0.66290966002478059546e+00, 0.67301883023041847920e+00,
        0.68298743109107922809e+00, 0.69281376977911470289e+00,
        0.70249620649152707861e+00, 0.71203315536225203459e+00,
        0.72142308537009891548e+00, 0.73066452124218126133e+00,
        0.73975604435269475868e+00, 0.74869629361693660282e+00,
        0.75748396638051363793e+00, 0.76611781930376009072e+00,
        0.77459666924148337704e+00, 0.78291939411828301639e+00,
        0.79108493379984836143e+00, 0.79909229096084140180e+00,
        0.80694053195021761186e+00, 0.81462878765513741344e+00,
        0.82215625436498040737e+00, 0.82952219463740140018e+00,
        0.83672593816886873550e+00, 0.84376688267270860104e+00,
        0.85064449476835027976e+00, 0.85735831088623215653e+00,
        0.86390793819369047715e+00, 0.87029305554811390585e+00,
        0.87651341448470526974e+00, 0.88256884024734190684e+00,
        0.88845923287225699889e+00, 0.89418456833555902286e+00,
        0.89974489977694003664e+00, 0.90514035881326159519e+00,
        0.91037115695700429250e+00, 0.91543758715576504064e+00,
        0.92034002547001242073e+00, 0.92507893290707565236e+00,
        0.92965485742974005667e+00, 0.93406843615772578800e+00,
        0.93832039777959288365e+00, 0.94241156519108305981e+00,
        0.94634285837340290515e+00, 0.95011529752129487656e+00,
        0.95373000642576113641e+00, 0.95718821610986096274e+00,
        0.96049126870802028342e+00, 0.96364062156981213252e+00,
        0.96663785155841656709e+00, 0.96948465950245923177e+00,
        0.97218287474858179658e+00, 0.97473445975240266776e+00,
        0.97714151463970571416e+00, 0.97940628167086268381e+00,
        0.98153114955374010687e+00, 0.98351865757863272876e+00,
        0.98537149959852037111e+00, 0.98709252795403406719e+00,
        0.98868475754742947994e+00, 0.99015137040077015918e+00,
        0.99149572117810613240e+00, 0.99272134428278861533e+00,
        0.99383196321275502221e+00, 0.99483150280062100052e+00,
        0.99572410469840718851e+00, 0.99651414591489027385e+00,
        0.99720625937222195908e+00, 0.99780535449595727456e+00,
        0.99831663531840739253e+00, 0.99874561446809511470e+00,
        0.99909812496766759766e+00, 0.99938033802502358193e+00,
        0.99959879967191068325e+00, 0.99976049092443204733e+00,
        0.99987288812035761194e+00, 0.99994399620705437576e+00,
        0.99998243035489159858e+00, 0.99999759637974846462e+00,
    ), (
        0.69379364324108267170e-05, 0.25157870384280661489e-04,
        0.53275293669780613125e-04, 0.90372734658751149261e-04,
        0.13575491094922871973e-03, 0.18887326450650491366e-03,
        0.24921240048299729402e-03, 0.31630366082226447689e-03,
        0.38974528447328229322e-03, 0.46918492424785040975e-03,
        0.55429531493037471492e-03, 0.64476204130572477933e-03,
        0.74028280424450333046e-03, 0.84057143271072246365e-03,
        0.94536151685852538246e-03, 0.10544076228633167722e-02,
        0.11674841174299594077e-02, 0.12843824718970101768e-02,
        0.14049079956551446427e-02, 0.15288767050877655684e-02,
        0.16561127281544526052e-02, 0.17864463917586498247e-02,
        0.19197129710138724125e-02, 0.20557519893273465236e-02,
        0.21944069253638388388e-02, 0.23355251860571608737e-02,
        0.24789582266575679307e-02, 0.26245617274044295626e-02,
        0.27721957645934509940e-02, 0.29217249379178197538e-02,
        0.30730184347025783234e-02, 0.32259500250878684614e-02,
        0.33803979910869203823e-02, 0.35362449977167777340e-02,
        0.36933779170256508183e-02, 0.38516876166398709241e-02,
        0.40110687240750233989e-02, 0.41714193769840788528e-02,
        0.43326409680929828545e-02, 0.44946378920320678616e-02,
        0.46573172997568547773e-02, 0.48205888648512683476e-02,
        0.49843645647655386012e-02, 0.51485584789781777618e-02,
        0.53130866051870565663e-02, 0.54778666939189508240e-02,
        0.56428181013844441585e-02, 0.58078616599775673635e-02,
        0.59729195655081658049e-02, 0.61379152800413850435e-02,
        0.63027734490857587172e-02, 0.64674198318036867274e-02,
        0.66317812429018878941e-02, 0.67957855048827733948e-02,
        0.69593614093904229394e-02, 0.71224386864583871532e-02,
        0.72849479805538070639e-02, 0.74468208324075910174e-02,
        0.76079896657190565832e-02, 0.77683877779219912200e-02,
        0.79279493342948491103e-02, 0.80866093647888599710e-02,
        0.82443037630328680306e-02, 0.84009692870519326354e-02,
        0.85565435613076896192e-02, 0.87109650797320868736e-02,
        0.88641732094824942641e-02, 0.90161081951956431600e-02,
        0.91667111635607884067e-02, 0.93159241280693950932e-02,
        0.94636899938300652943e-02, 0.96099525623638830097e-02,
        0.97546565363174114611e-02, 0.98977475240487497440e-02,
        0.10039172044056840798e-01, 0.10178877529236079733e-01,
        0.10316812330947621682e-01, 0.10452925722906011926e-01,
        0.10587167904885197931e-01, 0.10719490006251933623e-01,
        0.10849844089337314099e-01, 0.10978183152658912470e-01,
        0.11104461134006926537e-01, 0.11228632913408049354e-01,
        0.11350654315980596602e-01, 0.11470482114693874380e-01,
        0.11588074033043952568e-01, 0.11703388747657003101e-01,
        0.11816385890830235763e-01, 0.11927026053019270040e-01,
        0.12035270785279562630e-01, 0.12141082601668299679e-01,
        0.12244424981611985899e-01, 0.12345262372243838455e-01,
        0.12443560190714035263e-01, 0.12539284826474884353e-01,
        0.12632403643542078765e-01, 0.12722884982732382906e-01,
        0.12810698163877361967e-01, 0.12895813488012114694e-01,
        0.12978202239537399286e-01, 0.13057836688353048840e-01,
        0.13134690091960152836e-01, 0.13208736697529129966e-01,
        0.13279951743930530650e-01, 0.13348311463725179953e-01,
        0.13413793085110098513e-01, 0.13476374833816515982e-01,
        0.13536035934956213614e-01, 0.13592756614812395910e-01,
        0.13646518102571291428e-01, 0.13697302631990716258e-01,
        0.13745093443001896632e-01, 0.13789874783240936517e-01,
        0.13831631909506428676e-01, 0.13870351089139840997e-01,
        0.13906019601325461264e-01, 0.13938625738306850804e-01,
        0.13968158806516938516e-01, 0.13994609127619079852e-01,
        0.14017968039456608810e-01, 0.14038227896908623303e-01,
        0.14055382072649964277e-01, 0.14069424957813575318e-01,
        0.14080351962553661325e-01, 0.14088159516508301065e-01,
        0.14092845069160408355e-01, 0.14094407090096179347e-01,
        0.14092845069160408355e-01, 0.14088159516508301065e-01,
        0.14080351962553661325e-01, 0.14069424957813575318e-01,
        0.14055382072649964277e-01, 0.14038227896908623303e-01,
        0.14017968039456608810e-01, 0.13994609127619079852e-01,
        0.13968158806516938516e-01, 0.13938625738306850804e-01,
        0.13906019601325461264e-01, 0.13870351089139840997e-01,
        0.13831631909506428676e-01, 0.13789874783240936517e-01,
        0.13745093443001896632e-01, 0.13697302631990716258e-01,
        0.13646518102571291428e-01, 0.13592756614812395910e-01,
        0.13536035934956213614e-01, 0.13476374833816515982e-01,
        0.13413793085110098513e-01, 0.13348311463725179953e-01,
        0.13279951743930530650e-01, 0.13208736697529129966e-01,
        0.13134690091960152836e-01, 0.13057836688353048840e-01,
        0.12978202239537399286e-01, 0.12895813488012114694e-01,
        0.12810698163877361967e-01, 0.12722884982732382906e-01,
        0.12632403643542078765e-01, 0.12539284826474884353e-01,
        0.12443560190714035263e-01, 0.12345262372243838455e-01,
        0.12244424981611985899e-01, 0.12141082601668299679e-01,
        0.12035270785279562630e-01, 0.11927026053019270040e-01,
        0.11816385890830235763e-01, 0.11703388747657003101e-01,
        0.11588074033043952568e-01, 0.11470482114693874380e-01,
        0.11350654315980596602e-01, 0.11228632913408049354e-01,
        0.11104461134006926537e-01, 0.10978183152658912470e-01,
        0.10849844089337314099e-01, 0.10719490006251933623e-01,
        0.10587167904885197931e-01, 0.10452925722906011926e-01,
        0.10316812330947621682e-01, 0.10178877529236079733e-01,
        0.10039172044056840798e-01, 0.98977475240487497440e-02,
        0.97546565363174114611e-02, 0.96099525623638830097e-02,
        0.94636899938300652943e-02, 0.93159241280693950932e-02,
        0.91667111635607884067e-02, 0.90161081951956431600e-02,
        0.88641732094824942641e-02, 0.87109650797320868736e-02,
        0.85565435613076896192e-02, 0.84009692870519326354e-02,
        0.82443037630328680306e-02, 0.80866093647888599710e-02,
        0.79279493342948491103e-02, 0.77683877779219912200e-02,
        0.76079896657190565832e-02, 0.74468208324075910174e-02,
        0.72849479805538070639e-02, 0.71224386864583871532e-02,
        0.69593614093904229394e-02, 0.67957855048827733948e-02,
        0.66317812429018878941e-02, 0.64674198318036867274e-02,
        0.63027734490857587172e-02, 0.61379152800413850435e-02,
        0.59729195655081658049e-02, 0.58078616599775673635e-02,
        0.56428181013844441585e-02, 0.54778666939189508240e-02,
        0.53130866051870565663e-02, 0.51485584789781777618e-02,
        0.49843645647655386012e-02, 0.48205888648512683476e-02,
        0.46573172997568547773e-02, 0.44946378920320678616e-02,
        0.43326409680929828545e-02, 0.41714193769840788528e-02,
        0.40110687240750233989e-02, 0.38516876166398709241e-02,
        0.36933779170256508183e-02, 0.35362449977167777340e-02,
        0.33803979910869203823e-02, 0.32259500250878684614e-02,
        0.30730184347025783234e-02, 0.29217249379178197538e-02,
        0.27721957645934509940e-02, 0.26245617274044295626e-02,
        0.24789582266575679307e-02, 0.23355251860571608737e-02,
        0.21944069253638388388e-02, 0.20557519893273465236e-02,
        0.19197129710138724125e-02, 0.17864463917586498247e-02,
        0.16561127281544526052e-02, 0.15288767050877655684e-02,
        0.14049079956551446427e-02, 0.12843824718970101768e-02,
        0.11674841174299594077e-02, 0.10544076228633167722e-02,
        0.94536151685852538246e-03, 0.84057143271072246365e-03,
        0.74028280424450333046e-03, 0.64476204130572477933e-03,
        0.55429531493037471492e-03, 0.46918492424785040975e-03,
        0.38974528447328229322e-03, 0.31630366082226447689e-03,
        0.24921240048299729402e-03, 0.18887326450650491366e-03,
        0.13575491094922871973e-03, 0.90372734658751149261e-04,
        0.53275293669780613125e-04, 0.25157870384280661489e-04,
        0.69379364324108267170e-05,
    )),
    511 : ((
        -0.999999672956734384381e+00, -0.999997596379748464620e+00,
        -0.999992298136257588028e+00, -0.999982430354891598580e+00,
        -0.999966730098486276883e+00, -0.999943996207054375764e+00,
        -0.999913081144678282800e+00, -0.999872888120357611938e+00,
        -0.999822363679787739196e+00, -0.999760490924432047330e+00,
        -0.999686286448317731776e+00, -0.999598799671910683252e+00,
        -0.999497112467187190535e+00, -0.999380338025023581928e+00,
        -0.999247618943342473599e+00, -0.999098124967667597662e+00,
        -0.998931050830810562236e+00, -0.998745614468095114704e+00,
        -0.998541055697167906027e+00, -0.998316635318407392531e+00,
        -0.998071634524930323302e+00, -0.997805354495957274562e+00,
        -0.997517116063472399965e+00, -0.997206259372221959076e+00,
        -0.996872143485260161299e+00, -0.996514145914890273849e+00,
        -0.996131662079315037786e+00, -0.995724104698407188509e+00,
        -0.995290903148810302261e+00, -0.994831502800621000519e+00,
        -0.994345364356723405931e+00, -0.993831963212755022209e+00,
        -0.993290788851684966211e+00, -0.992721344282788615328e+00,
        -0.992123145530863117683e+00, -0.991495721178106132399e+00,
        -0.990838611958294243677e+00, -0.990151370400770159181e+00,
        -0.989433560520240838716e+00, -0.988684757547429479939e+00,
        -0.987904547695124280467e+00, -0.987092527954034067190e+00,
        -0.986248305913007552681e+00, -0.985371499598520371114e+00,
        -0.984461737328814534596e+00, -0.983518657578632728762e+00,
        -0.982541908851080604251e+00, -0.981531149553740106867e+00,
        -0.980486047876721339416e+00, -0.979406281670862683806e+00,
        -0.978291538324758539526e+00, -0.977141514639705714156e+00,
        -0.975955916702011753129e+00, -0.974734459752402667761e+00,
        -0.973476868052506926773e+00, -0.972182874748581796578e+00,
        -0.970852221732792443256e+00, -0.969484659502459231771e+00,
        -0.968079947017759947964e+00, -0.966637851558416567092e+00,
        -0.965158148579915665979e+00, -0.963640621569812132521e+00,
        -0.962085061904651475741e+00, -0.960491268708020283423e+00,
        -0.958859048710200221356e+00, -0.957188216109860962736e+00,
        -0.955478592438183697574e+00, -0.953730006425761136415e+00,
        -0.951942293872573589498e+00, -0.950115297521294876558e+00,
        -0.948248866934137357063e+00, -0.946342858373402905148e+00,
        -0.944397134685866648591e+00, -0.942411565191083059813e+00,
        -0.940386025573669721370e+00, -0.938320397779592883655e+00,
        -0.936214569916450806625e+00, -0.934068436157725787999e+00,
        -0.931881896650953639345e+00, -0.929654857429740056670e+00,
        -0.927387230329536696843e+00, -0.925078932907075652364e+00,
        -0.922729888363349241523e+00, -0.920340025470012420730e+00,
        -0.917909278499077501636e+00, -0.915437587155765040644e+00,
        -0.912924896514370590080e+00, -0.910371156957004292498e+00,
        -0.907776324115058903624e+00, -0.905140358813261595189e+00,
        -0.902463227016165675048e+00, -0.899744899776940036639e+00,
        -0.896985353188316590376e+00, -0.894184568335559022859e+00,
        -0.891342531251319871666e+00, -0.888459232872256998890e+00,
        -0.885534668997285008926e+00, -0.882568840247341906842e+00,
        -0.879561752026556262568e+00, -0.876513414484705269742e+00,
        -0.873423842480859310192e+00, -0.870293055548113905851e+00,
        -0.867121077859315215614e+00, -0.863907938193690477146e+00,
        -0.860653669904299969802e+00, -0.857358310886232156525e+00,
        -0.854021903545468625813e+00, -0.850644494768350279758e+00,
        -0.847226135891580884381e+00, -0.843766882672708601038e+00,
        -0.840266795261030442350e+00, -0.836725938168868735503e+00,
        -0.833144380243172624728e+00, -0.829522194637401400178e+00,
        -0.825859458783650001088e+00, -0.822156254364980407373e+00,
        -0.818412667287925807395e+00, -0.814628787655137413436e+00,
        -0.810804709738146594361e+00, -0.806940531950217611856e+00,
        -0.803036356819268687782e+00, -0.799092290960841401800e+00,
        -0.795108445051100526780e+00, -0.791084933799848361435e+00,
        -0.787021875923539422170e+00, -0.782919394118283016385e+00,
        -0.778777615032822744702e+00, -0.774596669241483377036e+00,
        -0.770376691217076824278e+00, -0.766117819303760090717e+00,
        -0.761820195689839149173e+00, -0.757483966380513637926e+00,
        -0.753109281170558142523e+00, -0.748696293616936602823e+00,
        -0.744245161011347082309e+00, -0.739756044352694758677e+00,
        -0.735229108319491547663e+00, -0.730664521242181261329e+00,
        -0.726062455075389632685e+00, -0.721423085370098915485e+00,
        -0.716746591245747095767e+00, -0.712033155362252034587e+00,
        -0.707282963891961103412e+00, -0.702496206491527078610e+00,
        -0.697673076273711232906e+00, -0.692813769779114702895e+00,
        -0.687918486947839325756e+00, -0.682987431091079228087e+00,
        -0.678020808862644517838e+00, -0.673018830230418479199e+00,
        -0.667981708447749702165e+00, -0.662909660024780595461e+00,
        -0.657802904699713735422e+00, -0.652661665410017496101e+00,
        -0.647486168263572388782e+00, -0.642276642509759513774e+00,
        -0.637033320510492495071e+00, -0.631756437711194230414e+00,
        -0.626446232611719746542e+00, -0.621102946737226402941e+00,
        -0.615726824608992638014e+00, -0.610318113715186400156e+00,
        -0.604877064481584353319e+00, -0.599403930242242892974e+00,
        -0.593898967210121954393e+00, -0.588362434447662541434e+00,
        -0.582794593837318850840e+00, -0.577195710052045814844e+00,
        -0.571566050525742833992e+00, -0.565905885423654422623e+00,
        -0.560215487612728441818e+00, -0.554495132631932548866e+00,
        -0.548745098662529448608e+00, -0.542965666498311490492e+00,
        -0.537157119515795115982e+00, -0.531319743644375623972e+00,
        -0.525453827336442687395e+00, -0.519559661537457021993e+00,
        -0.513637539655988578507e+00, -0.507687757533716602155e+00,
        -0.501710613415391878251e+00, -0.495706407918761460170e+00,
        -0.489675444004456155436e+00, -0.483618026945841027562e+00,
        -0.477534464298829155284e+00, -0.471425065871658876934e+00,
        -0.465290143694634735858e+00, -0.459130011989832332874e+00,
        -0.452944987140767283784e+00, -0.446735387662028473742e+00,
        -0.440501534168875795783e+00, -0.434243749346802558002e+00,
        -0.427962357921062742583e+00, -0.421657686626163300056e+00,
        -0.415330064175321663764e+00, -0.408979821229888672409e+00,
        -0.402607290368737092671e+00, -0.396212806057615939183e+00,
        -0.389796704618470795479e+00, -0.383359324198730346916e+00,
        -0.376901004740559344802e+00, -0.370422087950078230138e+00,
        -0.363922917266549655269e+00, -0.357403837831532152376e+00,
        -0.350865196458001209011e+00, -0.344307341599438022777e+00,
        -0.337730623318886219621e+00, -0.331135393257976833093e+00,
        -0.324522004605921855207e+00, -0.317890812068476683182e+00,
        -0.311242171836871800300e+00, -0.304576441556714043335e+00,
        -0.297893980296857823437e+00, -0.291195148518246681964e+00,
        -0.284480308042725577496e+00, -0.277749822021824315065e+00,
        -0.271004054905512543536e+00, -0.264243372410926761945e+00,
        -0.257468141491069790481e+00, -0.250678730303483176613e+00,
        -0.243875508178893021593e+00, -0.237058845589829727213e+00,
        -0.230229114119222177156e+00, -0.223386686428966881628e+00,
        -0.216531936228472628081e+00, -0.209665238243181194766e+00,
        -0.202786968183064697557e+00, -0.195897502711100153915e+00,
        -0.188997219411721861059e+00, -0.182086496759252198246e+00,
        -0.175165714086311475707e+00, -0.168235251552207464982e+00,
        -0.161295490111305257361e+00, -0.154346811481378108692e+00,
        -0.147389598111939940054e+00, -0.140424233152560174594e+00,
        -0.133451100421161601344e+00, -0.126470584372301966851e+00,
        -0.119483070065440005133e+00, -0.112488943133186625746e+00,
        -0.105488589749541988533e+00, -0.984823965981192020903e-01,
        -0.914707508403553909095e-01, -0.844540400837108837102e-01,
        -0.774326523498572825675e-01, -0.704069760428551790633e-01,
        -0.633773999173222898797e-01, -0.563443130465927899720e-01,
        -0.493081047908686267156e-01, -0.422691647653636032124e-01,
        -0.352278828084410232603e-01, -0.281846489497456943394e-01,
        -0.211398533783310883350e-01, -0.140938864107824626142e-01,
        -0.704713845933674648514e-02, +0.000000000000000000000e+00,
        +0.704713845933674648514e-02, +0.140938864107824626142e-01,
        +0.211398533783310883350e-01, +0.281846489497456943394e-01,
        +0.352278828084410232603e-01, +0.422691647653636032124e-01,
        +0.493081047908686267156e-01, +0.563443130465927899720e-01,
        +0.633773999173222898797e-01, +0.704069760428551790633e-01,
        +0.774326523498572825675e-01, +0.844540400837108837102e-01,
        +0.914707508403553909095e-01, +0.984823965981192020903e-01,
        +0.105488589749541988533e+00, +0.112488943133186625746e+00,
        +0.119483070065440005133e+00, +0.126470584372301966851e+00,
        +0.133451100421161601344e+00, +0.140424233152560174594e+00,
        +0.147389598111939940054e+00, +0.154346811481378108692e+00,
        +0.161295490111305257361e+00, +0.168235251552207464982e+00,
        +0.175165714086311475707e+00, +0.182086496759252198246e+00,
        +0.188997219411721861059e+00, +0.195897502711100153915e+00,
        +0.202786968183064697557e+00, +0.209665238243181194766e+00,
        +0.216531936228472628081e+00, +0.223386686428966881628e+00,
        +0.230229114119222177156e+00, +0.237058845589829727213e+00,
        +0.243875508178893021593e+00, +0.250678730303483176613e+00,
        +0.257468141491069790481e+00, +0.264243372410926761945e+00,
        +0.271004054905512543536e+00, +0.277749822021824315065e+00,
        +0.284480308042725577496e+00, +0.291195148518246681964e+00,
        +0.297893980296857823437e+00, +0.304576441556714043335e+00,
        +0.311242171836871800300e+00, +0.317890812068476683182e+00,
        +0.324522004605921855207e+00, +0.331135393257976833093e+00,
        +0.337730623318886219621e+00, +0.344307341599438022777e+00,
        +0.350865196458001209011e+00, +0.357403837831532152376e+00,
        +0.363922917266549655269e+00, +0.370422087950078230138e+00,
        +0.376901004740559344802e+00, +0.383359324198730346916e+00,
        +0.389796704618470795479e+00, +0.396212806057615939183e+00,
        +0.402607290368737092671e+00, +0.408979821229888672409e+00,
        +0.415330064175321663764e+00, +0.421657686626163300056e+00,
        +0.427962357921062742583e+00, +0.434243749346802558002e+00,
        +0.440501534168875795783e+00, +0.446735387662028473742e+00,
        +0.452944987140767283784e+00, +0.459130011989832332874e+00,
        +0.465290143694634735858e+00, +0.471425065871658876934e+00,
        +0.477534464298829155284e+00, +0.483618026945841027562e+00,
        +0.489675444004456155436e+00, +0.495706407918761460170e+00,
        +0.501710613415391878251e+00, +0.507687757533716602155e+00,
        +0.513637539655988578507e+00, +0.519559661537457021993e+00,
        +0.525453827336442687395e+00, +0.531319743644375623972e+00,
        +0.537157119515795115982e+00, +0.542965666498311490492e+00,
        +0.548745098662529448608e+00, +0.554495132631932548866e+00,
        +0.560215487612728441818e+00, +0.565905885423654422623e+00,
        +0.571566050525742833992e+00, +0.577195710052045814844e+00,
        +0.582794593837318850840e+00, +0.588362434447662541434e+00,
        +0.593898967210121954393e+00, +0.599403930242242892974e+00,
        +0.604877064481584353319e+00, +0.610318113715186400156e+00,
        +0.615726824608992638014e+00, +0.621102946737226402941e+00,
        +0.626446232611719746542e+00, +0.631756437711194230414e+00,
        +0.637033320510492495071e+00, +0.642276642509759513774e+00,
        +0.647486168263572388782e+00, +0.652661665410017496101e+00,
        +0.657802904699713735422e+00, +0.662909660024780595461e+00,
        +0.667981708447749702165e+00, +0.673018830230418479199e+00,
        +0.678020808862644517838e+00, +0.682987431091079228087e+00,
        +0.687918486947839325756e+00, +0.692813769779114702895e+00,
        +0.697673076273711232906e+00, +0.702496206491527078610e+00,
        +0.707282963891961103412e+00, +0.712033155362252034587e+00,
        +0.716746591245747095767e+00, +0.721423085370098915485e+00,
        +0.726062455075389632685e+00, +0.730664521242181261329e+00,
        +0.735229108319491547663e+00, +0.739756044352694758677e+00,
        +0.744245161011347082309e+00, +0.748696293616936602823e+00,
        +0.753109281170558142523e+00, +0.757483966380513637926e+00,
        +0.761820195689839149173e+00, +0.766117819303760090717e+00,
        +0.770376691217076824278e+00, +0.774596669241483377036e+00,
        +0.778777615032822744702e+00, +0.782919394118283016385e+00,
        +0.787021875923539422170e+00, +0.791084933799848361435e+00,
        +0.795108445051100526780e+00, +0.799092290960841401800e+00,
        +0.803036356819268687782e+00, +0.806940531950217611856e+00,
        +0.810804709738146594361e+00, +0.814628787655137413436e+00,
        +0.818412667287925807395e+00, +0.822156254364980407373e+00,
        +0.825859458783650001088e+00, +0.829522194637401400178e+00,
        +0.833144380243172624728e+00, +0.836725938168868735503e+00,
        +0.840266795261030442350e+00, +0.843766882672708601038e+00,
        +0.847226135891580884381e+00, +0.850644494768350279758e+00,
        +0.854021903545468625813e+00, +0.857358310886232156525e+00,
        +0.860653669904299969802e+00, +0.863907938193690477146e+00,
        +0.867121077859315215614e+00, +0.870293055548113905851e+00,
        +0.873423842480859310192e+00, +0.876513414484705269742e+00,
        +0.879561752026556262568e+00, +0.882568840247341906842e+00,
        +0.885534668997285008926e+00, +0.888459232872256998890e+00,
        +0.891342531251319871666e+00, +0.894184568335559022859e+00,
        +0.896985353188316590376e+00, +0.899744899776940036639e+00,
        +0.902463227016165675048e+00, +0.905140358813261595189e+00,
        +0.907776324115058903624e+00, +0.910371156957004292498e+00,
        +0.912924896514370590080e+00, +0.915437587155765040644e+00,
        +0.917909278499077501636e+00, +0.920340025470012420730e+00,
        +0.922729888363349241523e+00, +0.925078932907075652364e+00,
        +0.927387230329536696843e+00, +0.929654857429740056670e+00,
        +0.931881896650953639345e+00, +0.934068436157725787999e+00,
        +0.936214569916450806625e+00, +0.938320397779592883655e+00,
        +0.940386025573669721370e+00, +0.942411565191083059813e+00,
        +0.944397134685866648591e+00, +0.946342858373402905148e+00,
        +0.948248866934137357063e+00, +0.950115297521294876558e+00,
        +0.951942293872573589498e+00, +0.953730006425761136415e+00,
        +0.955478592438183697574e+00, +0.957188216109860962736e+00,
        +0.958859048710200221356e+00, +0.960491268708020283423e+00,
        +0.962085061904651475741e+00, +0.963640621569812132521e+00,
        +0.965158148579915665979e+00, +0.966637851558416567092e+00,
        +0.968079947017759947964e+00, +0.969484659502459231771e+00,
        +0.970852221732792443256e+00, +0.972182874748581796578e+00,
        +0.973476868052506926773e+00, +0.974734459752402667761e+00,
        +0.975955916702011753129e+00, +0.977141514639705714156e+00,
        +0.978291538324758539526e+00, +0.979406281670862683806e+00,
        +0.980486047876721339416e+00, +0.981531149553740106867e+00,
        +0.982541908851080604251e+00, +0.983518657578632728762e+00,
        +0.984461737328814534596e+00, +0.985371499598520371114e+00,
        +0.986248305913007552681e+00, +0.987092527954034067190e+00,
        +0.987904547695124280467e+00, +0.988684757547429479939e+00,
        +0.989433560520240838716e+00, +0.990151370400770159181e+00,
        +0.990838611958294243677e+00, +0.991495721178106132399e+00,
        +0.992123145530863117683e+00, +0.992721344282788615328e+00,
        +0.993290788851684966211e+00, +0.993831963212755022209e+00,
        +0.994345364356723405931e+00, +0.994831502800621000519e+00,
        +0.995290903148810302261e+00, +0.995724104698407188509e+00,
        +0.996131662079315037786e+00, +0.996514145914890273849e+00,
        +0.996872143485260161299e+00, +0.997206259372221959076e+00,
        +0.997517116063472399965e+00, +0.997805354495957274562e+00,
        +0.998071634524930323302e+00, +0.998316635318407392531e+00,
        +0.998541055697167906027e+00, +0.998745614468095114704e+00,
        +0.998931050830810562236e+00, +0.999098124967667597662e+00,
        +0.999247618943342473599e+00, +0.999380338025023581928e+00,
        +0.999497112467187190535e+00, +0.999598799671910683252e+00,
        +0.999686286448317731776e+00, +0.999760490924432047330e+00,
        +0.999822363679787739196e+00, +0.999872888120357611938e+00,
        +0.999913081144678282800e+00, +0.999943996207054375764e+00,
        +0.999966730098486276883e+00, +0.999982430354891598580e+00,
        +0.999992298136257588028e+00, +0.999997596379748464620e+00,
        +0.999999672956734384381e+00,
    ), (
        0.945715933950007048827e-06, 0.345456507169149134898e-05,
        0.736624069102321668857e-05, 0.125792781889592743525e-04,
        0.190213681905875816679e-04, 0.266376412339000901358e-04,
        0.353751372055189588628e-04, 0.451863674126296143105e-04,
        0.560319507856164252140e-04, 0.678774554733972416227e-04,
        0.806899228014035293851e-04, 0.944366322532705527066e-04,
        0.109085545645741522051e-03, 0.124606200241498368482e-03,
        0.140970302204104791413e-03, 0.158151830411132242924e-03,
        0.176126765545083195474e-03, 0.194872642236641146532e-03,
        0.214368090034216937149e-03, 0.234592462123925204879e-03,
        0.255525589595236862014e-03, 0.277147657465187357459e-03,
        0.299439176850911730874e-03, 0.322381020652862389664e-03,
        0.345954492129903871350e-03, 0.370141402122251665232e-03,
        0.394924138246873704434e-03, 0.420285716355361231823e-03,
        0.446209810101403247488e-03, 0.472680758429262691232e-03,
        0.499683553312800484519e-03, 0.527203811431658386125e-03,
        0.555227733977307579715e-03, 0.583742058714979703847e-03,
        0.612734008012225209294e-03, 0.642191235948505088403e-03,
        0.672101776960108194646e-03, 0.702453997827572321358e-03,
        0.733236554224767912055e-03, 0.764438352543882784191e-03,
        0.796048517297550871506e-03, 0.828056364077226302608e-03,
        0.860451377808527848128e-03, 0.893223195879324912340e-03,
        0.926361595613111283368e-03, 0.959856485506936206261e-03,
        0.993697899638760857945e-03, 0.102787599466367326179e-02,
        0.106238104885340071375e-02, 0.109720346268191941940e-02,
        0.113233376051597664917e-02, 0.116776259302858043685e-02,
        0.120348074001265964881e-02, 0.123947911332878396534e-02,
        0.127574875977346947345e-02, 0.131228086370221478128e-02,
        0.134906674928353113127e-02, 0.138609788229672549700e-02,
        0.142336587141720519900e-02, 0.146086246895890987689e-02,
        0.149857957106456636214e-02, 0.153650921735128916170e-02,
        0.157464359003212166189e-02, 0.161297501254393423070e-02,
        0.165149594771914570655e-02, 0.169019899554346019117e-02,
        0.172907689054461607168e-02, 0.176812249885838886701e-02,
        0.180732881501808930079e-02, 0.184668895851282540913e-02,
        0.188619617015808475394e-02, 0.192584380831993546204e-02,
        0.196562534503150547732e-02, 0.200553436203751169944e-02,
        0.204556454679958293446e-02, 0.208570968849203942640e-02,
        0.212596367401472533045e-02, 0.216632048404649142727e-02,
        0.220677418916003329194e-02, 0.224731894601603393082e-02,
        0.228794899365195972378e-02, 0.232865864987842738864e-02,
        0.236944230779380495146e-02, 0.241029443242563417382e-02,
        0.245120955750556483923e-02, 0.249218228238276930060e-02,
        0.253320726907925325750e-02, 0.257427923948908888092e-02,
        0.261539297272236109225e-02, 0.265654330259352828314e-02,
        0.269772511525294586667e-02, 0.273893334695947541201e-02,
        0.278016298199139435045e-02, 0.282140905069222207923e-02,
        0.286266662764757868253e-02, 0.290393082998878368175e-02,
        0.294519681581857582284e-02, 0.298645978275408290247e-02,
        0.302771496658198544480e-02, 0.306895764002069252174e-02,
        0.311018311158427546158e-02, 0.315138672454287935858e-02,
        0.319256385597434736790e-02, 0.323370991590184336368e-02,
        0.327482034651233969564e-02, 0.331589062145094394706e-02,
        0.335691624518616761342e-02, 0.339789275244138669739e-02,
        0.343881570768790591876e-02, 0.347968070469521146972e-02,
        0.352048336613417922682e-02, 0.356121934322919357659e-02,
        0.360188431545532431869e-02, 0.364247399027690353194e-02,
        0.368298410292403911967e-02, 0.372341041620379550870e-02,
        0.376374872034296338241e-02, 0.380399483285952829161e-02,
        0.384414459846013158917e-02, 0.388419388896099560998e-02,
        0.392413860322995774660e-02, 0.396397466714742455513e-02,
        0.400369803358421688562e-02, 0.404330468239442998549e-02,
        0.408279062042157838350e-02, 0.412215188151643401528e-02,
        0.416138452656509745764e-02, 0.420048464352596631772e-02,
        0.423944834747438184434e-02, 0.427827178065384480959e-02,
        0.431695111253279479928e-02, 0.435548253986604343679e-02,
        0.439386228676004195260e-02, 0.443208660474124713206e-02,
        0.447015177282692726900e-02, 0.450805409759782158001e-02,
        0.454578991327213285488e-02, 0.458335558178039420335e-02,
        0.462074749284080687482e-02, 0.465796206403469754658e-02,
        0.469499574088179046532e-02, 0.473184499691503264714e-02,
        0.476850633375474925263e-02, 0.480497628118194150483e-02,
        0.484125139721057135214e-02, 0.487732826815870573054e-02,
        0.491320350871841897367e-02, 0.494887376202437487201e-02,
        0.498433569972103029914e-02, 0.501958602202842039909e-02,
        0.505462145780650125058e-02, 0.508943876461803986674e-02,
        0.512403472879005351831e-02, 0.515840616547381084096e-02,
        0.519254991870341614863e-02, 0.522646286145300596306e-02,
        0.526014189569259311205e-02, 0.529358395244259896547e-02,
        0.532678599182711857974e-02, 0.535974500312596681161e-02,
        0.539245800482555593606e-02, 0.542492204466865704951e-02,
        0.545713419970309863995e-02, 0.548909157632945623482e-02,
        0.552079131034778706457e-02, 0.555223056700346326850e-02,
        0.558340654103215637610e-02, 0.561431645670402467678e-02,
        0.564495756786715368885e-02, 0.567532715799029830087e-02,
        0.570542254020497332312e-02, 0.573524105734693719020e-02,
        0.576478008199711142954e-02, 0.579403701652197628421e-02,
        0.582300929311348057702e-02, 0.585169437382850155033e-02,
        0.588008975062788803205e-02, 0.590819294541511788161e-02,
        0.593600151007459827614e-02, 0.596351302650963502011e-02,
        0.599072510668009471472e-02, 0.601763539263978131522e-02,
        0.604424155657354634589e-02, 0.607054130083414983949e-02,
        0.609653235797888692923e-02, 0.612221249080599294931e-02,
        0.614757949239083790214e-02, 0.617263118612191922727e-02,
        0.619736542573665996342e-02, 0.622178009535701763157e-02,
        0.624587310952490748541e-02, 0.626964241323744217671e-02,
        0.629308598198198836688e-02, 0.631620182177103938227e-02,
        0.633898796917690165912e-02, 0.636144249136619145314e-02,
        0.638356348613413709795e-02, 0.640534908193868098342e-02,
        0.642679743793437438922e-02, 0.644790674400605734710e-02,
        0.646867522080231481688e-02, 0.648910111976869964292e-02,
        0.650918272318071200827e-02, 0.652891834417652442012e-02,
        0.654830632678944064054e-02, 0.656734504598007641819e-02,
        0.658603290766824937794e-02, 0.660436834876456498276e-02,
        0.662234983720168509457e-02, 0.663997587196526532519e-02,
        0.665724498312454708217e-02, 0.667415573186258997654e-02,
        0.669070671050613006584e-02, 0.670689654255504925648e-02,
        0.672272388271144108036e-02, 0.673818741690825799086e-02,
        0.675328586233752529078e-02, 0.676801796747810680683e-02,
        0.678238251212300746082e-02, 0.679637830740619795480e-02,
        0.681000419582894688374e-02, 0.682325905128564571420e-02,
        0.683614177908911221841e-02, 0.684865131599535812903e-02,
        0.686078663022780697951e-02, 0.687254672150094831613e-02,
        0.688393062104341470995e-02, 0.689493739162046825872e-02,
        0.690556612755588354803e-02, 0.691581595475321433825e-02,
        0.692568603071643155621e-02, 0.693517554456992049848e-02,
        0.694428371707782549438e-02, 0.695300980066273063177e-02,
        0.696135307942366551493e-02, 0.696931286915342540213e-02,
        0.697688851735519545845e-02, 0.698407940325846925786e-02,
        0.699088493783425207545e-02, 0.699730456380953992594e-02,
        0.700333775568106572820e-02, 0.700898401972830440494e-02,
        0.701424289402572916425e-02, 0.701911394845431165171e-02,
        0.702359678471225911031e-02, 0.702769103632498213858e-02,
        0.703139636865428709508e-02, 0.703471247890678765907e-02,
        0.703763909614153052319e-02, 0.704017598127683066242e-02,
        0.704232292709631209597e-02, 0.704407975825415053266e-02,
        0.704544633127951476780e-02, 0.704642253458020417748e-02,
        0.704700828844548013730e-02, 0.704720354504808967346e-02,
        0.704700828844548013730e-02, 0.704642253458020417748e-02,
        0.704544633127951476780e-02, 0.704407975825415053266e-02,
        0.704232292709631209597e-02, 0.704017598127683066242e-02,
        0.703763909614153052319e-02, 0.703471247890678765907e-02,
        0.703139636865428709508e-02, 0.702769103632498213858e-02,
        0.702359678471225911031e-02, 0.701911394845431165171e-02,
        0.701424289402572916425e-02, 0.700898401972830440494e-02,
        0.700333775568106572820e-02, 0.699730456380953992594e-02,
        0.699088493783425207545e-02, 0.698407940325846925786e-02,
        0.697688851735519545845e-02, 0.696931286915342540213e-02,
        0.696135307942366551493e-02, 0.695300980066273063177e-02,
        0.694428371707782549438e-02, 0.693517554456992049848e-02,
        0.692568603071643155621e-02, 0.691581595475321433825e-02,
        0.690556612755588354803e-02, 0.689493739162046825872e-02,
        0.688393062104341470995e-02, 0.687254672150094831613e-02,
        0.686078663022780697951e-02, 0.684865131599535812903e-02,
        0.683614177908911221841e-02, 0.682325905128564571420e-02,
        0.681000419582894688374e-02, 0.679637830740619795480e-02,
        0.678238251212300746082e-02, 0.676801796747810680683e-02,
        0.675328586233752529078e-02, 0.673818741690825799086e-02,
        0.672272388271144108036e-02, 0.670689654255504925648e-02,
        0.669070671050613006584e-02, 0.667415573186258997654e-02,
        0.665724498312454708217e-02, 0.663997587196526532519e-02,
        0.662234983720168509457e-02, 0.660436834876456498276e-02,
        0.658603290766824937794e-02, 0.656734504598007641819e-02,
        0.654830632678944064054e-02, 0.652891834417652442012e-02,
        0.650918272318071200827e-02, 0.648910111976869964292e-02,
        0.646867522080231481688e-02, 0.644790674400605734710e-02,
        0.642679743793437438922e-02, 0.640534908193868098342e-02,
        0.638356348613413709795e-02, 0.636144249136619145314e-02,
        0.633898796917690165912e-02, 0.631620182177103938227e-02,
        0.629308598198198836688e-02, 0.626964241323744217671e-02,
        0.624587310952490748541e-02, 0.622178009535701763157e-02,
        0.619736542573665996342e-02, 0.617263118612191922727e-02,
        0.614757949239083790214e-02, 0.612221249080599294931e-02,
        0.609653235797888692923e-02, 0.607054130083414983949e-02,
        0.604424155657354634589e-02, 0.601763539263978131522e-02,
        0.599072510668009471472e-02, 0.596351302650963502011e-02,
        0.593600151007459827614e-02, 0.590819294541511788161e-02,
        0.588008975062788803205e-02, 0.585169437382850155033e-02,
        0.582300929311348057702e-02, 0.579403701652197628421e-02,
        0.576478008199711142954e-02, 0.573524105734693719020e-02,
        0.570542254020497332312e-02, 0.567532715799029830087e-02,
        0.564495756786715368885e-02, 0.561431645670402467678e-02,
        0.558340654103215637610e-02, 0.555223056700346326850e-02,
        0.552079131034778706457e-02, 0.548909157632945623482e-02,
        0.545713419970309863995e-02, 0.542492204466865704951e-02,
        0.539245800482555593606e-02, 0.535974500312596681161e-02,
        0.532678599182711857974e-02, 0.529358395244259896547e-02,
        0.526014189569259311205e-02, 0.522646286145300596306e-02,
        0.519254991870341614863e-02, 0.515840616547381084096e-02,
        0.512403472879005351831e-02, 0.508943876461803986674e-02,
        0.505462145780650125058e-02, 0.501958602202842039909e-02,
        0.498433569972103029914e-02, 0.494887376202437487201e-02,
        0.491320350871841897367e-02, 0.487732826815870573054e-02,
        0.484125139721057135214e-02, 0.480497628118194150483e-02,
        0.476850633375474925263e-02, 0.473184499691503264714e-02,
        0.469499574088179046532e-02, 0.465796206403469754658e-02,
        0.462074749284080687482e-02, 0.458335558178039420335e-02,
        0.454578991327213285488e-02, 0.450805409759782158001e-02,
        0.447015177282692726900e-02, 0.443208660474124713206e-02,
        0.439386228676004195260e-02, 0.435548253986604343679e-02,
        0.431695111253279479928e-02, 0.427827178065384480959e-02,
        0.423944834747438184434e-02, 0.420048464352596631772e-02,
        0.416138452656509745764e-02, 0.412215188151643401528e-02,
        0.408279062042157838350e-02, 0.404330468239442998549e-02,
        0.400369803358421688562e-02, 0.396397466714742455513e-02,
        0.392413860322995774660e-02, 0.388419388896099560998e-02,
        0.384414459846013158917e-02, 0.380399483285952829161e-02,
        0.376374872034296338241e-02, 0.372341041620379550870e-02,
        0.368298410292403911967e-02, 0.364247399027690353194e-02,
        0.360188431545532431869e-02, 0.356121934322919357659e-02,
        0.352048336613417922682e-02, 0.347968070469521146972e-02,
        0.343881570768790591876e-02, 0.339789275244138669739e-02,
        0.335691624518616761342e-02, 0.331589062145094394706e-02,
        0.327482034651233969564e-02, 0.323370991590184336368e-02,
        0.319256385597434736790e-02, 0.315138672454287935858e-02,
        0.311018311158427546158e-02, 0.306895764002069252174e-02,
        0.302771496658198544480e-02, 0.298645978275408290247e-02,
        0.294519681581857582284e-02, 0.290393082998878368175e-02,
        0.286266662764757868253e-02, 0.282140905069222207923e-02,
        0.278016298199139435045e-02, 0.273893334695947541201e-02,
        0.269772511525294586667e-02, 0.265654330259352828314e-02,
        0.261539297272236109225e-02, 0.257427923948908888092e-02,
        0.253320726907925325750e-02, 0.249218228238276930060e-02,
        0.245120955750556483923e-02, 0.241029443242563417382e-02,
        0.236944230779380495146e-02, 0.232865864987842738864e-02,
        0.228794899365195972378e-02, 0.224731894601603393082e-02,
        0.220677418916003329194e-02, 0.216632048404649142727e-02,
        0.212596367401472533045e-02, 0.208570968849203942640e-02,
        0.204556454679958293446e-02, 0.200553436203751169944e-02,
        0.196562534503150547732e-02, 0.192584380831993546204e-02,
        0.188619617015808475394e-02, 0.184668895851282540913e-02,
        0.180732881501808930079e-02, 0.176812249885838886701e-02,
        0.172907689054461607168e-02, 0.169019899554346019117e-02,
        0.165149594771914570655e-02, 0.161297501254393423070e-02,
        0.157464359003212166189e-02, 0.153650921735128916170e-02,
        0.149857957106456636214e-02, 0.146086246895890987689e-02,
        0.142336587141720519900e-02, 0.138609788229672549700e-02,
        0.134906674928353113127e-02, 0.131228086370221478128e-02,
        0.127574875977346947345e-02, 0.123947911332878396534e-02,
        0.120348074001265964881e-02, 0.116776259302858043685e-02,
        0.113233376051597664917e-02, 0.109720346268191941940e-02,
        0.106238104885340071375e-02, 0.102787599466367326179e-02,
        0.993697899638760857945e-03, 0.959856485506936206261e-03,
        0.926361595613111283368e-03, 0.893223195879324912340e-03,
        0.860451377808527848128e-03, 0.828056364077226302608e-03,
        0.796048517297550871506e-03, 0.764438352543882784191e-03,
        0.733236554224767912055e-03, 0.702453997827572321358e-03,
        0.672101776960108194646e-03, 0.642191235948505088403e-03,
        0.612734008012225209294e-03, 0.583742058714979703847e-03,
        0.555227733977307579715e-03, 0.527203811431658386125e-03,
        0.499683553312800484519e-03, 0.472680758429262691232e-03,
        0.446209810101403247488e-03, 0.420285716355361231823e-03,
        0.394924138246873704434e-03, 0.370141402122251665232e-03,
        0.345954492129903871350e-03, 0.322381020652862389664e-03,
        0.299439176850911730874e-03, 0.277147657465187357459e-03,
        0.255525589595236862014e-03, 0.234592462123925204879e-03,
        0.214368090034216937149e-03, 0.194872642236641146532e-03,
        0.176126765545083195474e-03, 0.158151830411132242924e-03,
        0.140970302204104791413e-03, 0.124606200241498368482e-03,
        0.109085545645741522051e-03, 0.944366322532705527066e-04,
        0.806899228014035293851e-04, 0.678774554733972416227e-04,
        0.560319507856164252140e-04, 0.451863674126296143105e-04,
        0.353751372055189588628e-04, 0.266376412339000901358e-04,
        0.190213681905875816679e-04, 0.125792781889592743525e-04,
        0.736624069102321668857e-05, 0.345456507169149134898e-05,
        0.945715933950007048827e-06,
    ))
}
