"""Hermite Genz-Keister 18 rule."""
import numpy


def quad_genz_keister_18(order):
    """
    Hermite Genz-Keister 18 rule.

    Args:
        order (int):
            The quadrature order. Must be in the interval (0, 8).

    Returns:
        (:py:data:typing.Tuple[numpy.ndarray, numpy.ndarray]):
            Abscissas and weights

    Examples:
        >>> abscissas, weights = quad_genz_keister_18(1)
        >>> abscissas.round(4)
        array([-1.7321,  0.    ,  1.7321])
        >>> weights.round(4)
        array([0.1667, 0.6667, 0.1667])
    """
    order = sorted(GENZ_KEISTER_18.keys())[order]

    abscissas, weights = GENZ_KEISTER_18[order]
    abscissas = numpy.array(abscissas)
    weights = numpy.array(weights)

    weights /= numpy.sum(weights)
    abscissas *= numpy.sqrt(2)

    return abscissas, weights


GENZ_KEISTER_18 = {
    1 : ((
        0.0000000000000000E+00,
    ), (
        1.7724538509055159E+00,
    )),
    3 : ((
        -1.2247448713915889E+00,
        0.0000000000000000E+00,
        1.2247448713915889E+00,
    ), (
        2.9540897515091930E-01,
        1.1816359006036772E+00,
        2.9540897515091930E-01,
    )),
    9 : ((
        -2.9592107790638380E+00,
        -2.0232301911005157E+00,
        -1.2247448713915889E+00,
        -5.2403354748695763E-01,
        0.0000000000000000E+00,
        5.2403354748695763E-01,
        1.2247448713915889E+00,
        2.0232301911005157E+00,
        2.9592107790638380E+00,
    ), (
        1.6708826306882348E-04,
        1.4173117873979098E-02,
        1.6811892894767771E-01,
        4.7869428549114124E-01,
        4.5014700975378197E-01,
        4.7869428549114124E-01,
        1.6811892894767771E-01,
        1.4173117873979098E-02,
        1.6708826306882348E-04,
    )),
    19 : ((
        -4.4995993983103881E+00,
        -3.6677742159463378E+00,
        -2.9592107790638380E+00,
        -2.2665132620567876E+00,
        -2.0232301911005157E+00,
        -1.8357079751751868E+00,
        -1.2247448713915889E+00,
        -8.7004089535290285E-01,
        -5.2403354748695763E-01,
        0.0000000000000000E+00,
        5.2403354748695763E-01,
        8.7004089535290285E-01,
        1.2247448713915889E+00,
        1.8357079751751868E+00,
        2.0232301911005157E+00,
        2.2665132620567876E+00,
        2.9592107790638380E+00,
        3.6677742159463378E+00,
        4.4995993983103881E+00,
    ), (
        1.5295717705322357E-09,
        1.0802767206624762E-06,
        1.0656589772852267E-04,
        5.1133174390883855E-03,
        -1.1232438489069229E-02,
        3.2055243099445879E-02,
        1.1360729895748269E-01,
        1.0838861955003017E-01,
        3.6924643368920851E-01,
        5.3788160700510168E-01,
        3.6924643368920851E-01,
        1.0838861955003017E-01,
        1.1360729895748269E-01,
        3.2055243099445879E-02,
        -1.1232438489069229E-02,
        5.1133174390883855E-03,
        1.0656589772852267E-04,
        1.0802767206624762E-06,
        1.5295717705322357E-09,
    )),
    37 : ((
        -6.853200069757519,
        -6.124527854622158,
        -5.521865209868350,
        -4.986551454150765,
        -4.499599398310388,
        -4.057956316089741,
        -3.667774215946338,
        -3.315584617593290,
        -2.959210779063838,
        -2.597288631188366,
        -2.266513262056788,
        -2.023230191100516,
        -1.835707975175187,
        -1.561553427651873,
        -1.224744871391589,
        -0.870040895352903,
        -0.524033547486958,
        -0.214618180588171,
        0.000000000000000,
        0.214618180588171,
        0.524033547486958,
        0.870040895352903,
        1.224744871391589,
        1.561553427651873,
        1.835707975175187,
        2.023230191100516,
        2.266513262056788,
        2.597288631188366,
        2.959210779063838,
        3.315584617593290,
        3.667774215946338,
        4.057956316089741,
        4.499599398310388,
        4.986551454150765,
        5.521865209868350,
        6.124527854622158,
        6.853200069757519,
    ), (
        0.19030350940130498E-20,
        0.187781893143728947E-16,
        0.182242751549129356E-13,
        0.45661763676186859E-11,
        0.422525843963111041E-09,
        0.16595448809389819E-07,
        0.295907520230744049E-06,
        0.330975870979203419E-05,
        0.32265185983739747E-04,
        0.234940366465975222E-03,
        0.985827582996483824E-03,
        0.176802225818295443E-02,
        0.43334988122723492E-02,
        0.15513109874859354E-01,
        0.442116442189845444E-01,
        0.937208280655245902E-01,
        0.143099302896833389E+00,
        0.147655710402686249E+00,
        0.968824552928425499E-01,
        0.147655710402686249E+00,
        0.143099302896833389E+00,
        0.937208280655245902E-01,
        0.442116442189845444E-01,
        0.15513109874859354E-01,
        0.43334988122723492E-02,
        0.176802225818295443E-02,
        0.985827582996483824E-03,
        0.234940366465975222E-03,
        0.32265185983739747E-04,
        0.330975870979203419E-05,
        0.295907520230744049E-06,
        0.16595448809389819E-07,
        0.422525843963111041E-09,
        0.45661763676186859E-11,
        0.182242751549129356E-13,
        0.187781893143728947E-16,
        0.19030350940130498E-20,
    )),
}
