# -*- coding: utf-8 -*-
from bunq.sdk.exception.bunq_exception import BunqException
from bunq.sdk.http.api_client import ApiClient
from bunq.sdk.http.bunq_response import BunqResponse
from bunq.sdk.json import converter
from bunq.sdk.model.core.anchor_object_interface import AnchorObjectInterface
from bunq.sdk.model.core.bunq_model import BunqModel
from bunq.sdk.model.generated import object_


class BillingContractSubscription(BunqModel):
    """
    Show the subscription billing contract for the authenticated user.
    
    :param _subscription_type: The subscription type of the user. Can be one of
    PERSON_SUPER_LIGHT_V1, PERSON_LIGHT_V1, PERSON_MORE_V1, PERSON_FREE_V1,
    PERSON_PREMIUM_V1, COMPANY_V1, or COMPANY_V2.
    :type _subscription_type: str
    :param _id_: The id of the billing contract.
    :type _id_: int
    :param _created: The timestamp when the billing contract was made.
    :type _created: str
    :param _updated: The timestamp when the billing contract was last updated.
    :type _updated: str
    :param _contract_date_start: The date from when the billing contract is
    valid.
    :type _contract_date_start: str
    :param _contract_date_end: The date until when the billing contract is
    valid.
    :type _contract_date_end: str
    :param _contract_version: The version of the billing contract.
    :type _contract_version: int
    :param _subscription_type_downgrade: The subscription type the user will
    have after a subscription downgrade. Will be null if downgrading is not
    possible.
    :type _subscription_type_downgrade: str
    :param _status: The subscription status.
    :type _status: str
    :param _sub_status: The subscription substatus.
    :type _sub_status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/billing-contract-subscription"

    # Field constants.
    FIELD_SUBSCRIPTION_TYPE = "subscription_type"

    # Object type.
    _OBJECT_TYPE_GET = "BillingContractSubscription"

    _id_ = None
    _created = None
    _updated = None
    _contract_date_start = None
    _contract_date_end = None
    _contract_version = None
    _subscription_type = None
    _subscription_type_downgrade = None
    _status = None
    _sub_status = None
    _subscription_type_field_for_request = None

    def __init__(self, subscription_type):
        """
        :param subscription_type: The subscription type of the user. Can be one of
        PERSON_LIGHT_V1, PERSON_MORE_V1, PERSON_FREE_V1, PERSON_PREMIUM_V1,
        COMPANY_V1, or COMPANY_V2.
        :type subscription_type: str
        """

        self._subscription_type_field_for_request = subscription_type

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get all subscription billing contract for the authenticated user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBillingContractSubscriptionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseBillingContractSubscriptionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def contract_date_start(self):
        """
        :rtype: str
        """

        return self._contract_date_start

    @property
    def contract_date_end(self):
        """
        :rtype: str
        """

        return self._contract_date_end

    @property
    def contract_version(self):
        """
        :rtype: int
        """

        return self._contract_version

    @property
    def subscription_type(self):
        """
        :rtype: str
        """

        return self._subscription_type

    @property
    def subscription_type_downgrade(self):
        """
        :rtype: str
        """

        return self._subscription_type_downgrade

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._contract_date_start is not None:
            return False

        if self._contract_date_end is not None:
            return False

        if self._contract_version is not None:
            return False

        if self._subscription_type is not None:
            return False

        if self._subscription_type_downgrade is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BillingContractSubscription
        """

        return converter.json_to_class(BillingContractSubscription, json_str)


class CustomerLimit(BunqModel):
    """
    Show the limits for the authenticated user.
    
    :param _limit_monetary_account: The limit of monetary accounts.
    :type _limit_monetary_account: int
    :param _limit_monetary_account_remaining: The amount of additional monetary
    accounts you can create.
    :type _limit_monetary_account_remaining: int
    :param _limit_card_debit_maestro: The limit of Maestro cards.
    :type _limit_card_debit_maestro: int
    :param _limit_card_debit_mastercard: The limit of MasterCard cards.
    :type _limit_card_debit_mastercard: int
    :param _limit_card_debit_wildcard: DEPRECTATED: The limit of wildcards, e.g.
    Maestro or MasterCard cards.
    :type _limit_card_debit_wildcard: int
    :param _limit_card_wildcard: The limit of wildcards, e.g. Maestro or
    MasterCard cards.
    :type _limit_card_wildcard: int
    :param _limit_card_replacement: The limit of free replacement cards.
    :type _limit_card_replacement: int
    :param _limit_amount_monthly: The maximum amount a user is allowed to spend
    in a month.
    :type _limit_amount_monthly: object_.Amount
    :param _spent_amount_monthly: The amount the user has spent in the last
    month.
    :type _spent_amount_monthly: object_.Amount
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/limit"

    # Object type.
    _OBJECT_TYPE_GET = "CustomerLimit"

    _limit_monetary_account = None
    _limit_monetary_account_remaining = None
    _limit_card_debit_maestro = None
    _limit_card_debit_mastercard = None
    _limit_card_debit_wildcard = None
    _limit_card_wildcard = None
    _limit_card_replacement = None
    _limit_amount_monthly = None
    _spent_amount_monthly = None

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get all limits for the authenticated user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCustomerLimitList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCustomerLimitList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def limit_monetary_account(self):
        """
        :rtype: int
        """

        return self._limit_monetary_account

    @property
    def limit_monetary_account_remaining(self):
        """
        :rtype: int
        """

        return self._limit_monetary_account_remaining

    @property
    def limit_card_debit_maestro(self):
        """
        :rtype: int
        """

        return self._limit_card_debit_maestro

    @property
    def limit_card_debit_mastercard(self):
        """
        :rtype: int
        """

        return self._limit_card_debit_mastercard

    @property
    def limit_card_debit_wildcard(self):
        """
        :rtype: int
        """

        return self._limit_card_debit_wildcard

    @property
    def limit_card_wildcard(self):
        """
        :rtype: int
        """

        return self._limit_card_wildcard

    @property
    def limit_card_replacement(self):
        """
        :rtype: int
        """

        return self._limit_card_replacement

    @property
    def limit_amount_monthly(self):
        """
        :rtype: object_.Amount
        """

        return self._limit_amount_monthly

    @property
    def spent_amount_monthly(self):
        """
        :rtype: object_.Amount
        """

        return self._spent_amount_monthly

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._limit_monetary_account is not None:
            return False

        if self._limit_monetary_account_remaining is not None:
            return False

        if self._limit_card_debit_maestro is not None:
            return False

        if self._limit_card_debit_mastercard is not None:
            return False

        if self._limit_card_debit_wildcard is not None:
            return False

        if self._limit_card_wildcard is not None:
            return False

        if self._limit_card_replacement is not None:
            return False

        if self._limit_amount_monthly is not None:
            return False

        if self._spent_amount_monthly is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CustomerLimit
        """

        return converter.json_to_class(CustomerLimit, json_str)


class InvoiceExportPdfContent(BunqModel):
    """
    Get a PDF export of an invoice.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/invoice/{}/pdf-content"

    # Object type.
    _OBJECT_TYPE_GET = "InvoiceExportPdfContent"

    @classmethod
    def list(cls, invoice_id, custom_headers=None):
        """
        :type user_id: int
        :type invoice_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), invoice_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: InvoiceExportPdfContent
        """

        return converter.json_to_class(InvoiceExportPdfContent, json_str)


class Invoice(BunqModel):
    """
    Used to view a bunq invoice.
    
    :param _status: The invoice status.
    :type _status: str
    :param _description: The description provided by the admin.
    :type _description: str
    :param _external_url: The external url provided by the admin.
    :type _external_url: str
    :param _id_: The id of the invoice object.
    :type _id_: int
    :param _created: The timestamp of the invoice object's creation.
    :type _created: str
    :param _updated: The timestamp of the invoice object's last update.
    :type _updated: str
    :param _invoice_date: The invoice date.
    :type _invoice_date: str
    :param _invoice_number: The invoice number.
    :type _invoice_number: str
    :param _category: The category to display to the user.
    :type _category: str
    :param _group: The invoice item groups.
    :type _group: list[object_.InvoiceItemGroup]
    :param _total_vat_inclusive: The total discounted item price including VAT.
    :type _total_vat_inclusive: object_.Amount
    :param _total_vat_exclusive: The total discounted item price excluding VAT.
    :type _total_vat_exclusive: object_.Amount
    :param _total_vat: The VAT on the total discounted item price.
    :type _total_vat: object_.Amount
    :param _alias: The label that's displayed to the counterparty with the
    invoice. Includes user.
    :type _alias: object_.MonetaryAccountReference
    :param _address: The customer's address.
    :type _address: object_.Address
    :param _counterparty_alias: The label of the counterparty of the invoice.
    Includes user.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _counterparty_address: The company's address.
    :type _counterparty_address: object_.Address
    :param _chamber_of_commerce_number: The company's chamber of commerce
    number.
    :type _chamber_of_commerce_number: str
    :param _vat_number: The company's chamber of commerce number.
    :type _vat_number: str
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/invoice"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/invoice/{}"

    # Field constants.
    FIELD_STATUS = "status"
    FIELD_DESCRIPTION = "description"
    FIELD_EXTERNAL_URL = "external_url"

    # Object type.
    _OBJECT_TYPE_GET = "Invoice"

    _id_ = None
    _created = None
    _updated = None
    _invoice_date = None
    _invoice_number = None
    _status = None
    _category = None
    _group = None
    _total_vat_inclusive = None
    _total_vat_exclusive = None
    _total_vat = None
    _alias = None
    _address = None
    _counterparty_alias = None
    _counterparty_address = None
    _chamber_of_commerce_number = None
    _vat_number = None
    _request_reference_split_the_bill = None
    _status_field_for_request = None
    _description_field_for_request = None
    _external_url_field_for_request = None

    def __init__(self, status=None, description=None, external_url=None):
        """
        :param status: The status of the invoice.
        :type status: str
        :param description: The description provided by the admin.
        :type description: str
        :param external_url: The external url provided by the admin.
        :type external_url: str
        """

        self._status_field_for_request = status
        self._description_field_for_request = description
        self._external_url_field_for_request = external_url

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInvoiceList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseInvoiceList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, invoice_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type invoice_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInvoice
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     invoice_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseInvoice.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def invoice_date(self):
        """
        :rtype: str
        """

        return self._invoice_date

    @property
    def invoice_number(self):
        """
        :rtype: str
        """

        return self._invoice_number

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def category(self):
        """
        :rtype: str
        """

        return self._category

    @property
    def group(self):
        """
        :rtype: list[object_.InvoiceItemGroup]
        """

        return self._group

    @property
    def total_vat_inclusive(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat_inclusive

    @property
    def total_vat_exclusive(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat_exclusive

    @property
    def total_vat(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def address(self):
        """
        :rtype: object_.Address
        """

        return self._address

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def counterparty_address(self):
        """
        :rtype: object_.Address
        """

        return self._counterparty_address

    @property
    def chamber_of_commerce_number(self):
        """
        :rtype: str
        """

        return self._chamber_of_commerce_number

    @property
    def vat_number(self):
        """
        :rtype: str
        """

        return self._vat_number

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._invoice_date is not None:
            return False

        if self._invoice_number is not None:
            return False

        if self._status is not None:
            return False

        if self._category is not None:
            return False

        if self._group is not None:
            return False

        if self._total_vat_inclusive is not None:
            return False

        if self._total_vat_exclusive is not None:
            return False

        if self._total_vat is not None:
            return False

        if self._alias is not None:
            return False

        if self._address is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._counterparty_address is not None:
            return False

        if self._chamber_of_commerce_number is not None:
            return False

        if self._vat_number is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Invoice
        """

        return converter.json_to_class(Invoice, json_str)


class InvoiceByUser(BunqModel):
    """
    Used to list bunq invoices by user.
    
    :param _id_: The id of the invoice object.
    :type _id_: int
    :param _created: The timestamp of the invoice object's creation.
    :type _created: str
    :param _updated: The timestamp of the invoice object's last update.
    :type _updated: str
    :param _invoice_date: The invoice date.
    :type _invoice_date: str
    :param _invoice_number: The invoice number.
    :type _invoice_number: str
    :param _status: The invoice status.
    :type _status: str
    :param _group: The invoice item groups.
    :type _group: list[object_.InvoiceItemGroup]
    :param _total_vat_inclusive: The total discounted item price including VAT.
    :type _total_vat_inclusive: object_.Amount
    :param _total_vat_exclusive: The total discounted item price excluding VAT.
    :type _total_vat_exclusive: object_.Amount
    :param _total_vat: The VAT on the total discounted item price.
    :type _total_vat: object_.Amount
    :param _alias: The label that's displayed to the counterparty with the
    invoice. Includes user.
    :type _alias: object_.MonetaryAccountReference
    :param _address: The customer's address.
    :type _address: object_.Address
    :param _counterparty_alias: The label of the counterparty of the invoice.
    Includes user.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _counterparty_address: The company's address.
    :type _counterparty_address: object_.Address
    :param _chamber_of_commerce_number: The company's chamber of commerce
    number.
    :type _chamber_of_commerce_number: str
    :param _vat_number: The company's chamber of commerce number.
    :type _vat_number: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/invoice"
    _ENDPOINT_URL_READ = "user/{}/invoice/{}"

    # Object type.
    _OBJECT_TYPE_GET = "Invoice"

    _id_ = None
    _created = None
    _updated = None
    _invoice_date = None
    _invoice_number = None
    _status = None
    _group = None
    _total_vat_inclusive = None
    _total_vat_exclusive = None
    _total_vat = None
    _alias = None
    _address = None
    _counterparty_alias = None
    _counterparty_address = None
    _chamber_of_commerce_number = None
    _vat_number = None

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInvoiceByUserList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseInvoiceByUserList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, invoice_by_user_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type invoice_by_user_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInvoiceByUser
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), invoice_by_user_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseInvoiceByUser.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def invoice_date(self):
        """
        :rtype: str
        """

        return self._invoice_date

    @property
    def invoice_number(self):
        """
        :rtype: str
        """

        return self._invoice_number

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def group(self):
        """
        :rtype: list[object_.InvoiceItemGroup]
        """

        return self._group

    @property
    def total_vat_inclusive(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat_inclusive

    @property
    def total_vat_exclusive(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat_exclusive

    @property
    def total_vat(self):
        """
        :rtype: object_.Amount
        """

        return self._total_vat

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def address(self):
        """
        :rtype: object_.Address
        """

        return self._address

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def counterparty_address(self):
        """
        :rtype: object_.Address
        """

        return self._counterparty_address

    @property
    def chamber_of_commerce_number(self):
        """
        :rtype: str
        """

        return self._chamber_of_commerce_number

    @property
    def vat_number(self):
        """
        :rtype: str
        """

        return self._vat_number

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._invoice_date is not None:
            return False

        if self._invoice_number is not None:
            return False

        if self._status is not None:
            return False

        if self._group is not None:
            return False

        if self._total_vat_inclusive is not None:
            return False

        if self._total_vat_exclusive is not None:
            return False

        if self._total_vat is not None:
            return False

        if self._alias is not None:
            return False

        if self._address is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._counterparty_address is not None:
            return False

        if self._chamber_of_commerce_number is not None:
            return False

        if self._vat_number is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: InvoiceByUser
        """

        return converter.json_to_class(InvoiceByUser, json_str)


class AttachmentConversationContent(BunqModel):
    """
    Fetch the raw content of an attachment with given ID. The raw content is the
    base64 of a file, without any JSON wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/chat-conversation/{}/attachment/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentConversationContent"

    @classmethod
    def list(cls, chat_conversation_id, attachment_id, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type user_id: int
        :type chat_conversation_id: int
        :type attachment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), chat_conversation_id, attachment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentConversationContent
        """

        return converter.json_to_class(AttachmentConversationContent, json_str)


class AttachmentMonetaryAccountContent(BunqModel):
    """
    Fetch the raw content of a monetary account attachment with given ID. The
    raw content is the binary representation of a file, without any JSON
    wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/attachment/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentMonetaryAccountContent"

    @classmethod
    def list(cls, attachment_id, monetary_account_id=None, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type user_id: int
        :type monetary_account_id: int
        :type attachment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        attachment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentMonetaryAccountContent
        """

        return converter.json_to_class(AttachmentMonetaryAccountContent, json_str)


class AttachmentPublicContent(BunqModel):
    """
    Fetch the raw content of a public attachment with given ID. The raw content
    is the binary representation of a file, without any JSON wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "attachment-public/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentPublicContent"

    @classmethod
    def list(cls, attachment_public_uuid, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type attachment_public_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(attachment_public_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentPublicContent
        """

        return converter.json_to_class(AttachmentPublicContent, json_str)


class AttachmentTabContent(BunqModel):
    """
    Fetch the raw content of a tab attachment with given ID. The raw content is
    the binary representation of a file, without any JSON wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/attachment-tab/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentTabContent"

    @classmethod
    def list(cls, attachment_tab_id, monetary_account_id=None, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type user_id: int
        :type monetary_account_id: int
        :type attachment_tab_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        attachment_tab_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentTabContent
        """

        return converter.json_to_class(AttachmentTabContent, json_str)


class AttachmentUserContent(BunqModel):
    """
    Fetch the raw content of a user attachment with given ID. The raw content is
    the binary representation of a file, without any JSON wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/attachment/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentUserContent"

    @classmethod
    def list(cls, attachment_id, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type user_id: int
        :type attachment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), attachment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentUserContent
        """

        return converter.json_to_class(AttachmentUserContent, json_str)


class TabAttachmentTabContent(BunqModel):
    """
    Fetch the raw content of a tab attachment with given ID. The raw content is
    the binary representation of a file, without any JSON wrapping.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "tab/{}/attachment/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "TabAttachmentTabContent"

    @classmethod
    def list(cls, tab_uuid, attachment_id, custom_headers=None):
        """
        Get the raw content of a specific attachment.
        
        :type tab_uuid: str
        :type attachment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(tab_uuid, attachment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabAttachmentTabContent
        """

        return converter.json_to_class(TabAttachmentTabContent, json_str)


class AttachmentMonetaryAccount(BunqModel):
    """
    This call is used to upload an attachment that can be referenced to in
    payment requests and payments sent from a specific monetary account.
    Attachments supported are png, jpg and gif.
    
    :param _attachment: The attachment.
    :type _attachment: object_.Attachment
    :param _id_: The ID of the attachment created.
    :type _id_: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/attachment"

    _attachment = None
    _id_ = None

    @classmethod
    def create(cls, request_bytes, monetary_account_id=None, custom_headers=None):
        """
        Create a new monetary account attachment. Create a POST request with a
        payload that contains the binary representation of the file, without any
        JSON wrapping. Make sure you define the MIME type (i.e. image/jpeg) in
        the Content-Type header. You are required to provide a description of
        the attachment using the X-Bunq-Attachment-Description header.
        
        :type user_id: int
        :type monetary_account_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._attachment is not None:
            return False

        if self._id_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentMonetaryAccount
        """

        return converter.json_to_class(AttachmentMonetaryAccount, json_str)


class AttachmentPublic(BunqModel):
    """
    This call is used to upload an attachment that can be referenced to as an
    avatar (through the Avatar endpoint) or in a tab sent. Attachments supported
    are png, jpg and gif.
    
    :param _uuid: The UUID of the attachment.
    :type _uuid: str
    :param _created: The timestamp of the attachment's creation.
    :type _created: str
    :param _updated: The timestamp of the attachment's last update.
    :type _updated: str
    :param _attachment: The attachment.
    :type _attachment: object_.Attachment
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "attachment-public"
    _ENDPOINT_URL_READ = "attachment-public/{}"

    # Object type.
    _OBJECT_TYPE_POST = "Uuid"
    _OBJECT_TYPE_GET = "AttachmentPublic"

    _uuid = None
    _created = None
    _updated = None
    _attachment = None

    @classmethod
    def create(cls, request_bytes, custom_headers=None):
        """
        Create a new public attachment. Create a POST request with a payload
        that contains a binary representation of the file, without any JSON
        wrapping. Make sure you define the MIME type (i.e. image/jpeg, or
        image/png) in the Content-Type header. You are required to provide a
        description of the attachment using the X-Bunq-Attachment-Description
        header.
        
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_CREATE
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def get(cls, attachment_public_uuid, custom_headers=None):
        """
        Get a specific attachment's metadata through its UUID. The Content-Type
        header of the response will describe the MIME type of the attachment
        file.
        
        :type api_context: ApiContext
        :type attachment_public_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseAttachmentPublic
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(attachment_public_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseAttachmentPublic.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def uuid(self):
        """
        :rtype: str
        """

        return self._uuid

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._uuid is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentPublic
        """

        return converter.json_to_class(AttachmentPublic, json_str)


class AttachmentTab(BunqModel):
    """
    This call is used to upload an attachment that will be accessible only
    through tabs. This can be used for example to upload special promotions or
    other attachments. Attachments supported are png, jpg and gif.
    
    :param _id_: The id of the attachment.
    :type _id_: int
    :param _created: The timestamp of the attachment's creation.
    :type _created: str
    :param _updated: The timestamp of the attachment's last update.
    :type _updated: str
    :param _attachment: The attachment.
    :type _attachment: object_.Attachment
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/attachment-tab"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/attachment-tab/{}"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentTab"

    _id_ = None
    _created = None
    _updated = None
    _attachment = None

    @classmethod
    def create(cls, request_bytes, monetary_account_id=None, custom_headers=None):
        """
        Upload a new attachment to use with a tab, and to read its metadata.
        Create a POST request with a payload that contains the binary
        representation of the file, without any JSON wrapping. Make sure you
        define the MIME type (i.e. image/jpeg) in the Content-Type header. You
        are required to provide a description of the attachment using the
        X-Bunq-Attachment-Description header.
        
        :type user_id: int
        :type monetary_account_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, attachment_tab_id, monetary_account_id=None, custom_headers=None):
        """
        Get a specific attachment. The header of the response contains the
        content-type of the attachment.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type attachment_tab_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseAttachmentTab
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     attachment_tab_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseAttachmentTab.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentTab
        """

        return converter.json_to_class(AttachmentTab, json_str)


class TabAttachmentTab(BunqModel):
    """
    This call is used to view an attachment that is linked to a tab.
    
    :param _id_: The id of the attachment.
    :type _id_: int
    :param _created: The timestamp of the attachment's creation.
    :type _created: str
    :param _updated: The timestamp of the attachment's last update.
    :type _updated: str
    :param _attachment: The attachment.
    :type _attachment: object_.Attachment
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "tab/{}/attachment/{}"

    # Object type.
    _OBJECT_TYPE_GET = "TabAttachmentTab"

    _id_ = None
    _created = None
    _updated = None
    _attachment = None

    @classmethod
    def get(cls, tab_uuid, tab_attachment_tab_id, custom_headers=None):
        """
        Get a specific attachment. The header of the response contains the
        content-type of the attachment.
        
        :type api_context: ApiContext
        :type tab_uuid: str
        :type tab_attachment_tab_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabAttachmentTab
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(tab_uuid, tab_attachment_tab_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTabAttachmentTab.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabAttachmentTab
        """

        return converter.json_to_class(TabAttachmentTab, json_str)


class AttachmentUser(BunqModel):
    """
    This call is used to upload an attachment that is accessible only by a
    specific user. This can be used for example to upload passport scans or
    other documents. Attachments supported are png, jpg and gif.
    
    :param _id_: The id of the attachment.
    :type _id_: int
    :param _created: The timestamp of the attachment's creation.
    :type _created: str
    :param _updated: The timestamp of the attachment's last update.
    :type _updated: str
    :param _attachment: The attachment.
    :type _attachment: object_.Attachment
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/attachment/{}"

    # Object type.
    _OBJECT_TYPE_GET = "AttachmentUser"

    _id_ = None
    _created = None
    _updated = None
    _attachment = None

    @classmethod
    def get(cls, attachment_user_id, custom_headers=None):
        """
        Get a specific attachment. The header of the response contains the
        content-type of the attachment.
        
        :type api_context: ApiContext
        :type user_id: int
        :type attachment_user_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseAttachmentUser
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), attachment_user_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseAttachmentUser.cast_from_bunq_response(
            cls._from_json(response_raw)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: AttachmentUser
        """

        return converter.json_to_class(AttachmentUser, json_str)


class Avatar(BunqModel):
    """
    Avatars are public images used to represent you or your company. Avatars are
    used to represent users, monetary accounts and cash registers. Avatars
    cannot be deleted, only replaced. Avatars can be updated after uploading the
    image you would like to use through AttachmentPublic. Using the
    attachment_public_uuid which is returned you can update your Avatar. Avatars
    used for cash registers and company accounts will be reviewed by bunq.
    
    :param _attachment_public_uuid: The public UUID of the public attachment
    from which an avatar image must be created.
    :type _attachment_public_uuid: str
    :param _uuid: The UUID of the created avatar.
    :type _uuid: str
    :param _image: The content type of the image.
    :type _image: list[object_.Image]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "avatar"
    _ENDPOINT_URL_READ = "avatar/{}"

    # Field constants.
    FIELD_ATTACHMENT_PUBLIC_UUID = "attachment_public_uuid"

    # Object type.
    _OBJECT_TYPE_POST = "Uuid"
    _OBJECT_TYPE_GET = "Avatar"

    _uuid = None
    _image = None
    _attachment_public_uuid_field_for_request = None

    def __init__(self, attachment_public_uuid):
        """
        :param attachment_public_uuid: The public UUID of the public attachment from
        which an avatar image must be created.
        :type attachment_public_uuid: str
        """

        self._attachment_public_uuid_field_for_request = attachment_public_uuid

    @classmethod
    def create(cls, attachment_public_uuid, custom_headers=None):
        """
        :param attachment_public_uuid: The public UUID of the public attachment
        from which an avatar image must be created.
        :type attachment_public_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_ATTACHMENT_PUBLIC_UUID: attachment_public_uuid
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def get(cls, avatar_uuid, custom_headers=None):
        """
        :type api_context: ApiContext
        :type avatar_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseAvatar
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(avatar_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseAvatar.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def uuid(self):
        """
        :rtype: str
        """

        return self._uuid

    @property
    def image(self):
        """
        :rtype: list[object_.Image]
        """

        return self._image

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._uuid is not None:
            return False

        if self._image is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Avatar
        """

        return converter.json_to_class(Avatar, json_str)


class BankSwitchServiceNetherlandsIncomingPayment(BunqModel):
    """
    An incoming payment made towards an account of an external bank and
    redirected to a bunq account via switch service.
    
    :param _bank_switch_service: The bank switch service details.
    :type _bank_switch_service: BankSwitchServiceNetherlandsIncoming
    :param _payment: The payment made using bank switch service.
    :type _payment: Payment
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/switch-service-payment/{}"

    # Object type.
    _OBJECT_TYPE_GET = "BankSwitchServiceNetherlandsIncomingPayment"

    _bank_switch_service = None
    _payment = None

    @classmethod
    def get(cls, bank_switch_service_netherlands_incoming_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bank_switch_service_netherlands_incoming_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBankSwitchServiceNetherlandsIncomingPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBankSwitchServiceNetherlandsIncomingPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def bank_switch_service(self):
        """
        :rtype: BankSwitchServiceNetherlandsIncoming
        """

        return self._bank_switch_service

    @property
    def payment(self):
        """
        :rtype: Payment
        """

        return self._payment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._bank_switch_service is not None:
            return False

        if self._payment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BankSwitchServiceNetherlandsIncomingPayment
        """

        return converter.json_to_class(BankSwitchServiceNetherlandsIncomingPayment, json_str)


class BankSwitchServiceNetherlandsIncoming(BunqModel):
    """
    Endpoint for using the Equens Bank Switch Service.
    
    :param _alias: The label of the monetary of this switch service.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The IBAN alias that's displayed for this switch
    service.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _status: The status of the switch service.
    :type _status: str
    :param _user_alias: The label of the user creator of this switch service.
    :type _user_alias: object_.LabelUser
    :param _sub_status: The sub status of the switch service.
    :type _sub_status: str
    :param _time_start_desired: The timestamp when the switch service desired to
    be start.
    :type _time_start_desired: str
    :param _time_start_actual: The timestamp when the switch service actually
    starts.
    :type _time_start_actual: str
    :param _time_end: The timestamp when the switch service ends.
    :type _time_end: str
    :param _attachment: Reference to the bank transfer form for this
    switch-service.
    :type _attachment: object_.Attachment
    """

    # Field constants.
    FIELD_ALIAS = "alias"
    FIELD_COUNTERPARTY_ALIAS = "counterparty_alias"
    FIELD_STATUS = "status"

    _user_alias = None
    _alias = None
    _counterparty_alias = None
    _status = None
    _sub_status = None
    _time_start_desired = None
    _time_start_actual = None
    _time_end = None
    _attachment = None
    _alias_field_for_request = None
    _counterparty_alias_field_for_request = None
    _status_field_for_request = None

    def __init__(self, alias, counterparty_alias, status=None):
        """
        :param alias: The alias of the Monetary Account this switch service is for.
        :type alias: object_.Pointer
        :param counterparty_alias: The Alias of the party we are switching from. Can
        only be an Alias of type IBAN (external bank account).
        :type counterparty_alias: object_.Pointer
        :param status: The status of the switch service. Ignored in POST requests
        (always set to REQUESTED) can be CANCELLED in PUT requests to cancel the
        switch service. Admin can set this to ACCEPTED, or REJECTED.
        :type status: str
        """

        self._alias_field_for_request = alias
        self._counterparty_alias_field_for_request = counterparty_alias
        self._status_field_for_request = status

    @property
    def user_alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def time_start_desired(self):
        """
        :rtype: str
        """

        return self._time_start_desired

    @property
    def time_start_actual(self):
        """
        :rtype: str
        """

        return self._time_start_actual

    @property
    def time_end(self):
        """
        :rtype: str
        """

        return self._time_end

    @property
    def attachment(self):
        """
        :rtype: object_.Attachment
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._user_alias is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._time_start_desired is not None:
            return False

        if self._time_start_actual is not None:
            return False

        if self._time_end is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BankSwitchServiceNetherlandsIncoming
        """

        return converter.json_to_class(BankSwitchServiceNetherlandsIncoming, json_str)


class Payment(BunqModel):
    """
    Using Payment, you can send payments to bunq and non-bunq users from your
    bunq MonetaryAccounts. This can be done using bunq Aliases or IBAN Aliases.
    When transferring money to other bunq MonetaryAccounts you can also refer to
    Attachments. These will be received by the counter-party as part of the
    Payment. You can also retrieve a single Payment or all executed Payments of
    a specific monetary account.
    
    :param _amount: The Amount transferred by the Payment. Will be negative for
    outgoing Payments and positive for incoming Payments (relative to the
    MonetaryAccount indicated by monetary_account_id).
    :type _amount: object_.Amount
    :param _counterparty_alias: The LabelMonetaryAccount containing the public
    information of the other (counterparty) side of the Payment.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _description: The description for the Payment. Maximum 140 characters
    for Payments to external IBANs, 9000 characters for Payments to only other
    bunq MonetaryAccounts.
    :type _description: str
    :param _attachment: The Attachments attached to the Payment.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    :param _merchant_reference: Optional data included with the Payment specific
    to the merchant.
    :type _merchant_reference: str
    :param _allow_bunqto: Whether or not sending a bunq.to payment is allowed.
    :type _allow_bunqto: bool
    :param _id_: The id of the created Payment.
    :type _id_: int
    :param _created: The timestamp when the Payment was done.
    :type _created: str
    :param _updated: The timestamp when the Payment was last updated (will be
    updated when chat messages are received).
    :type _updated: str
    :param _monetary_account_id: The id of the MonetaryAccount the Payment was
    made to or from (depending on whether this is an incoming or outgoing
    Payment).
    :type _monetary_account_id: int
    :param _alias: The LabelMonetaryAccount containing the public information of
    'this' (party) side of the Payment.
    :type _alias: object_.MonetaryAccountReference
    :param _type_: The type of Payment, can be BUNQ, EBA_SCT, EBA_SDD, IDEAL,
    SWIFT or FIS (card).
    :type _type_: str
    :param _sub_type: The sub-type of the Payment, can be PAYMENT, WITHDRAWAL,
    REVERSAL, REQUEST, BILLING, SCT, SDD or NLO.
    :type _sub_type: str
    :param _bunqto_status: The status of the bunq.to payment.
    :type _bunqto_status: str
    :param _bunqto_sub_status: The sub status of the bunq.to payment.
    :type _bunqto_sub_status: str
    :param _bunqto_share_url: The status of the bunq.to payment.
    :type _bunqto_share_url: str
    :param _bunqto_expiry: When bunq.to payment is about to expire.
    :type _bunqto_expiry: str
    :param _bunqto_time_responded: The timestamp of when the bunq.to payment was
    responded to.
    :type _bunqto_time_responded: str
    :param _batch_id: The id of the PaymentBatch if this Payment was part of
    one.
    :type _batch_id: int
    :param _scheduled_id: The id of the JobScheduled if the Payment was
    scheduled.
    :type _scheduled_id: int
    :param _address_shipping: A shipping Address provided with the Payment,
    currently unused.
    :type _address_shipping: object_.Address
    :param _address_billing: A billing Address provided with the Payment,
    currently unused.
    :type _address_billing: object_.Address
    :param _geolocation: The Geolocation where the Payment was done from.
    :type _geolocation: object_.Geolocation
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    :param _balance_after_mutation: The new balance of the monetary account
    after the mutation.
    :type _balance_after_mutation: object_.Amount
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment"

    # Field constants.
    FIELD_AMOUNT = "amount"
    FIELD_COUNTERPARTY_ALIAS = "counterparty_alias"
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT = "attachment"
    FIELD_MERCHANT_REFERENCE = "merchant_reference"
    FIELD_ALLOW_BUNQTO = "allow_bunqto"

    # Object type.
    _OBJECT_TYPE_GET = "Payment"

    _id_ = None
    _created = None
    _updated = None
    _monetary_account_id = None
    _amount = None
    _alias = None
    _counterparty_alias = None
    _description = None
    _type_ = None
    _sub_type = None
    _bunqto_status = None
    _bunqto_sub_status = None
    _bunqto_share_url = None
    _bunqto_expiry = None
    _bunqto_time_responded = None
    _attachment = None
    _merchant_reference = None
    _batch_id = None
    _scheduled_id = None
    _address_shipping = None
    _address_billing = None
    _geolocation = None
    _request_reference_split_the_bill = None
    _balance_after_mutation = None
    _amount_field_for_request = None
    _counterparty_alias_field_for_request = None
    _description_field_for_request = None
    _attachment_field_for_request = None
    _merchant_reference_field_for_request = None
    _allow_bunqto_field_for_request = None

    def __init__(self, amount, counterparty_alias, description, attachment=None, merchant_reference=None,
                 allow_bunqto=None):
        """
        :param amount: The Amount to transfer with the Payment. Must be bigger than
        0 and smaller than the MonetaryAccount's balance.
        :type amount: object_.Amount
        :param counterparty_alias: The Alias of the party we are transferring the
        money to. Can be an Alias of type EMAIL or PHONE_NUMBER (for bunq
        MonetaryAccounts or bunq.to payments) or IBAN (for external bank account).
        :type counterparty_alias: object_.Pointer
        :param description: The description for the Payment. Maximum 140 characters
        for Payments to external IBANs, 9000 characters for Payments to only other
        bunq MonetaryAccounts. Field is required but can be an empty string.
        :type description: str
        :param attachment: The Attachments to attach to the Payment.
        :type attachment: list[object_.AttachmentMonetaryAccountPayment]
        :param merchant_reference: Optional data to be included with the Payment
        specific to the merchant.
        :type merchant_reference: str
        :param allow_bunqto: Whether or not sending a bunq.to payment is allowed.
        :type allow_bunqto: bool
        """

        self._amount_field_for_request = amount
        self._counterparty_alias_field_for_request = counterparty_alias
        self._description_field_for_request = description
        self._attachment_field_for_request = attachment
        self._merchant_reference_field_for_request = merchant_reference
        self._allow_bunqto_field_for_request = allow_bunqto

    @classmethod
    def create(cls, amount, counterparty_alias, description, monetary_account_id=None, attachment=None,
               merchant_reference=None, allow_bunqto=None, custom_headers=None):
        """
        Create a new Payment.
        
        :type user_id: int
        :type monetary_account_id: int
        :param amount: The Amount to transfer with the Payment. Must be bigger
        than 0 and smaller than the MonetaryAccount's balance.
        :type amount: object_.Amount
        :param counterparty_alias: The Alias of the party we are transferring
        the money to. Can be an Alias of type EMAIL or PHONE_NUMBER (for bunq
        MonetaryAccounts or bunq.to payments) or IBAN (for external bank
        account).
        :type counterparty_alias: object_.Pointer
        :param description: The description for the Payment. Maximum 140
        characters for Payments to external IBANs, 9000 characters for Payments
        to only other bunq MonetaryAccounts. Field is required but can be an
        empty string.
        :type description: str
        :param attachment: The Attachments to attach to the Payment.
        :type attachment: list[object_.AttachmentMonetaryAccountPayment]
        :param merchant_reference: Optional data to be included with the Payment
        specific to the merchant.
        :type merchant_reference: str
        :param allow_bunqto: Whether or not sending a bunq.to payment is
        allowed.
        :type allow_bunqto: bool
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_AMOUNT: amount,
            cls.FIELD_COUNTERPARTY_ALIAS: counterparty_alias,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT: attachment,
            cls.FIELD_MERCHANT_REFERENCE: merchant_reference,
            cls.FIELD_ALLOW_BUNQTO: allow_bunqto
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, payment_id, monetary_account_id=None, custom_headers=None):
        """
        Get a specific previous Payment.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponsePayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponsePayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a listing of all Payments performed on a given MonetaryAccount
        (incoming and outgoing).
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponsePaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponsePaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def amount(self):
        """
        :rtype: object_.Amount
        """

        return self._amount

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def sub_type(self):
        """
        :rtype: str
        """

        return self._sub_type

    @property
    def bunqto_status(self):
        """
        :rtype: str
        """

        return self._bunqto_status

    @property
    def bunqto_sub_status(self):
        """
        :rtype: str
        """

        return self._bunqto_sub_status

    @property
    def bunqto_share_url(self):
        """
        :rtype: str
        """

        return self._bunqto_share_url

    @property
    def bunqto_expiry(self):
        """
        :rtype: str
        """

        return self._bunqto_expiry

    @property
    def bunqto_time_responded(self):
        """
        :rtype: str
        """

        return self._bunqto_time_responded

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    @property
    def merchant_reference(self):
        """
        :rtype: str
        """

        return self._merchant_reference

    @property
    def batch_id(self):
        """
        :rtype: int
        """

        return self._batch_id

    @property
    def scheduled_id(self):
        """
        :rtype: int
        """

        return self._scheduled_id

    @property
    def address_shipping(self):
        """
        :rtype: object_.Address
        """

        return self._address_shipping

    @property
    def address_billing(self):
        """
        :rtype: object_.Address
        """

        return self._address_billing

    @property
    def geolocation(self):
        """
        :rtype: object_.Geolocation
        """

        return self._geolocation

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    @property
    def balance_after_mutation(self):
        """
        :rtype: object_.Amount
        """

        return self._balance_after_mutation

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._amount is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._description is not None:
            return False

        if self._type_ is not None:
            return False

        if self._sub_type is not None:
            return False

        if self._bunqto_status is not None:
            return False

        if self._bunqto_sub_status is not None:
            return False

        if self._bunqto_share_url is not None:
            return False

        if self._bunqto_expiry is not None:
            return False

        if self._bunqto_time_responded is not None:
            return False

        if self._attachment is not None:
            return False

        if self._merchant_reference is not None:
            return False

        if self._batch_id is not None:
            return False

        if self._scheduled_id is not None:
            return False

        if self._address_shipping is not None:
            return False

        if self._address_billing is not None:
            return False

        if self._geolocation is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        if self._balance_after_mutation is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Payment
        """

        return converter.json_to_class(Payment, json_str)


class BunqMeFundraiserProfileUser(BunqModel):
    """
    bunq.me public profile of the user.
    
    :param _monetary_account_id: Id of the monetary account on which you want to
    receive bunq.me payments.
    :type _monetary_account_id: int
    :param _color: The color chosen for the bunq.me fundraiser profile in
    hexadecimal format.
    :type _color: str
    :param _description: The description of the bunq.me fundraiser profile.
    :type _description: str
    :param _attachment_public_uuid: The public UUID of the public attachment
    from which an avatar image must be created.
    :type _attachment_public_uuid: str
    :param _pointer: The pointer (url) which will be used to access the bunq.me
    fundraiser profile.
    :type _pointer: object_.MonetaryAccountReference
    :param _redirect_url: The URL which the user is sent to when a payment is
    completed.
    :type _redirect_url: str
    :param _status: The status of the bunq.me fundraiser profile, can be ACTIVE
    or DEACTIVATED.
    :type _status: str
    :param _owner_user_id: Id of the user owning the profile.
    :type _owner_user_id: int
    :param _alias: The LabelMonetaryAccount with the public information of the
    User and the MonetaryAccount that created the bunq.me fundraiser profile.
    :type _alias: object_.MonetaryAccountReference
    :param _attachment: The attachment used for the background of the bunq.me
    fundraiser profile.
    :type _attachment: object_.AttachmentPublic
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/bunqme-fundraiser-profile/{}"
    _ENDPOINT_URL_LISTING = "user/{}/bunqme-fundraiser-profile"

    # Field constants.
    FIELD_MONETARY_ACCOUNT_ID = "monetary_account_id"
    FIELD_COLOR = "color"
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_PUBLIC_UUID = "attachment_public_uuid"
    FIELD_POINTER = "pointer"
    FIELD_REDIRECT_URL = "redirect_url"
    FIELD_STATUS = "status"

    # Object type.
    _OBJECT_TYPE_GET = "BunqMeFundraiserProfileModel"

    _monetary_account_id = None
    _owner_user_id = None
    _color = None
    _alias = None
    _description = None
    _attachment = None
    _pointer = None
    _redirect_url = None
    _status = None
    _monetary_account_id_field_for_request = None
    _color_field_for_request = None
    _description_field_for_request = None
    _attachment_public_uuid_field_for_request = None
    _pointer_field_for_request = None
    _redirect_url_field_for_request = None
    _status_field_for_request = None

    def __init__(self, monetary_account_id=None, description=None, pointer=None, color=None,
                 attachment_public_uuid=None, redirect_url=None, status=None):
        """
        :param monetary_account_id: ID of the monetary account on which you want to
        receive bunq.me fundraiser payments.
        :type monetary_account_id: int
        :param description: The description of the bunq.me fundraiser profile.
        Maximum 9000 characters. Field is required but can be an empty string.
        :type description: str
        :param pointer: The pointer (url) which will be used to access the bunq.me
        fundraiser profile.
        :type pointer: object_.Pointer
        :param color: The color chosen for the bunq.me fundraiser profile in
        hexadecimal format.
        :type color: str
        :param attachment_public_uuid: The public UUID of the public attachment from
        which an avatar image must be created.
        :type attachment_public_uuid: str
        :param redirect_url: The URL which the user is sent to when a payment is
        completed.
        :type redirect_url: str
        :param status: The status of the bunq.me fundraiser profile.
        :type status: str
        """

        self._monetary_account_id_field_for_request = monetary_account_id
        self._description_field_for_request = description
        self._pointer_field_for_request = pointer
        self._color_field_for_request = color
        self._attachment_public_uuid_field_for_request = attachment_public_uuid
        self._redirect_url_field_for_request = redirect_url
        self._status_field_for_request = status

    @classmethod
    def get(cls, bunq_me_fundraiser_profile_user_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type bunq_me_fundraiser_profile_user_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeFundraiserProfileUser
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), bunq_me_fundraiser_profile_user_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBunqMeFundraiserProfileUser.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeFundraiserProfileUserList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseBunqMeFundraiserProfileUserList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def owner_user_id(self):
        """
        :rtype: int
        """

        return self._owner_user_id

    @property
    def color(self):
        """
        :rtype: str
        """

        return self._color

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: object_.AttachmentPublic
        """

        return self._attachment

    @property
    def pointer(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._pointer

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._monetary_account_id is not None:
            return False

        if self._owner_user_id is not None:
            return False

        if self._color is not None:
            return False

        if self._alias is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        if self._pointer is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeFundraiserProfileUser
        """

        return converter.json_to_class(BunqMeFundraiserProfileUser, json_str)


class BunqMeFundraiserResult(BunqModel):
    """
    bunq.me fundraiser result containing all payments.
    
    :param _id_: The id of the bunq.me.
    :type _id_: int
    :param _created: The timestamp when the bunq.me was created.
    :type _created: str
    :param _updated: The timestamp when the bunq.me was last updated.
    :type _updated: str
    :param _bunqme_fundraiser_profile: The bunq.me fundraiser profile.
    :type _bunqme_fundraiser_profile: BunqMeFundraiserProfile
    :param _payments: The list of payments, paid to the bunq.me fundraiser
    profile.
    :type _payments: list[Payment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}"

    # Object type.
    _OBJECT_TYPE_GET = "BunqMeFundraiserResult"

    _id_ = None
    _created = None
    _updated = None
    _bunqme_fundraiser_profile = None
    _payments = None

    @classmethod
    def get(cls, bunq_me_fundraiser_result_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bunq_me_fundraiser_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeFundraiserResult
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bunq_me_fundraiser_result_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBunqMeFundraiserResult.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def bunqme_fundraiser_profile(self):
        """
        :rtype: BunqMeFundraiserProfile
        """

        return self._bunqme_fundraiser_profile

    @property
    def payments(self):
        """
        :rtype: list[Payment]
        """

        return self._payments

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._bunqme_fundraiser_profile is not None:
            return False

        if self._payments is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeFundraiserResult
        """

        return converter.json_to_class(BunqMeFundraiserResult, json_str)


class BunqMeFundraiserProfile(BunqModel):
    """
    bunq.me public profile of the user.
    
    :param _pointer: The pointer (url) which will be used to access the bunq.me
    fundraiser profile.
    :type _pointer: object_.MonetaryAccountReference
    :param _color: The color chosen for the bunq.me fundraiser profile in
    hexadecimal format.
    :type _color: str
    :param _alias: The LabelMonetaryAccount with the public information of the
    User and the MonetaryAccount that created the bunq.me fundraiser profile.
    :type _alias: object_.MonetaryAccountReference
    :param _description: The description of the bunq.me fundraiser profile.
    :type _description: str
    :param _attachment: The attachment attached to the fundraiser profile.
    :type _attachment: object_.AttachmentPublic
    :param _status: The status of the bunq.me fundraiser profile, can be ACTIVE
    or DEACTIVATED.
    :type _status: str
    :param _redirect_url: The URL which the user is sent to when a payment is
    completed.
    :type _redirect_url: str
    :param _invite_profile_name: Provided if the user has enabled their invite
    link.
    :type _invite_profile_name: str
    """

    # Field constants.
    FIELD_POINTER = "pointer"

    _color = None
    _alias = None
    _description = None
    _attachment = None
    _pointer = None
    _status = None
    _redirect_url = None
    _invite_profile_name = None
    _pointer_field_for_request = None

    def __init__(self, pointer):
        """
        :param pointer: The pointer (url) which will be used to access the bunq.me
        fundraiser profile.
        :type pointer: object_.Pointer
        """

        self._pointer_field_for_request = pointer

    @property
    def color(self):
        """
        :rtype: str
        """

        return self._color

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: object_.AttachmentPublic
        """

        return self._attachment

    @property
    def pointer(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._pointer

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def invite_profile_name(self):
        """
        :rtype: str
        """

        return self._invite_profile_name

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._color is not None:
            return False

        if self._alias is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        if self._pointer is not None:
            return False

        if self._status is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._invite_profile_name is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeFundraiserProfile
        """

        return converter.json_to_class(BunqMeFundraiserProfile, json_str)


class BunqMeTabResultResponse(BunqModel):
    """
    Used to view bunq.me TabResultResponse objects belonging to a tab. A
    TabResultResponse is an object that holds details on a tab which has been
    paid from the provided monetary account.
    
    :param _payment: The payment made for the bunq.me tab.
    :type _payment: Payment
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/bunqme-tab-result-response/{}"

    # Object type.
    _OBJECT_TYPE_GET = "BunqMeTabResultResponse"

    _payment = None

    @classmethod
    def get(cls, bunq_me_tab_result_response_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bunq_me_tab_result_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeTabResultResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bunq_me_tab_result_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBunqMeTabResultResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def payment(self):
        """
        :rtype: Payment
        """

        return self._payment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._payment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeTabResultResponse
        """

        return converter.json_to_class(BunqMeTabResultResponse, json_str)


class BunqMeTab(BunqModel):
    """
    bunq.me tabs allows you to create a payment request and share the link
    through e-mail, chat, etc. Multiple persons are able to respond to the
    payment request and pay through bunq, iDeal or SOFORT.
    
    :param _bunqme_tab_entry: The bunq.me entry containing the payment
    information.
    :type _bunqme_tab_entry: BunqMeTabEntry
    :param _status: The status of the bunq.me. Can be WAITING_FOR_PAYMENT,
    CANCELLED or EXPIRED.
    :type _status: str
    :param _id_: The id of the created bunq.me.
    :type _id_: int
    :param _created: The timestamp when the bunq.me was created.
    :type _created: str
    :param _updated: The timestamp when the bunq.me was last updated.
    :type _updated: str
    :param _time_expiry: The timestamp of when the bunq.me expired or will
    expire.
    :type _time_expiry: str
    :param _monetary_account_id: The id of the MonetaryAccount the bunq.me was
    sent from.
    :type _monetary_account_id: int
    :param _bunqme_tab_share_url: The url that points to the bunq.me page.
    :type _bunqme_tab_share_url: str
    :param _result_inquiries: The list of bunq.me result Inquiries successfully
    made and paid.
    :type _result_inquiries: list[BunqMeTabResultInquiry]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/bunqme-tab"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/bunqme-tab/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/bunqme-tab"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/bunqme-tab/{}"

    # Field constants.
    FIELD_BUNQME_TAB_ENTRY = "bunqme_tab_entry"
    FIELD_STATUS = "status"

    # Object type.
    _OBJECT_TYPE_GET = "BunqMeTab"

    _id_ = None
    _created = None
    _updated = None
    _time_expiry = None
    _monetary_account_id = None
    _status = None
    _bunqme_tab_share_url = None
    _bunqme_tab_entry = None
    _result_inquiries = None
    _bunqme_tab_entry_field_for_request = None
    _status_field_for_request = None

    def __init__(self, bunqme_tab_entry, status=None):
        """
        :param bunqme_tab_entry: The bunq.me entry containing the payment
        information.
        :type bunqme_tab_entry: BunqMeTabEntry
        :param status: The status of the bunq.me. Ignored in POST requests but can
        be used for cancelling the bunq.me by setting status as CANCELLED with a PUT
        request.
        :type status: str
        """

        self._bunqme_tab_entry_field_for_request = bunqme_tab_entry
        self._status_field_for_request = status

    @classmethod
    def create(cls, bunqme_tab_entry, monetary_account_id=None, status=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param bunqme_tab_entry: The bunq.me entry containing the payment
        information.
        :type bunqme_tab_entry: BunqMeTabEntry
        :param status: The status of the bunq.me. Ignored in POST requests but
        can be used for cancelling the bunq.me by setting status as CANCELLED
        with a PUT request.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_BUNQME_TAB_ENTRY: bunqme_tab_entry,
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, bunq_me_tab_id, monetary_account_id=None, status=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunq_me_tab_id: int
        :param status: The status of the bunq.me. Ignored in POST requests but
        can be used for cancelling the bunq.me by setting status as CANCELLED
        with a PUT request.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunq_me_tab_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeTabList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseBunqMeTabList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, bunq_me_tab_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bunq_me_tab_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBunqMeTab
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bunq_me_tab_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBunqMeTab.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def time_expiry(self):
        """
        :rtype: str
        """

        return self._time_expiry

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def bunqme_tab_share_url(self):
        """
        :rtype: str
        """

        return self._bunqme_tab_share_url

    @property
    def bunqme_tab_entry(self):
        """
        :rtype: BunqMeTabEntry
        """

        return self._bunqme_tab_entry

    @property
    def result_inquiries(self):
        """
        :rtype: list[BunqMeTabResultInquiry]
        """

        return self._result_inquiries

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._time_expiry is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._status is not None:
            return False

        if self._bunqme_tab_share_url is not None:
            return False

        if self._bunqme_tab_entry is not None:
            return False

        if self._result_inquiries is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeTab
        """

        return converter.json_to_class(BunqMeTab, json_str)


class BunqMeTabEntry(BunqModel):
    """
    bunq.me tabs allows you to create a payment request and share the link
    through e-mail, chat, etc. Multiple persons are able to respond to the
    payment request and pay through bunq, iDeal or SOFORT.
    
    :param _amount_inquired: The requested Amount.
    :type _amount_inquired: object_.Amount
    :param _description: The description for the bunq.me. Maximum 9000
    characters.
    :type _description: str
    :param _redirect_url: The URL which the user is sent to when a payment is
    completed.
    :type _redirect_url: str
    :param _uuid: The uuid of the bunq.me.
    :type _uuid: str
    :param _alias: The LabelMonetaryAccount with the public information of the
    User and the MonetaryAccount that created the bunq.me link.
    :type _alias: object_.MonetaryAccountReference
    :param _status: The status of the bunq.me. Can be WAITING_FOR_PAYMENT,
    CANCELLED or EXPIRED.
    :type _status: str
    :param _merchant_available: List of available merchants.
    :type _merchant_available: list[object_.BunqMeMerchantAvailable]
    """

    # Field constants.
    FIELD_AMOUNT_INQUIRED = "amount_inquired"
    FIELD_DESCRIPTION = "description"
    FIELD_REDIRECT_URL = "redirect_url"

    _uuid = None
    _amount_inquired = None
    _alias = None
    _description = None
    _status = None
    _redirect_url = None
    _merchant_available = None
    _amount_inquired_field_for_request = None
    _description_field_for_request = None
    _redirect_url_field_for_request = None

    def __init__(self, amount_inquired, description, redirect_url=None):
        """
        :param amount_inquired: The Amount requested to be paid. Can be optional.
        :type amount_inquired: object_.Amount
        :param description: The description for the bunq.me. Maximum 9000
        characters. Field is required but can be an empty string.
        :type description: str
        :param redirect_url: The URL which the user is sent to after making a
        payment.
        :type redirect_url: str
        """

        self._amount_inquired_field_for_request = amount_inquired
        self._description_field_for_request = description
        self._redirect_url_field_for_request = redirect_url

    @property
    def uuid(self):
        """
        :rtype: str
        """

        return self._uuid

    @property
    def amount_inquired(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_inquired

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def merchant_available(self):
        """
        :rtype: list[object_.BunqMeMerchantAvailable]
        """

        return self._merchant_available

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._uuid is not None:
            return False

        if self._amount_inquired is not None:
            return False

        if self._alias is not None:
            return False

        if self._description is not None:
            return False

        if self._status is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._merchant_available is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeTabEntry
        """

        return converter.json_to_class(BunqMeTabEntry, json_str)


class BunqMeTabResultInquiry(BunqModel):
    """
    Used to view bunq.me TabResultInquiry objects belonging to a tab. A
    TabResultInquiry is an object that holds details on both the tab and a
    single payment made for that tab.
    
    :param _payment: The payment made for the Tab.
    :type _payment: Payment
    :param _bunq_me_tab_id: The Id of the bunq.me tab that this
    BunqMeTabResultInquiry belongs to.
    :type _bunq_me_tab_id: int
    """

    _payment = None
    _bunq_me_tab_id = None

    @property
    def payment(self):
        """
        :rtype: Payment
        """

        return self._payment

    @property
    def bunq_me_tab_id(self):
        """
        :rtype: int
        """

        return self._bunq_me_tab_id

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._payment is not None:
            return False

        if self._bunq_me_tab_id is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: BunqMeTabResultInquiry
        """

        return converter.json_to_class(BunqMeTabResultInquiry, json_str)


class CardBatch(BunqModel):
    """
    Used to update multiple cards in a batch.
    
    :param _cards: The cards that need to be updated.
    :type _cards: list[object_.CardBatchEntry]
    :param _updated_card_ids: The ids of the cards that have been updated.
    :type _updated_card_ids: list[object_.BunqId]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/card-batch"

    # Field constants.
    FIELD_CARDS = "cards"

    # Object type.
    _OBJECT_TYPE_POST = "CardBatch"

    _updated_card_ids = None
    _cards_field_for_request = None

    def __init__(self, cards):
        """
        :param cards: The cards that need to be updated.
        :type cards: list[object_.CardBatchEntry]
        """

        self._cards_field_for_request = cards

    @classmethod
    def create(cls, cards, custom_headers=None):
        """
        :type user_id: int
        :param cards: The cards that need to be updated.
        :type cards: list[object_.CardBatchEntry]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardBatch
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CARDS: cards
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseCardBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_POST)
        )

    @property
    def updated_card_ids(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._updated_card_ids

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._updated_card_ids is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardBatch
        """

        return converter.json_to_class(CardBatch, json_str)


class CardCredit(BunqModel):
    """
    With bunq it is possible to order credit cards that can then be connected
    with each one of the monetary accounts the user has access to (including
    connected accounts).
    
    :param _second_line: The second line of text on the card
    :type _second_line: str
    :param _name_on_card: The user's name on the card.
    :type _name_on_card: str
    :param _alias: The pointer to the monetary account that will be connected at
    first with the card. Its IBAN code is also the one that will be printed on
    the card itself. The pointer must be of type IBAN.
    :type _alias: object_.Pointer
    :param _type_: The type of the card. Can is MASTERCARD.
    :type _type_: str
    :param _product_type: The product type of the card.
    :type _product_type: str
    :param _pin_code_assignment: Array of Types, PINs, account IDs assigned to
    the card.
    :type _pin_code_assignment: list[object_.CardPinAssignment]
    :param _monetary_account_id_fallback: ID of the MA to be used as fallback
    for this card if insufficient balance. Fallback account is removed if not
    supplied.
    :type _monetary_account_id_fallback: int
    :param _id_: The id of the card.
    :type _id_: int
    :param _created: The timestamp of the card's creation.
    :type _created: str
    :param _updated: The timestamp of the card's last update.
    :type _updated: str
    :param _public_uuid: The public UUID of the card.
    :type _public_uuid: str
    :param _sub_type: The sub-type of the card.
    :type _sub_type: str
    :param _status: The status to set for the card. Can be ACTIVE, DEACTIVATED,
    LOST, STOLEN, CANCELLED, EXPIRED or PIN_TRIES_EXCEEDED.
    :type _status: str
    :param _sub_status: The sub-status of the card. Can be NONE or REPLACED.
    :type _sub_status: str
    :param _order_status: The order status of the card. Can be
    NEW_CARD_REQUEST_RECEIVED, CARD_REQUEST_PENDING, SENT_FOR_PRODUCTION,
    ACCEPTED_FOR_PRODUCTION, DELIVERED_TO_CUSTOMER, CARD_UPDATE_REQUESTED,
    CARD_UPDATE_PENDING, CARD_UPDATE_SENT, CARD_UPDATE_ACCEPTED,
    VIRTUAL_DELIVERY, NEW_CARD_REQUEST_PENDING_USER_APPROVAL, SENT_FOR_DELIVERY
    or NEW_CARD_REQUEST_CANCELLED.
    :type _order_status: str
    :param _expiry_date: Expiry date of the card.
    :type _expiry_date: str
    :param _card_limit: The spending limit for the card.
    :type _card_limit: object_.Amount
    :param _card_limit_atm: The ATM spending limit for the card.
    :type _card_limit_atm: object_.Amount
    :param _country_permission: The countries for which to grant (temporary)
    permissions to use the card.
    :type _country_permission: list[object_.CardCountryPermission]
    :param _label_monetary_account_ordered: The monetary account this card was
    ordered on and the label user that owns the card.
    :type _label_monetary_account_ordered: object_.MonetaryAccountReference
    :param _label_monetary_account_current: The monetary account that this card
    is currently linked to and the label user viewing it.
    :type _label_monetary_account_current: object_.MonetaryAccountReference
    :param _country: The country that is domestic to the card. Defaults to
    country of residence of user.
    :type _country: str
    :param _card_shipment_tracking_url: A tracking link provided by our shipment
    provider.
    :type _card_shipment_tracking_url: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/card-credit"

    # Field constants.
    FIELD_SECOND_LINE = "second_line"
    FIELD_NAME_ON_CARD = "name_on_card"
    FIELD_ALIAS = "alias"
    FIELD_TYPE = "type"
    FIELD_PRODUCT_TYPE = "product_type"
    FIELD_PIN_CODE_ASSIGNMENT = "pin_code_assignment"
    FIELD_MONETARY_ACCOUNT_ID_FALLBACK = "monetary_account_id_fallback"

    # Object type.
    _OBJECT_TYPE_POST = "CardCredit"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _type_ = None
    _sub_type = None
    _product_type = None
    _second_line = None
    _status = None
    _sub_status = None
    _order_status = None
    _expiry_date = None
    _name_on_card = None
    _card_limit = None
    _card_limit_atm = None
    _country_permission = None
    _label_monetary_account_ordered = None
    _label_monetary_account_current = None
    _pin_code_assignment = None
    _monetary_account_id_fallback = None
    _country = None
    _card_shipment_tracking_url = None
    _second_line_field_for_request = None
    _name_on_card_field_for_request = None
    _alias_field_for_request = None
    _type__field_for_request = None
    _product_type_field_for_request = None
    _pin_code_assignment_field_for_request = None
    _monetary_account_id_fallback_field_for_request = None

    def __init__(self, second_line, name_on_card, type_, alias=None, product_type=None, pin_code_assignment=None,
                 monetary_account_id_fallback=None):
        """
        :param second_line: The second line of text on the card, used as
        name/description for it. It can contain at most 17 characters and it can be
        empty.
        :type second_line: str
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param type_: The type of card to order. Can be MASTERCARD.
        :type type_: str
        :param alias: The pointer to the monetary account that will be connected at
        first with the card. Its IBAN code is also the one that will be printed on
        the card itself. The pointer must be of type IBAN.
        :type alias: object_.Pointer
        :param product_type: The product type of the card to order.
        :type product_type: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned to
        the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback for
        this card if insufficient balance. Fallback account is removed if not
        supplied.
        :type monetary_account_id_fallback: int
        """

        self._second_line_field_for_request = second_line
        self._name_on_card_field_for_request = name_on_card
        self._type__field_for_request = type_
        self._alias_field_for_request = alias
        self._product_type_field_for_request = product_type
        self._pin_code_assignment_field_for_request = pin_code_assignment
        self._monetary_account_id_fallback_field_for_request = monetary_account_id_fallback

    @classmethod
    def create(cls, second_line, name_on_card, type_, alias=None, product_type=None, pin_code_assignment=None,
               monetary_account_id_fallback=None, custom_headers=None):
        """
        Create a new credit card request.
        
        :type user_id: int
        :param second_line: The second line of text on the card, used as
        name/description for it. It can contain at most 17 characters and it can
        be empty.
        :type second_line: str
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param type_: The type of card to order. Can be MASTERCARD.
        :type type_: str
        :param alias: The pointer to the monetary account that will be connected
        at first with the card. Its IBAN code is also the one that will be
        printed on the card itself. The pointer must be of type IBAN.
        :type alias: object_.Pointer
        :param product_type: The product type of the card to order.
        :type product_type: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned
        to the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback
        for this card if insufficient balance. Fallback account is removed if
        not supplied.
        :type monetary_account_id_fallback: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardCredit
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_SECOND_LINE: second_line,
            cls.FIELD_NAME_ON_CARD: name_on_card,
            cls.FIELD_ALIAS: alias,
            cls.FIELD_TYPE: type_,
            cls.FIELD_PRODUCT_TYPE: product_type,
            cls.FIELD_PIN_CODE_ASSIGNMENT: pin_code_assignment,
            cls.FIELD_MONETARY_ACCOUNT_ID_FALLBACK: monetary_account_id_fallback
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseCardCredit.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_POST)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def sub_type(self):
        """
        :rtype: str
        """

        return self._sub_type

    @property
    def product_type(self):
        """
        :rtype: str
        """

        return self._product_type

    @property
    def second_line(self):
        """
        :rtype: str
        """

        return self._second_line

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def order_status(self):
        """
        :rtype: str
        """

        return self._order_status

    @property
    def expiry_date(self):
        """
        :rtype: str
        """

        return self._expiry_date

    @property
    def name_on_card(self):
        """
        :rtype: str
        """

        return self._name_on_card

    @property
    def card_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._card_limit

    @property
    def card_limit_atm(self):
        """
        :rtype: object_.Amount
        """

        return self._card_limit_atm

    @property
    def country_permission(self):
        """
        :rtype: list[object_.CardCountryPermission]
        """

        return self._country_permission

    @property
    def label_monetary_account_ordered(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_ordered

    @property
    def label_monetary_account_current(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_current

    @property
    def pin_code_assignment(self):
        """
        :rtype: list[object_.CardPinAssignment]
        """

        return self._pin_code_assignment

    @property
    def monetary_account_id_fallback(self):
        """
        :rtype: int
        """

        return self._monetary_account_id_fallback

    @property
    def country(self):
        """
        :rtype: str
        """

        return self._country

    @property
    def card_shipment_tracking_url(self):
        """
        :rtype: str
        """

        return self._card_shipment_tracking_url

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._type_ is not None:
            return False

        if self._sub_type is not None:
            return False

        if self._product_type is not None:
            return False

        if self._second_line is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._order_status is not None:
            return False

        if self._expiry_date is not None:
            return False

        if self._name_on_card is not None:
            return False

        if self._card_limit is not None:
            return False

        if self._card_limit_atm is not None:
            return False

        if self._country_permission is not None:
            return False

        if self._label_monetary_account_ordered is not None:
            return False

        if self._label_monetary_account_current is not None:
            return False

        if self._pin_code_assignment is not None:
            return False

        if self._monetary_account_id_fallback is not None:
            return False

        if self._country is not None:
            return False

        if self._card_shipment_tracking_url is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardCredit
        """

        return converter.json_to_class(CardCredit, json_str)


class CardDebit(BunqModel):
    """
    With bunq it is possible to order debit cards that can then be connected
    with each one of the monetary accounts the user has access to (including
    connected accounts).
    
    :param _second_line: The second line of text on the card
    :type _second_line: str
    :param _name_on_card: The user's name as will be on the card
    :type _name_on_card: str
    :param _alias: The label for the user who requested the card.
    :type _alias: object_.LabelUser
    :param _type_: The type of the card. Can be MAESTRO, MASTERCARD.
    :type _type_: str
    :param _product_type: The product type of the card to order.
    :type _product_type: str
    :param _pin_code_assignment: Array of Types, PINs, account IDs assigned to
    the card.
    :type _pin_code_assignment: list[object_.CardPinAssignment]
    :param _monetary_account_id_fallback: ID of the MA to be used as fallback
    for this card if insufficient balance. Fallback account is removed if not
    supplied.
    :type _monetary_account_id_fallback: int
    :param _id_: The id of the card.
    :type _id_: int
    :param _created: The timestamp when the card was crated.
    :type _created: str
    :param _updated: The timestamp when the card was last updated.
    :type _updated: str
    :param _public_uuid: The public UUID of the card.
    :type _public_uuid: str
    :param _sub_type: The sub_type of card.
    :type _sub_type: str
    :param _status: The status to set for the card. After ordering the card it
    will be DEACTIVATED.
    :type _status: str
    :param _order_status: The order status of the card. Can be
    NEW_CARD_REQUEST_RECEIVED, CARD_REQUEST_PENDING, SENT_FOR_PRODUCTION,
    ACCEPTED_FOR_PRODUCTION, DELIVERED_TO_CUSTOMER, CARD_UPDATE_REQUESTED,
    CARD_UPDATE_PENDING, CARD_UPDATE_SENT, CARD_UPDATE_ACCEPTED,
    VIRTUAL_DELIVERY, NEW_CARD_REQUEST_PENDING_USER_APPROVAL, SENT_FOR_DELIVERY
    or NEW_CARD_REQUEST_CANCELLED.
    :type _order_status: str
    :param _expiry_date: The expiry date of the card.
    :type _expiry_date: str
    :param _country_permission: The countries for which to grant (temporary)
    permissions to use the card.
    :type _country_permission: list[object_.CardCountryPermission]
    :param _label_monetary_account_ordered: The monetary account this card was
    ordered on and the label user that owns the card.
    :type _label_monetary_account_ordered: object_.MonetaryAccountReference
    :param _label_monetary_account_current: The monetary account that this card
    is currently linked to and the label user viewing it.
    :type _label_monetary_account_current: object_.MonetaryAccountReference
    :param _country: The country that is domestic to the card. Defaults to
    country of residence of user.
    :type _country: str
    :param _card_shipment_tracking_url: A tracking link provided by our shipment
    provider.
    :type _card_shipment_tracking_url: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/card-debit"

    # Field constants.
    FIELD_SECOND_LINE = "second_line"
    FIELD_NAME_ON_CARD = "name_on_card"
    FIELD_ALIAS = "alias"
    FIELD_TYPE = "type"
    FIELD_PRODUCT_TYPE = "product_type"
    FIELD_PIN_CODE_ASSIGNMENT = "pin_code_assignment"
    FIELD_MONETARY_ACCOUNT_ID_FALLBACK = "monetary_account_id_fallback"

    # Object type.
    _OBJECT_TYPE_POST = "CardDebit"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _type_ = None
    _sub_type = None
    _second_line = None
    _name_on_card = None
    _status = None
    _order_status = None
    _expiry_date = None
    _country_permission = None
    _label_monetary_account_ordered = None
    _label_monetary_account_current = None
    _alias = None
    _pin_code_assignment = None
    _monetary_account_id_fallback = None
    _country = None
    _card_shipment_tracking_url = None
    _second_line_field_for_request = None
    _name_on_card_field_for_request = None
    _alias_field_for_request = None
    _type__field_for_request = None
    _product_type_field_for_request = None
    _pin_code_assignment_field_for_request = None
    _monetary_account_id_fallback_field_for_request = None

    def __init__(self, second_line, name_on_card, type_, alias=None, product_type=None, pin_code_assignment=None,
                 monetary_account_id_fallback=None):
        """
        :param second_line: The second line of text on the card, used as
        name/description for it. It can contain at most 17 characters and it can be
        empty.
        :type second_line: str
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param type_: The type of card to order. Can be MAESTRO or MASTERCARD.
        :type type_: str
        :param alias: The pointer to the monetary account that will be connected at
        first with the card. Its IBAN code is also the one that will be printed on
        the card itself. The pointer must be of type IBAN.
        :type alias: object_.Pointer
        :param product_type: The product type of the card to order.
        :type product_type: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned to
        the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback for
        this card if insufficient balance. Fallback account is removed if not
        supplied.
        :type monetary_account_id_fallback: int
        """

        self._second_line_field_for_request = second_line
        self._name_on_card_field_for_request = name_on_card
        self._type__field_for_request = type_
        self._alias_field_for_request = alias
        self._product_type_field_for_request = product_type
        self._pin_code_assignment_field_for_request = pin_code_assignment
        self._monetary_account_id_fallback_field_for_request = monetary_account_id_fallback

    @classmethod
    def create(cls, second_line, name_on_card, type_, alias=None, product_type=None, pin_code_assignment=None,
               monetary_account_id_fallback=None, custom_headers=None):
        """
        Create a new debit card request.
        
        :type user_id: int
        :param second_line: The second line of text on the card, used as
        name/description for it. It can contain at most 17 characters and it can
        be empty.
        :type second_line: str
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param type_: The type of card to order. Can be MAESTRO or MASTERCARD.
        :type type_: str
        :param alias: The pointer to the monetary account that will be connected
        at first with the card. Its IBAN code is also the one that will be
        printed on the card itself. The pointer must be of type IBAN.
        :type alias: object_.Pointer
        :param product_type: The product type of the card to order.
        :type product_type: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned
        to the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback
        for this card if insufficient balance. Fallback account is removed if
        not supplied.
        :type monetary_account_id_fallback: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardDebit
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_SECOND_LINE: second_line,
            cls.FIELD_NAME_ON_CARD: name_on_card,
            cls.FIELD_ALIAS: alias,
            cls.FIELD_TYPE: type_,
            cls.FIELD_PRODUCT_TYPE: product_type,
            cls.FIELD_PIN_CODE_ASSIGNMENT: pin_code_assignment,
            cls.FIELD_MONETARY_ACCOUNT_ID_FALLBACK: monetary_account_id_fallback
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseCardDebit.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_POST)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def sub_type(self):
        """
        :rtype: str
        """

        return self._sub_type

    @property
    def second_line(self):
        """
        :rtype: str
        """

        return self._second_line

    @property
    def name_on_card(self):
        """
        :rtype: str
        """

        return self._name_on_card

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def order_status(self):
        """
        :rtype: str
        """

        return self._order_status

    @property
    def expiry_date(self):
        """
        :rtype: str
        """

        return self._expiry_date

    @property
    def country_permission(self):
        """
        :rtype: list[object_.CardCountryPermission]
        """

        return self._country_permission

    @property
    def label_monetary_account_ordered(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_ordered

    @property
    def label_monetary_account_current(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_current

    @property
    def alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._alias

    @property
    def pin_code_assignment(self):
        """
        :rtype: list[object_.CardPinAssignment]
        """

        return self._pin_code_assignment

    @property
    def monetary_account_id_fallback(self):
        """
        :rtype: int
        """

        return self._monetary_account_id_fallback

    @property
    def country(self):
        """
        :rtype: str
        """

        return self._country

    @property
    def card_shipment_tracking_url(self):
        """
        :rtype: str
        """

        return self._card_shipment_tracking_url

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._type_ is not None:
            return False

        if self._sub_type is not None:
            return False

        if self._second_line is not None:
            return False

        if self._name_on_card is not None:
            return False

        if self._status is not None:
            return False

        if self._order_status is not None:
            return False

        if self._expiry_date is not None:
            return False

        if self._country_permission is not None:
            return False

        if self._label_monetary_account_ordered is not None:
            return False

        if self._label_monetary_account_current is not None:
            return False

        if self._alias is not None:
            return False

        if self._pin_code_assignment is not None:
            return False

        if self._monetary_account_id_fallback is not None:
            return False

        if self._country is not None:
            return False

        if self._card_shipment_tracking_url is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardDebit
        """

        return converter.json_to_class(CardDebit, json_str)


class CardGeneratedCvc2(BunqModel):
    """
    Endpoint for generating and retrieving a new CVC2 code.
    
    :param _type_: The type of generated cvc2. Can be STATIC or GENERATED.
    :type _type_: str
    :param _id_: The id of the cvc code.
    :type _id_: int
    :param _created: The timestamp of the cvc code's creation.
    :type _created: str
    :param _updated: The timestamp of the cvc code's last update.
    :type _updated: str
    :param _cvc2: The cvc2 code.
    :type _cvc2: str
    :param _status: The status of the cvc2. Can be AVAILABLE, USED, EXPIRED,
    BLOCKED.
    :type _status: str
    :param _expiry_time: Expiry time of the cvc2.
    :type _expiry_time: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/card/{}/generated-cvc2"
    _ENDPOINT_URL_READ = "user/{}/card/{}/generated-cvc2/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/card/{}/generated-cvc2/{}"
    _ENDPOINT_URL_LISTING = "user/{}/card/{}/generated-cvc2"

    # Field constants.
    FIELD_TYPE = "type"

    # Object type.
    _OBJECT_TYPE_GET = "CardGeneratedCvc2"

    _id_ = None
    _created = None
    _updated = None
    _type_ = None
    _cvc2 = None
    _status = None
    _expiry_time = None
    _type__field_for_request = None

    def __init__(self, type_=None):
        """
        :param type_: The type of generated cvc2. Can be STATIC or GENERATED.
        :type type_: str
        """

        self._type__field_for_request = type_

    @classmethod
    def create(cls, card_id, type_=None, custom_headers=None):
        """
        Generate a new CVC2 code for a card.
        
        :type user_id: int
        :type card_id: int
        :param type_: The type of generated cvc2. Can be STATIC or GENERATED.
        :type type_: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_TYPE: type_
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(), card_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, card_id, card_generated_cvc2_id, custom_headers=None):
        """
        Get the details for a specific generated CVC2 code.
        
        :type api_context: ApiContext
        :type user_id: int
        :type card_id: int
        :type card_generated_cvc2_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardGeneratedCvc2
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), card_id, card_generated_cvc2_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCardGeneratedCvc2.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, card_id, card_generated_cvc2_id, type_=None, custom_headers=None):
        """
        :type user_id: int
        :type card_id: int
        :type card_generated_cvc2_id: int
        :param type_: The type of generated cvc2. Can be STATIC or GENERATED.
        :type type_: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_TYPE: type_
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), card_id, card_generated_cvc2_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, card_id, params=None, custom_headers=None):
        """
        Get all generated CVC2 codes for a card.
        
        :type user_id: int
        :type card_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardGeneratedCvc2List
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), card_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCardGeneratedCvc2List.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def cvc2(self):
        """
        :rtype: str
        """

        return self._cvc2

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def expiry_time(self):
        """
        :rtype: str
        """

        return self._expiry_time

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._type_ is not None:
            return False

        if self._cvc2 is not None:
            return False

        if self._status is not None:
            return False

        if self._expiry_time is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardGeneratedCvc2
        """

        return converter.json_to_class(CardGeneratedCvc2, json_str)


class CardName(BunqModel):
    """
    Endpoint for getting all the accepted card names for a user. As bunq do not
    allow total freedom in choosing the name that is going to be printed on the
    card, the following formats are accepted: Name Surname, N. Surname, N
    Surname or Surname.
    
    :param _possible_card_name_array: All possible variations (of suitable
    length) of user's legal name for the debit card.
    :type _possible_card_name_array: list[str]
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/card-name"

    # Object type.
    _OBJECT_TYPE_GET = "CardUserNameArray"

    _possible_card_name_array = None

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Return all the accepted card names for a specific user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardNameList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCardNameList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def possible_card_name_array(self):
        """
        :rtype: list[str]
        """

        return self._possible_card_name_array

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._possible_card_name_array is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardName
        """

        return converter.json_to_class(CardName, json_str)


class CardReplace(BunqModel):
    """
    It is possible to order a card replacement with the bunq API.<br/><br/>You
    can order up to one free card replacement per year. Additional replacement
    requests will be billed.<br/><br/>The card replacement will have the same
    expiry date and the same pricing as the old card, but it will have a new
    card number. You can change the description and optional the PIN through the
    card replacement endpoint.
    
    :param _name_on_card: The user's name as it will be on the card. Check
    'card-name' for the available card names for a user.
    :type _name_on_card: str
    :param _pin_code_assignment: Array of Types, PINs, account IDs assigned to
    the card.
    :type _pin_code_assignment: list[object_.CardPinAssignment]
    :param _second_line: The second line on the card.
    :type _second_line: str
    :param _id_: The id of the new card.
    :type _id_: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/card/{}/replace"

    # Field constants.
    FIELD_NAME_ON_CARD = "name_on_card"
    FIELD_PIN_CODE_ASSIGNMENT = "pin_code_assignment"
    FIELD_SECOND_LINE = "second_line"

    _id_ = None
    _name_on_card_field_for_request = None
    _pin_code_assignment_field_for_request = None
    _second_line_field_for_request = None

    def __init__(self, name_on_card=None, pin_code_assignment=None, second_line=None):
        """
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned to
        the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param second_line: The second line on the card.
        :type second_line: str
        """

        self._name_on_card_field_for_request = name_on_card
        self._pin_code_assignment_field_for_request = pin_code_assignment
        self._second_line_field_for_request = second_line

    @classmethod
    def create(cls, card_id, name_on_card=None, pin_code_assignment=None, second_line=None, custom_headers=None):
        """
        Request a card replacement.
        
        :type user_id: int
        :type card_id: int
        :param name_on_card: The user's name as it will be on the card. Check
        'card-name' for the available card names for a user.
        :type name_on_card: str
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned
        to the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param second_line: The second line on the card.
        :type second_line: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NAME_ON_CARD: name_on_card,
            cls.FIELD_PIN_CODE_ASSIGNMENT: pin_code_assignment,
            cls.FIELD_SECOND_LINE: second_line
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(), card_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CardReplace
        """

        return converter.json_to_class(CardReplace, json_str)


class Card(BunqModel):
    """
    Endpoint for retrieving details for the cards the user has access to.
    
    :param _pin_code: The plaintext pin code. Requests require encryption to be
    enabled.
    :type _pin_code: str
    :param _activation_code: DEPRECATED: Activate a card by setting status to
    ACTIVE when the order_status is ACCEPTED_FOR_PRODUCTION.
    :type _activation_code: str
    :param _status: The status to set for the card. Can be ACTIVE, DEACTIVATED,
    LOST, STOLEN, CANCELLED, EXPIRED or PIN_TRIES_EXCEEDED.
    :type _status: str
    :param _card_limit: The spending limit for the card.
    :type _card_limit: object_.Amount
    :param _card_limit_atm: The ATM spending limit for the card.
    :type _card_limit_atm: object_.Amount
    :param _country_permission: The countries for which to grant (temporary)
    permissions to use the card.
    :type _country_permission: list[object_.CardCountryPermission]
    :param _pin_code_assignment: Array of Types, PINs, account IDs assigned to
    the card.
    :type _pin_code_assignment: list[object_.CardPinAssignment]
    :param _primary_account_numbers: Array of PANs and their attributes.
    :type _primary_account_numbers: list[object_.CardPrimaryAccountNumber]
    :param _monetary_account_id_fallback: ID of the MA to be used as fallback
    for this card if insufficient balance. Fallback account is removed if not
    supplied.
    :type _monetary_account_id_fallback: int
    :param _id_: The id of the card.
    :type _id_: int
    :param _created: The timestamp of the card's creation.
    :type _created: str
    :param _updated: The timestamp of the card's last update.
    :type _updated: str
    :param _public_uuid: The public UUID of the card.
    :type _public_uuid: str
    :param _type_: The type of the card. Can be MAESTRO, MASTERCARD.
    :type _type_: str
    :param _sub_type: The sub-type of the card.
    :type _sub_type: str
    :param _second_line: The second line of text on the card
    :type _second_line: str
    :param _user_id: ID of the user who is owner of the card.
    :type _user_id: int
    :param _sub_status: The sub-status of the card. Can be NONE or REPLACED.
    :type _sub_status: str
    :param _order_status: The order status of the card. Can be
    NEW_CARD_REQUEST_RECEIVED, CARD_REQUEST_PENDING, SENT_FOR_PRODUCTION,
    ACCEPTED_FOR_PRODUCTION, DELIVERED_TO_CUSTOMER, CARD_UPDATE_REQUESTED,
    CARD_UPDATE_PENDING, CARD_UPDATE_SENT, CARD_UPDATE_ACCEPTED,
    VIRTUAL_DELIVERY, NEW_CARD_REQUEST_PENDING_USER_APPROVAL, SENT_FOR_DELIVERY
    or NEW_CARD_REQUEST_CANCELLED.
    :type _order_status: str
    :param _expiry_date: Expiry date of the card.
    :type _expiry_date: str
    :param _name_on_card: The user's name on the card.
    :type _name_on_card: str
    :param _label_monetary_account_ordered: The monetary account this card was
    ordered on and the label user that owns the card.
    :type _label_monetary_account_ordered: object_.MonetaryAccountReference
    :param _label_monetary_account_current: The monetary account that this card
    is currently linked to and the label user viewing it.
    :type _label_monetary_account_current: object_.MonetaryAccountReference
    :param _country: The country that is domestic to the card. Defaults to
    country of residence of user.
    :type _country: str
    :param _card_shipment_tracking_url: A tracking link provided by our shipment
    provider.
    :type _card_shipment_tracking_url: str
    :param _amount_saved_zero_fx: The amount saved through ZeroFX on this card.
    :type _amount_saved_zero_fx: object_.Amount
    """

    # Endpoint constants.
    _ENDPOINT_URL_UPDATE = "user/{}/card/{}"
    _ENDPOINT_URL_READ = "user/{}/card/{}"
    _ENDPOINT_URL_LISTING = "user/{}/card"

    # Field constants.
    FIELD_PIN_CODE = "pin_code"
    FIELD_ACTIVATION_CODE = "activation_code"
    FIELD_STATUS = "status"
    FIELD_CARD_LIMIT = "card_limit"
    FIELD_CARD_LIMIT_ATM = "card_limit_atm"
    FIELD_COUNTRY_PERMISSION = "country_permission"
    FIELD_PIN_CODE_ASSIGNMENT = "pin_code_assignment"
    FIELD_PRIMARY_ACCOUNT_NUMBERS = "primary_account_numbers"
    FIELD_MONETARY_ACCOUNT_ID_FALLBACK = "monetary_account_id_fallback"

    # Object type.
    _OBJECT_TYPE_PUT = "CardDebit"
    _OBJECT_TYPE_GET = "CardDebit"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _type_ = None
    _sub_type = None
    _second_line = None
    _user_id = None
    _status = None
    _sub_status = None
    _order_status = None
    _expiry_date = None
    _name_on_card = None
    _primary_account_numbers = None
    _card_limit = None
    _card_limit_atm = None
    _country_permission = None
    _label_monetary_account_ordered = None
    _label_monetary_account_current = None
    _pin_code_assignment = None
    _monetary_account_id_fallback = None
    _country = None
    _card_shipment_tracking_url = None
    _amount_saved_zero_fx = None
    _pin_code_field_for_request = None
    _activation_code_field_for_request = None
    _status_field_for_request = None
    _card_limit_field_for_request = None
    _card_limit_atm_field_for_request = None
    _country_permission_field_for_request = None
    _pin_code_assignment_field_for_request = None
    _primary_account_numbers_field_for_request = None
    _monetary_account_id_fallback_field_for_request = None

    def __init__(self, pin_code=None, activation_code=None, status=None, card_limit=None, card_limit_atm=None,
                 country_permission=None, pin_code_assignment=None, primary_account_numbers=None,
                 monetary_account_id_fallback=None):
        """
        :param pin_code: The plaintext pin code. Requests require encryption to be
        enabled.
        :type pin_code: str
        :param activation_code: DEPRECATED: Activate a card by setting status to
        ACTIVE when the order_status is ACCEPTED_FOR_PRODUCTION.
        :type activation_code: str
        :param status: The status to set for the card. Can be ACTIVE, DEACTIVATED,
        LOST, STOLEN or CANCELLED, and can only be set to LOST/STOLEN/CANCELLED when
        order status is
        ACCEPTED_FOR_PRODUCTION/DELIVERED_TO_CUSTOMER/CARD_UPDATE_REQUESTED/CARD_UPDATE_SENT/CARD_UPDATE_ACCEPTED.
        Can only be set to DEACTIVATED after initial activation, i.e. order_status
        is
        DELIVERED_TO_CUSTOMER/CARD_UPDATE_REQUESTED/CARD_UPDATE_SENT/CARD_UPDATE_ACCEPTED.
        Mind that all the possible choices (apart from ACTIVE and DEACTIVATED) are
        permanent and cannot be changed after.
        :type status: str
        :param card_limit: The spending limit for the card.
        :type card_limit: object_.Amount
        :param card_limit_atm: The ATM spending limit for the card.
        :type card_limit_atm: object_.Amount
        :param country_permission: The countries for which to grant (temporary)
        permissions to use the card.
        :type country_permission: list[object_.CardCountryPermission]
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned to
        the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param primary_account_numbers: Array of PANs and their attributes.
        :type primary_account_numbers: list[object_.CardPrimaryAccountNumber]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback for
        this card if insufficient balance. Fallback account is removed if not
        supplied.
        :type monetary_account_id_fallback: int
        """

        self._pin_code_field_for_request = pin_code
        self._activation_code_field_for_request = activation_code
        self._status_field_for_request = status
        self._card_limit_field_for_request = card_limit
        self._card_limit_atm_field_for_request = card_limit_atm
        self._country_permission_field_for_request = country_permission
        self._pin_code_assignment_field_for_request = pin_code_assignment
        self._primary_account_numbers_field_for_request = primary_account_numbers
        self._monetary_account_id_fallback_field_for_request = monetary_account_id_fallback

    @classmethod
    def update(cls, card_id, pin_code=None, activation_code=None, status=None, card_limit=None, card_limit_atm=None,
               country_permission=None, pin_code_assignment=None, primary_account_numbers=None,
               monetary_account_id_fallback=None, custom_headers=None):
        """
        Update the card details. Allow to change pin code, status, limits,
        country permissions and the monetary account connected to the card. When
        the card has been received, it can be also activated through this
        endpoint.
        
        :type user_id: int
        :type card_id: int
        :param pin_code: The plaintext pin code. Requests require encryption to
        be enabled.
        :type pin_code: str
        :param activation_code: DEPRECATED: Activate a card by setting status to
        ACTIVE when the order_status is ACCEPTED_FOR_PRODUCTION.
        :type activation_code: str
        :param status: The status to set for the card. Can be ACTIVE,
        DEACTIVATED, LOST, STOLEN or CANCELLED, and can only be set to
        LOST/STOLEN/CANCELLED when order status is
        ACCEPTED_FOR_PRODUCTION/DELIVERED_TO_CUSTOMER/CARD_UPDATE_REQUESTED/CARD_UPDATE_SENT/CARD_UPDATE_ACCEPTED.
        Can only be set to DEACTIVATED after initial activation, i.e.
        order_status is
        DELIVERED_TO_CUSTOMER/CARD_UPDATE_REQUESTED/CARD_UPDATE_SENT/CARD_UPDATE_ACCEPTED.
        Mind that all the possible choices (apart from ACTIVE and DEACTIVATED)
        are permanent and cannot be changed after.
        :type status: str
        :param card_limit: The spending limit for the card.
        :type card_limit: object_.Amount
        :param card_limit_atm: The ATM spending limit for the card.
        :type card_limit_atm: object_.Amount
        :param country_permission: The countries for which to grant (temporary)
        permissions to use the card.
        :type country_permission: list[object_.CardCountryPermission]
        :param pin_code_assignment: Array of Types, PINs, account IDs assigned
        to the card.
        :type pin_code_assignment: list[object_.CardPinAssignment]
        :param primary_account_numbers: Array of PANs and their attributes.
        :type primary_account_numbers: list[object_.CardPrimaryAccountNumber]
        :param monetary_account_id_fallback: ID of the MA to be used as fallback
        for this card if insufficient balance. Fallback account is removed if
        not supplied.
        :type monetary_account_id_fallback: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCard
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_PIN_CODE: pin_code,
            cls.FIELD_ACTIVATION_CODE: activation_code,
            cls.FIELD_STATUS: status,
            cls.FIELD_CARD_LIMIT: card_limit,
            cls.FIELD_CARD_LIMIT_ATM: card_limit_atm,
            cls.FIELD_COUNTRY_PERMISSION: country_permission,
            cls.FIELD_PIN_CODE_ASSIGNMENT: pin_code_assignment,
            cls.FIELD_PRIMARY_ACCOUNT_NUMBERS: primary_account_numbers,
            cls.FIELD_MONETARY_ACCOUNT_ID_FALLBACK: monetary_account_id_fallback
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), card_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseCard.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_PUT)
        )

    @classmethod
    def get(cls, card_id, custom_headers=None):
        """
        Return the details of a specific card.
        
        :type api_context: ApiContext
        :type user_id: int
        :type card_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCard
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), card_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCard.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Return all the cards available to the user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCardList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCardList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def sub_type(self):
        """
        :rtype: str
        """

        return self._sub_type

    @property
    def second_line(self):
        """
        :rtype: str
        """

        return self._second_line

    @property
    def user_id(self):
        """
        :rtype: int
        """

        return self._user_id

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def order_status(self):
        """
        :rtype: str
        """

        return self._order_status

    @property
    def expiry_date(self):
        """
        :rtype: str
        """

        return self._expiry_date

    @property
    def name_on_card(self):
        """
        :rtype: str
        """

        return self._name_on_card

    @property
    def primary_account_numbers(self):
        """
        :rtype: list[object_.CardPrimaryAccountNumber]
        """

        return self._primary_account_numbers

    @property
    def card_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._card_limit

    @property
    def card_limit_atm(self):
        """
        :rtype: object_.Amount
        """

        return self._card_limit_atm

    @property
    def country_permission(self):
        """
        :rtype: list[object_.CardCountryPermission]
        """

        return self._country_permission

    @property
    def label_monetary_account_ordered(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_ordered

    @property
    def label_monetary_account_current(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._label_monetary_account_current

    @property
    def pin_code_assignment(self):
        """
        :rtype: list[object_.CardPinAssignment]
        """

        return self._pin_code_assignment

    @property
    def monetary_account_id_fallback(self):
        """
        :rtype: int
        """

        return self._monetary_account_id_fallback

    @property
    def country(self):
        """
        :rtype: str
        """

        return self._country

    @property
    def card_shipment_tracking_url(self):
        """
        :rtype: str
        """

        return self._card_shipment_tracking_url

    @property
    def amount_saved_zero_fx(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_saved_zero_fx

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._type_ is not None:
            return False

        if self._sub_type is not None:
            return False

        if self._second_line is not None:
            return False

        if self._user_id is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._order_status is not None:
            return False

        if self._expiry_date is not None:
            return False

        if self._name_on_card is not None:
            return False

        if self._primary_account_numbers is not None:
            return False

        if self._card_limit is not None:
            return False

        if self._card_limit_atm is not None:
            return False

        if self._country_permission is not None:
            return False

        if self._label_monetary_account_ordered is not None:
            return False

        if self._label_monetary_account_current is not None:
            return False

        if self._pin_code_assignment is not None:
            return False

        if self._monetary_account_id_fallback is not None:
            return False

        if self._country is not None:
            return False

        if self._card_shipment_tracking_url is not None:
            return False

        if self._amount_saved_zero_fx is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Card
        """

        return converter.json_to_class(Card, json_str)


class CashRegisterQrCodeContent(BunqModel):
    """
    Show the raw contents of a QR code. First you need to created a QR code
    using ../cash-register/{id}/qr-code.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/qr-code/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "CashRegisterQrCodeContent"

    @classmethod
    def list(cls, cash_register_id, qr_code_id, monetary_account_id=None, custom_headers=None):
        """
        Show the raw contents of a QR code
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type qr_code_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id, qr_code_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CashRegisterQrCodeContent
        """

        return converter.json_to_class(CashRegisterQrCodeContent, json_str)


class CashRegisterQrCode(BunqModel):
    """
    Once your CashRegister has been activated you can create a QR code for it.
    The visibility of a tab can be modified to be linked to this QR code. If a
    user of the bunq app scans this QR code, the linked tab will be shown on his
    device.
    
    :param _status: The status of this QR code. If the status is "ACTIVE" the QR
    code can be scanned to see the linked CashRegister and tab. If the status is
    "INACTIVE" the QR code does not link to a anything.
    :type _status: str
    :param _id_: The id of the created QR code. Use this id to get the RAW
    content of the QR code with: ../qr-code/{id}/content
    :type _id_: int
    :param _created: The timestamp of the QR code's creation.
    :type _created: str
    :param _updated: The timestamp of the TokenQrCashRegister's last update.
    :type _updated: str
    :param _cash_register: The CashRegister that is linked to the token.
    :type _cash_register: CashRegister
    :param _tab_object: Holds the Tab object. Can be TabUsageSingle,
    TabUsageMultiple or null
    :type _tab_object: Tab
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/cash-register/{}/qr-code"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/cash-register/{}/qr-code/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}/qr-code/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/qr-code"

    # Field constants.
    FIELD_STATUS = "status"

    # Object type.
    _OBJECT_TYPE_GET = "TokenQrCashRegister"

    _id_ = None
    _created = None
    _updated = None
    _status = None
    _cash_register = None
    _tab_object = None
    _status_field_for_request = None

    def __init__(self, status=None):
        """
        :param status: The status of the QR code. ACTIVE or INACTIVE. Only one QR
        code can be ACTIVE for a CashRegister at any time. Setting a QR code to
        ACTIVE will deactivate any other CashRegister QR codes.
        :type status: str
        """

        self._status_field_for_request = status

    @classmethod
    def create(cls, cash_register_id, status, monetary_account_id=None, custom_headers=None):
        """
        Create a new QR code for this CashRegister. You can only have one ACTIVE
        CashRegister QR code at the time.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :param status: The status of the QR code. ACTIVE or INACTIVE. Only one
        QR code can be ACTIVE for a CashRegister at any time. Setting a QR code
        to ACTIVE will deactivate any other CashRegister QR codes.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, cash_register_id, cash_register_qr_code_id, monetary_account_id=None, status=None,
               custom_headers=None):
        """
        Modify a QR code in a given CashRegister. You can only have one ACTIVE
        CashRegister QR code at the time.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type cash_register_qr_code_id: int
        :param status: The status of the QR code. ACTIVE or INACTIVE. Only one
        QR code can be ACTIVE for a CashRegister at any time. Setting a QR code
        to ACTIVE will deactivate any other CashRegister QR codes.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id, cash_register_qr_code_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, cash_register_id, cash_register_qr_code_id, monetary_account_id=None, custom_headers=None):
        """
        Get the information of a specific QR code. To get the RAW content of the
        QR code use ../qr-code/{id}/content
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type cash_register_qr_code_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCashRegisterQrCode
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id, cash_register_qr_code_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCashRegisterQrCode.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, cash_register_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of QR code information from a given CashRegister
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCashRegisterQrCodeList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCashRegisterQrCodeList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def cash_register(self):
        """
        :rtype: CashRegister
        """

        return self._cash_register

    @property
    def tab_object(self):
        """
        :rtype: Tab
        """

        return self._tab_object

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._status is not None:
            return False

        if self._cash_register is not None:
            return False

        if self._tab_object is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CashRegisterQrCode
        """

        return converter.json_to_class(CashRegisterQrCode, json_str)


class CashRegister(BunqModel):
    """
    CashRegisters are virtual points of sale. They have a specific name and
    avatar, and optionally, a location.<br/>With a CashRegister you can create a
    Tab and then use a QR code to receive payments.<br/>Check out our Quickstart
    example to learn how you can easily <a
    href="/api/1/page/usecase-tab-payment">create Tab
    payments</a>.<br/><br/>Notification filters can be set on a CashRegister to
    receive callbacks. For more information check the <a
    href="/api/1/page/callbacks">dedicated callbacks page</a>.
    
    :param _name: The name of the CashRegister.
    :type _name: str
    :param _status: The status of the CashRegister. Can be PENDING_APPROVAL,
    ACTIVE, DENIED or CLOSED.
    :type _status: str
    :param _avatar_uuid: The UUID of the avatar of the CashRegister. Use the
    calls /attachment-public and /avatar to create a new Avatar and get its
    UUID.
    :type _avatar_uuid: str
    :param _location: The geolocation of the CashRegister. Can be null.
    :type _location: object_.Geolocation
    :param _tab_text_waiting_screen: The tab text for waiting screen of
    CashRegister.
    :type _tab_text_waiting_screen: list[object_.TabTextWaitingScreen]
    :param _id_: The id of the created CashRegister.
    :type _id_: int
    :param _created: The timestamp of the CashRegister's creation.
    :type _created: str
    :param _updated: The timestamp of the CashRegister's last update.
    :type _updated: str
    :param _avatar: The Avatar of the CashRegister.
    :type _avatar: object_.Avatar
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/cash-register"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/cash-register/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register"

    # Field constants.
    FIELD_NAME = "name"
    FIELD_STATUS = "status"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_LOCATION = "location"
    FIELD_TAB_TEXT_WAITING_SCREEN = "tab_text_waiting_screen"

    # Object type.
    _OBJECT_TYPE_GET = "CashRegister"

    _id_ = None
    _created = None
    _updated = None
    _name = None
    _status = None
    _avatar = None
    _location = None
    _tab_text_waiting_screen = None
    _name_field_for_request = None
    _status_field_for_request = None
    _avatar_uuid_field_for_request = None
    _location_field_for_request = None
    _tab_text_waiting_screen_field_for_request = None

    def __init__(self, name=None, status=None, avatar_uuid=None, location=None, tab_text_waiting_screen=None):
        """
        :param name: The name of the CashRegister. Must be unique for this
        MonetaryAccount.
        :type name: str
        :param status: The status of the CashRegister. Can only be created or
        updated with PENDING_APPROVAL or CLOSED.
        :type status: str
        :param avatar_uuid: The UUID of the avatar of the CashRegister. Use the
        calls /attachment-public and /avatar to create a new Avatar and get its
        UUID.
        :type avatar_uuid: str
        :param location: The geolocation of the CashRegister.
        :type location: object_.Geolocation
        :param tab_text_waiting_screen: The tab text for waiting screen of
        CashRegister.
        :type tab_text_waiting_screen: list[object_.TabTextWaitingScreen]
        """

        self._name_field_for_request = name
        self._status_field_for_request = status
        self._avatar_uuid_field_for_request = avatar_uuid
        self._location_field_for_request = location
        self._tab_text_waiting_screen_field_for_request = tab_text_waiting_screen

    @classmethod
    def create(cls, name, status, avatar_uuid, monetary_account_id=None, location=None, tab_text_waiting_screen=None,
               custom_headers=None):
        """
        Create a new CashRegister. Only an UserCompany can create a
        CashRegisters. They need to be created with status PENDING_APPROVAL, an
        bunq admin has to approve your CashRegister before you can use it. In
        the sandbox testing environment an CashRegister will be automatically
        approved immediately after creation.
        
        :type user_id: int
        :type monetary_account_id: int
        :param name: The name of the CashRegister. Must be unique for this
        MonetaryAccount.
        :type name: str
        :param status: The status of the CashRegister. Can only be created or
        updated with PENDING_APPROVAL or CLOSED.
        :type status: str
        :param avatar_uuid: The UUID of the avatar of the CashRegister. Use the
        calls /attachment-public and /avatar to create a new Avatar and get its
        UUID.
        :type avatar_uuid: str
        :param location: The geolocation of the CashRegister.
        :type location: object_.Geolocation
        :param tab_text_waiting_screen: The tab text for waiting screen of
        CashRegister.
        :type tab_text_waiting_screen: list[object_.TabTextWaitingScreen]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NAME: name,
            cls.FIELD_STATUS: status,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_LOCATION: location,
            cls.FIELD_TAB_TEXT_WAITING_SCREEN: tab_text_waiting_screen
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, cash_register_id, monetary_account_id=None, custom_headers=None):
        """
        Get a specific CashRegister.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCashRegister
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCashRegister.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, cash_register_id, monetary_account_id=None, name=None, status=None, avatar_uuid=None, location=None,
               tab_text_waiting_screen=None, custom_headers=None):
        """
        Modify or close an existing CashRegister. You must set the status back
        to PENDING_APPROVAL if you modify the name, avatar or location of a
        CashRegister. To close a cash register put its status to CLOSED.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :param name: The name of the CashRegister. Must be unique for this
        MonetaryAccount.
        :type name: str
        :param status: The status of the CashRegister. Can only be created or
        updated with PENDING_APPROVAL or CLOSED.
        :type status: str
        :param avatar_uuid: The UUID of the avatar of the CashRegister. Use the
        calls /attachment-public and /avatar to create a new Avatar and get its
        UUID.
        :type avatar_uuid: str
        :param location: The geolocation of the CashRegister.
        :type location: object_.Geolocation
        :param tab_text_waiting_screen: The tab text for waiting screen of
        CashRegister.
        :type tab_text_waiting_screen: list[object_.TabTextWaitingScreen]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_NAME: name,
            cls.FIELD_STATUS: status,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_LOCATION: location,
            cls.FIELD_TAB_TEXT_WAITING_SCREEN: tab_text_waiting_screen
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of CashRegister for a given user and monetary account.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCashRegisterList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCashRegisterList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def name(self):
        """
        :rtype: str
        """

        return self._name

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def location(self):
        """
        :rtype: object_.Geolocation
        """

        return self._location

    @property
    def tab_text_waiting_screen(self):
        """
        :rtype: list[object_.TabTextWaitingScreen]
        """

        return self._tab_text_waiting_screen

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._name is not None:
            return False

        if self._status is not None:
            return False

        if self._avatar is not None:
            return False

        if self._location is not None:
            return False

        if self._tab_text_waiting_screen is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CashRegister
        """

        return converter.json_to_class(CashRegister, json_str)


class Tab(BunqModel, AnchorObjectInterface):
    """
    Once your CashRegister has been activated you can use it to create Tabs. A
    Tab is a template for a payment. In contrast to requests a Tab is not
    pointed towards a specific user. Any user can pay the Tab as long as it is
    made visible by you. The creation of a Tab happens with /tab-usage-single or
    /tab-usage-multiple. A TabUsageSingle is a Tab that can be paid once. A
    TabUsageMultiple is a Tab that can be paid multiple times by different
    users.
    
    :param _TabUsageSingle: 
    :type _TabUsageSingle: TabUsageSingle
    :param _TabUsageMultiple: 
    :type _TabUsageMultiple: TabUsageMultiple
    """

    # Error constants.
    _ERROR_NULL_FIELDS = "All fields of an extended model or object are null."

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}/tab/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/tab"

    # Object type.
    _OBJECT_TYPE_GET = "Tab"

    _TabUsageSingle = None
    _TabUsageMultiple = None

    @classmethod
    def get(cls, cash_register_id, tab_uuid, monetary_account_id=None, custom_headers=None):
        """
        Get a specific tab. This returns a TabUsageSingle or TabUsageMultiple.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTab
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id, tab_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTab.cast_from_bunq_response(
            cls._from_json(response_raw)
        )

    @classmethod
    def list(cls, cash_register_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of tabs.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTabList.cast_from_bunq_response(
            cls._from_json_list(response_raw)
        )

    @property
    def TabUsageSingle(self):
        """
        :rtype: TabUsageSingle
        """

        return self._TabUsageSingle

    @property
    def TabUsageMultiple(self):
        """
        :rtype: TabUsageMultiple
        """

        return self._TabUsageMultiple

    def get_referenced_object(self):
        """
        :rtype: BunqModel
        :raise: BunqException
        """

        if self._TabUsageSingle is not None:
            return self._TabUsageSingle

        if self._TabUsageMultiple is not None:
            return self._TabUsageMultiple

        raise BunqException(self._ERROR_NULL_FIELDS)

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._TabUsageSingle is not None:
            return False

        if self._TabUsageMultiple is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Tab
        """

        return converter.json_to_class(Tab, json_str)


class TabUsageSingle(BunqModel):
    """
    TabUsageSingle is a Tab that can be paid once. The TabUsageSingle is created
    with the status OPEN. Optionally you can add TabItems to the tab using
    /tab/_/tab-item, TabItems don't affect the total amount of the Tab. However,
    if you've created any TabItems for a Tab the sum of the amounts of these
    items must be equal to the total_amount of the Tab when you change its
    status to WAITING_FOR_PAYMENT. By setting the visibility object a
    TabUsageSingle with the status OPEN or WAITING_FOR_PAYMENT can be made
    visible to customers. As soon as a customer pays the TabUsageSingle its
    status changes to PAID, and it can't be paid again.
    
    :param _merchant_reference: The merchant reference of the Tab, as defined by
    the owner.
    :type _merchant_reference: str
    :param _description: The description of the TabUsageMultiple. Maximum 9000
    characters.
    :type _description: str
    :param _status: The status of the Tab. Can be OPEN, WAITING_FOR_PAYMENT,
    PAID or CANCELED.
    :type _status: str
    :param _amount_total: The total amount of the Tab.
    :type _amount_total: object_.Amount
    :param _allow_amount_higher: [DEPRECATED] Whether or not a higher amount can
    be paid.
    :type _allow_amount_higher: bool
    :param _allow_amount_lower: [DEPRECATED] Whether or not a lower amount can
    be paid.
    :type _allow_amount_lower: bool
    :param _want_tip: [DEPRECATED] Whether or not the user paying the Tab should
    be asked if he wants to give a tip. When want_tip is set to true,
    allow_amount_higher must also be set to true and allow_amount_lower must be
    false.
    :type _want_tip: bool
    :param _minimum_age: The minimum age of the user paying the Tab.
    :type _minimum_age: bool
    :param _require_address: Whether or not an billing and shipping address must
    be provided when paying the Tab.
    :type _require_address: str
    :param _redirect_url: The URL which the user is sent to after paying the
    Tab.
    :type _redirect_url: str
    :param _visibility: The visibility of a Tab. A Tab can be visible trough
    NearPay, the QR code of the CashRegister and its own QR code.
    :type _visibility: object_.TabVisibility
    :param _expiration: The moment when this Tab expires.
    :type _expiration: str
    :param _tab_attachment: An array of attachments that describe the tab.
    Uploaded through the POST /user/{userid}/attachment-tab endpoint.
    :type _tab_attachment: list[object_.BunqId]
    :param _uuid: The uuid of the created TabUsageSingle.
    :type _uuid: str
    :param _created: The timestamp of the Tab's creation.
    :type _created: str
    :param _updated: The timestamp of the Tab's last update.
    :type _updated: str
    :param _amount_paid: The amount that has been paid for this Tab.
    :type _amount_paid: object_.Amount
    :param _qr_code_token: The token used to redirect mobile devices directly to
    the bunq app. Because they can't scan a QR code.
    :type _qr_code_token: str
    :param _tab_url: The URL redirecting user to the tab payment in the bunq
    app. Only works on mobile devices.
    :type _tab_url: str
    :param _alias: The alias of the party that owns this tab.
    :type _alias: object_.MonetaryAccountReference
    :param _cash_register_location: The location of the cash register that
    created this tab.
    :type _cash_register_location: object_.Geolocation
    :param _tab_item: The tab items of this tab.
    :type _tab_item: list[TabItem]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-single"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-single/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-single/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-single/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-single"

    # Field constants.
    FIELD_MERCHANT_REFERENCE = "merchant_reference"
    FIELD_DESCRIPTION = "description"
    FIELD_STATUS = "status"
    FIELD_AMOUNT_TOTAL = "amount_total"
    FIELD_ALLOW_AMOUNT_HIGHER = "allow_amount_higher"
    FIELD_ALLOW_AMOUNT_LOWER = "allow_amount_lower"
    FIELD_WANT_TIP = "want_tip"
    FIELD_MINIMUM_AGE = "minimum_age"
    FIELD_REQUIRE_ADDRESS = "require_address"
    FIELD_REDIRECT_URL = "redirect_url"
    FIELD_VISIBILITY = "visibility"
    FIELD_EXPIRATION = "expiration"
    FIELD_TAB_ATTACHMENT = "tab_attachment"

    # Object type.
    _OBJECT_TYPE_POST = "Uuid"
    _OBJECT_TYPE_PUT = "Uuid"
    _OBJECT_TYPE_GET = "TabUsageSingle"

    _uuid = None
    _created = None
    _updated = None
    _merchant_reference = None
    _description = None
    _status = None
    _amount_total = None
    _amount_paid = None
    _qr_code_token = None
    _tab_url = None
    _visibility = None
    _minimum_age = None
    _require_address = None
    _redirect_url = None
    _expiration = None
    _alias = None
    _cash_register_location = None
    _tab_item = None
    _tab_attachment = None
    _merchant_reference_field_for_request = None
    _description_field_for_request = None
    _status_field_for_request = None
    _amount_total_field_for_request = None
    _allow_amount_higher_field_for_request = None
    _allow_amount_lower_field_for_request = None
    _want_tip_field_for_request = None
    _minimum_age_field_for_request = None
    _require_address_field_for_request = None
    _redirect_url_field_for_request = None
    _visibility_field_for_request = None
    _expiration_field_for_request = None
    _tab_attachment_field_for_request = None

    def __init__(self, description, status=None, amount_total=None, merchant_reference=None, allow_amount_higher=None,
                 allow_amount_lower=None, want_tip=None, minimum_age=None, require_address=None, redirect_url=None,
                 visibility=None, expiration=None, tab_attachment=None):
        """
        :param description: The description of the Tab. Maximum 9000 characters.
        Field is required but can be an empty string.
        :type description: str
        :param status: The status of the Tab. On creation the status must be set to
        OPEN. You can change the status from OPEN to WAITING_FOR_PAYMENT.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive amount.
        As long as the tab has the status OPEN you can change the total amount. This
        amount is not affected by the amounts of the TabItems. However, if you've
        created any TabItems for a Tab the sum of the amounts of these items must be
        equal to the total_amount of the Tab when you change its status to
        WAITING_FOR_PAYMENT.
        :type amount_total: object_.Amount
        :param merchant_reference: The reference of the Tab, as defined by the
        owner. This reference will be set for any payment that is generated by this
        tab. Must be unique among all the owner's tabs for the used monetary
        account.
        :type merchant_reference: str
        :param allow_amount_higher: [DEPRECATED] Whether or not a higher amount can
        be paid.
        :type allow_amount_higher: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not a lower amount can be
        paid.
        :type allow_amount_lower: bool
        :param want_tip: [DEPRECATED] Whether or not the user paying the Tab should
        be asked if he wants to give a tip. When want_tip is set to true,
        allow_amount_higher must also be set to true and allow_amount_lower must be
        false.
        :type want_tip: bool
        :param minimum_age: The minimum age of the user paying the Tab.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the Tab. Possible values are: BILLING, SHIPPING,
        BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param redirect_url: The URL which the user is sent to after paying the Tab.
        :type redirect_url: str
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 1 hour
        into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        """

        self._description_field_for_request = description
        self._status_field_for_request = status
        self._amount_total_field_for_request = amount_total
        self._merchant_reference_field_for_request = merchant_reference
        self._allow_amount_higher_field_for_request = allow_amount_higher
        self._allow_amount_lower_field_for_request = allow_amount_lower
        self._want_tip_field_for_request = want_tip
        self._minimum_age_field_for_request = minimum_age
        self._require_address_field_for_request = require_address
        self._redirect_url_field_for_request = redirect_url
        self._visibility_field_for_request = visibility
        self._expiration_field_for_request = expiration
        self._tab_attachment_field_for_request = tab_attachment

    @classmethod
    def create(cls, cash_register_id, description, status, amount_total, monetary_account_id=None,
               merchant_reference=None, allow_amount_higher=None, allow_amount_lower=None, want_tip=None,
               minimum_age=None, require_address=None, redirect_url=None, visibility=None, expiration=None,
               tab_attachment=None, custom_headers=None):
        """
        Create a TabUsageSingle. The initial status must be OPEN
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :param description: The description of the Tab. Maximum 9000 characters.
        Field is required but can be an empty string.
        :type description: str
        :param status: The status of the Tab. On creation the status must be set
        to OPEN. You can change the status from OPEN to WAITING_FOR_PAYMENT.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive
        amount. As long as the tab has the status OPEN you can change the total
        amount. This amount is not affected by the amounts of the TabItems.
        However, if you've created any TabItems for a Tab the sum of the amounts
        of these items must be equal to the total_amount of the Tab when you
        change its status to WAITING_FOR_PAYMENT.
        :type amount_total: object_.Amount
        :param merchant_reference: The reference of the Tab, as defined by the
        owner. This reference will be set for any payment that is generated by
        this tab. Must be unique among all the owner's tabs for the used
        monetary account.
        :type merchant_reference: str
        :param allow_amount_higher: [DEPRECATED] Whether or not a higher amount
        can be paid.
        :type allow_amount_higher: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not a lower amount
        can be paid.
        :type allow_amount_lower: bool
        :param want_tip: [DEPRECATED] Whether or not the user paying the Tab
        should be asked if he wants to give a tip. When want_tip is set to true,
        allow_amount_higher must also be set to true and allow_amount_lower must
        be false.
        :type want_tip: bool
        :param minimum_age: The minimum age of the user paying the Tab.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the Tab. Possible values are: BILLING, SHIPPING,
        BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param redirect_url: The URL which the user is sent to after paying the
        Tab.
        :type redirect_url: str
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 1
        hour into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_MERCHANT_REFERENCE: merchant_reference,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_STATUS: status,
            cls.FIELD_AMOUNT_TOTAL: amount_total,
            cls.FIELD_ALLOW_AMOUNT_HIGHER: allow_amount_higher,
            cls.FIELD_ALLOW_AMOUNT_LOWER: allow_amount_lower,
            cls.FIELD_WANT_TIP: want_tip,
            cls.FIELD_MINIMUM_AGE: minimum_age,
            cls.FIELD_REQUIRE_ADDRESS: require_address,
            cls.FIELD_REDIRECT_URL: redirect_url,
            cls.FIELD_VISIBILITY: visibility,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_TAB_ATTACHMENT: tab_attachment
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def update(cls, cash_register_id, tab_usage_single_uuid, monetary_account_id=None, status=None, amount_total=None,
               visibility=None, expiration=None, tab_attachment=None, custom_headers=None):
        """
        Modify a specific TabUsageSingle. You can change the amount_total,
        status and visibility. Once you change the status to WAITING_FOR_PAYMENT
        the TabUsageSingle will expire after 5 minutes (default) or up to 1 hour
        if a different expiration is provided.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_single_uuid: str
        :param status: The status of the Tab. On creation the status must be set
        to OPEN. You can change the status from OPEN to WAITING_FOR_PAYMENT.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive
        amount. As long as the tab has the status OPEN you can change the total
        amount. This amount is not affected by the amounts of the TabItems.
        However, if you've created any TabItems for a Tab the sum of the amounts
        of these items must be equal to the total_amount of the Tab when you
        change its status to WAITING_FOR_PAYMENT.
        :type amount_total: object_.Amount
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 1
        hour into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_AMOUNT_TOTAL: amount_total,
            cls.FIELD_VISIBILITY: visibility,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_TAB_ATTACHMENT: tab_attachment
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id, tab_usage_single_uuid)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def delete(cls, cash_register_id, tab_usage_single_uuid, monetary_account_id=None, custom_headers=None):
        """
        Cancel a specific TabUsageSingle.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_single_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id, tab_usage_single_uuid)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def get(cls, cash_register_id, tab_usage_single_uuid, monetary_account_id=None, custom_headers=None):
        """
        Get a specific TabUsageSingle.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_single_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabUsageSingle
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id, tab_usage_single_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTabUsageSingle.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, cash_register_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of TabUsageSingle.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabUsageSingleList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTabUsageSingleList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def uuid(self):
        """
        :rtype: str
        """

        return self._uuid

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def merchant_reference(self):
        """
        :rtype: str
        """

        return self._merchant_reference

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def amount_total(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_total

    @property
    def amount_paid(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_paid

    @property
    def qr_code_token(self):
        """
        :rtype: str
        """

        return self._qr_code_token

    @property
    def tab_url(self):
        """
        :rtype: str
        """

        return self._tab_url

    @property
    def visibility(self):
        """
        :rtype: object_.TabVisibility
        """

        return self._visibility

    @property
    def minimum_age(self):
        """
        :rtype: bool
        """

        return self._minimum_age

    @property
    def require_address(self):
        """
        :rtype: str
        """

        return self._require_address

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def expiration(self):
        """
        :rtype: str
        """

        return self._expiration

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def cash_register_location(self):
        """
        :rtype: object_.Geolocation
        """

        return self._cash_register_location

    @property
    def tab_item(self):
        """
        :rtype: list[TabItem]
        """

        return self._tab_item

    @property
    def tab_attachment(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._tab_attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._uuid is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._merchant_reference is not None:
            return False

        if self._description is not None:
            return False

        if self._status is not None:
            return False

        if self._amount_total is not None:
            return False

        if self._amount_paid is not None:
            return False

        if self._qr_code_token is not None:
            return False

        if self._tab_url is not None:
            return False

        if self._visibility is not None:
            return False

        if self._minimum_age is not None:
            return False

        if self._require_address is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._expiration is not None:
            return False

        if self._alias is not None:
            return False

        if self._cash_register_location is not None:
            return False

        if self._tab_item is not None:
            return False

        if self._tab_attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabUsageSingle
        """

        return converter.json_to_class(TabUsageSingle, json_str)


class TabItem(BunqModel):
    """
    Used to get items on a tab.
    
    :param _id_: The id of the tab item.
    :type _id_: int
    :param _description: The item's brief description.
    :type _description: str
    :param _ean_code: The item's EAN code.
    :type _ean_code: str
    :param _avatar_attachment: A struct with an AttachmentPublic UUID that used
    as an avatar for the TabItem.
    :type _avatar_attachment: object_.AttachmentPublic
    :param _tab_attachment: A list of AttachmentTab attached to the TabItem.
    :type _tab_attachment: list[object_.AttachmentTab]
    :param _quantity: The quantity of the item. Formatted as a number containing
    up to 15 digits, up to 15 decimals and using a dot.
    :type _quantity: str
    :param _amount: The money amount of the item.
    :type _amount: object_.Amount
    """

    _id_ = None
    _description = None
    _ean_code = None
    _avatar_attachment = None
    _tab_attachment = None
    _quantity = None
    _amount = None

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def ean_code(self):
        """
        :rtype: str
        """

        return self._ean_code

    @property
    def avatar_attachment(self):
        """
        :rtype: object_.AttachmentPublic
        """

        return self._avatar_attachment

    @property
    def tab_attachment(self):
        """
        :rtype: list[object_.AttachmentTab]
        """

        return self._tab_attachment

    @property
    def quantity(self):
        """
        :rtype: str
        """

        return self._quantity

    @property
    def amount(self):
        """
        :rtype: object_.Amount
        """

        return self._amount

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._description is not None:
            return False

        if self._ean_code is not None:
            return False

        if self._avatar_attachment is not None:
            return False

        if self._tab_attachment is not None:
            return False

        if self._quantity is not None:
            return False

        if self._amount is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabItem
        """

        return converter.json_to_class(TabItem, json_str)


class TabUsageMultiple(BunqModel):
    """
    TabUsageMultiple is a Tab that can be paid by multiple users. Just like the
    TabUsageSingle it is created with the status OPEN, the visibility can be
    defined in the visibility object and TabItems can be added as long as the
    status is OPEN. When you change the status to PAYABLE any bunq user can use
    the tab to make a payment to your account. After an user has paid your
    TabUsageMultiple the status will not change, it will stay PAYABLE. For
    example: you can create a TabUsageMultiple with require_address set to true.
    Now show the QR code of this Tab on your webshop, and any bunq user can
    instantly pay and order something from your webshop.
    
    :param _description: The description of the TabUsageMultiple. Maximum 9000
    characters.
    :type _description: str
    :param _status: The status of the Tab. Can be OPEN, PAYABLE or CLOSED.
    :type _status: str
    :param _amount_total: The total amount of the Tab.
    :type _amount_total: object_.Amount
    :param _allow_amount_higher: [DEPRECATED] Whether or not a higher amount can
    be paid.
    :type _allow_amount_higher: bool
    :param _allow_amount_lower: [DEPRECATED] Whether or not a lower amount can
    be paid.
    :type _allow_amount_lower: bool
    :param _want_tip: [DEPRECATED] Whether or not the user paying the Tab should
    be asked if he wants to give a tip. When want_tip is set to true,
    allow_amount_higher must also be set to true and allow_amount_lower must be
    false.
    :type _want_tip: bool
    :param _minimum_age: The minimum age of the user paying the Tab.
    :type _minimum_age: bool
    :param _require_address: Whether or not an billing and shipping address must
    be provided when paying the Tab.
    :type _require_address: str
    :param _redirect_url: The URL which the user is sent to after paying the
    Tab.
    :type _redirect_url: str
    :param _visibility: The visibility of a Tab. A Tab can be visible trough
    NearPay, the QR code of the CashRegister and its own QR code.
    :type _visibility: object_.TabVisibility
    :param _expiration: The moment when this Tab expires.
    :type _expiration: str
    :param _tab_attachment: An array of attachments that describe the tab.
    Viewable through the GET /tab/{tabid}/attachment/{attachmentid}/content
    endpoint.
    :type _tab_attachment: list[object_.BunqId]
    :param _uuid: The uuid of the created TabUsageMultiple.
    :type _uuid: str
    :param _created: The timestamp of the Tab's creation.
    :type _created: str
    :param _updated: The timestamp of the Tab's last update.
    :type _updated: str
    :param _qr_code_token: The token used to redirect mobile devices directly to
    the bunq app. Because they can't scan a QR code.
    :type _qr_code_token: str
    :param _tab_url: The URL redirecting user to the tab payment in the bunq
    app. Only works on mobile devices.
    :type _tab_url: str
    :param _alias: The alias of the party that owns this tab.
    :type _alias: object_.MonetaryAccountReference
    :param _cash_register_location: The location of the cash register that
    created this tab.
    :type _cash_register_location: object_.Geolocation
    :param _tab_item: The tab items of this tab.
    :type _tab_item: list[TabItem]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-multiple"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-multiple/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-multiple/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-multiple/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/tab-usage-multiple"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_STATUS = "status"
    FIELD_AMOUNT_TOTAL = "amount_total"
    FIELD_ALLOW_AMOUNT_HIGHER = "allow_amount_higher"
    FIELD_ALLOW_AMOUNT_LOWER = "allow_amount_lower"
    FIELD_WANT_TIP = "want_tip"
    FIELD_MINIMUM_AGE = "minimum_age"
    FIELD_REQUIRE_ADDRESS = "require_address"
    FIELD_REDIRECT_URL = "redirect_url"
    FIELD_VISIBILITY = "visibility"
    FIELD_EXPIRATION = "expiration"
    FIELD_TAB_ATTACHMENT = "tab_attachment"

    # Object type.
    _OBJECT_TYPE_POST = "Uuid"
    _OBJECT_TYPE_PUT = "Uuid"
    _OBJECT_TYPE_GET = "TabUsageMultiple"

    _uuid = None
    _created = None
    _updated = None
    _description = None
    _status = None
    _amount_total = None
    _qr_code_token = None
    _tab_url = None
    _visibility = None
    _minimum_age = None
    _require_address = None
    _redirect_url = None
    _expiration = None
    _alias = None
    _cash_register_location = None
    _tab_item = None
    _tab_attachment = None
    _description_field_for_request = None
    _status_field_for_request = None
    _amount_total_field_for_request = None
    _allow_amount_higher_field_for_request = None
    _allow_amount_lower_field_for_request = None
    _want_tip_field_for_request = None
    _minimum_age_field_for_request = None
    _require_address_field_for_request = None
    _redirect_url_field_for_request = None
    _visibility_field_for_request = None
    _expiration_field_for_request = None
    _tab_attachment_field_for_request = None

    def __init__(self, description, status=None, amount_total=None, allow_amount_higher=None, allow_amount_lower=None,
                 want_tip=None, minimum_age=None, require_address=None, redirect_url=None, visibility=None,
                 expiration=None, tab_attachment=None):
        """
        :param description: The description of the TabUsageMultiple. Maximum 9000
        characters. Field is required but can be an empty string.
        :type description: str
        :param status: The status of the TabUsageMultiple. On creation the status
        must be set to OPEN. You can change the status from OPEN to PAYABLE. If the
        TabUsageMultiple gets paid the status will remain PAYABLE.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive amount.
        As long as the tab has the status OPEN you can change the total amount. This
        amount is not affected by the amounts of the TabItems. However, if you've
        created any TabItems for a Tab the sum of the amounts of these items must be
        equal to the total_amount of the Tab when you change its status to PAYABLE
        :type amount_total: object_.Amount
        :param allow_amount_higher: [DEPRECATED] Whether or not a higher amount can
        be paid.
        :type allow_amount_higher: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not a lower amount can be
        paid.
        :type allow_amount_lower: bool
        :param want_tip: [DEPRECATED] Whether or not the user paying the Tab should
        be asked if he wants to give a tip. When want_tip is set to true,
        allow_amount_higher must also be set to true and allow_amount_lower must be
        false.
        :type want_tip: bool
        :param minimum_age: The minimum age of the user paying the Tab.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the Tab. Possible values are: BILLING, SHIPPING,
        BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param redirect_url: The URL which the user is sent to after paying the Tab.
        :type redirect_url: str
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 365 days
        into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        """

        self._description_field_for_request = description
        self._status_field_for_request = status
        self._amount_total_field_for_request = amount_total
        self._allow_amount_higher_field_for_request = allow_amount_higher
        self._allow_amount_lower_field_for_request = allow_amount_lower
        self._want_tip_field_for_request = want_tip
        self._minimum_age_field_for_request = minimum_age
        self._require_address_field_for_request = require_address
        self._redirect_url_field_for_request = redirect_url
        self._visibility_field_for_request = visibility
        self._expiration_field_for_request = expiration
        self._tab_attachment_field_for_request = tab_attachment

    @classmethod
    def create(cls, cash_register_id, description, status, amount_total, monetary_account_id=None,
               allow_amount_higher=None, allow_amount_lower=None, want_tip=None, minimum_age=None, require_address=None,
               redirect_url=None, visibility=None, expiration=None, tab_attachment=None, custom_headers=None):
        """
        Create a TabUsageMultiple. On creation the status must be set to OPEN
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :param description: The description of the TabUsageMultiple. Maximum
        9000 characters. Field is required but can be an empty string.
        :type description: str
        :param status: The status of the TabUsageMultiple. On creation the
        status must be set to OPEN. You can change the status from OPEN to
        PAYABLE. If the TabUsageMultiple gets paid the status will remain
        PAYABLE.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive
        amount. As long as the tab has the status OPEN you can change the total
        amount. This amount is not affected by the amounts of the TabItems.
        However, if you've created any TabItems for a Tab the sum of the amounts
        of these items must be equal to the total_amount of the Tab when you
        change its status to PAYABLE
        :type amount_total: object_.Amount
        :param allow_amount_higher: [DEPRECATED] Whether or not a higher amount
        can be paid.
        :type allow_amount_higher: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not a lower amount
        can be paid.
        :type allow_amount_lower: bool
        :param want_tip: [DEPRECATED] Whether or not the user paying the Tab
        should be asked if he wants to give a tip. When want_tip is set to true,
        allow_amount_higher must also be set to true and allow_amount_lower must
        be false.
        :type want_tip: bool
        :param minimum_age: The minimum age of the user paying the Tab.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the Tab. Possible values are: BILLING, SHIPPING,
        BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param redirect_url: The URL which the user is sent to after paying the
        Tab.
        :type redirect_url: str
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 365
        days into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_STATUS: status,
            cls.FIELD_AMOUNT_TOTAL: amount_total,
            cls.FIELD_ALLOW_AMOUNT_HIGHER: allow_amount_higher,
            cls.FIELD_ALLOW_AMOUNT_LOWER: allow_amount_lower,
            cls.FIELD_WANT_TIP: want_tip,
            cls.FIELD_MINIMUM_AGE: minimum_age,
            cls.FIELD_REQUIRE_ADDRESS: require_address,
            cls.FIELD_REDIRECT_URL: redirect_url,
            cls.FIELD_VISIBILITY: visibility,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_TAB_ATTACHMENT: tab_attachment
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def update(cls, cash_register_id, tab_usage_multiple_uuid, monetary_account_id=None, status=None, amount_total=None,
               visibility=None, expiration=None, tab_attachment=None, custom_headers=None):
        """
        Modify a specific TabUsageMultiple. You can change the amount_total,
        status and visibility. Once you change the status to PAYABLE the
        TabUsageMultiple will expire after a year (default). If you've created
        any TabItems for a Tab the sum of the amounts of these items must be
        equal to the total_amount of the Tab when you change its status to
        PAYABLE.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_multiple_uuid: str
        :param status: The status of the TabUsageMultiple. On creation the
        status must be set to OPEN. You can change the status from OPEN to
        PAYABLE. If the TabUsageMultiple gets paid the status will remain
        PAYABLE.
        :type status: str
        :param amount_total: The total amount of the Tab. Must be a positive
        amount. As long as the tab has the status OPEN you can change the total
        amount. This amount is not affected by the amounts of the TabItems.
        However, if you've created any TabItems for a Tab the sum of the amounts
        of these items must be equal to the total_amount of the Tab when you
        change its status to PAYABLE
        :type amount_total: object_.Amount
        :param visibility: The visibility of a Tab. A Tab can be visible trough
        NearPay, the QR code of the CashRegister and its own QR code.
        :type visibility: object_.TabVisibility
        :param expiration: The moment when this Tab expires. Can be at most 365
        days into the future.
        :type expiration: str
        :param tab_attachment: An array of attachments that describe the tab.
        Uploaded through the POST /user/{userid}/attachment-tab endpoint.
        :type tab_attachment: list[object_.BunqId]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseStr
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_AMOUNT_TOTAL: amount_total,
            cls.FIELD_VISIBILITY: visibility,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_TAB_ATTACHMENT: tab_attachment
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id, tab_usage_multiple_uuid)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseStr.cast_from_bunq_response(
            cls._process_for_uuid(response_raw)
        )

    @classmethod
    def delete(cls, cash_register_id, tab_usage_multiple_uuid, monetary_account_id=None, custom_headers=None):
        """
        Close a specific TabUsageMultiple.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_multiple_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       cash_register_id, tab_usage_multiple_uuid)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def get(cls, cash_register_id, tab_usage_multiple_uuid, monetary_account_id=None, custom_headers=None):
        """
        Get a specific TabUsageMultiple.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_usage_multiple_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabUsageMultiple
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id, tab_usage_multiple_uuid)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTabUsageMultiple.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, cash_register_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of TabUsageMultiple.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabUsageMultipleList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTabUsageMultipleList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def uuid(self):
        """
        :rtype: str
        """

        return self._uuid

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def amount_total(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_total

    @property
    def qr_code_token(self):
        """
        :rtype: str
        """

        return self._qr_code_token

    @property
    def tab_url(self):
        """
        :rtype: str
        """

        return self._tab_url

    @property
    def visibility(self):
        """
        :rtype: object_.TabVisibility
        """

        return self._visibility

    @property
    def minimum_age(self):
        """
        :rtype: bool
        """

        return self._minimum_age

    @property
    def require_address(self):
        """
        :rtype: str
        """

        return self._require_address

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def expiration(self):
        """
        :rtype: str
        """

        return self._expiration

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def cash_register_location(self):
        """
        :rtype: object_.Geolocation
        """

        return self._cash_register_location

    @property
    def tab_item(self):
        """
        :rtype: list[TabItem]
        """

        return self._tab_item

    @property
    def tab_attachment(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._tab_attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._uuid is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._description is not None:
            return False

        if self._status is not None:
            return False

        if self._amount_total is not None:
            return False

        if self._qr_code_token is not None:
            return False

        if self._tab_url is not None:
            return False

        if self._visibility is not None:
            return False

        if self._minimum_age is not None:
            return False

        if self._require_address is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._expiration is not None:
            return False

        if self._alias is not None:
            return False

        if self._cash_register_location is not None:
            return False

        if self._tab_item is not None:
            return False

        if self._tab_attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabUsageMultiple
        """

        return converter.json_to_class(TabUsageMultiple, json_str)


class CertificatePinned(BunqModel):
    """
    This endpoint allow you to pin the certificate chains to your account. These
    certificate chains are used for SSL validation whenever a callback is
    initiated to one of your https callback urls.
    
    :param _certificate_chain: The certificate chain in .PEM format.
    Certificates are glued with newline characters.
    :type _certificate_chain: str
    :param _id_: The id generated for the pinned certificate chain.
    :type _id_: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/certificate-pinned"
    _ENDPOINT_URL_DELETE = "user/{}/certificate-pinned/{}"
    _ENDPOINT_URL_LISTING = "user/{}/certificate-pinned"
    _ENDPOINT_URL_READ = "user/{}/certificate-pinned/{}"

    # Field constants.
    FIELD_CERTIFICATE_CHAIN = "certificate_chain"

    # Object type.
    _OBJECT_TYPE_GET = "CertificatePinned"

    _certificate_chain = None
    _id_ = None
    _certificate_chain_field_for_request = None

    def __init__(self, certificate_chain):
        """
        :param certificate_chain: The certificate chain in .PEM format.
        :type certificate_chain: list[object_.Certificate]
        """

        self._certificate_chain_field_for_request = certificate_chain

    @classmethod
    def create(cls, certificate_chain, custom_headers=None):
        """
        Pin the certificate chain.
        
        :type user_id: int
        :param certificate_chain: The certificate chain in .PEM format.
        :type certificate_chain: list[object_.Certificate]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CERTIFICATE_CHAIN: certificate_chain
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, certificate_pinned_id, custom_headers=None):
        """
        Remove the pinned certificate chain with the specific ID.
        
        :type user_id: int
        :type certificate_pinned_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(), certificate_pinned_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        List all the pinned certificate chain for the given user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCertificatePinnedList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCertificatePinnedList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, certificate_pinned_id, custom_headers=None):
        """
        Get the pinned certificate chain with the specified ID.
        
        :type api_context: ApiContext
        :type user_id: int
        :type certificate_pinned_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCertificatePinned
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), certificate_pinned_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCertificatePinned.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def certificate_chain(self):
        """
        :rtype: str
        """

        return self._certificate_chain

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._certificate_chain is not None:
            return False

        if self._id_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: CertificatePinned
        """

        return converter.json_to_class(CertificatePinned, json_str)


class Company(BunqModel):
    """
    Create and manage companies.
    
    :param _name: The company name.
    :type _name: str
    :param _address_main: The company's main address.
    :type _address_main: object_.Address
    :param _address_postal: The company's postal address.
    :type _address_postal: object_.Address
    :param _country: The country where the company is registered.
    :type _country: str
    :param _ubo: The names and birth dates of the company's ultimate beneficiary
    owners. Minimum zero, maximum four.
    :type _ubo: list[object_.Ubo]
    :param _chamber_of_commerce_number: The company's chamber of commerce
    number.
    :type _chamber_of_commerce_number: str
    :param _legal_form: The company's legal form.
    :type _legal_form: str
    :param _avatar_uuid: The public UUID of the company's avatar.
    :type _avatar_uuid: str
    :param _UserCompany: 
    :type _UserCompany: UserCompany
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/company"
    _ENDPOINT_URL_READ = "user/{}/company/{}"
    _ENDPOINT_URL_LISTING = "user/{}/company"
    _ENDPOINT_URL_UPDATE = "user/{}/company/{}"

    # Field constants.
    FIELD_NAME = "name"
    FIELD_ADDRESS_MAIN = "address_main"
    FIELD_ADDRESS_POSTAL = "address_postal"
    FIELD_COUNTRY = "country"
    FIELD_UBO = "ubo"
    FIELD_CHAMBER_OF_COMMERCE_NUMBER = "chamber_of_commerce_number"
    FIELD_LEGAL_FORM = "legal_form"
    FIELD_AVATAR_UUID = "avatar_uuid"

    # Object type.
    _OBJECT_TYPE_GET = "UserCompany"

    _UserCompany = None
    _name_field_for_request = None
    _address_main_field_for_request = None
    _address_postal_field_for_request = None
    _country_field_for_request = None
    _ubo_field_for_request = None
    _chamber_of_commerce_number_field_for_request = None
    _legal_form_field_for_request = None
    _avatar_uuid_field_for_request = None

    def __init__(self, name, address_main, address_postal, country, legal_form, ubo=None,
                 chamber_of_commerce_number=None, avatar_uuid=None):
        """
        :param name: The company name.
        :type name: str
        :param address_main: The company's main address.
        :type address_main: object_.Address
        :param address_postal: The company's postal address.
        :type address_postal: object_.Address
        :param country: The country where the company is registered.
        :type country: str
        :param legal_form: The company's legal form.
        :type legal_form: str
        :param ubo: The names and birth dates of the company's ultimate beneficiary
        owners. Minimum zero, maximum four.
        :type ubo: list[object_.Ubo]
        :param chamber_of_commerce_number: The company's chamber of commerce number.
        :type chamber_of_commerce_number: str
        :param avatar_uuid: The public UUID of the company's avatar.
        :type avatar_uuid: str
        """

        self._name_field_for_request = name
        self._address_main_field_for_request = address_main
        self._address_postal_field_for_request = address_postal
        self._country_field_for_request = country
        self._legal_form_field_for_request = legal_form
        self._ubo_field_for_request = ubo
        self._chamber_of_commerce_number_field_for_request = chamber_of_commerce_number
        self._avatar_uuid_field_for_request = avatar_uuid

    @classmethod
    def create(cls, name, address_main, address_postal, country, legal_form, ubo=None, chamber_of_commerce_number=None,
               avatar_uuid=None, custom_headers=None):
        """
        :type user_id: int
        :param name: The company name.
        :type name: str
        :param address_main: The company's main address.
        :type address_main: object_.Address
        :param address_postal: The company's postal address.
        :type address_postal: object_.Address
        :param country: The country where the company is registered.
        :type country: str
        :param legal_form: The company's legal form.
        :type legal_form: str
        :param ubo: The names and birth dates of the company's ultimate
        beneficiary owners. Minimum zero, maximum four.
        :type ubo: list[object_.Ubo]
        :param chamber_of_commerce_number: The company's chamber of commerce
        number.
        :type chamber_of_commerce_number: str
        :param avatar_uuid: The public UUID of the company's avatar.
        :type avatar_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NAME: name,
            cls.FIELD_ADDRESS_MAIN: address_main,
            cls.FIELD_ADDRESS_POSTAL: address_postal,
            cls.FIELD_COUNTRY: country,
            cls.FIELD_UBO: ubo,
            cls.FIELD_CHAMBER_OF_COMMERCE_NUMBER: chamber_of_commerce_number,
            cls.FIELD_LEGAL_FORM: legal_form,
            cls.FIELD_AVATAR_UUID: avatar_uuid
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, company_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type company_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCompany
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), company_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseCompany.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseCompanyList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseCompanyList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, company_id, avatar_uuid=None, custom_headers=None):
        """
        :type user_id: int
        :type company_id: int
        :param avatar_uuid: The public UUID of the company's avatar.
        :type avatar_uuid: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_AVATAR_UUID: avatar_uuid
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), company_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @property
    def UserCompany(self):
        """
        :rtype: UserCompany
        """

        return self._UserCompany

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._UserCompany is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Company
        """

        return converter.json_to_class(Company, json_str)


class UserCompany(BunqModel):
    """
    With UserCompany you can retrieve information regarding the authenticated
    UserCompany and update specific fields.<br/><br/>Notification filters can be
    set on a UserCompany level to receive callbacks. For more information check
    the <a href="/api/1/page/callbacks">dedicated callbacks page</a>.
    
    :param _name: The company name.
    :type _name: str
    :param _public_nick_name: The company's public nick name.
    :type _public_nick_name: str
    :param _avatar_uuid: The public UUID of the company's avatar.
    :type _avatar_uuid: str
    :param _address_main: The company's main address.
    :type _address_main: object_.Address
    :param _address_postal: The company's postal address.
    :type _address_postal: object_.Address
    :param _language: The person's preferred language. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _language: str
    :param _region: The person's preferred region. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _region: str
    :param _country: The country as an ISO 3166-1 alpha-2 country code.
    :type _country: str
    :param _ubo: The names of the company's ultimate beneficiary owners. Minimum
    zero, maximum four.
    :type _ubo: list[object_.Ubo]
    :param _chamber_of_commerce_number: The company's chamber of commerce
    number.
    :type _chamber_of_commerce_number: str
    :param _legal_form: The company's legal form.
    :type _legal_form: str
    :param _status: The user status. Can be: ACTIVE, SIGNUP, RECOVERY.
    :type _status: str
    :param _sub_status: The user sub-status. Can be: NONE, FACE_RESET, APPROVAL,
    APPROVAL_DIRECTOR, APPROVAL_PARENT, APPROVAL_SUPPORT, COUNTER_IBAN, IDEAL or
    SUBMIT.
    :type _sub_status: str
    :param _session_timeout: The setting for the session timeout of the company
    in seconds.
    :type _session_timeout: int
    :param _daily_limit_without_confirmation_login: The amount the company can
    pay in the session without asking for credentials.
    :type _daily_limit_without_confirmation_login: object_.Amount
    :param _id_: The id of the modified company.
    :type _id_: int
    :param _created: The timestamp of the company object's creation.
    :type _created: str
    :param _updated: The timestamp of the company object's last update.
    :type _updated: str
    :param _public_uuid: The company's public UUID.
    :type _public_uuid: str
    :param _display_name: The company's display name.
    :type _display_name: str
    :param _alias: The aliases of the account.
    :type _alias: list[object_.Pointer]
    :param _type_of_business_entity: The type of business entity.
    :type _type_of_business_entity: str
    :param _sector_of_industry: The sector of industry.
    :type _sector_of_industry: str
    :param _counter_bank_iban: The company's other bank account IBAN, through
    which we verify it.
    :type _counter_bank_iban: str
    :param _avatar: The company's avatar.
    :type _avatar: object_.Avatar
    :param _version_terms_of_service: The version of the terms of service
    accepted by the user.
    :type _version_terms_of_service: str
    :param _director_alias: The existing bunq alias for the company's primary
    director.
    :type _director_alias: object_.LabelUser
    :param _directors: The existing bunq aliases for the company's directors.
    :type _directors: list[object_.LabelUser]
    :param _notification_filters: The types of notifications that will result in
    a push notification or URL callback for this UserCompany.
    :type _notification_filters: list[object_.NotificationFilter]
    :param _customer: The customer profile of the company.
    :type _customer: Customer
    :param _customer_limit: The customer limits of the company.
    :type _customer_limit: CustomerLimit
    :param _billing_contract: The subscription of the company.
    :type _billing_contract: list[BillingContractSubscription]
    :param _deny_reason: The user deny reason.
    :type _deny_reason: str
    :param _relations: The relations for this user.
    :type _relations: list[RelationUser]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user-company/{}"
    _ENDPOINT_URL_UPDATE = "user-company/{}"

    # Field constants.
    FIELD_NAME = "name"
    FIELD_PUBLIC_NICK_NAME = "public_nick_name"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_ADDRESS_MAIN = "address_main"
    FIELD_ADDRESS_POSTAL = "address_postal"
    FIELD_LANGUAGE = "language"
    FIELD_REGION = "region"
    FIELD_COUNTRY = "country"
    FIELD_UBO = "ubo"
    FIELD_CHAMBER_OF_COMMERCE_NUMBER = "chamber_of_commerce_number"
    FIELD_LEGAL_FORM = "legal_form"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_SESSION_TIMEOUT = "session_timeout"
    FIELD_DAILY_LIMIT_WITHOUT_CONFIRMATION_LOGIN = "daily_limit_without_confirmation_login"

    # Object type.
    _OBJECT_TYPE_GET = "UserCompany"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _name = None
    _display_name = None
    _public_nick_name = None
    _alias = None
    _chamber_of_commerce_number = None
    _legal_form = None
    _type_of_business_entity = None
    _sector_of_industry = None
    _counter_bank_iban = None
    _avatar = None
    _address_main = None
    _address_postal = None
    _version_terms_of_service = None
    _director_alias = None
    _directors = None
    _language = None
    _country = None
    _region = None
    _ubo = None
    _status = None
    _sub_status = None
    _session_timeout = None
    _daily_limit_without_confirmation_login = None
    _notification_filters = None
    _customer = None
    _customer_limit = None
    _billing_contract = None
    _deny_reason = None
    _relations = None
    _name_field_for_request = None
    _public_nick_name_field_for_request = None
    _avatar_uuid_field_for_request = None
    _address_main_field_for_request = None
    _address_postal_field_for_request = None
    _language_field_for_request = None
    _region_field_for_request = None
    _country_field_for_request = None
    _ubo_field_for_request = None
    _chamber_of_commerce_number_field_for_request = None
    _legal_form_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _session_timeout_field_for_request = None
    _daily_limit_without_confirmation_login_field_for_request = None

    def __init__(self, address_main=None, language=None, region=None, name=None, public_nick_name=None,
                 avatar_uuid=None, address_postal=None, country=None, ubo=None, chamber_of_commerce_number=None,
                 legal_form=None, status=None, sub_status=None, session_timeout=None,
                 daily_limit_without_confirmation_login=None):
        """
        :param address_main: The user's main address.
        :type address_main: object_.Address
        :param language: The person's preferred language. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type language: str
        :param region: The person's preferred region. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type region: str
        :param name: The company name.
        :type name: str
        :param public_nick_name: The company's nick name.
        :type public_nick_name: str
        :param avatar_uuid: The public UUID of the company's avatar.
        :type avatar_uuid: str
        :param address_postal: The company's postal address.
        :type address_postal: object_.Address
        :param country: The country where the company is registered.
        :type country: str
        :param ubo: The names and birth dates of the company's ultimate beneficiary
        owners. Minimum zero, maximum four.
        :type ubo: list[object_.Ubo]
        :param chamber_of_commerce_number: The company's chamber of commerce number.
        :type chamber_of_commerce_number: str
        :param legal_form: The company's legal form.
        :type legal_form: str
        :param status: The user status. Can be: ACTIVE, SIGNUP, RECOVERY.
        :type status: str
        :param sub_status: The user sub-status. Can be: NONE, FACE_RESET, APPROVAL,
        APPROVAL_DIRECTOR, APPROVAL_PARENT, APPROVAL_SUPPORT, COUNTER_IBAN, IDEAL or
        SUBMIT.
        :type sub_status: str
        :param session_timeout: The setting for the session timeout of the company
        in seconds.
        :type session_timeout: int
        :param daily_limit_without_confirmation_login: The amount the company can
        pay in the session without asking for credentials.
        :type daily_limit_without_confirmation_login: object_.Amount
        """

        self._address_main_field_for_request = address_main
        self._language_field_for_request = language
        self._region_field_for_request = region
        self._name_field_for_request = name
        self._public_nick_name_field_for_request = public_nick_name
        self._avatar_uuid_field_for_request = avatar_uuid
        self._address_postal_field_for_request = address_postal
        self._country_field_for_request = country
        self._ubo_field_for_request = ubo
        self._chamber_of_commerce_number_field_for_request = chamber_of_commerce_number
        self._legal_form_field_for_request = legal_form
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._session_timeout_field_for_request = session_timeout
        self._daily_limit_without_confirmation_login_field_for_request = daily_limit_without_confirmation_login

    @classmethod
    def get(cls, custom_headers=None):
        """
        Get a specific company.
        
        :type api_context: ApiContext
        :type user_company_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseUserCompany
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseUserCompany.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, name=None, public_nick_name=None, avatar_uuid=None, address_main=None, address_postal=None,
               language=None, region=None, country=None, ubo=None, chamber_of_commerce_number=None, legal_form=None,
               status=None, sub_status=None, session_timeout=None, daily_limit_without_confirmation_login=None,
               custom_headers=None):
        """
        Modify a specific company's data.
        
        :type user_company_id: int
        :param name: The company name.
        :type name: str
        :param public_nick_name: The company's nick name.
        :type public_nick_name: str
        :param avatar_uuid: The public UUID of the company's avatar.
        :type avatar_uuid: str
        :param address_main: The user's main address.
        :type address_main: object_.Address
        :param address_postal: The company's postal address.
        :type address_postal: object_.Address
        :param language: The person's preferred language. Formatted as a ISO
        639-1 language code plus a ISO 3166-1 alpha-2 country code, seperated by
        an underscore.
        :type language: str
        :param region: The person's preferred region. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type region: str
        :param country: The country where the company is registered.
        :type country: str
        :param ubo: The names and birth dates of the company's ultimate
        beneficiary owners. Minimum zero, maximum four.
        :type ubo: list[object_.Ubo]
        :param chamber_of_commerce_number: The company's chamber of commerce
        number.
        :type chamber_of_commerce_number: str
        :param legal_form: The company's legal form.
        :type legal_form: str
        :param status: The user status. Can be: ACTIVE, SIGNUP, RECOVERY.
        :type status: str
        :param sub_status: The user sub-status. Can be: NONE, FACE_RESET,
        APPROVAL, APPROVAL_DIRECTOR, APPROVAL_PARENT, APPROVAL_SUPPORT,
        COUNTER_IBAN, IDEAL or SUBMIT.
        :type sub_status: str
        :param session_timeout: The setting for the session timeout of the
        company in seconds.
        :type session_timeout: int
        :param daily_limit_without_confirmation_login: The amount the company
        can pay in the session without asking for credentials.
        :type daily_limit_without_confirmation_login: object_.Amount
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_NAME: name,
            cls.FIELD_PUBLIC_NICK_NAME: public_nick_name,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_ADDRESS_MAIN: address_main,
            cls.FIELD_ADDRESS_POSTAL: address_postal,
            cls.FIELD_LANGUAGE: language,
            cls.FIELD_REGION: region,
            cls.FIELD_COUNTRY: country,
            cls.FIELD_UBO: ubo,
            cls.FIELD_CHAMBER_OF_COMMERCE_NUMBER: chamber_of_commerce_number,
            cls.FIELD_LEGAL_FORM: legal_form,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_SESSION_TIMEOUT: session_timeout,
            cls.FIELD_DAILY_LIMIT_WITHOUT_CONFIRMATION_LOGIN: daily_limit_without_confirmation_login
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id())
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def name(self):
        """
        :rtype: str
        """

        return self._name

    @property
    def display_name(self):
        """
        :rtype: str
        """

        return self._display_name

    @property
    def public_nick_name(self):
        """
        :rtype: str
        """

        return self._public_nick_name

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def chamber_of_commerce_number(self):
        """
        :rtype: str
        """

        return self._chamber_of_commerce_number

    @property
    def legal_form(self):
        """
        :rtype: str
        """

        return self._legal_form

    @property
    def type_of_business_entity(self):
        """
        :rtype: str
        """

        return self._type_of_business_entity

    @property
    def sector_of_industry(self):
        """
        :rtype: str
        """

        return self._sector_of_industry

    @property
    def counter_bank_iban(self):
        """
        :rtype: str
        """

        return self._counter_bank_iban

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def address_main(self):
        """
        :rtype: object_.Address
        """

        return self._address_main

    @property
    def address_postal(self):
        """
        :rtype: object_.Address
        """

        return self._address_postal

    @property
    def version_terms_of_service(self):
        """
        :rtype: str
        """

        return self._version_terms_of_service

    @property
    def director_alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._director_alias

    @property
    def directors(self):
        """
        :rtype: list[object_.LabelUser]
        """

        return self._directors

    @property
    def language(self):
        """
        :rtype: str
        """

        return self._language

    @property
    def country(self):
        """
        :rtype: str
        """

        return self._country

    @property
    def region(self):
        """
        :rtype: str
        """

        return self._region

    @property
    def ubo(self):
        """
        :rtype: list[object_.Ubo]
        """

        return self._ubo

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def session_timeout(self):
        """
        :rtype: int
        """

        return self._session_timeout

    @property
    def daily_limit_without_confirmation_login(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit_without_confirmation_login

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilter]
        """

        return self._notification_filters

    @property
    def customer(self):
        """
        :rtype: Customer
        """

        return self._customer

    @property
    def customer_limit(self):
        """
        :rtype: CustomerLimit
        """

        return self._customer_limit

    @property
    def billing_contract(self):
        """
        :rtype: list[BillingContractSubscription]
        """

        return self._billing_contract

    @property
    def deny_reason(self):
        """
        :rtype: str
        """

        return self._deny_reason

    @property
    def relations(self):
        """
        :rtype: list[RelationUser]
        """

        return self._relations

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._name is not None:
            return False

        if self._display_name is not None:
            return False

        if self._public_nick_name is not None:
            return False

        if self._alias is not None:
            return False

        if self._chamber_of_commerce_number is not None:
            return False

        if self._legal_form is not None:
            return False

        if self._type_of_business_entity is not None:
            return False

        if self._sector_of_industry is not None:
            return False

        if self._counter_bank_iban is not None:
            return False

        if self._avatar is not None:
            return False

        if self._address_main is not None:
            return False

        if self._address_postal is not None:
            return False

        if self._version_terms_of_service is not None:
            return False

        if self._director_alias is not None:
            return False

        if self._directors is not None:
            return False

        if self._language is not None:
            return False

        if self._country is not None:
            return False

        if self._region is not None:
            return False

        if self._ubo is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._session_timeout is not None:
            return False

        if self._daily_limit_without_confirmation_login is not None:
            return False

        if self._notification_filters is not None:
            return False

        if self._customer is not None:
            return False

        if self._customer_limit is not None:
            return False

        if self._billing_contract is not None:
            return False

        if self._deny_reason is not None:
            return False

        if self._relations is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: UserCompany
        """

        return converter.json_to_class(UserCompany, json_str)


class Customer(BunqModel):
    """
    Used to view a customer.
    
    :param _billing_account_id: The primary billing account account's id.
    :type _billing_account_id: str
    :param _invoice_notification_preference: The preferred notification type for
    invoices.
    :type _invoice_notification_preference: str
    :param _id_: The id of the customer.
    :type _id_: int
    :param _created: The timestamp of the customer object's creation.
    :type _created: str
    :param _updated: The timestamp of the customer object's last update.
    :type _updated: str
    """

    # Field constants.
    FIELD_BILLING_ACCOUNT_ID = "billing_account_id"
    FIELD_INVOICE_NOTIFICATION_PREFERENCE = "invoice_notification_preference"

    _id_ = None
    _created = None
    _updated = None
    _billing_account_id = None
    _invoice_notification_preference = None
    _billing_account_id_field_for_request = None
    _invoice_notification_preference_field_for_request = None

    def __init__(self, billing_account_id=None, invoice_notification_preference=None):
        """
        :param billing_account_id: The primary billing account account's id.
        :type billing_account_id: str
        :param invoice_notification_preference: The preferred notification type for
        invoices
        :type invoice_notification_preference: str
        """

        self._billing_account_id_field_for_request = billing_account_id
        self._invoice_notification_preference_field_for_request = invoice_notification_preference

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def billing_account_id(self):
        """
        :rtype: str
        """

        return self._billing_account_id

    @property
    def invoice_notification_preference(self):
        """
        :rtype: str
        """

        return self._invoice_notification_preference

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._billing_account_id is not None:
            return False

        if self._invoice_notification_preference is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Customer
        """

        return converter.json_to_class(Customer, json_str)


class RelationUser(BunqModel):
    """
    Manage the relation user details.
    
    :param _user_id: The user's ID.
    :type _user_id: str
    :param _counter_user_id: The counter user's ID.
    :type _counter_user_id: str
    :param _label_user: The user's label.
    :type _label_user: object_.LabelUser
    :param _counter_label_user: The counter user's label.
    :type _counter_label_user: object_.LabelUser
    :param _relationship: The requested relation type.
    :type _relationship: str
    :param _status: The request's status, only for UPDATE.
    :type _status: str
    """

    _user_id = None
    _counter_user_id = None
    _label_user = None
    _counter_label_user = None
    _relationship = None
    _status = None

    @property
    def user_id(self):
        """
        :rtype: str
        """

        return self._user_id

    @property
    def counter_user_id(self):
        """
        :rtype: str
        """

        return self._counter_user_id

    @property
    def label_user(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user

    @property
    def counter_label_user(self):
        """
        :rtype: object_.LabelUser
        """

        return self._counter_label_user

    @property
    def relationship(self):
        """
        :rtype: str
        """

        return self._relationship

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._user_id is not None:
            return False

        if self._counter_user_id is not None:
            return False

        if self._label_user is not None:
            return False

        if self._counter_label_user is not None:
            return False

        if self._relationship is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RelationUser
        """

        return converter.json_to_class(RelationUser, json_str)


class ConfirmationOfFunds(BunqModel):
    """
    Used to confirm availability of funds on an account.
    
    :param _pointer_iban: The pointer (IBAN) of the account we're querying.
    :type _pointer_iban: object_.Pointer
    :param _amount: The amount we want to check for.
    :type _amount: object_.Amount
    :param _has_sufficient_funds: Whether the account has sufficient funds.
    :type _has_sufficient_funds: bool
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/confirmation-of-funds"

    # Field constants.
    FIELD_POINTER_IBAN = "pointer_iban"
    FIELD_AMOUNT = "amount"

    # Object type.
    _OBJECT_TYPE_POST = "ConfirmationOfFunds"

    _has_sufficient_funds = None
    _pointer_iban_field_for_request = None
    _amount_field_for_request = None

    def __init__(self, pointer_iban, amount):
        """
        :param pointer_iban: The pointer (IBAN) of the account we're querying.
        :type pointer_iban: object_.Pointer
        :param amount: The amount we want to check for.
        :type amount: object_.Amount
        """

        self._pointer_iban_field_for_request = pointer_iban
        self._amount_field_for_request = amount

    @classmethod
    def create(cls, pointer_iban, amount, custom_headers=None):
        """
        :type user_id: int
        :param pointer_iban: The pointer (IBAN) of the account we're querying.
        :type pointer_iban: object_.Pointer
        :param amount: The amount we want to check for.
        :type amount: object_.Amount
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseConfirmationOfFunds
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_POINTER_IBAN: pointer_iban,
            cls.FIELD_AMOUNT: amount
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseConfirmationOfFunds.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_POST)
        )

    @property
    def has_sufficient_funds(self):
        """
        :rtype: bool
        """

        return self._has_sufficient_funds

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._has_sufficient_funds is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ConfirmationOfFunds
        """

        return converter.json_to_class(ConfirmationOfFunds, json_str)


class DeviceServer(BunqModel):
    """
    After having created an Installation you can now create a DeviceServer. A
    DeviceServer is needed to do a login call with session-server.
    
    :param _description: The description of the DeviceServer.
    :type _description: str
    :param _secret: The API key. You can request an API key in the bunq app.
    :type _secret: str
    :param _permitted_ips: An array of IPs (v4 or v6) this DeviceServer will be
    able to do calls from. These will be linked to the API key.
    :type _permitted_ips: list[str]
    :param _id_: The id of the DeviceServer as created on the server.
    :type _id_: int
    :param _created: The timestamp of the DeviceServer's creation.
    :type _created: str
    :param _updated: The timestamp of the DeviceServer's last update.
    :type _updated: str
    :param _ip: The ip address which was used to create the DeviceServer.
    :type _ip: str
    :param _status: The status of the DeviceServer. Can be ACTIVE, BLOCKED,
    NEEDS_CONFIRMATION or OBSOLETE.
    :type _status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "device-server"
    _ENDPOINT_URL_READ = "device-server/{}"
    _ENDPOINT_URL_LISTING = "device-server"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_SECRET = "secret"
    FIELD_PERMITTED_IPS = "permitted_ips"

    # Object type.
    _OBJECT_TYPE_GET = "DeviceServer"

    _id_ = None
    _created = None
    _updated = None
    _description = None
    _ip = None
    _status = None
    _description_field_for_request = None
    _secret_field_for_request = None
    _permitted_ips_field_for_request = None

    def __init__(self, description, secret, permitted_ips=None):
        """
        :param description: The description of the DeviceServer. This is only for
        your own reference when reading the DeviceServer again.
        :type description: str
        :param secret: The API key. You can request an API key in the bunq app.
        :type secret: str
        :param permitted_ips: An array of IPs (v4 or v6) this DeviceServer will be
        able to do calls from. These will be linked to the API key.
        :type permitted_ips: list[str]
        """

        self._description_field_for_request = description
        self._secret_field_for_request = secret
        self._permitted_ips_field_for_request = permitted_ips

    @classmethod
    def create(cls, description, secret, permitted_ips=None, custom_headers=None):
        """
        Create a new DeviceServer providing the installation token in the header
        and signing the request with the private part of the key you used to
        create the installation. The API Key that you are using will be bound to
        the IP address of the DeviceServer which you have
        created.<br/><br/>Using a Wildcard API Key gives you the freedom to make
        API calls even if the IP address has changed after the POST
        device-server.<br/><br/>Find out more at this link <a
        href="https://bunq.com/en/apikey-dynamic-ip"
        target="_blank">https://bunq.com/en/apikey-dynamic-ip</a>.
        
        :param description: The description of the DeviceServer. This is only
        for your own reference when reading the DeviceServer again.
        :type description: str
        :param secret: The API key. You can request an API key in the bunq app.
        :type secret: str
        :param permitted_ips: An array of IPs (v4 or v6) this DeviceServer will
        be able to do calls from. These will be linked to the API key.
        :type permitted_ips: list[str]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_SECRET: secret,
            cls.FIELD_PERMITTED_IPS: permitted_ips
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, device_server_id, custom_headers=None):
        """
        Get one of your DeviceServers.
        
        :type api_context: ApiContext
        :type device_server_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDeviceServer
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(device_server_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseDeviceServer.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get a collection of all the DeviceServers you have created.
        
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDeviceServerList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseDeviceServerList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def ip(self):
        """
        :rtype: str
        """

        return self._ip

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._description is not None:
            return False

        if self._ip is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: DeviceServer
        """

        return converter.json_to_class(DeviceServer, json_str)


class Device(BunqModel, AnchorObjectInterface):
    """
    Used to get a Device or a listing of Devices. Creating a DeviceServer should
    happen via /device-server
    
    :param _DeviceServer: 
    :type _DeviceServer: DeviceServer
    """

    # Error constants.
    _ERROR_NULL_FIELDS = "All fields of an extended model or object are null."

    # Endpoint constants.
    _ENDPOINT_URL_READ = "device/{}"
    _ENDPOINT_URL_LISTING = "device"

    # Object type.
    _OBJECT_TYPE_GET = "Device"

    _DeviceServer = None

    @classmethod
    def get(cls, device_id, custom_headers=None):
        """
        Get a single Device. A Device is either a DevicePhone or a DeviceServer.
        
        :type api_context: ApiContext
        :type device_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDevice
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(device_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseDevice.cast_from_bunq_response(
            cls._from_json(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get a collection of Devices. A Device is either a DevicePhone or a
        DeviceServer.
        
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDeviceList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseDeviceList.cast_from_bunq_response(
            cls._from_json_list(response_raw)
        )

    @property
    def DeviceServer(self):
        """
        :rtype: DeviceServer
        """

        return self._DeviceServer

    def get_referenced_object(self):
        """
        :rtype: BunqModel
        :raise: BunqException
        """

        if self._DeviceServer is not None:
            return self._DeviceServer

        raise BunqException(self._ERROR_NULL_FIELDS)

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._DeviceServer is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Device
        """

        return converter.json_to_class(Device, json_str)


class DraftPayment(BunqModel):
    """
    A DraftPayment is like a regular Payment, but it needs to be accepted by the
    sending party before the actual Payment is done.
    
    :param _status: The status of the DraftPayment.
    :type _status: str
    :param _entries: The entries in the DraftPayment.
    :type _entries: list[object_.DraftPaymentEntry]
    :param _previous_updated_timestamp: The last updated_timestamp that you
    received for this DraftPayment. This needs to be provided to prevent race
    conditions.
    :type _previous_updated_timestamp: str
    :param _number_of_required_accepts: The number of accepts that are required
    for the draft payment to receive status ACCEPTED. Currently only 1 is valid.
    :type _number_of_required_accepts: int
    :param _schedule: The schedule details.
    :type _schedule: Schedule
    :param _id_: The id of the created DrafPayment.
    :type _id_: int
    :param _monetary_account_id: The id of the MonetaryAccount the DraftPayment
    applies to.
    :type _monetary_account_id: int
    :param _user_alias_created: The label of the User who created the
    DraftPayment.
    :type _user_alias_created: object_.LabelUser
    :param _responses: All responses to this draft payment.
    :type _responses: list[object_.DraftPaymentResponse]
    :param _type_: The type of the DraftPayment.
    :type _type_: str
    :param _object_: The Payment or PaymentBatch. This will only be present
    after the DraftPayment has been accepted.
    :type _object_: object_.DraftPaymentAnchorObject
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/draft-payment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/draft-payment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/draft-payment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/draft-payment/{}"

    # Field constants.
    FIELD_STATUS = "status"
    FIELD_ENTRIES = "entries"
    FIELD_PREVIOUS_UPDATED_TIMESTAMP = "previous_updated_timestamp"
    FIELD_NUMBER_OF_REQUIRED_ACCEPTS = "number_of_required_accepts"
    FIELD_SCHEDULE = "schedule"

    # Object type.
    _OBJECT_TYPE_GET = "DraftPayment"

    _id_ = None
    _monetary_account_id = None
    _user_alias_created = None
    _responses = None
    _status = None
    _type_ = None
    _entries = None
    _object_ = None
    _request_reference_split_the_bill = None
    _schedule = None
    _status_field_for_request = None
    _entries_field_for_request = None
    _previous_updated_timestamp_field_for_request = None
    _number_of_required_accepts_field_for_request = None
    _schedule_field_for_request = None

    def __init__(self, number_of_required_accepts, entries=None, status=None, previous_updated_timestamp=None,
                 schedule=None):
        """
        :param entries: The list of entries in the DraftPayment. Each entry will
        result in a payment when the DraftPayment is accepted.
        :type entries: list[object_.DraftPaymentEntry]
        :param number_of_required_accepts: The number of accepts that are required
        for the draft payment to receive status ACCEPTED. Currently only 1 is valid.
        :type number_of_required_accepts: int
        :param status: The status of the DraftPayment.
        :type status: str
        :param previous_updated_timestamp: The last updated_timestamp that you
        received for this DraftPayment. This needs to be provided to prevent race
        conditions.
        :type previous_updated_timestamp: str
        :param schedule: The schedule details when creating or updating a scheduled
        payment.
        :type schedule: Schedule
        """

        self._entries_field_for_request = entries
        self._number_of_required_accepts_field_for_request = number_of_required_accepts
        self._status_field_for_request = status
        self._previous_updated_timestamp_field_for_request = previous_updated_timestamp
        self._schedule_field_for_request = schedule

    @classmethod
    def create(cls, entries, number_of_required_accepts, monetary_account_id=None, status=None,
               previous_updated_timestamp=None, schedule=None, custom_headers=None):
        """
        Create a new DraftPayment.
        
        :type user_id: int
        :type monetary_account_id: int
        :param entries: The list of entries in the DraftPayment. Each entry will
        result in a payment when the DraftPayment is accepted.
        :type entries: list[object_.DraftPaymentEntry]
        :param number_of_required_accepts: The number of accepts that are
        required for the draft payment to receive status ACCEPTED. Currently
        only 1 is valid.
        :type number_of_required_accepts: int
        :param status: The status of the DraftPayment.
        :type status: str
        :param previous_updated_timestamp: The last updated_timestamp that you
        received for this DraftPayment. This needs to be provided to prevent
        race conditions.
        :type previous_updated_timestamp: str
        :param schedule: The schedule details when creating or updating a
        scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_ENTRIES: entries,
            cls.FIELD_PREVIOUS_UPDATED_TIMESTAMP: previous_updated_timestamp,
            cls.FIELD_NUMBER_OF_REQUIRED_ACCEPTS: number_of_required_accepts,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, draft_payment_id, monetary_account_id=None, status=None, entries=None,
               previous_updated_timestamp=None, schedule=None, custom_headers=None):
        """
        Update a DraftPayment.
        
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :param status: The status of the DraftPayment.
        :type status: str
        :param entries: The list of entries in the DraftPayment. Each entry will
        result in a payment when the DraftPayment is accepted.
        :type entries: list[object_.DraftPaymentEntry]
        :param previous_updated_timestamp: The last updated_timestamp that you
        received for this DraftPayment. This needs to be provided to prevent
        race conditions.
        :type previous_updated_timestamp: str
        :param schedule: The schedule details when creating or updating a
        scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_ENTRIES: entries,
            cls.FIELD_PREVIOUS_UPDATED_TIMESTAMP: previous_updated_timestamp,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a listing of all DraftPayments from a given MonetaryAccount.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDraftPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseDraftPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, draft_payment_id, monetary_account_id=None, custom_headers=None):
        """
        Get a specific DraftPayment.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDraftPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     draft_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseDraftPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def user_alias_created(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_created

    @property
    def responses(self):
        """
        :rtype: list[object_.DraftPaymentResponse]
        """

        return self._responses

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def entries(self):
        """
        :rtype: list[object_.DraftPaymentEntry]
        """

        return self._entries

    @property
    def object_(self):
        """
        :rtype: object_.DraftPaymentAnchorObject
        """

        return self._object_

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    @property
    def schedule(self):
        """
        :rtype: Schedule
        """

        return self._schedule

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._user_alias_created is not None:
            return False

        if self._responses is not None:
            return False

        if self._status is not None:
            return False

        if self._type_ is not None:
            return False

        if self._entries is not None:
            return False

        if self._object_ is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        if self._schedule is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: DraftPayment
        """

        return converter.json_to_class(DraftPayment, json_str)


class PaymentBatch(BunqModel):
    """
    Create a payment batch, or show the payment batches of a monetary account.
    
    :param _payments: The list of mutations that were made.
    :type _payments: object_.PaymentBatchAnchoredPayment
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment-batch"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/payment-batch/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment-batch/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment-batch"

    # Field constants.
    FIELD_PAYMENTS = "payments"

    # Object type.
    _OBJECT_TYPE_GET = "PaymentBatch"

    _payments = None
    _payments_field_for_request = None

    def __init__(self, payments):
        """
        :param payments: The list of payments we want to send in a single batch.
        :type payments: list[Payment]
        """

        self._payments_field_for_request = payments

    @classmethod
    def create(cls, payments, monetary_account_id=None, custom_headers=None):
        """
        Create a payment batch by sending an array of single payment objects,
        that will become part of the batch.
        
        :type user_id: int
        :type monetary_account_id: int
        :param payments: The list of payments we want to send in a single batch.
        :type payments: list[Payment]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_PAYMENTS: payments
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        Revoke a bunq.to payment batch. The status of all the payments will be
        set to REVOKED.
        
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {

        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        Return the details of a specific payment batch.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponsePaymentBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponsePaymentBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Return all the payment batches for a monetary account.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponsePaymentBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponsePaymentBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def payments(self):
        """
        :rtype: object_.PaymentBatchAnchoredPayment
        """

        return self._payments

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._payments is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: PaymentBatch
        """

        return converter.json_to_class(PaymentBatch, json_str)


class Schedule(BunqModel):
    """
    view for reading the scheduled definitions.
    
    :param _time_start: The schedule start time (UTC).
    :type _time_start: str
    :param _time_end: The schedule end time (UTC).
    :type _time_end: str
    :param _recurrence_unit: The schedule recurrence unit, options: ONCE,
    HOURLY, DAILY, WEEKLY, MONTHLY, YEARLY
    :type _recurrence_unit: str
    :param _recurrence_size: The schedule recurrence size. For example size 4
    and unit WEEKLY means the recurrence is every 4 weeks.
    :type _recurrence_size: int
    :param _status: The schedule status, options: ACTIVE, FINISHED, CANCELLED.
    :type _status: str
    :param _object_: The scheduled object. (Payment, PaymentBatch)
    :type _object_: object_.ScheduleAnchorObject
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule"

    # Field constants.
    FIELD_TIME_START = "time_start"
    FIELD_TIME_END = "time_end"
    FIELD_RECURRENCE_UNIT = "recurrence_unit"
    FIELD_RECURRENCE_SIZE = "recurrence_size"

    # Object type.
    _OBJECT_TYPE_GET = "Schedule"

    _time_start = None
    _time_end = None
    _recurrence_unit = None
    _recurrence_size = None
    _status = None
    _object_ = None
    _time_start_field_for_request = None
    _time_end_field_for_request = None
    _recurrence_unit_field_for_request = None
    _recurrence_size_field_for_request = None

    def __init__(self, time_start=None, recurrence_unit=None, recurrence_size=None, time_end=None):
        """
        :param time_start: The schedule start time (UTC).
        :type time_start: str
        :param recurrence_unit: The schedule recurrence unit, options: ONCE, HOURLY,
        DAILY, WEEKLY, MONTHLY, YEARLY
        :type recurrence_unit: str
        :param recurrence_size: The schedule recurrence size. For example size 4 and
        unit WEEKLY means the recurrence is every 4 weeks.
        :type recurrence_size: int
        :param time_end: The schedule end time (UTC).
        :type time_end: str
        """

        self._time_start_field_for_request = time_start
        self._recurrence_unit_field_for_request = recurrence_unit
        self._recurrence_size_field_for_request = recurrence_size
        self._time_end_field_for_request = time_end

    @classmethod
    def get(cls, schedule_id, monetary_account_id=None, custom_headers=None):
        """
        Get a specific schedule definition for a given monetary account.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSchedule
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseSchedule.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get a collection of scheduled definition for a given monetary account.
        You can add the parameter type to filter the response. When
        type={SCHEDULE_DEFINITION_PAYMENT,SCHEDULE_DEFINITION_PAYMENT_BATCH} is
        provided only schedule definition object that relate to these
        definitions are returned.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseScheduleList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseScheduleList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def time_start(self):
        """
        :rtype: str
        """

        return self._time_start

    @property
    def time_end(self):
        """
        :rtype: str
        """

        return self._time_end

    @property
    def recurrence_unit(self):
        """
        :rtype: str
        """

        return self._recurrence_unit

    @property
    def recurrence_size(self):
        """
        :rtype: int
        """

        return self._recurrence_size

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def object_(self):
        """
        :rtype: object_.ScheduleAnchorObject
        """

        return self._object_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._time_start is not None:
            return False

        if self._time_end is not None:
            return False

        if self._recurrence_unit is not None:
            return False

        if self._recurrence_size is not None:
            return False

        if self._status is not None:
            return False

        if self._object_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Schedule
        """

        return converter.json_to_class(Schedule, json_str)


class DraftShareInviteBankQrCodeContent(BunqModel):
    """
    This call returns the raw content of the QR code that links to this draft
    share invite. When a bunq user scans this QR code with the bunq app the
    draft share invite will be shown on his/her device.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/draft-share-invite-bank/{}/qr-code-content"

    # Object type.
    _OBJECT_TYPE_GET = "DraftShareInviteBankQrCodeContent"

    @classmethod
    def list(cls, draft_share_invite_bank_id, custom_headers=None):
        """
        Returns the raw content of the QR code that links to this draft share
        invite. The raw content is the binary representation of a file, without
        any JSON wrapping.
        
        :type user_id: int
        :type draft_share_invite_bank_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), draft_share_invite_bank_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: DraftShareInviteBankQrCodeContent
        """

        return converter.json_to_class(DraftShareInviteBankQrCodeContent, json_str)


class DraftShareInviteBank(BunqModel):
    """
    Used to create a draft share invite for a monetary account with another bunq
    user, as in the 'Connect' feature in the bunq app. The user that accepts the
    invite can share one of their MonetaryAccounts with the user that created
    the invite.
    
    :param _status: The status of the draft share invite. Can be USED, CANCELLED
    and PENDING.
    :type _status: str
    :param _expiration: The moment when this draft share invite expires.
    :type _expiration: str
    :param _draft_share_settings: The draft share invite details.
    :type _draft_share_settings: object_.DraftShareInviteEntry
    :param _user_alias_created: The user who created the draft share invite.
    :type _user_alias_created: object_.LabelUser
    :param _share_invite_bank_response_id: The id of the share invite bank
    response this draft share belongs to.
    :type _share_invite_bank_response_id: int
    :param _draft_share_url: The URL redirecting user to the draft share invite
    in the app. Only works on mobile devices.
    :type _draft_share_url: str
    :param _id_: The id of the newly created draft share invite.
    :type _id_: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/draft-share-invite-bank"
    _ENDPOINT_URL_READ = "user/{}/draft-share-invite-bank/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/draft-share-invite-bank/{}"
    _ENDPOINT_URL_LISTING = "user/{}/draft-share-invite-bank"

    # Field constants.
    FIELD_STATUS = "status"
    FIELD_EXPIRATION = "expiration"
    FIELD_DRAFT_SHARE_SETTINGS = "draft_share_settings"

    # Object type.
    _OBJECT_TYPE_GET = "DraftShareInviteBank"

    _user_alias_created = None
    _status = None
    _expiration = None
    _share_invite_bank_response_id = None
    _draft_share_url = None
    _draft_share_settings = None
    _id_ = None
    _status_field_for_request = None
    _expiration_field_for_request = None
    _draft_share_settings_field_for_request = None

    def __init__(self, expiration=None, draft_share_settings=None, status=None):
        """
        :param expiration: The moment when this draft share invite expires.
        :type expiration: str
        :param draft_share_settings: The draft share invite details.
        :type draft_share_settings: object_.DraftShareInviteEntry
        :param status: The status of the draft share invite. Can be CANCELLED (the
        user cancels the draft share before it's used).
        :type status: str
        """

        self._expiration_field_for_request = expiration
        self._draft_share_settings_field_for_request = draft_share_settings
        self._status_field_for_request = status

    @classmethod
    def create(cls, expiration, draft_share_settings, status=None, custom_headers=None):
        """
        :type user_id: int
        :param expiration: The moment when this draft share invite expires.
        :type expiration: str
        :param draft_share_settings: The draft share invite details.
        :type draft_share_settings: object_.DraftShareInviteEntry
        :param status: The status of the draft share invite. Can be CANCELLED
        (the user cancels the draft share before it's used).
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_DRAFT_SHARE_SETTINGS: draft_share_settings
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, draft_share_invite_bank_id, custom_headers=None):
        """
        Get the details of a specific draft of a share invite.
        
        :type api_context: ApiContext
        :type user_id: int
        :type draft_share_invite_bank_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDraftShareInviteBank
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), draft_share_invite_bank_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseDraftShareInviteBank.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, draft_share_invite_bank_id, status=None, expiration=None, draft_share_settings=None,
               custom_headers=None):
        """
        Update a draft share invite. When sending status CANCELLED it is
        possible to cancel the draft share invite.
        
        :type user_id: int
        :type draft_share_invite_bank_id: int
        :param status: The status of the draft share invite. Can be CANCELLED
        (the user cancels the draft share before it's used).
        :type status: str
        :param expiration: The moment when this draft share invite expires.
        :type expiration: str
        :param draft_share_settings: The draft share invite details.
        :type draft_share_settings: object_.DraftShareInviteEntry
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_EXPIRATION: expiration,
            cls.FIELD_DRAFT_SHARE_SETTINGS: draft_share_settings
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), draft_share_invite_bank_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseDraftShareInviteBankList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseDraftShareInviteBankList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def user_alias_created(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_created

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def expiration(self):
        """
        :rtype: str
        """

        return self._expiration

    @property
    def share_invite_bank_response_id(self):
        """
        :rtype: int
        """

        return self._share_invite_bank_response_id

    @property
    def draft_share_url(self):
        """
        :rtype: str
        """

        return self._draft_share_url

    @property
    def draft_share_settings(self):
        """
        :rtype: object_.DraftShareInviteEntry
        """

        return self._draft_share_settings

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._user_alias_created is not None:
            return False

        if self._status is not None:
            return False

        if self._expiration is not None:
            return False

        if self._share_invite_bank_response_id is not None:
            return False

        if self._draft_share_url is not None:
            return False

        if self._draft_share_settings is not None:
            return False

        if self._id_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: DraftShareInviteBank
        """

        return converter.json_to_class(DraftShareInviteBank, json_str)


class Event(BunqModel):
    """
    Used to view events. Events are automatically created and contain
    information about everything that happens to your bunq account. In the bunq
    app events are shown in your 'overview'. Examples of when events are created
    or modified: payment sent, payment received, request for payment received or
    connect invite received.
    
    :param _id_: The id of the event.
    :type _id_: int
    :param _created: The timestamp of the event's creation.
    :type _created: str
    :param _updated: The timestamp of the event's last update.
    :type _updated: str
    :param _action: The performed action. Can be: CREATE or UPDATE.
    :type _action: str
    :param _user_id: The id of the user the event applied to (if it was a user
    event).
    :type _user_id: str
    :param _monetary_account_id: The id of the monetary account the event
    applied to (if it was a monetary account event).
    :type _monetary_account_id: str
    :param _object_: The details of the external object the event was created
    for.
    :type _object_: object_.EventObject
    :param _status: The event status. Can be: FINALIZED or AWAITING_REPLY. An
    example of FINALIZED event is a payment received event, while an
    AWAITING_REPLY event is a request received event.
    :type _status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/event/{}"
    _ENDPOINT_URL_LISTING = "user/{}/event"

    # Object type.
    _OBJECT_TYPE_GET = "Event"

    _id_ = None
    _created = None
    _updated = None
    _action = None
    _user_id = None
    _monetary_account_id = None
    _object_ = None
    _status = None

    @classmethod
    def get(cls, event_id, custom_headers=None):
        """
        Get a specific event for a given user.
        
        :type api_context: ApiContext
        :type user_id: int
        :type event_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseEvent
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), event_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseEvent.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get a collection of events for a given user. You can add query the
        parameters monetary_account_id, status and/or display_user_event to
        filter the response. When monetary_account_id={id,id} is provided only
        events that relate to these monetary account ids are returned. When
        status={AWAITING_REPLY/FINALIZED} is provided the response only contains
        events with the status AWAITING_REPLY or FINALIZED. When
        display_user_event={true/false} is set to false user events are excluded
        from the response, when not provided user events are displayed. User
        events are events that are not related to a monetary account (for
        example: connect invites).
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseEventList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseEventList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def action(self):
        """
        :rtype: str
        """

        return self._action

    @property
    def user_id(self):
        """
        :rtype: str
        """

        return self._user_id

    @property
    def monetary_account_id(self):
        """
        :rtype: str
        """

        return self._monetary_account_id

    @property
    def object_(self):
        """
        :rtype: object_.EventObject
        """

        return self._object_

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._action is not None:
            return False

        if self._user_id is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._object_ is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Event
        """

        return converter.json_to_class(Event, json_str)


class FeatureAnnouncement(BunqModel):
    """
    view for updating the feature display.
    
    :param _avatar: The Avatar of the event overview.
    :type _avatar: object_.Avatar
    :param _title: The event overview title of the feature display
    :type _title: str
    :param _sub_title: The event overview subtitle of the feature display
    :type _sub_title: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/feature-announcement/{}"

    # Object type.
    _OBJECT_TYPE_GET = "FeatureAnnouncement"

    _avatar = None
    _title = None
    _sub_title = None

    @classmethod
    def get(cls, feature_announcement_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type feature_announcement_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseFeatureAnnouncement
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), feature_announcement_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseFeatureAnnouncement.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def title(self):
        """
        :rtype: str
        """

        return self._title

    @property
    def sub_title(self):
        """
        :rtype: str
        """

        return self._sub_title

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._avatar is not None:
            return False

        if self._title is not None:
            return False

        if self._sub_title is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: FeatureAnnouncement
        """

        return converter.json_to_class(FeatureAnnouncement, json_str)


class IdealMerchantTransaction(BunqModel):
    """
    View for requesting iDEAL transactions and polling their status.
    
    :param _amount_requested: The requested amount of money to add.
    :type _amount_requested: object_.Amount
    :param _issuer: The BIC of the issuer.
    :type _issuer: str
    :param _monetary_account_id: The id of the monetary account this ideal
    merchant transaction links to.
    :type _monetary_account_id: int
    :param _alias: The alias of the monetary account to add money to.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The alias of the monetary account the money
    comes from.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _amount_guaranteed: In case of a successful transaction, the amount
    of money that will be transferred.
    :type _amount_guaranteed: object_.Amount
    :param _expiration: When the transaction will expire.
    :type _expiration: str
    :param _issuer_name: The Name of the issuer.
    :type _issuer_name: str
    :param _issuer_authentication_url: The URL to visit to 
    :type _issuer_authentication_url: str
    :param _purchase_identifier: The 'purchase ID' of the iDEAL transaction.
    :type _purchase_identifier: str
    :param _status: The status of the transaction.
    :type _status: str
    :param _status_timestamp: When the status was last updated.
    :type _status_timestamp: str
    :param _transaction_identifier: The 'transaction ID' of the iDEAL
    transaction.
    :type _transaction_identifier: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/ideal-merchant-transaction"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/ideal-merchant-transaction"

    # Field constants.
    FIELD_AMOUNT_REQUESTED = "amount_requested"
    FIELD_ISSUER = "issuer"

    # Object type.
    _OBJECT_TYPE_GET = "IdealMerchantTransaction"

    _monetary_account_id = None
    _alias = None
    _counterparty_alias = None
    _amount_guaranteed = None
    _amount_requested = None
    _expiration = None
    _issuer = None
    _issuer_name = None
    _issuer_authentication_url = None
    _purchase_identifier = None
    _status = None
    _status_timestamp = None
    _transaction_identifier = None
    _amount_requested_field_for_request = None
    _issuer_field_for_request = None

    def __init__(self, amount_requested, issuer):
        """
        :param amount_requested: The requested amount of money to add.
        :type amount_requested: object_.Amount
        :param issuer: The BIC of the issuing bank to ask for money.
        :type issuer: str
        """

        self._amount_requested_field_for_request = amount_requested
        self._issuer_field_for_request = issuer

    @classmethod
    def create(cls, amount_requested, issuer, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param amount_requested: The requested amount of money to add.
        :type amount_requested: object_.Amount
        :param issuer: The BIC of the issuing bank to ask for money.
        :type issuer: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_AMOUNT_REQUESTED: amount_requested,
            cls.FIELD_ISSUER: issuer
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, ideal_merchant_transaction_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseIdealMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     ideal_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseIdealMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseIdealMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseIdealMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def amount_guaranteed(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_guaranteed

    @property
    def amount_requested(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_requested

    @property
    def expiration(self):
        """
        :rtype: str
        """

        return self._expiration

    @property
    def issuer(self):
        """
        :rtype: str
        """

        return self._issuer

    @property
    def issuer_name(self):
        """
        :rtype: str
        """

        return self._issuer_name

    @property
    def issuer_authentication_url(self):
        """
        :rtype: str
        """

        return self._issuer_authentication_url

    @property
    def purchase_identifier(self):
        """
        :rtype: str
        """

        return self._purchase_identifier

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def status_timestamp(self):
        """
        :rtype: str
        """

        return self._status_timestamp

    @property
    def transaction_identifier(self):
        """
        :rtype: str
        """

        return self._transaction_identifier

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._monetary_account_id is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._amount_guaranteed is not None:
            return False

        if self._amount_requested is not None:
            return False

        if self._expiration is not None:
            return False

        if self._issuer is not None:
            return False

        if self._issuer_name is not None:
            return False

        if self._issuer_authentication_url is not None:
            return False

        if self._purchase_identifier is not None:
            return False

        if self._status is not None:
            return False

        if self._status_timestamp is not None:
            return False

        if self._transaction_identifier is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: IdealMerchantTransaction
        """

        return converter.json_to_class(IdealMerchantTransaction, json_str)


class SchedulePayment(BunqModel):
    """
    Endpoint for schedule payments.
    
    :param _payment: The payment details.
    :type _payment: object_.SchedulePaymentEntry
    :param _schedule: The schedule details.
    :type _schedule: Schedule
    :param _status: The schedule status, options: ACTIVE, FINISHED, CANCELLED.
    :type _status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule-payment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule-payment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment/{}"

    # Field constants.
    FIELD_PAYMENT = "payment"
    FIELD_SCHEDULE = "schedule"

    # Object type.
    _OBJECT_TYPE_GET = "ScheduledPayment"
    _OBJECT_TYPE_PUT = "ScheduledPayment"

    _payment = None
    _schedule = None
    _status = None
    _payment_field_for_request = None
    _schedule_field_for_request = None

    def __init__(self, payment=None, schedule=None):
        """
        :param payment: The payment details.
        :type payment: object_.SchedulePaymentEntry
        :param schedule: The schedule details when creating or updating a scheduled
        payment.
        :type schedule: Schedule
        """

        self._payment_field_for_request = payment
        self._schedule_field_for_request = schedule

    @classmethod
    def create(cls, payment, schedule, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param payment: The payment details.
        :type payment: object_.SchedulePaymentEntry
        :param schedule: The schedule details when creating or updating a
        scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_PAYMENT: payment,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def get(cls, schedule_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSchedulePayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseSchedulePayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSchedulePaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseSchedulePaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, schedule_payment_id, monetary_account_id=None, payment=None, schedule=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :param payment: The payment details.
        :type payment: object_.SchedulePaymentEntry
        :param schedule: The schedule details when creating or updating a
        scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSchedulePayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_PAYMENT: payment,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseSchedulePayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_PUT)
        )

    @property
    def payment(self):
        """
        :rtype: object_.SchedulePaymentEntry
        """

        return self._payment

    @property
    def schedule(self):
        """
        :rtype: Schedule
        """

        return self._schedule

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._payment is not None:
            return False

        if self._schedule is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: SchedulePayment
        """

        return converter.json_to_class(SchedulePayment, json_str)


class SchedulePaymentBatch(BunqModel):
    """
    Endpoint for schedule payment batches.
    
    :param _payments: The payment details.
    :type _payments: list[object_.SchedulePaymentEntry]
    :param _schedule: The schedule details.
    :type _schedule: Schedule
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment-batch"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment-batch/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment-batch/{}"

    # Field constants.
    FIELD_PAYMENTS = "payments"
    FIELD_SCHEDULE = "schedule"

    _payments = None
    _schedule = None
    _payments_field_for_request = None
    _schedule_field_for_request = None

    def __init__(self, payments=None, schedule=None):
        """
        :param payments: The payment details.
        :type payments: list[object_.SchedulePaymentEntry]
        :param schedule: The schedule details when creating a scheduled payment.
        :type schedule: Schedule
        """

        self._payments_field_for_request = payments
        self._schedule_field_for_request = schedule

    @classmethod
    def create(cls, payments, schedule, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param payments: The payment details.
        :type payments: list[object_.SchedulePaymentEntry]
        :param schedule: The schedule details when creating a scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_PAYMENTS: payments,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_payment_batch_id, monetary_account_id=None, payments=None, schedule=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :param payments: The payment details.
        :type payments: list[object_.SchedulePaymentEntry]
        :param schedule: The schedule details when creating a scheduled payment.
        :type schedule: Schedule
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_PAYMENTS: payments,
            cls.FIELD_SCHEDULE: schedule
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @property
    def payments(self):
        """
        :rtype: list[object_.SchedulePaymentEntry]
        """

        return self._payments

    @property
    def schedule(self):
        """
        :rtype: Schedule
        """

        return self._schedule

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._payments is not None:
            return False

        if self._schedule is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: SchedulePaymentBatch
        """

        return converter.json_to_class(SchedulePaymentBatch, json_str)


class ScheduleInstance(BunqModel):
    """
    view for reading, updating and listing the scheduled instance.
    
    :param _state: The state of the scheduleInstance. (FINISHED_SUCCESSFULLY,
    RETRY, FAILED_USER_ERROR)
    :type _state: str
    :param _time_start: The schedule start time (UTC).
    :type _time_start: str
    :param _time_end: The schedule end time (UTC).
    :type _time_end: str
    :param _error_message: The message when the scheduled instance has run and
    failed due to user error.
    :type _error_message: list[object_.Error]
    :param _scheduled_object: The scheduled object. (Payment, PaymentBatch)
    :type _scheduled_object: object_.ScheduleAnchorObject
    :param _result_object: The result object of this schedule instance.
    (Payment, PaymentBatch)
    :type _result_object: object_.ScheduleInstanceAnchorObject
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule/{}/schedule-instance"

    # Field constants.
    FIELD_STATE = "state"

    # Object type.
    _OBJECT_TYPE_GET = "ScheduledInstance"

    _state = None
    _time_start = None
    _time_end = None
    _error_message = None
    _scheduled_object = None
    _result_object = None
    _request_reference_split_the_bill = None
    _state_field_for_request = None

    def __init__(self, state=None):
        """
        :param state: Change the state of the scheduleInstance from
        FAILED_USER_ERROR to RETRY.
        :type state: str
        """

        self._state_field_for_request = state

    @classmethod
    def get(cls, schedule_id, schedule_instance_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseScheduleInstance
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_id, schedule_instance_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseScheduleInstance.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, schedule_id, schedule_instance_id, monetary_account_id=None, state=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :param state: Change the state of the scheduleInstance from
        FAILED_USER_ERROR to RETRY.
        :type state: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATE: state
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, schedule_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseScheduleInstanceList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseScheduleInstanceList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def state(self):
        """
        :rtype: str
        """

        return self._state

    @property
    def time_start(self):
        """
        :rtype: str
        """

        return self._time_start

    @property
    def time_end(self):
        """
        :rtype: str
        """

        return self._time_end

    @property
    def error_message(self):
        """
        :rtype: list[object_.Error]
        """

        return self._error_message

    @property
    def scheduled_object(self):
        """
        :rtype: object_.ScheduleAnchorObject
        """

        return self._scheduled_object

    @property
    def result_object(self):
        """
        :rtype: object_.ScheduleInstanceAnchorObject
        """

        return self._result_object

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._state is not None:
            return False

        if self._time_start is not None:
            return False

        if self._time_end is not None:
            return False

        if self._error_message is not None:
            return False

        if self._scheduled_object is not None:
            return False

        if self._result_object is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ScheduleInstance
        """

        return converter.json_to_class(ScheduleInstance, json_str)


class MasterCardAction(BunqModel):
    """
    MasterCard transaction view.
    
    :param _id_: The id of the MastercardAction.
    :type _id_: int
    :param _monetary_account_id: The id of the monetary account this action
    links to.
    :type _monetary_account_id: int
    :param _card_id: The id of the card this action links to.
    :type _card_id: int
    :param _amount_local: The amount of the transaction in local currency.
    :type _amount_local: object_.Amount
    :param _amount_converted: The amount of the transaction in local currency.
    :type _amount_converted: object_.Amount
    :param _amount_billing: The amount of the transaction in the monetary
    account's currency.
    :type _amount_billing: object_.Amount
    :param _amount_original_local: The original amount in local currency.
    :type _amount_original_local: object_.Amount
    :param _amount_original_billing: The original amount in the monetary
    account's currency.
    :type _amount_original_billing: object_.Amount
    :param _amount_fee: The fee amount as charged by the merchant, if
    applicable.
    :type _amount_fee: object_.Amount
    :param _card_authorisation_id_response: The response code by which
    authorised transaction can be identified as authorised by bunq.
    :type _card_authorisation_id_response: str
    :param _decision: Why the transaction was denied, if it was denied, or just
    ALLOWED.
    :type _decision: str
    :param _payment_status: The payment status of the transaction. For example
    PAYMENT_SUCCESSFUL, for a successful payment.
    :type _payment_status: str
    :param _decision_description: Empty if allowed, otherwise a textual
    explanation of why it was denied.
    :type _decision_description: str
    :param _decision_description_translated: Empty if allowed, otherwise a
    textual explanation of why it was denied in user's language.
    :type _decision_description_translated: str
    :param _description: The description for this transaction to display.
    :type _description: str
    :param _authorisation_status: The status in the authorisation process.
    :type _authorisation_status: str
    :param _authorisation_type: The type of transaction that was delivered using
    the card.
    :type _authorisation_type: str
    :param _pan_entry_mode_user: The type of entry mode the user used. Can be
    'ATM', 'ICC', 'MAGNETIC_STRIPE' or 'E_COMMERCE'.
    :type _pan_entry_mode_user: str
    :param _settlement_status: The setlement status in the authorisation
    process.
    :type _settlement_status: str
    :param _clearing_status: The clearing status of the authorisation. Can be
    'PENDING', 'FIRST_PRESENTMENT_COMPLETE' or 'REFUND_LENIENCY'.
    :type _clearing_status: str
    :param _maturity_date: The maturity date.
    :type _maturity_date: str
    :param _city: The city where the message originates from as announced by the
    terminal.
    :type _city: str
    :param _alias: The monetary account label of the account that this action is
    created for.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The monetary account label of the counterparty.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _label_card: The label of the card.
    :type _label_card: object_.LabelCard
    :param _token_status: If this is a tokenisation action, this shows the
    status of the token.
    :type _token_status: str
    :param _reservation_expiry_time: If this is a reservation, the moment the
    reservation will expire.
    :type _reservation_expiry_time: str
    :param _clearing_expiry_time: The time when the processing of the clearing
    is expired, refunding the authorisation.
    :type _clearing_expiry_time: str
    :param _applied_limit: The type of the limit applied to validate if this
    MasterCardAction was within the spending limits. The returned string matches
    the limit types as defined in the card endpoint.
    :type _applied_limit: str
    :param _secure_code_id: The secure code id for this mastercard action or
    null.
    :type _secure_code_id: int
    :param _wallet_provider_id: The ID of the wallet provider as defined by
    MasterCard. 420 = bunq Android app with Tap&Pay; 103 = Apple Pay.
    :type _wallet_provider_id: str
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/mastercard-action/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/mastercard-action"

    # Object type.
    _OBJECT_TYPE_GET = "MasterCardAction"

    _id_ = None
    _monetary_account_id = None
    _card_id = None
    _amount_local = None
    _amount_converted = None
    _amount_billing = None
    _amount_original_local = None
    _amount_original_billing = None
    _amount_fee = None
    _card_authorisation_id_response = None
    _decision = None
    _payment_status = None
    _decision_description = None
    _decision_description_translated = None
    _description = None
    _authorisation_status = None
    _authorisation_type = None
    _pan_entry_mode_user = None
    _settlement_status = None
    _clearing_status = None
    _maturity_date = None
    _city = None
    _alias = None
    _counterparty_alias = None
    _label_card = None
    _token_status = None
    _reservation_expiry_time = None
    _clearing_expiry_time = None
    _applied_limit = None
    _secure_code_id = None
    _wallet_provider_id = None
    _request_reference_split_the_bill = None

    @classmethod
    def get(cls, master_card_action_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type master_card_action_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMasterCardAction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     master_card_action_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseMasterCardAction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMasterCardActionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseMasterCardActionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def card_id(self):
        """
        :rtype: int
        """

        return self._card_id

    @property
    def amount_local(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_local

    @property
    def amount_converted(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_converted

    @property
    def amount_billing(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_billing

    @property
    def amount_original_local(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_original_local

    @property
    def amount_original_billing(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_original_billing

    @property
    def amount_fee(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_fee

    @property
    def card_authorisation_id_response(self):
        """
        :rtype: str
        """

        return self._card_authorisation_id_response

    @property
    def decision(self):
        """
        :rtype: str
        """

        return self._decision

    @property
    def payment_status(self):
        """
        :rtype: str
        """

        return self._payment_status

    @property
    def decision_description(self):
        """
        :rtype: str
        """

        return self._decision_description

    @property
    def decision_description_translated(self):
        """
        :rtype: str
        """

        return self._decision_description_translated

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def authorisation_status(self):
        """
        :rtype: str
        """

        return self._authorisation_status

    @property
    def authorisation_type(self):
        """
        :rtype: str
        """

        return self._authorisation_type

    @property
    def pan_entry_mode_user(self):
        """
        :rtype: str
        """

        return self._pan_entry_mode_user

    @property
    def settlement_status(self):
        """
        :rtype: str
        """

        return self._settlement_status

    @property
    def clearing_status(self):
        """
        :rtype: str
        """

        return self._clearing_status

    @property
    def maturity_date(self):
        """
        :rtype: str
        """

        return self._maturity_date

    @property
    def city(self):
        """
        :rtype: str
        """

        return self._city

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def label_card(self):
        """
        :rtype: object_.LabelCard
        """

        return self._label_card

    @property
    def token_status(self):
        """
        :rtype: str
        """

        return self._token_status

    @property
    def reservation_expiry_time(self):
        """
        :rtype: str
        """

        return self._reservation_expiry_time

    @property
    def clearing_expiry_time(self):
        """
        :rtype: str
        """

        return self._clearing_expiry_time

    @property
    def applied_limit(self):
        """
        :rtype: str
        """

        return self._applied_limit

    @property
    def secure_code_id(self):
        """
        :rtype: int
        """

        return self._secure_code_id

    @property
    def wallet_provider_id(self):
        """
        :rtype: str
        """

        return self._wallet_provider_id

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._card_id is not None:
            return False

        if self._amount_local is not None:
            return False

        if self._amount_converted is not None:
            return False

        if self._amount_billing is not None:
            return False

        if self._amount_original_local is not None:
            return False

        if self._amount_original_billing is not None:
            return False

        if self._amount_fee is not None:
            return False

        if self._card_authorisation_id_response is not None:
            return False

        if self._decision is not None:
            return False

        if self._payment_status is not None:
            return False

        if self._decision_description is not None:
            return False

        if self._decision_description_translated is not None:
            return False

        if self._description is not None:
            return False

        if self._authorisation_status is not None:
            return False

        if self._authorisation_type is not None:
            return False

        if self._pan_entry_mode_user is not None:
            return False

        if self._settlement_status is not None:
            return False

        if self._clearing_status is not None:
            return False

        if self._maturity_date is not None:
            return False

        if self._city is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._label_card is not None:
            return False

        if self._token_status is not None:
            return False

        if self._reservation_expiry_time is not None:
            return False

        if self._clearing_expiry_time is not None:
            return False

        if self._applied_limit is not None:
            return False

        if self._secure_code_id is not None:
            return False

        if self._wallet_provider_id is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MasterCardAction
        """

        return converter.json_to_class(MasterCardAction, json_str)


class RequestInquiryBatch(BunqModel):
    """
    Create a batch of requests for payment, or show the request batches of a
    monetary account.
    
    :param _request_inquiries: The list of requests that were made.
    :type _request_inquiries: list[RequestInquiry]
    :param _status: The status of the request.
    :type _status: str
    :param _total_amount_inquired: The total amount originally inquired for this
    batch.
    :type _total_amount_inquired: object_.Amount
    :param _event_id: The ID of the associated event if the request batch was
    made using 'split the bill'.
    :type _event_id: int
    :param _reference_split_the_bill: The reference to the object used for split
    the bill. Can be Payment, PaymentBatch, ScheduleInstance, RequestResponse
    and MasterCardAction
    :type _reference_split_the_bill:
    object_.RequestReferenceSplitTheBillAnchorObject
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry-batch"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry-batch/{}"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry-batch/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry-batch"

    # Field constants.
    FIELD_REQUEST_INQUIRIES = "request_inquiries"
    FIELD_STATUS = "status"
    FIELD_TOTAL_AMOUNT_INQUIRED = "total_amount_inquired"
    FIELD_EVENT_ID = "event_id"

    # Object type.
    _OBJECT_TYPE_GET = "RequestInquiryBatch"

    _request_inquiries = None
    _total_amount_inquired = None
    _reference_split_the_bill = None
    _request_inquiries_field_for_request = None
    _status_field_for_request = None
    _total_amount_inquired_field_for_request = None
    _event_id_field_for_request = None

    def __init__(self, request_inquiries, total_amount_inquired, status=None, event_id=None):
        """
        :param request_inquiries: The list of request inquiries we want to send in 1
        batch.
        :type request_inquiries: list[RequestInquiry]
        :param total_amount_inquired: The total amount originally inquired for this
        batch.
        :type total_amount_inquired: object_.Amount
        :param status: The status of the request.
        :type status: str
        :param event_id: The ID of the associated event if the request batch was
        made using 'split the bill'.
        :type event_id: int
        """

        self._request_inquiries_field_for_request = request_inquiries
        self._total_amount_inquired_field_for_request = total_amount_inquired
        self._status_field_for_request = status
        self._event_id_field_for_request = event_id

    @classmethod
    def create(cls, request_inquiries, total_amount_inquired, monetary_account_id=None, status=None, event_id=None,
               custom_headers=None):
        """
        Create a request batch by sending an array of single request objects,
        that will become part of the batch.
        
        :type user_id: int
        :type monetary_account_id: int
        :param request_inquiries: The list of request inquiries we want to send
        in 1 batch.
        :type request_inquiries: list[RequestInquiry]
        :param total_amount_inquired: The total amount originally inquired for
        this batch.
        :type total_amount_inquired: object_.Amount
        :param status: The status of the request.
        :type status: str
        :param event_id: The ID of the associated event if the request batch was
        made using 'split the bill'.
        :type event_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_REQUEST_INQUIRIES: request_inquiries,
            cls.FIELD_STATUS: status,
            cls.FIELD_TOTAL_AMOUNT_INQUIRED: total_amount_inquired,
            cls.FIELD_EVENT_ID: event_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_batch_id, monetary_account_id=None, status=None, custom_headers=None):
        """
        Revoke a request batch. The status of all the requests will be set to
        REVOKED.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :param status: The status of the request.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, request_inquiry_batch_id, monetary_account_id=None, custom_headers=None):
        """
        Return the details of a specific request batch.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestInquiryBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseRequestInquiryBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Return all the request batches for a monetary account.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestInquiryBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseRequestInquiryBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def request_inquiries(self):
        """
        :rtype: list[RequestInquiry]
        """

        return self._request_inquiries

    @property
    def total_amount_inquired(self):
        """
        :rtype: object_.Amount
        """

        return self._total_amount_inquired

    @property
    def reference_split_the_bill(self):
        """
        :rtype: object_.RequestReferenceSplitTheBillAnchorObject
        """

        return self._reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._request_inquiries is not None:
            return False

        if self._total_amount_inquired is not None:
            return False

        if self._reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RequestInquiryBatch
        """

        return converter.json_to_class(RequestInquiryBatch, json_str)


class RequestInquiry(BunqModel):
    """
    RequestInquiry, aka 'RFP' (Request for Payment), is one of the innovative
    features that bunq offers. To request payment from another bunq account a
    new Request Inquiry is created. As with payments you can add attachments to
    a RFP. Requests for Payment are the foundation for a number of consumer
    features like 'Split the bill' and 'Request forwarding'. We invite you to
    invent your own based on the bunq api!
    
    :param _amount_inquired: The requested amount.
    :type _amount_inquired: object_.Amount
    :param _counterparty_alias: The LabelMonetaryAccount with the public
    information of the MonetaryAccount the money was requested from.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _description: The description of the inquiry.
    :type _description: str
    :param _attachment: The attachments attached to the payment.
    :type _attachment: list[object_.BunqId]
    :param _merchant_reference: The client's custom reference that was attached
    to the request and the mutation.
    :type _merchant_reference: str
    :param _status: The status of the request.
    :type _status: str
    :param _minimum_age: The minimum age the user accepting the RequestInquiry
    must have.
    :type _minimum_age: int
    :param _require_address: Whether or not an address must be provided on
    accept.
    :type _require_address: str
    :param _want_tip: [DEPRECATED] Whether or not the accepting user can give an
    extra tip on top of the requested Amount. Defaults to false.
    :type _want_tip: bool
    :param _allow_amount_lower: [DEPRECATED] Whether or not the accepting user
    can choose to accept with a lower amount than requested. Defaults to false.
    :type _allow_amount_lower: bool
    :param _allow_amount_higher: [DEPRECATED] Whether or not the accepting user
    can choose to accept with a higher amount than requested. Defaults to false.
    :type _allow_amount_higher: bool
    :param _allow_bunqme: Whether or not sending a bunq.me request is allowed.
    :type _allow_bunqme: bool
    :param _redirect_url: The URL which the user is sent to after accepting or
    rejecting the Request.
    :type _redirect_url: str
    :param _event_id: The ID of the associated event if the request was made
    using 'split the bill'.
    :type _event_id: int
    :param _id_: The id of the created RequestInquiry.
    :type _id_: int
    :param _created: The timestamp of the payment request's creation.
    :type _created: str
    :param _updated: The timestamp of the payment request's last update.
    :type _updated: str
    :param _time_responded: The timestamp of when the payment request was
    responded to.
    :type _time_responded: str
    :param _time_expiry: The timestamp of when the payment request expired.
    :type _time_expiry: str
    :param _monetary_account_id: The id of the monetary account the request
    response applies to.
    :type _monetary_account_id: int
    :param _amount_responded: The responded amount.
    :type _amount_responded: object_.Amount
    :param _user_alias_created: The label that's displayed to the counterparty
    with the mutation. Includes user.
    :type _user_alias_created: object_.LabelUser
    :param _user_alias_revoked: The label that's displayed to the counterparty
    with the mutation. Includes user.
    :type _user_alias_revoked: object_.LabelUser
    :param _batch_id: The id of the batch if the request was part of a batch.
    :type _batch_id: int
    :param _scheduled_id: The id of the scheduled job if the request was
    scheduled.
    :type _scheduled_id: int
    :param _bunqme_share_url: The url that points to the bunq.me request.
    :type _bunqme_share_url: str
    :param _address_shipping: The shipping address provided by the accepting
    user if an address was requested.
    :type _address_shipping: object_.Address
    :param _address_billing: The billing address provided by the accepting user
    if an address was requested.
    :type _address_billing: object_.Address
    :param _geolocation: The geolocation where the payment was done.
    :type _geolocation: object_.Geolocation
    :param _reference_split_the_bill: The reference to the object used for split
    the bill. Can be Payment, PaymentBatch, ScheduleInstance, RequestResponse
    and MasterCardAction
    :type _reference_split_the_bill:
    object_.RequestReferenceSplitTheBillAnchorObject
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry/{}"

    # Field constants.
    FIELD_AMOUNT_INQUIRED = "amount_inquired"
    FIELD_COUNTERPARTY_ALIAS = "counterparty_alias"
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT = "attachment"
    FIELD_MERCHANT_REFERENCE = "merchant_reference"
    FIELD_STATUS = "status"
    FIELD_MINIMUM_AGE = "minimum_age"
    FIELD_REQUIRE_ADDRESS = "require_address"
    FIELD_WANT_TIP = "want_tip"
    FIELD_ALLOW_AMOUNT_LOWER = "allow_amount_lower"
    FIELD_ALLOW_AMOUNT_HIGHER = "allow_amount_higher"
    FIELD_ALLOW_BUNQME = "allow_bunqme"
    FIELD_REDIRECT_URL = "redirect_url"
    FIELD_EVENT_ID = "event_id"

    # Object type.
    _OBJECT_TYPE_PUT = "RequestInquiry"
    _OBJECT_TYPE_GET = "RequestInquiry"

    _id_ = None
    _created = None
    _updated = None
    _time_responded = None
    _time_expiry = None
    _monetary_account_id = None
    _amount_inquired = None
    _amount_responded = None
    _user_alias_created = None
    _user_alias_revoked = None
    _counterparty_alias = None
    _description = None
    _merchant_reference = None
    _attachment = None
    _status = None
    _batch_id = None
    _scheduled_id = None
    _minimum_age = None
    _require_address = None
    _bunqme_share_url = None
    _redirect_url = None
    _address_shipping = None
    _address_billing = None
    _geolocation = None
    _reference_split_the_bill = None
    _amount_inquired_field_for_request = None
    _counterparty_alias_field_for_request = None
    _description_field_for_request = None
    _attachment_field_for_request = None
    _merchant_reference_field_for_request = None
    _status_field_for_request = None
    _minimum_age_field_for_request = None
    _require_address_field_for_request = None
    _want_tip_field_for_request = None
    _allow_amount_lower_field_for_request = None
    _allow_amount_higher_field_for_request = None
    _allow_bunqme_field_for_request = None
    _redirect_url_field_for_request = None
    _event_id_field_for_request = None

    def __init__(self, amount_inquired, counterparty_alias, description, allow_bunqme, attachment=None,
                 merchant_reference=None, status=None, minimum_age=None, require_address=None, want_tip=None,
                 allow_amount_lower=None, allow_amount_higher=None, redirect_url=None, event_id=None):
        """
        :param amount_inquired: The Amount requested to be paid by the person the
        RequestInquiry is sent to. Must be bigger than 0.
        :type amount_inquired: object_.Amount
        :param counterparty_alias: The Alias of the party we are requesting the
        money from. Can be an Alias of type EMAIL, PHONE_NUMBER or IBAN. In case the
        EMAIL or PHONE_NUMBER Alias does not refer to a bunq monetary account,
        'allow_bunqme' needs to be 'true' in order to trigger the creation of a
        bunq.me request. Otherwise no request inquiry will be sent.
        :type counterparty_alias: object_.Pointer
        :param description: The description for the RequestInquiry. Maximum 9000
        characters. Field is required but can be an empty string.
        :type description: str
        :param allow_bunqme: Whether or not sending a bunq.me request is allowed.
        :type allow_bunqme: bool
        :param attachment: The Attachments to attach to the RequestInquiry.
        :type attachment: list[object_.BunqId]
        :param merchant_reference: Optional data to be included with the
        RequestInquiry specific to the merchant. Has to be unique for the same
        source MonetaryAccount.
        :type merchant_reference: str
        :param status: The status of the RequestInquiry. Ignored in POST requests
        but can be used for revoking (cancelling) the RequestInquiry by setting
        REVOKED with a PUT request.
        :type status: str
        :param minimum_age: The minimum age the user accepting the RequestInquiry
        must have. Defaults to not checking. If set, must be between 12 and 100
        inclusive.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the request. Possible values are: BILLING, SHIPPING,
        BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param want_tip: [DEPRECATED] Whether or not the accepting user can give an
        extra tip on top of the requested Amount. Defaults to false.
        :type want_tip: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not the accepting user
        can choose to accept with a lower amount than requested. Defaults to false.
        :type allow_amount_lower: bool
        :param allow_amount_higher: [DEPRECATED] Whether or not the accepting user
        can choose to accept with a higher amount than requested. Defaults to false.
        :type allow_amount_higher: bool
        :param redirect_url: The URL which the user is sent to after accepting or
        rejecting the Request.
        :type redirect_url: str
        :param event_id: The ID of the associated event if the request was made
        using 'split the bill'.
        :type event_id: int
        """

        self._amount_inquired_field_for_request = amount_inquired
        self._counterparty_alias_field_for_request = counterparty_alias
        self._description_field_for_request = description
        self._allow_bunqme_field_for_request = allow_bunqme
        self._attachment_field_for_request = attachment
        self._merchant_reference_field_for_request = merchant_reference
        self._status_field_for_request = status
        self._minimum_age_field_for_request = minimum_age
        self._require_address_field_for_request = require_address
        self._want_tip_field_for_request = want_tip
        self._allow_amount_lower_field_for_request = allow_amount_lower
        self._allow_amount_higher_field_for_request = allow_amount_higher
        self._redirect_url_field_for_request = redirect_url
        self._event_id_field_for_request = event_id

    @classmethod
    def create(cls, amount_inquired, counterparty_alias, description, allow_bunqme, monetary_account_id=None,
               attachment=None, merchant_reference=None, status=None, minimum_age=None, require_address=None,
               want_tip=None, allow_amount_lower=None, allow_amount_higher=None, redirect_url=None, event_id=None,
               custom_headers=None):
        """
        Create a new payment request.
        
        :type user_id: int
        :type monetary_account_id: int
        :param amount_inquired: The Amount requested to be paid by the person
        the RequestInquiry is sent to. Must be bigger than 0.
        :type amount_inquired: object_.Amount
        :param counterparty_alias: The Alias of the party we are requesting the
        money from. Can be an Alias of type EMAIL, PHONE_NUMBER or IBAN. In case
        the EMAIL or PHONE_NUMBER Alias does not refer to a bunq monetary
        account, 'allow_bunqme' needs to be 'true' in order to trigger the
        creation of a bunq.me request. Otherwise no request inquiry will be
        sent.
        :type counterparty_alias: object_.Pointer
        :param description: The description for the RequestInquiry. Maximum 9000
        characters. Field is required but can be an empty string.
        :type description: str
        :param allow_bunqme: Whether or not sending a bunq.me request is
        allowed.
        :type allow_bunqme: bool
        :param attachment: The Attachments to attach to the RequestInquiry.
        :type attachment: list[object_.BunqId]
        :param merchant_reference: Optional data to be included with the
        RequestInquiry specific to the merchant. Has to be unique for the same
        source MonetaryAccount.
        :type merchant_reference: str
        :param status: The status of the RequestInquiry. Ignored in POST
        requests but can be used for revoking (cancelling) the RequestInquiry by
        setting REVOKED with a PUT request.
        :type status: str
        :param minimum_age: The minimum age the user accepting the
        RequestInquiry must have. Defaults to not checking. If set, must be
        between 12 and 100 inclusive.
        :type minimum_age: int
        :param require_address: Whether a billing and shipping address must be
        provided when paying the request. Possible values are: BILLING,
        SHIPPING, BILLING_SHIPPING, NONE, OPTIONAL. Default is NONE.
        :type require_address: str
        :param want_tip: [DEPRECATED] Whether or not the accepting user can give
        an extra tip on top of the requested Amount. Defaults to false.
        :type want_tip: bool
        :param allow_amount_lower: [DEPRECATED] Whether or not the accepting
        user can choose to accept with a lower amount than requested. Defaults
        to false.
        :type allow_amount_lower: bool
        :param allow_amount_higher: [DEPRECATED] Whether or not the accepting
        user can choose to accept with a higher amount than requested. Defaults
        to false.
        :type allow_amount_higher: bool
        :param redirect_url: The URL which the user is sent to after accepting
        or rejecting the Request.
        :type redirect_url: str
        :param event_id: The ID of the associated event if the request was made
        using 'split the bill'.
        :type event_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_AMOUNT_INQUIRED: amount_inquired,
            cls.FIELD_COUNTERPARTY_ALIAS: counterparty_alias,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT: attachment,
            cls.FIELD_MERCHANT_REFERENCE: merchant_reference,
            cls.FIELD_STATUS: status,
            cls.FIELD_MINIMUM_AGE: minimum_age,
            cls.FIELD_REQUIRE_ADDRESS: require_address,
            cls.FIELD_WANT_TIP: want_tip,
            cls.FIELD_ALLOW_AMOUNT_LOWER: allow_amount_lower,
            cls.FIELD_ALLOW_AMOUNT_HIGHER: allow_amount_higher,
            cls.FIELD_ALLOW_BUNQME: allow_bunqme,
            cls.FIELD_REDIRECT_URL: redirect_url,
            cls.FIELD_EVENT_ID: event_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_id, monetary_account_id=None, status=None, custom_headers=None):
        """
        Revoke a request for payment, by updating the status to REVOKED.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :param status: The status of the RequestInquiry. Ignored in POST
        requests but can be used for revoking (cancelling) the RequestInquiry by
        setting REVOKED with a PUT request.
        :type status: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseRequestInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_PUT)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get all payment requests for a user's monetary account. bunqme_share_url
        is always null if the counterparty is a bunq user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestInquiryList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseRequestInquiryList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        Get the details of a specific payment request, including its status.
        bunqme_share_url is always null if the counterparty is a bunq user.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseRequestInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def time_responded(self):
        """
        :rtype: str
        """

        return self._time_responded

    @property
    def time_expiry(self):
        """
        :rtype: str
        """

        return self._time_expiry

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def amount_inquired(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_inquired

    @property
    def amount_responded(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_responded

    @property
    def user_alias_created(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_created

    @property
    def user_alias_revoked(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_revoked

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def merchant_reference(self):
        """
        :rtype: str
        """

        return self._merchant_reference

    @property
    def attachment(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._attachment

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def batch_id(self):
        """
        :rtype: int
        """

        return self._batch_id

    @property
    def scheduled_id(self):
        """
        :rtype: int
        """

        return self._scheduled_id

    @property
    def minimum_age(self):
        """
        :rtype: int
        """

        return self._minimum_age

    @property
    def require_address(self):
        """
        :rtype: str
        """

        return self._require_address

    @property
    def bunqme_share_url(self):
        """
        :rtype: str
        """

        return self._bunqme_share_url

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def address_shipping(self):
        """
        :rtype: object_.Address
        """

        return self._address_shipping

    @property
    def address_billing(self):
        """
        :rtype: object_.Address
        """

        return self._address_billing

    @property
    def geolocation(self):
        """
        :rtype: object_.Geolocation
        """

        return self._geolocation

    @property
    def reference_split_the_bill(self):
        """
        :rtype: object_.RequestReferenceSplitTheBillAnchorObject
        """

        return self._reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._time_responded is not None:
            return False

        if self._time_expiry is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._amount_inquired is not None:
            return False

        if self._amount_responded is not None:
            return False

        if self._user_alias_created is not None:
            return False

        if self._user_alias_revoked is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._description is not None:
            return False

        if self._merchant_reference is not None:
            return False

        if self._attachment is not None:
            return False

        if self._status is not None:
            return False

        if self._batch_id is not None:
            return False

        if self._scheduled_id is not None:
            return False

        if self._minimum_age is not None:
            return False

        if self._require_address is not None:
            return False

        if self._bunqme_share_url is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._address_shipping is not None:
            return False

        if self._address_billing is not None:
            return False

        if self._geolocation is not None:
            return False

        if self._reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RequestInquiry
        """

        return converter.json_to_class(RequestInquiry, json_str)


class RequestResponse(BunqModel):
    """
    A RequestResponse is what a user on the other side of a RequestInquiry gets
    when he is sent one. So a RequestInquiry is the initiator and visible for
    the user that sent it and that wants to receive the money. A RequestResponse
    is what the other side sees, i.e. the user that pays the money to accept the
    request. The content is almost identical.
    
    :param _amount_responded: The Amount the RequestResponse was accepted with.
    :type _amount_responded: object_.Amount
    :param _status: The status of the RequestResponse. Can be ACCEPTED, PENDING,
    REJECTED, REFUND_REQUESTED, REFUNDED or REVOKED.
    :type _status: str
    :param _address_shipping: The shipping address provided by the accepting
    user if an address was requested.
    :type _address_shipping: object_.Address
    :param _address_billing: The billing address provided by the accepting user
    if an address was requested.
    :type _address_billing: object_.Address
    :param _id_: The id of the Request Response.
    :type _id_: int
    :param _created: The timestamp when the Request Response was created.
    :type _created: str
    :param _updated: The timestamp when the Request Response was last updated
    (will be updated when chat messages are received).
    :type _updated: str
    :param _time_responded: The timestamp of when the RequestResponse was
    responded to.
    :type _time_responded: str
    :param _time_expiry: The timestamp of when the RequestResponse expired or
    will expire.
    :type _time_expiry: str
    :param _time_refund_requested: The timestamp of when a refund request for
    the RequestResponse was claimed.
    :type _time_refund_requested: str
    :param _time_refunded: The timestamp of when the RequestResponse was
    refunded.
    :type _time_refunded: str
    :param _user_refund_requested: The label of the user that requested the
    refund.
    :type _user_refund_requested: object_.LabelUser
    :param _monetary_account_id: The id of the MonetaryAccount the
    RequestResponse was received on.
    :type _monetary_account_id: int
    :param _amount_inquired: The requested Amount.
    :type _amount_inquired: object_.Amount
    :param _description: The description for the RequestResponse provided by the
    requesting party. Maximum 9000 characters.
    :type _description: str
    :param _alias: The LabelMonetaryAccount with the public information of the
    MonetaryAccount this RequestResponse was received on.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The LabelMonetaryAccount with the public
    information of the MonetaryAccount that is requesting money with this
    RequestResponse.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _attachment: The Attachments attached to the RequestResponse.
    :type _attachment: list[object_.Attachment]
    :param _minimum_age: The minimum age the user accepting the RequestResponse
    must have.
    :type _minimum_age: int
    :param _require_address: Whether or not an address must be provided on
    accept.
    :type _require_address: str
    :param _geolocation: The Geolocation where the RequestResponse was created.
    :type _geolocation: object_.Geolocation
    :param _type_: The type of the RequestInquiry. Can be DIRECT_DEBIT,
    DIRECT_DEBIT_B2B, IDEAL, SOFORT or INTERNAL.
    :type _type_: str
    :param _sub_type: The subtype of the RequestInquiry. Can be ONCE or
    RECURRING for DIRECT_DEBIT RequestInquiries and NONE for all other.
    :type _sub_type: str
    :param _redirect_url: The URL which the user is sent to after accepting or
    rejecting the Request.
    :type _redirect_url: str
    :param _credit_scheme_identifier: The credit scheme id provided by the
    counterparty for DIRECT_DEBIT inquiries.
    :type _credit_scheme_identifier: str
    :param _mandate_identifier: The mandate id provided by the counterparty for
    DIRECT_DEBIT inquiries.
    :type _mandate_identifier: str
    :param _eligible_whitelist_id: The whitelist id for this action or null.
    :type _eligible_whitelist_id: int
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-response/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-response"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-response/{}"

    # Field constants.
    FIELD_AMOUNT_RESPONDED = "amount_responded"
    FIELD_STATUS = "status"
    FIELD_ADDRESS_SHIPPING = "address_shipping"
    FIELD_ADDRESS_BILLING = "address_billing"

    # Object type.
    _OBJECT_TYPE_PUT = "RequestResponse"
    _OBJECT_TYPE_GET = "RequestResponse"

    _id_ = None
    _created = None
    _updated = None
    _time_responded = None
    _time_expiry = None
    _time_refund_requested = None
    _time_refunded = None
    _user_refund_requested = None
    _monetary_account_id = None
    _amount_inquired = None
    _amount_responded = None
    _status = None
    _description = None
    _alias = None
    _counterparty_alias = None
    _attachment = None
    _minimum_age = None
    _require_address = None
    _geolocation = None
    _type_ = None
    _sub_type = None
    _redirect_url = None
    _address_billing = None
    _address_shipping = None
    _credit_scheme_identifier = None
    _mandate_identifier = None
    _eligible_whitelist_id = None
    _request_reference_split_the_bill = None
    _amount_responded_field_for_request = None
    _status_field_for_request = None
    _address_shipping_field_for_request = None
    _address_billing_field_for_request = None

    def __init__(self, status=None, amount_responded=None, address_shipping=None, address_billing=None):
        """
        :param status: The responding status of the RequestResponse. Can be ACCEPTED
        or REJECTED.
        :type status: str
        :param amount_responded: The Amount the user decides to pay.
        :type amount_responded: object_.Amount
        :param address_shipping: The shipping Address to return to the user who
        created the RequestInquiry. Should only be provided if 'require_address' is
        set to SHIPPING, BILLING_SHIPPING or OPTIONAL.
        :type address_shipping: object_.Address
        :param address_billing: The billing Address to return to the user who
        created the RequestInquiry. Should only be provided if 'require_address' is
        set to BILLING, BILLING_SHIPPING or OPTIONAL.
        :type address_billing: object_.Address
        """

        self._status_field_for_request = status
        self._amount_responded_field_for_request = amount_responded
        self._address_shipping_field_for_request = address_shipping
        self._address_billing_field_for_request = address_billing

    @classmethod
    def update(cls, request_response_id, monetary_account_id=None, amount_responded=None, status=None,
               address_shipping=None, address_billing=None, custom_headers=None):
        """
        Update the status to accept or reject the RequestResponse.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :param amount_responded: The Amount the user decides to pay.
        :type amount_responded: object_.Amount
        :param status: The responding status of the RequestResponse. Can be
        ACCEPTED or REJECTED.
        :type status: str
        :param address_shipping: The shipping Address to return to the user who
        created the RequestInquiry. Should only be provided if 'require_address'
        is set to SHIPPING, BILLING_SHIPPING or OPTIONAL.
        :type address_shipping: object_.Address
        :param address_billing: The billing Address to return to the user who
        created the RequestInquiry. Should only be provided if 'require_address'
        is set to BILLING, BILLING_SHIPPING or OPTIONAL.
        :type address_billing: object_.Address
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_AMOUNT_RESPONDED: amount_responded,
            cls.FIELD_STATUS: status,
            cls.FIELD_ADDRESS_SHIPPING: address_shipping,
            cls.FIELD_ADDRESS_BILLING: address_billing
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseRequestResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_PUT)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Get all RequestResponses for a MonetaryAccount.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestResponseList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseRequestResponseList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_response_id, monetary_account_id=None, custom_headers=None):
        """
        Get the details for a specific existing RequestResponse.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRequestResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseRequestResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def time_responded(self):
        """
        :rtype: str
        """

        return self._time_responded

    @property
    def time_expiry(self):
        """
        :rtype: str
        """

        return self._time_expiry

    @property
    def time_refund_requested(self):
        """
        :rtype: str
        """

        return self._time_refund_requested

    @property
    def time_refunded(self):
        """
        :rtype: str
        """

        return self._time_refunded

    @property
    def user_refund_requested(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_refund_requested

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def amount_inquired(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_inquired

    @property
    def amount_responded(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_responded

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def attachment(self):
        """
        :rtype: list[object_.Attachment]
        """

        return self._attachment

    @property
    def minimum_age(self):
        """
        :rtype: int
        """

        return self._minimum_age

    @property
    def require_address(self):
        """
        :rtype: str
        """

        return self._require_address

    @property
    def geolocation(self):
        """
        :rtype: object_.Geolocation
        """

        return self._geolocation

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def sub_type(self):
        """
        :rtype: str
        """

        return self._sub_type

    @property
    def redirect_url(self):
        """
        :rtype: str
        """

        return self._redirect_url

    @property
    def address_billing(self):
        """
        :rtype: object_.Address
        """

        return self._address_billing

    @property
    def address_shipping(self):
        """
        :rtype: object_.Address
        """

        return self._address_shipping

    @property
    def credit_scheme_identifier(self):
        """
        :rtype: str
        """

        return self._credit_scheme_identifier

    @property
    def mandate_identifier(self):
        """
        :rtype: str
        """

        return self._mandate_identifier

    @property
    def eligible_whitelist_id(self):
        """
        :rtype: int
        """

        return self._eligible_whitelist_id

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._time_responded is not None:
            return False

        if self._time_expiry is not None:
            return False

        if self._time_refund_requested is not None:
            return False

        if self._time_refunded is not None:
            return False

        if self._user_refund_requested is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._amount_inquired is not None:
            return False

        if self._amount_responded is not None:
            return False

        if self._status is not None:
            return False

        if self._description is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._attachment is not None:
            return False

        if self._minimum_age is not None:
            return False

        if self._require_address is not None:
            return False

        if self._geolocation is not None:
            return False

        if self._type_ is not None:
            return False

        if self._sub_type is not None:
            return False

        if self._redirect_url is not None:
            return False

        if self._address_billing is not None:
            return False

        if self._address_shipping is not None:
            return False

        if self._credit_scheme_identifier is not None:
            return False

        if self._mandate_identifier is not None:
            return False

        if self._eligible_whitelist_id is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RequestResponse
        """

        return converter.json_to_class(RequestResponse, json_str)


class TabResultResponse(BunqModel):
    """
    Used to view TabResultResponse objects belonging to a tab. A
    TabResultResponse is an object that holds details on a tab which has been
    paid from the provided monetary account.
    
    :param _tab: The Tab details.
    :type _tab: Tab
    :param _payment: The payment made for the Tab.
    :type _payment: Payment
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/tab-result-response/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/tab-result-response"

    # Object type.
    _OBJECT_TYPE_GET = "TabResultResponse"

    _tab = None
    _payment = None
    _request_reference_split_the_bill = None

    @classmethod
    def get(cls, tab_result_response_id, monetary_account_id=None, custom_headers=None):
        """
        Used to view a single TabResultResponse belonging to a tab.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type tab_result_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabResultResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     tab_result_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTabResultResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        Used to view a list of TabResultResponse objects belonging to a tab.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabResultResponseList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTabResultResponseList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def tab(self):
        """
        :rtype: Tab
        """

        return self._tab

    @property
    def payment(self):
        """
        :rtype: Payment
        """

        return self._payment

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._tab is not None:
            return False

        if self._payment is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabResultResponse
        """

        return converter.json_to_class(TabResultResponse, json_str)


class WhitelistResult(BunqModel):
    """
    Whitelist an SDD so that when one comes in, it is automatically accepted.
    
    :param _id_: The ID of the whitelist entry.
    :type _id_: int
    :param _monetary_account_paying_id: The account from which payments will be
    deducted when a transaction is matched with this whitelist.
    :type _monetary_account_paying_id: int
    :param _status: The status of the WhitelistResult.
    :type _status: str
    :param _sub_status: The subStatus of the WhitelistResult.
    :type _sub_status: str
    :param _error_message: The message when the whitelist result has failed due
    to user error.
    :type _error_message: list[object_.Error]
    :param _whitelist: The corresponding whitelist.
    :type _whitelist: Whitelist
    :param _object_: The details of the external object the event was created
    for.
    :type _object_: object_.WhitelistResultViewAnchoredObject
    :param _request_reference_split_the_bill: The reference to the object used
    for split the bill. Can be RequestInquiry or RequestInquiryBatch
    :type _request_reference_split_the_bill:
    list[object_.RequestInquiryReference]
    """

    _id_ = None
    _monetary_account_paying_id = None
    _status = None
    _sub_status = None
    _error_message = None
    _whitelist = None
    _object_ = None
    _request_reference_split_the_bill = None

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def monetary_account_paying_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_paying_id

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def error_message(self):
        """
        :rtype: list[object_.Error]
        """

        return self._error_message

    @property
    def whitelist(self):
        """
        :rtype: Whitelist
        """

        return self._whitelist

    @property
    def object_(self):
        """
        :rtype: object_.WhitelistResultViewAnchoredObject
        """

        return self._object_

    @property
    def request_reference_split_the_bill(self):
        """
        :rtype: list[object_.RequestInquiryReference]
        """

        return self._request_reference_split_the_bill

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._monetary_account_paying_id is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._error_message is not None:
            return False

        if self._whitelist is not None:
            return False

        if self._object_ is not None:
            return False

        if self._request_reference_split_the_bill is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: WhitelistResult
        """

        return converter.json_to_class(WhitelistResult, json_str)


class Whitelist(BunqModel):
    """
    Whitelist a Request so that when one comes in, it is automatically accepted.
    """

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Whitelist
        """

        return converter.json_to_class(Whitelist, json_str)


class TransferwiseTransfer(BunqModel):
    """
    Used to create Transferwise payments.
    
    :param _monetary_account_id: The id of the monetary account the payment
    should be made from.
    :type _monetary_account_id: str
    :param _recipient_id: The id of the target account.
    :type _recipient_id: str
    :param _alias: The LabelMonetaryAccount containing the public information of
    'this' (party) side of the Payment.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The LabelMonetaryAccount containing the public
    information of the other (counterparty) side of the Payment.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _status: The status.
    :type _status: str
    :param _sub_status: The subStatus.
    :type _sub_status: str
    :param _status_transferwise: The status as Transferwise reports it.
    :type _status_transferwise: str
    :param _status_transferwise_issue: A status to indicatie if Transferwise has
    an issue with this payment and requires more information.
    :type _status_transferwise_issue: str
    :param _amount_source: The source amount.
    :type _amount_source: object_.Amount
    :param _amount_target: The target amount.
    :type _amount_target: object_.Amount
    :param _rate: The rate of the payment.
    :type _rate: str
    :param _reference: The reference of the payment.
    :type _reference: str
    :param _pay_in_reference: The Pay-In reference of the payment.
    :type _pay_in_reference: str
    :param _time_delivery_estimate: The estimated delivery time.
    :type _time_delivery_estimate: str
    :param _quote: The quote details used to created the payment.
    :type _quote: TransferwiseQuote
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/transferwise-quote/{}/transferwise-transfer"
    _ENDPOINT_URL_READ = "user/{}/transferwise-quote/{}/transferwise-transfer/{}"
    _ENDPOINT_URL_LISTING = "user/{}/transferwise-quote/{}/transferwise-transfer"

    # Field constants.
    FIELD_MONETARY_ACCOUNT_ID = "monetary_account_id"
    FIELD_RECIPIENT_ID = "recipient_id"

    # Object type.
    _OBJECT_TYPE_GET = "TransferwisePayment"

    _alias = None
    _counterparty_alias = None
    _status = None
    _sub_status = None
    _status_transferwise = None
    _status_transferwise_issue = None
    _amount_source = None
    _amount_target = None
    _rate = None
    _reference = None
    _pay_in_reference = None
    _time_delivery_estimate = None
    _quote = None
    _monetary_account_id_field_for_request = None
    _recipient_id_field_for_request = None

    def __init__(self, monetary_account_id, recipient_id):
        """
        :param monetary_account_id: The id of the monetary account the payment
        should be made from.
        :type monetary_account_id: str
        :param recipient_id: The id of the target account.
        :type recipient_id: str
        """

        self._monetary_account_id_field_for_request = monetary_account_id
        self._recipient_id_field_for_request = recipient_id

    @classmethod
    def create(cls, transferwise_quote_id, monetary_account_id, recipient_id, custom_headers=None):
        """
        :type user_id: int
        :type transferwise_quote_id: int
        :param monetary_account_id: The id of the monetary account the payment
        should be made from.
        :type monetary_account_id: str
        :param recipient_id: The id of the target account.
        :type recipient_id: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_MONETARY_ACCOUNT_ID: monetary_account_id,
            cls.FIELD_RECIPIENT_ID: recipient_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(), transferwise_quote_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, transferwise_quote_id, transferwise_transfer_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type transferwise_quote_id: int
        :type transferwise_transfer_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTransferwiseTransfer
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), transferwise_quote_id,
                                                     transferwise_transfer_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTransferwiseTransfer.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, transferwise_quote_id, params=None, custom_headers=None):
        """
        :type user_id: int
        :type transferwise_quote_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTransferwiseTransferList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), transferwise_quote_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTransferwiseTransferList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def status_transferwise(self):
        """
        :rtype: str
        """

        return self._status_transferwise

    @property
    def status_transferwise_issue(self):
        """
        :rtype: str
        """

        return self._status_transferwise_issue

    @property
    def amount_source(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_source

    @property
    def amount_target(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_target

    @property
    def rate(self):
        """
        :rtype: str
        """

        return self._rate

    @property
    def reference(self):
        """
        :rtype: str
        """

        return self._reference

    @property
    def pay_in_reference(self):
        """
        :rtype: str
        """

        return self._pay_in_reference

    @property
    def time_delivery_estimate(self):
        """
        :rtype: str
        """

        return self._time_delivery_estimate

    @property
    def quote(self):
        """
        :rtype: TransferwiseQuote
        """

        return self._quote

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._status_transferwise is not None:
            return False

        if self._status_transferwise_issue is not None:
            return False

        if self._amount_source is not None:
            return False

        if self._amount_target is not None:
            return False

        if self._rate is not None:
            return False

        if self._reference is not None:
            return False

        if self._pay_in_reference is not None:
            return False

        if self._time_delivery_estimate is not None:
            return False

        if self._quote is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TransferwiseTransfer
        """

        return converter.json_to_class(TransferwiseTransfer, json_str)


class TransferwiseQuote(BunqModel):
    """
    Used to get quotes from Transferwise. These can be used to initiate
    payments.
    
    :param _currency_source: The source currency.
    :type _currency_source: str
    :param _currency_target: The target currency.
    :type _currency_target: str
    :param _amount_source: The source amount.
    :type _amount_source: object_.Amount
    :param _amount_target: The target amount.
    :type _amount_target: object_.Amount
    :param _id_: The id of the quote.
    :type _id_: int
    :param _created: The timestamp of the quote's creation.
    :type _created: str
    :param _updated: The timestamp of the quote's last update.
    :type _updated: str
    :param _time_expiry: The expiration timestamp of the quote.
    :type _time_expiry: str
    :param _quote_id: The quote id Transferwise needs.
    :type _quote_id: str
    :param _amount_fee: The fee amount.
    :type _amount_fee: object_.Amount
    :param _rate: The rate.
    :type _rate: str
    :param _time_delivery_estimate: The estimated delivery time.
    :type _time_delivery_estimate: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/transferwise-quote"
    _ENDPOINT_URL_READ = "user/{}/transferwise-quote/{}"

    # Field constants.
    FIELD_CURRENCY_SOURCE = "currency_source"
    FIELD_CURRENCY_TARGET = "currency_target"
    FIELD_AMOUNT_SOURCE = "amount_source"
    FIELD_AMOUNT_TARGET = "amount_target"

    # Object type.
    _OBJECT_TYPE_GET = "TransferwiseQuote"

    _id_ = None
    _created = None
    _updated = None
    _time_expiry = None
    _quote_id = None
    _amount_source = None
    _amount_target = None
    _amount_fee = None
    _rate = None
    _time_delivery_estimate = None
    _currency_source_field_for_request = None
    _currency_target_field_for_request = None
    _amount_source_field_for_request = None
    _amount_target_field_for_request = None

    def __init__(self, currency_source, currency_target, amount_source=None, amount_target=None):
        """
        :param currency_source: The source currency.
        :type currency_source: str
        :param currency_target: The target currency.
        :type currency_target: str
        :param amount_source: The source amount. Required if target amount is left
        empty.
        :type amount_source: object_.Amount
        :param amount_target: The target amount. Required if source amount is left
        empty.
        :type amount_target: object_.Amount
        """

        self._currency_source_field_for_request = currency_source
        self._currency_target_field_for_request = currency_target
        self._amount_source_field_for_request = amount_source
        self._amount_target_field_for_request = amount_target

    @classmethod
    def create(cls, currency_source, currency_target, amount_source=None, amount_target=None, custom_headers=None):
        """
        :type user_id: int
        :param currency_source: The source currency.
        :type currency_source: str
        :param currency_target: The target currency.
        :type currency_target: str
        :param amount_source: The source amount. Required if target amount is
        left empty.
        :type amount_source: object_.Amount
        :param amount_target: The target amount. Required if source amount is
        left empty.
        :type amount_target: object_.Amount
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CURRENCY_SOURCE: currency_source,
            cls.FIELD_CURRENCY_TARGET: currency_target,
            cls.FIELD_AMOUNT_SOURCE: amount_source,
            cls.FIELD_AMOUNT_TARGET: amount_target
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, transferwise_quote_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type transferwise_quote_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTransferwiseQuote
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), transferwise_quote_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTransferwiseQuote.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def time_expiry(self):
        """
        :rtype: str
        """

        return self._time_expiry

    @property
    def quote_id(self):
        """
        :rtype: str
        """

        return self._quote_id

    @property
    def amount_source(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_source

    @property
    def amount_target(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_target

    @property
    def amount_fee(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_fee

    @property
    def rate(self):
        """
        :rtype: str
        """

        return self._rate

    @property
    def time_delivery_estimate(self):
        """
        :rtype: str
        """

        return self._time_delivery_estimate

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._time_expiry is not None:
            return False

        if self._quote_id is not None:
            return False

        if self._amount_source is not None:
            return False

        if self._amount_target is not None:
            return False

        if self._amount_fee is not None:
            return False

        if self._rate is not None:
            return False

        if self._time_delivery_estimate is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TransferwiseQuote
        """

        return converter.json_to_class(TransferwiseQuote, json_str)


class RewardRecipient(BunqModel):
    """
    Used to view Rewards.
    
    :param _id_: The id of the reward.
    :type _id_: int
    :param _created: The time the reward was created.
    :type _created: str
    :param _updated: The time the reward was last updated.
    :type _updated: str
    :param _status: The status of the reward.
    :type _status: str
    :param _sub_status: The subStatus of the reward.
    :type _sub_status: str
    :param _type_: The type of the reward.
    :type _type_: str
    :param _counterparty_alias: The alias of the other user eligible for the
    reward award.
    :type _counterparty_alias: object_.LabelUser
    :param _amount_reward: The amount that will be/was awarded as reward for the
    reward.
    :type _amount_reward: object_.Amount
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/reward-recipient/{}"
    _ENDPOINT_URL_LISTING = "user/{}/reward-recipient"

    # Object type.
    _OBJECT_TYPE_GET = "RewardRecipient"

    _id_ = None
    _created = None
    _updated = None
    _status = None
    _sub_status = None
    _type_ = None
    _counterparty_alias = None
    _amount_reward = None

    @classmethod
    def get(cls, reward_recipient_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type reward_recipient_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRewardRecipient
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), reward_recipient_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseRewardRecipient.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRewardRecipientList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseRewardRecipientList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._counterparty_alias

    @property
    def amount_reward(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_reward

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._type_ is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._amount_reward is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RewardRecipient
        """

        return converter.json_to_class(RewardRecipient, json_str)


class RewardSender(BunqModel):
    """
    Used to view Rewards.
    
    :param _id_: The id of the reward.
    :type _id_: int
    :param _created: The time the reward was created.
    :type _created: str
    :param _updated: The time the reward was last updated.
    :type _updated: str
    :param _status: The status of the reward.
    :type _status: str
    :param _sub_status: The subStatus of the reward.
    :type _sub_status: str
    :param _type_: The type of the reward.
    :type _type_: str
    :param _counterparty_alias: The alias of the other user eligible for the
    reward award.
    :type _counterparty_alias: object_.LabelUser
    :param _amount_reward: The amount that will be/was awarded as reward for the
    reward.
    :type _amount_reward: object_.Amount
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/reward-sender/{}"
    _ENDPOINT_URL_LISTING = "user/{}/reward-sender"

    # Object type.
    _OBJECT_TYPE_GET = "RewardSender"

    _id_ = None
    _created = None
    _updated = None
    _status = None
    _sub_status = None
    _type_ = None
    _counterparty_alias = None
    _amount_reward = None

    @classmethod
    def get(cls, reward_sender_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type reward_sender_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRewardSender
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), reward_sender_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseRewardSender.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseRewardSenderList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseRewardSenderList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def type_(self):
        """
        :rtype: str
        """

        return self._type_

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._counterparty_alias

    @property
    def amount_reward(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_reward

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._type_ is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._amount_reward is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: RewardSender
        """

        return converter.json_to_class(RewardSender, json_str)


class ShareInviteBankInquiryBatch(BunqModel):
    """
    Used to share a monetary account with another bunq user, as in the 'Connect'
    feature in the bunq app. Allow the creation of share inquiries that, in the
    same way as request inquiries, can be revoked by the user creating them or
    accepted/rejected by the other party.
    
    :param _share_invite_bank_inquiries: The list of share invite bank inquiries
    that were made.
    :type _share_invite_bank_inquiries: list[ShareInviteMonetaryAccountInquiry]
    :param _alias: The LabelMonetaryAccount containing the public information of
    this share invite inquiry batch.
    :type _alias: object_.MonetaryAccountReference
    """

    _share_invite_bank_inquiries = None
    _alias = None

    @property
    def share_invite_bank_inquiries(self):
        """
        :rtype: list[ShareInviteMonetaryAccountInquiry]
        """

        return self._share_invite_bank_inquiries

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._share_invite_bank_inquiries is not None:
            return False

        if self._alias is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ShareInviteBankInquiryBatch
        """

        return converter.json_to_class(ShareInviteBankInquiryBatch, json_str)


class ShareInviteMonetaryAccountInquiry(BunqModel):
    """
    [DEPRECATED - use /share-invite-monetary-account-response] Used to share a
    monetary account with another bunq user, as in the 'Connect' feature in the
    bunq app. Allow the creation of share inquiries that, in the same way as
    request inquiries, can be revoked by the user creating them or
    accepted/rejected by the other party.
    
    :param _counter_user_alias: The label of the user to share with.
    :type _counter_user_alias: object_.LabelUser
    :param _draft_share_invite_bank_id: The id of the draft share invite bank.
    :type _draft_share_invite_bank_id: int
    :param _share_detail: The share details. Only one of these objects is
    returned.
    :type _share_detail: object_.ShareDetail
    :param _status: The status of the share. Can be PENDING, REVOKED (the user
    deletes the share inquiry before it's accepted), ACCEPTED, CANCELLED (the
    user deletes an active share) or CANCELLATION_PENDING,
    CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual connects)
    :type _status: str
    :param _relationship: The relationship: COMPANY_DIRECTOR, COMPANY_EMPLOYEE,
    etc
    :type _relationship: str
    :param _share_type: The share type, either STANDARD or MUTUAL.
    :type _share_type: str
    :param _start_date: The start date of this share.
    :type _start_date: str
    :param _end_date: The expiration date of this share.
    :type _end_date: str
    :param _alias: The label of the monetary account that's being shared.
    :type _alias: object_.MonetaryAccountReference
    :param _user_alias_created: The user who created the share.
    :type _user_alias_created: object_.LabelUser
    :param _user_alias_revoked: The user who revoked the share.
    :type _user_alias_revoked: object_.LabelUser
    :param _monetary_account_id: The id of the monetary account the share
    applies to.
    :type _monetary_account_id: int
    :param _id_: The id of the newly created share invite.
    :type _id_: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/share-invite-monetary-account-inquiry"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/share-invite-monetary-account-inquiry/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/share-invite-monetary-account-inquiry/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/share-invite-monetary-account-inquiry"

    # Field constants.
    FIELD_COUNTER_USER_ALIAS = "counter_user_alias"
    FIELD_DRAFT_SHARE_INVITE_BANK_ID = "draft_share_invite_bank_id"
    FIELD_SHARE_DETAIL = "share_detail"
    FIELD_STATUS = "status"
    FIELD_RELATIONSHIP = "relationship"
    FIELD_SHARE_TYPE = "share_type"
    FIELD_START_DATE = "start_date"
    FIELD_END_DATE = "end_date"

    # Object type.
    _OBJECT_TYPE_GET = "ShareInviteMonetaryAccountInquiry"

    _alias = None
    _user_alias_created = None
    _user_alias_revoked = None
    _counter_user_alias = None
    _monetary_account_id = None
    _draft_share_invite_bank_id = None
    _share_detail = None
    _status = None
    _relationship = None
    _share_type = None
    _start_date = None
    _end_date = None
    _id_ = None
    _counter_user_alias_field_for_request = None
    _draft_share_invite_bank_id_field_for_request = None
    _share_detail_field_for_request = None
    _status_field_for_request = None
    _relationship_field_for_request = None
    _share_type_field_for_request = None
    _start_date_field_for_request = None
    _end_date_field_for_request = None

    def __init__(self, counter_user_alias, share_detail=None, status=None, draft_share_invite_bank_id=None,
                 relationship=None, share_type=None, start_date=None, end_date=None):
        """
        :param counter_user_alias: The pointer of the user to share with.
        :type counter_user_alias: object_.Pointer
        :param share_detail: The share details. Only one of these objects may be
        passed.
        :type share_detail: object_.ShareDetail
        :param status: The status of the share. Can be PENDING, REVOKED (the user
        deletes the share inquiry before it's accepted), ACCEPTED, CANCELLED (the
        user deletes an active share) or CANCELLATION_PENDING,
        CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual
        connects).
        :type status: str
        :param draft_share_invite_bank_id: The id of the draft share invite bank.
        :type draft_share_invite_bank_id: int
        :param relationship: The relationship: COMPANY_DIRECTOR, COMPANY_EMPLOYEE,
        etc
        :type relationship: str
        :param share_type: The share type, either STANDARD or MUTUAL.
        :type share_type: str
        :param start_date: The start date of this share.
        :type start_date: str
        :param end_date: The expiration date of this share.
        :type end_date: str
        """

        self._counter_user_alias_field_for_request = counter_user_alias
        self._share_detail_field_for_request = share_detail
        self._status_field_for_request = status
        self._draft_share_invite_bank_id_field_for_request = draft_share_invite_bank_id
        self._relationship_field_for_request = relationship
        self._share_type_field_for_request = share_type
        self._start_date_field_for_request = start_date
        self._end_date_field_for_request = end_date

    @classmethod
    def create(cls, counter_user_alias, share_detail, status, monetary_account_id=None, draft_share_invite_bank_id=None,
               relationship=None, share_type=None, start_date=None, end_date=None, custom_headers=None):
        """
        [DEPRECATED - use /share-invite-monetary-account-response] Create a new
        share inquiry for a monetary account, specifying the permission the
        other bunq user will have on it.
        
        :type user_id: int
        :type monetary_account_id: int
        :param counter_user_alias: The pointer of the user to share with.
        :type counter_user_alias: object_.Pointer
        :param share_detail: The share details. Only one of these objects may be
        passed.
        :type share_detail: object_.ShareDetail
        :param status: The status of the share. Can be PENDING, REVOKED (the
        user deletes the share inquiry before it's accepted), ACCEPTED,
        CANCELLED (the user deletes an active share) or CANCELLATION_PENDING,
        CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual
        connects).
        :type status: str
        :param draft_share_invite_bank_id: The id of the draft share invite
        bank.
        :type draft_share_invite_bank_id: int
        :param relationship: The relationship: COMPANY_DIRECTOR,
        COMPANY_EMPLOYEE, etc
        :type relationship: str
        :param share_type: The share type, either STANDARD or MUTUAL.
        :type share_type: str
        :param start_date: The start date of this share.
        :type start_date: str
        :param end_date: The expiration date of this share.
        :type end_date: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_COUNTER_USER_ALIAS: counter_user_alias,
            cls.FIELD_DRAFT_SHARE_INVITE_BANK_ID: draft_share_invite_bank_id,
            cls.FIELD_SHARE_DETAIL: share_detail,
            cls.FIELD_STATUS: status,
            cls.FIELD_RELATIONSHIP: relationship,
            cls.FIELD_SHARE_TYPE: share_type,
            cls.FIELD_START_DATE: start_date,
            cls.FIELD_END_DATE: end_date
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, share_invite_monetary_account_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        [DEPRECATED - use /share-invite-monetary-account-response] Get the
        details of a specific share inquiry.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type share_invite_monetary_account_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseShareInviteMonetaryAccountInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     share_invite_monetary_account_inquiry_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseShareInviteMonetaryAccountInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, share_invite_monetary_account_inquiry_id, monetary_account_id=None, share_detail=None, status=None,
               start_date=None, end_date=None, custom_headers=None):
        """
        [DEPRECATED - use /share-invite-monetary-account-response] Update the
        details of a share. This includes updating status (revoking or
        cancelling it), granted permission and validity period of this share.
        
        :type user_id: int
        :type monetary_account_id: int
        :type share_invite_monetary_account_inquiry_id: int
        :param share_detail: The share details. Only one of these objects may be
        passed.
        :type share_detail: object_.ShareDetail
        :param status: The status of the share. Can be PENDING, REVOKED (the
        user deletes the share inquiry before it's accepted), ACCEPTED,
        CANCELLED (the user deletes an active share) or CANCELLATION_PENDING,
        CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual
        connects).
        :type status: str
        :param start_date: The start date of this share.
        :type start_date: str
        :param end_date: The expiration date of this share.
        :type end_date: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_SHARE_DETAIL: share_detail,
            cls.FIELD_STATUS: status,
            cls.FIELD_START_DATE: start_date,
            cls.FIELD_END_DATE: end_date
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       share_invite_monetary_account_inquiry_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        [DEPRECATED - use /share-invite-monetary-account-response] Get a list
        with all the share inquiries for a monetary account, only if the
        requesting user has permission to change the details of the various
        ones.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseShareInviteMonetaryAccountInquiryList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseShareInviteMonetaryAccountInquiryList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def user_alias_created(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_created

    @property
    def user_alias_revoked(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_revoked

    @property
    def counter_user_alias(self):
        """
        :rtype: object_.LabelUser
        """

        return self._counter_user_alias

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def draft_share_invite_bank_id(self):
        """
        :rtype: int
        """

        return self._draft_share_invite_bank_id

    @property
    def share_detail(self):
        """
        :rtype: object_.ShareDetail
        """

        return self._share_detail

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def relationship(self):
        """
        :rtype: str
        """

        return self._relationship

    @property
    def share_type(self):
        """
        :rtype: str
        """

        return self._share_type

    @property
    def start_date(self):
        """
        :rtype: str
        """

        return self._start_date

    @property
    def end_date(self):
        """
        :rtype: str
        """

        return self._end_date

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._alias is not None:
            return False

        if self._user_alias_created is not None:
            return False

        if self._user_alias_revoked is not None:
            return False

        if self._counter_user_alias is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._draft_share_invite_bank_id is not None:
            return False

        if self._share_detail is not None:
            return False

        if self._status is not None:
            return False

        if self._relationship is not None:
            return False

        if self._share_type is not None:
            return False

        if self._start_date is not None:
            return False

        if self._end_date is not None:
            return False

        if self._id_ is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ShareInviteMonetaryAccountInquiry
        """

        return converter.json_to_class(ShareInviteMonetaryAccountInquiry, json_str)


class ShareInviteMonetaryAccountResponse(BunqModel):
    """
    Used to view or respond to shares a user was invited to. See
    'share-invite-bank-inquiry' for more information about the inquiring
    endpoint.
    
    :param _status: The status of the share. Can be PENDING, REVOKED (the user
    deletes the share inquiry before it's accepted), ACCEPTED, CANCELLED (the
    user deletes an active share) or CANCELLATION_PENDING,
    CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual connects)
    :type _status: str
    :param _card_id: The card to link to the shared monetary account. Used only
    if share_detail is ShareDetailCardPayment.
    :type _card_id: int
    :param _id_: The id of the ShareInviteBankResponse.
    :type _id_: int
    :param _created: The timestamp of the ShareInviteBankResponse creation.
    :type _created: str
    :param _updated: The timestamp of the ShareInviteBankResponse last update.
    :type _updated: str
    :param _counter_alias: The monetary account and user who created the share.
    :type _counter_alias: object_.MonetaryAccountReference
    :param _user_alias_cancelled: The user who cancelled the share if it has
    been revoked or rejected.
    :type _user_alias_cancelled: object_.LabelUser
    :param _monetary_account_id: The id of the monetary account the ACCEPTED
    share applies to. null otherwise.
    :type _monetary_account_id: int
    :param _draft_share_invite_bank_id: The id of the draft share invite bank.
    :type _draft_share_invite_bank_id: int
    :param _share_detail: The share details.
    :type _share_detail: object_.ShareDetail
    :param _relation_user: All of the relation users towards this MA.
    :type _relation_user: RelationUser
    :param _share_type: The share type, either STANDARD or MUTUAL.
    :type _share_type: str
    :param _start_date: The start date of this share.
    :type _start_date: str
    :param _end_date: The expiration date of this share.
    :type _end_date: str
    :param _description: The description of this share. It is basically the
    monetary account description.
    :type _description: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/share-invite-monetary-account-response/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/share-invite-monetary-account-response/{}"
    _ENDPOINT_URL_LISTING = "user/{}/share-invite-monetary-account-response"

    # Field constants.
    FIELD_STATUS = "status"
    FIELD_CARD_ID = "card_id"

    # Object type.
    _OBJECT_TYPE_GET = "ShareInviteMonetaryAccountResponse"

    _id_ = None
    _created = None
    _updated = None
    _counter_alias = None
    _user_alias_cancelled = None
    _monetary_account_id = None
    _draft_share_invite_bank_id = None
    _share_detail = None
    _status = None
    _relation_user = None
    _share_type = None
    _start_date = None
    _end_date = None
    _description = None
    _status_field_for_request = None
    _card_id_field_for_request = None

    def __init__(self, status=None, card_id=None):
        """
        :param status: The status of the share. Can be PENDING, REVOKED (the user
        deletes the share inquiry before it's accepted), ACCEPTED, CANCELLED (the
        user deletes an active share) or CANCELLATION_PENDING,
        CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual connects)
        :type status: str
        :param card_id: The card to link to the shared monetary account. Used only
        if share_detail is ShareDetailCardPayment.
        :type card_id: int
        """

        self._status_field_for_request = status
        self._card_id_field_for_request = card_id

    @classmethod
    def get(cls, share_invite_monetary_account_response_id, custom_headers=None):
        """
        Return the details of a specific share a user was invited to.
        
        :type api_context: ApiContext
        :type user_id: int
        :type share_invite_monetary_account_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseShareInviteMonetaryAccountResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     share_invite_monetary_account_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseShareInviteMonetaryAccountResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, share_invite_monetary_account_response_id, status=None, card_id=None, custom_headers=None):
        """
        Accept or reject a share a user was invited to.
        
        :type user_id: int
        :type share_invite_monetary_account_response_id: int
        :param status: The status of the share. Can be PENDING, REVOKED (the
        user deletes the share inquiry before it's accepted), ACCEPTED,
        CANCELLED (the user deletes an active share) or CANCELLATION_PENDING,
        CANCELLATION_ACCEPTED, CANCELLATION_REJECTED (for canceling mutual
        connects)
        :type status: str
        :param card_id: The card to link to the shared monetary account. Used
        only if share_detail is ShareDetailCardPayment.
        :type card_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_STATUS: status,
            cls.FIELD_CARD_ID: card_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       share_invite_monetary_account_response_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Return all the shares a user was invited to.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseShareInviteMonetaryAccountResponseList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseShareInviteMonetaryAccountResponseList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def counter_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counter_alias

    @property
    def user_alias_cancelled(self):
        """
        :rtype: object_.LabelUser
        """

        return self._user_alias_cancelled

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def draft_share_invite_bank_id(self):
        """
        :rtype: int
        """

        return self._draft_share_invite_bank_id

    @property
    def share_detail(self):
        """
        :rtype: object_.ShareDetail
        """

        return self._share_detail

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def relation_user(self):
        """
        :rtype: RelationUser
        """

        return self._relation_user

    @property
    def share_type(self):
        """
        :rtype: str
        """

        return self._share_type

    @property
    def start_date(self):
        """
        :rtype: str
        """

        return self._start_date

    @property
    def end_date(self):
        """
        :rtype: str
        """

        return self._end_date

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._counter_alias is not None:
            return False

        if self._user_alias_cancelled is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._draft_share_invite_bank_id is not None:
            return False

        if self._share_detail is not None:
            return False

        if self._status is not None:
            return False

        if self._relation_user is not None:
            return False

        if self._share_type is not None:
            return False

        if self._start_date is not None:
            return False

        if self._end_date is not None:
            return False

        if self._description is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ShareInviteMonetaryAccountResponse
        """

        return converter.json_to_class(ShareInviteMonetaryAccountResponse, json_str)


class SofortMerchantTransaction(BunqModel):
    """
    View for requesting Sofort transactions and polling their status.
    
    :param _amount_requested: The requested amount of money to add.
    :type _amount_requested: object_.Amount
    :param _issuer: The BIC of the issuer.
    :type _issuer: str
    :param _monetary_account_id: The id of the monetary account this sofort
    merchant transaction links to.
    :type _monetary_account_id: int
    :param _alias: The alias of the monetary account to add money to.
    :type _alias: object_.MonetaryAccountReference
    :param _counterparty_alias: The alias of the monetary account the money
    comes from.
    :type _counterparty_alias: object_.MonetaryAccountReference
    :param _amount_guaranteed: In case of a successful transaction, the amount
    of money that will be transferred.
    :type _amount_guaranteed: object_.Amount
    :param _issuer_authentication_url: The URL to visit to 
    :type _issuer_authentication_url: str
    :param _status: The status of the transaction.
    :type _status: str
    :param _error_message: The error message of the transaction.
    :type _error_message: list[object_.Error]
    :param _transaction_identifier: The 'transaction ID' of the Sofort
    transaction.
    :type _transaction_identifier: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/sofort-merchant-transaction"

    # Field constants.
    FIELD_AMOUNT_REQUESTED = "amount_requested"
    FIELD_ISSUER = "issuer"

    # Object type.
    _OBJECT_TYPE_GET = "SofortMerchantTransaction"

    _monetary_account_id = None
    _alias = None
    _counterparty_alias = None
    _amount_guaranteed = None
    _amount_requested = None
    _issuer = None
    _issuer_authentication_url = None
    _status = None
    _error_message = None
    _transaction_identifier = None
    _amount_requested_field_for_request = None
    _issuer_field_for_request = None

    def __init__(self, amount_requested, issuer=None):
        """
        :param amount_requested: The requested amount of money to add.
        :type amount_requested: object_.Amount
        :param issuer: The BIC of the issuing bank to ask for money.
        :type issuer: str
        """

        self._amount_requested_field_for_request = amount_requested
        self._issuer_field_for_request = issuer

    @classmethod
    def get(cls, sofort_merchant_transaction_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSofortMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     sofort_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseSofortMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseSofortMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseSofortMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def monetary_account_id(self):
        """
        :rtype: int
        """

        return self._monetary_account_id

    @property
    def alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias

    @property
    def counterparty_alias(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._counterparty_alias

    @property
    def amount_guaranteed(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_guaranteed

    @property
    def amount_requested(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_requested

    @property
    def issuer(self):
        """
        :rtype: str
        """

        return self._issuer

    @property
    def issuer_authentication_url(self):
        """
        :rtype: str
        """

        return self._issuer_authentication_url

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def error_message(self):
        """
        :rtype: list[object_.Error]
        """

        return self._error_message

    @property
    def transaction_identifier(self):
        """
        :rtype: str
        """

        return self._transaction_identifier

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._monetary_account_id is not None:
            return False

        if self._alias is not None:
            return False

        if self._counterparty_alias is not None:
            return False

        if self._amount_guaranteed is not None:
            return False

        if self._amount_requested is not None:
            return False

        if self._issuer is not None:
            return False

        if self._issuer_authentication_url is not None:
            return False

        if self._status is not None:
            return False

        if self._error_message is not None:
            return False

        if self._transaction_identifier is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: SofortMerchantTransaction
        """

        return converter.json_to_class(SofortMerchantTransaction, json_str)


class TabResultInquiry(BunqModel):
    """
    Used to view TabResultInquiry objects belonging to a tab. A TabResultInquiry
    is an object that holds details on both the tab and a single payment made
    for that tab.
    
    :param _tab: The Tab details.
    :type _tab: Tab
    :param _payment: The payment made for the Tab.
    :type _payment: Payment
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/cash-register/{}/tab/{}/tab-result-inquiry/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/cash-register/{}/tab/{}/tab-result-inquiry"

    # Object type.
    _OBJECT_TYPE_GET = "TabResultInquiry"

    _tab = None
    _payment = None

    @classmethod
    def get(cls, cash_register_id, tab_uuid, tab_result_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        Used to view a single TabResultInquiry belonging to a tab.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_uuid: str
        :type tab_result_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabResultInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     cash_register_id, tab_uuid, tab_result_inquiry_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseTabResultInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, cash_register_id, tab_uuid, monetary_account_id=None, params=None, custom_headers=None):
        """
        Used to view a list of TabResultInquiry objects belonging to a tab.
        
        :type user_id: int
        :type monetary_account_id: int
        :type cash_register_id: int
        :type tab_uuid: str
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseTabResultInquiryList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        cash_register_id, tab_uuid)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseTabResultInquiryList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def tab(self):
        """
        :rtype: Tab
        """

        return self._tab

    @property
    def payment(self):
        """
        :rtype: Payment
        """

        return self._payment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._tab is not None:
            return False

        if self._payment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: TabResultInquiry
        """

        return converter.json_to_class(TabResultInquiry, json_str)


class ExportAnnualOverviewContent(BunqModel):
    """
    Fetch the raw content of an annual overview. The annual overview is always
    in PDF format. Doc won't display the response of a request to get the
    content of an annual overview.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/export-annual-overview/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "ExportAnnualOverviewContent"

    @classmethod
    def list(cls, export_annual_overview_id, custom_headers=None):
        """
        Used to retrieve the raw content of an annual overview.
        
        :type user_id: int
        :type export_annual_overview_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(), export_annual_overview_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportAnnualOverviewContent
        """

        return converter.json_to_class(ExportAnnualOverviewContent, json_str)


class ExportAnnualOverview(BunqModel):
    """
    Used to create new and read existing annual overviews of all the user's
    monetary accounts. Once created, annual overviews can be downloaded in PDF
    format via the 'export-annual-overview/{id}/content' endpoint.
    
    :param _year: The year for which the overview is.
    :type _year: int
    :param _id_: The id of the annual overview as created on the server.
    :type _id_: int
    :param _created: The timestamp of the annual overview 's creation.
    :type _created: str
    :param _updated: The timestamp of the annual overview 's last update.
    :type _updated: str
    :param _alias_user: The user to which this annual overview belongs.
    :type _alias_user: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/export-annual-overview"
    _ENDPOINT_URL_READ = "user/{}/export-annual-overview/{}"
    _ENDPOINT_URL_DELETE = "user/{}/export-annual-overview/{}"
    _ENDPOINT_URL_LISTING = "user/{}/export-annual-overview"

    # Field constants.
    FIELD_YEAR = "year"

    # Object type.
    _OBJECT_TYPE_GET = "ExportAnnualOverview"

    _id_ = None
    _created = None
    _updated = None
    _year = None
    _alias_user = None
    _year_field_for_request = None

    def __init__(self, year):
        """
        :param year: The year for which the overview is.
        :type year: int
        """

        self._year_field_for_request = year

    @classmethod
    def create(cls, year, custom_headers=None):
        """
        Create a new annual overview for a specific year. An overview can be
        generated only for a past year.
        
        :type user_id: int
        :param year: The year for which the overview is.
        :type year: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_YEAR: year
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, export_annual_overview_id, custom_headers=None):
        """
        Get an annual overview for a user by its id.
        
        :type api_context: ApiContext
        :type user_id: int
        :type export_annual_overview_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportAnnualOverview
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), export_annual_overview_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseExportAnnualOverview.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def delete(cls, export_annual_overview_id, custom_headers=None):
        """
        :type user_id: int
        :type export_annual_overview_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(), export_annual_overview_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        List all the annual overviews for a user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportAnnualOverviewList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseExportAnnualOverviewList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def year(self):
        """
        :rtype: int
        """

        return self._year

    @property
    def alias_user(self):
        """
        :rtype: object_.LabelUser
        """

        return self._alias_user

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._year is not None:
            return False

        if self._alias_user is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportAnnualOverview
        """

        return converter.json_to_class(ExportAnnualOverview, json_str)


class ExportRibContent(BunqModel):
    """
    Fetch the raw content of an RIB. The RIB is always in PDF format.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/export-rib/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "ExportRibContent"

    @classmethod
    def list(cls, export_rib_id, monetary_account_id=None, custom_headers=None):
        """
        Used to retrieve the raw content of an RIB.
        
        :type user_id: int
        :type monetary_account_id: int
        :type export_rib_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        export_rib_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportRibContent
        """

        return converter.json_to_class(ExportRibContent, json_str)


class ExportRib(BunqModel):
    """
    Used to create new and read existing RIBs of a monetary account
    
    :param _id_: The id of the rib as created on the server.
    :type _id_: int
    :param _created: The timestamp of the RIB's creation.
    :type _created: str
    :param _updated: The timestamp of the RIB's last update.
    :type _updated: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/export-rib"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/export-rib/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/export-rib/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/export-rib"

    # Object type.
    _OBJECT_TYPE_GET = "ExportRib"

    _id_ = None
    _created = None
    _updated = None

    @classmethod
    def create(cls, monetary_account_id=None, custom_headers=None):
        """
        Create a new RIB.
        
        :type user_id: int
        :type monetary_account_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {

        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, export_rib_id, monetary_account_id=None, custom_headers=None):
        """
        Get a RIB for a monetary account by its id.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type export_rib_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportRib
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     export_rib_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseExportRib.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def delete(cls, export_rib_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type export_rib_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       export_rib_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        List all the RIBs for a monetary account.
        
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportRibList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseExportRibList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportRib
        """

        return converter.json_to_class(ExportRib, json_str)


class ExportStatementContent(BunqModel):
    """
    Fetch the raw content of a statement export. The returned file format could
    be MT940, CSV or PDF depending on the statement format specified during the
    statement creation. The doc won't display the response of a request to get
    the content of a statement export.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/customer-statement/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "ExportStatementContent"

    @classmethod
    def list(cls, customer_statement_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type customer_statement_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        customer_statement_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportStatementContent
        """

        return converter.json_to_class(ExportStatementContent, json_str)


class ExportStatementPaymentContent(BunqModel):
    """
    Fetch the raw content of a payment statement export.
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/event/{}/statement/{}/content"

    # Object type.
    _OBJECT_TYPE_GET = "ExportStatementPayment"

    @classmethod
    def list(cls, event_id, statement_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type event_id: int
        :type statement_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseBytes
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        event_id, statement_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseBytes.cast_from_bunq_response(
            BunqResponse(response_raw.body_bytes, response_raw.headers)
        )

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportStatementPaymentContent
        """

        return converter.json_to_class(ExportStatementPaymentContent, json_str)


class ExportStatementPayment(BunqModel):
    """
    Used to create a statement export of a single payment.
    
    :param _id_: The id of the single payment statement model.
    :type _id_: int
    :param _created: The timestamp of the statement model's creation.
    :type _created: str
    :param _updated: The timestamp of the statement model's last update.
    :type _updated: str
    :param _status: The status of the export.
    :type _status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/event/{}/statement"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/event/{}/statement/{}"

    # Object type.
    _OBJECT_TYPE_GET = "ExportStatementPayment"

    _id_ = None
    _created = None
    _updated = None
    _status = None

    @classmethod
    def create(cls, event_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type event_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {

        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       event_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, event_id, export_statement_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type event_id: int
        :type export_statement_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportStatementPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id), event_id,
                                                     export_statement_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseExportStatementPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportStatementPayment
        """

        return converter.json_to_class(ExportStatementPayment, json_str)


class ExportStatement(BunqModel):
    """
    Used to create new and read existing statement exports. Statement exports
    can be created in either CSV, MT940 or PDF file format.
    
    :param _statement_format: The format of statement.
    :type _statement_format: str
    :param _date_start: The date from when this statement shows transactions.
    :type _date_start: str
    :param _date_end: The date until which statement shows transactions.
    :type _date_end: str
    :param _regional_format: The regional format of a CSV statement.
    :type _regional_format: str
    :param _include_attachment: Only for PDF exports. Includes attachments to
    mutations in the export, such as scanned receipts.
    :type _include_attachment: bool
    :param _id_: The id of the customer statement model.
    :type _id_: int
    :param _created: The timestamp of the statement model's creation.
    :type _created: str
    :param _updated: The timestamp of the statement model's last update.
    :type _updated: str
    :param _status: The status of the export.
    :type _status: str
    :param _statement_number: MT940 Statement number. Unique per monetary
    account.
    :type _statement_number: int
    :param _alias_monetary_account: The monetary account for which this
    statement was created.
    :type _alias_monetary_account: object_.MonetaryAccountReference
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/customer-statement"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/customer-statement/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/customer-statement"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/customer-statement/{}"

    # Field constants.
    FIELD_STATEMENT_FORMAT = "statement_format"
    FIELD_DATE_START = "date_start"
    FIELD_DATE_END = "date_end"
    FIELD_REGIONAL_FORMAT = "regional_format"
    FIELD_INCLUDE_ATTACHMENT = "include_attachment"

    # Object type.
    _OBJECT_TYPE_GET = "CustomerStatement"

    _id_ = None
    _created = None
    _updated = None
    _date_start = None
    _date_end = None
    _status = None
    _statement_number = None
    _statement_format = None
    _regional_format = None
    _alias_monetary_account = None
    _statement_format_field_for_request = None
    _date_start_field_for_request = None
    _date_end_field_for_request = None
    _regional_format_field_for_request = None
    _include_attachment_field_for_request = None

    def __init__(self, statement_format, date_start, date_end, regional_format=None, include_attachment=None):
        """
        :param statement_format: The format type of statement. Allowed values:
        MT940, CSV, PDF.
        :type statement_format: str
        :param date_start: The start date for making statements.
        :type date_start: str
        :param date_end: The end date for making statements.
        :type date_end: str
        :param regional_format: Required for CSV exports. The regional format of the
        statement, can be UK_US (comma-separated) or EUROPEAN (semicolon-separated).
        :type regional_format: str
        :param include_attachment: Only for PDF exports. Includes attachments to
        mutations in the export, such as scanned receipts.
        :type include_attachment: bool
        """

        self._statement_format_field_for_request = statement_format
        self._date_start_field_for_request = date_start
        self._date_end_field_for_request = date_end
        self._regional_format_field_for_request = regional_format
        self._include_attachment_field_for_request = include_attachment

    @classmethod
    def create(cls, statement_format, date_start, date_end, monetary_account_id=None, regional_format=None,
               include_attachment=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param statement_format: The format type of statement. Allowed values:
        MT940, CSV, PDF.
        :type statement_format: str
        :param date_start: The start date for making statements.
        :type date_start: str
        :param date_end: The end date for making statements.
        :type date_end: str
        :param regional_format: Required for CSV exports. The regional format of
        the statement, can be UK_US (comma-separated) or EUROPEAN
        (semicolon-separated).
        :type regional_format: str
        :param include_attachment: Only for PDF exports. Includes attachments to
        mutations in the export, such as scanned receipts.
        :type include_attachment: bool
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_STATEMENT_FORMAT: statement_format,
            cls.FIELD_DATE_START: date_start,
            cls.FIELD_DATE_END: date_end,
            cls.FIELD_REGIONAL_FORMAT: regional_format,
            cls.FIELD_INCLUDE_ATTACHMENT: include_attachment
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, export_statement_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type export_statement_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportStatement
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     export_statement_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseExportStatement.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseExportStatementList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseExportStatementList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def delete(cls, export_statement_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type export_statement_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       export_statement_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def date_start(self):
        """
        :rtype: str
        """

        return self._date_start

    @property
    def date_end(self):
        """
        :rtype: str
        """

        return self._date_end

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def statement_number(self):
        """
        :rtype: int
        """

        return self._statement_number

    @property
    def statement_format(self):
        """
        :rtype: str
        """

        return self._statement_format

    @property
    def regional_format(self):
        """
        :rtype: str
        """

        return self._regional_format

    @property
    def alias_monetary_account(self):
        """
        :rtype: object_.MonetaryAccountReference
        """

        return self._alias_monetary_account

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._date_start is not None:
            return False

        if self._date_end is not None:
            return False

        if self._status is not None:
            return False

        if self._statement_number is not None:
            return False

        if self._statement_format is not None:
            return False

        if self._regional_format is not None:
            return False

        if self._alias_monetary_account is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ExportStatement
        """

        return converter.json_to_class(ExportStatement, json_str)


class InsightEvent(BunqModel):
    """
    Used to get events based on time and insight category.
    
    :param _id_: The id of the event.
    :type _id_: int
    :param _created: The timestamp of the event's creation.
    :type _created: str
    :param _updated: The timestamp of the event's last update.
    :type _updated: str
    :param _action: The performed action. Can be: CREATE or UPDATE.
    :type _action: str
    :param _user_id: The id of the user the event applied to (if it was a user
    event).
    :type _user_id: str
    :param _monetary_account_id: The id of the monetary account the event
    applied to (if it was a monetary account event).
    :type _monetary_account_id: str
    :param _object_: The details of the external object the event was created
    for.
    :type _object_: object_.EventObject
    :param _status: The event status. Can be: FINALIZED or AWAITING_REPLY. An
    example of FINALIZED event is a payment received event, while an
    AWAITING_REPLY event is a request received event.
    :type _status: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/insights-search"

    # Object type.
    _OBJECT_TYPE_GET = "Event"

    _id_ = None
    _created = None
    _updated = None
    _action = None
    _user_id = None
    _monetary_account_id = None
    _object_ = None
    _status = None

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInsightEventList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseInsightEventList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def action(self):
        """
        :rtype: str
        """

        return self._action

    @property
    def user_id(self):
        """
        :rtype: str
        """

        return self._user_id

    @property
    def monetary_account_id(self):
        """
        :rtype: str
        """

        return self._monetary_account_id

    @property
    def object_(self):
        """
        :rtype: object_.EventObject
        """

        return self._object_

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._action is not None:
            return False

        if self._user_id is not None:
            return False

        if self._monetary_account_id is not None:
            return False

        if self._object_ is not None:
            return False

        if self._status is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: InsightEvent
        """

        return converter.json_to_class(InsightEvent, json_str)


class Insight(BunqModel):
    """
    Used to get insights about transactions between given time range.
    
    :param _category: The category.
    :type _category: str
    :param _category_translated: The translated category.
    :type _category_translated: str
    :param _amount_total: The total amount of the transactions in the category.
    :type _amount_total: object_.Amount
    :param _number_of_transactions: The number of the transactions in the
    category.
    :type _number_of_transactions: float
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "user/{}/insights"

    # Object type.
    _OBJECT_TYPE_GET = "InsightCategory"

    _category = None
    _category_translated = None
    _amount_total = None
    _number_of_transactions = None

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInsightList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseInsightList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def category(self):
        """
        :rtype: str
        """

        return self._category

    @property
    def category_translated(self):
        """
        :rtype: str
        """

        return self._category_translated

    @property
    def amount_total(self):
        """
        :rtype: object_.Amount
        """

        return self._amount_total

    @property
    def number_of_transactions(self):
        """
        :rtype: float
        """

        return self._number_of_transactions

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._category is not None:
            return False

        if self._category_translated is not None:
            return False

        if self._amount_total is not None:
            return False

        if self._number_of_transactions is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: Insight
        """

        return converter.json_to_class(Insight, json_str)


class InstallationServerPublicKey(BunqModel):
    """
    Using /installation/_/server-public-key you can request the ServerPublicKey
    again. This is done by referring to the id of the Installation.
    
    :param _server_public_key: The server's public key for this Installation.
    :type _server_public_key: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_LISTING = "installation/{}/server-public-key"

    # Object type.
    _OBJECT_TYPE_GET = "ServerPublicKey"

    _server_public_key = None

    @classmethod
    def list(cls, installation_id, params=None, custom_headers=None):
        """
        Show the ServerPublicKey for this Installation.
        
        :type installation_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInstallationServerPublicKeyList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(installation_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseInstallationServerPublicKeyList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def server_public_key(self):
        """
        :rtype: str
        """

        return self._server_public_key

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._server_public_key is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: InstallationServerPublicKey
        """

        return converter.json_to_class(InstallationServerPublicKey, json_str)


class MonetaryAccountBank(BunqModel):
    """
    With MonetaryAccountBank you can create a new bank account, retrieve
    information regarding your existing MonetaryAccountBanks and update specific
    fields of an existing MonetaryAccountBank. Examples of fields that can be
    updated are the description, the daily limit and the avatar of the
    account.<br/><br/>Notification filters can be set on a monetary account
    level to receive callbacks. For more information check the <a
    href="/api/1/page/callbacks">dedicated callbacks page</a>.
    
    :param _currency: The currency of the MonetaryAccountBank as an ISO 4217
    formatted currency code.
    :type _currency: str
    :param _description: The description of the MonetaryAccountBank. Defaults to
    'bunq account'.
    :type _description: str
    :param _daily_limit: The daily spending limit Amount of the
    MonetaryAccountBank. Defaults to 1000 EUR. Currency must match the
    MonetaryAccountBank's currency. Limited to 10000 EUR.
    :type _daily_limit: object_.Amount
    :param _avatar_uuid: The UUID of the Avatar of the MonetaryAccountBank.
    :type _avatar_uuid: str
    :param _status: The status of the MonetaryAccountBank. Can be: ACTIVE,
    BLOCKED, CANCELLED or PENDING_REOPEN
    :type _status: str
    :param _sub_status: The sub-status of the MonetaryAccountBank providing
    extra information regarding the status. Will be NONE for ACTIVE or
    PENDING_REOPEN, COMPLETELY or ONLY_ACCEPTING_INCOMING for BLOCKED and
    REDEMPTION_INVOLUNTARY, REDEMPTION_VOLUNTARY or PERMANENT for CANCELLED.
    :type _sub_status: str
    :param _reason: The reason for voluntarily cancelling (closing) the
    MonetaryAccountBank, can only be OTHER.
    :type _reason: str
    :param _reason_description: The optional free-form reason for voluntarily
    cancelling (closing) the MonetaryAccountBank. Can be any user provided
    message.
    :type _reason_description: str
    :param _display_name: The legal name of the user / company using this
    monetary account.
    :type _display_name: str
    :param _setting: The settings of the MonetaryAccountBank.
    :type _setting: object_.MonetaryAccountSetting
    :param _id_: The id of the MonetaryAccountBank.
    :type _id_: int
    :param _created: The timestamp of the MonetaryAccountBank's creation.
    :type _created: str
    :param _updated: The timestamp of the MonetaryAccountBank's last update.
    :type _updated: str
    :param _avatar: The Avatar of the MonetaryAccountBank.
    :type _avatar: object_.Avatar
    :param _overdraft_limit: The maximum Amount the MonetaryAccountBank can be
    'in the red'.
    :type _overdraft_limit: object_.Amount
    :param _balance: The current available balance Amount of the
    MonetaryAccountBank.
    :type _balance: object_.Amount
    :param _alias: The Aliases for the MonetaryAccountBank.
    :type _alias: list[object_.Pointer]
    :param _public_uuid: The MonetaryAccountBank's public UUID.
    :type _public_uuid: str
    :param _user_id: The id of the User who owns the MonetaryAccountBank.
    :type _user_id: int
    :param _monetary_account_profile: The profile of the account.
    :type _monetary_account_profile: MonetaryAccountProfile
    :param _all_auto_save_id: The ids of the AutoSave.
    :type _all_auto_save_id: list[object_.BunqId]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account-bank"
    _ENDPOINT_URL_READ = "user/{}/monetary-account-bank/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account-bank/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account-bank"

    # Field constants.
    FIELD_CURRENCY = "currency"
    FIELD_DESCRIPTION = "description"
    FIELD_DAILY_LIMIT = "daily_limit"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_REASON = "reason"
    FIELD_REASON_DESCRIPTION = "reason_description"
    FIELD_DISPLAY_NAME = "display_name"
    FIELD_SETTING = "setting"

    # Object type.
    _OBJECT_TYPE_GET = "MonetaryAccountBank"

    _id_ = None
    _created = None
    _updated = None
    _avatar = None
    _currency = None
    _description = None
    _daily_limit = None
    _overdraft_limit = None
    _balance = None
    _alias = None
    _public_uuid = None
    _status = None
    _sub_status = None
    _reason = None
    _reason_description = None
    _user_id = None
    _monetary_account_profile = None
    _display_name = None
    _setting = None
    _all_auto_save_id = None
    _currency_field_for_request = None
    _description_field_for_request = None
    _daily_limit_field_for_request = None
    _avatar_uuid_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _reason_field_for_request = None
    _reason_description_field_for_request = None
    _display_name_field_for_request = None
    _setting_field_for_request = None

    def __init__(self, currency, description=None, daily_limit=None, avatar_uuid=None, status=None, sub_status=None,
                 reason=None, reason_description=None, display_name=None, setting=None):
        """
        :param currency: The currency of the MonetaryAccountBank as an ISO 4217
        formatted currency code.
        :type currency: str
        :param description: The description of the MonetaryAccountBank. Defaults to
        'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountBank. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountBank's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountBank.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountBank. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in PUT
        requests to cancel (close) or reopen the MonetaryAccountBank. When updating
        the status and/or sub_status no other fields can be updated in the same
        request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountBank providing extra
        information regarding the status. Should be ignored for POST requests. In
        case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be NONE.
        When updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountBank, can only be OTHER. Should only be specified if updating
        the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountBank. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param display_name: The legal name of the user / company using this
        monetary account.
        :type display_name: str
        :param setting: The settings of the MonetaryAccountBank.
        :type setting: object_.MonetaryAccountSetting
        """

        self._currency_field_for_request = currency
        self._description_field_for_request = description
        self._daily_limit_field_for_request = daily_limit
        self._avatar_uuid_field_for_request = avatar_uuid
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._reason_field_for_request = reason
        self._reason_description_field_for_request = reason_description
        self._display_name_field_for_request = display_name
        self._setting_field_for_request = setting

    @classmethod
    def create(cls, currency, description=None, daily_limit=None, avatar_uuid=None, status=None, sub_status=None,
               reason=None, reason_description=None, display_name=None, setting=None, custom_headers=None):
        """
        Create new MonetaryAccountBank.
        
        :type user_id: int
        :param currency: The currency of the MonetaryAccountBank as an ISO 4217
        formatted currency code.
        :type currency: str
        :param description: The description of the MonetaryAccountBank. Defaults
        to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountBank. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountBank's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountBank.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountBank. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountBank. When
        updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountBank providing
        extra information regarding the status. Should be ignored for POST
        requests. In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountBank, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountBank. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param display_name: The legal name of the user / company using this
        monetary account.
        :type display_name: str
        :param setting: The settings of the MonetaryAccountBank.
        :type setting: object_.MonetaryAccountSetting
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CURRENCY: currency,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_DISPLAY_NAME: display_name,
            cls.FIELD_SETTING: setting
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, monetary_account_bank_id, custom_headers=None):
        """
        Get a specific MonetaryAccountBank.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_bank_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountBank
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), monetary_account_bank_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseMonetaryAccountBank.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, monetary_account_bank_id, description=None, daily_limit=None, avatar_uuid=None, status=None,
               sub_status=None, reason=None, reason_description=None, display_name=None, setting=None,
               custom_headers=None):
        """
        Update a specific existing MonetaryAccountBank.
        
        :type user_id: int
        :type monetary_account_bank_id: int
        :param description: The description of the MonetaryAccountBank. Defaults
        to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountBank. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountBank's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountBank.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountBank. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountBank. When
        updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountBank providing
        extra information regarding the status. Should be ignored for POST
        requests. In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountBank, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountBank. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param display_name: The legal name of the user / company using this
        monetary account.
        :type display_name: str
        :param setting: The settings of the MonetaryAccountBank.
        :type setting: object_.MonetaryAccountSetting
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_DISPLAY_NAME: display_name,
            cls.FIELD_SETTING: setting
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), monetary_account_bank_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Gets a listing of all MonetaryAccountBanks of a given user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountBankList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseMonetaryAccountBankList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def currency(self):
        """
        :rtype: str
        """

        return self._currency

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def daily_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit

    @property
    def overdraft_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._overdraft_limit

    @property
    def balance(self):
        """
        :rtype: object_.Amount
        """

        return self._balance

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def reason(self):
        """
        :rtype: str
        """

        return self._reason

    @property
    def reason_description(self):
        """
        :rtype: str
        """

        return self._reason_description

    @property
    def user_id(self):
        """
        :rtype: int
        """

        return self._user_id

    @property
    def monetary_account_profile(self):
        """
        :rtype: MonetaryAccountProfile
        """

        return self._monetary_account_profile

    @property
    def display_name(self):
        """
        :rtype: str
        """

        return self._display_name

    @property
    def setting(self):
        """
        :rtype: object_.MonetaryAccountSetting
        """

        return self._setting

    @property
    def all_auto_save_id(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._all_auto_save_id

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._avatar is not None:
            return False

        if self._currency is not None:
            return False

        if self._description is not None:
            return False

        if self._daily_limit is not None:
            return False

        if self._overdraft_limit is not None:
            return False

        if self._balance is not None:
            return False

        if self._alias is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._reason is not None:
            return False

        if self._reason_description is not None:
            return False

        if self._user_id is not None:
            return False

        if self._monetary_account_profile is not None:
            return False

        if self._display_name is not None:
            return False

        if self._setting is not None:
            return False

        if self._all_auto_save_id is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccountBank
        """

        return converter.json_to_class(MonetaryAccountBank, json_str)


class MonetaryAccountProfile(BunqModel):
    """
    Used to update and read up monetary account profiles, to keep the balance
    between specific thresholds.
    
    :param _profile_fill: The profile settings for triggering the fill of a
    monetary account.
    :type _profile_fill: object_.MonetaryAccountProfileFill
    :param _profile_drain: The profile settings for moving excesses to a savings
    account
    :type _profile_drain: object_.MonetaryAccountProfileDrain
    """

    # Field constants.
    FIELD_PROFILE_FILL = "profile_fill"
    FIELD_PROFILE_DRAIN = "profile_drain"

    _profile_fill = None
    _profile_drain = None
    _profile_fill_field_for_request = None
    _profile_drain_field_for_request = None

    def __init__(self, profile_fill=None, profile_drain=None):
        """
        :param profile_fill: The profile settings for triggering the fill of a
        monetary account.
        :type profile_fill: object_.MonetaryAccountProfileFill
        :param profile_drain: The profile settings for moving excesses to a savings
        account
        :type profile_drain: object_.MonetaryAccountProfileDrain
        """

        self._profile_fill_field_for_request = profile_fill
        self._profile_drain_field_for_request = profile_drain

    @property
    def profile_fill(self):
        """
        :rtype: object_.MonetaryAccountProfileFill
        """

        return self._profile_fill

    @property
    def profile_drain(self):
        """
        :rtype: object_.MonetaryAccountProfileDrain
        """

        return self._profile_drain

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._profile_fill is not None:
            return False

        if self._profile_drain is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccountProfile
        """

        return converter.json_to_class(MonetaryAccountProfile, json_str)


class MonetaryAccountJoint(BunqModel):
    """
    The endpoint for joint monetary accounts.
    
    :param _currency: The currency of the MonetaryAccountJoint as an ISO 4217
    formatted currency code.
    :type _currency: str
    :param _description: The description of the MonetaryAccountJoint. Defaults
    to 'bunq account'.
    :type _description: str
    :param _daily_limit: The daily spending limit Amount of the
    MonetaryAccountJoint. Defaults to 1000 EUR. Currency must match the
    MonetaryAccountJoint's currency. Limited to 10000 EUR.
    :type _daily_limit: object_.Amount
    :param _overdraft_limit: The maximum Amount the MonetaryAccountJoint can be
    'in the red'.
    :type _overdraft_limit: object_.Amount
    :param _alias: The Aliases for the MonetaryAccountJoint.
    :type _alias: list[object_.Pointer]
    :param _avatar_uuid: The UUID of the Avatar of the MonetaryAccountJoint.
    :type _avatar_uuid: str
    :param _status: The status of the MonetaryAccountJoint. Can be: ACTIVE,
    BLOCKED, CANCELLED or PENDING_REOPEN
    :type _status: str
    :param _sub_status: The sub-status of the MonetaryAccountJoint providing
    extra information regarding the status. Will be NONE for ACTIVE or
    PENDING_REOPEN, COMPLETELY or ONLY_ACCEPTING_INCOMING for BLOCKED and
    REDEMPTION_INVOLUNTARY, REDEMPTION_VOLUNTARY or PERMANENT for CANCELLED.
    :type _sub_status: str
    :param _reason: The reason for voluntarily cancelling (closing) the
    MonetaryAccountJoint, can only be OTHER.
    :type _reason: str
    :param _reason_description: The optional free-form reason for voluntarily
    cancelling (closing) the MonetaryAccountJoint. Can be any user provided
    message.
    :type _reason_description: str
    :param _all_co_owner: The users the account will be joint with.
    :type _all_co_owner: list[object_.CoOwner]
    :param _setting: The settings of the MonetaryAccountJoint.
    :type _setting: object_.MonetaryAccountSetting
    :param _id_: The id of the MonetaryAccountJoint.
    :type _id_: int
    :param _created: The timestamp of the MonetaryAccountJoint's creation.
    :type _created: str
    :param _updated: The timestamp of the MonetaryAccountJoint's last update.
    :type _updated: str
    :param _avatar: The Avatar of the MonetaryAccountJoint.
    :type _avatar: object_.Avatar
    :param _balance: The current available balance Amount of the
    MonetaryAccountJoint.
    :type _balance: object_.Amount
    :param _public_uuid: The MonetaryAccountJoint's public UUID.
    :type _public_uuid: str
    :param _user_id: The id of the User who owns the MonetaryAccountJoint.
    :type _user_id: int
    :param _monetary_account_profile: The profile of the account.
    :type _monetary_account_profile: MonetaryAccountProfile
    :param _all_auto_save_id: The ids of the AutoSave.
    :type _all_auto_save_id: list[object_.BunqId]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account-joint"
    _ENDPOINT_URL_READ = "user/{}/monetary-account-joint/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account-joint/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account-joint"

    # Field constants.
    FIELD_CURRENCY = "currency"
    FIELD_DESCRIPTION = "description"
    FIELD_DAILY_LIMIT = "daily_limit"
    FIELD_OVERDRAFT_LIMIT = "overdraft_limit"
    FIELD_ALIAS = "alias"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_REASON = "reason"
    FIELD_REASON_DESCRIPTION = "reason_description"
    FIELD_ALL_CO_OWNER = "all_co_owner"
    FIELD_SETTING = "setting"

    # Object type.
    _OBJECT_TYPE_GET = "MonetaryAccountJoint"

    _id_ = None
    _created = None
    _updated = None
    _avatar = None
    _currency = None
    _description = None
    _daily_limit = None
    _overdraft_limit = None
    _balance = None
    _alias = None
    _public_uuid = None
    _status = None
    _sub_status = None
    _reason = None
    _reason_description = None
    _all_co_owner = None
    _user_id = None
    _monetary_account_profile = None
    _setting = None
    _all_auto_save_id = None
    _currency_field_for_request = None
    _description_field_for_request = None
    _daily_limit_field_for_request = None
    _overdraft_limit_field_for_request = None
    _alias_field_for_request = None
    _avatar_uuid_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _reason_field_for_request = None
    _reason_description_field_for_request = None
    _all_co_owner_field_for_request = None
    _setting_field_for_request = None

    def __init__(self, currency, all_co_owner, description=None, daily_limit=None, overdraft_limit=None, alias=None,
                 avatar_uuid=None, status=None, sub_status=None, reason=None, reason_description=None, setting=None):
        """
        :param currency: The currency of the MonetaryAccountJoint as an ISO 4217
        formatted currency code.
        :type currency: str
        :param all_co_owner: The users the account will be joint with.
        :type all_co_owner: list[object_.CoOwner]
        :param description: The description of the MonetaryAccountJoint. Defaults to
        'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountJoint. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountJoint's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param overdraft_limit: The maximum Amount the MonetaryAccountJoint can be
        'in the red'. Must be 0 EUR or omitted.
        :type overdraft_limit: object_.Amount
        :param alias: The Aliases to add to MonetaryAccountJoint. Must all be
        confirmed first. Can mostly be ignored.
        :type alias: list[object_.Pointer]
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountJoint.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountJoint. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in PUT
        requests to cancel (close) or reopen the MonetaryAccountJoint. When updating
        the status and/or sub_status no other fields can be updated in the same
        request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountJoint providing
        extra information regarding the status. Should be ignored for POST requests.
        In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be NONE.
        When updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountJoint, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountJoint. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param setting: The settings of the MonetaryAccountJoint.
        :type setting: object_.MonetaryAccountSetting
        """

        self._currency_field_for_request = currency
        self._all_co_owner_field_for_request = all_co_owner
        self._description_field_for_request = description
        self._daily_limit_field_for_request = daily_limit
        self._overdraft_limit_field_for_request = overdraft_limit
        self._alias_field_for_request = alias
        self._avatar_uuid_field_for_request = avatar_uuid
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._reason_field_for_request = reason
        self._reason_description_field_for_request = reason_description
        self._setting_field_for_request = setting

    @classmethod
    def create(cls, currency, all_co_owner, description=None, daily_limit=None, overdraft_limit=None, alias=None,
               avatar_uuid=None, status=None, sub_status=None, reason=None, reason_description=None, setting=None,
               custom_headers=None):
        """
        :type user_id: int
        :param currency: The currency of the MonetaryAccountJoint as an ISO 4217
        formatted currency code.
        :type currency: str
        :param all_co_owner: The users the account will be joint with.
        :type all_co_owner: list[object_.CoOwner]
        :param description: The description of the MonetaryAccountJoint.
        Defaults to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountJoint. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountJoint's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param overdraft_limit: The maximum Amount the MonetaryAccountJoint can
        be 'in the red'. Must be 0 EUR or omitted.
        :type overdraft_limit: object_.Amount
        :param alias: The Aliases to add to MonetaryAccountJoint. Must all be
        confirmed first. Can mostly be ignored.
        :type alias: list[object_.Pointer]
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountJoint.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountJoint. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountJoint. When
        updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountJoint providing
        extra information regarding the status. Should be ignored for POST
        requests. In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountJoint, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountJoint. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param setting: The settings of the MonetaryAccountJoint.
        :type setting: object_.MonetaryAccountSetting
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CURRENCY: currency,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_OVERDRAFT_LIMIT: overdraft_limit,
            cls.FIELD_ALIAS: alias,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_ALL_CO_OWNER: all_co_owner,
            cls.FIELD_SETTING: setting
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, monetary_account_joint_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_joint_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountJoint
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), monetary_account_joint_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseMonetaryAccountJoint.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, monetary_account_joint_id, description=None, daily_limit=None, avatar_uuid=None, status=None,
               sub_status=None, reason=None, reason_description=None, setting=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_joint_id: int
        :param description: The description of the MonetaryAccountJoint.
        Defaults to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountJoint. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountJoint's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountJoint.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountJoint. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountJoint. When
        updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountJoint providing
        extra information regarding the status. Should be ignored for POST
        requests. In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountJoint, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountJoint. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param setting: The settings of the MonetaryAccountJoint.
        :type setting: object_.MonetaryAccountSetting
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_SETTING: setting
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), monetary_account_joint_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountJointList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseMonetaryAccountJointList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def currency(self):
        """
        :rtype: str
        """

        return self._currency

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def daily_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit

    @property
    def overdraft_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._overdraft_limit

    @property
    def balance(self):
        """
        :rtype: object_.Amount
        """

        return self._balance

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def reason(self):
        """
        :rtype: str
        """

        return self._reason

    @property
    def reason_description(self):
        """
        :rtype: str
        """

        return self._reason_description

    @property
    def all_co_owner(self):
        """
        :rtype: list[object_.CoOwner]
        """

        return self._all_co_owner

    @property
    def user_id(self):
        """
        :rtype: int
        """

        return self._user_id

    @property
    def monetary_account_profile(self):
        """
        :rtype: MonetaryAccountProfile
        """

        return self._monetary_account_profile

    @property
    def setting(self):
        """
        :rtype: object_.MonetaryAccountSetting
        """

        return self._setting

    @property
    def all_auto_save_id(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._all_auto_save_id

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._avatar is not None:
            return False

        if self._currency is not None:
            return False

        if self._description is not None:
            return False

        if self._daily_limit is not None:
            return False

        if self._overdraft_limit is not None:
            return False

        if self._balance is not None:
            return False

        if self._alias is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._reason is not None:
            return False

        if self._reason_description is not None:
            return False

        if self._all_co_owner is not None:
            return False

        if self._user_id is not None:
            return False

        if self._monetary_account_profile is not None:
            return False

        if self._setting is not None:
            return False

        if self._all_auto_save_id is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccountJoint
        """

        return converter.json_to_class(MonetaryAccountJoint, json_str)


class MonetaryAccountSavings(BunqModel):
    """
    With MonetaryAccountSavings you can create a new savings account.
    
    :param _currency: The currency of the MonetaryAccountSavings as an ISO 4217
    formatted currency code.
    :type _currency: str
    :param _description: The description of the MonetaryAccountSavings. Defaults
    to 'bunq account'.
    :type _description: str
    :param _daily_limit: The daily spending limit Amount of the
    MonetaryAccountSavings. Defaults to 1000 EUR. Currency must match the
    MonetaryAccountSavings's currency. Limited to 10000 EUR.
    :type _daily_limit: object_.Amount
    :param _avatar_uuid: The UUID of the Avatar of the MonetaryAccountSavings.
    :type _avatar_uuid: str
    :param _status: The status of the MonetaryAccountSavings. Can be: ACTIVE,
    BLOCKED, CANCELLED or PENDING_REOPEN
    :type _status: str
    :param _sub_status: The sub-status of the MonetaryAccountSavings providing
    extra information regarding the status. Will be NONE for ACTIVE or
    PENDING_REOPEN, COMPLETELY or ONLY_ACCEPTING_INCOMING for BLOCKED and
    REDEMPTION_INVOLUNTARY, REDEMPTION_VOLUNTARY or PERMANENT for CANCELLED.
    :type _sub_status: str
    :param _reason: The reason for voluntarily cancelling (closing) the
    MonetaryAccountSavings, can only be OTHER.
    :type _reason: str
    :param _reason_description: The optional free-form reason for voluntarily
    cancelling (closing) the MonetaryAccountSavings. Can be any user provided
    message.
    :type _reason_description: str
    :param _all_co_owner: The users the account will be joint with.
    :type _all_co_owner: list[object_.CoOwner]
    :param _setting: The settings of the MonetaryAccountSavings.
    :type _setting: object_.MonetaryAccountSetting
    :param _savings_goal: The Savings Goal set for this MonetaryAccountSavings.
    :type _savings_goal: object_.Amount
    :param _id_: The id of the MonetaryAccountSavings.
    :type _id_: int
    :param _created: The timestamp of the MonetaryAccountSavings's creation.
    :type _created: str
    :param _updated: The timestamp of the MonetaryAccountSavings's last update.
    :type _updated: str
    :param _avatar: The Avatar of the MonetaryAccountSavings.
    :type _avatar: object_.Avatar
    :param _overdraft_limit: The maximum Amount the MonetaryAccountSavings can
    be 'in the red'. Must be 0 EUR or omitted.
    :type _overdraft_limit: object_.Amount
    :param _balance: The current available balance Amount of the
    MonetaryAccountSavings.
    :type _balance: object_.Amount
    :param _alias: The Aliases for the MonetaryAccountSavings.
    :type _alias: list[object_.Pointer]
    :param _public_uuid: The MonetaryAccountSavings's public UUID.
    :type _public_uuid: str
    :param _user_id: The id of the User who owns the MonetaryAccountSavings.
    :type _user_id: int
    :param _monetary_account_profile: The profile of the account.
    :type _monetary_account_profile: MonetaryAccountProfile
    :param _savings_goal_progress: The progress in percentages for the Savings
    Goal set for this MonetaryAccountSavings.
    :type _savings_goal_progress: float
    :param _all_auto_save_id: The ids of the AutoSave.
    :type _all_auto_save_id: list[object_.BunqId]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account-savings"
    _ENDPOINT_URL_READ = "user/{}/monetary-account-savings/{}"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account-savings/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account-savings"

    # Field constants.
    FIELD_CURRENCY = "currency"
    FIELD_DESCRIPTION = "description"
    FIELD_DAILY_LIMIT = "daily_limit"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_REASON = "reason"
    FIELD_REASON_DESCRIPTION = "reason_description"
    FIELD_ALL_CO_OWNER = "all_co_owner"
    FIELD_SETTING = "setting"
    FIELD_SAVINGS_GOAL = "savings_goal"

    # Object type.
    _OBJECT_TYPE_GET = "MonetaryAccountSavings"

    _id_ = None
    _created = None
    _updated = None
    _avatar = None
    _currency = None
    _description = None
    _daily_limit = None
    _overdraft_limit = None
    _balance = None
    _alias = None
    _public_uuid = None
    _status = None
    _sub_status = None
    _reason = None
    _reason_description = None
    _all_co_owner = None
    _user_id = None
    _monetary_account_profile = None
    _setting = None
    _savings_goal = None
    _savings_goal_progress = None
    _all_auto_save_id = None
    _currency_field_for_request = None
    _description_field_for_request = None
    _daily_limit_field_for_request = None
    _avatar_uuid_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _reason_field_for_request = None
    _reason_description_field_for_request = None
    _all_co_owner_field_for_request = None
    _setting_field_for_request = None
    _savings_goal_field_for_request = None

    def __init__(self, currency, description=None, daily_limit=None, avatar_uuid=None, status=None, sub_status=None,
                 reason=None, reason_description=None, all_co_owner=None, setting=None, savings_goal=None):
        """
        :param currency: The currency of the MonetaryAccountSavings as an ISO 4217
        formatted currency code.
        :type currency: str
        :param description: The description of the MonetaryAccountSavings. Defaults
        to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountSavings. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountSavings's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountSavings.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountSavings. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in PUT
        requests to cancel (close) or reopen the MonetaryAccountSavings. When
        updating the status and/or sub_status no other fields can be updated in the
        same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountSavings providing
        extra information regarding the status. Should be ignored for POST requests.
        In case of PUT requests with status CANCELLED it can only be
        REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be NONE.
        When updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountSavings, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountSavings. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param all_co_owner: The users the account will be joint with.
        :type all_co_owner: list[object_.CoOwner]
        :param setting: The settings of the MonetaryAccountSavings.
        :type setting: object_.MonetaryAccountSetting
        :param savings_goal: The Savings Goal set for this MonetaryAccountSavings.
        :type savings_goal: object_.Amount
        """

        self._currency_field_for_request = currency
        self._description_field_for_request = description
        self._daily_limit_field_for_request = daily_limit
        self._avatar_uuid_field_for_request = avatar_uuid
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._reason_field_for_request = reason
        self._reason_description_field_for_request = reason_description
        self._all_co_owner_field_for_request = all_co_owner
        self._setting_field_for_request = setting
        self._savings_goal_field_for_request = savings_goal

    @classmethod
    def create(cls, currency, description=None, daily_limit=None, avatar_uuid=None, status=None, sub_status=None,
               reason=None, reason_description=None, all_co_owner=None, setting=None, savings_goal=None,
               custom_headers=None):
        """
        Create new MonetaryAccountSavings.
        
        :type user_id: int
        :param currency: The currency of the MonetaryAccountSavings as an ISO
        4217 formatted currency code.
        :type currency: str
        :param description: The description of the MonetaryAccountSavings.
        Defaults to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountSavings. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountSavings's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the
        MonetaryAccountSavings.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountSavings. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountSavings.
        When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountSavings
        providing extra information regarding the status. Should be ignored for
        POST requests. In case of PUT requests with status CANCELLED it can only
        be REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountSavings, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountSavings. Can be any user
        provided message. Should only be specified if updating the status to
        CANCELLED.
        :type reason_description: str
        :param all_co_owner: The users the account will be joint with.
        :type all_co_owner: list[object_.CoOwner]
        :param setting: The settings of the MonetaryAccountSavings.
        :type setting: object_.MonetaryAccountSetting
        :param savings_goal: The Savings Goal set for this
        MonetaryAccountSavings.
        :type savings_goal: object_.Amount
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CURRENCY: currency,
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_ALL_CO_OWNER: all_co_owner,
            cls.FIELD_SETTING: setting,
            cls.FIELD_SAVINGS_GOAL: savings_goal
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def get(cls, monetary_account_savings_id, custom_headers=None):
        """
        Get a specific MonetaryAccountSavings.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_savings_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountSavings
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(), monetary_account_savings_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseMonetaryAccountSavings.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, monetary_account_savings_id, description=None, daily_limit=None, avatar_uuid=None, status=None,
               sub_status=None, reason=None, reason_description=None, setting=None, savings_goal=None,
               custom_headers=None):
        """
        Update a specific existing MonetaryAccountSavings.
        
        :type user_id: int
        :type monetary_account_savings_id: int
        :param description: The description of the MonetaryAccountSavings.
        Defaults to 'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountSavings. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountSavings's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the
        MonetaryAccountSavings.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountSavings. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in
        PUT requests to cancel (close) or reopen the MonetaryAccountSavings.
        When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountSavings
        providing extra information regarding the status. Should be ignored for
        POST requests. In case of PUT requests with status CANCELLED it can only
        be REDEMPTION_VOLUNTARY, while with status PENDING_REOPEN it can only be
        NONE. When updating the status and/or sub_status no other fields can be
        updated in the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountSavings, can only be OTHER. Should only be specified if
        updating the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountSavings. Can be any user
        provided message. Should only be specified if updating the status to
        CANCELLED.
        :type reason_description: str
        :param setting: The settings of the MonetaryAccountSavings.
        :type setting: object_.MonetaryAccountSetting
        :param savings_goal: The Savings Goal set for this
        MonetaryAccountSavings.
        :type savings_goal: object_.Amount
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_DAILY_LIMIT: daily_limit,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_REASON: reason,
            cls.FIELD_REASON_DESCRIPTION: reason_description,
            cls.FIELD_SETTING: setting,
            cls.FIELD_SAVINGS_GOAL: savings_goal
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(), monetary_account_savings_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Gets a listing of all MonetaryAccountSavingss of a given user.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountSavingsList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseMonetaryAccountSavingsList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def currency(self):
        """
        :rtype: str
        """

        return self._currency

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def daily_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit

    @property
    def overdraft_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._overdraft_limit

    @property
    def balance(self):
        """
        :rtype: object_.Amount
        """

        return self._balance

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def reason(self):
        """
        :rtype: str
        """

        return self._reason

    @property
    def reason_description(self):
        """
        :rtype: str
        """

        return self._reason_description

    @property
    def all_co_owner(self):
        """
        :rtype: list[object_.CoOwner]
        """

        return self._all_co_owner

    @property
    def user_id(self):
        """
        :rtype: int
        """

        return self._user_id

    @property
    def monetary_account_profile(self):
        """
        :rtype: MonetaryAccountProfile
        """

        return self._monetary_account_profile

    @property
    def setting(self):
        """
        :rtype: object_.MonetaryAccountSetting
        """

        return self._setting

    @property
    def savings_goal(self):
        """
        :rtype: object_.Amount
        """

        return self._savings_goal

    @property
    def savings_goal_progress(self):
        """
        :rtype: float
        """

        return self._savings_goal_progress

    @property
    def all_auto_save_id(self):
        """
        :rtype: list[object_.BunqId]
        """

        return self._all_auto_save_id

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._avatar is not None:
            return False

        if self._currency is not None:
            return False

        if self._description is not None:
            return False

        if self._daily_limit is not None:
            return False

        if self._overdraft_limit is not None:
            return False

        if self._balance is not None:
            return False

        if self._alias is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._reason is not None:
            return False

        if self._reason_description is not None:
            return False

        if self._all_co_owner is not None:
            return False

        if self._user_id is not None:
            return False

        if self._monetary_account_profile is not None:
            return False

        if self._setting is not None:
            return False

        if self._savings_goal is not None:
            return False

        if self._savings_goal_progress is not None:
            return False

        if self._all_auto_save_id is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccountSavings
        """

        return converter.json_to_class(MonetaryAccountSavings, json_str)


class MonetaryAccount(BunqModel, AnchorObjectInterface):
    """
    Used to show the MonetaryAccounts that you can access. Currently the only
    MonetaryAccount type is MonetaryAccountBank. See also:
    monetary-account-bank.<br/><br/>Notification filters can be set on a
    monetary account level to receive callbacks. For more information check the
    <a href="/api/2/page/callbacks">dedicated callbacks page</a>.
    
    :param _MonetaryAccountBank: 
    :type _MonetaryAccountBank: MonetaryAccountBank
    :param _MonetaryAccountJoint: 
    :type _MonetaryAccountJoint: MonetaryAccountJoint
    :param _MonetaryAccountLight: 
    :type _MonetaryAccountLight: MonetaryAccountLight
    :param _MonetaryAccountSavings: 
    :type _MonetaryAccountSavings: MonetaryAccountSavings
    """

    # Error constants.
    _ERROR_NULL_FIELDS = "All fields of an extended model or object are null."

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account"

    # Object type.
    _OBJECT_TYPE_GET = "MonetaryAccount"

    _MonetaryAccountBank = None
    _MonetaryAccountJoint = None
    _MonetaryAccountLight = None
    _MonetaryAccountSavings = None

    @classmethod
    def get(cls, monetary_account_id, custom_headers=None):
        """
        Get a specific MonetaryAccount.
        
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccount
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseMonetaryAccount.cast_from_bunq_response(
            cls._from_json(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get a collection of all your MonetaryAccounts.
        
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseMonetaryAccountList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseMonetaryAccountList.cast_from_bunq_response(
            cls._from_json_list(response_raw)
        )

    @property
    def MonetaryAccountBank(self):
        """
        :rtype: MonetaryAccountBank
        """

        return self._MonetaryAccountBank

    @property
    def MonetaryAccountJoint(self):
        """
        :rtype: MonetaryAccountJoint
        """

        return self._MonetaryAccountJoint

    @property
    def MonetaryAccountLight(self):
        """
        :rtype: MonetaryAccountLight
        """

        return self._MonetaryAccountLight

    @property
    def MonetaryAccountSavings(self):
        """
        :rtype: MonetaryAccountSavings
        """

        return self._MonetaryAccountSavings

    def get_referenced_object(self):
        """
        :rtype: BunqModel
        :raise: BunqException
        """

        if self._MonetaryAccountBank is not None:
            return self._MonetaryAccountBank

        if self._MonetaryAccountJoint is not None:
            return self._MonetaryAccountJoint

        if self._MonetaryAccountLight is not None:
            return self._MonetaryAccountLight

        if self._MonetaryAccountSavings is not None:
            return self._MonetaryAccountSavings

        raise BunqException(self._ERROR_NULL_FIELDS)

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._MonetaryAccountBank is not None:
            return False

        if self._MonetaryAccountJoint is not None:
            return False

        if self._MonetaryAccountLight is not None:
            return False

        if self._MonetaryAccountSavings is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccount
        """

        return converter.json_to_class(MonetaryAccount, json_str)


class MonetaryAccountLight(BunqModel):
    """
    With MonetaryAccountLight is a monetary account for bunq light users.
    Through this endpoint you can retrieve information regarding your existing
    MonetaryAccountLights and update specific fields of an existing
    MonetaryAccountLight. Examples of fields that can be updated are the
    description, the daily limit and the avatar of the account.
    
    :param _currency: The currency of the MonetaryAccountLight as an ISO 4217
    formatted currency code.
    :type _currency: str
    :param _description: The description of the MonetaryAccountLight. Defaults
    to 'bunq account'.
    :type _description: str
    :param _daily_limit: The daily spending limit Amount of the
    MonetaryAccountLight. Defaults to 1000 EUR. Currency must match the
    MonetaryAccountLight's currency. Limited to 10000 EUR.
    :type _daily_limit: object_.Amount
    :param _avatar_uuid: The UUID of the Avatar of the MonetaryAccountLight.
    :type _avatar_uuid: str
    :param _status: The status of the MonetaryAccountLight. Can be: ACTIVE,
    BLOCKED, CANCELLED or PENDING_REOPEN
    :type _status: str
    :param _sub_status: The sub-status of the MonetaryAccountLight providing
    extra information regarding the status. Will be NONE for ACTIVE or
    PENDING_REOPEN, COMPLETELY or ONLY_ACCEPTING_INCOMING for BLOCKED and
    REDEMPTION_INVOLUNTARY, REDEMPTION_VOLUNTARY or PERMANENT for CANCELLED.
    :type _sub_status: str
    :param _reason: The reason for voluntarily cancelling (closing) the
    MonetaryAccountBank, can only be OTHER.
    :type _reason: str
    :param _reason_description: The optional free-form reason for voluntarily
    cancelling (closing) the MonetaryAccountBank. Can be any user provided
    message.
    :type _reason_description: str
    :param _setting: The settings of the MonetaryAccountLight.
    :type _setting: object_.MonetaryAccountSetting
    :param _id_: The id of the MonetaryAccountLight.
    :type _id_: int
    :param _created: The timestamp of the MonetaryAccountLight's creation.
    :type _created: str
    :param _updated: The timestamp of the MonetaryAccountLight's last update.
    :type _updated: str
    :param _avatar: The Avatar of the MonetaryAccountLight.
    :type _avatar: object_.Avatar
    :param _balance: The current available balance Amount of the
    MonetaryAccountLight.
    :type _balance: object_.Amount
    :param _alias: The Aliases for the MonetaryAccountLight.
    :type _alias: list[object_.Pointer]
    :param _public_uuid: The MonetaryAccountLight's public UUID.
    :type _public_uuid: str
    :param _user_id: The id of the User who owns the MonetaryAccountLight.
    :type _user_id: int
    :param _balance_maximum: The maximum balance Amount of the
    MonetaryAccountLight.
    :type _balance_maximum: object_.Amount
    :param _budget_month_used: The amount of the monthly budget used.
    :type _budget_month_used: object_.Amount
    :param _budget_month_maximum: The total amount of the monthly budget.
    :type _budget_month_maximum: object_.Amount
    :param _budget_year_used: The amount of the yearly budget used.
    :type _budget_year_used: object_.Amount
    :param _budget_year_maximum: The total amount of the yearly budget.
    :type _budget_year_maximum: object_.Amount
    :param _budget_withdrawal_year_used: The amount of the yearly withdrawal
    budget used.
    :type _budget_withdrawal_year_used: object_.Amount
    :param _budget_withdrawal_year_maximum: The total amount of the yearly
    withdrawal budget.
    :type _budget_withdrawal_year_maximum: object_.Amount
    """

    # Field constants.
    FIELD_CURRENCY = "currency"
    FIELD_DESCRIPTION = "description"
    FIELD_DAILY_LIMIT = "daily_limit"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_REASON = "reason"
    FIELD_REASON_DESCRIPTION = "reason_description"
    FIELD_SETTING = "setting"

    _id_ = None
    _created = None
    _updated = None
    _avatar = None
    _currency = None
    _description = None
    _daily_limit = None
    _balance = None
    _alias = None
    _public_uuid = None
    _status = None
    _sub_status = None
    _reason = None
    _reason_description = None
    _user_id = None
    _balance_maximum = None
    _budget_month_used = None
    _budget_month_maximum = None
    _budget_year_used = None
    _budget_year_maximum = None
    _budget_withdrawal_year_used = None
    _budget_withdrawal_year_maximum = None
    _setting = None
    _currency_field_for_request = None
    _description_field_for_request = None
    _daily_limit_field_for_request = None
    _avatar_uuid_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _reason_field_for_request = None
    _reason_description_field_for_request = None
    _setting_field_for_request = None

    def __init__(self, currency, description=None, daily_limit=None, avatar_uuid=None, status=None, sub_status=None,
                 reason=None, reason_description=None, setting=None):
        """
        :param currency: The currency of the MonetaryAccountLight as an ISO 4217
        formatted currency code.
        :type currency: str
        :param description: The description of the MonetaryAccountLight. Defaults to
        'bunq account'.
        :type description: str
        :param daily_limit: The daily spending limit Amount of the
        MonetaryAccountLight. Defaults to 1000 EUR. Currency must match the
        MonetaryAccountLight's currency. Limited to 10000 EUR.
        :type daily_limit: object_.Amount
        :param avatar_uuid: The UUID of the Avatar of the MonetaryAccountLight.
        :type avatar_uuid: str
        :param status: The status of the MonetaryAccountLight. Ignored in POST
        requests (always set to ACTIVE) can be CANCELLED or PENDING_REOPEN in PUT
        requests to cancel (close) or reopen the MonetaryAccountLight. When updating
        the status and/or sub_status no other fields can be updated in the same
        request (and vice versa).
        :type status: str
        :param sub_status: The sub-status of the MonetaryAccountLight providing
        extra information regarding the status. Should be ignored for POST requests
        and can only be REDEMPTION_VOLUNTARY for PUT requests with status CANCELLED.
        When updating the status and/or sub_status no other fields can be updated in
        the same request (and vice versa).
        :type sub_status: str
        :param reason: The reason for voluntarily cancelling (closing) the
        MonetaryAccountBank, can only be OTHER. Should only be specified if updating
        the status to CANCELLED.
        :type reason: str
        :param reason_description: The optional free-form reason for voluntarily
        cancelling (closing) the MonetaryAccountBank. Can be any user provided
        message. Should only be specified if updating the status to CANCELLED.
        :type reason_description: str
        :param setting: The settings of the MonetaryAccountLight.
        :type setting: object_.MonetaryAccountSetting
        """

        self._currency_field_for_request = currency
        self._description_field_for_request = description
        self._daily_limit_field_for_request = daily_limit
        self._avatar_uuid_field_for_request = avatar_uuid
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._reason_field_for_request = reason
        self._reason_description_field_for_request = reason_description
        self._setting_field_for_request = setting

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def currency(self):
        """
        :rtype: str
        """

        return self._currency

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def daily_limit(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit

    @property
    def balance(self):
        """
        :rtype: object_.Amount
        """

        return self._balance

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def reason(self):
        """
        :rtype: str
        """

        return self._reason

    @property
    def reason_description(self):
        """
        :rtype: str
        """

        return self._reason_description

    @property
    def user_id(self):
        """
        :rtype: int
        """

        return self._user_id

    @property
    def balance_maximum(self):
        """
        :rtype: object_.Amount
        """

        return self._balance_maximum

    @property
    def budget_month_used(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_month_used

    @property
    def budget_month_maximum(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_month_maximum

    @property
    def budget_year_used(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_year_used

    @property
    def budget_year_maximum(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_year_maximum

    @property
    def budget_withdrawal_year_used(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_withdrawal_year_used

    @property
    def budget_withdrawal_year_maximum(self):
        """
        :rtype: object_.Amount
        """

        return self._budget_withdrawal_year_maximum

    @property
    def setting(self):
        """
        :rtype: object_.MonetaryAccountSetting
        """

        return self._setting

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._avatar is not None:
            return False

        if self._currency is not None:
            return False

        if self._description is not None:
            return False

        if self._daily_limit is not None:
            return False

        if self._balance is not None:
            return False

        if self._alias is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._reason is not None:
            return False

        if self._reason_description is not None:
            return False

        if self._user_id is not None:
            return False

        if self._balance_maximum is not None:
            return False

        if self._budget_month_used is not None:
            return False

        if self._budget_month_maximum is not None:
            return False

        if self._budget_year_used is not None:
            return False

        if self._budget_year_maximum is not None:
            return False

        if self._budget_withdrawal_year_used is not None:
            return False

        if self._budget_withdrawal_year_maximum is not None:
            return False

        if self._setting is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: MonetaryAccountLight
        """

        return converter.json_to_class(MonetaryAccountLight, json_str)


class NoteAttachmentBankSwitchServiceNetherlandsIncomingPayment(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/switch-service-payment/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/switch-service-payment/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, switch_service_payment_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, switch_service_payment_id, note_attachment_bank_switch_service_netherlands_incoming_payment_id,
               monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type
        note_attachment_bank_switch_service_netherlands_incoming_payment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id,
                                                       note_attachment_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, switch_service_payment_id, note_attachment_bank_switch_service_netherlands_incoming_payment_id,
               monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type
        note_attachment_bank_switch_service_netherlands_incoming_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id,
                                                       note_attachment_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, switch_service_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype:
        BunqResponseNoteAttachmentBankSwitchServiceNetherlandsIncomingPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        switch_service_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentBankSwitchServiceNetherlandsIncomingPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, switch_service_payment_id, note_attachment_bank_switch_service_netherlands_incoming_payment_id,
            monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type
        note_attachment_bank_switch_service_netherlands_incoming_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype:
        BunqResponseNoteAttachmentBankSwitchServiceNetherlandsIncomingPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     switch_service_payment_id,
                                                     note_attachment_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentBankSwitchServiceNetherlandsIncomingPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentBankSwitchServiceNetherlandsIncomingPayment
        """

        return converter.json_to_class(NoteAttachmentBankSwitchServiceNetherlandsIncomingPayment, json_str)


class NoteTextBankSwitchServiceNetherlandsIncomingPayment(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/switch-service-payment/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/switch-service-payment/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/switch-service-payment/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, switch_service_payment_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, switch_service_payment_id, note_text_bank_switch_service_netherlands_incoming_payment_id,
               monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type note_text_bank_switch_service_netherlands_incoming_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id,
                                                       note_text_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, switch_service_payment_id, note_text_bank_switch_service_netherlands_incoming_payment_id,
               monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type note_text_bank_switch_service_netherlands_incoming_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       switch_service_payment_id,
                                                       note_text_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, switch_service_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype:
        BunqResponseNoteTextBankSwitchServiceNetherlandsIncomingPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        switch_service_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextBankSwitchServiceNetherlandsIncomingPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, switch_service_payment_id, note_text_bank_switch_service_netherlands_incoming_payment_id,
            monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type switch_service_payment_id: int
        :type note_text_bank_switch_service_netherlands_incoming_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextBankSwitchServiceNetherlandsIncomingPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     switch_service_payment_id,
                                                     note_text_bank_switch_service_netherlands_incoming_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextBankSwitchServiceNetherlandsIncomingPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextBankSwitchServiceNetherlandsIncomingPayment
        """

        return converter.json_to_class(NoteTextBankSwitchServiceNetherlandsIncomingPayment, json_str)


class NoteAttachmentBunqMeFundraiserResult(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, bunqme_fundraiser_result_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, bunqme_fundraiser_result_id, note_attachment_bunq_me_fundraiser_result_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_attachment_bunq_me_fundraiser_result_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id,
                                                       note_attachment_bunq_me_fundraiser_result_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, bunqme_fundraiser_result_id, note_attachment_bunq_me_fundraiser_result_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_attachment_bunq_me_fundraiser_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id,
                                                       note_attachment_bunq_me_fundraiser_result_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, bunqme_fundraiser_result_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentBunqMeFundraiserResultList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        bunqme_fundraiser_result_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentBunqMeFundraiserResultList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, bunqme_fundraiser_result_id, note_attachment_bunq_me_fundraiser_result_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_attachment_bunq_me_fundraiser_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentBunqMeFundraiserResult
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bunqme_fundraiser_result_id,
                                                     note_attachment_bunq_me_fundraiser_result_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentBunqMeFundraiserResult.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentBunqMeFundraiserResult
        """

        return converter.json_to_class(NoteAttachmentBunqMeFundraiserResult, json_str)


class NoteTextBunqMeFundraiserResult(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/bunqme-fundraiser-result/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, bunqme_fundraiser_result_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, bunqme_fundraiser_result_id, note_text_bunq_me_fundraiser_result_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_text_bunq_me_fundraiser_result_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id,
                                                       note_text_bunq_me_fundraiser_result_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, bunqme_fundraiser_result_id, note_text_bunq_me_fundraiser_result_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_text_bunq_me_fundraiser_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       bunqme_fundraiser_result_id,
                                                       note_text_bunq_me_fundraiser_result_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, bunqme_fundraiser_result_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextBunqMeFundraiserResultList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        bunqme_fundraiser_result_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextBunqMeFundraiserResultList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, bunqme_fundraiser_result_id, note_text_bunq_me_fundraiser_result_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type bunqme_fundraiser_result_id: int
        :type note_text_bunq_me_fundraiser_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextBunqMeFundraiserResult
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     bunqme_fundraiser_result_id,
                                                     note_text_bunq_me_fundraiser_result_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextBunqMeFundraiserResult.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextBunqMeFundraiserResult
        """

        return converter.json_to_class(NoteTextBunqMeFundraiserResult, json_str)


class NoteAttachmentDraftPayment(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/draft-payment/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/draft-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/draft-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/draft-payment/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/draft-payment/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, draft_payment_id, attachment_id, monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, draft_payment_id, note_attachment_draft_payment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_attachment_draft_payment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id, note_attachment_draft_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, draft_payment_id, note_attachment_draft_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_attachment_draft_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id, note_attachment_draft_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, draft_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentDraftPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        draft_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentDraftPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, draft_payment_id, note_attachment_draft_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_attachment_draft_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentDraftPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     draft_payment_id, note_attachment_draft_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentDraftPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentDraftPayment
        """

        return converter.json_to_class(NoteAttachmentDraftPayment, json_str)


class NoteTextDraftPayment(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/draft-payment/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/draft-payment/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/draft-payment/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/draft-payment/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/draft-payment/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, draft_payment_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, draft_payment_id, note_text_draft_payment_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_text_draft_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id, note_text_draft_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, draft_payment_id, note_text_draft_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_text_draft_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       draft_payment_id, note_text_draft_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, draft_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextDraftPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        draft_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextDraftPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, draft_payment_id, note_text_draft_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type draft_payment_id: int
        :type note_text_draft_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextDraftPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     draft_payment_id, note_text_draft_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextDraftPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextDraftPayment
        """

        return converter.json_to_class(NoteTextDraftPayment, json_str)


class NoteAttachmentIdealMerchantTransaction(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, ideal_merchant_transaction_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, ideal_merchant_transaction_id, note_attachment_ideal_merchant_transaction_id,
               monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_attachment_ideal_merchant_transaction_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id,
                                                       note_attachment_ideal_merchant_transaction_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, ideal_merchant_transaction_id, note_attachment_ideal_merchant_transaction_id,
               monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_attachment_ideal_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id,
                                                       note_attachment_ideal_merchant_transaction_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, ideal_merchant_transaction_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentIdealMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        ideal_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentIdealMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, ideal_merchant_transaction_id, note_attachment_ideal_merchant_transaction_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_attachment_ideal_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentIdealMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     ideal_merchant_transaction_id,
                                                     note_attachment_ideal_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentIdealMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentIdealMerchantTransaction
        """

        return converter.json_to_class(NoteAttachmentIdealMerchantTransaction, json_str)


class NoteTextIdealMerchantTransaction(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/ideal-merchant-transaction/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, ideal_merchant_transaction_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, ideal_merchant_transaction_id, note_text_ideal_merchant_transaction_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_text_ideal_merchant_transaction_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id,
                                                       note_text_ideal_merchant_transaction_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, ideal_merchant_transaction_id, note_text_ideal_merchant_transaction_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_text_ideal_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       ideal_merchant_transaction_id,
                                                       note_text_ideal_merchant_transaction_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, ideal_merchant_transaction_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextIdealMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        ideal_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextIdealMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, ideal_merchant_transaction_id, note_text_ideal_merchant_transaction_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type ideal_merchant_transaction_id: int
        :type note_text_ideal_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextIdealMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     ideal_merchant_transaction_id,
                                                     note_text_ideal_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextIdealMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextIdealMerchantTransaction
        """

        return converter.json_to_class(NoteTextIdealMerchantTransaction, json_str)


class NoteAttachmentMasterCardAction(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/mastercard-action/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/mastercard-action/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/mastercard-action/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/mastercard-action/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/mastercard-action/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, mastercard_action_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, mastercard_action_id, note_attachment_master_card_action_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_attachment_master_card_action_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id, note_attachment_master_card_action_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, mastercard_action_id, note_attachment_master_card_action_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_attachment_master_card_action_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id, note_attachment_master_card_action_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, mastercard_action_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentMasterCardActionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        mastercard_action_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentMasterCardActionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, mastercard_action_id, note_attachment_master_card_action_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_attachment_master_card_action_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentMasterCardAction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     mastercard_action_id, note_attachment_master_card_action_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentMasterCardAction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentMasterCardAction
        """

        return converter.json_to_class(NoteAttachmentMasterCardAction, json_str)


class NoteTextMasterCardAction(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/mastercard-action/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/mastercard-action/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/mastercard-action/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/mastercard-action/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/mastercard-action/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, mastercard_action_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, mastercard_action_id, note_text_master_card_action_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_text_master_card_action_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id, note_text_master_card_action_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, mastercard_action_id, note_text_master_card_action_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_text_master_card_action_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       mastercard_action_id, note_text_master_card_action_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, mastercard_action_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextMasterCardActionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        mastercard_action_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextMasterCardActionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, mastercard_action_id, note_text_master_card_action_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type mastercard_action_id: int
        :type note_text_master_card_action_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextMasterCardAction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     mastercard_action_id, note_text_master_card_action_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextMasterCardAction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextMasterCardAction
        """

        return converter.json_to_class(NoteTextMasterCardAction, json_str)


class NoteAttachmentPaymentBatch(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment-batch/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/payment-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/payment-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment-batch/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment-batch/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, payment_batch_id, attachment_id, monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, payment_batch_id, note_attachment_payment_batch_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_attachment_payment_batch_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id, note_attachment_payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, payment_batch_id, note_attachment_payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_attachment_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id, note_attachment_payment_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, payment_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentPaymentBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        payment_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentPaymentBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, payment_batch_id, note_attachment_payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_attachment_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentPaymentBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_batch_id, note_attachment_payment_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentPaymentBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentPaymentBatch
        """

        return converter.json_to_class(NoteAttachmentPaymentBatch, json_str)


class NoteTextPaymentBatch(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment-batch/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/payment-batch/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/payment-batch/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment-batch/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment-batch/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, payment_batch_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, payment_batch_id, note_text_payment_batch_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_text_payment_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id, note_text_payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, payment_batch_id, note_text_payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_text_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_batch_id, note_text_payment_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, payment_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextPaymentBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        payment_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextPaymentBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, payment_batch_id, note_text_payment_batch_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_batch_id: int
        :type note_text_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextPaymentBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_batch_id, note_text_payment_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextPaymentBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextPaymentBatch
        """

        return converter.json_to_class(NoteTextPaymentBatch, json_str)


class NoteAttachmentPayment(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/payment/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/payment/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, payment_id, attachment_id, monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, payment_id, note_attachment_payment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_attachment_payment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id, note_attachment_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, payment_id, note_attachment_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_attachment_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id, note_attachment_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, payment_id, note_attachment_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_attachment_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_id, note_attachment_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentPayment
        """

        return converter.json_to_class(NoteAttachmentPayment, json_str)


class NoteTextPayment(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/payment/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/payment/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/payment/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/payment/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/payment/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, payment_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, payment_id, note_text_payment_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_text_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id, note_text_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, payment_id, note_text_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_text_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       payment_id, note_text_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextPaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextPaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, payment_id, note_text_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type payment_id: int
        :type note_text_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextPayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     payment_id, note_text_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextPayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextPayment
        """

        return converter.json_to_class(NoteTextPayment, json_str)


class NoteAttachmentRequestInquiryBatch(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, request_inquiry_batch_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_batch_id, note_attachment_request_inquiry_batch_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_attachment_request_inquiry_batch_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id,
                                                       note_attachment_request_inquiry_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_inquiry_batch_id, note_attachment_request_inquiry_batch_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_attachment_request_inquiry_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id,
                                                       note_attachment_request_inquiry_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_inquiry_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestInquiryBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_inquiry_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentRequestInquiryBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_inquiry_batch_id, note_attachment_request_inquiry_batch_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_attachment_request_inquiry_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestInquiryBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_batch_id, note_attachment_request_inquiry_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentRequestInquiryBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentRequestInquiryBatch
        """

        return converter.json_to_class(NoteAttachmentRequestInquiryBatch, json_str)


class NoteTextRequestInquiryBatch(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry-batch/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, request_inquiry_batch_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_batch_id, note_text_request_inquiry_batch_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_text_request_inquiry_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id, note_text_request_inquiry_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_inquiry_batch_id, note_text_request_inquiry_batch_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_text_request_inquiry_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_batch_id, note_text_request_inquiry_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_inquiry_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestInquiryBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_inquiry_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextRequestInquiryBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_inquiry_batch_id, note_text_request_inquiry_batch_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_batch_id: int
        :type note_text_request_inquiry_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestInquiryBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_batch_id, note_text_request_inquiry_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextRequestInquiryBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextRequestInquiryBatch
        """

        return converter.json_to_class(NoteTextRequestInquiryBatch, json_str)


class NoteAttachmentRequestInquiry(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-inquiry/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, request_inquiry_id, attachment_id, monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_id, note_attachment_request_inquiry_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_attachment_request_inquiry_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id, note_attachment_request_inquiry_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_inquiry_id, note_attachment_request_inquiry_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_attachment_request_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id, note_attachment_request_inquiry_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_inquiry_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestInquiryList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_inquiry_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentRequestInquiryList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_inquiry_id, note_attachment_request_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_attachment_request_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_id, note_attachment_request_inquiry_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentRequestInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentRequestInquiry
        """

        return converter.json_to_class(NoteAttachmentRequestInquiry, json_str)


class NoteTextRequestInquiry(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-inquiry/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-inquiry/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-inquiry/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-inquiry/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-inquiry/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, request_inquiry_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_inquiry_id, note_text_request_inquiry_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_text_request_inquiry_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id, note_text_request_inquiry_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_inquiry_id, note_text_request_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_text_request_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_inquiry_id, note_text_request_inquiry_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_inquiry_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestInquiryList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_inquiry_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextRequestInquiryList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_inquiry_id, note_text_request_inquiry_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_inquiry_id: int
        :type note_text_request_inquiry_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestInquiry
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_inquiry_id, note_text_request_inquiry_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextRequestInquiry.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextRequestInquiry
        """

        return converter.json_to_class(NoteTextRequestInquiry, json_str)


class NoteAttachmentRequestResponse(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-response/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-response/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-response/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-response/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-response/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, request_response_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_response_id, note_attachment_request_response_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_attachment_request_response_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id, note_attachment_request_response_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_response_id, note_attachment_request_response_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_attachment_request_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id, note_attachment_request_response_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_response_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestResponseList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_response_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentRequestResponseList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_response_id, note_attachment_request_response_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_attachment_request_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentRequestResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_response_id, note_attachment_request_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentRequestResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentRequestResponse
        """

        return converter.json_to_class(NoteAttachmentRequestResponse, json_str)


class NoteTextRequestResponse(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/request-response/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/request-response/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/request-response/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/request-response/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/request-response/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, request_response_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, request_response_id, note_text_request_response_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_text_request_response_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id, note_text_request_response_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, request_response_id, note_text_request_response_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_text_request_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       request_response_id, note_text_request_response_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, request_response_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestResponseList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        request_response_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextRequestResponseList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, request_response_id, note_text_request_response_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type request_response_id: int
        :type note_text_request_response_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextRequestResponse
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     request_response_id, note_text_request_response_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextRequestResponse.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextRequestResponse
        """

        return converter.json_to_class(NoteTextRequestResponse, json_str)


class NoteAttachmentScheduleInstance(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, schedule_id, schedule_instance_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_id, schedule_instance_id, note_attachment_schedule_instance_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_attachment_schedule_instance_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id,
                                                       note_attachment_schedule_instance_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_id, schedule_instance_id, note_attachment_schedule_instance_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_attachment_schedule_instance_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id,
                                                       note_attachment_schedule_instance_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_id, schedule_instance_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentScheduleInstanceList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_id, schedule_instance_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentScheduleInstanceList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_id, schedule_instance_id, note_attachment_schedule_instance_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_attachment_schedule_instance_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentScheduleInstance
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_id, schedule_instance_id,
                                                     note_attachment_schedule_instance_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentScheduleInstance.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentScheduleInstance
        """

        return converter.json_to_class(NoteAttachmentScheduleInstance, json_str)


class NoteTextScheduleInstance(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule/{}/schedule-instance/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, schedule_id, schedule_instance_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_id, schedule_instance_id, note_text_schedule_instance_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_text_schedule_instance_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id,
                                                       note_text_schedule_instance_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_id, schedule_instance_id, note_text_schedule_instance_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_text_schedule_instance_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_id, schedule_instance_id,
                                                       note_text_schedule_instance_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_id, schedule_instance_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextScheduleInstanceList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_id, schedule_instance_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextScheduleInstanceList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_id, schedule_instance_id, note_text_schedule_instance_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_id: int
        :type schedule_instance_id: int
        :type note_text_schedule_instance_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextScheduleInstance
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_id, schedule_instance_id, note_text_schedule_instance_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextScheduleInstance.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextScheduleInstance
        """

        return converter.json_to_class(NoteTextScheduleInstance, json_str)


class NoteAttachmentSchedulePaymentBatch(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, schedule_payment_batch_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_payment_batch_id, note_attachment_schedule_payment_batch_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_attachment_schedule_payment_batch_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id,
                                                       note_attachment_schedule_payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_batch_id, note_attachment_schedule_payment_batch_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_attachment_schedule_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id,
                                                       note_attachment_schedule_payment_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_payment_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSchedulePaymentBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_payment_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentSchedulePaymentBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_payment_batch_id, note_attachment_schedule_payment_batch_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_attachment_schedule_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSchedulePaymentBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_payment_batch_id,
                                                     note_attachment_schedule_payment_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentSchedulePaymentBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentSchedulePaymentBatch
        """

        return converter.json_to_class(NoteAttachmentSchedulePaymentBatch, json_str)


class NoteTextSchedulePaymentBatch(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule-payment-batch/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, schedule_payment_batch_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_payment_batch_id, note_text_schedule_payment_batch_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_text_schedule_payment_batch_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id, note_text_schedule_payment_batch_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_batch_id, note_text_schedule_payment_batch_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_text_schedule_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_batch_id, note_text_schedule_payment_batch_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_payment_batch_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSchedulePaymentBatchList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_payment_batch_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextSchedulePaymentBatchList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_payment_batch_id, note_text_schedule_payment_batch_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_batch_id: int
        :type note_text_schedule_payment_batch_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSchedulePaymentBatch
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_payment_batch_id, note_text_schedule_payment_batch_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextSchedulePaymentBatch.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextSchedulePaymentBatch
        """

        return converter.json_to_class(NoteTextSchedulePaymentBatch, json_str)


class NoteAttachmentSchedulePayment(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule-payment/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule-payment/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, schedule_payment_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_payment_id, note_attachment_schedule_payment_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_attachment_schedule_payment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id, note_attachment_schedule_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_id, note_attachment_schedule_payment_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_attachment_schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id, note_attachment_schedule_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSchedulePaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentSchedulePaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_payment_id, note_attachment_schedule_payment_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_attachment_schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSchedulePayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_payment_id, note_attachment_schedule_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentSchedulePayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentSchedulePayment
        """

        return converter.json_to_class(NoteAttachmentSchedulePayment, json_str)


class NoteTextSchedulePayment(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/schedule-payment/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/schedule-payment/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/schedule-payment/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/schedule-payment/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/schedule-payment/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, schedule_payment_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, schedule_payment_id, note_text_schedule_payment_id, monetary_account_id=None, content=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_text_schedule_payment_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id, note_text_schedule_payment_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, schedule_payment_id, note_text_schedule_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_text_schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       schedule_payment_id, note_text_schedule_payment_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, schedule_payment_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSchedulePaymentList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        schedule_payment_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextSchedulePaymentList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, schedule_payment_id, note_text_schedule_payment_id, monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type schedule_payment_id: int
        :type note_text_schedule_payment_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSchedulePayment
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     schedule_payment_id, note_text_schedule_payment_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextSchedulePayment.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextSchedulePayment
        """

        return converter.json_to_class(NoteTextSchedulePayment, json_str)


class NoteAttachmentSofortMerchantTransaction(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, sofort_merchant_transaction_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, sofort_merchant_transaction_id, note_attachment_sofort_merchant_transaction_id,
               monetary_account_id=None, description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_attachment_sofort_merchant_transaction_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id,
                                                       note_attachment_sofort_merchant_transaction_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, sofort_merchant_transaction_id, note_attachment_sofort_merchant_transaction_id,
               monetary_account_id=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_attachment_sofort_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id,
                                                       note_attachment_sofort_merchant_transaction_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, sofort_merchant_transaction_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSofortMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        sofort_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentSofortMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, sofort_merchant_transaction_id, note_attachment_sofort_merchant_transaction_id,
            monetary_account_id=None, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_attachment_sofort_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentSofortMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     sofort_merchant_transaction_id,
                                                     note_attachment_sofort_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentSofortMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentSofortMerchantTransaction
        """

        return converter.json_to_class(NoteAttachmentSofortMerchantTransaction, json_str)


class NoteTextSofortMerchantTransaction(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/sofort-merchant-transaction/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, sofort_merchant_transaction_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, sofort_merchant_transaction_id, note_text_sofort_merchant_transaction_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_text_sofort_merchant_transaction_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id,
                                                       note_text_sofort_merchant_transaction_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, sofort_merchant_transaction_id, note_text_sofort_merchant_transaction_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_text_sofort_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       sofort_merchant_transaction_id,
                                                       note_text_sofort_merchant_transaction_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, sofort_merchant_transaction_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSofortMerchantTransactionList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        sofort_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextSofortMerchantTransactionList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, sofort_merchant_transaction_id, note_text_sofort_merchant_transaction_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type sofort_merchant_transaction_id: int
        :type note_text_sofort_merchant_transaction_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextSofortMerchantTransaction
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     sofort_merchant_transaction_id,
                                                     note_text_sofort_merchant_transaction_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextSofortMerchantTransaction.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextSofortMerchantTransaction
        """

        return converter.json_to_class(NoteTextSofortMerchantTransaction, json_str)


class NoteAttachmentWhitelistResult(BunqModel):
    """
    Used to manage attachment notes.
    
    :param _description: Optional description of the attachment.
    :type _description: str
    :param _attachment_id: The reference to the uploaded file to attach to this
    note.
    :type _attachment_id: int
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    :param _attachment: The attachment attached to the note.
    :type _attachment: list[object_.AttachmentMonetaryAccountPayment]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-attachment"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-attachment/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-attachment/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-attachment"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-attachment/{}"

    # Field constants.
    FIELD_DESCRIPTION = "description"
    FIELD_ATTACHMENT_ID = "attachment_id"

    # Object type.
    _OBJECT_TYPE_GET = "NoteAttachment"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _description = None
    _attachment = None
    _description_field_for_request = None
    _attachment_id_field_for_request = None

    def __init__(self, attachment_id, description=None):
        """
        :param attachment_id: The reference to the uploaded file to attach to this
        note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        """

        self._attachment_id_field_for_request = attachment_id
        self._description_field_for_request = description

    @classmethod
    def create(cls, whitelist_id, whitelist_result_id, attachment_id, monetary_account_id=None, description=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :param attachment_id: The reference to the uploaded file to attach to
        this note.
        :type attachment_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_DESCRIPTION: description,
            cls.FIELD_ATTACHMENT_ID: attachment_id
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, whitelist_id, whitelist_result_id, note_attachment_whitelist_result_id, monetary_account_id=None,
               description=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_attachment_whitelist_result_id: int
        :param description: Optional description of the attachment.
        :type description: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_DESCRIPTION: description
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id,
                                                       note_attachment_whitelist_result_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, whitelist_id, whitelist_result_id, note_attachment_whitelist_result_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_attachment_whitelist_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id,
                                                       note_attachment_whitelist_result_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, whitelist_id, whitelist_result_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentWhitelistResultList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        whitelist_id, whitelist_result_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteAttachmentWhitelistResultList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, whitelist_id, whitelist_result_id, note_attachment_whitelist_result_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_attachment_whitelist_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteAttachmentWhitelistResult
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     whitelist_id, whitelist_result_id,
                                                     note_attachment_whitelist_result_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteAttachmentWhitelistResult.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def description(self):
        """
        :rtype: str
        """

        return self._description

    @property
    def attachment(self):
        """
        :rtype: list[object_.AttachmentMonetaryAccountPayment]
        """

        return self._attachment

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._description is not None:
            return False

        if self._attachment is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteAttachmentWhitelistResult
        """

        return converter.json_to_class(NoteAttachmentWhitelistResult, json_str)


class NoteTextWhitelistResult(BunqModel):
    """
    Used to manage text notes.
    
    :param _content: The content of the note.
    :type _content: str
    :param _id_: The id of the note.
    :type _id_: int
    :param _created: The timestamp of the note's creation.
    :type _created: str
    :param _updated: The timestamp of the note's last update.
    :type _updated: str
    :param _label_user_creator: The label of the user who created this note.
    :type _label_user_creator: object_.LabelUser
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-text"
    _ENDPOINT_URL_UPDATE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-text/{}"
    _ENDPOINT_URL_DELETE = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-text/{}"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-text"
    _ENDPOINT_URL_READ = "user/{}/monetary-account/{}/whitelist/{}/whitelist-result/{}/note-text/{}"

    # Field constants.
    FIELD_CONTENT = "content"

    # Object type.
    _OBJECT_TYPE_GET = "NoteText"

    _id_ = None
    _created = None
    _updated = None
    _label_user_creator = None
    _content = None
    _content_field_for_request = None

    def __init__(self, content=None):
        """
        :param content: The content of the note.
        :type content: str
        """

        self._content_field_for_request = content

    @classmethod
    def create(cls, whitelist_id, whitelist_result_id, monetary_account_id=None, content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id)
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def update(cls, whitelist_id, whitelist_result_id, note_text_whitelist_result_id, monetary_account_id=None,
               content=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_text_whitelist_result_id: int
        :param content: The content of the note.
        :type content: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_CONTENT: content
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id, note_text_whitelist_result_id)
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def delete(cls, whitelist_id, whitelist_result_id, note_text_whitelist_result_id, monetary_account_id=None,
               custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_text_whitelist_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNone
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_DELETE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id),
                                                       whitelist_id, whitelist_result_id, note_text_whitelist_result_id)
        response_raw = api_client.delete(endpoint_url, custom_headers)

        return BunqResponseNone.cast_from_bunq_response(
            BunqResponse(None, response_raw.headers)
        )

    @classmethod
    def list(cls, whitelist_id, whitelist_result_id, monetary_account_id=None, params=None, custom_headers=None):
        """
        Manage the notes for a given user.
        
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextWhitelistResultList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id),
                                                        whitelist_id, whitelist_result_id)
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNoteTextWhitelistResultList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def get(cls, whitelist_id, whitelist_result_id, note_text_whitelist_result_id, monetary_account_id=None,
            custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_id: int
        :type monetary_account_id: int
        :type whitelist_id: int
        :type whitelist_result_id: int
        :type note_text_whitelist_result_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNoteTextWhitelistResult
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id(),
                                                     cls._determine_monetary_account_id(monetary_account_id),
                                                     whitelist_id, whitelist_result_id, note_text_whitelist_result_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseNoteTextWhitelistResult.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def label_user_creator(self):
        """
        :rtype: object_.LabelUser
        """

        return self._label_user_creator

    @property
    def content(self):
        """
        :rtype: str
        """

        return self._content

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._label_user_creator is not None:
            return False

        if self._content is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NoteTextWhitelistResult
        """

        return converter.json_to_class(NoteTextWhitelistResult, json_str)


class NotificationFilterPushUser(BunqModel):
    """
    Manage the push notification filters for a user.
    
    :param _notification_filters: The types of notifications that will result in
    a push notification for this user.
    :type _notification_filters: list[object_.NotificationFilterPush]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/notification-filter-push"
    _ENDPOINT_URL_LISTING = "user/{}/notification-filter-push"

    # Field constants.
    FIELD_NOTIFICATION_FILTERS = "notification_filters"

    # Object type.
    _OBJECT_TYPE_POST = "NotificationFilterPush"
    _OBJECT_TYPE_GET = "NotificationFilterPush"

    _notification_filters = None
    _notification_filters_field_for_request = None

    def __init__(self, notification_filters=None):
        """
        :param notification_filters: The types of notifications that will result in
        a push notification for this user.
        :type notification_filters: list[object_.NotificationFilterPush]
        """

        self._notification_filters_field_for_request = notification_filters

    @classmethod
    def create(cls, notification_filters=None, custom_headers=None):
        """
        :type user_id: int
        :param notification_filters: The types of notifications that will result
        in a push notification for this user.
        :type notification_filters: list[object_.NotificationFilterPush]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNotificationFilterPushUser
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NOTIFICATION_FILTERS: notification_filters
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseNotificationFilterPushUser.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_POST)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNotificationFilterPushUserList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNotificationFilterPushUserList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilterPush]
        """

        return self._notification_filters

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._notification_filters is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NotificationFilterPushUser
        """

        return converter.json_to_class(NotificationFilterPushUser, json_str)


class NotificationFilterUrlMonetaryAccount(BunqModel):
    """
    Manage the url notification filters for a user.
    
    :param _notification_filters: The types of notifications that will result in
    a url notification for this monetary account.
    :type _notification_filters: list[object_.NotificationFilterUrl]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/monetary-account/{}/notification-filter-url"
    _ENDPOINT_URL_LISTING = "user/{}/monetary-account/{}/notification-filter-url"

    # Field constants.
    FIELD_NOTIFICATION_FILTERS = "notification_filters"

    # Object type.
    _OBJECT_TYPE_GET = "NotificationFilterUrl"

    _notification_filters = None
    _notification_filters_field_for_request = None

    def __init__(self, notification_filters=None):
        """
        :param notification_filters: The types of notifications that will result in
        a url notification for this monetary account.
        :type notification_filters: list[object_.NotificationFilterUrl]
        """

        self._notification_filters_field_for_request = notification_filters

    @classmethod
    def create(cls, monetary_account_id=None, notification_filters=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :param notification_filters: The types of notifications that will result
        in a url notification for this monetary account.
        :type notification_filters: list[object_.NotificationFilterUrl]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NOTIFICATION_FILTERS: notification_filters
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id(),
                                                       cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, monetary_account_id=None, params=None, custom_headers=None):
        """
        :type user_id: int
        :type monetary_account_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNotificationFilterUrlMonetaryAccountList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id(),
                                                        cls._determine_monetary_account_id(monetary_account_id))
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNotificationFilterUrlMonetaryAccountList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilterUrl]
        """

        return self._notification_filters

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._notification_filters is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NotificationFilterUrlMonetaryAccount
        """

        return converter.json_to_class(NotificationFilterUrlMonetaryAccount, json_str)


class NotificationFilterUrlUser(BunqModel):
    """
    Manage the url notification filters for a user.
    
    :param _notification_filters: The types of notifications that will result in
    a url notification for this user.
    :type _notification_filters: list[object_.NotificationFilterUrl]
    """

    # Endpoint constants.
    _ENDPOINT_URL_CREATE = "user/{}/notification-filter-url"
    _ENDPOINT_URL_LISTING = "user/{}/notification-filter-url"

    # Field constants.
    FIELD_NOTIFICATION_FILTERS = "notification_filters"

    # Object type.
    _OBJECT_TYPE_GET = "NotificationFilterUrl"

    _notification_filters = None
    _notification_filters_field_for_request = None

    def __init__(self, notification_filters=None):
        """
        :param notification_filters: The types of notifications that will result in
        a url notification for this user.
        :type notification_filters: list[object_.NotificationFilterUrl]
        """

        self._notification_filters_field_for_request = notification_filters

    @classmethod
    def create(cls, notification_filters=None, custom_headers=None):
        """
        :type user_id: int
        :param notification_filters: The types of notifications that will result
        in a url notification for this user.
        :type notification_filters: list[object_.NotificationFilterUrl]
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        request_map = {
            cls.FIELD_NOTIFICATION_FILTERS: notification_filters
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        api_client = ApiClient(cls._get_api_context())
        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_CREATE.format(cls._determine_user_id())
        response_raw = api_client.post(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        :type user_id: int
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseNotificationFilterUrlUserList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseNotificationFilterUrlUserList.cast_from_bunq_response(
            cls._from_json_list(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilterUrl]
        """

        return self._notification_filters

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._notification_filters is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: NotificationFilterUrlUser
        """

        return converter.json_to_class(NotificationFilterUrlUser, json_str)


class ChatMessage(BunqModel):
    """
    Endpoint for retrieving the messages that are part of a conversation.
    """

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: ChatMessage
        """

        return converter.json_to_class(ChatMessage, json_str)


class User(BunqModel, AnchorObjectInterface):
    """
    Using this call you can retrieve information of the user you are logged in
    as. This includes your user id, which is referred to in endpoints.
    
    :param _UserLight: 
    :type _UserLight: UserLight
    :param _UserPerson: 
    :type _UserPerson: UserPerson
    :param _UserCompany: 
    :type _UserCompany: UserCompany
    :param _UserApiKey: 
    :type _UserApiKey: UserApiKey
    :param _UserPaymentServiceProvider: 
    :type _UserPaymentServiceProvider: UserPaymentServiceProvider
    """

    # Error constants.
    _ERROR_NULL_FIELDS = "All fields of an extended model or object are null."

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}"
    _ENDPOINT_URL_LISTING = "user"

    # Object type.
    _OBJECT_TYPE_GET = "User"

    _UserLight = None
    _UserPerson = None
    _UserCompany = None
    _UserApiKey = None
    _UserPaymentServiceProvider = None

    @classmethod
    def get(cls, custom_headers=None):
        """
        Get a specific user.
        
        :type api_context: ApiContext
        :type user_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseUser
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseUser.cast_from_bunq_response(
            cls._from_json(response_raw)
        )

    @classmethod
    def list(cls, params=None, custom_headers=None):
        """
        Get a collection of all available users.
        
        :type params: dict[str, str]|None
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseUserList
        """

        if params is None:
            params = {}

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_LISTING
        response_raw = api_client.get(endpoint_url, params, custom_headers)

        return BunqResponseUserList.cast_from_bunq_response(
            cls._from_json_list(response_raw)
        )

    @property
    def UserLight(self):
        """
        :rtype: UserLight
        """

        return self._UserLight

    @property
    def UserPerson(self):
        """
        :rtype: UserPerson
        """

        return self._UserPerson

    @property
    def UserCompany(self):
        """
        :rtype: UserCompany
        """

        return self._UserCompany

    @property
    def UserApiKey(self):
        """
        :rtype: UserApiKey
        """

        return self._UserApiKey

    @property
    def UserPaymentServiceProvider(self):
        """
        :rtype: UserPaymentServiceProvider
        """

        return self._UserPaymentServiceProvider

    def get_referenced_object(self):
        """
        :rtype: BunqModel
        :raise: BunqException
        """

        if self._UserLight is not None:
            return self._UserLight

        if self._UserPerson is not None:
            return self._UserPerson

        if self._UserCompany is not None:
            return self._UserCompany

        if self._UserApiKey is not None:
            return self._UserApiKey

        if self._UserPaymentServiceProvider is not None:
            return self._UserPaymentServiceProvider

        raise BunqException(self._ERROR_NULL_FIELDS)

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._UserLight is not None:
            return False

        if self._UserPerson is not None:
            return False

        if self._UserCompany is not None:
            return False

        if self._UserApiKey is not None:
            return False

        if self._UserPaymentServiceProvider is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: User
        """

        return converter.json_to_class(User, json_str)


class UserLight(BunqModel):
    """
    Show the authenticated user, if it is a light user.
    
    :param _first_name: The user's first name.
    :type _first_name: str
    :param _middle_name: The user's middle name.
    :type _middle_name: str
    :param _last_name: The user's last name.
    :type _last_name: str
    :param _public_nick_name: The public nick name for the user.
    :type _public_nick_name: str
    :param _address_main: The user's main address.
    :type _address_main: object_.Address
    :param _address_postal: The user's postal address.
    :type _address_postal: object_.Address
    :param _avatar_uuid: The public UUID of the user's avatar.
    :type _avatar_uuid: str
    :param _social_security_number: The user's social security number.
    :type _social_security_number: str
    :param _tax_resident: The user's tax residence numbers for different
    countries.
    :type _tax_resident: list[object_.TaxResident]
    :param _document_type: The type of identification document the user
    registered with.
    :type _document_type: str
    :param _document_number: The identification document number the user
    registered with.
    :type _document_number: str
    :param _document_country_of_issuance: The country which issued the
    identification document the user registered with.
    :type _document_country_of_issuance: str
    :param _document_front_attachment_id: The reference to the uploaded
    picture/scan of the front side of the identification document.
    :type _document_front_attachment_id: int
    :param _document_back_attachment_id: The reference to the uploaded
    picture/scan of the back side of the identification document.
    :type _document_back_attachment_id: int
    :param _date_of_birth: The user's date of birth. Accepts ISO8601 date
    formats.
    :type _date_of_birth: str
    :param _place_of_birth: The user's place of birth.
    :type _place_of_birth: str
    :param _country_of_birth: The user's country of birth. Formatted as a SO
    3166-1 alpha-2 country code.
    :type _country_of_birth: str
    :param _nationality: The user's nationality. Formatted as a SO 3166-1
    alpha-2 country code.
    :type _nationality: str
    :param _language: The user's preferred language. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _language: str
    :param _region: The user's preferred region. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _region: str
    :param _gender: The user's gender. Can be MALE, FEMALE or UNKNOWN.
    :type _gender: str
    :param _status: The user status. The user status. Can be: ACTIVE, BLOCKED,
    SIGNUP, DENIED or ABORTED.
    :type _status: str
    :param _sub_status: The user sub-status. Can be: NONE, FACE_RESET, APPROVAL,
    APPROVAL_PARENT, AWAITING_PARENT, APPROVAL_SUPPORT, COUNTER_IBAN, IDEAL or
    SUBMIT.
    :type _sub_status: str
    :param _legal_guardian_alias: The legal guardian of the user. Required for
    minors.
    :type _legal_guardian_alias: object_.Pointer
    :param _session_timeout: The setting for the session timeout of the user in
    seconds.
    :type _session_timeout: int
    :param _daily_limit_without_confirmation_login: The amount the user can pay
    in the session without asking for credentials.
    :type _daily_limit_without_confirmation_login: object_.Amount
    :param _id_: The id of the user.
    :type _id_: int
    :param _created: The timestamp of the user object's creation.
    :type _created: str
    :param _updated: The timestamp of the user object's last update.
    :type _updated: str
    :param _public_uuid: The user's public UUID.
    :type _public_uuid: str
    :param _legal_name: The user's legal name.
    :type _legal_name: str
    :param _display_name: The display name for the user.
    :type _display_name: str
    :param _alias: The aliases of the user.
    :type _alias: list[object_.Pointer]
    :param _avatar: The user's avatar.
    :type _avatar: object_.Avatar
    :param _version_terms_of_service: The version of the terms of service
    accepted by the user.
    :type _version_terms_of_service: str
    :param _notification_filters: The types of notifications that will result in
    a push notification or URL callback for this UserLight.
    :type _notification_filters: list[object_.NotificationFilter]
    :param _deny_reason: The user deny reason.
    :type _deny_reason: str
    :param _relations: The relations for this user.
    :type _relations: list[RelationUser]
    """

    # Field constants.
    FIELD_FIRST_NAME = "first_name"
    FIELD_MIDDLE_NAME = "middle_name"
    FIELD_LAST_NAME = "last_name"
    FIELD_PUBLIC_NICK_NAME = "public_nick_name"
    FIELD_ADDRESS_MAIN = "address_main"
    FIELD_ADDRESS_POSTAL = "address_postal"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_SOCIAL_SECURITY_NUMBER = "social_security_number"
    FIELD_TAX_RESIDENT = "tax_resident"
    FIELD_DOCUMENT_TYPE = "document_type"
    FIELD_DOCUMENT_NUMBER = "document_number"
    FIELD_DOCUMENT_COUNTRY_OF_ISSUANCE = "document_country_of_issuance"
    FIELD_DOCUMENT_FRONT_ATTACHMENT_ID = "document_front_attachment_id"
    FIELD_DOCUMENT_BACK_ATTACHMENT_ID = "document_back_attachment_id"
    FIELD_DATE_OF_BIRTH = "date_of_birth"
    FIELD_PLACE_OF_BIRTH = "place_of_birth"
    FIELD_COUNTRY_OF_BIRTH = "country_of_birth"
    FIELD_NATIONALITY = "nationality"
    FIELD_LANGUAGE = "language"
    FIELD_REGION = "region"
    FIELD_GENDER = "gender"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_LEGAL_GUARDIAN_ALIAS = "legal_guardian_alias"
    FIELD_SESSION_TIMEOUT = "session_timeout"
    FIELD_DAILY_LIMIT_WITHOUT_CONFIRMATION_LOGIN = "daily_limit_without_confirmation_login"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _first_name = None
    _middle_name = None
    _last_name = None
    _legal_name = None
    _display_name = None
    _public_nick_name = None
    _alias = None
    _social_security_number = None
    _tax_resident = None
    _address_main = None
    _address_postal = None
    _date_of_birth = None
    _place_of_birth = None
    _country_of_birth = None
    _nationality = None
    _language = None
    _region = None
    _gender = None
    _avatar = None
    _version_terms_of_service = None
    _status = None
    _sub_status = None
    _session_timeout = None
    _daily_limit_without_confirmation_login = None
    _notification_filters = None
    _deny_reason = None
    _relations = None
    _first_name_field_for_request = None
    _middle_name_field_for_request = None
    _last_name_field_for_request = None
    _public_nick_name_field_for_request = None
    _address_main_field_for_request = None
    _address_postal_field_for_request = None
    _avatar_uuid_field_for_request = None
    _social_security_number_field_for_request = None
    _tax_resident_field_for_request = None
    _document_type_field_for_request = None
    _document_number_field_for_request = None
    _document_country_of_issuance_field_for_request = None
    _document_front_attachment_id_field_for_request = None
    _document_back_attachment_id_field_for_request = None
    _date_of_birth_field_for_request = None
    _place_of_birth_field_for_request = None
    _country_of_birth_field_for_request = None
    _nationality_field_for_request = None
    _language_field_for_request = None
    _region_field_for_request = None
    _gender_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _legal_guardian_alias_field_for_request = None
    _session_timeout_field_for_request = None
    _daily_limit_without_confirmation_login_field_for_request = None

    def __init__(self, document_back_attachment_id=None, session_timeout=None, legal_guardian_alias=None,
                 sub_status=None, status=None, gender=None, region=None, language=None, nationality=None,
                 country_of_birth=None, place_of_birth=None, date_of_birth=None, document_front_attachment_id=None,
                 first_name=None, document_country_of_issuance=None, document_number=None, document_type=None,
                 tax_resident=None, social_security_number=None, avatar_uuid=None, address_postal=None,
                 address_main=None, public_nick_name=None, last_name=None, middle_name=None,
                 daily_limit_without_confirmation_login=None):
        """
        :param first_name: The user's first name.
        :type first_name: str
        :param middle_name: The user's middle name.
        :type middle_name: str
        :param last_name: The user's last name.
        :type last_name: str
        :param public_nick_name: The user's public nick name.
        :type public_nick_name: str
        :param address_main: The user's main address.
        :type address_main: object_.Address
        :param address_postal: The user's postal address.
        :type address_postal: object_.Address
        :param avatar_uuid: The public UUID of the user's avatar.
        :type avatar_uuid: str
        :param social_security_number: The user's social security number.
        :type social_security_number: str
        :param tax_resident: The user's tax residence numbers for different
        countries.
        :type tax_resident: list[object_.TaxResident]
        :param document_type: The type of identification document the user
        registered with.
        :type document_type: str
        :param document_number: The identification document number the user
        registered with.
        :type document_number: str
        :param document_country_of_issuance: The country which issued the
        identification document the user registered with.
        :type document_country_of_issuance: str
        :param document_front_attachment_id: The reference to the uploaded
        picture/scan of the front side of the identification document.
        :type document_front_attachment_id: int
        :param document_back_attachment_id: The reference to the uploaded
        picture/scan of the back side of the identification document.
        :type document_back_attachment_id: int
        :param date_of_birth: The user's date of birth. Accepts ISO8601 date
        formats.
        :type date_of_birth: str
        :param place_of_birth: The user's place of birth.
        :type place_of_birth: str
        :param country_of_birth: The user's country of birth. Formatted as a SO
        3166-1 alpha-2 country code.
        :type country_of_birth: str
        :param nationality: The user's nationality. Formatted as a SO 3166-1 alpha-2
        country code.
        :type nationality: str
        :param language: The user's preferred language. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type language: str
        :param region: The user's preferred region. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type region: str
        :param gender: The user's gender. Can be: MALE, FEMALE and UNKNOWN.
        :type gender: str
        :param status: The user status. You are not allowed to update the status via
        PUT.
        :type status: str
        :param sub_status: The user sub-status. Can be updated to SUBMIT to apply
        for a full bunq account.
        :type sub_status: str
        :param legal_guardian_alias: The legal guardian of the user. Required for
        minors.
        :type legal_guardian_alias: object_.Pointer
        :param session_timeout: The setting for the session timeout of the user in
        seconds.
        :type session_timeout: int
        :param daily_limit_without_confirmation_login: The amount the user can pay
        in the session without asking for credentials.
        :type daily_limit_without_confirmation_login: object_.Amount
        """

        self._first_name_field_for_request = first_name
        self._middle_name_field_for_request = middle_name
        self._last_name_field_for_request = last_name
        self._public_nick_name_field_for_request = public_nick_name
        self._address_main_field_for_request = address_main
        self._address_postal_field_for_request = address_postal
        self._avatar_uuid_field_for_request = avatar_uuid
        self._social_security_number_field_for_request = social_security_number
        self._tax_resident_field_for_request = tax_resident
        self._document_type_field_for_request = document_type
        self._document_number_field_for_request = document_number
        self._document_country_of_issuance_field_for_request = document_country_of_issuance
        self._document_front_attachment_id_field_for_request = document_front_attachment_id
        self._document_back_attachment_id_field_for_request = document_back_attachment_id
        self._date_of_birth_field_for_request = date_of_birth
        self._place_of_birth_field_for_request = place_of_birth
        self._country_of_birth_field_for_request = country_of_birth
        self._nationality_field_for_request = nationality
        self._language_field_for_request = language
        self._region_field_for_request = region
        self._gender_field_for_request = gender
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._legal_guardian_alias_field_for_request = legal_guardian_alias
        self._session_timeout_field_for_request = session_timeout
        self._daily_limit_without_confirmation_login_field_for_request = daily_limit_without_confirmation_login

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def first_name(self):
        """
        :rtype: str
        """

        return self._first_name

    @property
    def middle_name(self):
        """
        :rtype: str
        """

        return self._middle_name

    @property
    def last_name(self):
        """
        :rtype: str
        """

        return self._last_name

    @property
    def legal_name(self):
        """
        :rtype: str
        """

        return self._legal_name

    @property
    def display_name(self):
        """
        :rtype: str
        """

        return self._display_name

    @property
    def public_nick_name(self):
        """
        :rtype: str
        """

        return self._public_nick_name

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def social_security_number(self):
        """
        :rtype: str
        """

        return self._social_security_number

    @property
    def tax_resident(self):
        """
        :rtype: list[object_.TaxResident]
        """

        return self._tax_resident

    @property
    def address_main(self):
        """
        :rtype: object_.Address
        """

        return self._address_main

    @property
    def address_postal(self):
        """
        :rtype: object_.Address
        """

        return self._address_postal

    @property
    def date_of_birth(self):
        """
        :rtype: str
        """

        return self._date_of_birth

    @property
    def place_of_birth(self):
        """
        :rtype: str
        """

        return self._place_of_birth

    @property
    def country_of_birth(self):
        """
        :rtype: str
        """

        return self._country_of_birth

    @property
    def nationality(self):
        """
        :rtype: str
        """

        return self._nationality

    @property
    def language(self):
        """
        :rtype: str
        """

        return self._language

    @property
    def region(self):
        """
        :rtype: str
        """

        return self._region

    @property
    def gender(self):
        """
        :rtype: str
        """

        return self._gender

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def version_terms_of_service(self):
        """
        :rtype: str
        """

        return self._version_terms_of_service

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def session_timeout(self):
        """
        :rtype: int
        """

        return self._session_timeout

    @property
    def daily_limit_without_confirmation_login(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit_without_confirmation_login

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilter]
        """

        return self._notification_filters

    @property
    def deny_reason(self):
        """
        :rtype: str
        """

        return self._deny_reason

    @property
    def relations(self):
        """
        :rtype: list[RelationUser]
        """

        return self._relations

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._first_name is not None:
            return False

        if self._middle_name is not None:
            return False

        if self._last_name is not None:
            return False

        if self._legal_name is not None:
            return False

        if self._display_name is not None:
            return False

        if self._public_nick_name is not None:
            return False

        if self._alias is not None:
            return False

        if self._social_security_number is not None:
            return False

        if self._tax_resident is not None:
            return False

        if self._address_main is not None:
            return False

        if self._address_postal is not None:
            return False

        if self._date_of_birth is not None:
            return False

        if self._place_of_birth is not None:
            return False

        if self._country_of_birth is not None:
            return False

        if self._nationality is not None:
            return False

        if self._language is not None:
            return False

        if self._region is not None:
            return False

        if self._gender is not None:
            return False

        if self._avatar is not None:
            return False

        if self._version_terms_of_service is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._session_timeout is not None:
            return False

        if self._daily_limit_without_confirmation_login is not None:
            return False

        if self._notification_filters is not None:
            return False

        if self._deny_reason is not None:
            return False

        if self._relations is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: UserLight
        """

        return converter.json_to_class(UserLight, json_str)


class UserPerson(BunqModel):
    """
    With UserPerson you can retrieve information regarding the authenticated
    UserPerson and update specific fields.<br/><br/>Notification filters can be
    set on a UserPerson level to receive callbacks. For more information check
    the <a href="/api/1/page/callbacks">dedicated callbacks page</a>.
    
    :param _first_name: The person's first name.
    :type _first_name: str
    :param _middle_name: The person's middle name.
    :type _middle_name: str
    :param _last_name: The person's last name.
    :type _last_name: str
    :param _public_nick_name: The public nick name for the person.
    :type _public_nick_name: str
    :param _address_main: The person's main address.
    :type _address_main: object_.Address
    :param _address_postal: The person's postal address.
    :type _address_postal: object_.Address
    :param _avatar_uuid: The public UUID of the user's avatar.
    :type _avatar_uuid: str
    :param _tax_resident: The user's tax residence numbers for different
    countries.
    :type _tax_resident: list[object_.TaxResident]
    :param _document_type: The type of identification document the person
    registered with.
    :type _document_type: str
    :param _document_number: The identification document number the person
    registered with.
    :type _document_number: str
    :param _document_country_of_issuance: The country which issued the
    identification document the person registered with.
    :type _document_country_of_issuance: str
    :param _document_front_attachment_id: The reference to the uploaded
    picture/scan of the front side of the identification document.
    :type _document_front_attachment_id: int
    :param _document_back_attachment_id: The reference to the uploaded
    picture/scan of the back side of the identification document.
    :type _document_back_attachment_id: int
    :param _date_of_birth: The person's date of birth. Accepts ISO8601 date
    formats.
    :type _date_of_birth: str
    :param _place_of_birth: The person's place of birth.
    :type _place_of_birth: str
    :param _country_of_birth: The person's country of birth. Formatted as a SO
    3166-1 alpha-2 country code.
    :type _country_of_birth: str
    :param _nationality: The person's nationality. Formatted as a SO 3166-1
    alpha-2 country code.
    :type _nationality: str
    :param _language: The person's preferred language. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _language: str
    :param _region: The person's preferred region. Formatted as a ISO 639-1
    language code plus a ISO 3166-1 alpha-2 country code, seperated by an
    underscore.
    :type _region: str
    :param _gender: The person's gender. Can be MALE, FEMALE or UNKNOWN.
    :type _gender: str
    :param _status: The user status. The user status. Can be: ACTIVE, BLOCKED,
    SIGNUP, RECOVERY, DENIED or ABORTED.
    :type _status: str
    :param _sub_status: The user sub-status. Can be: NONE, FACE_RESET, APPROVAL,
    APPROVAL_DIRECTOR, APPROVAL_PARENT, APPROVAL_SUPPORT, COUNTER_IBAN, IDEAL or
    SUBMIT.
    :type _sub_status: str
    :param _legal_guardian_alias: The legal guardian of the user. Required for
    minors.
    :type _legal_guardian_alias: object_.Pointer
    :param _session_timeout: The setting for the session timeout of the user in
    seconds.
    :type _session_timeout: int
    :param _daily_limit_without_confirmation_login: The amount the user can pay
    in the session without asking for credentials.
    :type _daily_limit_without_confirmation_login: object_.Amount
    :param _display_name: The display name for the person.
    :type _display_name: str
    :param _id_: The id of the modified person object.
    :type _id_: int
    :param _created: The timestamp of the person object's creation.
    :type _created: str
    :param _updated: The timestamp of the person object's last update.
    :type _updated: str
    :param _public_uuid: The person's public UUID.
    :type _public_uuid: str
    :param _legal_name: The person's legal name.
    :type _legal_name: str
    :param _alias: The aliases of the user.
    :type _alias: list[object_.Pointer]
    :param _avatar: The user's avatar.
    :type _avatar: object_.Avatar
    :param _version_terms_of_service: The version of the terms of service
    accepted by the user.
    :type _version_terms_of_service: str
    :param _notification_filters: The types of notifications that will result in
    a push notification or URL callback for this UserPerson.
    :type _notification_filters: list[object_.NotificationFilter]
    :param _relations: The relations for this user.
    :type _relations: list[RelationUser]
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user-person/{}"
    _ENDPOINT_URL_UPDATE = "user-person/{}"

    # Field constants.
    FIELD_FIRST_NAME = "first_name"
    FIELD_MIDDLE_NAME = "middle_name"
    FIELD_LAST_NAME = "last_name"
    FIELD_PUBLIC_NICK_NAME = "public_nick_name"
    FIELD_ADDRESS_MAIN = "address_main"
    FIELD_ADDRESS_POSTAL = "address_postal"
    FIELD_AVATAR_UUID = "avatar_uuid"
    FIELD_TAX_RESIDENT = "tax_resident"
    FIELD_DOCUMENT_TYPE = "document_type"
    FIELD_DOCUMENT_NUMBER = "document_number"
    FIELD_DOCUMENT_COUNTRY_OF_ISSUANCE = "document_country_of_issuance"
    FIELD_DOCUMENT_FRONT_ATTACHMENT_ID = "document_front_attachment_id"
    FIELD_DOCUMENT_BACK_ATTACHMENT_ID = "document_back_attachment_id"
    FIELD_DATE_OF_BIRTH = "date_of_birth"
    FIELD_PLACE_OF_BIRTH = "place_of_birth"
    FIELD_COUNTRY_OF_BIRTH = "country_of_birth"
    FIELD_NATIONALITY = "nationality"
    FIELD_LANGUAGE = "language"
    FIELD_REGION = "region"
    FIELD_GENDER = "gender"
    FIELD_STATUS = "status"
    FIELD_SUB_STATUS = "sub_status"
    FIELD_LEGAL_GUARDIAN_ALIAS = "legal_guardian_alias"
    FIELD_SESSION_TIMEOUT = "session_timeout"
    FIELD_DAILY_LIMIT_WITHOUT_CONFIRMATION_LOGIN = "daily_limit_without_confirmation_login"
    FIELD_DISPLAY_NAME = "display_name"

    # Object type.
    _OBJECT_TYPE_GET = "UserPerson"

    _id_ = None
    _created = None
    _updated = None
    _public_uuid = None
    _first_name = None
    _middle_name = None
    _last_name = None
    _legal_name = None
    _display_name = None
    _public_nick_name = None
    _alias = None
    _tax_resident = None
    _address_main = None
    _address_postal = None
    _date_of_birth = None
    _place_of_birth = None
    _country_of_birth = None
    _nationality = None
    _language = None
    _region = None
    _gender = None
    _avatar = None
    _version_terms_of_service = None
    _status = None
    _sub_status = None
    _session_timeout = None
    _daily_limit_without_confirmation_login = None
    _notification_filters = None
    _relations = None
    _first_name_field_for_request = None
    _middle_name_field_for_request = None
    _last_name_field_for_request = None
    _public_nick_name_field_for_request = None
    _address_main_field_for_request = None
    _address_postal_field_for_request = None
    _avatar_uuid_field_for_request = None
    _tax_resident_field_for_request = None
    _document_type_field_for_request = None
    _document_number_field_for_request = None
    _document_country_of_issuance_field_for_request = None
    _document_front_attachment_id_field_for_request = None
    _document_back_attachment_id_field_for_request = None
    _date_of_birth_field_for_request = None
    _place_of_birth_field_for_request = None
    _country_of_birth_field_for_request = None
    _nationality_field_for_request = None
    _language_field_for_request = None
    _region_field_for_request = None
    _gender_field_for_request = None
    _status_field_for_request = None
    _sub_status_field_for_request = None
    _legal_guardian_alias_field_for_request = None
    _session_timeout_field_for_request = None
    _daily_limit_without_confirmation_login_field_for_request = None
    _display_name_field_for_request = None

    def __init__(self, status=None, document_back_attachment_id=None, tax_resident=None, address_postal=None,
                 public_nick_name=None, last_name=None, middle_name=None, first_name=None,
                 daily_limit_without_confirmation_login=None, session_timeout=None, legal_guardian_alias=None,
                 sub_status=None, gender=None, address_main=None, region=None, language=None, nationality=None,
                 country_of_birth=None, place_of_birth=None, date_of_birth=None, document_front_attachment_id=None,
                 document_country_of_issuance=None, document_number=None, document_type=None, avatar_uuid=None,
                 display_name=None):
        """
        :param address_main: The user's main address.
        :type address_main: object_.Address
        :param avatar_uuid: The public UUID of the user's avatar.
        :type avatar_uuid: str
        :param document_type: The type of identification document the person
        registered with.
        :type document_type: str
        :param document_number: The identification document number the person
        registered with.
        :type document_number: str
        :param document_country_of_issuance: The country which issued the
        identification document the person registered with.
        :type document_country_of_issuance: str
        :param document_front_attachment_id: The reference to the uploaded
        picture/scan of the front side of the identification document.
        :type document_front_attachment_id: int
        :param date_of_birth: The person's date of birth. Accepts ISO8601 date
        formats.
        :type date_of_birth: str
        :param place_of_birth: The person's place of birth.
        :type place_of_birth: str
        :param country_of_birth: The person's country of birth. Formatted as a SO
        3166-1 alpha-2 country code.
        :type country_of_birth: str
        :param nationality: The person's nationality. Formatted as a SO 3166-1
        alpha-2 country code.
        :type nationality: str
        :param language: The person's preferred language. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type language: str
        :param region: The person's preferred region. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type region: str
        :param gender: The person's gender. Can be: MALE, FEMALE and UNKNOWN.
        :type gender: str
        :param status: The user status. You are not allowed to update the status via
        PUT.
        :type status: str
        :param sub_status: The user sub-status. Can be updated to SUBMIT if status
        is RECOVERY.
        :type sub_status: str
        :param legal_guardian_alias: The legal guardian of the user. Required for
        minors.
        :type legal_guardian_alias: object_.Pointer
        :param session_timeout: The setting for the session timeout of the user in
        seconds.
        :type session_timeout: int
        :param daily_limit_without_confirmation_login: The amount the user can pay
        in the session without asking for credentials.
        :type daily_limit_without_confirmation_login: object_.Amount
        :param first_name: The person's first name.
        :type first_name: str
        :param middle_name: The person's middle name.
        :type middle_name: str
        :param last_name: The person's last name.
        :type last_name: str
        :param public_nick_name: The person's public nick name.
        :type public_nick_name: str
        :param address_postal: The person's postal address.
        :type address_postal: object_.Address
        :param tax_resident: The user's tax residence numbers for different
        countries.
        :type tax_resident: list[object_.TaxResident]
        :param document_back_attachment_id: The reference to the uploaded
        picture/scan of the back side of the identification document.
        :type document_back_attachment_id: int
        :param display_name: The person's legal name. Available legal names can be
        listed via the 'user/{user_id}/legal-name' endpoint.
        :type display_name: str
        """

        self._address_main_field_for_request = address_main
        self._avatar_uuid_field_for_request = avatar_uuid
        self._document_type_field_for_request = document_type
        self._document_number_field_for_request = document_number
        self._document_country_of_issuance_field_for_request = document_country_of_issuance
        self._document_front_attachment_id_field_for_request = document_front_attachment_id
        self._date_of_birth_field_for_request = date_of_birth
        self._place_of_birth_field_for_request = place_of_birth
        self._country_of_birth_field_for_request = country_of_birth
        self._nationality_field_for_request = nationality
        self._language_field_for_request = language
        self._region_field_for_request = region
        self._gender_field_for_request = gender
        self._status_field_for_request = status
        self._sub_status_field_for_request = sub_status
        self._legal_guardian_alias_field_for_request = legal_guardian_alias
        self._session_timeout_field_for_request = session_timeout
        self._daily_limit_without_confirmation_login_field_for_request = daily_limit_without_confirmation_login
        self._first_name_field_for_request = first_name
        self._middle_name_field_for_request = middle_name
        self._last_name_field_for_request = last_name
        self._public_nick_name_field_for_request = public_nick_name
        self._address_postal_field_for_request = address_postal
        self._tax_resident_field_for_request = tax_resident
        self._document_back_attachment_id_field_for_request = document_back_attachment_id
        self._display_name_field_for_request = display_name

    @classmethod
    def get(cls, custom_headers=None):
        """
        Get a specific person.
        
        :type api_context: ApiContext
        :type user_person_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseUserPerson
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(cls._determine_user_id())
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseUserPerson.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @classmethod
    def update(cls, first_name=None, middle_name=None, last_name=None, public_nick_name=None, address_main=None,
               address_postal=None, avatar_uuid=None, tax_resident=None, document_type=None, document_number=None,
               document_country_of_issuance=None, document_front_attachment_id=None, document_back_attachment_id=None,
               date_of_birth=None, place_of_birth=None, country_of_birth=None, nationality=None, language=None,
               region=None, gender=None, status=None, sub_status=None, legal_guardian_alias=None, session_timeout=None,
               daily_limit_without_confirmation_login=None, display_name=None, custom_headers=None):
        """
        Modify a specific person object's data.
        
        :type user_person_id: int
        :param first_name: The person's first name.
        :type first_name: str
        :param middle_name: The person's middle name.
        :type middle_name: str
        :param last_name: The person's last name.
        :type last_name: str
        :param public_nick_name: The person's public nick name.
        :type public_nick_name: str
        :param address_main: The user's main address.
        :type address_main: object_.Address
        :param address_postal: The person's postal address.
        :type address_postal: object_.Address
        :param avatar_uuid: The public UUID of the user's avatar.
        :type avatar_uuid: str
        :param tax_resident: The user's tax residence numbers for different
        countries.
        :type tax_resident: list[object_.TaxResident]
        :param document_type: The type of identification document the person
        registered with.
        :type document_type: str
        :param document_number: The identification document number the person
        registered with.
        :type document_number: str
        :param document_country_of_issuance: The country which issued the
        identification document the person registered with.
        :type document_country_of_issuance: str
        :param document_front_attachment_id: The reference to the uploaded
        picture/scan of the front side of the identification document.
        :type document_front_attachment_id: int
        :param document_back_attachment_id: The reference to the uploaded
        picture/scan of the back side of the identification document.
        :type document_back_attachment_id: int
        :param date_of_birth: The person's date of birth. Accepts ISO8601 date
        formats.
        :type date_of_birth: str
        :param place_of_birth: The person's place of birth.
        :type place_of_birth: str
        :param country_of_birth: The person's country of birth. Formatted as a
        SO 3166-1 alpha-2 country code.
        :type country_of_birth: str
        :param nationality: The person's nationality. Formatted as a SO 3166-1
        alpha-2 country code.
        :type nationality: str
        :param language: The person's preferred language. Formatted as a ISO
        639-1 language code plus a ISO 3166-1 alpha-2 country code, seperated by
        an underscore.
        :type language: str
        :param region: The person's preferred region. Formatted as a ISO 639-1
        language code plus a ISO 3166-1 alpha-2 country code, seperated by an
        underscore.
        :type region: str
        :param gender: The person's gender. Can be: MALE, FEMALE and UNKNOWN.
        :type gender: str
        :param status: The user status. You are not allowed to update the status
        via PUT.
        :type status: str
        :param sub_status: The user sub-status. Can be updated to SUBMIT if
        status is RECOVERY.
        :type sub_status: str
        :param legal_guardian_alias: The legal guardian of the user. Required
        for minors.
        :type legal_guardian_alias: object_.Pointer
        :param session_timeout: The setting for the session timeout of the user
        in seconds.
        :type session_timeout: int
        :param daily_limit_without_confirmation_login: The amount the user can
        pay in the session without asking for credentials.
        :type daily_limit_without_confirmation_login: object_.Amount
        :param display_name: The person's legal name. Available legal names can
        be listed via the 'user/{user_id}/legal-name' endpoint.
        :type display_name: str
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseInt
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())

        request_map = {
            cls.FIELD_FIRST_NAME: first_name,
            cls.FIELD_MIDDLE_NAME: middle_name,
            cls.FIELD_LAST_NAME: last_name,
            cls.FIELD_PUBLIC_NICK_NAME: public_nick_name,
            cls.FIELD_ADDRESS_MAIN: address_main,
            cls.FIELD_ADDRESS_POSTAL: address_postal,
            cls.FIELD_AVATAR_UUID: avatar_uuid,
            cls.FIELD_TAX_RESIDENT: tax_resident,
            cls.FIELD_DOCUMENT_TYPE: document_type,
            cls.FIELD_DOCUMENT_NUMBER: document_number,
            cls.FIELD_DOCUMENT_COUNTRY_OF_ISSUANCE: document_country_of_issuance,
            cls.FIELD_DOCUMENT_FRONT_ATTACHMENT_ID: document_front_attachment_id,
            cls.FIELD_DOCUMENT_BACK_ATTACHMENT_ID: document_back_attachment_id,
            cls.FIELD_DATE_OF_BIRTH: date_of_birth,
            cls.FIELD_PLACE_OF_BIRTH: place_of_birth,
            cls.FIELD_COUNTRY_OF_BIRTH: country_of_birth,
            cls.FIELD_NATIONALITY: nationality,
            cls.FIELD_LANGUAGE: language,
            cls.FIELD_REGION: region,
            cls.FIELD_GENDER: gender,
            cls.FIELD_STATUS: status,
            cls.FIELD_SUB_STATUS: sub_status,
            cls.FIELD_LEGAL_GUARDIAN_ALIAS: legal_guardian_alias,
            cls.FIELD_SESSION_TIMEOUT: session_timeout,
            cls.FIELD_DAILY_LIMIT_WITHOUT_CONFIRMATION_LOGIN: daily_limit_without_confirmation_login,
            cls.FIELD_DISPLAY_NAME: display_name
        }
        request_map_string = converter.class_to_json(request_map)
        request_map_string = cls._remove_field_for_request(request_map_string)

        request_bytes = request_map_string.encode()
        endpoint_url = cls._ENDPOINT_URL_UPDATE.format(cls._determine_user_id())
        response_raw = api_client.put(endpoint_url, request_bytes, custom_headers)

        return BunqResponseInt.cast_from_bunq_response(
            cls._process_for_id(response_raw)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def first_name(self):
        """
        :rtype: str
        """

        return self._first_name

    @property
    def middle_name(self):
        """
        :rtype: str
        """

        return self._middle_name

    @property
    def last_name(self):
        """
        :rtype: str
        """

        return self._last_name

    @property
    def legal_name(self):
        """
        :rtype: str
        """

        return self._legal_name

    @property
    def display_name(self):
        """
        :rtype: str
        """

        return self._display_name

    @property
    def public_nick_name(self):
        """
        :rtype: str
        """

        return self._public_nick_name

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def tax_resident(self):
        """
        :rtype: list[object_.TaxResident]
        """

        return self._tax_resident

    @property
    def address_main(self):
        """
        :rtype: object_.Address
        """

        return self._address_main

    @property
    def address_postal(self):
        """
        :rtype: object_.Address
        """

        return self._address_postal

    @property
    def date_of_birth(self):
        """
        :rtype: str
        """

        return self._date_of_birth

    @property
    def place_of_birth(self):
        """
        :rtype: str
        """

        return self._place_of_birth

    @property
    def country_of_birth(self):
        """
        :rtype: str
        """

        return self._country_of_birth

    @property
    def nationality(self):
        """
        :rtype: str
        """

        return self._nationality

    @property
    def language(self):
        """
        :rtype: str
        """

        return self._language

    @property
    def region(self):
        """
        :rtype: str
        """

        return self._region

    @property
    def gender(self):
        """
        :rtype: str
        """

        return self._gender

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def version_terms_of_service(self):
        """
        :rtype: str
        """

        return self._version_terms_of_service

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def session_timeout(self):
        """
        :rtype: int
        """

        return self._session_timeout

    @property
    def daily_limit_without_confirmation_login(self):
        """
        :rtype: object_.Amount
        """

        return self._daily_limit_without_confirmation_login

    @property
    def notification_filters(self):
        """
        :rtype: list[object_.NotificationFilter]
        """

        return self._notification_filters

    @property
    def relations(self):
        """
        :rtype: list[RelationUser]
        """

        return self._relations

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._first_name is not None:
            return False

        if self._middle_name is not None:
            return False

        if self._last_name is not None:
            return False

        if self._legal_name is not None:
            return False

        if self._display_name is not None:
            return False

        if self._public_nick_name is not None:
            return False

        if self._alias is not None:
            return False

        if self._tax_resident is not None:
            return False

        if self._address_main is not None:
            return False

        if self._address_postal is not None:
            return False

        if self._date_of_birth is not None:
            return False

        if self._place_of_birth is not None:
            return False

        if self._country_of_birth is not None:
            return False

        if self._nationality is not None:
            return False

        if self._language is not None:
            return False

        if self._region is not None:
            return False

        if self._gender is not None:
            return False

        if self._avatar is not None:
            return False

        if self._version_terms_of_service is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._session_timeout is not None:
            return False

        if self._daily_limit_without_confirmation_login is not None:
            return False

        if self._notification_filters is not None:
            return False

        if self._relations is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: UserPerson
        """

        return converter.json_to_class(UserPerson, json_str)


class UserApiKey(BunqModel):
    """
    Used to view OAuth request detais in events.
    
    :param _id_: The id of the user.
    :type _id_: int
    :param _created: The timestamp of the user object's creation.
    :type _created: str
    :param _updated: The timestamp of the user object's last update.
    :type _updated: str
    :param _requested_by_user: The user who requested access.
    :type _requested_by_user: object_.UserApiKeyAnchoredUser
    :param _granted_by_user: The user who granted access.
    :type _granted_by_user: object_.UserApiKeyAnchoredUser
    """

    _id_ = None
    _created = None
    _updated = None
    _requested_by_user = None
    _granted_by_user = None

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def requested_by_user(self):
        """
        :rtype: object_.UserApiKeyAnchoredUser
        """

        return self._requested_by_user

    @property
    def granted_by_user(self):
        """
        :rtype: object_.UserApiKeyAnchoredUser
        """

        return self._granted_by_user

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._requested_by_user is not None:
            return False

        if self._granted_by_user is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: UserApiKey
        """

        return converter.json_to_class(UserApiKey, json_str)


class UserPaymentServiceProvider(BunqModel):
    """
    Used to view UserPaymentServiceProvider for session creation.
    
    :param _id_: The id of the user.
    :type _id_: int
    :param _created: The timestamp of the user object's creation.
    :type _created: str
    :param _updated: The timestamp of the user object's last update.
    :type _updated: str
    :param _certificate_distinguished_name: The distinguished name from the
    certificate used to identify this user.
    :type _certificate_distinguished_name: str
    :param _alias: The aliases of the user.
    :type _alias: list[object_.Pointer]
    :param _avatar: The user's avatar.
    :type _avatar: object_.Avatar
    :param _status: The user status. The user status. Can be: ACTIVE, BLOCKED or
    DENIED.
    :type _status: str
    :param _sub_status: The user sub-status. Can be: NONE
    :type _sub_status: str
    :param _public_uuid: The providers's public UUID.
    :type _public_uuid: str
    :param _display_name: The display name for the provider.
    :type _display_name: str
    :param _public_nick_name: The public nick name for the provider.
    :type _public_nick_name: str
    :param _language: The provider's language. Formatted as a ISO 639-1 language
    code plus a ISO 3166-1 alpha-2 country code, separated by an underscore.
    :type _language: str
    :param _region: The provider's region. Formatted as a ISO 639-1 language
    code plus a ISO 3166-1 alpha-2 country code, separated by an underscore.
    :type _region: str
    :param _session_timeout: The setting for the session timeout of the user in
    seconds.
    :type _session_timeout: int
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user-payment-service-provider/{}"

    # Object type.
    _OBJECT_TYPE_GET = "UserPaymentServiceProvider"

    _id_ = None
    _created = None
    _updated = None
    _certificate_distinguished_name = None
    _alias = None
    _avatar = None
    _status = None
    _sub_status = None
    _public_uuid = None
    _display_name = None
    _public_nick_name = None
    _language = None
    _region = None
    _session_timeout = None

    @classmethod
    def get(cls, user_payment_service_provider_id, custom_headers=None):
        """
        :type api_context: ApiContext
        :type user_payment_service_provider_id: int
        :type custom_headers: dict[str, str]|None
        
        :rtype: BunqResponseUserPaymentServiceProvider
        """

        if custom_headers is None:
            custom_headers = {}

        api_client = ApiClient(cls._get_api_context())
        endpoint_url = cls._ENDPOINT_URL_READ.format(user_payment_service_provider_id)
        response_raw = api_client.get(endpoint_url, {}, custom_headers)

        return BunqResponseUserPaymentServiceProvider.cast_from_bunq_response(
            cls._from_json(response_raw, cls._OBJECT_TYPE_GET)
        )

    @property
    def id_(self):
        """
        :rtype: int
        """

        return self._id_

    @property
    def created(self):
        """
        :rtype: str
        """

        return self._created

    @property
    def updated(self):
        """
        :rtype: str
        """

        return self._updated

    @property
    def certificate_distinguished_name(self):
        """
        :rtype: str
        """

        return self._certificate_distinguished_name

    @property
    def alias(self):
        """
        :rtype: list[object_.Pointer]
        """

        return self._alias

    @property
    def avatar(self):
        """
        :rtype: object_.Avatar
        """

        return self._avatar

    @property
    def status(self):
        """
        :rtype: str
        """

        return self._status

    @property
    def sub_status(self):
        """
        :rtype: str
        """

        return self._sub_status

    @property
    def public_uuid(self):
        """
        :rtype: str
        """

        return self._public_uuid

    @property
    def display_name(self):
        """
        :rtype: str
        """

        return self._display_name

    @property
    def public_nick_name(self):
        """
        :rtype: str
        """

        return self._public_nick_name

    @property
    def language(self):
        """
        :rtype: str
        """

        return self._language

    @property
    def region(self):
        """
        :rtype: str
        """

        return self._region

    @property
    def session_timeout(self):
        """
        :rtype: int
        """

        return self._session_timeout

    def is_all_field_none(self):
        """
        :rtype: bool
        """

        if self._id_ is not None:
            return False

        if self._created is not None:
            return False

        if self._updated is not None:
            return False

        if self._certificate_distinguished_name is not None:
            return False

        if self._alias is not None:
            return False

        if self._avatar is not None:
            return False

        if self._status is not None:
            return False

        if self._sub_status is not None:
            return False

        if self._public_uuid is not None:
            return False

        if self._display_name is not None:
            return False

        if self._public_nick_name is not None:
            return False

        if self._language is not None:
            return False

        if self._region is not None:
            return False

        if self._session_timeout is not None:
            return False

        return True

    @staticmethod
    def from_json(json_str):
        """
        :type json_str: str
        
        :rtype: UserPaymentServiceProvider
        """

        return converter.json_to_class(UserPaymentServiceProvider, json_str)


class OauthCallbackUrl(BunqModel):
    """
    Used for managing OAuth Client Callback URLs.
    
    :param _url: The URL for this callback.
    :type _url: str
    """

    # Endpoint constants.
    _ENDPOINT_URL_READ = "user/{}/oauth-client/{}/callback-url/{}"
    _ENDPOINT_URL_CREATE = "user/{}/oauth-client/{}/callback-url"
    _ENDPOINT_URL_UPDATE = "user/{}/oauth-client/{}/callback-url/{}"
    _ENDPOINT_URL_LISTING = "user/{}/oauth-client/{}/callback-url"
    _ENDPOINT_URL_DELETE = "user/{}/oauth-client/{}/callback-url/{}"

    # Field constants.
    FIELD_URL = "url"

    # Object type.
    _OBJECT_TYPE_GET = "OauthCallbackUrl"

    _url = None
    _url_f