"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultCloudFrontDisributionForMediaStoreProps = exports.DefaultCloudFrontWebDistributionForS3Props = exports.DefaultCloudFrontWebDistributionForApiGatewayProps = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const cdk = require("@aws-cdk/core");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
function DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, setHttpSecurityHeaders, cfFunction) {
    const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
    const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: new origins.HttpOrigin(apiEndPointDomainName, {
                    originPath: `/${apiEndPoint.deploymentStage.stageName}`
                }),
                functionAssociations: [
                    {
                        eventType: aws_cloudfront_1.FunctionEventType.VIEWER_RESPONSE,
                        function: cfFunction
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: new origins.HttpOrigin(apiEndPointDomainName, {
                    originPath: `/${apiEndPoint.deploymentStage.stageName}`
                }),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
        };
    }
}
exports.DefaultCloudFrontWebDistributionForApiGatewayProps = DefaultCloudFrontWebDistributionForApiGatewayProps;
function DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, setHttpSecurityHeaders, cfFunction) {
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: new origins.S3Origin(sourceBucket),
                functionAssociations: [
                    {
                        eventType: aws_cloudfront_1.FunctionEventType.VIEWER_RESPONSE,
                        function: cfFunction
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
            defaultRootObject: 'index.html'
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: new origins.S3Origin(sourceBucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
            defaultRootObject: 'index.html'
        };
    }
}
exports.DefaultCloudFrontWebDistributionForS3Props = DefaultCloudFrontWebDistributionForS3Props;
function DefaultCloudFrontDisributionForMediaStoreProps(mediastoreContainer, loggingBucket, originRequestPolicy, setHttpSecurityHeaders, customHeaders, cfFunction) {
    const mediaStoreContainerUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split('://', mediastoreContainer.attrEndpoint));
    const mediaStoreContainerDomainName = cdk.Fn.select(0, cdk.Fn.split('/', mediaStoreContainerUrlWithoutProtocol));
    const httpOrigin = customHeaders ?
        new origins.HttpOrigin(mediaStoreContainerDomainName, { customHeaders }) :
        new origins.HttpOrigin(mediaStoreContainerDomainName);
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: httpOrigin,
                functionAssociations: [
                    {
                        eventType: aws_cloudfront_1.FunctionEventType.VIEWER_RESPONSE,
                        function: cfFunction
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                originRequestPolicy
            },
            enableLogging: true,
            logBucket: loggingBucket
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: httpOrigin,
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                originRequestPolicy
            },
            enableLogging: true,
            logBucket: loggingBucket
        };
    }
}
exports.DefaultCloudFrontDisributionForMediaStoreProps = DefaultCloudFrontDisributionForMediaStoreProps;
//# sourceMappingURL=data:application/json;base64,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