"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders) {
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    let defaultprops;
    let cloudfrontFunction;
    let loggingBucket;
    if (_httpSecurityHeaders) {
        cloudfrontFunction = defaultCloudfrontFunction(scope);
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, cloudFrontDistributionProps.logBucket, _httpSecurityHeaders, cloudfrontFunction);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, _httpSecurityHeaders, cloudfrontFunction);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders) {
    let defaultprops;
    let cloudfrontFunction;
    let loggingBucket;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        cloudfrontFunction = defaultCloudfrontFunction(scope);
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, cloudFrontDistributionProps.logBucket, _httpSecurityHeaders, cloudfrontFunction);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, _httpSecurityHeaders, cloudfrontFunction);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders) {
    let defaultprops;
    let originRequestPolicy;
    let loggingBucket;
    let cloudfrontFunction;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        cloudfrontFunction = defaultCloudfrontFunction(scope);
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        loggingBucket = cloudFrontDistributionProps.logBucket;
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
    }
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, _httpSecurityHeaders, cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.customHeaders, cloudfrontFunction);
    let cfprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false);
    }
    else {
        cfprops = defaultprops;
    }
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, cloudfrontFunction];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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