"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildLogGroup = void 0;
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const logs = require("@aws-cdk/aws-logs");
const utils_1 = require("./utils");
function buildLogGroup(scope, logGroupId, logGroupProps) {
    let _logGroupProps;
    // Override user provided CW LogGroup props with the DefaultLogGroupProps
    if (logGroupProps !== undefined) {
        _logGroupProps = utils_1.overrideProps(cloudwatch_log_group_defaults_1.DefaultLogGroupProps(), logGroupProps);
    }
    else {
        _logGroupProps = cloudwatch_log_group_defaults_1.DefaultLogGroupProps();
    }
    // Set the LogGroup Id
    const _logGroupId = logGroupId ? logGroupId : 'CloudWatchLogGroup';
    // Create the CW Log Group
    const logGroup = new logs.LogGroup(scope, _logGroupId, _logGroupProps);
    // If required, suppress the Cfn Nag WARNINGS
    if (_logGroupProps.retention === logs.RetentionDays.INFINITE) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W86',
                reason: 'Retention period for CloudWatchLogs LogGroups are set to \'Never Expire\' to preserve customer data indefinitely'
            }
        ]);
    }
    if (!_logGroupProps.encryptionKey) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W84',
                reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
            }
        ]);
    }
    return logGroup;
}
exports.buildLogGroup = buildLogGroup;
//# sourceMappingURL=data:application/json;base64,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