"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupCognitoForElasticSearch = exports.buildIdentityPool = exports.buildUserPoolClient = exports.buildUserPool = void 0;
const cognito = require("@aws-cdk/aws-cognito");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    if (userPoolProps) {
        cognitoUserPoolProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    }
    else {
        cognitoUserPoolProps = cognito_defaults_1.DefaultUserPoolProps;
    }
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    // Add Cfn Nag suppress for the cognito SMS role policy
    const userPoolSmsRole = userPool.node.tryFindChild('smsRole');
    if (userPoolSmsRole) {
        utils_1.addCfnSuppressRules(userPool, [
            {
                id: 'W11',
                reason: `Allowing * resource on permissions policy since its used by Cognito to send SMS messages via sns:Publish`
            }
        ]);
    }
    return userPool;
}
exports.buildUserPool = buildUserPool;
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    if (cognitoUserPoolClientProps) {
        userPoolClientProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    }
    else {
        userPoolClientProps = cognito_defaults_1.DefaultUserPoolClientProps(userPool);
    }
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    if (identityPoolProps) {
        cognitoIdentityPoolProps = utils_1.overrideProps(cognitoIdentityPoolProps, identityPoolProps);
    }
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
function setupCognitoForElasticSearch(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependsOn(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForElasticSearch = setupCognitoForElasticSearch;
//# sourceMappingURL=data:application/json;base64,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