"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("@aws-cdk/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    if (cfnDomainProps) {
        _cfnDomainProps = utils_1.overrideProps(_cfnDomainProps, cfnDomainProps);
    }
    const esDomain = new elasticsearch.CfnDomain(scope, "ElasticsearchDomain", _cfnDomainProps);
    utils_1.addCfnSuppressRules(esDomain, [
        {
            id: "W28",
            reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return [esDomain, cognitoKibanaConfigureRole];
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace'
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        statistic: 'Minimum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization'
        }),
        threshold: 80,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure'
        }),
        threshold: 80,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization'
        }),
        threshold: 50,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure'
        }),
        threshold: 50,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,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