"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
const duration_1 = require("@aws-cdk/core/lib/duration");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`,
            `${s3Bucket.bucketArn}`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId, removalPolicy) {
    let loggingBucketProps;
    if (removalPolicy) {
        loggingBucketProps = utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), { removalPolicy });
    }
    else {
        loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
    }
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: duration_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.removalPolicy);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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